/*
 * CdfFileDataSetDescriptor.java
 *
 * Created on August 12, 2005, 3:07 PM
 *
 *
 */
package org.virbo.metatree;

import org.das2.datum.DatumRange;
import org.das2.datum.Units;
import java.lang.reflect.Array;
import java.text.ParseException;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.Map;
import org.das2.datum.UnitsUtil;
import org.virbo.dataset.QDataSet;
import org.virbo.dataset.SemanticOps;
import org.virbo.datasource.MetadataModel;
import org.virbo.datasource.DataSourceUtil;

/**
 *
 * @author Jeremy
 *
 */
public class IstpMetadataModel extends MetadataModel {

    /**
     * Non-null, non-empty String if it is virtual.  The string will be like "compute_magnitude(B_xyz_gse)"
     */
    public static final String USER_PROP_VIRTUAL_FUNCTION= "FUNCTION";

    public static final String USER_PROP_VIRTUAL_COMPONENT_= "COMPONENT_";

    public static final Object VALUE_MIN= "MIN";
    public static final Object VALUE_MAX= "MAX";

    /**
     * returns the Entry that is convertible to double as a double.
     * @throws IllegalArgumentException for strings
     */
    private static double doubleValue(Object o, Units units, Object minmax ) {
        return doubleValue(o, units, Double.NaN, minmax ); 
    }

    /**
     * returns the Entry that is convertible to double as a double.
     * When there is an array, throw IllegalArgumentException
     * @param o the datum in double, int, String, array, etc.
     * @param units the units of the datum
     * @param deflt the default value
     * @param minmax VALUE_MIN or VALUE_MAX or null.
     * @throws IllegalArgumentException for strings
     */
    public static double doubleValue(Object o, Units units, double deflt, Object minmax ) {
        if (o == null) {
            return deflt;
        }
        if (o instanceof Float) {
            return ((Float) o).doubleValue();
        } else if (o instanceof Double) {
            return ((Double) o).doubleValue();
        } else if (o instanceof Short) {
            return ((Short) o).doubleValue();
        } else if (o instanceof Integer) {
            return ((Integer) o).doubleValue();
        } else if (o instanceof Long) {
            return ((Long) o).doubleValue();
        } else if (o instanceof Byte) {
            return ((Byte) o).doubleValue();
        } else if (o instanceof String) {
            String s = (String) o;
            if (s.startsWith("CDF_PARSE_EPOCH(")) {  // hack for Onera CDFs
                try {
                    // hack for Onera CDFs
                    return units.parse(s.substring(16, s.length() - 1)).doubleValue(units);
                } catch (ParseException ex) {
                    throw new IllegalArgumentException("unable to parse " + o);
                }
            } else {
                try {
                    return units.parse(DataSourceUtil.unquote((String) o)).doubleValue(units);
                } catch (ParseException ex) {
                    try {
                        return Double.parseDouble((String) o);
                    } catch (NumberFormatException ex2) {
                        throw new IllegalArgumentException("unable to parse " + o);
                    }
                }
            }
        } else {
            Class c = o.getClass();
            if (c.isArray()) {
                if (units == Units.cdfEpoch && Array.getLength(o) == 2) { // kludge for Epoch16
                    double cdfEpoch = Array.getDouble(o, 0) * 1000 + Array.getDouble(o, 1) / 1e9;
                    Units.cdfEpoch.createDatum(cdfEpoch);
                    return cdfEpoch;
                } else {
                    double v= Array.getDouble(o, 0);
                    int n= Array.getLength(o);
                    for ( int i=1; i<n; i++ ) {
                        if ( minmax==VALUE_MAX ) {
                            v= Math.max( v, Array.getDouble(o,i) );
                        } else if ( minmax==VALUE_MIN ) {
                            v= Math.min( v, Array.getDouble(o,i) );
                        } else {
                            throw new IllegalArgumentException("object is array: "+o+", and minmax is not set");
                        }
                    }
                    return v;
                }
            } else {
                throw new RuntimeException("Unsupported Data Type: " + o.getClass().getName());
            }
        }
    }

    /**
     * Return the range from VALIDMIN to VALIDMAX.
     * Note QDataSet only allows times from 1000AD to 9000AD when Units are TimeLocationUnits.
     */
    public static DatumRange getValidRange(Map attrs, Units units) {
        double max = doubleValue(attrs.get("VALIDMAX"), units, Double.MAX_VALUE, VALUE_MAX );
        double min = doubleValue(attrs.get("VALIDMIN"), units, -1e29, VALUE_MIN ); //TODO: remove limitation
        if ( units.isFill(min) ) min= min / 100; // kludge because DatumRanges cannot contain fill.
        if ( UnitsUtil.isTimeLocation(units) ) {
            DatumRange vrange= new DatumRange( 3.15569952E13, 2.840126112E14, Units.cdfEpoch ); // approx 1000AD to 9000AD
            if ( vrange.min().doubleValue(units)>min ) min= vrange.min().doubleValue(units);
            if ( vrange.max().doubleValue(units)<max ) max= vrange.max().doubleValue(units);
            if ( vrange.min().doubleValue(units)>max ) max= vrange.max().doubleValue(units); //vap+cdaweb:ds=IM_HK_FSW&id=BF_DramMbeCnt&timerange=2005-12-18
        }
        return DatumRange.newDatumRange(min, max, units);
    }

    /**
     * returns the range of the data by looking for the SCALEMIN/SCALEMAX params,
     * or the required VALIDMIN/VALIDMAX parameters.  Checks for valid range when
     * SCALETYP=log.
     * Note QDataSet only allows times from 1000AD to 9000AD when Units are TimeLocationUnits.
     */
    private static DatumRange getRange(Map attrs, Units units) {
        DatumRange range;

        double min, max;
        if (attrs.containsKey("SCALEMIN") && attrs.containsKey("SCALEMAX")) {
            max = doubleValue(attrs.get("SCALEMAX"), units, VALUE_MAX );
            min = doubleValue(attrs.get("SCALEMIN"), units, VALUE_MIN );
        } else {
            if (attrs.containsKey("SCALEMAX")) {
                max = doubleValue(attrs.get("SCALEMAX"), units, VALUE_MAX );
                min = 0;
            } else {
                max = doubleValue(attrs.get("VALIDMAX"), units, Double.MAX_VALUE, VALUE_MAX );
                min = doubleValue(attrs.get("VALIDMIN"), units, -1e29, VALUE_MIN );
                if ( min>0 && max/min>1e20 ) {
                    return null;
                }
            }
        }
        if ("log".equals(getScaleType(attrs)) && min <= 0) {
            min = max / 10000;
        }
        if ( units.isFill(min) ) min= min / 100 ;  // kludge because DatumRanges cannot contain -1e31
        if ( max<min ) max= Double.MAX_VALUE; //vap+cdaweb:ds=I2_AV_AME&id=ampl&timerange=1978-01-23+7:28:21+to+7:28:22
        if ( UnitsUtil.isTimeLocation(units) ) {
            DatumRange vrange= new DatumRange( 3.15569952E13, 2.840126112E14, Units.cdfEpoch ); // approx 1000AD to 9000AD
            if ( vrange.min().doubleValue(units)>min ) min= vrange.min().doubleValue(units);
            if ( vrange.max().doubleValue(units)<max ) max= vrange.max().doubleValue(units);
            if ( vrange.min().doubleValue(units)>max ) max= vrange.max().doubleValue(units); //vap+cdaweb:ds=IM_HK_FSW&id=BF_DramMbeCnt&timerange=2005-12-18
        }
        range = new DatumRange(min, max, units);
        return range;
    }

    /**
     * return null or the scale type if found.
     * @param attrs
     * @return
     */
    private static String getScaleType(Map attrs) {
        String type = null;
        if (attrs.containsKey("SCALETYP") && attrs.get("SCALETYP") instanceof String ) { // CAA STAFF
            type = (String) attrs.get("SCALETYP");
        }
        return type;
    }

    public Map<String, Object> properties(Map<String, Object> meta) {
        return properties( meta, true );
    }

    public Map<String, Object> properties(Map<String, Object> meta, boolean recurse ) {

        Map attrs;
        if ( meta==null ) {
            new Exception("null attributes").printStackTrace();
            attrs= Collections.emptyMap();
        } else {
            attrs= new HashMap(meta);
        }

        Map<String,Object> user= new LinkedHashMap<String,Object>();

        Map<String, Object> properties = new LinkedHashMap<String, Object>();

        if (attrs.containsKey("LABLAXIS")) {
            properties.put(QDataSet.LABEL, attrs.get("LABLAXIS"));
        }

        if (attrs.containsKey("CATDESC")) {
            properties.put(QDataSet.TITLE, attrs.get("CATDESC"));
        }

        if (attrs.containsKey("DISPLAY_TYPE")) {
            String type = (String) attrs.get("DISPLAY_TYPE");
            properties.put(QDataSet.RENDER_TYPE, type);
        }

        if (attrs.containsKey("VIRTUAL") ) {
            String v= (String) attrs.get("VIRTUAL");
            String function= (String)attrs.get("FUNCTION");
            user.put( IstpMetadataModel.USER_PROP_VIRTUAL_FUNCTION, function );
            for ( int i=0; i<4; i++ ) {
                if ( attrs.get("COMPONENT_"+i)!=null ) {
                    user.put( IstpMetadataModel.USER_PROP_VIRTUAL_COMPONENT_ + i, attrs.get("COMPONENT_"+i) );
                } else {
                    break;
                }
            }
            
        }

        Units units = Units.dimensionless;
        if (attrs.containsKey("UNITS")) {
            String sunits = (String) attrs.get("UNITS");

            try {
                units = SemanticOps.lookupUnits(DataSourceUtil.unquote(sunits));
            } catch (IllegalArgumentException e) {
                units = Units.dimensionless;
            }

            // we need to distinguish between ms and epoch times.
            boolean isMillis=false;
            Object ovalidMax= attrs.get("VALIDMAX");
            Object ovalidMin= attrs.get("VALIDMIN");
            if ( ovalidMax!=null && ovalidMin!=null 
                    && ovalidMax instanceof Number && ovalidMin instanceof Number
                    && units==Units.milliseconds ) {
                double validMax= ((Number)ovalidMax).doubleValue();
                double validMin= ((Number)ovalidMin).doubleValue();
                isMillis= validMin<validMax && validMin < 1e8 && validMax < 1e12 ; // java cdf would get zeros for these  rbsp-b_HFR-waveform_emfisis-L1_20110405154808_v1.1.1.cdf?HFRsamples
            }

            Object ofv= attrs.get( "FILLVAL" );
            double dv= doubleValue( ofv, units, Double.NaN, IstpMetadataModel.VALUE_MIN );
            if ( !Double.isNaN(dv) ) {
                properties.put(QDataSet.FILL_VALUE, dv );
            }

            boolean isEpoch = ( units == Units.milliseconds && !isMillis ) || "Epoch".equals(attrs.get(QDataSet.NAME)) || "Epoch".equalsIgnoreCase(DataSourceUtil.unquote((String) attrs.get("LABLAXIS")));
            if (isEpoch) {
                units = Units.cdfEpoch;
                properties.put(QDataSet.LABEL, "");
            } else {
                String label = (String) attrs.get("LABLAXIS");
                if (label == null) {
                    label = sunits;
                } else {
                    if (!sunits.equals("")) {
                        label += " (" + sunits + ")";
                    }
                }
                properties.put(QDataSet.LABEL, label);
            }
            properties.put(QDataSet.UNITS, units);
        }

        try {

            DatumRange range = getRange(attrs, units);
            if (!attrs.containsKey("COMPONENT_0")) { // Themis kludge
                if ( range!=null ) properties.put(QDataSet.TYPICAL_MIN, range.min().doubleValue(units));
                if ( range!=null ) properties.put(QDataSet.TYPICAL_MAX, range.max().doubleValue(units));

                range = getValidRange(attrs, units);
                properties.put(QDataSet.VALID_MIN, range.min().doubleValue(units));
                properties.put(QDataSet.VALID_MAX, range.max().doubleValue(units));

                Object ofv= attrs.get( "FILLVAL" );
                if ( ofv!=null && ofv instanceof Number ) {
                    Number fillVal= (Number) ofv;
                    double fillVald= fillVal.doubleValue();
                    if( fillVald>=range.min().doubleValue(units) && fillVald<=range.max().doubleValue(units) ) {
                        properties.put( QDataSet.FILL_VALUE, fillVal );
                    }
                } else if ( ofv!=null && ofv.getClass().isArray() ) {
                    // try to reduce it to one number.
                    Number fillVal= (Number) Array.get(ofv,0);
                    int n= Array.getLength(ofv);
                    for ( int i=1; i<n; i++ ) {
                        if ( ! Array.get(ofv,i).equals(fillVal) ) {
                            fillVal= Double.NaN;
                        }
                    }
                    double fillVald= fillVal.doubleValue();
                    if( fillVald>=range.min().doubleValue(units) && fillVald<=range.max().doubleValue(units) ) {
                        properties.put( QDataSet.FILL_VALUE, fillVal );
                    }
                    
                }

            }

            properties.put(QDataSet.SCALE_TYPE, getScaleType(attrs));
        } catch (IllegalArgumentException ex) {
            ex.printStackTrace();

        }

        if ( recurse ) {
            for (int i = 0; i < QDataSet.MAX_RANK; i++) {
                String key = "DEPEND_" + i;
                Object o= attrs.get(key);
                if ( o==null ) continue;
                if ( !( o instanceof Map ) ) {
                    new RuntimeException("String where Map was expected").printStackTrace();
                    //TODO: track this down: vap:http://cdaweb.gsfc.nasa.gov/istp_public/data/fast/ies/1998/fa_k0_ies_19980102_v02.cdf?ion_0
                    continue;
                }
                Map<String, Object> props = (Map<String, Object>) o;
                for ( int j=0; j<QDataSet.MAX_RANK; j++ ) {
                    if ( props.containsKey("DEPEND_"+j ) ) {
                        props.remove("DEPEND_"+j); // remove DEPEND property from DEPEND property.
                    }
                }
                properties.put(key, properties(props,false));
            }
        }

        if ( !user.isEmpty() ) {
            properties.put( QDataSet.USER_PROPERTIES, user );
        }
        
        return properties;

    }

    @Override
    public String getLabel() {
        return "ISTP-CDF";
    }
}
