/*
 * MetadataPanel.java
 *
 * Created on July 27, 2007, 11:54 AM
 */
package org.virbo.autoplot;

import java.awt.EventQueue;
import org.das2.datum.Datum;
import org.das2.datum.DatumUtil;
import org.das2.datum.Units;
import org.das2.util.CombinedTreeModel;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.text.DecimalFormat;
import java.util.LinkedHashMap;
import java.util.Map;
import javax.swing.SwingUtilities;
import javax.swing.tree.TreeModel;
import org.autoplot.help.AutoplotHelpSystem;
import org.das2.system.RequestProcessor;
import org.das2.util.monitor.NullProgressMonitor;
import org.das2.util.monitor.ProgressMonitor;
import org.virbo.autoplot.dom.Application;
import org.virbo.autoplot.dom.ApplicationController;
import org.virbo.autoplot.dom.DataSourceController;
import org.virbo.autoplot.dom.DataSourceFilter;
import org.virbo.autoplot.dom.PlotElement;
import org.virbo.autoplot.dom.PlotElementController;
import org.virbo.dataset.DataSetUtil;
import org.virbo.dataset.QDataSet;
import org.virbo.dataset.RankZeroDataSet;
import org.virbo.datasource.DataSource;
import org.virbo.datasource.MetadataModel;
import org.virbo.dsutil.AutoHistogram;
import org.virbo.dsutil.PropertiesTreeModel;
import org.virbo.metatree.NameValueTreeModel;

/**
 *
 * @author  jbf
 */
public class MetadataPanel extends javax.swing.JPanel {

    ApplicationModel applicationModel;
    Application dom;
    CombinedTreeModel tree;
    TreeModel dsTree;
    TreeModel componentDataSetTree=null;
    DataSourceFilter bindToDataSourceFilter = null;  //TODO: these should be weak references or such.
    PlotElement bindToPlotElement =null;
    private QDataSet dsTreeDs;
    private QDataSet componentDs;

    /** Creates new form MetadataPanel */
    public MetadataPanel(ApplicationModel applicationModel) {
        this.applicationModel = applicationModel;
        this.dom = applicationModel.getDocumentModel();
        initComponents();

        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                metaDataTree.setModel(null);
            }
        });

        dom.getController().addPropertyChangeListener(ApplicationController.PROP_DATASOURCEFILTER, new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                bindToDataSourceFilter( dom.getController().getDataSourceFilter() );
            }
        });

        dom.getController().addPropertyChangeListener(ApplicationController.PROP_PLOT_ELEMENT, new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent evt) {
                bindToPlotElement( dom.getController().getPlotElement() );
            }
        });

        DataSourceFilter dsf= dom.getController().getDataSourceFilter();
        if ( dsf!=null ) bindToDataSourceFilter(dsf);

        //applicationModel.addPropertyChangeListener(this.appModelListener);
        updateProperties();
        updateStatistics();
        updateComponentDataSet();

        AutoplotHelpSystem.getHelpSystem().registerHelpID(this, "metadataPanel");
    }

    private void bindToDataSourceFilter( DataSourceFilter dsf ) {
        if (bindToDataSourceFilter != null) {
            DataSourceController dsc = bindToDataSourceFilter.getController();
            dsc.removePropertyChangeListener(propertiesListener);
            dsc.removePropertyChangeListener(fillListener);
        }
        dsf.getController().addPropertyChangeListener(DataSourceController.PROP_RAWPROPERTIES, propertiesListener);
        dsf.getController().addPropertyChangeListener(DataSourceController.PROP_FILLDATASET, fillListener);
        bindToDataSourceFilter= dsf; // BUGFIX
        updateProperties();
        updateStatistics();
    }

    private void bindToPlotElement( PlotElement pe ) {
        if (bindToPlotElement != null) {
            PlotElementController pec = bindToPlotElement.getController();
            pec.removePropertyChangeListener(componentListener);
        }
        pe.getController().addPropertyChangeListener(DataSourceController.PROP_DATASET, componentListener );
        bindToPlotElement= pe;
        updateComponentDataSet();

    }

    private void updateProperties() {

        try {
            DataSourceFilter dsf = dom.getController().getDataSourceFilter();
            DataSourceController dsfc= null;
            DataSource dataSource = null;
            if (dsf != null) {
                dsfc= dsf.getController();
                dataSource = dsfc.getDataSource();
            }
            if ( dsfc==null ) {
                String label = "(data source controller is null)";
                tree = new CombinedTreeModel(label);
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        metaDataTree.setModel(tree);
                    }
                });

            } else if (dataSource != null) {
                tree = new CombinedTreeModel("" + dataSource.getURI());
                Map<String, Object> meta = dsfc.getRawProperties(); //TODO: consider that dataSource.getMetadata() might be better.  This might result in extra read for some sources.
                MetadataModel model = dataSource.getMetadataModel();
                String root = "Metadata";
                if (model != null) {
                    if (!model.getLabel().equals("")) {
                        root = root + "(" + model.getLabel() + ")";
                    }
                }

                final TreeModel dsrcMeta = NameValueTreeModel.create(root, meta);
                if (dsrcMeta != null) {
                    SwingUtilities.invokeLater(new Runnable() {

                        public void run() {
                            tree.mountTree(dsrcMeta, 10);
                            metaDataTree.setModel(tree);
                        }
                    });
                }

            } else {
                String label = "(no data source)";
                if ( dsfc.getDataSet() != null) {  
                    label = "dataset";
                }
                tree = new CombinedTreeModel(label);
                SwingUtilities.invokeLater(new Runnable() {
                    public void run() {
                        metaDataTree.setModel(tree);
                    }
                });
                
            }
        } catch (Exception e) {
            tree = new CombinedTreeModel("Exception: " + e);
            applicationModel.getExceptionHandler().handle(e);
        }
    }
    transient PropertyChangeListener propertiesListener = new PropertyChangeListener() {

        public void propertyChange(PropertyChangeEvent evt) {
            if (evt.getPropertyName().equals(DataSourceController.PROP_RAWPROPERTIES)) {
                updateProperties();
            }
        }
    };
    /**
     * update when the fill dataset changes.
     */
    transient PropertyChangeListener fillListener = new PropertyChangeListener() {

        public void propertyChange(PropertyChangeEvent evt) {
            if (evt.getPropertyName().equals(DataSourceController.PROP_FILLDATASET)) {
                //System.err.println("fillChanged: "+evt+" "+evt.getPropertyName()+" "+evt.getOldValue()+" "+evt.getNewValue());
                updateStatistics();
            }
        }
    };

    /**
     * update when the fill dataset changes.
     */
    transient PropertyChangeListener componentListener = new PropertyChangeListener() {

        public void propertyChange(PropertyChangeEvent evt) {
            if (evt.getPropertyName().equals(PlotElementController.PROP_DATASET )) {
                updateComponentDataSet();
            }
        }
    };

    private void updateComponentDataSet() {
        Runnable run= new Runnable() {
            public void run() {
                updateComponentDataSetPropertiesView();
            }
        };
        RequestProcessor.invokeLater(run);
    }

    private String format(double d) {
        if (Math.abs(Math.log(d) / Math.log(10)) < 3) {
            DecimalFormat df1 = new DecimalFormat("0.00");
            return df1.format(d);
        } else {
            DecimalFormat df = new DecimalFormat("0.00E0");
            return df.format(d);
        }
    }
    boolean statisticsDirty;

//    private String histStr(QDataSet ds) {
//        QDataSet hist = Ops.histogram(ds, 20);
//        QDataSet dep0 = (QDataSet) hist.property(QDataSet.DEPEND_0);
//        Datum res = DataSetUtil.asDatum((RankZeroDataSet) dep0.property(QDataSet.CADENCE));
//        Units u = (Units) dep0.property(QDataSet.UNITS);
//        if (u == null) {
//            u = Units.dimensionless;
//        }
//
//        String scale;
//        if (metaDataTree.getFont().canDisplay((char) 2581)) {
//            scale = "\u2581\u2582\u2583\u2584\u2585\u2586\u2587\u2588";
//        } else {
//            scale = " .:!#";
//        }
//        int scaleMax = scale.length();
//
//        Integer max = (Integer) hist.property("max");
//
//        StringBuffer s = new StringBuffer();
//
//        s.append("" + Datum.create(dep0.value(0), u, res.doubleValue(u.getOffsetUnits())) + " ");
//        for (int i = 0; i < hist.length(); i++) {
//            int norm = (int) hist.value(i) * scaleMax / max;
//            if (norm == scaleMax) {
//                norm = scaleMax - 1; // make last bin inclusive
//            }            //s.append( (char)(2581+norm) );
//            s.append(scale.charAt(norm));
//        }
//        s.append(" " + Datum.create(dep0.value(dep0.length() - 1), u, res.doubleValue(u.getOffsetUnits())));
//        if ("log".equals(dep0.property(QDataSet.SCALE_TYPE))) {
//            s.append(" log");
//        }
//        return s.toString();
//    }

    private void updateStatistics() {
        statisticsDirty = true;
        Runnable run = new Runnable() {

            public void run() {
                if (statisticsDirty) {
                    updateStatisticsImmediately();
                    updateDataSetPropertiesView();
                }
            }
        };
        RequestProcessor.invokeLater(run);
    }

    private synchronized void updateDataSetPropertiesView() {
        assert EventQueue.isDispatchThread() == false;
        final TreeModel unmount;
        DataSourceFilter dsf = dom.getController().getDataSourceFilter();
        QDataSet ds= dsf.getController().getDataSet();
        if ( ds == null) {
            unmount = dsTree;
            dsTree= NameValueTreeModel.create("Dataset", java.util.Collections.singletonMap("dataset", "(no dataset)") );
            this.dsTreeDs = null;
            //(PropertiesTreeModel( "no dataset", null );
        } else {
            if ( ds != this.dsTreeDs) {
                unmount = dsTree;
                dsTree = new PropertiesTreeModel("Dataset= ", ds, 20);
                this.dsTreeDs = ds;
            } else {
                unmount = null;
            }
        }
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                if (unmount != null) {
                    tree.unmountTree(unmount);
                }
                tree.mountTree(dsTree, 30);
            }
        });
    }

    private synchronized void updateComponentDataSetPropertiesView() {
        assert EventQueue.isDispatchThread() == false;
        final TreeModel unmount;
        PlotElement pe= dom.getController().getPlotElement();
        if ( pe==null ) {
            unmount = componentDataSetTree;
            componentDataSetTree= NameValueTreeModel.create("Processed Dataset", java.util.Collections.singletonMap("dataset", "(no dataset)") );
            this.componentDs = null;
            //(PropertiesTreeModel( "no dataset", null );
        } else {
            QDataSet ds= pe.getController().getDataSet();
            if ( ds == null) {
                unmount = componentDataSetTree;
                componentDataSetTree= NameValueTreeModel.create("Processed Dataset", java.util.Collections.singletonMap("dataset", "(no dataset)") );
                this.componentDs = null;
                //(PropertiesTreeModel( "no dataset", null );
            //} else if ( ds==this.dsTreeDs ) {
            //    unmount = componentDataSetTree;
            //    componentDataSetTree= NameValueTreeModel.create("Processed Dataset", java.util.Collections.singletonMap("dataset", "(no additional processing)") );
            } else {
                if ( ds != this.componentDs) {
                    unmount = componentDataSetTree;
                    componentDataSetTree = new PropertiesTreeModel("Processed Dataset= ", ds, 20);
                    this.componentDs = ds;
                } else {
                    unmount = null;
                }
            }
        }
        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                if (unmount != null) {
                    tree.unmountTree(unmount);
                }
                tree.mountTree(componentDataSetTree, 40);
            }
        });
    }

    @SuppressWarnings("unchecked")
    private synchronized void updateStatisticsImmediately() {
        assert EventQueue.isDispatchThread() == false;

        DataSourceFilter dsf = dom.getController().getDataSourceFilter();
        final LinkedHashMap map = new LinkedHashMap();

        QDataSet ds = dsf.getController().getDataSet();
        if (ds == null) {
            map.put("dataset", "(no dataset)");

        } else {

            //int nelements= DataSetUtil.totalLength( ds );
            RankZeroDataSet moments ;
            long validCount;
            long invalidCount;
            String s;

            QDataSet hist = dsf.getController().getHistogram();
            map.put("Histogram", hist);

            if (hist != null) {
                moments = AutoHistogram.moments(hist);

                validCount = (Long) moments.property("validCount");
                invalidCount= (Long) moments.property("invalidCount");
                
                map.put("# invalid", "" + invalidCount + " of " + String.valueOf(validCount + invalidCount));

                if (validCount > 0) {
                    s = String.valueOf(moments);
                } else {
                    s = "";
                }
                map.put("Mean", s);

                if (validCount > 1 && moments.property("stddev")!=null ) {
                    s = String.valueOf(DatumUtil.asOrderOneUnits(DataSetUtil.asDatum((RankZeroDataSet) moments.property("stddev"))));
                } else {
                    s = "";
                }
                map.put("Std Dev", s);

                QDataSet range= AutoHistogram.simpleRange( hist );
                map.put("min", range.slice(0) );
                map.put("max", range.slice(1) );
            }

            QDataSet dep0 = (QDataSet) ds.property(QDataSet.DEPEND_0);

            RankZeroDataSet cadence;

            if (dep0 == null) {
                cadence = DataSetUtil.asDataSet(1);
            } else {
                cadence = (RankZeroDataSet) dep0.property(QDataSet.CADENCE);
            }

            if (cadence != null) {
                Datum d = DatumUtil.asOrderOneUnits(DataSetUtil.asDatum(cadence));
                Units u = d.getUnits();
                map.put("Cadence", format(d.doubleValue(u)) + " " + u);
            } else {
                map.put("Cadence", "null");
            }

        //s= histStr( ds );
        //map.put( "Histogram", s );

        }

        SwingUtilities.invokeLater(new Runnable() {

            public void run() {
                tree.mountTree(NameValueTreeModel.create("Statistics", map), 20);
            }
        });

        statisticsDirty = false;
    }

    /** This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        jScrollPane1 = new javax.swing.JScrollPane();
        metaDataTree = new javax.swing.JTree();

        jScrollPane1.setViewportView(metaDataTree);

        org.jdesktop.layout.GroupLayout layout = new org.jdesktop.layout.GroupLayout(this);
        this.setLayout(layout);
        layout.setHorizontalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 285, Short.MAX_VALUE)
        );
        layout.setVerticalGroup(
            layout.createParallelGroup(org.jdesktop.layout.GroupLayout.LEADING)
            .add(jScrollPane1, org.jdesktop.layout.GroupLayout.DEFAULT_SIZE, 254, Short.MAX_VALUE)
        );
    }// </editor-fold>//GEN-END:initComponents

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JScrollPane jScrollPane1;
    private javax.swing.JTree metaDataTree;
    // End of variables declaration//GEN-END:variables
}
