/*
 * PlasmaModel.java
 *
 * Created on October 9, 2005, 6:46 AM
 *
 * To change this template, choose Tools | Options and locate the template under
 * the Source Creation and Management node. Right-click the template and choose
 * Open. You can then make changes to the template in the Source Editor.
 */

package org.das2.dataset.model;

import org.das2.datum.Datum;
import org.das2.datum.Units;
import org.das2.math.PoissonDistribution;
import org.das2.util.DasMath;
import java.util.Random;
import org.virbo.dataset.DDataSet;
import org.virbo.dataset.DataSetIterator;
import org.virbo.dataset.DataSetUtil;
import org.virbo.dataset.QDataSet;
import org.virbo.dataset.QubeDataSetIterator;
import org.virbo.dataset.SemanticOps;

/**
 * Model of plasma distribution function for given density, temperature, speed.
 * A java.util.Random object is passed in so that the data may be reproducible
 * (by using a given starting seed).
 *
 * This was resurrected from an ancient version of das2.
 *
 * @author Jeremy
 */
public class PlasmaModel {
    
    public double nc= 1.2; // core density 1/cm^3
    public double wcperp = 8000.*1e5; // core thermal velocity cm/s
    public double wcparl = 8000.*1e5;
    public double mass= 9.11e-28;
    boolean isotropic= true;
    
    double geomFactor= 1000e-40;
    
    public PlasmaModel() {
    }
    
    public void setDensity( Datum density ) {
        this.nc= density.doubleValue(Units.pcm3);
    }
    
    public Datum getDensity() {
        return Units.pcm3.createDatum(nc);
    }
    
    public double f( double energy, Units units ) {
        if ( units!=Units.eV ) throw new IllegalArgumentException("units must be in eV");
        if ( !isotropic ) throw new IllegalArgumentException("distribution is not isotropic, need theta,phi");
        double v= Math.sqrt(2*energy*1.6e-19*1e7/mass);               
        double logfc= DasMath.log10(nc/(Math.pow(Math.PI,(3./2.))*wcparl*wcparl)) - 3*Math.pow(v/wcparl,2) ;        
        return DasMath.exp10(logfc);
    }
    
    public double fcounts( double energy, Units units, Random random ) {
        if ( units!=Units.eV ) throw new IllegalArgumentException("units must be in eV");
        double fcount= 2.*(energy/mass)*(energy/mass)*geomFactor*f(energy,units);
        return fcount;
    }

    public QDataSet fcounts( QDataSet energy, Random random ) {
        DDataSet result= DDataSet.create( DataSetUtil.qubeDims(energy) );
        Units u= SemanticOps.getUnits(energy);
        DataSetIterator it= new QubeDataSetIterator(energy);
        while ( it.hasNext() ) {
            it.next();
            it.putValue( result, fcounts( it.getValue(energy), u, random ) );
        }
        return result;
    }

    public int counts( double energy, Units units, Random random ) {
        if ( units!=Units.eV ) throw new IllegalArgumentException("units must be in eV");
        double fcount= 2.*(energy/mass)*(energy/mass)*geomFactor*f(energy,units);
        return PoissonDistribution.poisson(fcount,random);
    }

    public QDataSet counts( QDataSet energy, Random random ) {
        DDataSet result= DDataSet.create( DataSetUtil.qubeDims(energy) );
        Units u= SemanticOps.getUnits(energy);
        DataSetIterator it= new QubeDataSetIterator(energy);
        while ( it.hasNext() ) {
            it.next();
            it.putValue( result, counts( it.getValue(energy), u, random ) );
        }
        return result;
    }

}
