/*
 * DataSetUtil.java
 *
 * Created on April 1, 2007, 4:28 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.virbo.dataset;

import edu.uiowa.physics.pw.das.datum.Units;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

/**
 *
 * @author jbf
 */
public class DataSetUtil {
    
    /**
     * creates a dataset of integers 0,1,2,...,n-1.
     */
    public static DataSet indexGenDataSet(int n) {
        return new IndexGenDataSet(n);
    }
    
    /**
     * creates a dataset with the given cadence, start and length.
     */
    public static DataSet tagGenDataSet( int n, final double start, final double cadence ) {
        return new IndexGenDataSet(n) {
            public double value(int i) {
                return i*cadence + start;
            }
        };
    }
    
    /**
     * returns true if the dataset is monotonically increasing, and contains no fill.
     * An empty dataset is monotonic.
     */
    public static boolean isMonotonic( DataSet ds ) {
        if ( ds.rank()!=1 ) return false;
        int i=0;
        
        final Units u= (Units) ds.property( DataSet.UNITS );
        
        if ( ds.length()==0 ) return false;
        
        double last= ds.value(i);
        
        if ( u!=null && u.isFill(last) ) return false;
        
        for ( i=1; i<ds.length(); i++ ) {
            double d=  ds.value(i);
            if ( d <= last || ( u!=null && u.isFill(d) ) )  return false;
            last= d;
        }
        return true;
    }
    
    /**
     * perform a binary search for key within ds, constraining the search to between low and high.
     * @param ds a rank 1 monotonic dataset.
     * @param key the value to find.
     * @param low
     * @param high
     * @return a positive index of the found value or -index-1 the insertion point.
     */
    public static int binarySearch( DataSet ds, double key, int low, int high ) {
        while (low <= high) {
            int mid = (low + high) >> 1;
            double midVal = ds.value(mid);
            int cmp;
            if (midVal < key) {
                cmp = -1;   // Neither val is NaN, thisVal is smaller
            } else if (midVal > key) {
                cmp = 1;    // Neither val is NaN, thisVal is larger
            } else {
                long midBits = Double.doubleToLongBits(midVal);
                long keyBits = Double.doubleToLongBits(key);
                cmp = (midBits == keyBits ?  0 : // Values are equal
                    (midBits < keyBits ? -1 : // (-0.0, 0.0) or (!NaN, NaN)
                        1));                     // (0.0, -0.0) or (NaN, !NaN)
            }
            
            if (cmp < 0)
                low = mid + 1;
            else if (cmp > 0)
                high = mid - 1;
            else
                return mid; // key found
        }
        return -(low + 1);  // key not found.
    }
    
    /**
     * return the index of the closest value in ds to d, using guess as a starting point.  This
     * implementation ignores guess, but wise clients will supply it as a parameter.
     * @param ds a rank 1, monotonic dataset.
     * @param d the value to find.
     * @param guess a guess at a close index, or -1 if no guess should be made.  In this case, a binary search is performed.
     * @return index of the closest.
     */
    public static int closest( DataSet ds, double d, int guess ) {
        int result= binarySearch( ds, d, 0, ds.length()-1 );
        if (result == -1) {
            result = 0; //insertion point is 0
        } else if (result < 0) {
            result= ~result; // usually this is the case
            if ( result >= ds.length()-1 ) {
                result= ds.length()-1;
            } else {
                double x= d;
                double x0= ds.value( result-1 );
                double x1= ds.value( result );
                result= ( ( x-x0 ) / ( x1 - x0 ) < 0.5 ? result-1 : result );
            }
        }
        return result;
    }
    
    /**
     * gets all the properties of the dataset.  This is a shallow
     * copy of properties.
     */
    public static Map<String,Object> getProperties( DataSet ds ) {
        Map result= new HashMap();
        result.put( DataSet.VALID_RANGE, ds.property(DataSet.VALID_RANGE) );
        result.put( DataSet.UNITS, ds.property(DataSet.UNITS) );
        for ( int i=0; i<ds.rank(); i++ ) {
            DataSet dep= (DataSet) ds.property( "DEPEND_"+i );
            if ( dep!=null ) {
                result.put( "DEPEND_"+i, dep );
            }
        }
        DataSet plane0= (DataSet) ds.property( DataSet.PLANE_0 );
        if ( plane0!=null ) result.put( DataSet.PLANE_0, plane0 );
        //TODO: correlated PLANEs
        return result;
    }
    
    /**
     * copy all properties into the dataset by iterating through the map.
     */
    public static void putProperties( Map<String,Object> properties, MutablePropertyDataSet ds ) {
        for ( Iterator i= properties.entrySet().iterator(); i.hasNext(); ) {
            Map.Entry e= (Map.Entry)i.next();
            ds.putProperty( (String)e.getKey(), e.getValue() );
        }
    }
    
    /**
     * cleans up code by doing the cast, and handles default value
     */
    /*public static <T> getProperty( DataSet ds, String propertyName, Class<T> clazz, Object<T> defaultValue ) {
        T p = ds.property( propertyName );
        if ( p==null ) p= defaultValue;
        return p;
        ArrayList o;
    }*/
    
    public static String toString( DataSet ds ) {
        StringBuffer dimStr= new StringBuffer( ""+ds.length() );
        if ( ds.rank()>1 ) dimStr.append(","+ds.length(0)+"*" );
        if ( ds.rank()>2 ) dimStr.append(","+ds.length(0,0)+"*" );
        
        String u= String.valueOf( ds.property(DataSet.UNITS) );
        if ( u.equals("null") || u=="" ) u="dimensionless";
        return "dataSet["+dimStr.toString()+"] ("+u+")";
    }
    
    /**
     * calculate cadence by averaging the smallest set of consistent inter-point 
     * distance.
     */
    public static double guessCadence( DataSet xds ) {
        
        double cadence = Double.MAX_VALUE;
        
        // calculate average cadence for consistent points.  Preload to avoid extra branch.
        double cadenceS= Double.MAX_VALUE;
        int cadenceN= 1;
        
        double x0 = 0;

        for ( int i=0; i < xds.length(); i++) {
            double cadenceAvg;
            cadenceAvg= cadenceS/cadenceN;
            cadence = xds.value(i) - x0;
            if ( cadence < 0.5 * cadenceAvg) {
                cadenceS= cadence;
                cadenceN= 1;
            } else if ( cadence < 1.5 * cadenceAvg ) {
                cadenceS+= cadence;
                cadenceN+= 1;
            }
            x0 = xds.value(i);
        }
        return  cadenceS/cadenceN;
    }
}

