/*
 * DataSetURL.java
 *
 * Created on March 31, 2007, 7:54 AM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.virbo.datasource;

import edu.uiowa.physics.pw.das.util.DasProgressMonitor;
import edu.uiowa.physics.pw.das.util.fileSystem.FileObject;
import edu.uiowa.physics.pw.das.util.fileSystem.FileSystem;
import ftp.FtpBean;
import ftp.FtpException;
import ftp.FtpObserver;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.virbo.aggragator.AggregatingDataSourceFactory;
import org.virbo.ascii.AsciiTableDataSource;
import org.virbo.das2Stream.Das2StreamDataSource;
import org.virbo.dods.DodsDataSource;
import org.virbo.excel.ExcelSpreadsheetDataSource;
import org.virbo.netCDF.NetCDFDataSource;

/**
 *
 * Works with DataSourceRegistry to translate a URL into a DataSource.  Also,
 * will provide completions.
 *
 * @author jbf
 *
 */

public class DataSetURL {

    private static int MAX_POSITIONAL_ARGS=10;
    
    static {
        DataSourceRegistry registry= DataSourceRegistry.getInstance();
        registry.register( AsciiTableDataSource.getFactory(), ".dat" );
        registry.register( ExcelSpreadsheetDataSource.getFactory(), ".xls" );
        registry.register( Das2StreamDataSource.getFactory(), ".d2s", "application/x-das2stream" );
        registry.register( Das2StreamDataSource.getFactory(), ".das2Stream", "application/x-das2stream" );
        registry.register( NetCDFDataSource.getFactory(), ".nc" );
        registry.register( DodsDataSource.getFactory(), ".dds" );
        registry.register( DodsDataSource.getFactory(), ".dods" );
        registry.register( DodsDataSource.getFactory(), ".html" );
        registry.register( "org.virbo.spase.SpaseRecordDataSourceFactory", ".xml" );
        
    }
    
    
    
    public static class URLSplit {
        public String path;
        public String file;
        public String ext;
        public String params;
        public String toString() {
            return path + "\n"+ file + "\n"+ ext + "\n"+ params;
        }
    }
    
    public static String maybeAddFile( String surl ) {
        if ( surl.length()==0 ) return "file:/";
        if ( !( surl.startsWith("file:/") || surl.startsWith("ftp://") || surl.startsWith("http://") || surl.startsWith("https://") ) ) {
            surl= "file://"+ ( ( surl.charAt(0)=='/' ) ? surl : ( '/' + surl ) ); // Windows c:
        }
        return surl;
    }
    
    /**
     * split the url string (http://www.example.com/data/myfile.nc?myVariable) into:
     *   path, the directory with http://www.example.com/data/
     *   file, the file, http://www.example.com/data/myfile.nc
     *   ext, the extenion, .nc
     *   params, myVariable
     */
    public static URLSplit parse( String surl ) {
        URL url;
        try {
            surl= maybeAddFile( surl );
            url = new URL(surl);
            
            String file=  url.getPath();
            int i= file.lastIndexOf(".");
            String ext= i==-1 ? "" : file.substring(i);
            
            String params=null;
            
            int fileEnd;
            // check for just one ?
            i= surl.indexOf( "?" );
            if ( i != -1 ) {
                fileEnd= i;
                params= surl.substring(i+1);
                i= surl.indexOf("?",i+1);
                if ( i!=-1 ) {
                    throw new IllegalArgumentException("too many ??'s!");
                }
            } else {
                fileEnd= surl.length();
            }
            
            i= surl.lastIndexOf("/");
            String surlDir= surl.substring(0,i);
            
            int i2= surl.indexOf("://");
            
            URLSplit result= new URLSplit();
            result.path= surlDir+"/";
            result.file= surl.substring(0,fileEnd);
            result.ext= ext;
            result.params=params;
            
            return result;
            
        } catch (MalformedURLException ex) {
            return null;
            
        }
        
    }
    
    /**
     * get the data source for the URL.
     * @throws IllegalArgumentException if the url extension is not supported.
     */
    public static DataSource getDataSource( URL url ) throws Exception {
        DataSourceFactory factory = getDataSourceFactory(url);
        DataSource result= factory.getDataSource( url );
        
        return result;
        
    }

    /**
     * get the datasource factory for the URL.
     */
    public static DataSourceFactory getDataSourceFactory(final URL url) throws IOException, IllegalArgumentException {
        
        if ( url.toString().indexOf("%Y") != -1 ) {
            return new AggregatingDataSourceFactory();
        }
        
        String file= url.getPath();
        int i= file.lastIndexOf(".");
        String ext= i==-1 ? "" : file.substring(i);
        
        // check for just one ?
        String surl= url.toString();
        i= surl.indexOf( "?" );
        if ( i != -1 ) {
            i= surl.indexOf("?",i+1);
            if ( i!=-1 ) {
                throw new IllegalArgumentException("too many ??'s!");
            }
        }
        
        DataSourceFactory factory= null;
        if ( url.getProtocol().equals("http") ) { // get the mime type
            URLConnection c= url.openConnection();
            String mime= c.getContentType();
            factory= DataSourceRegistry.getInstance().getSourceByMime( mime );
        }
        
        if ( factory==null ) factory= DataSourceRegistry.getInstance().getSource( ext );
        if ( factory==null ) {
            throw new IllegalArgumentException( "Unsupported extension: "+ext );
        }
        return factory;
    }
    
    
    
    /**
     *
     * split the parameters into name,value pairs.
     *
     * items without equals (=) are inserted both as "arg_N"=name
     *
     */
    public static Map parseParams( String params ) {
        Map result= new HashMap();
        if ( params==null ) return result;
        
        String[] ss= params.split( "&" );
        
        int argc=0;
        
        for ( int i=0;i<ss.length; i++ ) {
            int j= ss[i].indexOf("=");
            String name, value;
            if ( j==-1 ) {
                name= ss[i];
                value="";
                result.put( "arg_"+(argc++), name );
            } else {
                name= ss[i].substring(0,j);
                value= ss[i].substring(j+1);
                result.put(name,value);
            }
        }
        return result;
    }

    public static String formatParams(Map parms) {
        StringBuffer result= new StringBuffer("");
        boolean notdone=true;
        for ( int i=0; notdone && i<MAX_POSITIONAL_ARGS; i++ ) {
            if ( parms.get("arg_"+i) !=null ) {
                result.append( "&" + parms.get("arg_"+i) );
            } else {
                notdone= false;
            }
        }
        for ( Iterator i=parms.keySet().iterator(); i.hasNext(); ) {
            String key= (String) i.next();
            if ( key.startsWith("arg_") ) continue;
            String value= (String)parms.get(key);
            if ( value!=null ) result.append("&"+key+"="+value);
        }
        return result.substring(1);
    }
        
    private static void transferFile( URL url, File file, final DasProgressMonitor mon ) throws IOException {
        if ( url.getProtocol().equals("ftp") ) {
            try {
                FtpBean bean= new FtpBean();
                bean.ftpConnect( url.getHost(), "ftp" );
                bean.setDirectory( url.getPath() );
                mon.setTaskSize( -1 );
                FtpObserver observer= new FtpObserver() {
                    int totalBytes=0;
                    public void byteRead(int bytes) {
                        totalBytes+= bytes;
                        mon.setTaskProgress( totalBytes );
                    }
                    public void byteWrite(int bytes) {
                        totalBytes+= bytes;
                        mon.setTaskProgress( totalBytes );
                    }
                };
                bean.getBinaryFile( url.getFile(), file.toString(), observer );
                mon.finished();
                
            } catch ( FtpException ex ) {
                throw new RuntimeException( ex );
                
            }
        } else if ( url.getProtocol().equals("http") ) {
            HttpURLConnection connect= (HttpURLConnection) url.openConnection();
            mon.setTaskSize( connect.getContentLength() );
            mon.started();
            InputStream in= connect.getInputStream();
            
            FileOutputStream fo= new FileOutputStream( file );
            
            byte[] buf= new byte[4096];
            int bytesRead=in.read(buf);
            int totalBytesRead=0;
            while ( bytesRead!=-1 ) {
                totalBytesRead+= bytesRead;
                fo.write( buf,0,bytesRead);
                mon.setTaskProgress(totalBytesRead);
                bytesRead=in.read(buf);
            }
            mon.finished();
            fo.close();
            in.close();
        }
    }
    
    /**
     * return a file reference for the url.  This is initially to fix the problem
     * for Windows where new URL( "file://c:/myfile.dat" ).getPath() -> "/myfile.dat".
     * This may eventually be how remote files are downloaded as well, and
     * may block until the file is downloaded.
     * Linux: file:/home/jbf/fun/realEstate/to1960.latlon.xls?column=C[1:]&depend0=H[1:]
     *
     */
    public static File getFile( URL url, DasProgressMonitor mon ) throws IOException {
        URLSplit split= parse( url.toString() );
        url= new URL( split.file );
        
        String proto= url.getProtocol();
        if ( proto.equals("file") ) {
            String surl= url.toString();
            int idx1=surl.indexOf("?");
            if ( idx1 == -1 ) idx1= surl.length();
            surl= surl.substring(0,idx1);
            
            String sfile;
            int idx0=surl.indexOf("file:///");
            if( idx0==-1 ) {
                idx0= surl.indexOf("file:/");
                sfile= surl.substring( idx0+5 );
            } else {
                sfile= surl.substring( idx0+7 );
            }
            
            return new File( sfile );
            
        } else {
            FileSystem fs= FileSystem.create( new URL( split.path ) ) ;
            FileObject fo= fs.getFileObject( split.file.substring( split.path.length() ) );
            
            File tfile= fo.getFile(mon);
            return tfile;
        }
    }
    
    /**
     * canonical method for getting the URL.  If no protocol is specified, then file:// is
     * used.
     */
    public static URL getURL( String surl ) throws MalformedURLException {
        surl= maybeAddFile( surl );
        if ( surl.endsWith( "://" ) ) surl+= "/";  // what strange case is this?
        return new URL( surl );
    }
    
    public static List getExamples() {
        List result= new ArrayList();
        result.add( "L:/ct/virbo/sampexTimeL/sampex.dat?fixedColumns=90&rank2");
        result.add( "http://www.sarahandjeremy.net:8080/thredds/dodsC/testAll/poes_n15_20060111.nc.dds?proton_6_dome_16_MeV" );
        result.add( "http://www.sarahandjeremy.net:8080/thredds/dodsC/test/poesaggLittle.nc.dds?proton_6_dome_16_MeV[0:10:400000]" );
        result.add( "http://cdaweb.gsfc.nasa.gov/cgi-bin/opendap/nph-dods/istp_public/data/genesis/3dl2_gim/2003/genesis_3dl2_gim_20030501_v01.cdf.dds?Proton_Density" );
        result.add( "file://C:/iowaCitySales2004-2006.latlong.xls?column=M[1:]" );
        result.add( "file://c:/Documents and Settings/jbf/My Documents/xx.d2s" );
        result.add( "L:/fun/realEstate/to1960.latlon.xls?column=C[1:]&depend0=H[1:]" );
        result.add( "L:/fun/realEstate/to1960.latlon.xls?column=M[1:]&depend0=N[1:]&plane0=C[1:]" );
        result.add( "L:/ct/virbo/autoplot/data/610008002FE00410.20060901.das2Stream" );
        result.add( "P:/poes/poes_n15_20060212.nc?proton-6_dome_16_MeV" );
        result.add( "L:/ct/virbo/autoplot/data/asciiTab.dat" );
        result.add( "L:/ct/virbo/autoplot/data/2490lintest90005.dat" );
        result.add( "http://www.sarahandjeremy.net:8080/thredds/dodsC/test/LanlGPSAgg.nc.dds?FEIO[0:1:1000][0:0]" );
        result.add( "file://P:/cdf/fast/tms/1996/fa_k0_tms_19961021_v01.cdf?L");
        return result;
    }
    
    public static String[] getCompletions( String surl , DasProgressMonitor mon) throws Exception {
        String[] result;
        try {
            URLSplit split= parse( surl );
            if ( surl.contains("?") ) {
                int i= surl.indexOf("?");
                DataSourceFactory factory= getDataSourceFactory( Util.newURL( Util.FS_URL, surl ) );
                mon.setProgressMessage("getting completions for "+factory);
                mon.started();
                result = factory.getCompletions(surl.substring(0,i+1));
                
            } else {
                try {
                    mon.setProgressMessage("listing directory");
                    mon.started();
                    int i= surl.lastIndexOf("/");
                    String surlDir;
                    if ( i<=0 || surl.charAt(i-1)=='/' ) {
                        surlDir=surl;
                    } else {
                        surlDir= surl.substring(0,i);
                    }
                    URL url= getURL(surlDir);
                    String prefix= surl.substring(i+1);
                    FileSystem fs= FileSystem.create(url);
                    String[] s= fs.listDirectory("/");
                    List<String> result1= new ArrayList<String>(s.length);
                    for ( int j=0; j<s.length; j++ ) {
                        if ( s[j].startsWith(prefix) ) {
                            result1.add(surlDir+"/"+s[j]);
                        }
                    }
                    result= result1.toArray( new String[result1.size()] );
                } catch ( MalformedURLException ex ) {
                    result= new String[] { "Malformed URL" };
                } catch ( FileSystem.FileSystemOfflineException ex) {
                    result= new String[] {"FileSystem offline" };
                }
            }
        } catch ( Exception e ) {
            e.printStackTrace();
            result= new String[] { e.getMessage() };
        } finally {
            mon.finished();
        }
        return result;
    }

}

