;==============================================================================
;
;  NAME:
;     
;     ddeis_shells
;
;  PURPOSE:
;     
;     This function sorts Hydra DDEIS data according to increasing energy.
;
;  ARGUMENTS:
;     
;     Input
;     -----
;
;     z (STRUCTURE):
;
;        an element of a structure array output by
;        'ddeis_block_process'
;
;     gz (STRUCTURE):
;
;        an element of a structure array output by 'ddeis_fp_to_gz'
;     
;     Ouput
;     -----
;
;     z_en (STRUCTURE):
;
;        contains the energy sorted data; the fields are described
;        below:
;
;           time: dblarr(ndata, nen, ndet)
;
;              time in seconds
;
;           theta: fltarr(ndata, nen, ndet)
;
;              particle polar angle in fixed payload (FP) coordinates
;              (radians)
;
;           phi: fltarr(ndata, nen, ndet)
;
;              particle azimuthal angle in fixed payload (FP)
;              coordinates (radians)
;
;           thetab: fltarr(ndata, nen, ndet)
;
;              magnetic field polar angle in FP coordinates (radians)
;
;           phib: fltarr(ndata, nen, ndet)
;
;              magnetic field azimuthal angle in FP coordinates
;              (radians)
;
;           alpha: fltarr(ndata, nen, ndet)
;
;              particle pitch angle in radians
;
;           z: dblarr(ndata, nen, ndet)
;
;              distribution function or count data (CGS units)
;
;           sig: dblarr(ndata, nen, ndet)
;
;              distribution function or count sigmas (CGS units)
;
;           dm: bytarr(ndata, nen, ndet)
;
;              a 'data mask' which eliminates invalid counts (e.g.,
;              glint); 0 -> invalid, 1 -> valid 
; 
;           idx: intarr(ndata, nen)
;
;              time ordered measurement indices
;
;           ndata: intarr(nen, ndet)
;
;              number of data points per detector per energy level
;              
;           energy: fltarr(nen)
;
;              unique energy levels (eV) 
;    
;  USAGE:
;
;     IDL> ddeis_shells, z, gz, z_en, gz_en
;  
;  HISTORY:
;
;     VERSION 1.00 -- 1/21/98 -- Written, John C. Dorelli
;
;=============================================================================

PRO ddeis_shells, z, gz, z_en, gz_en

; for each detector, sort data according to increasing energy

ndet = 12

z2 = z
gz2 = gz

FOR i = 1, ndet DO BEGIN
    energy_old = z.energy(*, i-1)
    idx_energy = sort(energy_old)
    energy_new = energy_old(idx_energy)
    idx_uniq = uniq(energy_new)
    z2.energy(*, i-1) = z.energy(idx_energy, i-1)
    z2.time(*, i-1) = z.time(idx_energy, i-1)
    z2.theta(*, i-1) = z.time(idx_energy, i-1)
    z2.phi(*, i-1) = z.phi(idx_energy, i-1)
    z2.thetab(*, i-1) = z.thetab(idx_energy, i-1)
    z2.phib(*, i-1) = z.phib(idx_energy, i-1)
    z2.alpha(*, i-1) = z.alpha(idx_energy, i-1)
    z2.z(*, i-1) = z.z(idx_energy, i-1)
    z2.sig(*, i-1) = z.sig(idx_energy, i-1)
    z2.dm(*, i-1) = z.dm(idx_energy, i-1)
    z2.idx = z.idx(idx_energy)

    gz2.alpha(*, i-1) = gz.alpha(idx_energy, i-1)
    gz2.beta(*, i-1) = gz.beta(idx_energy, i-1)
ENDFOR

; compress energy ordered arrays
        
nen = n_elements(idx_uniq)
ndata = 12
z_en = {$
         time: dblarr(ndata, nen, ndet), $
         theta: fltarr(ndata, nen, ndet), $
         phi: fltarr(ndata, nen, ndet), $
         thetab: fltarr(ndata, nen, ndet), $
         phib: fltarr(ndata, nen, ndet), $
         alpha: fltarr(ndata, nen, ndet), $
         z: dblarr(ndata, nen, ndet), $
         sig: dblarr(ndata, nen, ndet), $
         dm: bytarr(ndata, nen, ndet), $
         idx: intarr(ndata, nen), $
         ndata: intarr(nen, ndet), $
         energy: fltarr(nen)}

gz_en = {$
          alpha: fltarr(ndata, nen, ndet), $
          beta: fltarr(ndata, nen, ndet), $
          thetab_avg: gz.thetab_avg, $
          phib_avg: gz.phib_avg}
       
FOR i = 1, ndet DO BEGIN
    FOR j = 1, nen DO BEGIN

        idx = idx_uniq
        IF j EQ 1 THEN BEGIN
            ndata = idx(j-1)+1
            idx1 = 0
            idx2 = idx(j-1)
            data_idx1 = idx1
            data_idx2 = idx2
        ENDIF ELSE BEGIN
            ndata = idx(j-1)-idx(j-2)
            idx1 = idx(j-2)
            idx2 = idx(j-1)-1
            data_idx1 = 0
            data_idx2 = ndata-1
        ENDELSE

        z_en.ndata(j-1, i-1) = ndata
        z_en.time(data_idx1:data_idx2, j-1, i-1) = z2.time(idx1:idx2, i-1)
        z_en.theta(data_idx1:data_idx2, j-1, i-1) = z2.theta(idx1:idx2, i-1)
        z_en.phi(data_idx1:data_idx2, j-1, i-1) = z2.phi(idx1:idx2, i-1)
        z_en.thetab(data_idx1:data_idx2, j-1, i-1) = z2.thetab(idx1:idx2, i-1)
        z_en.alpha(data_idx1:data_idx2, j-1, i-1) = z2.alpha(idx1:idx2, i-1)
        z_en.z(data_idx1:data_idx2, j-1, i-1) = z2.z(idx1:idx2, i-1)
        z_en.sig(data_idx1:data_idx2, j-1, i-1) = z2.sig(idx1:idx2, i-1)
        z_en.dm(data_idx1:data_idx2, j-1, i-1) = z2.dm(idx1:idx2, i-1)
        z_en.idx(data_idx1:data_idx2, j-1) = z2.idx(idx1:idx2)
        z_en.energy(j-1) = z2.energy(idx2)
       
        gz_en.alpha(data_idx1:data_idx2, j-1, i-1) = gz2.alpha(idx1:idx2, i-1)
        gz_en.beta(data_idx1:data_idx2, j-1, i-1) = gz2.beta(idx1:idx2, i-1)

    ENDFOR
ENDFOR

END

