;******************************************************************************
;* PROCEDURE:   cluster_createOrbitInfo 
;* 
;* DESCRIPTION: for all CLUSTER-ephemeris-files in the directory CLUSTER_EPH, 
;*	        orbit-infos are created and written to the file  
;*              $CLUSTER_EPH/orbits.info 
;*
;* INPUTS:      none
;* 
;* OUTPUT:	none
;*
;* CALLING SEQUENCE: cluster_createOrbitInfo
;* 
;* MODIFICATION HISTORY: December 1997, written by R. Friedel 
;******************************************************************************
PRO cluster_createOrbitInfo

COMMON cluster_ephem, cluster_ephem

; search for the CLUSTER-data-files. Use this code to get a list of all the
; ephemeris files available for CLUSTER. Replace with correct file template
; for your files in findfile below.

message, 'Compiling list of all CLUSTER ephemeris files:',  /cont
cluster_path=papco_getenv('CLUSTER_EPH_SETS')
allCLUSTERFiles=findfile(cluster_path+'*.idl', count = c)
IF c EQ 0 THEN BEGIN
    print, '  found no CLUSTER ephemeris files'
    return
ENDIF ELSE print, '  ...found ', varprt(c), ' ephemeris-files'

; Depending on your CLUSTER ephemeris files, loop through them to find orbit
; start and end times. One way of doing this is to construct an array of
; radial distance. Max. of radius are then apogee times, Min perigee times.

; CLUSTER ephemeris idl savesets are per month, and contain data for
; the orbit number: SC_ORBIT_NUM. Make a long array of this variable
; for the whole mission, together with the time (here it's cdf EPOCH)

FOR i = 0,  c-1 DO BEGIN
    restore, allCLUSTERFiles(i), /verbose
    IF i EQ 0 THEN BEGIN
        epoch = input_data.EPOCH
        orbit_num = input_data.SC_ORBIT_NUM 
    ENDIF ELSE BEGIN
        epoch = [epoch, input_data.EPOCH]
        orbit_num = [orbit_num, input_data.SC_ORBIT_NUM]
    ENDELSE 
ENDFOR 

;get min, max orbit numbers, then find start times for all orbits
min_orbit = fix(min(orbit_num))
max_orbit = fix(max(orbit_num))
orbits = indgen(max_orbit-min_orbit+1)+min_orbit + 0.5 ;(apogee to apogee)
;interpolate times to start of orbits
orbit_start_epoch = interpol(epoch, orbit_num, orbits)
ndat = n_elements(orbit_start_epoch)

; create a CLUSTER_ORBIT_INFO-structure and create an array of those

tmp=papco_getOrbitInfoStruct()	  
orbitInfo=replicate(tmp, ndat-1)

FOR i=0l,ndat-2 do BEGIN 
    
    ;convert times from cdf epoch
    cdf_epoch, orbit_start_epoch(i), $
      year, mo, day_om, hr, min, sec, milli, /BREAKDOWN_EPOCH
    day_oy = UTC2DOY({YEAR:year, MONTH:mo, DAY:day_om, $
                      HOUR:hr, MINUTE:min, SECOND:sec, MILLISECOND:milli})

    ;xut1   = is the no of sec from the start day of the start of orbit
    ;xut2   = is the no of sec from the start day of the end of orbit
    xut1 = hr*3600 + min*60 + sec
    period = (orbit_start_epoch(i+1) - orbit_start_epoch(i)) / 1000.0
    xut2 = xut1+period

    print, orbits(i), day_oy, year, mo, day_om,  period/3600.0
          ;
          ; format your orbit start /end times such that
          ; i      = orbit number
          ; day_oy = is the day of year of start of orbit
          ; year   = is the year of start of orbit
          ; xut1   = is the no of sec from the start day of the start of orbit
          ; xut2   = is the no of sec from the start day of the end of orbit
          ; 
          ; Its up to you to define orbits from apogee to apogee or 
          ; from perigee to perigee
          ;
          ; save the data in the orbitInfo-structure
          ;
    orbitInfo(i).number=orbits(i)
    orbitInfo(i).day=day_oy
    orbitInfo(i).year=year
    orbitInfo(i).xut1=xut1
    orbitInfo(i).xut2=xut2
    orbitInfo(i).xdata=0 ;dummy, don't use, here for old compatibility
    orbitInfo(i).ydata=0 ;dummy, don't use, here for old compatibility
ENDFOR 


stop
; write the orbits.info-file
result=papco_writeOrbitInfo('CLUSTER', orbitInfo)
IF result EQ 'OK' THEN $
      message, 'OrbitInfo successfully created.', /cont $
ELSE print, result

END; (cluster_createOrbitInfo)
