;******************************************************************************
;******************************************************************************
;* FILE:
;*    papco_XX.X/papco/papco.pro
;*
;* DESCRIPTION:
;*    This file contains the main widget of PAPCO, the Panelplot-Composer.
;*    contained are :      - the widget-construction-routines
;*                          - the event-handlers.
;*
;*    The Panelplot-composer is a window in which you can compose plots
;*    from various modules. You can add new plots, resize the plots, edit or
;*    delete the plots, zoom into the plots, save list of plots, load them,
;*    print them to various devices.
;*
;* FILES BELONGING TO THE MAIN-WINDOW:
;*
;*      papco.pro
;*         - main-window widget-construction
;*         - main-window event-handlers
;*
;*      papco_plotedit.pro
;*         - adding and editing plotinfos
;*         - deleting plotinfos
;*         - routine for obtaining verbose description of plotinfos
;*
;*      papco_files.pro
;*         - procedures for saving and loading plotinfos
;*         - procedures for writing and reading default-settings
;*         - batch-mode
;*
;*      papco_settime.pro
;*         - procedures for selecting an orbit and time-range
;*         - procedure for displaying time-range when using time-slider
;*
;*      papco_draw.pro
;*         - routines for outputting plots.
;*
;*      papco_draw_window.pro
;*         - routines for handling the graphics output window
;*
;*      papco_cursor.pro
;*         - routines for handling the mouse-cursor
;*         - routines for zooming into plots
;*         - routines for doing slices of plots
;*         - routines for doing level zero access
;*
;*      papco_edit_type.pro
;*         - changing the plot-type of a plotinfo
;*
;*      papco_opsys.pro
;*         - implements architecture independence for some routines
;*
;* COMMON-BLOCKS:
;*    COMMON PLOT_COMPOSER, widgetdata
;*
;* STRUCTURES:
;*    The structures defined in this file are:
;*
;*        papco_PLOTINFO     -> a structure to hold the description of
;*                              one plot-panel
;*
;*        PLOT_COMPOSER      -> a structure to hold the global variables
;*                              of the papco main-window.
;*
;*        PLOT_DRAWN         -> when a plot is drawn to the screen,
;*                              variables of this type are used
;*                              for storing info about this plot
;*
;*    They are described in the file
;*    papco_XX/papco_doc/papco_variables.text
;*
;*    Those procedures are contained here:
;*       PRO papco_error_handler
;*       FUNCTION papco_getPlotInfoStruct
;*       FUNCTION papco_DescriptionFor, plotInfo, totalHeight
;*       PRO papco_EditPlotType, plotNumber
;*       PRO papco_DeletePlot, plotNumber
;*       PRO papco_PrintMenu_event, event
;*       FUNCTION papco_SetOrbit, anOrbitNr
;*       PRO papco_setTime, newTime, STARTTIME=starttime, ENDTIME=endtime
;*       PRO papco_DrawTimeRange, newStart, newEnd, CLEAR_ONLY=CLEAR_ONLY
;*       PRO papco_main_Event, Event
;*       PRO papco_Refresh, TIME=TIME, LIST=LIST, NEW_TIMERANGE=NEW_TIMERANGE,
;*             ALL=ALL, PRINTER=PRINTER,
;*             SELECT_TIME_METHOD=SELECT_TIME_METHOD
;*       PRO papco_RefreshList
;*       PRO papco_Input, DISABLE=DISABLE, ENABLE=ENABLE
;*       PRO papco, GROUP=Group
;*
;* MODIFICATION HISTORY: (applies to all routines in this file)
;*     august,september 1995, written by A.Keese
;*     november 1995, modified by R.Friedel
;*     march 1996, modified by H.Korth
;*     october 1996, modified by R.Friedel
;*     january 1997, modified by R.Friedel
;*     october 1997, modified by R.Friedel
;*     constant updates by R Friedel, latest January 2003
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      papco_error_handler
;*
;* DESCRIPTION:
;*	This routine is called when an error occurs on calling a
;*      user-defined interface routien. A message is printed to the IDL
;*      transcript window and into an empty panel in the draw window. This is
;*      useful for trapping errors in the draw loop without crashing PAPCO.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_Clear_Drawing
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO papco_error_handler, error_status, proc_call, panelkind, panelVector, $
                         NO_ERR_DRAW=NO_ERR_DRAW

COMMON PLOT_COMPOSER, widgetdata
COMMON get_error, get_err_no, get_err_msg

IF error_status eq -60 THEN BEGIN
    msg2=[' Looks like PAPCO could find module: '+ panelkind,$
          ' But there is a naming convention problem in the ',$
          ' modules papcoadd_'+panelkind+' file.','']
ENDIF ELSE msg2=''
    msg=[' ','% PAPCO: Error trapped calling user interface routine', $
         '         "'+ proc_call + '" or below.','', $
         ' ERROR INDEX: '+strtrim(string(error_status),2)+', ERROR MESSAGE:', $
         ' '+!err_string, '', $
         ' To trace this error fully, you can turn off error trapping', $
         ' in the Defaults widget. PAPCO will then crash at the error.','', $
         ' To continue use @papco_reset in main widget and do not use this', $
         ' module until the problem is fixed.', $
         ' ',msg2]
FOR j=0,n_elements(msg)-1 DO print,msg(j)

help,  /traceback

IF NOT keyword_set(NO_ERR_DRAW) THEN BEGIN
    get_err_no=99
    get_err_msg=msg
    textArray=['Unable to draw panel for: '+panelKind, get_err_msg]
    papco_Draw_ErrorPanel, panelVector, textArray
ENDIF

END

;******************************************************************************
;* FUNCTION:
;*      papco_getPlotInfoStruct
;*
;* DESCRIPTION:
;*      This function is used whenever an empty structure of type
;*      PAPCO_PLOTINFO is needed. To enable later modifications of PAPCO,
;*      structures of this kind are NEVER defined explicitly - they are
;*      always created using papco_getPlotInfoStruct(). This struture
;*      contains all the information needed for a given panel.
;*
;*      TO MODIFY this structure only ADD tags that you might need. You will
;*      need to restart idl after changing this structure.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* OUTPUT:
;*      a default-structure of type PAPCO_PLOTINFO
;*      for its description see the top of this file and the file
;*      papco_variables.text
;*
;* CALLING SEQUENCE:
;*      aPlotInfoStruct=papco_getPlotInfoStruct()
;*
;* MODIFICATION HISTORY:
;*      written august 1995, Andreas Keese
;*      modified May 2003, R. Friedel, added POINTER to struct
;******************************************************************************
FUNCTION papco_getPlotInfoStruct

return, $
    {PAPCO_PLOTINFO_17, $
     manualYScaling:0,  manualZScaling:0, $  ; \
     yscaleLb:0,        zscaleext:0,      $  ;  \
     yscaleMin:1.0,     yscaleMax:2.0,    $  ;   \
     zscaleMin:1.0,     zscaleMax:2.0,    $  ;    \
     yLinLog:1,         zLinLog:1,        $  ;     \
     overplot:0,        plotstyle:0,      $  ;      \
     color:0,  psym:0,  thick:1,          $  ;       \
     linestyle:0,       symsize:1,        $  ;        \
     xut1:0.0,          xut2:0.0,         $  ;         > papco_panel keywords
     panelPosition:0,   panelHeight:1,    $  ;        /
     panelKind:'',      channel:0,        $  ;       /
     dataUnits:0,       fileName:'',      $  ;      /
     descr_str:'',      descr:0,          $  ;     /
     descr_extend:1,    modified:0,       $  ;    /
     ylbl_str:'',       ylbl:0,           $  ;   /
     ylbl_extend:1,     nogap:0,          $  ;  /
     zlbl_str:'',       zlbl:0,           $  ; /
     zlbl_extend:1,     descr_type:0,     $  ;/

     typeVector:papco_lon64arr(4), $  ; user tag (old, for compat.)
     options:dblarr(20),           $  ; user tag (float options)
     ioptions:lonarr(20),          $  ; user tag (long  options)
     soptions:strarr(20),          $  ; user tag (string options)
     usr_ptr1:PTR_NEW(0)           $  ; user tag (pointer to panel struct)
     }
end

;******************************************************************************
;* FUNCTION:
;*      papco_getDrawnPlotInfoStruct
;*
;* DESCRIPTION:
;*      This function is used whenever an empty structure of type
;*      PAPCO_DRAWN_PLOTINFO is needed. To enable later modifications of PAPCO,
;*      structures of this kind are NEVER defined explicitly - they are
;*      always created using papco_getPlotInfoStruct(). This struture
;*      contains all the information needed for a given panel.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* OUTPUT:
;*      a default-structure of type PAPCO_DRAWN_PLOTINFO
;*      for its description see the top of this file and the file
;*      papco_variables.text
;*
;* CALLING SEQUENCE:
;*      aPlotInfoStruct=papco_getDrawnPlotInfoStruct()
;*
;* MODIFICATION HISTORY:
;*      written June 1998, Reiner Friedel
;******************************************************************************
FUNCTION papco_getDrawnPlotInfoStruct

a=papco_getPlotInfoStruct()

tmp={PLOT_DRAWN_1, $
       panelVector:[0,0,0], $
       data_xRange:[0.0,0.0], $ ; this is the range in IDL-data-coordinates
       data_yRange:[0.0,0.0], $
       yscl_min:0.0, $          ; this is the y-range from the yscl-array.
       yscl_max:0.0, $
       ylinlog:0, $
       zscl_min:0.0, $          ; this is the z-range from the zscl-array.
       zscl_max:0.0, $
       zlinlog:0, $
       x_s:!x.s, $
       y_s:!y.s, $
       x_type:!x.type, $
       y_type:!y.type, $
       normal_xRange:[0.0,0.0], $
       normal_yRange:[0.0,0.0], $
       get_data_Call:'', $
       slice_type:0l, $
       orbit:0, $
       changed:0, $
       panelKind:a.panelKind, $
       typeVector:a.typeVector, $
       right_side_label:''}

return, tmp

END

;******************************************************************************
;* PROCEDURE:
;*   pro papco_show_widgetdata
;*
;* DESCRIPTION:
;*   Outputs verbose description of the contents of the main
;*   widgetdata structure that can be overwritten on the papco command
;*   line by using a keyword equal to the tag name.
;*
;* INPUTS: none
;*
;* KEYWORDS: none
;*
;* CALLING SEQUENCE:
;*   papco_show_widgetdata
;*   Called by papco only. May be called from anywhere in PAPCO environment.
;*
;* MODIFICATION HISTORY:
;*   written January 2001, R. Friedel
;******************************************************************************
PRO papco_show_widgetdata

COMMON plot_composer, widgetdata

message, 'Settings of PAPCO that can be overwritten:',/info
print,''
print,'  Use the name of the corresponding widgetdata tag'
print,'  as a keyword to a call to papco and set to new value'
print,''

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_PrintMenu_event, event
;*
;* DESCRIPTION:
;*      Handles events that are invoked, when the user selects an entry
;*      from the printer-menu.
;*
;* INPUTS:
;*      Event       a string indicating the caller.
;*                  'papco' if caller is papco main
;*                  'slice' if caller is a slice.
;*
;* KEYWORDS:
;*  none
;*
;* CALLING SEQUENCE:
;*  papco_PrintMenu_event, anXManagerEvent
;*  Called by papco_main_Event only.
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_PrintMenu_event, event, CLOSE_DEV=CLOSE_DEV

COMMON plot_composer, widgetdata

papco_Input, /DISABLE
default_print_choice=widgetdata.default_print_choice
SLICE=event
;CLOSE_DEV controls if device is kept open for multiple plots.
if keyword_set(CLOSE_DEV) then close_dev=CLOSE_DEV else close_dev=[0,0]

CASE default_print_choice OF
    1: BEGIN                    ;-----print to Standard printer
       IF papco_CanYouPlot() THEN papco_PrintPlots, widgetdata.printerInfo, $
         /printer, SLICE=slice, CLOSE_DEV=close_dev
    END
    2 : BEGIN                   ;-----print to default-file
       IF papco_CanYouPlot() THEN papco_PrintPlots, widgetdata.printerInfo, $
         /file, SLICE=slice, CLOSE_DEV=close_dev
    END
    3 : BEGIN                   ;-----select another printer
       IF papco_CanYouPlot() THEN BEGIN
         printer=papco_Printer(widgetdata.printerInfo,$
                               GROUP=widgetdata.main_Base)
         IF NOT printer.canceled THEN papco_PrintPlots, printer, $
           SLICE=slice, CLOSE_DEV=close_dev
       ENDIF
    END
    4 : BEGIN ;-----select the default printer
       print, 'not handled here, handled directly in papco.pro'
    END
    5 : PRINT, 'empty line...'
    6 : BEGIN ;-----BATCH - print -> preview
       title='BATCH->SCREEN'
       preview=1  &  widgetdata.output=0
    END
    7 : BEGIN ;-----BATCH - output data -> to file
       title='BATCH->DATA OUTPUT'
       preview=2  &  widgetdata.output=1  &  widgetdata.quiet=1
    END
    8 : BEGIN                    ;-----BATCH - print -> to file
       title='BATCH->PRINT FILE'
       preview=0  &  widgetdata.output=0  &  widgetdata.quiet=1
    END
ENDCASE

IF default_print_choice ge 6 then begin
    ;check if PATH to batch files exists, use default otherwise
    IF not papco_finddir(widgetdata.batchFilePath) then $
      path=papco_getenv('PAPCO_HOME') ELSE path=widgetdata.batchFilePath
    fName=dialog_pickfile(/READ, /MUST_EXIST, $
                   GROUP=widgetdata.main_base, $
                   PATH=path,$
                   FILE=widgetdata.BATCHFILENAME, $
                   FILTER='*.papco_batch', $
                   TITLE=title+' - Pick file', $
                   GET_PATH=pathSelected)
    IF fName NE '' THEN BEGIN
      widgetdata.batchFilePath=pathSelected
      widgetdata.BATCHFILENAME=fname
      IF strpos(fName(0), '/') GT -1 THEN pathSelected=''
      papco_Batch, pathSelected+fName, PREVIEW=preview
    ENDIF
ENDIF

papco_Refresh, /NEW_TIMERANGE
papco_Input, /ENABLE

END

;******************************************************************************
;* PROCEDURE:
;*     PRO papco_main_Event, Event
;*
;* DESCRIPTION:
;*     The main XMANAGER-event-handler.
;*     All events to the main-window are processed here
;*
;* INPUTS:
;*     event       an XManager-Event
;*
;* KEYWORDS:
;*     USER_EVENT set to the event uvalue for user call of this routine
;*     NOASK      set to skip any queries
;*
;* CALLING SEQUENCE:
;*     Called by XMANAGER only !
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     november 1995, modified by R.Friedel
;******************************************************************************
PRO papco_main_Event, Event, USER_EVENT=USER_EVENT, NOASK=NOASK

COMMON PLOT_COMPOSER, widgetdata
COMMON PAPCO_DRAW, drawWidgetData
COMMON MISSIONS, missions
COMMON ADDINS, plot_types
COMMON ADDBTN, addbuttons
COMMON MOUSE_INFORMATION, mouseData ;general mouse information
COMMON config, configkind
COMMON papco_graphics, papco_drawn_window
COMMON select_from_batch_data, thiswidgetdata

IF keyword_set(USER_EVENT) THEN Ev=USER_EVENT ELSE $
  WIDGET_CONTROL, Event.Id, GET_UVALUE=Ev

; === handle Events for adding an add-in plot =================================

; Call the routine which adds the plot to the paneleditor.The list of
; included plottypes is in common ADDINS

if addbuttons(0) ne '' then begin ;do addbutton events only if there are any!

    result=strpos(strlowcase([addbuttons,plot_types]), $
                  strlowcase(strplotdesc(ev)),0)
    typeindex=where(result eq 0,count)

    if count ne 0 then begin
      panelkind=strplotdesc(ev) & configkind=panelkind
      proc_call='addpanel_'+panelkind
      if widgetdata.default_trap eq 1 then catch, error_status $
        else error_status=0
      if error_status ne 0 then begin     ;call error handler
        papco_error_handler, error_status, proc_call, panelkind, panelVector, $
          /NO_ERR_DRAW
        error_status=0 &  catch, /CANCEL  ;disable error catching
        goto, outevent
      endif
      CALL_PROCEDURE,proc_call, Event
      catch, /CANCEL                      ;disable error catching
      widgetdata.needto_read_data=1       ;new data added: re-read data
      widgetdata.need_to_draw=1
      papco_refresh,/NEED_READ
      ;reset overplot button
      WIDGET_CONTROL, widgetdata.pb_overPlot, set_button=0
      goto,outevent
    endif

ENDIF

CASE Ev OF

; === handle timer-events =====================================================
    'clearTimeRange' : BEGIN
      papco_DrawTimeRange, 0, 0, /CLEAR_ONLY
    END

; === handle Events for empty buttons =========================================
    'pb_empty_button': BEGIN & END

; === handle Events for choosing the time =====================================
    'ef_startTime': BEGIN
        WIDGET_CONTROL, widgetdata.ef_startTime, GET_VALUE=val
        tstart=convert_timeToSeconds(val(0), /T90, /MJDT)
        IF tstart.err THEN dummy=messageBox(tstart.errtext, ['OK'], /CENTER, $
                           TITLE='Enter start time') $
        ELSE BEGIN
          ; convert tstart to seconds since minTime start of day
          sec=(tstart.value.mjd-widgetdata.minTime.mjd)*86400+tstart.value.t
          papco_setTime, sec, /STARTTIME
          papco_Refresh, /TIME
          widgetdata.need_to_draw=1
        ENDELSE
    END

    'sl_startTime': BEGIN
        widgetdata.need_to_draw=1
        papco_setTime, event.value, /STARTTIME
        papco_Refresh, /TIME
    END

    'ef_endTime': BEGIN
        WIDGET_CONTROL, widgetdata.ef_endTime, GET_VALUE=val
        tend=convert_timeToSeconds(val(0), /T90, /MJDT)
        IF tend.err THEN dummy=messageBox(tend.errtext, ['OK'], /CENTER, $
                           TITLE='Enter end time') $
        ELSE BEGIN
          ; convert tstart to seconds since minTime start of day
          sec=(tend.value.mjd-widgetdata.minTime.mjd)*86400+tend.value.t
          papco_setTime, sec, /ENDTIME
          papco_Refresh, /TIME
          widgetdata.need_to_draw=1
        ENDELSE
    END

    'sl_endTime': BEGIN
        widgetdata.need_to_draw=1
        papco_setTime, event.value, /ENDTIME
        papco_Refresh, /TIME
    END

    'pb_previous': BEGIN
       case 1 of
         (widgetdata.time_choice_action eq 3): begin        ; from batch
           papco_main_Event, 0, USER_EVENT='pb_prev_batch'
         end
         (widgetdata.time_choice_action eq 1): begin        ; by orbit
           WIDGET_CONTROL, widgetdata.ef_orbit, get_value=this_orbit
           new_orbit=this_orbit-1 > 1
           dummy=papco_setOrbit(new_orbit)
           WIDGET_CONTROL, widgetdata.ef_orbit, set_value=new_orbit
         end
         (widgetdata.time_choice_action eq 0) or $
           (widgetdata.time_choice_action eq 2): begin      ; manual / day
           WIDGET_CONTROL, widgetdata.ef_minTime, GET_VALUE=startTime
           WIDGET_CONTROL, widgetdata.ef_maxTime, GET_VALUE=endTime
           startTime=convert_timeToSeconds(startTime(0), /T90, /MJDT)
           endTime=convert_timeToSeconds(endTime(0), /T90, /MJDT)
           deltaTime=addSubTime(endTime.value, startTime.value, /SUBTRACT)
           newStartTime=addSubTime(startTime.value, deltatime, /SUBTRACT)
           newEndTime=startTime.value
           newStartTime=convert_secondsToTime(newStartTime, /T90, /MJDT)
           newEndTime=convert_secondsToTime(newEndTime, /T90, /MJDT)
           WIDGET_CONTROL, widgetdata.ef_minTime, SET_VALUE=newStartTime
           WIDGET_CONTROL, widgetdata.ef_maxTime, SET_VALUE=newEndTime
           dummy=SET_MANUAL_TIME(/MINTIME)
           dummy=SET_MANUAL_TIME(/MAXTIME)
         end
       endcase

       widgetdata.need_to_draw=1
       widgetdata.needto_read_data=1   ;new time range: re-read data
       papco_refresh,/NEED_READ
       ;also draw the new time range if requested
       if widgetdata.default_draw_on_step then $
         papco_main_Event, 0, USER_EVENT='pb_draw'
    END

    'pb_next': BEGIN
       case 1 of
         (widgetdata.time_choice_action eq 3): begin        ; from batch
           papco_main_Event, 0, USER_EVENT='pb_next_batch'
           papco_Refresh, /new_timeRange
         end
         (widgetdata.time_choice_action eq 1): begin        ; by orbit
           WIDGET_CONTROL, widgetdata.ef_orbit, get_value=this_orbit
           new_orbit=this_orbit+1
           dummy=papco_setOrbit(new_orbit)
           WIDGET_CONTROL, widgetdata.ef_orbit, set_value=new_orbit
         end
         (widgetdata.time_choice_action eq 0) or $
           (widgetdata.time_choice_action eq 2): begin ; manual / day
           WIDGET_CONTROL, widgetdata.ef_minTime, GET_VALUE=startTime
           WIDGET_CONTROL, widgetdata.ef_maxTime, GET_VALUE=endTime
           startTime=convert_timeToSeconds(startTime(0), /T90, /MJDT)
           endTime=convert_timeToSeconds(endTime(0), /T90, /MJDT)
           deltaTime=addSubTime(endTime.value, startTime.value, /SUBTRACT)
           newEndTime=addSubTime(endTime.value, deltatime, /ADD)
           newStartTime=endTime.value
           newStartTime=convert_secondsToTime(newStartTime, /T90, /MJDT)
           newEndTime=convert_secondsToTime(newEndTime, /T90, /MJDT)
           WIDGET_CONTROL, widgetdata.ef_minTime, SET_VALUE=newStartTime
           WIDGET_CONTROL, widgetdata.ef_maxTime, SET_VALUE=newEndTime
           dummy=SET_MANUAL_TIME(/MINTIME)
           dummy=SET_MANUAL_TIME(/MAXTIME)
         end
       endcase

       widgetdata.need_to_draw=1
       widgetdata.needto_read_data=1   ;new time range: re-read data
       papco_refresh,/NEED_READ
       ;also draw the new time range if requested
       if widgetdata.default_draw_on_step then $
         papco_main_Event, 0, USER_EVENT='pb_draw'
    END

; === handle Events for selecting Orbit/time-range ============================
    'ef_orbit' : BEGIN
       IF event.value GE 0 THEN BEGIN
         papco_input, /DISABLE
         dummy=papco_setOrbit(event.value)
         papco_input, /ENABLE
         widgetdata.needto_read_data=1    ;new orbit: re-read data
         widgetdata.need_to_draw=1
         if widgetdata.default_adjust_time ne 0 then papco_time_adjust
         papco_refresh,/NEED_READ
       endif
    END

    'pdm_satellite': BEGIN
        WIDGET_CONTROL,widgetdata.ef_orbit,SENSITIVE=0
        WIDGET_CONTROL,widgetdata.pb_orbitSelect,SENSITIVE=0
        widgetdata.default_satellite=Event.Value-1
        sat_label=missions(widgetdata.default_satellite)
        WIDGET_CONTROL,widgetdata.ef_orbit,SENSITIVE=1
        WIDGET_CONTROL,widgetdata.pb_orbitSelect,SENSITIVE=1
        widgetdata.satellite_name=sat_label
        WIDGET_CONTROL, widgetdata.label_satellite, SET_VALUE=sat_label
        widget_control, widgetdata.ef_orbit, set_value=0
        widgetdata.default_orbitInfo.number=0
    END

    'pb_orbitSelect': BEGIN
        papco_selectOrbit, widgetdata.satellite_name, $
                           "papco_Orbit_Selected", $
                           GROUP=widgetdata.main_Base
    END

    'pdm_time_choice_action': BEGIN
        widgetdata.time_choice_action=event.value-1
        papco_Refresh, /SELECT_TIME_METHOD
    END

     'ef_minTime' : BEGIN
        dummy=SET_MANUAL_TIME(/MINTIME)
        widgetdata.needto_read_data=1   ;new time range: re-read data
        widgetdata.need_to_draw=1
        papco_refresh,/NEED_READ
    END

     'ef_maxTime': BEGIN
        dummy=SET_MANUAL_TIME(/MAXTIME)
        widgetdata.needto_read_data=1   ;new time range: re-read data
        widgetdata.need_to_draw=1
        papco_refresh,/NEED_READ
    END

     'ef_DayTime': BEGIN
        dummy=SET_MANUAL_TIME(/DAYTIME)
        widgetdata.needto_read_data=1   ;new time range: re-read data
        widgetdata.need_to_draw=1
        papco_refresh,/NEED_READ
    END

    'pb_read_batch': BEGIN
       fln_old=widgetdata.BATCHFILEPATH+widgetdata.BATCHFILENAME
       papco_edit_batch, /FILENAME
       fln=widgetdata.BATCHFILEPATH+widgetdata.BATCHFILENAME
       if fln ne fln_old then begin
         if XREGISTERED('papco_select_from_batch') then begin
           widget_control,thiswidgetdata.panel_base, /DESTROY
           papco_select_from_batch, fln, GROUP=widgetdata.base
         endif
       endif
       widgetdata.batch_count=0
       papco_read_batch, fln, time_str
       widget_control,widgetdata.label_batch_name, $
         set_value=widgetdata.BATCHFILENAME
       widget_control,widgetdata.label_batch_time, $
         set_value=time_str
    END

    'pb_next_batch': BEGIN
       widgetdata.batch_count=widgetdata.batch_count+1
       fln=widgetdata.BATCHFILEPATH+widgetdata.BATCHFILENAME
       papco_read_batch, fln, time_str
       widget_control,widgetdata.label_batch_time, set_value=time_str
       papco_Refresh, /new_timeRange
    END

    'pb_prev_batch': BEGIN
       widgetdata.batch_count=(widgetdata.batch_count-1)>0
       fln=widgetdata.BATCHFILEPATH+widgetdata.BATCHFILENAME
       papco_read_batch, fln, time_str
       widget_control,widgetdata.label_batch_time, set_value=time_str
       papco_Refresh, /new_timeRange
    END

    'pb_pick_batch': BEGIN
       if widgetdata.BATCHFILENAME eq '' then return
       fln=widgetdata.BATCHFILEPATH+widgetdata.BATCHFILENAME
       papco_select_from_batch, fln, GROUP=widgetdata.base
    END

    'pb_batch' : BEGIN
       homedir=papco_getenv('PAPCO_HOME')
       fln=homedir+'papco_products/batch/interactive.papco_batch'

       if (widgetdata.minTime.mjd EQ widgetdata.maxTime.mjd) and $
          (widgetdata.minTime.t EQ widgetdata.maxTime.t) then begin
          dummy=messagebox(['before adding a time-range to ' + $
                             fln + ' you should choose one !'], ['o.k.'])
       endif else begin
          orb_str=''
          time_str=''
          if widgetdata.default_orbitInfo.number ne 0 then $
            orb_str=strtrim(string(widgetdata.default_orbitInfo.number),2)+', '
          timeFrom=convert_secondsToTime(widgetdata.startTime, /t90, /mjdt, $
                                         COUNTRY=widgetdata.default_us_europe)
          timeTo=convert_secondsToTime(widgetdata.endTime, /t90, /mjdt, $
                                       COUNTRY=widgetdata.default_us_europe)
          time_str=timeFrom + ' - '+timeTo
          PRINT, '%PAPCO (batch): Adding to ' +fln+': '+orb_str+time_str
          if widgetdata.time_choice_action EQ 0 then orb_str=''
          openw, out, fln, /APPEND, /GET_LUN
          printf, out, orb_str+time_str
          close, out & free_lun, out
       endelse
    END

; === handle mouse option events ==============================================
    'pdm_mouse_action': BEGIN
        papco_Cursor_Reset, /FULLY
        widgetdata.mouse_doesSlice=Event.Value
        mouseData.mouseactiontyp=Event.Value
        papco_Refresh,/MOUSE
        n_slice=n_elements(widgetdata.mouseAction)
        if (Event.Value+1) eq n_slice then begin
          slice_set=widgetdata.default_multiple_slice(0:n_slice-3)
          result=papco_multiple_slice(widgetdata.mouseAction,slice_set )
          widgetdata.default_multiple_slice(0:n_slice-3)=result
        endif
     END

; === handle slice option events ==============================================
    'pdm_slice_action': BEGIN
        widgetdata.slice_choice=Event.Value
        papco_Refresh, /SLICE
    END

; === handle print option events ==============================================
    'pdm_print_action': BEGIN
       if Event.Value eq 5 then return
       if Event.Value eq 4 then begin
         printer=papco_Printer(widgetdata.printerInfo, $
            GROUP=widgetdata.main_Base, /SELECT_DEFAULT)
         IF NOT printer.canceled THEN BEGIN
           widgetdata.printerInfo.usePrinter=printer.usePrinter
           widgetdata.printerInfo.printer=printer.printer
           widgetdata.printerInfo.askForFileName=printer.askForFileName
           widgetdata.printerInfo.fileType=printer.fileType
           widgetdata.printerInfo.fileName_tmpl=printer.fileName_tmpl
           papco_refresh, /printer
         ENDIF
         choice=widgetdata.default_print_choice
       endif else choice=Event.Value

       widgetdata.default_print_choice=choice
       papco_Refresh,/printer
    END

; === handle print option events ==============================================
    'pdm_edit_action': BEGIN
       widgetdata.edit_choice=Event.Value
       papco_Refresh,/edit
    END

; === handle events for the list of choosen plots  ============================
    'lb_plotInfos': BEGIN     ;clicks in the list - box
        widgetdata.selectedPlot=event.index / 2
        sPlot=widgetdata.selectedPlot
        if sPlot GE widgetdata.numberOfPlots THEN BEGIN
           widgetdata.selectedPlot=-1
           widget_control, widgetdata.lb_plotInfos, set_list_select=-1
           for i=0, n_elements(widgetdata.edit_buttons)-1 do $
             widget_control, widgetdata.edit_buttons(i), sensitive=0
           widget_control, widgetdata.pb_overPlot, set_button=0
           widget_control, widgetdata.pb_anchored, set_button=0
        endif else begin
           for i=0, n_elements(widgetdata.edit_buttons)-1 do $
             widget_control, widgetdata.edit_buttons(i), sensitive=1
           widget_control, widgetdata.pb_overplot, $
             set_button=widgetdata.plotInfos(sPlot).overplot
           widget_control, widgetdata.pb_anchored, $
             set_button=widgetdata.anchored(sPlot)
        endelse

        IF event.clicks GT 1 THEN BEGIN
           sPlot=widgetdata.selectedPlot
           IF sPlot GE 0 THEN papco_EditPlot, sPlot
        ENDIF
    END

    'pb_MoveUp': BEGIN        ; -- pressing the up-arrow-button -------------
       papco_MoveUp
    END

    'pb_MoveDown': BEGIN      ; -- pressing the down-arrow-button ------------
       papco_MoveDown
    END

    'pb_Enlarge' : BEGIN      ; -- pressing the enlarge-arrow-button ---------
       papco_EnlargeMe, /NOJOIN, /ALONE
    END

    'pb_Shrink' : BEGIN       ; -- pressing the shrink-arrow-button ----------
       papco_ShrinkMe, /NOSTRIP, /ALONE
    END

    'pb_EnlargeOthers': BEGIN ; -- pressing the enlarge-others-arrow-button --
       papco_EnlargeOthers
    END

    'pb_ShrinkOthers':  BEGIN ; -- on pressing the shrinkothers-arrow-button -
       papco_ShrinkOthers
    END

    'pb_EditPlot': BEGIN       ; -- on pressing the edit-button --------------
        sPlot=widgetdata.selectedPlot
        IF sPlot GE 0 THEN BEGIN
         case event.value of
           1 : papco_EditPlot, sPlot               ; edit the plot-properties
           2 : papco_EditPlot, sPlot, /CHANGETYPE  ; change the plot-type
           3 : begin
             papco_DeletePlot, sPlot
             widgetdata.need_to_draw=1
           end
           4 : BEGIN
               max_panels=n_elements(widgetdata.PLOTINFOS)
               n=widgetdata.NUMBEROFPLOTS
               IF n LE max_panels THEN BEGIN

                   ;make sure a new instance of the
                   ;plotinfo pointer variable gets made, and set to the
                   ;contents of the exisiting one.

                   ;check that plotinfo pointer variable exists!
                   r = size(*widgetdata.PLOTINFOS(sPlot).USR_PTR1, /ST)
                   IF r.TYPE_NAME EQ 'INT' THEN BEGIN ;just copy
                       new_plotinfo = widgetdata.PLOTINFOS(sPlot)
                   ENDIF ELSE BEGIN ;make new structure / copy
                       old_control = *widgetdata.PLOTINFOS(sPlot).USR_PTR1
                       new_plotinfo = widgetdata.PLOTINFOS(sPlot)
                       new_plotinfo.USR_PTR1 = ptr_new(old_control)
                   ENDELSE

                   widgetdata.PLOTINFOS(n) = new_plotinfo
                   IF n GT 0 THEN BEGIN
                       widgetdata.plotInfos(0:(n-1)).panelPosition= $
                         widgetdata.plotInfos(0:(n-1)).panelPosition + $
                         widgetdata.plotInfos(n).panelHeight
                       widgetdata.plotInfos(n).panelPosition=0
                   ENDIF
                   widgetdata.NUMBEROFPLOTS=n+1
                   papco_refresh, /LIST
               ENDIF ELSE BEGIN
                   result=messageBox(['Reached maximum number of ', $
                          'panels ('+varprt(max_panels)+').' ],  $
                          ['Cancel copy'], /center, title='Too many panels')
                   message,'Reached maximum number of panels ('+ $
                            varprt(max_panels)+').' ,/info
               ENDELSE
           END
         endcase
        ENDIF
    END

    'pb_Reformat' : BEGIN      ; -- selecting the 'reformat'-menu ------------
       case event.value of
          1 : papco_stripEmptySlots     ; strip all empty slots
          2 : papco_resetSlots          ; reset all panels to height 1
          3 : papco_resortslots         ; resort the panels ascending order
          4 : papco_joinslots           ; remove all overlaps/overplots
       endcase
       widgetdata.need_to_draw=1
    END

    'pb_overplot' : begin
        sPlot=widgetdata.selectedPlot
        IF sPlot GE 0 THEN begin
          if widgetdata.plotInfos(sPlot).overplot EQ 1 then $
            widgetdata.plotInfos(sPlot).overplot=0 $
          else $
            widgetdata.plotInfos(sPlot).overplot=1
          widget_control, widgetdata.pb_overplot, $
            set_button=widgetdata.plotInfos(sPlot).overplot
          papco_Refresh, /LIST
       endif else $
         widget_control, widgetdata.pb_overplot, set_button=1, sensitive=0
     end

    'pb_anchored' : begin
        sPlot=widgetdata.selectedPlot
        IF sPlot GE 0 THEN begin
          if widgetdata.anchored(sPlot) EQ 1 then $
            widgetdata.anchored(sPlot)=0 else $
            widgetdata.anchored(sPlot)=1
          widget_control, widgetdata.pb_anchored, $
            set_button=widgetdata.anchored(sPlot)
          papco_Refresh, /LIST
       endif
     end

; === handle draw on step =====================================================
    'spb_draw_on_step':  if widgetdata.default_draw_on_step then $
       widgetdata.default_draw_on_step=0 else widgetdata.default_draw_on_step=1

; === handle the action buttons ===============================================
    'pb_draw' : BEGIN
        widgetdata.output=0
        IF papco_canYouPlot() THEN BEGIN
            IF widgetdata.no_draw_window eq 0 then widgetdata.no_draw_window=1
            papco_Input, /DISABLE
            draw_win_name='papco_draw_window_' + $
                string(widgetdata.no_draw_window,format="(i2.2)")
            papco_draw_window, draw_win_name  ;create or show draw window
            papco_DrawPlots
            papco_Input, /ENABLE
            widget_control, widgetdata.pb_draw_new, sensitive=1
            widget_control, widgetdata.pb_annotate, sensitive=1
            if widgetdata.annotatefilename ne '' then $
              widget_control, widgetdata.pb_rm_annot, sensitive=1 else $
              widget_control, widgetdata.pb_rm_annot, sensitive=0
       ENDIF
    END

    'pb_draw_new' : BEGIN
       widgetdata.output=0
       papco_Input, /DISABLE
       if widgetdata.dw_draw ne 0 then begin  ;desensitise old window
         widget_control,widgetdata.dw_draw, sensitive=0
         widget_control,drawWidgetData.pb_del_all, sensitive=0
         widget_control,drawWidgetData.pb_help, sensitive=0
         widget_control,drawWidgetData.pb_del_old, sensitive=0
         widget_control,drawWidgetData.pb_draw, sensitive=0
       endif
       widgetdata.no_draw_window=widgetdata.no_draw_window+1
       draw_win_name='papco_draw_window_' + $
          string(widgetdata.no_draw_window,format="(i2.2)")
       papco_draw_window,draw_win_name ;create or show draw window
       widgetdata.need_to_draw=1
       papco_DrawPlots
       papco_Refresh, /new_timeRange
       papco_Input, /ENABLE
    END

    'pb_annotate': BEGIN
       papco_Input, /DISABLE
       ;disable draw window events (they get buffered and fire up the
       ;papco cursor procedure after annotate is done!
       dw_Draw=widgetdata.dw_Draw
       WIDGET_CONTROL, dw_Draw, DRAW_BUTTON_EVENTS=0, DRAW_MOTION_EVENTS=0
       ;set colors that annotate can use from papco color table
       inidices=papco_get_Colorindices(3)
       fixed_color=indgen(16)+inidices(1)+1
       ;see if an annotate file is set for this product
       if widgetdata.annotateFileName ne '' then begin
         ;clear drawing and restore from buffer, then annotate
         papco_clear_drawing
         TV,papco_drawn_window, true=widgetData.truecolor
         papco_annotate, COLOR_INDICES=fixed_color, $
                   LOAD_FILE=widgetdata.annotateFileName
       endif else papco_annotate, COLOR_INDICES=fixed_color
       papco_Input, /ENABLE
       WIDGET_CONTROL, dw_Draw, /DRAW_BUTTON_EVENTS, /DRAW_MOTION_EVENTS
    END

    'pb_rm_annot' : BEGIN
       if (widgetdata.annotatefilename ne '') AND $
          (widgetdata.no_draw_window ne 0) then begin
         widgetdata.annotatefilename=''
         ;clear drawing and restore from buffer, then annotate
         papco_clear_drawing
         if n_elements(papco_drawn_window) ne 0 then $
           TV,papco_drawn_window, true=widgetData.truecolor
         widget_control, widgetdata.pb_rm_annot, sensitive=0
       endif
    END

    'pb_output': BEGIN
       papco_Input, /DISABLE
       IF widgetdata.output EQ 0 THEN widgetdata.output=1
       widgetdata.need_to_draw=1
       IF widgetdata.no_draw_window eq 0 then widgetdata.no_draw_window=1
       IF papco_canYouPlot() THEN BEGIN
         draw_win_name='papco_draw_window_' + $
             string(widgetdata.no_draw_window,format="(i2.2)")
         papco_draw_window,draw_win_name  ;create or show draw window
         papco_DrawPlots
      ENDIF
       papco_Input, /ENABLE
    END

    'pb_font' : papco_set_default_widget_font

    'pb_default' : papco_defaults

    'pb_print' : BEGIN
        widgetdata.need_to_draw=1
        papco_PrintMenu_Event, 'papco'
    END

    'pb_reset' : BEGIN
       IF messageBox(['Do you want to delete all plots?'], ['Yes', 'No'], $
                       title='Reset papco') EQ 0 THEN papco_resetstate
    END

    'pb_new_config': BEGIN
       papco_reconfigure, GROUP_LEADER=widgetdata.base
       dummy=messageBox(['PAPCO will now exit.',  $
                         'run "@papco_recon" from command line', $
                         'to compile new modules and addins.', $
                         'PAPCO will restart automatically.'], $
                        ['OK'], /CENTER, $
                        TITLE='PAPCO New Config')
       papco_main_Event, 0, USER_EVENT='pb_done', /NOASK
    END

    'pb_forward' : BEGIN
       papco_load_next_state
       ;also draw the new product range if requested
       if widgetdata.default_draw_on_step then $
         papco_main_Event, 0, USER_EVENT='pb_draw'
    END

    'pb_back' : BEGIN
       papco_load_previous_state
       ;also draw the new product range if requested
       if widgetdata.default_draw_on_step then $
         papco_main_Event, 0, USER_EVENT='pb_draw'
    END

    'pb_save' : BEGIN
        papco_Input, /DISABLE
        papco_saveState
        papco_Input, /ENABLE
    END

    'pb_load' : BEGIN
        papco_Input, /DISABLE
        papco_LoadState
        papco_Input, /ENABLE
        if widgetdata.needto_read_data eq 1 then papco_refresh,/NEED_READ
    END

    'pb_edit_batch': BEGIN
       papco_edit_batch, GROUP=widgetdata.base
    END

    'pb_edit_color': BEGIN
       papco_subtable_editor, GROUP=widgetdata.base
    END

    'pb_about' : BEGIN
       dummy=messageBox(['PAPCO is a free IDL package to combine data plots',$
                    'writen and maintained by A. Keese and R.Friedel', $
                    '','For more information, check the papco home page at', $
                    'http://leadbelly/ccr/software/papco/papco.html', $
                    'for online Manual and downloadble latest version.', $
                    '','Comments and questions can be directed to', $
                    'friedel@lanl.gov'], $
                    ['Back'], $
                    title='About papco')
    END

    'pb_help' : BEGIN
        xdisplayfile, papco_helpfile('papco_main.help'), $
          title='papco Online-Help', group=event.top
    END

    'pb_done' : BEGIN
       if not keyword_set(NOASK) then begin
         if messageBox(['Do you want to exit papco ?'], $
                       ['Yes', 'No'], title='Leaving papco') EQ 0 $
         then begin
           WIDGET_CONTROL, Event.top, /DESTROY
           papco_finish
           return
         endif
       endif else begin
         WIDGET_CONTROL, widgetdata.base, /DESTROY
         message,'run "@papco_recon" from command line.',/info
       endelse
    END

    'pb_done_save' : BEGIN
        IF messageBox(['Do you want to exit papco and save all routines ?'], $
                      ['Yes', 'No'], title='Leaving and Saving papco') EQ 0 $
          THEN BEGIN
            WIDGET_CONTROL, Event.top, /DESTROY
            papco_save_compiled
            papco_finish
            RETURN
         ENDIF
    END

     ELSE : BEGIN
       message, 'no match found: '+ ev, /info
       help, /struct, event
    END
   ENDCASE
  outevent:

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_refresh, TIME=TIME, LIST=LIST, NEW_TIMERANGE=NEW_TIMERANGE,
;*             ALL=ALL, PRINTER=PRINTER,SELECT_TIME_METHOD=SELECT_TIME_METHOD
;*                         NEED_READ=NEED_READ
;* DESCRIPTION:
;*      Refresh the screen to display the contents of the actual
;*  widgetdata
;*
;* INPUTS:
;*  none
;*
;* KEYWORDS:
;*  TIME          if set, the time-entry-fields and the time-
;*                   sliders are updated
;*  LIST          if set, the list containing the plot-descriptions
;*                   is updated
;*  ORBIT         if set, the displayed orbit is updated.
;*                   Specifying ORBIT implicitly specifies TIME
;*  PRINTER       the printer-menus are updated to display the
;*                   default-printer and default-print-file
;*  SLICE         the mouse does window is updated according to
;*                   the last set option
;*  ALL           everything will be updated, when ALL is set.
;*
;* CALLING SEQUENCE:
;*  e.g.: papco_Refresh, /LIST
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     updated for SLICE, May 1997, Reiner Friedel
;*     updated for NEED_READ, Oct 1997, Reiner Friedel
;******************************************************************************
PRO papco_refresh, TIME=TIME, LIST=LIST, NEW_TIMERANGE=NEW_TIMERANGE, $
                   SATTELITE=SATTELITE, MOUSE=MOUSE, PRINTER=PRINTER, $
                   EDIT=EDIT, SELECT_TIME_METHOD=SELECT_TIME_METHOD, ALL=ALL, $
                   NEED_READ=NEED_READ, DRAW_ON_STEP=DRAW_ON_STEP, $
                   SLICE=SLICE

COMMON PLOT_COMPOSER, widgetdata
COMMON MOUSE_INFORMATION, mouseData ;general mouse information
COMMON MISSIONS, missions
COMMON time, xut1,xut2
COMMON mjdt, mjdt_start, mjdt_end

;------------------------------------------------------------------------------
  IF KEYWORD_SET(ALL) THEN BEGIN
    LIST=1     & NEW_TIMERANGE=1 & TIME=1 & PRINTER=1   & SELECT_TIME_METHOD=1
    MOUSE=1    & EDIT=1 & SATTELITE=1 & DRAW_ON_STEP=1  & SLICE=1
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(MOUSE) THEN BEGIN
    choice= widgetdata.mouse_doesSlice < (n_elements(widgetdata.mouseaction)-1)
    mouseAction=widgetdata.mouseaction(choice).NAME
    WIDGET_CONTROL, widgetdata.text_mouse_action, $
                    SET_VALUE=strcenter(strtrim(mouseAction,2),35)
    mouseData.mouseactiontyp=widgetdata.mouse_doesSlice
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(SLICE) THEN BEGIN
    choice= widgetdata.slice_choice < (n_elements(widgetdata.sliceaction)-1)
    sliceAction=widgetdata.sliceaction(choice).NAME
    WIDGET_CONTROL, widgetdata.text_slice_action, $
                    SET_VALUE=strcenter(strtrim(sliceAction,2),35)
  ENDIF

;------------------------------------------------------------------------------
; when a new orbit is choosen, the print-file-name may have to be altered
  IF KEYWORD_SET(SELECT_TIME_METHOD) THEN BEGIN
    alt=widgetdata.time_choice_action
    time_choice=widgetdata.time_choice(widgetdata.time_choice_action+1).name
    WIDGET_CONTROL, widgetdata.text_time_choice_action, $
      SET_VALUE=strcenter(strmid(time_choice,1,10),12)

    case widgetdata.time_choice_action of
      0: BEGIN
        widget_control, widgetdata.orbitSelect_base, MAP=0
        widget_control, widgetdata.manual_time_base, MAP=1
        widget_control, widgetdata.day_time_base, MAP=0
        widget_control, widgetdata.batch_time_base, MAP=0
      END
      1: BEGIN
        widget_control, widgetdata.orbitSelect_base, MAP=1
        widget_control, widgetdata.manual_time_base, MAP=0
        widget_control, widgetdata.day_time_base, MAP=0
        widget_control, widgetdata.batch_time_base, MAP=0
      END
      2: BEGIN
        widget_control, widgetdata.orbitSelect_base, MAP=0
        widget_control, widgetdata.manual_time_base, MAP=0
        widget_control, widgetdata.day_time_base, MAP=1
        widget_control, widgetdata.batch_time_base, MAP=0
      END
      3:BEGIN
        widget_control, widgetdata.orbitSelect_base, MAP=0
        widget_control, widgetdata.manual_time_base, MAP=0
        widget_control, widgetdata.day_time_base, MAP=0
        widget_control, widgetdata.batch_time_base, MAP=1
      END
     ENDCASE
     IF alt NE widgetdata.time_choice_action THEN NEW_TIMERANGE=1
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(NEW_TIMERANGE) OR KEYWORD_SET(TIME) THEN PRINTER=1

  IF KEYWORD_SET(NEW_TIMERANGE) THEN BEGIN
    TIME=1
    IF (widgetdata.minTime.mjd EQ widgetdata.maxTime.mjd) AND $
        (widgetdata.minTime.T EQ widgetdata.maxTime.T) THEN BEGIN
      WIDGET_CONTROL, widgetdata.pb_previous, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.pb_next, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.ef_stepTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.sl_startTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.sl_startTime, $
                      SET_SLIDER_MIN=0l, SET_SLIDER_MAX=1l
      WIDGET_CONTROL, widgetdata.ef_startTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.sl_endTime, SENSITIVE=0, $
                      SET_SLIDER_MIN=0l, SET_SLIDER_MAX=1l
      WIDGET_CONTROL, widgetdata.ef_endTime, SENSITIVE=0
      widgetdata.startTime.mjd=0
      widgetdata.startTime.t=0
      widgetdata.endTime.mjd=0
      widgetdata.endTime.t=0
    ENDIF ELSE BEGIN
      WIDGET_CONTROL, widgetdata.pb_previous, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.pb_next, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.ef_stepTime, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.sl_startTime, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.ef_startTime, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.sl_endTime, SENSITIVE=1
      WIDGET_CONTROL, widgetdata.ef_endTime, SENSITIVE=1
      ;set time range for slider in seconds since start time
      ut1=long(widgetdata.minTime.t)
      ut2=long(widgetdata.maxTime.mjd-widgetdata.minTime.mjd)*86400l+ $
                                long(widgetdata.maxTime.t)
      WIDGET_CONTROL, widgetdata.sl_startTime, $
                      SET_SLIDER_MIN=ut1, $
                      SET_SLIDER_MAX=ut2
      WIDGET_CONTROL, widgetdata.sl_endTime, $
                      SET_SLIDER_MIN=ut1, $
                      SET_SLIDER_MAX=ut2
      deltaTime=addSubTime(widgetdata.maxTime, widgetdata.mintime, /SUB)
      IF deltaTime.mjd LT 100 THEN BEGIN
        IF deltaTime.mjd LT 10 THEN $
          days='00'+STRTRIM(STRING(deltaTime.mjd),2) $
        ELSE $
          days='0'+STRTRIM(STRING(deltaTime.mjd),2)
      ENDIF ELSE days=STRTRIM(STRING(deltaTime.mjd),2)
      deltaTime.mjd=0
      hours=convert_secondsToTime(deltaTime, /MJDT)
      deltaTimeStr=days+' '+hours
      WIDGET_CONTROL, widgetdata.ef_stepTime, SET_VALUE=deltaTimeStr
    ENDELSE

     WIDGET_CONTROL, widgetdata.ef_orbit, $
          SET_VALUE=strtrim(string(widgetdata.default_orbitInfo.number),2)
     widgetdata.needto_read_data=1  ;new orbit: re-read data
     papco_refresh,/NEED_READ

     IF (widgetdata.minTime.mjd NE widgetdata.maxTime.mjd) OR $
        (widgetdata.minTime.T NE widgetdata.maxTime.T) THEN BEGIN
        time={MJDT, MJD:0l, T:0.0 }
        time.mjd=widgetdata.minTime.mjd
        time.t=widgetdata.minTime.t
        WIDGET_CONTROL, widgetdata.ef_minTime, $
          SET_VALUE=convert_secondsToTime(time, /T90, /MJDT, $
                                          COUNTRY=widgetdata.default_us_europe)
        time.mjd=widgetdata.maxTime.mjd
        time.t=widgetdata.maxTime.t
        WIDGET_CONTROL, widgetdata.ef_maxTime, $
          SET_VALUE=convert_secondsToTime(time, /T90, /MJDT, $
                                          COUNTRY=widgetdata.default_us_europe)
     ENDIF

;-- set up common start and end time for MJDT users
  mjdt_start=widgetdata.startTime
  mjdt_end=widgetdata.endTime
;--- set up common start and end time for old T90 users
  xut1=widgetdata.startTime.t90
  xut2=widgetdata.endTime.t90

  ENDIF
;------------------------------------------------------------------------------
  IF KEYWORD_SET(TIME) THEN BEGIN
    ;set time range for slider in seconds since start time
     ut1=long(widgetdata.startTime.mjd-widgetdata.minTime.mjd)*86400l+ $
                             long(widgetdata.startTime.t)
     ut2=long(widgetdata.endTime.mjd-widgetdata.minTime.mjd)*86400l+ $
                             long(widgetdata.endTime.t)

     if (ut2 lt 0) or (ut1 lt 0) then begin
        dummy=messageBox('End Time before Start Time', ['OK'], $
                           /CENTER, TITLE='Wrong time')
        return
     endif

     if (ut2 eq ut1) then begin
        dummy=messageBox('End Time equals Start Time', ['OK'], $
                           /CENTER, TITLE='Wrong time')
        return
     endif

     WIDGET_CONTROL, widgetdata.ef_startTime, $
    SET_VALUE=convert_secondsToTime(widgetdata.startTime, /T90, /MJDT, $
                                        COUNTRY=widgetdata.default_us_europe)
     WIDGET_CONTROL, widgetdata.sl_startTime, $
    SET_VALUE=ut1
     WIDGET_CONTROL, widgetdata.ef_endTime, $
    SET_VALUE=convert_secondsToTime(widgetdata.endTime, /T90, /MJDT, $
                                        COUNTRY=widgetdata.default_us_europe)
     WIDGET_CONTROL, widgetdata.sl_endTime, $
    SET_VALUE=ut2
     WIDGET_CONTROL, widgetdata.ef_dayTime, $
          SET_VALUE=strmid(convert_secondsToTime( $
                           widgetdata.startTime, /T90, /MJDT, $
                           COUNTRY=widgetdata.default_us_europe),0,10)
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(SATTELITE) THEN BEGIN
    WIDGET_CONTROL,widgetdata.ef_orbit,SENSITIVE=0
    WIDGET_CONTROL,widgetdata.pb_orbitSelect,SENSITIVE=0
    sat_label=missions(widgetdata.default_satellite)
    WIDGET_CONTROL,widgetdata.ef_orbit,SENSITIVE=1
    WIDGET_CONTROL,widgetdata.pb_orbitSelect,SENSITIVE=1
    widgetdata.satellite_name=sat_label
    WIDGET_CONTROL, widgetdata.label_satellite, SET_VALUE=sat_label
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(LIST) THEN BEGIN
     papco_RefreshList, LIST=LIST
     sPlot=widgetdata.selectedPlot
     IF sPlot GE 0 THEN BEGIN
       widget_control, widgetdata.lb_plotInfos, set_list_select=sPlot*2
       for i=0, n_elements(widgetdata.edit_buttons)-1 do $
         widget_control, widgetdata.edit_buttons(i), sensitive=1
       widget_control, widgetdata.pb_overplot, $
         set_button=widgetdata.plotInfos(sPlot).overplot
       widget_control, widgetdata.pb_anchored, $
         set_button=widgetdata.anchored(sPlot)
    ENDIF ELSE BEGIN
       for i=0, n_elements(widgetdata.edit_buttons)-1 do $
         widget_control, widgetdata.edit_buttons(i), sensitive=0
       widget_control, widgetdata.pb_overplot, set_button=0
       widget_control, widgetdata.pb_anchored, set_button=0
    ENDELSE
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(PRINTER) THEN BEGIN
    printerName= $
      papco_printerDescription(widgetdata.printerInfo, /PRINTER)
    fileName=papco_printerDescription(widgetdata.printerInfo, /FILE)
    printAction=widgetdata.printAction(widgetdata.default_print_choice).name

    WIDGET_CONTROL, widgetdata.text_print_action, $
       SET_VALUE=strcenter(strtrim(printAction,2),35)

    printer_info_str=printerName+' - '+fileName
    WIDGET_CONTROL, widgetdata.set_printer_file, $
       SET_VALUE=strcenter(printer_info_str,125)

    if (widgetdata.default_print_choice eq 3) or $
       (widgetdata.default_print_choice eq 4) then begin
       WIDGET_CONTROL,widgetdata.set_printer_file, SENSITIVE=0
    endif else begin
      WIDGET_CONTROL,widgetdata.set_printer_file, SENSITIVE=1
    endelse
  ENDIF

;------------------------------------------------------------------------------
  if KEYWORD_SET(EDIT) then begin
    choice= widgetdata.edit_choice
    editAction=widgetdata.editAction(choice).name
    WIDGET_CONTROL, widgetdata.text_edit_action, $
                    SET_VALUE=strcenter(strtrim(editAction,2),35)
  endif

;------------------------------------------------------------------------------
  IF KEYWORD_SET(NEED_READ) THEN BEGIN
     if widgetdata.needto_read_data eq 1 then begin
        WIDGET_CONTROL, widgetdata.ef_startTime ,sensitive=0
        WIDGET_CONTROL, widgetdata.sl_startTime ,sensitive=0
        WIDGET_CONTROL, widgetdata.ef_endTime ,sensitive=0
        WIDGET_CONTROL, widgetdata.sl_endTime ,sensitive=0
     endif else begin
        WIDGET_CONTROL, widgetdata.ef_startTime ,sensitive=1
        WIDGET_CONTROL, widgetdata.sl_startTime ,sensitive=1
        WIDGET_CONTROL, widgetdata.ef_endTime ,sensitive=1
        WIDGET_CONTROL, widgetdata.sl_endTime ,sensitive=1
     endelse
  ENDIF

;------------------------------------------------------------------------------
  IF KEYWORD_SET(DRAW_ON_STEP) THEN BEGIN
    if widgetdata.default_draw_on_step then $
      WIDGET_CONTROL, widgetdata.spb_step, SET_BUTTON=1 else $
      WIDGET_CONTROL, widgetdata.spb_step, SET_BUTTON=0
  ENDIF

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Input, DISABLE=DISABLE, ENABLE=ENABLE
;*
;* DESCRIPTION:
;*      Enable or disable further input to the papco-mainwindow.
;*
;* INPUTS:
;*  none
;*
;* KEYWORDS:
;*  DISABLE     if set, all widgets are made insensitive
;*  ENABLE      if set, all widgets are made sensitive
;*
;* CALLING SEQUENCE:
;*  to disable input:
;*  papco_Input, /DISABLE
;*
;*  to enable input:
;*  papco_Input, /ENABLE
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_Input, DISABLE=DISABLE, ENABLE=ENABLE

 COMMON PLOT_COMPOSER, widgetdata

   IF widgetdata.default_input_enable THEN return

   IF KEYWORD_SET(DISABLE) THEN setEnable=0 ELSE setEnable=1
   IF setEnable THEN WIDGET_CONTROL, widgetdata.main_base, HOURGLASS=1 $
   ELSE  WIDGET_CONTROL, widgetdata.main_base, HOURGLASS=0

   IF (widgetdata.minTime.mjd NE widgetdata.maxTime.mjd) OR $
      (widgetdata.minTime.t NE widgetdata.maxTime.t)  THEN BEGIN
      WIDGET_CONTROL, widgetdata.sl_startTime, SENSITIVE=setEnable
      WIDGET_CONTROL, widgetdata.sl_endTime, SENSITIVE=setEnable
      WIDGET_CONTROL, widgetdata.ef_endTime, SENSITIVE=setEnable
      WIDGET_CONTROL, widgetdata.ef_startTime, SENSITIVE=setEnable
   ENDIF ELSE BEGIN
      WIDGET_CONTROL, widgetdata.sl_startTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.sl_endTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.ef_endTime, SENSITIVE=0
      WIDGET_CONTROL, widgetdata.ef_startTime, SENSITIVE=0
   ENDELSE

   WIDGET_CONTROL, widgetdata.ef_orbit, SENSITIVE=setEnable
   WIDGET_CONTROL, widgetdata.lb_plotInfos, SENSITIVE=setEnable
   WIDGET_CONTROL, widgetdata.dw_plotInfos, SENSITIVE=setEnable
   WIDGET_CONTROL, widgetdata.pb_print, SENSITIVE=setEnable
   WIDGET_CONTROL, widgetdata.main_base, HOURGLASS=0

   FOR i=0, N_ELEMENTS(widgetdata.wid_toEnable)-1 DO $
      WIDGET_CONTROL, widgetdata.wid_toEnable(i), SENSITIVE=setEnable

   setEnableEditButton=setEnable
   IF KEYWORD_SET(enable) then begin
      sPlot=widgetdata.selectedPlot
      if sPlot LT 0 then $
        setEnableEditButton=0 $
      else $
        widget_control, widgetdata.lb_plotInfos, set_list_select=sPlot*2
   endif
   FOR i=0, N_ELEMENTS(widgetdata.edit_buttons) - 1 DO $
      WIDGET_CONTROL, widgetdata.edit_buttons(i), SENSITIVE=setEnableEditButton

   draw_win_name='papco_draw_window_' + $
                  string(widgetdata.no_draw_window,format="(i2.2)")
   IF not XREGISTERED(draw_win_name) THEN begin
     WIDGET_CONTROL, widgetdata.pb_draw_new, SENSITIVE=0
     WIDGET_CONTROL, widgetdata.pb_annotate, SENSITIVE=0
     WIDGET_CONTROL, widgetdata.pb_rm_annot, SENSITIVE=0
   endif

end

;******************************************************************************
;* CALL (papco): papco.pro
;*
;* DESCRIPTION:
;*   The main - program. Initialize the widgets-hierarchy, set the
;*   COMMON PLOT_COMPOSER, widgetdata, read the defaults, call XManager
;*
;* INPUTS:           none
;*
;* KEYWORDS:
;*   GROUP       You may specify a GROUP_LEADER here.
;*   PRODUCT     If set, papco is loaded with this Product at startup
;*   BATCH       If set, also needs a valid PRODUCT name. papco is
;*               started, the PRODUCT is loaded, and then the BATCH
;*               is executed and produces postcript output files.
;*   OUTPUT      If set, and batch is set, data is output to file as well.
;*   DRAW        if set and PRODUCT set that product is drawn on startup
;*   SET_ONLY    sets up structures and common blocks only. Used for working
;*               in the "PAPCO environment"  without starting the widget.
;*   _EXTRA      collects all other keywors in a structure. Used to
;*               overwrite defaults in widgetdata
;*   INFO        prints out information for this routine
;*
;* CALLING SEQUENCE:
;*   papco, [optional keywords]
;*   To run papco, you must execute @papco_setup or @papco_startup first !
;*
;* MODIFICATION HISTORY:
;*   written august 1995, Andreas Keese
;*   november 1995, modified by R.Friedel
;*   september 1997, modified by Torsten Heise
;*   February 1998, modified by R.Friedel
;*   January 2001, modified by R.Friedel - updated command line features
;*****************************************************************************
PRO papco, GROUP=Group, INFO=info, PRODUCT=product, BATCH=batch, DRAW=draw, $
           SET_ONLY=set_only, OUTPUT=output, VERBOSE=verbose, $
           _EXTRA=cmdl_defaults
;;
COMMON plot_composer, widgetdata
COMMON mjdt, mjdt_start, mjdt_end
COMMON PRODUCT_HISTORY, current, last
COMMON MISSIONS, missions     ;contains all configured mission names
COMMON ADDINS, plot_types     ;contains lables of all plot types
COMMON ADDBTN, addbuttons     ;contains labels of all add-in buttons
COMMON ADDSUB, sorted_modules ;sorted module names and no of submenu entries
COMMON MODULE_PATHS, paths    ;contains the paths found by papco_find_modules
COMMON GET_ERROR, get_err_no, get_err_msg


if ( n_elements( widgetData ) ne 0 ) then begin
	if ( widget_info( widgetData.base, /valid ) ) then begin
	   message, 'There appears to be another valid papco widget running.  Close this widget before restarting', /cont
	   return
	endif
endif

;initialize papco error common block
IF keyword_set(VERBOSE) THEN default_verbose = 1 ELSE default_verbose = 0
get_err_no = 0  &  get_err_msg = 'No Errors'

outstr1=';***************************************'
outstr2='***************************************'
comment_stars_line=outstr1+outstr2
papco_prt,''

; print info on routine if needed
IF keyword_set(INFO) THEN BEGIN
    papco_routine_info,'papco','papco', ROOT_DIR=getenv('PAPCO_PATH')
    papco_show_widgetdata
    return
ENDIF

;-- set the maximum number of panels allowed in PAPCO -------------------------
max_no_of_panels=60

;-- initialize the product history --------------------------------------------
current=0  &  last=0

;--- some general initialization ----------------------------------------------
if not keyword_set(SET_ONLY) then CLOSE, /ALL else $
    if SET_ONLY le 1 then if n_elements(widgetdata) ne 0 then return
papco_set_plot, 'x'
if keyword_set(Group) then group=GROUP else GROUP=0
junk = { CW_PDMENU_S, flags:0, name:'' }

;--- Get list of modules papcoadd files, sorted alphabetically! ---------------
fn=n_elements(paths)
module_names=papco_get_module_names(paths)
index=sort(module_names)
sorted_paths=paths(index)
dummy={name:'',n_submenu:0}
sorted_modules=replicate(dummy,fn)
sorted_modules.name=module_names(index)

if (fn eq 1) AND (paths(0) eq '') then BEGIN
    message,'No modules found in !path'
    fn=0
endif else BEGIN
    addinfiles=strarr(fn)
    for i = 0, fn-1 do begin
      fname=papco_addpath(sorted_paths(i),'papco_interface')
      fname=fname+'papcoadd*.pro'
      f= findfile( fname,count=count )
      if count gt 1 then begin
        message, 'found multiple papcoadd routines:', /info
        for ii=1,count do message, f(ii-1), /info
        stop
      endif
      addinfiles(i)=findfile(fname)
    endfor
ENDELSE

;!!!!NEW PAPCO 10.5: This is now also done during startup compile of modules.
;--- read module defaults -----------------------------------------------------
IF fn ne 0 then BEGIN
    FOR i=0,fn-1 DO BEGIN
      module=strplottype(strlowcase(addinfiles(i)))
      r_module_config, module, config_data, msg        ;get user defaults first
      if get_err_no eq 0 then begin
        for ii=0,n_elements(config_data)-1 do $
          papco_setenv,/FS,config_data(ii).env_var+'='+config_data(ii).path
      endif else begin
        msg='No user module config for: '+ module
      endelse
      message, msg, /info
    ENDFOR
END

;--- define MOUSE_INFORMATION -------------------------------------------------
COMMON MOUSE_INFORMATION, mouseData
papco_Cursor_SetMouseDataStruct
mouseData.CurrentPosition=[0,0]
mouseData.LastPosition=[0,0]
mouseData.mouseActionTyp=1

; -----------------------------------------------------------------------------
; ---- MAIN BASE --------------------------------------------------------------
; -----------------------------------------------------------------------------

base = WIDGET_BASE(GROUP_LEADER=Group, COLUMN=1, /MAP, $
                   TITLE=papco_Version(),  UVALUE='main_base')

DEFAULT_FONT={id:'DEFAULT_FONT',struct:{DEFAULT_FONT:''}}
papco_readDefaultSettings, TAGID=DEFAULT_FONT
FONT_ONLY=DEFAULT_FONT.struct.DEFAULT_FONT

DEFAULT_TRAP={id:'DEFAULT_TRAP',struct:{DEFAULT_TRAP:0}}
papco_readDefaultSettings, TAGID=DEFAULT_TRAP
ERR_CATCH_ONLY=DEFAULT_TRAP.struct.DEFAULT_TRAP

WIDGET_CONTROL, base, default_font=FONT_ONLY

; -----------------------------------------------------------------------------
; ---- TOP BASE ---------------------------------------------------------------
; -----------------------------------------------------------------------------

top_base = WIDGET_BASE(base, COLUMN=1, /MAP)

;--- define plot types --------------------------------------------------------
; count number of buttons to compute number of rows: only 10 buttons a row
no_buttons=0
IF fn ne 0 then begin
    plot_types=strarr(fn)
    FOR i=0,fn-1 DO no_buttons=no_buttons+1
ENDIF
no_row = (no_buttons+9)/10  &  no_buttons=0

;--- Create hidden base of one button for sizing ------------------------------
modules_out_base = WIDGET_BASE(top_base, column=1)
modules_top_base= WIDGET_BASE(modules_out_base)

hidden_base = WIDGET_BASE(modules_top_base, ROW=1,  map = 0)
;set up a button for debugging, user edit of plot programs
MenuDescStop = [{ CW_PDMENU_S, 3, strcenter('STOP PAPCO',12) }, $
                { CW_PDMENU_S, 2, 'hidden button' } ]

pdm_hidden = CW_PDMENU(hidden_base, MenuDescStop, UVALUE='pdm_stop_papco')

;--- Module Buttons -----------------------------------------------------------
modules_base = WIDGET_BASE(modules_top_base, ROW=no_row)

;set size for module buttons
geometry = WIDGET_INFO(pdm_hidden, /GEOMETRY)
buttonGeometry={ysize:geometry.ysize,     xsize:geometry.xsize,    $
                yoffset:geometry.yoffset, xoffset:geometry.xoffset }

; read modules directories for a list of plot type files. This gives the plot
; type names used to construct the add in buttons from procedures contained
; in the modules directory in the corresponding papcoadd_plottype.pro files.
; Some add-in buttons themselves call a submenu of a range of plot types. These
; plotypes are read in from the top of the files and added to the array
; plot_types.

;--- add buttons for each module ----------------------------------------------
IF fn ne 0 then BEGIN
    plot_types=strarr(fn)
    pb_plottype_arr=intarr(fn)
    FOR i=0,fn-1 DO BEGIN
        plot_types(i)=strplottype(strlowcase(addinfiles(i)))
        proc_call='addplottype_'+plot_types(i)
        ; trap errors on call to user defined interface routine.
        IF ERR_CATCH_ONLY eq 1 then catch, error_status else error_status=0

        IF error_status ne 0 then begin ;call error handler
            papco_error_handler, error_status, proc_call, plot_types(i), $
              panelVector,  /NO_ERR_DRAW
            error_status=0 &  catch, /CANCEL          ;disable error catching
            CONTINUE
        ENDIF

        CALL_PROCEDURE, proc_call, modules_base, buttonGeometry, pb_plottype
        catch, /CANCEL                     ;disable error catching
        pb_plottype_arr(i)=pb_plottype
        ;now resize the button to a standard size
        widget_control, pb_plottype, xsize = geometry.xsize
        no_buttons=no_buttons+1
    ENDFOR

    addbuttons=plot_types

;--- enquire modules if they contain submenus with more plot types ------------
    plot_types=strarr(1)
    papco_prt,outstr1+outstr2
    papco_prt,';* checking for sub plot_types'
    IF fn ne 0 then BEGIN
      FOR i=0,fn-1 DO BEGIN
        instring=';*********' & n_submenu=0
        openr, unit, addinfiles(i), /get_lun
        readf,unit,instring & papco_prt,instring & sepstr=instring
        readf,unit,instring & papco_prt,instring
        readf,unit,instring & papco_prt,instring
        while instring ne ';* end' do begin
          n_submenu=n_submenu+1
          plot_types=[plot_types,strplottype(instring+'.pro')]
          readf,unit,instring & papco_prt,instring
        endwhile
        free_lun,unit
        sorted_modules(i).n_submenu=n_submenu
      ENDFOR
    END
    plot_types=plot_types(1:n_elements(plot_types)-1)
    papco_prt,sepstr
    papco_prt,';* All Add-in modules read'
    papco_prt,sepstr
ENDIF ELSE BEGIN
    papco_prt,';* No Modules Configured'
    papco_prt,outstr1+outstr2
    papco_prt,''
    pb_plottype_arr=[pdm_stop_papco]
    addbuttons=''
ENDELSE

;---add empty buttons to always make 10 bttons a row --------------------------
WHILE no_buttons LT no_row * 10 DO BEGIN
    pb_empty_button = WIDGET_BUTTON(modules_base,  $
                                    VALUE=strcenter('',12), $
                                    UVALUE='pb_empty_button', $
                                    _extra=buttonGeometry)
    no_buttons=no_buttons+1
ENDWHILE

; -----------------------------------------------------------------------------
; ---- MIDDLE BASE ------------------------------------------------------------
; -----------------------------------------------------------------------------

  middle_base = WIDGET_BASE(base,  ROW=1, /MAP)

; -----------------------------------------------------------------------------
; ---- LEFT SIDE MIDDLE -------------------------------------------------------
; -----------------------------------------------------------------------------

leftSide_base = WIDGET_BASE(middle_base, COLUMN=1, /MAP)

; ---- time choice / adjustment / step base -----------------------------------
time_select_base=WIDGET_BASE(leftSide_base,  COLUMN=1, /MAP, /FRAME)

; -- Time Range choice widgets  ---------------------------------------------
time_entry_base=WIDGET_BASE(time_select_base, COLUMN=1)
time_entry_label=WIDGET_LABEL(time_entry_base, /ALIGN_CENTER, $
                              VALUE='Enter timerange (then hit return)')

  ;set up mappable bases ---
  time_method_outer_base=WIDGET_BASE(time_entry_base,  ROW=1)
  time_method_base=WIDGET_BASE(time_method_outer_base)
  orbitSelect_base=WIDGET_BASE(time_method_base, $
                               COLUMN=1, /ALIGN_CENTER)
  manual_time_Base=WIDGET_BASE(time_method_base, $
                               COLUMN=1, /ALIGN_CENTER, MAP=0)
  day_time_Base=WIDGET_BASE(time_method_base, $
                               COLUMN=1, /ALIGN_CENTER, MAP=0)
  batch_time_Base=WIDGET_BASE(time_method_base, $
                               COLUMN=1, /ALIGN_CENTER, MAP=0)

  ;By orbit entry base ---
  enter_orbit_base = WIDGET_BASE(orbitSelect_base, ROW=1, /ALIGN_CENTER)
  path=papco_getenv('PAPCO_PATH')      ;read a list of "createorbitinfo" files
  files=findfile(path+'*createorbitinfo.pro',count=n)
  missions=strarr(n)  &  l1=strlen(path)
  for i=0,n-1 do $
    missions(i)=strupcase(strmid(files(i),l1,strlen(files(i))-20-l1))
  label_satellite = WIDGET_LABEL(enter_orbit_base, $
                                 VALUE=strcenter(missions(0),12))
  ef_orbit = CW_FIELD(enter_orbit_base, xsize=6, ysize=1, $
                      /INTEGER, return_events=1, TITLE='Orbit:',$
                      uvalue='ef_orbit')

  choose_sat_orb_base=WIDGET_BASE(orbitSelect_base, ROW=1, /ALIGN_CENTER)
  Satellite_list = [ { CW_PDMENU_S, 3, strcenter('Satellite',10) } ]
  for i=0,n-1 do $
    Satellite_list=[Satellite_list, { CW_PDMENU_S, 0, missions(i) } ]
  Satellite_list(n).(0)=2

  pdm_satellite=CW_PDMENU(choose_sat_orb_base, Satellite_list, $
                          UVALUE='pdm_satellite')
  pb_orbitSelect = WIDGET_BUTTON(choose_sat_orb_base, $
                                 UVALUE='pb_orbitSelect', $
                                 VALUE='List Orbits')

  ;Manual entry base ---
  ef_minTime=CW_FIELD(manual_time_base, ysize=1, string=1, return_events=1, $
                      uvalue='ef_minTime', title='from:')
  ef_maxTime=CW_FIELD(manual_time_base, ysize=1, string=1,return_events=1, $
                      uvalue='ef_maxTime', title='  to:')

  ;By day entry base---
  ef_DayTime=CW_FIELD(day_time_base, xsize=10, ysize=1, $
                      string=1, return_events=1, $
                      uvalue='ef_DayTime', title='Select Day:')

  ;By batch entry base---
  label_batch_base=WIDGET_BASE(batch_time_base,COLUMN=1)
  label_batch_name=WIDGET_LABEL(label_batch_base, $
    VALUE=strcenter('no batch file chosen',39))
  label_batch_time=WIDGET_LABEL(label_batch_base, $
    VALUE=strcenter('no time read',39))
  batch_action=WIDGET_BASE(batch_time_base,ROW=1)
  pb_read_batch=WIDGET_BUTTON(batch_action,$
                              UVALUE='pb_read_batch', $
                              VALUE='Select File')
  pb_next_batch=WIDGET_BUTTON(batch_action,$
                              UVALUE='pb_next_batch', $
                              VALUE=' Next ')
  pb_prev_batch=WIDGET_BUTTON(batch_action,$
                              UVALUE='pb_prev_batch', $
                              VALUE=' Prev ')
  pb_pick_batch=WIDGET_BUTTON(batch_action,$
                              UVALUE='pb_pick_batch', $
                              VALUE=' Pick ')

  ; -- Time choice method -----------------------------------------------------
  time_choice_base=WIDGET_BASE(time_method_outer_base, COLUMN=1)
  ; set papco time choice methods
  time_choice= [ { CW_PDMENU_S, 3, strcenter('Method:',8)}, $
                 { CW_PDMENU_S, 0, ' Manual     (start / end)       '}, $
                 { CW_PDMENU_S, 0, ' By Orbit   (Satellite orbit #) '}, $
                 { CW_PDMENU_S, 0, ' By Day     (00:00 -> 24:00)    '}, $
                 { CW_PDMENU_S, 2, ' From Batch (Times from file)   '}  ]

  pdm_time_choice_action=CW_PDMENU(time_choice_base, time_choice, $
                                   UVALUE='pdm_time_choice_action')
  text_time_choice_action=WIDGET_LABEL(time_choice_base, $
                                   VALUE=strcenter(' not  set ',12) )

  ; -- Add to Batch button ----------------------------------------------------
  pb_batch = WIDGET_BUTTON(time_choice_base,VALUE='--> Batch', $
                           UVALUE='pb_batch', /ALIGN_CENTER)

  ; -- time-select-sliders ----------------------------------------------------
  sl_startTime = WIDGET_SLIDER(time_select_base, /DRAG, /SUPPRESS, $
                               MAXIMUM=1, MINIMUM=0, $
                               UVALUE='sl_startTime', $
                               XSIZE=buttonGeometry.xsize*3.4, VALUE=0)

  timerange_label_base = WIDGET_BASE(time_select_base, /ROW, /MAP)
  ef_startTime = CW_FIELD(timerange_label_base, /RETURN_EVENTS, $
                        TITLE='From',  VALUE='00:00', UVALUE='ef_startTime')
  ef_endTime = CW_FIELD(timerange_label_base,  /RETURN_EVENTS, $
                        TITLE='to',  VALUE='00:00', UVALUE='ef_endTime')

  sl_endTime = WIDGET_SLIDER(time_select_base, /DRAG, /SUPPRESS, $
                             MAXIMUM=1,  MINIMUM=0, $
                             UVALUE='sl_endTime', $
                             XSIZE=buttonGeometry.xsize*3.4, VALUE=0)

  ; this empty base generates a timer-event that clears the drawn time-range
  clearTimeRange=WIDGET_BASE(time_select_base, UVALUE='clearTimeRange')

  ; -- Time Step Buttons ------------------------------------------------------
  steptime_base = WIDGET_BASE(time_select_base, /COLUMN, /MAP)
  dummy_base = WIDGET_BASE(steptime_base, /ROW)

  pb_previous = WIDGET_BUTTON(dummy_base, VALUE=' <-- ', UVALUE='pb_previous')
  pb_next = WIDGET_BUTTON(dummy_base,     VALUE=' --> ', UVALUE='pb_next')

  lb_steptime = WIDGET_LABEL(dummy_base,  VALUE='Step:')
  ef_steptime = WIDGET_TEXT(dummy_base,   VALUE='000 00:00:00', XSIZE=13)

  step_toggle_base=WIDGET_BASE(dummy_base, ROW=1, /NONEXCLUSIVE)
  spb_step=WIDGET_BUTTON(step_toggle_base, /FRAME,$
                         VALUE='Draw on step', UVALUE='spb_draw_on_step')

; --- Mouse-behaviour ---------------------------------------------------------
mouse_behaviour_Base=WIDGET_BASE(leftSide_base, ROW=1, /FRAME)
; The mouse actions zoom, slice, level zero ,Paneldata and haircursor are
; provided by papco. Here further user defined mouse actions can be added.
; User mouse functions are read from the env. var. USER_MOUSE_ACTIONS.
; Format for user added slice: 'user_slice_a/user_slice_b/user_slice_c'
; Seperate words by underscores, slices types by backslash.

; set papco mouse actions
mouseAction= [ { CW_PDMENU_S, 3, strcenter('Mouse Action:',13)}, $
               { CW_PDMENU_S, 0, ' papco Zoom              '}, $
               { CW_PDMENU_S, 0, ' papco Vertical Slice    '}, $
               { CW_PDMENU_S, 0, ' papco Access Level Zero '}, $
               { CW_PDMENU_S, 0, ' papco Write Paneldata   '}, $
               { CW_PDMENU_S, 0, ' papco Hair Cursor       '}, $
               { CW_PDMENU_S, 0, ' papco Horizontal Slice  '} ]

; check for env. variable for user mouse actions - expand mouseAction
papco_prt,''
u_actions=getenv('USER_MOUSE_ACTIONS')
if strlen(u_actions) eq 0 then message,'No user mouse actions', /info $
else begin
    u_actions=u_actions+'/ '
    u_actions=str_sep(u_actions,'/')
    for i=0,n_elements(u_actions)-2 do begin
      new_action=' USER '+strsubst(u_actions(i),'_',' ')+' '
      mouseAction = [mouseAction, { CW_PDMENU_S, 0, new_action } ]
      message,'Added user slice '+$
        strformat(new_action,30,/left )+' slice = '+varprt(2l^(5+i)), /info
    endfor
endelse
mouseAction=[mouseAction, { CW_PDMENU_S, 0, ' papco Multiple Slice ' } ]
mouseAction(n_elements(mouseAction)-1).FLAGS=2
papco_prt,''

pdm_mouse_action=CW_PDMENU(mouse_behaviour_Base, mouseAction, $
                           UVALUE='pdm_mouse_action')
text_mouse_action=WIDGET_LABEL(mouse_behaviour_Base, $
                               VALUE=strcenter('   not set yet   ',35) )

; --- Slice behaviour ---------------------------------------------------------
slice_behaviour_Base=WIDGET_BASE(leftSide_base, ROW=1, /FRAME)

; set papco slice actions
sliceAction= [ { CW_PDMENU_S, 3, strcenter('Slice Action:',13)}, $
               { CW_PDMENU_S, 0, ' single panel         '}, $
               { CW_PDMENU_S, 0, ' global per plot_type '}, $
               { CW_PDMENU_S, 2, ' global all panels    '}  ]

pdm_slice_action=CW_PDMENU(slice_behaviour_Base, sliceAction, $
                                 UVALUE='pdm_slice_action')
text_slice_action=WIDGET_LABEL(slice_behaviour_Base, $
                                 VALUE=strcenter('   not set yet   ',35) )

; --- Papco_Prt behaviour------------------------------------------------------
print_behaviour_Base=WIDGET_BASE(leftSide_base, ROW=1, /FRAME)

  ; set papco print actions
  printAction= [ { CW_PDMENU_S, 3, strcenter('Print Action:',13)}, $
                 { CW_PDMENU_S, 0, ' -> default printer       '}, $
                 { CW_PDMENU_S, 0, ' -> default file          '}, $
                 { CW_PDMENU_S, 0, ' -> select printer / file '}, $
                 { CW_PDMENU_S, 0, ' select default output    '}, $
                 { CW_PDMENU_S, 0, ' ------------------------ '}, $
                 { CW_PDMENU_S, 0, ' batch -> screen preview  '}, $
                 { CW_PDMENU_S, 0, ' batch -> data output     '}, $
                 { CW_PDMENU_S, 2, ' batch -> default file    '}  ]

  pdm_print_action=CW_PDMENU(print_behaviour_Base, printAction, $
                                                   UVALUE='pdm_print_action')
  text_print_action=WIDGET_LABEL(print_behaviour_Base, $
                                 VALUE=strcenter('   not set yet   ',35) )

; --- Edit behaviour ---------------------------------------------------------
  edit_behaviour_Base=WIDGET_BASE(leftSide_base, ROW=1, /FRAME)

  ; set papco edit actions
  editAction= [ { CW_PDMENU_S, 3, strcenter('Edit Action:',13)}, $
                { CW_PDMENU_S, 0, ' local                '}, $
                { CW_PDMENU_S, 0, ' global per plot_type '}, $
                { CW_PDMENU_S, 2, ' global all           '}  ]

  pdm_edit_action=CW_PDMENU(edit_behaviour_Base, editAction, $
                                                   UVALUE='pdm_edit_action')
  text_edit_action=WIDGET_LABEL(edit_behaviour_Base, $
                                 VALUE=strcenter('   not set yet   ',35) )

; -----------------------------------------------------------------------------
; ---- RIGHT SIDE MIDDLE ------------------------------------------------------
; -----------------------------------------------------------------------------

geometry=WIDGET_INFO(leftSide_base, /geometry)
rightSide_base = WIDGET_BASE(middle_base, COLUMN=1, /MAP, $
                             ysize=geometry.YSIZE+3)

; --- Panel Editor ------------------------------------------------------------
PanelEdit_base = WIDGET_BASE(rightSide_Base,  ROW=1, /MAP, $
                             YSIZE=geometry.YSIZE)

; --- Panel list and Draw schematic ------------------------------------------
xsize=540

;work out size of panel list and draw window needed for max no of panels
lines_in_list=max_no_of_panels*2
dummy=strarr(lines_in_list) & dummy(1)='moment...'
ysize=lines_in_list*16+8

plotList_Base=widget_base(PanelEdit_Base, ROW=1, $
                          scr_xsize=xsize, /scroll)

dw_plotInfos = WIDGET_DRAW(plotList_base, xsize=125)
list_base = widget_base(plotList_base)

lb_plotInfos = WIDGET_LIST(list_base, VALUE = dummy, $
                           YSIZE=lines_in_list,  XSIZE=80, $
                           UVALUE='lb_plotInfos')

geom=WIDGET_INFO(lb_plotInfos, /geometry)
WIDGET_CONTROL, dw_plotInfos, YSIZE=geom.scr_ysize+2*geom.margin

; --- Panel Edit buttons ------------------------------------------------------
infoButtons_base = WIDGET_BASE(PanelEdit_base, column=1, MAP=1)

bitmapButtons_base=WIDGET_BASE(infoButtons_base, COLUMN=1)

; --- Position UP/DOWN buttons ------------------------------------------------
  updownbuttons_base=WIDGET_BASE(bitmapButtons_base, COLUMN=1, /FRAME)
  dummy = WIDGET_LABEL(updownbuttons_base, VALUE='Position')
  upButton_base=WIDGET_BASE(updownbuttons_base, ROW=1)
  BMPUP = get_new_button(/BMPUP)
  pb_MoveUp = WIDGET_BUTTON(upButton_base, VALUE=BMPUP, $
                            UVALUE='pb_MoveUp')
  dummy = WIDGET_LABEL(upButton_base, VALUE='  Up  ')
  downButton_base=WIDGET_BASE(updownbuttons_base, ROW=1)
  BMPDOWN = get_new_button(/BMPDOWN)
  pb_MoveDown = WIDGET_BUTTON(downButton_base, VALUE=BMPDOWN, $
                              UVALUE='pb_MoveDown')
  dummy = WIDGET_LABEL(downButton_base, VALUE=' Down ')

; --- absolute size INCREAS/DECREASE buttons ----------------------------------
  enlargeShrinkButtons_Base= WIDGET_BASE(bitmapButtons_base, COLUMN=1, /FRAME)
  dummy = WIDGET_LABEL(enlargeShrinkButtons_Base, VALUE='Absolute size')
  enlargeButton_Base= WIDGET_BASE(enlargeShrinkButtons_Base, ROW=1)
  BMP_ENLARGE = get_new_button(/BMP_ENLARGE)
  pb_Enlarge = WIDGET_BUTTON(enlargeButton_Base, VALUE=BMP_ENLARGE, $
                             UVALUE='pb_Enlarge')
  dummy = WIDGET_LABEL(enlargeButton_Base, VALUE='Increase')
  shrinkButton_Base= WIDGET_BASE(enlargeShrinkButtons_Base, ROW=1)
  BMP_SHRINK = get_new_button(/BMP_SHRINK)
  pb_Shrink = WIDGET_BUTTON(shrinkButton_Base,VALUE=BMP_SHRINK, $
                            UVALUE='pb_Shrink')
  dummy = WIDGET_LABEL(shrinkButton_Base, VALUE='Decrease')

; --- relative size INCREAS/DECREASE buttons ----------------------------------
  enlShrOthers_Base= WIDGET_BASE(bitmapButtons_base, COLUMN=1, /FRAME)
  dummy = WIDGET_LABEL(enlShrOthers_Base, VALUE='Relative size')
  enlOthers_Base= WIDGET_BASE(enlShrOthers_Base, ROW=1)
  BMP_OTHERS_ENLARGE = get_new_button(/BMP_OTHERS_ENLARGE)
  pb_ShrinkOthers = WIDGET_BUTTON(enlOthers_base,$
                                   VALUE=BMP_OTHERS_ENLARGE, $
                                   UVALUE='pb_ShrinkOthers')
  dummy = WIDGET_LABEL(enlOthers_Base, VALUE='Increase')
  shrOthers_Base= WIDGET_BASE(enlShrOthers_Base, ROW=1)
  BMP_OTHERS_SHRINK = get_new_button(/BMP_OTHERS_SHRINK)
  pb_EnlargeOthers= WIDGET_BUTTON(shrOthers_base,$
                                  VALUE=BMP_OTHERS_SHRINK, $
                                  UVALUE='pb_EnlargeOthers')
  dummy = WIDGET_LABEL(shrOthers_base, VALUE='Decrease')

;--- edit button base ---------------------------------------------------------
  editButtons_base = WIDGET_BASE(infoButtons_base, COLUMN=1, MAP=1, /FRAME)
  EditMenu = [ { CW_PDMENU_S, 3, strcenter('Edit',6) }, $
               { CW_PDMENU_S, 0,       'Edit panel  ' }, $
               { CW_PDMENU_S, 0,       'Change Type' }, $
               { CW_PDMENU_S, 0,       'Delete panel'}, $
               { CW_PDMENU_S, 2,       'Copy panel' }   ]
  pb_EditPlot = CW_PDMENU(editButtons_base, EditMenu, $
                          UVALUE='pb_EditPlot')

  ReformatMenu = [ { CW_PDMENU_S, 3, strcenter('Reformat',6) }, $
                   { CW_PDMENU_S, 0,       'remove gaps'     }, $
                   { CW_PDMENU_S, 0,       'reset all'       }, $
                   { CW_PDMENU_S, 0,       'resort all'      }, $
                   { CW_PDMENU_S, 2,       'undo overlaps'   }  ]
  pb_Reformat = CW_PDMENU(editButtons_base, ReformatMenu, UVALUE='pb_Reformat')

  dummy=WIDGET_BASE(editButtons_base, /NONEXCLUSIVE, /COLUMN)
  pb_overplot=WIDGET_BUTTON(dummy, VALUE=strcenter('overplot', 8), $
                            UVALUE='pb_overplot')
  pb_anchored=WIDGET_BUTTON(dummy, VALUE=strcenter('anchored', 8), $
                            UVALUE='pb_anchored')
; -----------------------------------------------------------------------------
; ---- BOTTOM PART ------------------------------------------------------------
; -----------------------------------------------------------------------------

bottom_base = WIDGET_BASE(base, COLUMN=1, /MAP)

;--- print settings output labels ---------------------------------------------

print_set_base = WIDGET_BASE(bottom_base, ROW=1, /MAP, /FRAME)

printer_info     = WIDGET_LABEL(print_set_base, $
                          VALUE=strformat('Printer/file: ',15,/LEFT) )
set_printer_file = WIDGET_LABEL(print_set_base, $
                          VALUE=strcenter('Printer / file not set yet',125) )

;--- set action buttons -------------------------------------------------------
action_base = WIDGET_BASE(bottom_base, COLUMN=10, /MAP)

; empty button for expansion
;  pb_empty_button = WIDGET_BUTTON(action_base,  $
;                        VALUE=strcenter('',10), $
;                        UVALUE='pb_empty_button', _extra=buttonGeometry)

; button 1 ---
pb_draw = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Draw', 10), $
                        UVALUE='pb_draw', _extra=buttonGeometry)
; button 2 ---
pb_draw_new = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Draw New', 10), $
                        UVALUE='pb_draw_new', _extra=buttonGeometry)
WIDGET_CONTROL, pb_draw_new, sensitive=0

; button 3 ---
pb_print=WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Print', 10), $
                        UVALUE='pb_print', _extra=buttonGeometry)
; button 4 ---
pb_output = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('Output',10), $
                        UVALUE='pb_output', _extra=buttonGeometry)
; button 5 ---
pb_annotate = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Annotate', 10), $
                        UVALUE='pb_annotate', _extra=buttonGeometry)
WIDGET_CONTROL, pb_annotate, sensitive=0

; button 6 ---
pb_rm_annot = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Rm. Annot.', 10), $
                        UVALUE='pb_rm_annot', _extra=buttonGeometry)
WIDGET_CONTROL, pb_rm_annot, sensitive=0

; button 7 ---
pb_forward = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('Forward',10), $
                        UVALUE='pb_forward', _extra=buttonGeometry)
; button 8 ---
pb_back = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('Back',10), $
                        UVALUE='pb_back', _extra=buttonGeometry)
; button 9 ---
 pb_reset = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Reset', 10), $
                        UVALUE='pb_reset', _extra=buttonGeometry)
; button 10 ---
pb_new_config = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('New Config',10), $
                        UVALUE='pb_new_config', _extra=buttonGeometry)
; button 11 ---
  pb_default = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Defaults', 10), $
                        UVALUE='pb_default', _extra=buttonGeometry)
; button 12 ---
pb_font = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Font', 10), $
                        UVALUE='pb_font',  _extra=buttonGeometry)
; button 13 ---
pb_help = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Help', 10), $
                        UVALUE='pb_help', _extra=buttonGeometry)
; button 14 ---
pb_about = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('About...', 10), $
                        UVALUE='pb_about', _extra=buttonGeometry)
; button 15 ---
pb_edit_batch = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('Edit Batch',10), $
                        UVALUE='pb_edit_batch', _extra=buttonGeometry)
; button 16 ---
pb_edit_color = WIDGET_BUTTON(action_base,  $
                        VALUE=strcenter('Edit Color',10), $
                        UVALUE='pb_edit_color', _extra=buttonGeometry)
; button 17 ---
pb_load = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Load', 10), $
                        UVALUE='pb_load', _extra=buttonGeometry)
; button 18 ---
pb_save = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Save', 10), $
                        UVALUE='pb_save', _extra=buttonGeometry)
; button 19 ---
pb_done_save = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Save&Done', 10), $
                        UVALUE='pb_done_save',  _extra=buttonGeometry)
; button 20 ---
pb_done = WIDGET_BUTTON(action_base, $
                        VALUE=strcenter('Done', 10), $
                        UVALUE='pb_done', _extra=buttonGeometry)

; -----------------------------------------------------------------------------
; --- DONE SETTING UP MAIN WIDGET ---------------------------------------------
; -----------------------------------------------------------------------------

; --- Set widgets that are made (in)sensitive in papco_Input when papco is busy
wid_toEnable=[base]
edit_Buttons=[updownbuttons_base, enlargeShrinkButtons_Base, $
              enlShrOthers_Base, editButtons_base]

; ---- Initialize Data --------------------------------------------------------

; an orbit is not yet entered - so, get an empty orbitInfo-structure
default_orbitInfo=papco_getOrbitInfoStruct()

; a printer is not yet defined - so, get the printer-structure
printerInfo=papco_getPrinterInfoStruct()
printerno = n_elements(printerInfo(0, *))

; create an array of empty plot-infos. hint : If you want to add more
; than 20 plots, simply increase the number for max_no_of_panels below.
plotInfoTemplate=papco_getPlotInfoStruct()
plotInfos=replicate(plotInfoTemplate, max_no_of_panels)
anchored=bytarr(max_no_of_panels)

; create the array for storing information about drawn plots
drawnplotInfoTemplate=papco_getDrawnPlotInfoStruct()
plotsDrawn = replicate(drawnplotInfoTemplate, N_ELEMENTS(plotInfos))
defaultdir = papco_getenv('PAPCO_HOME')  ;startup directory
homedir = papco_getenv('HOME')

;get screensize
screen=get_screen_size()

widgetdata= {PLOT_COMPOSER_11_0, $
    base:base,  $                             ;ID of top level widget
    composer_type: 'papco', $                 ;to switch papco/scratch pad
    quiet:0, $                                ;1 for quiet, batch operation
    saveFileName:'none', $
    saveStatePath:papco_addpath(defaultdir, 'papco_products'), $
    PrintName:'', $
    printerInfo:printerInfo, $
    batchFileName:'', $
    batchFilePath:papco_addpath(defaultdir, 'papco_products'+papco_dirsep()+'batch'), $
    batchseries:['','',''], $
    outputFilePath:homedir, $
    annotateFileName:'', $
    annotateFilePath:papco_addpath(defaultdir, 'papco_products'+papco_dirsep()+'annotate'), $
    fname:'', $
    mouse_doesSlice:1, $
    mouse_clicked:0, $
    mouse_clicked_position:[0.0, 0.0], $
    mouse_position:[0.0, 0.0], $
    mouse_selectionDrawn:0, $
    mouse_panelNr:-1, $
    panel_editor_size:lonarr(n_elements(plot_types), 2), $
    plotInfos:plotInfos, $
    old_plotinfos:plotInfos, $
    plotsDrawn:plotsDrawn, $
    numberOfplotsDrawn:0, $
    timeRangeDrawn:0, $
    clearTimeRange:clearTimeRange, $          ;empty widget-base
    startTime:{papco_time, mjd:47892l, T:0.0, T90:0l}, $
    endTime:{papco_time, mjd:47893l, T:0.0, T90:86400l}, $
    minTime:{papco_time, mjd:47892l, T:0.0, T90:0l}, $
    maxTime:{papco_time, mjd:47893l, T:0.0, T90:86400l}, $
    satellite_name:'', $
    numberOfPlots:0, $
    selectedPlot:-1, $
    main_base:base, $
    orbitSelect_base:orbitSelect_base, $
    manual_time_Base:manual_time_Base, $
    day_time_Base:day_time_Base, $
    batch_time_Base:batch_time_Base, $
    batch_count:0, $
    label_batch_name:label_batch_name, $
    label_batch_time:label_batch_time, $
    needto_read_data:1, $                     ;set if data needs re-reading
    pb_overplot:pb_overplot, $
    pb_anchored:pb_anchored, $
    anchored:anchored, $
    ef_startTime:ef_startTime, $
    sl_startTime:sl_startTime, $
    ef_endTime:ef_endTime, $
    sl_endTime:sl_endTime, $
    pb_previous:pb_previous, $
    pb_next:pb_next, $
    ef_stepTime:ef_stepTime, $
    ef_orbit:ef_orbit, $
    pdm_satellite:pdm_satellite, $
    pb_orbitSelect:pb_orbitSelect, $
    label_satellite:label_satellite, $
    ef_minTime:ef_minTime, $
    ef_maxTime:ef_maxTime, $
    ef_DayTime:ef_DayTime, $
    lb_plotInfos:lb_plotInfos, $
    dw_draw:0l, $
    dw_plotInfos:dw_plotInfos, $
    spb_step:spb_step, $
    time_choice:time_choice, $
    time_choice_action:0, $
    text_time_choice_action:text_time_choice_action, $
    pdm_mouse_action:pdm_mouse_action, $
    text_mouse_action:text_mouse_action, $
    mouseAction:mouseAction, $
    text_print_action:text_print_action, $
    text_edit_action:text_edit_action, $
    text_slice_action:text_slice_action, $
    printAction:printAction, $
    editAction:editAction, $
    sliceAction:sliceAction, $
    set_printer_file:set_printer_file, $
    edit_choice:1, $
    slice_choice:1, $
    pb_print:pb_print, $
    wid_toEnable:wid_toEnable, $
    edit_buttons:edit_buttons, $
    draw_window:0l, $
    no_draw_window:0l, $
    need_to_draw:1, $                       ;set if redraw needed
    output:0, $                             ;set to output data to file
    old_draw_id:lonarr(100), $
    pb_draw_new:pb_draw_new, $
    pb_output:pb_output, $
    pb_annotate:pb_annotate, $
    pb_rm_annot:pb_rm_annot, $
    current_plotNr:0, $
    panelset:[0, 0, 0], $                   ;contains current panel vector
    screensize:screen, $                    ;contains size of screen, pixels
    truecolor:0, $                          ; 0=8bit, 1,2,3=true color
    unilib:0, $                             ;set to 1 if unilib available
    ;--------------------------------------------------------------------------
    ;all tag names containing "default" will be saved in defaults.dat
    default_redraw:1, $            ;enable redraw logic
    default_vert_slice_size: [500,500], $ ;vertical slice draw window size
    default_ask_wget:1,  $         ;ask confirmation before wget
    default_scroll:1,  $           ;disable scrolling inside papco windows
    default_save_bitmap:1,  $      ;enable saving of bitmap
    default_input_enable:0, $      ;allow disable of widgets
    default_do_plots:1, $          ;do plots or just do read of data
    default_printerno:printerno, $ ;actual number of printers loaded
    default_states:0, $            ;0 turns on papco_state files, 1 turns off
    default_png_format:0, $
    default_right_label:0, $
    default_slice_reload:1, $
    default_multiple_slice:intarr(16), $
    default_draw_on_step:0, $     ;draw on time step or not
    default_adjust_time:0.0, $              ;hours to add to ends of time range
    default_numbering:0, $
    default_satellite:0, $
    default_trap:1, $                       ;set to trap user routine errors
    default_wget:0, $                       ;set to enable remote get
    default_print_choice:1, $
    default_useprinter:printerInfo.useprinter, $
    default_printer:printerInfo.printer, $
    default_askforfilename:printerInfo.askforfilename, $
    default_filetype:printerInfo.filetype, $
    default_filename_tmpl:printerInfo.filename_tmpl, $
    default_orbitInfo:default_orbitInfo, $
    default_layout:0, $
    default_verbose:0, $ ;sets verbose flag in get_error common flag for diag
    default_error_box:1, $
    default_font:FONT_ONLY, $
    default_charfont:3, $
    default_charsize:1.0, $
    default_charthick:1.0, $
    default_output_format:'TXT', $
    default_us_europe:1, $                  ;used to set US/Europe defaults
    default_plot_size:[100.0,100.0],$       ;default plot scaling of full size
    default_gif_size:[570,760], $           ;default output gif size
    default_draw_size:[570,760], $          ;default draw window size
    default_ps_mult:1.0, $                  ;scaling fonts to ps factor
    default_ps_eps:0, $                     ;encapsulated postscript = 1
    default_draw_ident:1, $
    default_bckg_color:200, $
    default_forg_color:0, $
    default_TICKLEN:-0.015, $
    default_XTICKS:6, $
    default_XLINES:2, $
    default_XSTYLE:1, $
    default_XMINOR:10     }

;--- Load and test papco color tables - ---------------------------------------
test_colors=1
papco_SetColorTable, TO_FEW_COLORS=test_colors
if test_colors ne 0 then return

; the default-settings (printer, paths, ...) are read
papco_readDefaultSettings
widgetdata.PrintName=papco_checkdirname(widgetdata.outputFilePath) + $
    widgetdata.default_fileName_tmpl

; ---- Start the widget -------------------------------------------------------
IF KEYWORD_SET(SET_ONLY) THEN BEGIN
    ;-- set up common start and end time for MJDT users
    mjdt_start=widgetdata.startTime
    mjdt_end=widgetdata.endTime
    ;--- set up common start and end time for old T90 users
    xut1=widgetdata.startTime.t90
    xut2=widgetdata.endTime.t90
    papco_finish &  return
ENDIF

; set position of widget (top left of screen) and realize. Don't map for batch
IF keyword_set(BATCH) THEN map=0 ELSE map=1
WIDGET_CONTROL, base, default_font=widgetdata.default_font, $
                      XOFFSET = 20, YOFFSET = 20, /REALIZE, MAP=map

;--- Parse the input line to calling papco ------------------------------------
; Here the command line driven version of papco is anabled. Input consists of
; a papco product filename and / or a papco batch filename. For both, path is
; taken from stored default settings if not part of filename).
; All default setting entered as keywords are used to overide the defaults.

;check for any default overides
IF keyword_set(cmdl_defaults) THEN BEGIN
    papco_prt,''
    cmdl_default_names=tag_names(cmdl_defaults)
    widgetdata_names=tag_names(widgetdata)
    ;update widgetdata tags, check for existance
    FOR i=0, n_elements(cmdl_default_names)-1 DO BEGIN
        result=strpos(widgetdata_names,cmdl_default_names(i))
        index=where(result ne -1, c)
        IF c NE 1 THEN BEGIN
            message, 'Keyword Tag '+cmdl_default_names(i)+' is unknown',/info
            goto, next_cmdl_default_name
        ENDIF ELSE begin
            message, 'Updating Keyword Tag '+cmdl_default_names(i),/info
            widgetdata.(index(0))=cmdl_defaults.(i)
        endelse
        next_cmdl_default_name:
    ENDFOR
    papco_prt,''
ENDIF

  ;update the printerinfo struct
widgetdata.printerinfo.useprinter=widgetdata.default_useprinter
widgetdata.printerinfo.printer=widgetdata.default_printer
widgetdata.printerinfo.askforfilename=widgetdata.default_askforfilename
widgetdata.printerinfo.filetype=widgetdata.default_filetype
widgetdata.printerinfo.filename_tmpl=widgetdata.default_filename_tmpl

IF keyword_set(VERBOSE) THEN widgetdata.default_verbose = VERBOSE

;check for product / batch / draw keywords
if KEYWORD_SET(PRODUCT) then begin
    papco_LoadState, FILENAME=product
    if KEYWORD_SET(BATCH) then begin
      message,'Batch processing with product file:',/info
      papco_prt,'    '+product
      papco_prt,'  and batch file:'
      papco_prt,'    '+batch
      ;command line batch can only output to file using a filename template
      widgetdata.printerinfo.USEPRINTER=0
      widgetdata.printerinfo.ASKFORFILENAME=0
      widgetdata.PrintName=widgetdata.OUTPUTFILEPATH + '/' +$
        widgetdata.DEFAULT_FILENAME_TMPL
      if KEYWORD_SET(OUTPUT) then widgetdata.output=1
      batch_fln=batch
      papco_Batch, batch_fln, PREVIEW=0, QUIET=1
      widgetdata.output=0
      message,'Batch processing finished',/info
      papco_main_Event, 0, USER_EVENT='pb_done', /NOASK
      papco_finish,  /terminate & return
    endif
    if KEYWORD_SET(DRAW) then begin
      message,'Autodrawing product',/info
      papco_refresh, /ALL
      papco_main_Event, 0, USER_EVENT='pb_draw'
    endif
endif else begin
    ;load up last state from crashed session if it exists.
    papco_load_last_state
    widgetdata.mintime=widgetdata.starttime
    widgetdata.maxtime=widgetdata.endtime
endelse

papco_refresh, /ALL
XMANAGER, 'papco_main', base, /NO_BLOCK

END
;******************************************************************************
;******************************************************************************

PRO papco_finish,  terminate = terminate

IF keyword_set(terminate) THEN GOTO,  papco_terminate

;--- clean up after the main widget has exited --------------------------------
papco_prt,''  &  message,'Cleaning up',/info  &  papco_prt,''
papco_cleanup
papco_terminate:
papco_prt,''  &  message,'Terminated normally',/info  &  papco_prt,''

END

;******************************************************************************
