;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_color.pro
;*
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:
;*    Contained are procedures for setting and maintaining the papco color
;*    subtables. Four subtables are provided for the simultaneous display
;*    of up to four different color tables each using a part of the single
;*    255 color table provided by IDL. The default papco color table resides
;*    at the bottom of the table.
;*
;*    Color Table use:
;*         0         background color (grey)
;*         1         drawing color, default line and annotation color (black)
;*         2         start of subtable 0 (c_index=0, offset=2+c_index*nc)
;*          .
;*          .         nc is no of colors in subtable
;*          .
;*    2+c_index*nc   start of subtable 1 (c_index=1)
;*          .
;*          .
;*          .
;*    2+c_index*nc   start of subtable 2 (c_index=2)
;*          .
;*          .
;*          .
;*    2+c_index*nc   start of subtable 3 (c_index=3)
;*          .
;*          .
;*          .
;*    2+c_index*4    start of fixed primary color table for linestyle colors
;*          .
;*          .          16 pre-defined primary colors
;*          .
;*    2+c_index*4+16 end of internal colortable - same as nc_total=!D.N_colors
;*
;*
;*    Common block description:
;*       COMMON papco_color, papco_color, r_papco, g_papco, b_papco $
;*          black,red,green,yellow,blue,magenta,cyan,white, $
;*          burgundy,olive,dark_green,teal,royal_blue,violet,dark_grey,grey
;*
;*          r_papco, g_papco, b_papco are the primary color arrays
;*          papco_color is an array of 3 element vectors, the first element
;*                      gives the sub_color table index (c_index). The
;*                      next element is used to manage to color tables:
;*          papco_color(*,0)= 0 color index  NOT used, 1 USED
;*          papco_color(*,1)= type of color bar. Currently used assignments:
;*                                               0 : UNIPOLAR COLOR
;*                                               1 : BIPOLAR COLOR
;*                                               2 : UNIPOLAR BLACK&WHITE
;*                                               3 : BIPOLAR BLACK&WHITE
;*          papco_color(*,2)= 0 color index  INactive, 1 ACTIVE
;*          black,red,green,yellow,blue,magenta,cyan,white,
;*          burgundy,olive,dark_green,teal,royal_blue,violet,dark_grey,grey
;*              color name variable containing its color index
;*
;*
;* FILES BELONGING TO THE MAIN-WINDOW:
;*
;*      see header of papco.pro
;*
;*    Those procedures are contained:
;*	 PRO papco_SetColorTable
;*	     -> loads the papco sub_table structure with defaults at startup
;*
;*	 PRO papco_Load_SubTable,c_index, r_user, g_user, b_user, TABLE=table
;*	     -> folds user or system color tables into subtable
;*
;*	 PRO papco_Set_SubTable,c_index
;*	     -> sets the active subtable
;*
;*	 PRO papco_Release_SubTable
;*	     -> sets all or one subtable to unused
;*
;*       PRO papco_Occupy_SubTable, c_index
;*	     -> sets one subtable to used
;*
;*	 FUNCTION papco_SubTable_used,c_index
;*	     -> queries is subtable used already
;*
;*       FUNCTION papco_get_Colorindices, c_index
;*	     -> returns the start and end index of the sub-colortable
;*
;*       FUNCTION papco_GetColor_table
;*	     -> returns the currently active color sub table index
;*
;*      FUNCTION papco_Get_zvalue
;*	     ->
;*
;*      FUNCTION papco_Get_subtable_index
;*	     ->
;*
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_SetColorTable
;*
;* DESCRIPTION:
;*	Set the default color-table for the plot-widget, widgetData.dw_draw
;*      papco internally supports a maximum of four simultaneous color tables.
;*      They are made by partitioning the existing single 255 color table.
;*      The X-windows system uses some of these colors, the remaining
;*      number is used by papco.
;*      The top 16 colors are hardwired to some nice primary colors for
;*      coloring line plots, the remainder subdivided into four tables.
;*      The papco default table (rainbow) is loaded into the bottom of the
;*      four subtables, and is the color table used unless the user
;*      specifies another color table.
;*      The remaining three slots are filled with further pre-defined
;*      color tables. It is these that the user may change and substitute
;*      with his own.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_SetColorTable
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_SetColorTable, TO_FEW_COLORS=TO_FEW_COLORS

  COMMON PLOT_COMPOSER, widgetData
  COMMON colors, r_orig,g_orig,b_orig,r_curr,g_curr,b_curr
  COMMON papco_color, papco_color, r_papco, g_papco, b_papco
  COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
                            white, burgundy, olive, dark_green, teal, $
                            royal_blue, violet, dark_grey, grey, $
                            foreground, background, color_bar_names, $
                            fixed_color_index, fixed_color_names

; set the color table range indices to be used. Allow for any number of
; colors! But need at least 100 for any meaningful subtables.
  nc_total=!D.N_colors     ;get total number of colors available
  nc_total=nc_total<256    ;for cases with 24 bit color
  nc_avail=nc_total-16-2   ;the top 16 color indices are reserved
                            ;the lower two colors are reserved (0=black for
                            ;axis and annotation; 1=200 for background)
  nc=nc_avail/4            ;divide by 4 to get no of colors for now.

; See if there are enough colors!!
  if nc_total lt 20 then begin
    print,'% papco_colors: Not enough colours available!'
    print,'                Available colors:' ,nc_total
    print,'                Exit color-hungry applications,'
    print,'                restart IDL and try again'
    if keyword_set(TO_FEW_COLORS) then begin
      TO_FEW_COLORS=99
      return
    endif
  endif else begin
    if keyword_set(TO_FEW_COLORS) then TO_FEW_COLORS=0
  endelse

; initialize common block papco_color
  papco_color=intarr(4,3)

; Set up arrays for the papco color table
  r_papco=bytarr(nc_total)
  g_papco=bytarr(nc_total)
  b_papco=bytarr(nc_total)

; Set up array for color bar names
  color_bar_names=strarr(4)

; Define color table file name
  tbl_path=papco_getenv('PAPCO_PATH')
  file_table=tbl_path+'papco.tbl'
; Get names of color tables in file
  color_table_names='none'
  LOADCT, file=file_table, GET_NAMES=color_table_names, 0

;-- FIRST LOAD THE MAIN PAPACO DEFAULT TABLE INTO C_INDEX 0 (BOTTOM) ----------
;   GSFC RAINBOW (from papco.tbl color table file)
  LOADCT,file=file_table,1,/SILENT
  c_index=0                ;set the sub_index to the first (bottom) subtable
  color_bar_names(c_index)=color_table_names(1)

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
  c_offset=2+c_index*nc

; fold exisiting color table into this limited range
  r_subset=congrid(r_curr(0:nc_total-1),nc,/interp)
  r_papco(c_offset:c_offset+nc-1)=r_subset(0:nc-1)

  g_subset=congrid(g_curr(0:nc_total-1),nc,/interp)
  g_papco(c_offset:c_offset+nc-1)=g_subset(0:nc-1)

  b_subset=congrid(b_curr(0:nc_total-1),nc,/interp)
  b_papco(c_offset:c_offset+nc-1)=b_subset(0:nc-1)

;-- SET THE CORRECT SWITCHES IN THE PAPCO_COLORS COMMON BLOCK TO MAKE THE -----
;   FIRST SUBTABLE THE CURRENT ONE.
  papco_color(c_index,0)=1      ;set bottom default color index as being used
  papco_color(c_index,1)=0      ;set type to unipolar color
  papco_color(c_index,2)=1      ;set bottom default color as active

;-- NOW LOAD SECOND PRE-DEFINED COLOR TABLE (Bipolar) ------------------------
  LOADCT,file=file_table,2,/SILENT
  c_index=1                ;set the sub_index to the second subtable
  color_bar_names(c_index)=color_table_names(2)

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
  c_offset=2+c_index*nc

; fold exisiting color table into this limited range
  r_subset=congrid(r_curr(0:nc_total-1),nc,/interp)
  r_papco(c_offset:c_offset+nc-1)=r_subset(0:nc-1)

  g_subset=congrid(g_curr(0:nc_total-1),nc,/interp)
  g_papco(c_offset:c_offset+nc-1)=g_subset(0:nc-1)

  b_subset=congrid(b_curr(0:nc_total-1),nc,/interp)
  b_papco(c_offset:c_offset+nc-1)=b_subset(0:nc-1)

;-- SET THE CORRECT SWITCHES IN THE PAPCO_COLORS COMMON BLOCK -----------------
  papco_color(c_index,0)=0      ;set color index as being NOT used
  papco_color(c_index,1)=1      ;set type to BIPOLAR color
  papco_color(c_index,2)=0      ;set color index as INactive

;-- NOW LOAD THIRD PRE-DEFINED COLOR TABLE (BLUE-GREEN-RED-YELLOW) ------------
  LOADCT, 4,/SILENT
; Get names of color tables in file
  LOADCT, GET_NAMES=color_table_names
  c_index=2                    ;set the sub_index to the third subtable
  color_bar_names(c_index)=color_table_names(4)

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
  c_offset=2+c_index*nc

; fold exisiting color table into this limited range
  r_subset=congrid(r_curr(0:nc_total-1),nc,/interp)
  r_papco(c_offset:c_offset+nc-1)=r_subset(0:nc-1)

  g_subset=congrid(g_curr(0:nc_total-1),nc,/interp)
  g_papco(c_offset:c_offset+nc-1)=g_subset(0:nc-1)

  b_subset=congrid(b_curr(0:nc_total-1),nc,/interp)
  b_papco(c_offset:c_offset+nc-1)=b_subset(0:nc-1)

;-- SET THE CORRECT SWITCHES IN THE papco_COLORS COMMON BLOCK -----------------
  papco_color(c_index,0)=0      ;set color index as being NOT used
  papco_color(c_index,1)=0      ;set type to UNIPOLAR color
  papco_color(c_index,2)=0      ;set color index as INactive

;-- NOW LOAD FOURTH PRE-DEFINED COLOR TABLE (GREYSCALE) -----------------------
  LOADCT, 0,/SILENT
  c_index=3                ;set the sub_index to the fourth (top) subtable
  color_bar_names(c_index)=color_table_names(0)

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
  c_offset=2+c_index*nc

; fold exisiting color table into this limited range
  r_subset=congrid(r_curr(0:nc_total-1),nc,/interp)
  r_papco(c_offset:c_offset+nc-1)=r_subset(0:nc-1)

  g_subset=congrid(g_curr(0:nc_total-1),nc,/interp)
  g_papco(c_offset:c_offset+nc-1)=g_subset(0:nc-1)

  b_subset=congrid(b_curr(0:nc_total-1),nc,/interp)
  b_papco(c_offset:c_offset+nc-1)=b_subset(0:nc-1)

;-- SET THE CORRECT SWITCHES IN THE papco_COLORS COMMON BLOCK -----------------
  papco_color(c_index,0)=0      ;set color index as being NOT used
  papco_color(c_index,1)=2      ;set type to UNIPOLAR black&white
  papco_color(c_index,2)=0      ;set color index as INactive

;-- SET THE TOP 16 DEFINED PRIMARY COLORS -------------------------------------
  c_index=4                ;set the sub_index the defined 16-color table

; set starting offset for color table. Offset to Offset+16 defines range
; of color indices this sub_table may use.
  c_offset=2+c_index*nc

; make a simple fixed color table for this application.
  r_papco(c_offset:c_offset+15)=[0,1,0,1,0,1,0,1,.7,.7, 0, 0, 0,.7,.3,.6]*255
  g_papco(c_offset:c_offset+15)=[0,0,1,1,0,0,1,1, 0,.7,.7,.7, 0, 0,.3,.6]*255
  b_papco(c_offset:c_offset+15)=[0,0,0,0,1,1,1,1, 0, 0, 0,.7,.7,.7,.3,.6]*255

  black=c_offset+0 & red=c_offset+1 & green=c_offset+2 & yellow=c_offset+3
  blue=c_offset+4 & magenta=c_offset+5 & cyan=c_offset+6 & white=c_offset+7
  burgundy=c_offset+8 & olive=c_offset+9 & dark_green=c_offset+10
  teal=c_offset+11 & royal_blue=c_offset+12 & violet=c_offset+13
  dark_grey=c_offset+14 & grey=c_offset+15
  foreground=1 & background=0

  fixed_color_index=[black, red, green, yellow, blue, magenta,  $
                     cyan, white, burgundy, olive, dark_green,  $
                     teal, royal_blue, violet, dark_grey, grey]
  fixed_color_names=['black', 'red', 'green', 'yellow', 'blue', 'magenta', $
                     'cyan', 'white', 'burgundy', 'olive', 'dark_green', $
                     'teal', 'royal_blue', 'violet', 'dark_grey', 'grey']

;-- SET THE BOTTOM VALUES FOR LINE/ANNOTATION (BLACK) AND BACKGROUND (GREY) ---
;   i.e for foreground and background
  bckg=widgetdata.default_bckg_color
  forg=widgetdata.default_forg_color
  r_papco(0)=bckg & g_papco(0)=bckg & b_papco(0)=bckg
  r_papco(1)=forg & g_papco(1)=forg & b_papco(1)=forg

; IDL 5.0 requires that last color be white.  jbf
  last= nc_total-1
  r_papco(last)=255 & g_papco(last)=255 & b_papco(last)=255

;-- LOAD THE NEW COLOR TABLE --------------------------------------------------
  TVLCT,r_papco,g_papco,b_papco

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Load_SubTable
;*
;* DESCRIPTION:
;*	Folds a user-specified color table into one of the top three slots
;*      us the papco internally subdivided colortable. Selects the new
;*      color table as the active one, and sets it to being unsed.
;*
;* INPUTS:
;*	c_index    the index of the papco internal subtable. Allowed Values
;*                 are 1,2,3. 0 is reserved for to papco default table abd
;*                 cannot be changed.
;*      r_user, g_user b_user  the three rgb color index arrays, 255
;*                 colors each.
;*
;* KEYWORDS:
;* 	TABLE      if set it contains the number of a standard IDL color
;*                 table to be loaded
;*
;* CALLING SEQUENCE:
;*	papco_Load_SubTable,c_index, r_user, g_user, b_user, TABLE=table
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_Load_SubTable,c_index, r_user, g_user, b_user, TABLE=table

   COMMON colors, r_orig,g_orig,b_orig,r_curr,g_curr,b_curr
   COMMON papco_color, papco_color, r_papco, g_papco, b_papco
   COMMON PLOT_COMPOSER, widgetData

; set the color table range indices to be used.
   nc_total=!D.N_colors     ;get total number of colors available
   nc_total=nc_total<255    ;for cases with 24 bit color
   nc_avail=nc_total-16-2   ;the top 16 color indices are reserved
                            ;the lower two colors are reserved (0=black for
                            ;axis and annotation; 1=200 for background)
   nc=nc_avail/4            ;divide by 4 to get no of colors for now.

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
   c_offset=2+c_index*nc

   IF KEYWORD_SET(TABLE) THEN BEGIN
      LOADCT,table ,/SILENT
      r_user=r_orig
      g_user=g_orig
      b_user=b_orig
   ENDIF

; fold user color table into this limited range
   r_subset=congrid(r_user,nc,/interp)
   r_papco(c_offset:c_offset+nc-1)=r_subset(0:nc-1)

   g_subset=congrid(g_user,nc,/interp)
   g_papco(c_offset:c_offset+nc-1)=g_subset(0:nc-1)

   b_subset=congrid(b_user,nc,/interp)
   b_papco(c_offset:c_offset+nc-1)=b_subset(0:nc-1)

;-- SET THE BOTTOM VALUES FOR LINE/ANNOTATION (BLACK) AND BACKGROUND (GREY) ---

  bckg=widgetData.default_bckg_color
  forg=widgetData.default_forg_color

  r_papco(0)=bckg & g_papco(0)=bckg & b_papco(0)=bckg
  r_papco(1)=forg & g_papco(1)=forg & b_papco(1)=forg

;-- LOAD THE NEW COLOR TABLE --------------------------------------------------

  TVLCT,r_papco,g_papco,b_papco

;-- SET THE CORRECT SWITCHES IN THE papco_COLORS COMMON BLOCK TO MAKE THE -----
;   FIRST SUBTABLE THE CURRENT ONE.

   papco_color(c_index,0)=1      ;set new color sub_table as being used
   papco_color(c_index,1)=0      ;set type to unipolar color
   papco_color(*,2)=0            ;unset old active subtable
   papco_color(c_index,2)=1      ;set new sub_table as active

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Set_SubTable
;*
;* DESCRIPTION:
;*	Sets the internal papco sub colortable to be used for plotting.
;*
;* INPUTS:
;*	c_index    the index of the papco internal subtable. Allowed Values
;*                 are 1,2,3. 0 is reserved for to papco default table abd
;*                 cannot be changed.
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*	papco_Set_SubTable,c_index
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_Set_SubTable,c_index

   COMMON papco_color, papco_color, r_papco, g_papco, b_papco

   c_index= c_index < 3 > 0      ;restrict valid values

   papco_color(c_index,0)=1      ;set new color sub_table as being used
   papco_color(c_index,1)=0      ;set type to unipolar color
   papco_color(*,2)=0            ;unset old subtables
   papco_color(c_index,2)=1      ;set new sub_table as active

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Release_SubTable
;*
;* DESCRIPTION:
;*	Sets the chosen suntable as being unused: allows reassignment with
;*      a new color table.
;*      Default is to do nothing: needs keywords.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;*      TABLE    if set, mark that subtable as unused (1,2,3)
;*      ALL      if set, mark all user definable subtables as unused
;*
;* CALLING SEQUENCE:
;*	papco_Release_SubTable
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_Release_SubTable,TABLE=table, ALL=all

   COMMON papco_color, papco_color, r_papco, g_papco, b_papco

   IF KEYWORD_SET(TABLE) THEN BEGIN
      table= table < 3 > 1      ;restrict valid values
      papco_color(table,0)=0    ;set color sub_table as being unused
   ENDIF

   IF KEYWORD_SET(ALL) THEN BEGIN
      papco_color(*,0)=0        ;set all color sub_tables as being unused
      papco_color(0,0)=1        ;apart from the papco default one
   ENDIF

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_SubTable_used
;*
;* DESCRIPTION:
;*	Test if the specified subtable is currently in use.
;*
;* INPUTS:
;*	c_index    the index of the papco internal subtable. Allowed Values
;*                 are 1,2,3. 0 is reserved for to papco default table
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*	res = papco_SubTable_used(c_index)
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_SubTable_used, c_index

   COMMON papco_color, papco_color, r_papco, g_papco, b_papco

   c_index= c_index < 3 > 0      ;restrict valid values

   IF papco_color(c_index,0) eq 1 then return,1 else return,0

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Occupy_SubTable
;*
;* DESCRIPTION:
;*	Set the specified subtable as currently in use.
;*
;* INPUTS:
;*	c_index    the index of the papco internal subtable. Allowed Values
;*                 are 1,2,3. 0 is reserved for to papco default table
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*	papco_Occupy_SubTable, c_index
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_Occupy_SubTable, c_index

   COMMON papco_color, papco_color, r_papco, g_papco, b_papco

   c_index= c_index < 3 > 0      ;restrict valid values
   papco_color(c_index,0)=1

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_get_Colorindices
;*
;* DESCRIPTION:
;*	Return the start and end index for the sub color table in use.
;*
;* INPUTS:
;*	c_index    the index of the papco internal subtable. Allowed Values
;*                 are 1,2,3. 0 is reserved for to papco default table
;*
;* KEYWORDS:
;*      none
;*
;* OUTPUT:
;*      a two element vector containig the start and end index.
;*
;* CALLING SEQUENCE:
;*	colorindicies=papco_get_Colorindices(c_index)
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_get_Colorindices, c_index

   COMMON papco_color, papco_color, r_papco, g_papco, b_papco

   c_index= c_index < 3 > 0      ;restrict valid values

; get the color table range indices used for the sub-table
   nc_total=!D.N_colors     ;get total number of colors available
   nc_total=nc_total<255    ;for cases with 24 bit color
   nc_avail=nc_total-16-2   ;the top 16 color indices are reserved
                            ;the lower two colors are reserved (0=black for
                            ;axis and annotation; 1=200 for background)
   nc=nc_avail/4            ;divide by 4 to get no of colors for now.

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this sub_table  may use.
   c_offset=2+c_index*nc

   indicies=[c_offset,c_offset+nc-1]
   return,indicies

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_GetColor_table
;*
;* DESCRIPTION:
;*	Returns the currently set sub color table index.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* OUTPUT:
;*      the current set sub color table index
;*
;* CALLING SEQUENCE:
;*	colorindicies=papco_GetColor_table()
;*
;* MODIFICATION HISTORY:
;*     february 1997, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_GetColor_table

COMMON papco_color, papco_color, r_papco, g_papco, b_papco

; search common block for active color table (where papco_color(*,2) is set)
active_vector=papco_color(*,2)
result=where(active_vector eq 1)      ;active index for papco color table
return, result(0)

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Get_subtable_index
;*
;* DESCRIPTION:
;*	Returns the index of the color within the current color
;*      subtable for a given data value, given the data range.
;*
;* INPUTS:
;*      data    : the data range min, max assigned to the sub color
;*                table, two element vector
;*      val     : the data value
;*
;* KEYWORDS:
;*      LOG     : 0 uses linear scale (default), 1 uses log scale
;*
;* OUTPUT:
;*      the sub color table index for the data value.
;*
;* CALLING SEQUENCE:
;*	subtableindex=papco_Get_subtable_index(data,val)
;*
;* MODIFICATION HISTORY:
;*     September 1997, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_Get_subtable_index,data,val, LOG=LOG

c_index=papco_GetColor_table()
indicies=papco_get_Colorindices(c_index)

if keyword_set(LOG) then log=LOG else log=0

if log then begin
    if val le 0 then return, -1
    fac=(alog10(val)-alog10(data(0)))/(alog10(data(1))-alog10(data(0)))
    col=indicies(0)+(indicies(1)-indicies(0))*fac
    if col gt indicies(1) then col=indicies(1)
    if col lt indicies(0) then col=indicies(0)
endif else begin
    fac=(val-data(0))/(data(1)-data(0))
    col=indicies(0)+(indicies(1)-indicies(0))*fac
    if col gt indicies(1) then col=indicies(1)
    if col lt indicies(0) then col=indicies(0)
endelse

  return,fix(col)

end

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Get_zvalue
;*
;* DESCRIPTION:
;*	Finds the current color sub-table and computes the corresponding
;*      zvalue for the color index supplied.
;*
;* INPUTS:
;*      index    : the color index to be converted to zvalue
;*      zrange   : the color bar range used
;*
;* KEYWORDS:
;*      LOG      : if set, then the z-axis is logarithmic
;*
;* OUTPUT:
;*      zval     : [from,to] the range of z-values corresponding to index
;*
;* CALLING SEQUENCE:
;*	zvals=papco_Get_zvalue(index,zrange)
;*
;* MODIFICATION HISTORY:
;*     September 1997, written by Reiner Friedel
;*     August 2003, modified to handle true color displays, Reiner Friedel
;******************************************************************************
FUNCTION papco_Get_zvalue, x, y, zrange, LOG=LOG

COMMON papco_graphics, papco_drawn_window     ;current graphics output array
COMMON papco_color, papco_color, r_papco, g_papco, b_papco

idx_rng = papco_get_Colorindices(papco_getColor_table())

IF keyword_set(papco_drawn_window) THEN BEGIN
    r = size(papco_drawn_window, /st)
    IF r.N_DIMENSIONS EQ 3 THEN BEGIN ;true color
        zpix = papco_drawn_window(*, x,y)
        idx = where((zpix(0) EQ r_papco(idx_rng(0):idx_rng(1))) AND $
                    (zpix(1) EQ g_papco(idx_rng(0):idx_rng(1))) AND $
                    (zpix(2) EQ b_papco(idx_rng(0):idx_rng(1))), c)
    ENDIF ELSE BEGIN
        zpix = papco_drawn_window(x,y)
        idx_color = (idx_rng(1)-idx_rng(0)+1)+idx_rng(0)
        idx = where(zpix EQ idx_color, c)
    ENDELSE
ENDIF  ELSE  return, -99

;get "normal" position of found color idx in table idx range
frac = float(idx(0)-idx_rng(0))/float(idx_rng(1)-idx_rng(0))

IF c EQ 0 OR frac LT 0 THEN return, 0

;now that we hav "normal" position if color index in color table,
;comput the zvalue according to the xrange set.
IF keyword_set(LOG) THEN BEGIN  ;logarithmic scale
    zr=alog10(zrange)
    zval = 10^(frac*(zr(1)-zr(0))+zr(0))
ENDIF ELSE BEGIN                ;linear scale
    zval = frac*(zr(1)-zr(0))
ENDELSE

return,zval

END

function papco_get_rgb_for_name, name
  rr=long( [0,1,0,1,0,1,0,1,.7,.7, 0, 0, 0,.7,.3,.6]*255 )
  gg=long( [0,0,1,1,0,0,1,1, 0,.7,.7,.7, 0, 0,.3,.6]*255 )
  bb=long( [0,0,0,0,1,1,1,1, 0, 0, 0,.7,.7,.7,.3,.6]*255 )

  names= [ 'black', 'red', 'green', 'yellow', $
           'blue', 'magenta', 'cyan', 'white',$
           'burgundy', 'olive', 'dark_green',$
           'teal', 'royal_blue', 'violet',$
           'dark_grey', 'grey' ]
  r= where( name eq names )
  if ( r[0] eq -1 ) then begin
      print, 'unrecognized color name: '+name
      print, 'use one of:'
      print, names
      stop
  endif else begin
      idx= r[0]
      return, ishft( bb[idx], 16 ) + $
        ishft( gg[idx], 8 ) + $
        rr[idx]
  endelse
end

;;;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Get_color
;*
;* DESCRIPTION:
;*	finds an RGB color in the existing colorspace
;*
;* INPUTS:
;*      int red, int green, int blue
;*      int hue, int sat, int value, /hsv
;*      string name
;*
;* KEYWORDS:
;*      hsv=hsv  arg1,arg2,arg3   is hsv
;*
;* OUTPUT:
;*      int, the index in an 8-bit colorspace or the rgb int if true
;*         color.  For 8-bit color space, it is not guarenteed you
;*         will get the requested color.  (Presently it grabs the
;*         closest available color in rgb.)
;*
;* CALLING SEQUENCE:
;*	erase, papco_get_color( "yellow" )
;*      erase, papco_get_color( 255, 255, 0 )
;*      erase, papco_get_color( 0,0.,1.0, /hsv )
;*      for i=0.,1.0,0.001 do erase, papco_get_color( 0,1.0,i, /hsv)
;*      for i=0.,360,0.1 do erase, papco_get_color( i,1.0,1, /hsv)
;* MODIFICATION HISTORY:
;*     September 2004, written by Jeremy Faden
;******************************************************************************
function papco_get_color, red_in, green_in, blue_in, hsv=hsv
  if ( size( red_in, /type ) eq 7 ) then begin
      rgb= papco_get_rgb_for_name(red_in)
      blue= ishft( rgb,-16 ) and 255
      green= ishft( rgb,-8 ) and 255
      red= rgb and 255
  endif else if ( keyword_set(hsv) ) then begin
      hue= red_in & sat=green_in & value=blue_in
      color_convert, hue, sat, value, red, green, blue, /hsv_rgb
  endif else begin
      red= red_in & green=green_in & blue=blue_in
  endelse

  if ( !d.name eq 'PS' ) then begin
      decomposed=0  ;; TODO: check on this
  endif else if ( !d.name eq 'PRINTER' ) then begin
      decomposed=0
  endif else begin
      device, get_decomposed=decomposed
  endelse

  if ( decomposed ) then begin
      return, ishft( long(blue), 16 ) + $
        ishft( long(green), 8 ) + $
        red
  endif else begin
      tvlct, /get, r, g, b
      dist= sqrt( ( r-long(red) )^2 + ( g-long(green) )^2 + ( b-long(blue) )^2 )
      x= min( dist, imin )
      if ( x gt 5 ) then begin
          message, 'failed to find close color', /cont
      endif
      return, imin
  endelse

end




