;******************************************************************************
;* PROCEDURE:
;*     PRO s_papco_include_addins
;*
;* DESCRIPTION:
;*     writes new papco_include_addins file to location PAPCO_CONFIG
;*
;* INPUTS:
;*     none
;*
;* KEYWORDS:
;*     FILE      if set, save to this addins file name
;*
;* MODIFICATION HISTORY:
;*     written December 1998, Reiner Friedel
;*     modified January 2002, reiner Friedel, allow multiple conf's
;******************************************************************************
PRO s_papco_include_addins, FILE=FILE

COMMON papco_addin, found_addins, addins, edit_txt, entries

; check for env. var. pointing to PAPCO home directory
if not (papco_check_data_env('PAPCO_CONFIG', path=PATH)) then return
fln='papco_include_addins.pro'
IF keyword_set(FILE) then fln=file

home_path = papco_getenv('PAPCO_HOME')

message,'Writing file:',/cont
print, '  ', path+fln

OPENW, u, path+fln, /GET_LUN

  printf,u,';*****************************************' + $
           '************************************'
  printf,u,';* FILE:'
  printf,u,';*    $PAPCO_HOME/'+fln
  printf,u,';*'
  printf,u,';* DESCRIPTION:'
  printf,u,';*    Sets up other IDL applications to be included with PAPCO'
  printf,u,';*'
  printf,u,';* CALLED BY: papco_startup'
  printf,u,';*'
  printf,u,';* CREATED BY: papco_configure.pro at ' + systime(0)
  printf,u,';*****************************************' + $
           '************************************'
  printf,u,' '
  out='--- Start papco_include_addins ---------------------------------------'
  printf,u,"print,'"+out+"'"

  printf,u,' '
  printf,u,'; Each add-in applications is "included" by executing its own '
  printf,u,'; startup batch-file. This batch file typically expands the IDL '
  printf,u,'; search path to point to the location of the .pro files, and '
  printf,u,'; then compiles all .pro files needed.'

  if addins(0) ne '' then BEGIN
    for i=0,n_elements(addins)-1 do BEGIN
      pos=strpos(addins(i),'@')
      addin_name=strmid(addins(i),pos+1,strlen(addins(i)))
      printf,u,' '
      printf,u,'print, "Executing ',addin_name,'"'
      addin_batch='@'+home_path+'papco_addins/'+addin_name
      printf,u,addin_batch
    endfor
  endif else BEGIN
    printf,u,' '
    printf,u,'print, "no add-inn applications specified"'
  endelse

  printf,u,' '
  out='--- End papco_include_addins ----------------------------------------'
  printf,u,"print,'"+out+"'"

CLOSE, u &  FREE_LUN, u

END

;******************************************************************************
;* PROCEDURE:
;*      PRO r_papco_include_addins
;*
;* DESCRIPTION:
;*      reads papco_include_addins.pro file
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*      written December 1998, Reiner Friedel
;******************************************************************************
PRO r_papco_include_addins

  COMMON papco_addin, found_addins, addins, edit_txt, entries

; check for env. var. pointing to directory of file papco_include_addins.pro
  if not (papco_check_data_env('PAPCO_CONFIG', path=PATH)) then return

  OPENR, u, path+'papco_include_addins.pro', /GET_LUN, ERROR=err
  IF err NE 0 THEN BEGIN
      message, 'No papco_include_addins file found',/cont
      addins = ['']
      return
  endif
  i=0 & junk='' & inarr=['']
  while not eof(u) do BEGIN
    readf,u,junk
    pos=strpos(junk,'@')  &  if pos eq -1 then continue
    filename=strmid(junk,pos+1,strlen(junk))
    papco_sepdir,filename, path, file
    if i eq 0 then inarr=file else inarr=[inarr,file]
    i=i+1
  endwhile
  addins=inarr

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_change_startup
;*
;* DESCRIPTION:
;*      Changes papco_startup file to restore or compile on startup
;*      depending on how the env. variable papco_STARTUP_MODE is set
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*      written July 1997, Reiner Friedel
;******************************************************************************
PRO papco_change_startup

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules

papco_home=papco_getenv('PAPCO_HOME')

; make choices according to env. variables
; PAPCO_STARTUP_MODE, PAPCO_QUIET and PAPCO_ADDINS,
; PAPCO_CONFIG_FILE, PAPCO_ADDINS_FILE
startup_mode=getenv('PAPCO_STARTUP_MODE')
if startup_mode eq 'RESTORE' then BEGIN
    batch_job='@papco_restore'
    default="papco_setenv,      'PAPCO_STARTUP_MODE=RESTORE' "
endif else BEGIN
    batch_job='@papco_compile'
    default="papco_setenv,      'PAPCO_STARTUP_MODE=COMPILE' "
endelse
  quiet_mode=getenv('PAPCO_QUIET')
if quiet_mode eq '0' then BEGIN
    quiet="papco_setenv,      'PAPCO_QUIET=0' "
endif else BEGIN
    quiet="papco_setenv,      'PAPCO_QUIET=1' "
endelse
addins_mode=getenv('PAPCO_ADDINS')
if addins_mode eq 'YES' then BEGIN
    papco_addins_file=getenv('PAPCO_ADDINS_FILE')
    if papco_addins_file ne '' then $
      batch_line='@papco_include_addins_cmd'   else $
      batch_line='@papco_include_addins'
    addins="papco_setenv,      'PAPCO_ADDINS=YES' "
endif else BEGIN
    batch_line=';no addin batch file chosen'
    addins="papco_setenv,      'PAPCO_ADDINS=NO' "
endelse
papco_config_file=getenv('PAPCO_CONFIG_FILE')
if papco_config_file ne '' then $
    modules_line='@papco_include_modules_cmd.pro' else $
    modules_line='@papco_include_modules.pro'

; read papco_startup.pro into an array
startup_arr=strarr(200) & instr='' & i=-1
fname=papco_fname(papco_home+'papco_startup.pro')
openr,unit,fname,/get_lun
while not eof(unit) do BEGIN
     i=i+1 & readf,unit,instr & startup_arr(i)=instr
endwhile

startup_arr=startup_arr(0:i)
close,unit & free_lun,unit

; change contents of file papco_startup.pro and rewrite
res=strpos(startup_arr,'@papco_compile')
index=where(res eq 0,count)
if count eq 0 then BEGIN
    res=strpos(startup_arr,'@papco_restore')
    index=where(res eq 0,count)
endif
startup_arr(index(0))=batch_job

res=strpos(startup_arr,'@papco_include_addins')
index=where(res eq 0,count)
if count eq 0 then BEGIN
    res=strpos(startup_arr,';no addin batch file chosen')
    index=where(res eq 0,count)
endif
startup_arr(index(0))=batch_line

res=strpos(startup_arr,'@papco_include_modules')
index=where(res eq 0,count)
if count eq 0 then BEGIN
    res=strpos(startup_arr,';no include modules file chosen')
    index=where(res eq 0,count)
endif
startup_arr(index(0))=modules_line

; write the new array to file
openw,unit,fname,/get_lun
for j=0,i do printf,unit,startup_arr(j)
close,unit & free_lun,unit

; read papco_set_variables.pro into an array
set_var_arr=strarr(200) & instr='' & k=-1
fname2=papco_fname(papco_home+'papco_set_variables.pro')
openr,unit,fname2,/get_lun
while not eof(unit) do BEGIN
    k=k+1 & readf,unit,instr & set_var_arr(k)=instr
endwhile
set_var_arr=set_var_arr(0:k)
close,unit & free_lun,unit

; change contents of file papco_set_variables.pro and rewrite
res=strpos(set_var_arr,'PAPCO_STARTUP_MODE=RESTORE')
index=where(res ne -1,count)
if count eq 0 then BEGIN
    res=strpos(set_var_arr,'PAPCO_STARTUP_MODE=COMPILE')
    index=where(res ne -1,count)
endif
set_var_arr(index(0))=default

res=strpos(set_var_arr,'PAPCO_QUIET=0')
index=where(res ne -1,count)
if count eq 0 then BEGIN
    res=strpos(set_var_arr,'PAPCO_QUIET=1')
    index=where(res ne -1,count)
endif
set_var_arr(index(0))=quiet

res=strpos(set_var_arr,'PAPCO_ADDINS=YES')
index=where(res ne -1,count)
if count eq 0 then BEGIN
    res=strpos(set_var_arr,'PAPCO_ADDINS=NO')
    index=where(res ne -1,count)
endif
set_var_arr(index(0))=addins

res = strpos(set_var_arr, 'PAPCO_CONFIG =  PAPCO_HOME')
idx = where(res ne -1,count)
set_var_arr(idx(0)) = "PAPCO_CONFIG =  PAPCO_HOME + '"+ papco_get_pathsep() + $
    config_data.config_dir + "'"

; write the new array to file
openw,unit,fname2,/get_lun
for j=0,k do  printf,unit,set_var_arr(j)
close,unit & free_lun,unit

END

;******************************************************************************
;* PROCEDURE:
;*      PRO r_papco_include_modules
;*
;*      Writes papco_include_modules.pro to dir pointed at by PAPCO_CONFIG
;*
;* DESCRIPTION:
;*      Reads previous papco_include_modules.pro if exists
;*
;* INPUTS:
;*      none. All passed in common block papco_configure
;*
;* OUTPUT:
;*      none
;*
;* MODIFICATION HISTORY:
;*      written July 1997, Haje Korth
;*      modified January 2002, Reiner Friedel for multiple configs
;******************************************************************************
PRO r_papco_include_modules

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules

; check for env. variable -> directory containing papco_include_modules.pro
if not (papco_check_data_env('PAPCO_CONFIG', PATH=config_path)) then return

dummystr='' & modules=STRARR(50) & dummystruct={name:'', path:''}
userModules=REPLICATE(dummystruct,50)
i=-1 & j=-1

fname=papco_fname(config_path+'papco_include_modules.pro')
OPENR, unit, fname, /get_lun, ERROR=err

IF err NE 0 THEN BEGIN
    message, '', /cont
    print, '  ' + config_path + 'papco_include_modules.pro -> not available.'
    print, ''
    print, '  Note !!! :'
    print, '  If this is your first time starting this version of PAPCO'
    print, '  or this is a new configuration of PAPCO, you do'
    print, '  not have a corresponding papco_include_modules.pro -'
    print, '  it will be produced by papco Configure when you "Go On".'
    print, ''
    print, '  If you HAVE configured this version of PAPCO before, and'
    print, '  papco_include_modules.pro cannot be found in your'
    print, '  papco_user directory, then you have a path setup problem - '
    print, '  check papco_set_variables.pro in your papco_user directory.'
    print, ''
    RETURN
ENDIF

message, config_path+'papco_include_modules.pro', /INFO

WHILE NOT EOF(unit) DO BEGIN
     READF, unit, dummystr
     id=STRMID(dummystr,0,2)
     IF id EQ '; ' THEN BEGIN
       i=i+1
       pos=STRPOS(dummystr,'module')
       module_name=STRMID(dummystr,2,pos-3)
       len=STRLEN(module_name)
       FOR k=0,len-1 DO $
         IF STRMID(module_name,k,1) EQ ' ' THEN STRPUT,module_name,'_',k
       index=WHERE(standardModules EQ module_name,count)
       IF count GT 0 THEN modules(i)=module_name
       IF count EQ 0 THEN BEGIN
         j=j+1
         readpath=0
         userModules(j).name=module_name
         REPEAT BEGIN
           READF, unit, dummystr
           IF STRMID(dummystr,0,2) EQ '!p' then begin
             readpath=1
             pos=STRPOS(dummystr,'expand')
             len=STRLEN(dummystr)-(pos+16)
             userModules(j).path=STRMID(dummystr, pos+14, len)
           ENDIF else if strpos(dummystr,'papco_add_to_path') ne -1 THEN BEGIN
             readpath=1
             s= str_sep( dummystr, ',' )
             pathquote= strtrim(s[2],2)
             userModules(j).path=STRMID(pathquote, 1, strlen(pathquote)-2 )
            endif
         ENDREP UNTIL readpath

       ENDIF
     ENDIF
ENDWHILE

CLOSE, unit  &  FREE_LUN, unit

if i eq -1 then BEGIN
    message,'No modules found', /cont
    return
ENDIF

modules=modules(0:i)

END

;******************************************************************************
;* PROCEDURE:
;*      PRO s_papco_include_modules
;*
;* DESCRIPTION:
;*      Writes papco_include_modules.pro to dir pointed at by PAPCO_CONFIG
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      FILE   if set use this as the filename instead of
;*             papco_include_modules.pro - to load configuration files
;*
;* MODIFICATION HISTORY:
;*      written July 1997, Haje Korth
;*      modified January 2002, Reiner Friedel, allows multiple conf's
;******************************************************************************
PRO s_papco_include_modules

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules

; check env. variable pointing to dir containing papco_include_modules.pro
if not (papco_check_data_env('PAPCO_MODULES', PATH=module_path)) then return

; check env. variable pointing to config directory
if not (papco_check_data_env('PAPCO_CONFIG', PATH=path)) then return

module_path="papco_getenv('PAPCO_MODULES')+"
fln = config_data.config_dir+'/papco_include_modules.pro'

message,'Writing file:',/cont
print, '  ', path+'papco_include_modules.pro'

OPENW, u, path+'papco_include_modules.pro', /GET_LUN

  star_str= '**************************' & len=strlen(star_str)

  printf,u,';' + star_str + star_str + star_str
  printf,u,';* FILE:'
  printf,u,';*    $PAPCO_HOME/'+fln
  printf,u,';*'
  printf,u,';* DESCRIPTION:'
  printf,u,';*    Expands IDL search path by the modules configured'
  printf,u,';*'
  printf,u,';* CALLED BY: papco_startup'
  printf,u,';*'
  printf,u,';* CREATED BY: papco_configure.pro at ' + systime(0)
  printf,u,';' + star_str + star_str + star_str
  printf,u,' '

    IF userModules(0).name NE '' THEN BEGIN
      printf,u,';' + star_str + strcenter('User Modules',len) + star_str
      FOR i=0,N_ELEMENTS(userModules)-1 DO BEGIN
        IF userModules(i).name NE '' THEN BEGIN
          module_name=userModules(i).name
          path_name=userModules(i).path
          printf,u,'; '+module_name+' module'
          printf,u,"PRINT, 'Adding "+module_name+" module...'"
          printf,u, papco_add_module_path(path_name)
          printf,u,''
        ENDIF
      ENDFOR
    ENDIF

    index=WHERE(config_data.active EQ 1, count)
    IF count NE 0 THEN BEGIN
      printf,u,';' + star_str + strcenter('Standard Modules',len) + star_str
      FOR i=0,N_ELEMENTS(index)-1 DO BEGIN
        module_name=STRTRIM(standardModules(index(i)),2)
        printf,u,'; '+module_name+' module'
        printf,u,"PRINT, 'Adding "+module_name+" module...'"
        actual_module_path="'+"+module_path+"'"+module_name
        printf,u, papco_add_module_path(actual_module_path)
        printf,u,''
        FOR j=0,STRLEN(module_name)-1 DO $
          IF STRMID(module_name,j,1) EQ '_' THEN STRPUT,module_name,' ',j
      ENDFOR
    ENDIF

    printf,u,';' + star_str + star_str + star_str

  CLOSE, u &  FREE_LUN, u

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_addUserModule
;*
;* DESCRIPTION:
;*      Adds user module to the list.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*  EDIT     Edit user module instead of add.
;*
;* OUTPUT:
;*      Modifies the common variable userModules.
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
pro papco_addUserModule, EDIT=EDIT

  common papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules

  common papco_addUserModule, base

  base=papco_widget_base(/COLUMN, TITLE='Add User Module', /MODAL, $
                         GROUP_LEADER=papcocfgWidgetData.base)

  modBase=WIDGET_BASE(base, /ROW)
  txtBase=WIDGET_BASE(modBase, /COLUMN)
  txt_modname=CW_FIELD(txtBase, TITLE='Module Name:', UVALUE='txt_modname', $
                       XSIZE=80)
  txt_pathname=CW_FIELD(txtBase, TITLE='Module Path:', UVALUE='txt_pathname', $
                        XSIZE=80)
  pb_browse=WIDGET_BUTTON(modBase, VALUE='Browse', UVALUE='pb_browse')
  pb_cancel=WIDGET_BUTTON(base, VALUE='Cancel', UVALUE='pb_cancel')
  IF NOT KEYWORD_SET(EDIT) THEN BEGIN
    pb_add_cont=WIDGET_BUTTON(base, VALUE='Add and Continue', $
                              UVALUE='pb_add_cont')
    pb_add=WIDGET_BUTTON(base, VALUE='Add and Close', UVALUE='pb_add')
  ENDIF ELSE $
    pb_edit=WIDGET_BUTTON(base, VALUE='Done', UVALUE='pb_edit')

  ;check if TAGS txt_modname and txt_pathname already exist. If not, add them.
  tagnames=tag_names(papcocfgWidgetData)
  result=strpos(tagnames,'TXT_MODNAME')
  index=where(result ne -1,count)

  if count eq 0 then BEGIN
    papcocfgWidgetData=CREATE_STRUCT({txt_modname:txt_modname, $
                                      txt_pathname:txt_pathname}, $
                                      papcocfgWidgetData)
  endif else BEGIN
    papcocfgWidgetData.txt_modname=txt_modname
    papcocfgWidgetData.txt_pathname=txt_pathname
  endelse


  IF KEYWORD_SET(EDIT) THEN BEGIN
    config_data.edit=1
    index=config_data.index
    WIDGET_CONTROL, papcocfgWidgetData.txt_modname, $
                    SET_VALUE=userModules(index).name
    WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, $
                    SET_VALUE=userModules(index).path
  ENDIF ELSE $
    config_data.edit=0

  geo=WIDGET_INFO(base, /GEO)
  x=600-geo.xsize/2
  y=300-geo.ysize/2
  WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y

  WIDGET_CONTROL, base, /REALIZE
  papco_xmanager, 'papco_addUserModule', base

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_addUserModule_event
;*
;* DESCRIPTION:
;*      Event handler for papco_addUserModule.pro
;*
;* INPUTS:
;*      event     Event
;*
;* KEYWORDS:
;*  none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
PRO papco_addUserModule_event, event

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules
COMMON papco_addUserModule, base

WIDGET_CONTROL, event.id, get_uval= uval

CASE uval OF
    'pb_browse': BEGIN
        IF config_data.edit EQ 0 THEN BEGIN
          home_path=papco_getenv('PAPCO_HOME')
          user_mod_path='papco_user_modules'
          path=papco_addpath(home_path,user_mod_path)
        ENDIF ELSE BEGIN
          WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, GET_VALUE=path
          path=path(0)
          res=findfile(path,count=cc)
          if cc eq 0 then BEGIN
            dummy=messageBox('Directory '+path+' not found', ['OK'], /CENTER, $
                             TITLE='Error',GROUP_LEADER=base)
            WIDGET_CONTROL, base, show=1
            return
          endif
        ENDELSE
        r=PICKFILE(path=path, get_path=new)
        WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, set_value=new
        ;strip off last directory off module path and use as module name
        name=papco_lastdir(new)
        name=strsubst(name,'_',' ')
        WIDGET_CONTROL, papcocfgWidgetData.txt_modname,set_value=name
    END
    'pb_cancel': BEGIN
        WIDGET_CONTROL, event.top, /DESTROY
    END

    'pb_add': BEGIN
        WIDGET_CONTROL, papcocfgWidgetData.txt_modname, GET_VALUE=module_name
        WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, GET_VALUE=path_name
        IF (module_name(0) NE '') AND (path_name(0) NE '') THEN BEGIN
          index=WHERE(userModules.name EQ '', count)
          IF count NE 0 THEN BEGIN
            userModules(index(0)).name=module_name(0)
            userModules(index(0)).path=path_name(0)
          ENDIF
        ENDIF
        WIDGET_CONTROL, event.top, /DESTROY
     END

    'pb_add_cont': BEGIN
       WIDGET_CONTROL, papcocfgWidgetData.txt_modname, GET_VALUE=module_name
       WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, GET_VALUE=path_name
       IF (module_name(0) NE '') AND (path_name(0) NE '') THEN BEGIN
         index=WHERE(userModules.name EQ '', count)
         IF count NE 0 THEN BEGIN
           userModules(index(0)).name=module_name(0)
           userModules(index(0)).path=path_name(0)
         ENDIF
       ENDIF
       ;WIDGET_CONTROL, event.top, /DESTROY
       papco_configure_refresh, /USER_LIST
       WIDGET_CONTROL, papcocfgWidgetData.txt_modname, SET_VALUE=''
       WIDGET_CONTROL, papcocfgWidgetData.txt_pathname,SET_VALUE=''

       ;papco_addUserModule
    END

     'pb_edit': BEGIN
        WIDGET_CONTROL, papcocfgWidgetData.txt_modname, GET_VALUE=module_name
        WIDGET_CONTROL, papcocfgWidgetData.txt_pathname, GET_VALUE=path_name
        IF (module_name(0) NE '') AND (path_name(0) NE '') THEN BEGIN
          index=config_data.index
          userModules(index).name=module_name(0)
          userModules(index).path=path_name(0)
        ENDIF
        WIDGET_CONTROL, event.top, /DESTROY
     END

     ELSE: message, 'Event not handled.', /cont
ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*     PRO papco_comp_user_mod_list
;*
;* DESCRIPTION:
;*     Removes empty positions in between the components of
;*     the userModules variable.
;*
;* INPUTS:
;*     none
;*
;* KEYWORDS:
;*     none
;*
;* OUTPUT:
;*     Modifies the userModules variable.
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
PRO papco_comp_user_mod_list

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules

j=-1
index=WHERE(userModules.name NE '', count)
IF count NE 0 THEN BEGIN
    FOR i=0,N_ELEMENTS(index)-1 DO BEGIN
        j=j+1
        userModules(j).name=userModules(index(i)).name
        userModules(j).path=userModules(index(i)).path
        IF index(i) GT j THEN BEGIN
            userModules(index(i)).name=''
            userModules(index(i)).path=''
        ENDIF
    ENDFOR
ENDIF

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_configure
;*
;* DESCRIPTION:
;*      Creates papco_include_modules.pro for papco startup.
;*      Makes PAPCO configuration widget.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      RECONFIG   disables some parts of widget for reconfiguration work
;*
;* MODIFICATION HISTORY:
;*      written July 1997, Haje Korth
;*      modified August 1999, Reiner Friedel
;******************************************************************************
PRO papco_configure, RECONFIG=RECONFIG, GROUP_LEADER=GROUP_LEADER, $
                     CONFIGDIR = CONFIGDIR

COMMON papco_configure, papcocfgWidgetData, config_data, $
                        modules, standardModules, userModules
COMMON papco_addsn, found_addins, addins, edit_txt, entries


module_path=papco_getenv('PAPCO_MODULES')
standardModules=papco_get_dirlist(module_path)

;take out any CVS modules
idx = where(standardModules NE 'CVS', c)
IF c NE 0 THEN standardModules = standardModules(idx)

; check for env. var. pointing to PAPCO home directory
if not (papco_check_data_env('PAPCO_HOME', PATH=home_path)) then return

;--- check for CONFIGDIR keyword. If set, use CONFIGDIR subdirectory
;    of PAPCO_HOME to read configuration files, default files, printer
;    files, addin files.
IF keyword_set(CONFIGDIR) THEN BEGIN
    config_dir = CONFIGDIR
    config_path = papco_addpath(home_path, config_dir)
ENDIF ELSE BEGIN
    config_path = papco_getenv('PAPCO_CONFIG')
    IF config_path EQ home_path THEN BEGIN ;no config subdir set
        config_dir = ''
    ENDIF ELSE BEGIN
        config_dir = strmid(config_path, strlen(home_path), $
                            strlen(config_path)-strlen(home_path))
    ENDELSE
ENDELSE
message, 'Using config directory: ',  /cont
print, '  ',config_path
papco_setenv, 'PAPCO_CONFIG='+config_path

;-- check if command line paramaters were set - this is done with
;  environmental variables PAPCO_CONFIG_FILE and PAPCO_ADDINS_FILE
papco_config_file=getenv('PAPCO_CONFIG_FILE')
papco_addins_file=getenv('PAPCO_ADDINS_FILE')

if (papco_config_file ne '') or (papco_addins_file ne '') then BEGIN
    if papco_config_file ne '' then BEGIN
      papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
      message,'Starting PAPCO with configuration: '+ papco_config_file, /cont
      r_papco_include_modules, FILE=papco_config_file+'.papco_config'
      active=INTARR(N_ELEMENTS(standardModules))

      FOR i=0,N_ELEMENTS(modules)-1 DO BEGIN
        index=WHERE(STRTRIM(standardModules,2) EQ modules(i),count)
        IF count NE 0 THEN active(index(0))=1
      ENDFOR
      config_data={active:active}
      s_papco_include_modules, FILE='papco_include_modules_cmd.pro'
    endif
    if papco_addins_file ne '' then BEGIN
      papco_setenv,'PAPCO_ADDINS=YES'
      result=str_sep(PAPCO_ADDINS_FILE,'/')
      message,'Starting PAPCO with addins: ', /cont
      for i=0,n_elements(result)-1 do print, '    ',result(i)
      addins='@'+result
      s_papco_include_addins, FILE='papco_include_addins_cmd.pro'
    endif
    papco_change_startup
    return
endif

r_papco_include_modules
active = bytarr(n_elements(standardModules))
FOR i = 0, N_ELEMENTS(modules)-1 DO BEGIN
    idx = where(STRTRIM(standardModules,2) EQ modules(i), c)
    IF c NE 0 THEN active(idx(0)) = 1
ENDFOR

;-- top-level widget ----------------------------------------------------------
if keyword_set(GROUP_LEADER) then $
    base=papco_widget_base(ROW=1, TITLE='papco Configure', $
                           /MODAL, GROUP_LEADER=GROUP_LEADER) else $
    base=widget_base(ROW=1, TITLE='papco Configure')

;-- setup left, right bases ---------------------------------------------------
left_base=widget_base(base,/column)  &  right_base=widget_base(base,/column)

;-- Config load/save widget ---------------------------------------------------
file_base=WIDGET_BASE(left_base, column=1, /FRAME)
result=FINDFILE(config_path+'papco_include_modules.pro',count=count)
IF count GT 0 THEN status=1 ELSE status=0

enter_base=WIDGET_BASE(file_base, column=1)
direc=WIDGET_BASE(enter_base, ROW=1)
direc_label=WIDGET_LABEL(direc, $
                           VALUE=strformat('PAPCO User Directory:', 21, /LEFT))
direc_txt=WIDGET_TEXT(direc, XSIZE=55, YSIZE=1, VALUE=home_path, $
                                UVALUE='direc_txt', EDITABLE=0)

subdir=WIDGET_BASE(enter_base, ROW=1)
subdir_label=WIDGET_LABEL(subdir, $
                          VALUE=strformat('Config Subdirectory:', 21, /LEFT))
subdir_txt=WIDGET_TEXT(subdir, XSIZE=55, YSIZE=1, $
                        VALUE=config_dir, UVALUE='subdir_txt', EDITABLE=1)

status_base=WIDGET_BASE(file_base, ROW=1)
status_label=WIDGET_LABEL(status_base, $
                            VALUE=strformat('Config File Status:', 21, /LEFT))
status_txt=WIDGET_TEXT(status_base, XSIZE=18, YSIZE=1)
file_save=WIDGET_BUTTON(status_base, VALUE=strformat('Create',10,/CEN), $
                              UVALUE='file_save')
file_load=WIDGET_BUTTON(status_base,VALUE=strformat('Load',10,/CEN),  $
                            UVALUE='file_load')
file_browse=WIDGET_BUTTON(status_base,VALUE=strformat('Browse',10,/CEN), $
                            UVALUE='file_browse')

WIDGET_CONTROL, direc_txt, SET_TEXT_SELECT=[strlen(home_path)]
WIDGET_CONTROL, subdir_txt, SET_TEXT_SELECT=[strlen(config_dir)]

;-- edit addins base ----------------------------------------------------------
addins_base=WIDGET_BASE(right_base, /COLUMN)
status_label=WIDGET_LABEL(addins_base, $
                            VALUE=strformat('Edit PAPCO Addins:', 18, /LEFT))
pb_addins=  WIDGET_BUTTON(addins_base, $
                        VALUE=strformat('papco_include_addins.pro',24), $
                        UVALUE='pb_addins')

;-- standard module widgets ---------------------------------------------------
standard_base = CW_BGROUP(left_base, standardModules, COLUMN=5, $
                          /FRAME, /NONEXCLUSIVE, $
                          LABEL_TOP = 'Standard Modules', $
                          UVALUE = 'spb_standard', set_value=active )

geometry=WIDGET_INFO(standard_base, /GEOMETRY)

;-- user module  widgets ------------------------------------------------------
index=WHERE(userModules.name NE '',count)
IF count NE 0 THEN BEGIN
    moduleList=userModules(index)
    user_list = strarr(count)
    FOR i = 0, count-1 DO user_list(i) = moduleList(i).name+ $
                    '   (PATH: '+moduleList(i).path+')'
ENDIF ELSE user_list = ['']

userBase=WIDGET_BASE(left_base, /COLUMN, /FRAME)
lb_userModules=WIDGET_LABEL(userBase, VALUE='User Modules')
lst_userModules=WIDGET_LIST(userBase, VALUE = user_list, $
                            UVALUE='lst_userModules', $
                            scr_xsize=geometry.xsize, $
                            YSIZE=10, /FRAME )
userButtonBase=WIDGET_BASE(userBase, /ROW, /ALIGN_CENTER)
pb_edit=WIDGET_BUTTON(userButtonBase, $
                        VALUE='    EDIT User Module    ', UVALUE='pb_edit')
pb_add=WIDGET_BUTTON(userButtonBase, $
                        VALUE='    ADD User Module     ', UVALUE='pb_add')
pb_remove=WIDGET_BUTTON(userButtonBase, $
                          VALUE='   REMOVE User Module   ', $
                          UVALUE='pb_remove')

;-- startup mode widgets ------------------------------------------------------
  lb_dummy=WIDGET_LABEL(right_base, VALUE='-----------------')
  lb_startupmode=WIDGET_LABEL(right_base, VALUE='Startup Mode:')
  startupmodeBase=WIDGET_BASE(right_base, /COLUMN, /ALIGN_CENTER)
  pb_startupmode_base=WIDGET_BASE(startupmodeBase, $
                                  /FRAME, /COLUMN, /EXCLUSIVE)
  type_names=['Compile               ', $
              'Restore               ' ]
  pb_startupmode_type=LONARR(N_ELEMENTS(type_names))

  FOR i=0, N_ELEMENTS(type_names)-1 DO BEGIN
    pb_startupmode_type(i)=WIDGET_BUTTON(pb_startupmode_base, $
           VALUE=type_names(i), $
           UVALUE='pb_startup_type'+strtrim(string(i), 2))
  ENDFOR

  startup_mode=getenv('PAPCO_STARTUP_MODE')
  if startup_mode eq 'RESTORE' then $
     WIDGET_CONTROL,pb_startupmode_type(1),SET_BUTTON=1 $
  else $
     WIDGET_CONTROL,pb_startupmode_type(0),SET_BUTTON=1

;-- quiet / verbose  mode widgets ---------------------------------------------
  quiet_base=WIDGET_BASE(startupmodeBase, $
                         /FRAME, /COLUMN, /EXCLUSIVE, /ALIGN_CENTER)

  pb_verbose=WIDGET_BUTTON(quiet_base, $
                           VALUE='Verbose compile       ', $
                           UVALUE='pb_verbose')
  pb_quiet=  WIDGET_BUTTON(quiet_base, $
                           VALUE='Quiet compile         ', $
                           UVALUE='pb_quiet')

  quiet=getenv('PAPCO_QUIET')
  if quiet eq '0' then WIDGET_CONTROL,pb_verbose,SET_BUTTON=1 $
    else WIDGET_CONTROL,pb_quiet,SET_BUTTON=1
  if keyword_set(RECONFIG) then WIDGET_CONTROL, startupmodeBase, sensitive=0

;-- include/not include addins mode widgets -----------------------------------
  addins_base=WIDGET_BASE(startupmodeBase, $
                          /FRAME, /COLUMN, /EXCLUSIVE, /ALIGN_CENTER)

  pb_addins_yes=WIDGET_BUTTON(addins_base, $
                           VALUE='Include Add-Ins       ', $
                           UVALUE='pb_addins_yes')
  pb_addins_no =WIDGET_BUTTON(addins_base, $
                           VALUE='Do not include Add-Ins', $
                           UVALUE='pb_addins_no')

  addins=getenv('PAPCO_ADDINS')
  if addins  eq 'YES' then $
     WIDGET_CONTROL,pb_addins_yes,SET_BUTTON=1 $
  else $
     WIDGET_CONTROL,pb_addins_no,SET_BUTTON=1
  if keyword_set(RECONFIG) then WIDGET_CONTROL, startupmodeBase, sensitive=0

;-- action buttons ------------------------------------------------------------
lb_dummy=WIDGET_LABEL(right_base, VALUE='-----------------')
lb_action=WIDGET_LABEL(right_base, VALUE='Exit buttons:')
action_base=WIDGET_BASE(right_base, COLUMN=1, /FRAME, /ALIGN_CENTER)
pb_stop=  WIDGET_BUTTON(action_base, VALUE='   Exit & Stop Config   ', $
                           UVALUE='pb_stop')
pb_cancel=WIDGET_BUTTON(action_base, VALUE='   Old Config & Go on   ', $
                          UVALUE='pb_cancel')
pb_done=  WIDGET_BUTTON(action_base, VALUE='   New Config & Go on   ', $
                         UVALUE='pb_done')
if keyword_set(RECONFIG) then WIDGET_CONTROL, pb_stop, sensitive=0

papcocfgWidgetData={lst_userModules:lst_userModules, $
                    standard_base:standard_base, $
                    pb_edit:pb_edit, $
                    pb_startupmode_type:pb_startupmode_type, $
                    pb_cancel:pb_cancel, $
                    pb_done:pb_done,$
                    base:base}

config_data={base:base,$
             direc_txt:direc_txt, $
             subdir_txt:subdir_txt, $
             home_path:home_path, $
             config_path:config_path, $
             config_dir: config_dir, $
             file_browse:file_browse, $
             file_load:file_load, $
             status:status, $
             status_txt:status_txt, $
             file_save:file_save, $
             active:active, $
             index:0, $
             edit:0}

geo=WIDGET_INFO(base, /GEO)
x = 50 &  y=300-geo.ysize/2
WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y
WIDGET_CONTROL, base, /REALIZE

papco_configure_refresh, /FILE, /USER_LIST

papco_xmanager, 'papco_configure', base

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_configure_refresh
;*
;* DESCRIPTION:
;*      Event handler for papco_configure.pro
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      ALL
;*      FILE
;*      USER_LIST
;*
;* MODIFICATION HISTORY:
;*      written December 1998, Reiner Friedel
;******************************************************************************
PRO papco_configure_refresh, ALL=ALL, FILE=FILE, USER_LIST=USER_LIST, $
                             STANDARD_MODULES = STANDARD_MODULES

COMMON papco_configure, papcocfgWidgetData, config_data, $
                        modules, standardModules, userModules

IF keyword_set(ALL) THEN BEGIN
    FILE=1 & SER_LIST=1 & USER_LIST = 1 & STANDARD_MODULES = 1
ENDIF

IF keyword_set(STANDARD_MODULES) THEN BEGIN
    active = config_data.active
    papco_module_dependence, active
    WIDGET_CONTROL, papcocfgWidgetData.standard_base, SET_VALUE = active
    config_data.active = active
ENDIF

IF keyword_set(FILE) THEN BEGIN
    state_txt = ['File NOT Found!','File Found!','Directory NOT Found!', $
                 'File written.', 'File loaded.', 'Needs saving.']
    file_save = ['Create', 'Save', '----', 'Save', 'Save', 'Save']

    CASE config_data.status OF
      0: BEGIN
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=0
          WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=1
          if config_data.subdir_txt ne '' then $
            WIDGET_CONTROL, config_data.file_save, SENSITIVE=1 else $
            WIDGET_CONTROL, config_data.file_save, SENSITIVE=0
      end
      1: BEGIN
          WIDGET_CONTROL, config_data.file_save, SENSITIVE=1
          WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=1
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=1
      end
      2: BEGIN
          WIDGET_CONTROL, config_data.file_save, SENSITIVE=0
          WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=0
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=0
      end
      3: BEGIN
          WIDGET_CONTROL, config_data.file_save, SENSITIVE=1
          WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=1
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=1
      end
      4: BEGIN
          WIDGET_CONTROL, config_data.file_save, SENSITIVE=1
         WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=1
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=1
      end
      5: BEGIN
          WIDGET_CONTROL, config_data.file_save, SENSITIVE=1
          WIDGET_CONTROL, config_data.subdir_txt, SENSITIVE=1
          WIDGET_CONTROL, config_data.file_load, SENSITIVE=1
        END
    ENDCASE
    WIDGET_CONTROL, config_data.status_txt, $
                    SET_VALUE = state_txt(config_data.status)
    WIDGET_CONTROL, config_data.file_save, $
                    SET_VALUE = file_save(config_data.status)
ENDIF

if keyword_set(USER_LIST) then BEGIN
    index=WHERE(userModules.name NE '',count)
    IF count NE 0 THEN BEGIN
      moduleList=userModules(index)
      len=max(strlen(moduleList.name))+1
      listentry=strarr(count)
      for i=0,count-1 do listentry(i)=strformat(moduleList(i).name,len,/LEFT)+$
                      '(PATH: '+moduleList(i).path+')'
      WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, SET_VALUE=listentry
      WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, SET_LIST_SELECT=0
    ENDIF ELSE BEGIN
      WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, SET_VALUE=''
      WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=0
    endelse
ENDIF

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_configure_addins_event
;*
;* DESCRIPTION:
;*      event handler called by XMANAGER
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*
;* CALLING SEQUENCE:
;*      handled by XMANAGER
;*
;* MODIFICATION HISTORY:
;*      written December 1998, Reiner Friedel
;******************************************************************************
PRO papco_configure_addins_event, event

  COMMON papco_addin, found_addins, addins, edit_txt, entries

  WIDGET_CONTROL,GET_UVALUE=uval, event.id

  CASE uval OF
    'edit_txt' : BEGIN
      WIDGET_CONTROL, edit_txt, GET_VALUE=entries
    END
    'mpb_addin' :BEGIN
      WIDGET_CONTROL,event.id, GET_VALUE=addins_set
      index=where(addins_set ne 0,c)
      if c ne 0 then BEGIN
        addins=found_addins(index)
        addins='@'+addins
      endif else addins=''
      WIDGET_CONTROL, edit_txt, SET_VALUE=addins
    end
    'edit_done' : BEGIN
      WIDGET_CONTROL, edit_txt, GET_VALUE=entries
      valid_entries=WHERE(entries NE '',count)
      IF count GT 0 THEN addins=entries(valid_entries) $
        else addins=''
      WIDGET_CONTROL, event.top, /DESTROY
    END
  ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_configure_event
;*
;* DESCRIPTION:
;*      Event handler for papco_configure.pro
;*
;* INPUTS:
;*      event    Event
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*      written July 1997, Haje Korth
;*      added module dependency checks, Octobner 2003, Reiner Friedel
;******************************************************************************
PRO papco_configure_event, event

COMMON papco_configure, papcocfgWidgetData, config_data, $
                        modules, standardModules, userModules
COMMON papco_addin, found_addins, addins, edit_txt, entries

WIDGET_CONTROL, event.id, get_uval= uval

IF uval EQ 'lst_userModules' THEN IF event.clicks eq 2 then uval='pb_edit'

IF STRMID(uval,0,7) EQ 'pb_type' THEN BEGIN
    index=FIX(STRMID(uval,7,STRLEN(uval)-6))
    IF config_data.active(index) EQ 0 THEN config_data.active(index)=1 $
    ELSE config_data.active(index)=0
    WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(1),SENSITIVE=0
    WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(0),SET_BUTTON=1
    papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
ENDIF else BEGIN

CASE uval OF

    'spb_standard': BEGIN
        WIDGET_CONTROL, papcocfgWidgetData.standard_base, GET_VALUE = set
        config_data.active = set
        config_data.status = 5
        papco_configure_refresh, /STANDARD_MODULES, /FILE
    END

    'file_browse' : BEGIN
      if papco_hasdir(config_data.config_path) then BEGIN
        fname=PICKFILE(/READ, GROUP=config_data.base, $
                       TITLE='Config Path', $
                       PATH=config_data.config_path, $
                       GET_PATH=getpath)
        if fname eq '' then return
        config_data.config_path = getpath
        config_data.config_dir = papco_lastdir(config_data.config_path)
        WIDGET_CONTROL,config_data.subdir_txt, SET_VALUE=config_data.config_dir
        WIDGET_CONTROL,config_data.subdir_txt, $
          SET_TEXT_SELECT=[strlen(config_data.config_dir)]
        message, 'Using config directory: ',  /cont
        print, '  ',config_data.config_path
        papco_setenv, 'PAPCO_CONFIG='+config_data.config_path
        config_data.status=1
        papco_configure_Refresh, /FILE
      endif else BEGIN
        config_data.status=2
        papco_configure_Refresh, /FILE
      endelse
    END

    'subdir_txt' : BEGIN
      WIDGET_CONTROL, config_data.subdir_txt, GET_VALUE=subdir
      subdir = subdir(0)
      config_data.config_dir = subdir
      config_data.config_path = config_data.home_path+subdir
      message, 'Using config directory: ',  /cont
      print, '  ',config_data.config_path
      papco_setenv, 'PAPCO_CONFIG='+config_data.config_path
      papco_setenv, 'PAPCO_TEMP_CONFIG='+subdir
      ;check if "new" home_path exisits, if not, create it.
      if not papco_finddir(config_data.config_path) then $
        papco_create_dir, config_data.config_path, /NOASK

      if not papco_finddir(config_data.config_path) then BEGIN
        config_data.status=0
        papco_configure_Refresh, /FILE
      endif else BEGIN
        papco_file_status,config_data.home_path+'/'+subdir+'/'+$
          'papco_include_modules.pro', exist=exist
        config_data.status=exist
        papco_configure_Refresh, /FILE
      endelse
    END

    'file_save' : BEGIN
      ;check if any modules are configured, if not, don't write
      ; papco_include_modules file
      index=where(config_data.active ne 0,n_standard_modules)
      index=where(userModules.name ne '',n_user_modules)
      n_modules=n_user_modules+n_standard_modules
      if n_modules ne 0 then BEGIN
        s_papco_include_modules
        config_data.status=3
        papco_configure_Refresh, /FILE
      endif
    END

    'file_load' : BEGIN         ;load the file and update all widgets
        r_papco_include_modules
        active = bytarr(n_elements(standardModules))
        FOR i = 0, N_ELEMENTS(modules)-1 DO BEGIN
            idx = where(STRTRIM(standardModules,2) EQ modules(i), c)
            IF c NE 0 THEN active(idx(0)) = 1
        ENDFOR
        config_data.active = active
        config_data.status=4
        papco_configure_refresh, /ALL
    END

    'pb_addins': BEGIN
      r_papco_include_addins
      ;get list of all available papco addin batch files
      path=papco_getenv('PAPCO_HOME')  &  path=path+'papco_addins/'
      result=findfile(path+'papco_addin_*.pro',count=c)
      IF c EQ 0 THEN BEGIN
          message, 'No addins found', /cont
          found_addins = ['']
          addins = ['']
          GOTO, no_addins
      ENDIF
      addin_files=strmid(result,strlen(path),100)
      pos=strpos(addin_files,'.pro')
      found_addins=strarr(c)
      for i=0,c-1 do found_addins(i)=strmid(addin_files(i),0,pos(i))
      addin_set=bytarr(c)
      for i=0,c-1 do $
        FOR j = 0, n_elements(addins)-1 DO $
          IF addins(j) EQ found_addins(i) THEN addin_set(i)=1
      no_addins:
      edit_base=papco_WIDGET_BASE(TITLE='Edit PAPCO Addins', /COLUMN, /MODAL, $
                           GROUP_LEADER=config_data.base)
      edit_txt=WIDGET_TEXT(edit_base, XSIZE=40, YSIZE=10, VALUE=addins, $
                           UVALUE='edit_txt', /EDITABLE,  /ALL_EVENTS, /SCROLL)
      mpb_addin=CW_BGROUP(edit_base, found_addins, $
                          /FRAME, COLUMN=2,  /NONEXCLUSIVE, $
                          LABEL_TOP='Available PAPCO addins:', $
                          UVALUE='mpb_addin', SET_VALUE=addin_set)

      edit_done=WIDGET_BUTTON(edit_base, VALUE='Done', UVALUE='edit_done')
      WIDGET_CONTROL,edit_base,/REALIZE
      papco_xmanager, "papco_configure_addins", edit_base, $
                      GROUP_LEADER=config_data.base
      s_papco_include_addins
    end

    'lst_userModules': BEGIN
      config_data.index=event.index
      WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(1),SENSITIVE=0
      WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(0),SET_BUTTON=1
      papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
    END

    'pb_edit': BEGIN
      WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(1),SENSITIVE=0
      WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(0),SET_BUTTON=1
      papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
      papco_addUserModule, /EDIT

      index=WHERE(userModules.name NE '',count)
      IF count NE 0 THEN BEGIN
        moduleList=userModules(index)
        WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, $
                        SET_VALUE=moduleList.name+ $
                        '   (PATH: '+moduleList.path+')'
        WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=1
      ENDIF ELSE BEGIN
        WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, SET_VALUE=''
        WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=0
      ENDELSE
    END

    'pb_add': BEGIN
          WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(1),SENSITIVE=0
          WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(0),SET_BUTTON=1
          papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
          papco_addUserModule
          index=WHERE(userModules.name NE '',count)

          IF count NE 0 THEN BEGIN
            moduleList=userModules(index)
            WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, $
                            SET_VALUE=moduleList.name+ $
                            '   (PATH: '+moduleList.path+')'
            WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=1
          ENDIF ELSE BEGIN
            WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, $
                            SET_VALUE=''
            WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=0
          ENDELSE
    END

    'pb_remove': BEGIN
          WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(1),SENSITIVE=0
          WIDGET_CONTROL,papcocfgWidgetData.pb_startupmode_type(0),SET_BUTTON=1
          papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'
          index=config_data.index
          userModules(index).name=''
          userModules(index).path=''
          papco_comp_user_mod_list
          index=WHERE(userModules.name NE '',count)
          IF count NE 0 THEN BEGIN
            moduleList=userModules(index)
            WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, $
                            SET_VALUE=moduleList.name+ $
                            '   (PATH: '+moduleList.path+')'
            WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=1
          ENDIF ELSE BEGIN
            WIDGET_CONTROL, papcocfgWidgetData.lst_userModules, $
                            SET_VALUE=''
            WIDGET_CONTROL, papcocfgWidgetData.pb_edit, SENSITIVE=0
          ENDELSE
    END

    'pb_startup_type0': papco_setenv,'PAPCO_STARTUP_MODE=COMPILE'

    'pb_startup_type1': papco_setenv,'PAPCO_STARTUP_MODE=RESTORE'

    'pb_verbose': papco_setenv,'PAPCO_QUIET=0'

    'pb_quiet': papco_setenv,'PAPCO_QUIET=1'

    'pb_addins_yes': papco_setenv,'PAPCO_ADDINS=YES'

    'pb_addins_no': papco_setenv,'PAPCO_ADDINS=NO'

    'pb_cancel': WIDGET_CONTROL, event.top, /DESTROY

    'pb_done': BEGIN
        ;check if any modules are configured, if not, don't write
        ;papco_include_modules file
        index=where(config_data.active ne 0, n_standard_modules)
        index=where(userModules.name ne '',n_user_modules)
        n_modules=n_user_modules+n_standard_modules
        if n_modules ne 0 then s_papco_include_modules
        WIDGET_CONTROL, event.top, /DESTROY
    END

    'pb_stop': BEGIN
        WIDGET_CONTROL, event.top, /DESTROY
        !quiet=0 & message, 'Config Aborted'
    END

    ELSE: meassage, 'internal error.', /INFO

    ENDCASE

ENDELSE

END


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_module_dependence
;*
;* DESCRIPTION:
;*      Looks up a list of modules that depend on other modules, sets
;*      those so that they also get compiled in.
;*
;* INPUTS:
;*      set     logical array for standard modules 0 - not set, 1 - set
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*     written August 1999, Reiner Friedel
;******************************************************************************
PRO papco_module_dependence, set, VERBOSE = VERBOSE

COMMON papco_configure, papcocfgWidgetData, config_data, $
                        modules, standardModules, userModules

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0

verbose = 1

;make a list of module dependencies
dep_array = ['aero_heo:heo_eph', $
             'cluster_rapid:cluster_eph/papco_cdf/omni/papco_event', $
             'lanl_geo:lanl_geo_eph', $
             'lanl_gps:gps_eph', $
             'polar_ceppad:polar_eph', $
             'goes_eph:papco_cdf']

;now loop through dep_array and check each module listed there.
FOR i = 0, n_elements(dep_array)-1 DO BEGIN
    r1 = strsplit(dep_array(i), ':', /EXTRACT)
    name = r1(0)
    dep_names = strsplit(r1(1), '/', /EXTRACT)

    ;get index of module name in set, standardModules
    idx = where(standardModules EQ name, c)
    IF c EQ 0 THEN BEGIN
        message, 'module '+name+' not found'
    ENDIF
    ;check if this module is selected or not, and if so, set depent module
    IF set(idx(0)) EQ 0 THEN CONTINUE
    FOR j = 0, n_elements(dep_names)-1 DO BEGIN
        idx2 = where(standardModules EQ dep_names(j), c)
        IF c EQ 0 THEN BEGIN
            message, 'module '+dep_name(j)+' not found'
        ENDIF ELSE BEGIN
            IF set(idx2(0)) EQ 1 THEN CONTINUE ;dep. module already set
            IF verbose THEN message, name+' depends on: '+dep_names(j), /info
            set(idx2(0)) = 1
        ENDELSE
    ENDFOR
ENDFOR

END


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_reconfigure
;*
;* DESCRIPTION:
;*      Once PAPCO has been started, this routine allows the user to
;*      add or delete modules from PAPCo without needing to restart IDL.
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*     written August 1999, Reiner Friedel
;******************************************************************************
PRO papco_reconfigure, GROUP_LEADER=GROUP_LEADER

COMMON papco_configure, papcocfgWidgetData, config_data, $
    modules, standardModules, userModules
  COMMON module_paths, paths
  COMMON papco_addin, found_addins, addins, edit_txt, entries

; keep list of exisiting addins.
  r_papco_include_addins
  old_addins=addins

; call the configuration widget
  papco_configure, /RECONFIG, GROUP_LEADER=GROUP_LEADER

; get list of new addins - only those not already in old_addins
  r_papco_include_addins
  for i=0, n_elements(old_addins)-1 do BEGIN
    pos=strpos(addins,old_addins(i))
    index=where(pos ne -1,n)
    if n ne 0 then addins(index(0))=''
  endfor
  index=where(addins ne '',n)

  if n ne 0 then BEGIN
    addins=addins(index)
  endif else addins=''
  s_papco_include_addins, FILE='papco_include_new_addins.pro'

; get list of existing modules
  papco_find_modules, /NO_WRITE, /QUIET
  old_paths=paths

; delete all paths in !path that contain old modules
  idl_search_path=!path
; make an array of all the paths. The are seperated by dirsep
  idl_search_paths=str_sep(idl_search_path,papco_get_dirsep())
  idl_search_paths=strlowcase(idl_search_paths)

  message,'No of paths with modules:     '+ $
    varprt(n_elements(idl_search_paths)),/cont

  new_idl_search_path_count=0
  for i=0, n_elements(old_paths)-1 do BEGIN
    test_path=strmid(old_paths(i),0,strlen(old_paths(i))-1)
    result=strpos(idl_search_paths,test_path)
    index=where(result eq  -1,c) ;find paths NOT containing old_paths(i)!
    idl_search_paths=idl_search_paths(index)
  endfor

;make new path variable without any of the modules
  new_path=idl_search_paths(0)
  for i=1,n_elements(idl_search_paths)-1 do $
    new_path=new_path+papco_get_dirsep()+idl_search_paths(i)
  new_paths=str_sep(new_path,papco_get_dirsep())

  message,'No of paths without modules:  '+varprt(n_elements(new_paths)),/cont

; make new !path based on new papco_include_modules.pro and new path
  !path=new_path
  dir=papco_getenv('PAPCO_HOME') & instr=''
  openr,unit,dir+'papco_include_modules.pro', /get_lun
  while not eof(unit) do BEGIN
    readf,unit,instr
    if strmid(instr,0,5) eq '!path' then BEGIN
      result=execute(instr)
    endif
  endwhile
  close,unit  &  free_lun,unit

  new_paths=str_sep(!path,papco_get_dirsep())
  message,'No of paths with new modules: '+varprt(n_elements(new_paths)),/cont

; get list of new modules
  papco_find_modules, /NO_WRITE, /QUIET
  new_paths=paths

; make list of new modules added. Only those need to be compiled now.
  add_mod=0
  for i=0, n_elements(new_paths)-1 do BEGIN
    result=strpos(old_paths,new_paths(i))
    index=where(result ne -1 ,c)
    if c eq 0 then BEGIN       ;found new module
      if add_mod eq 0 then add_mod_list=[new_paths(i)] else $
        add_mod_list=[add_mod_list,new_paths(i)]
      add_mod=add_mod+1
      message,'New module found: ...'+strright(new_paths(i),30), /cont
    endif
  endfor

  if add_mod eq 0 then BEGIN
    message,'No new modules added',/cont
    paths=''
    papco_w_add_init,FILE='papco_new_add_init.pro',/EMPTY
  endif else BEGIN
    ;make a "papco_new_add_init" batch file to compile new modules
    paths=add_mod_list
    papco_w_add_init,FILE='papco_new_add_init.pro'
  endelse

; restore current  module_paths
  paths=new_paths

; make new batch file to compile new modules and addins
; check for env. var. pointing to directory of new batch file
  if not (papco_check_data_env('PAPCO_HOME', path=PATH)) then return
  fln='papco_recon.pro'
  message,'Writing file papco_recon.pro',/cont
  OPENW, u, path+fln, /GET_LUN

  printf,u,';*****************************************' + $
           '************************************'
  printf,u,';* FILE:'
  printf,u,';*    $PAPCO_HOME/'+fln
  printf,u,';*'
  printf,u,';* DESCRIPTION:'
  printf,u,';*    Calls batch files that compile a reconfigured PAPCO session'
  printf,u,';*'
  printf,u,';* CALLED BY: user from command line'
  printf,u,';*'
  printf,u,';* CREATED BY: papco_reconfigure at ' + systime(0)
  printf,u,';*****************************************' + $
           '************************************'
  printf,u,' '
  printf,u,';--- Start papco_recon -----'+ $
           '---------------------------------------'

  printf,u,' '
  printf,u,'; runs papco_new_add_init.pro which compiles newly added modules'
  printf,u,'; runs papco_include_new_addins.pro which compiles new addins '
  printf,u,' '

  printf,u,'retall'
  printf,u,'@papco_new_add_init'
  printf,u,'@papco_include_new_addins'

  printf,u,''
  printf,u,'print, "re-starting PAPCO"'
  printf,u,'papco'

  printf,u,' '
  printf,u,';--- End papco_recon -----'+ $
           '-----------------------------------------'

  CLOSE, u &  FREE_LUN, u

end
