;******************************************************************************
;******************************************************************************
;* FILE:
;*    $PAPCO_PATH/papco_cursor.pro
;*
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:
;*    implements the handling of mouse-events for the widgetData.dw_draw-
;*    widget.
;*
;*    There are two main functionalities in this file:
;*
;*         1. Providing all the procedures for standard papco slices
;*         2. Calling user defined routines for user added slices.
;*
;*    Adding of user slices is done in the papco main routine papco.pro
;*    The way this works: the user sets up an environmnetal variable
;*    called USER_MOUSE_ACTIONS in papco_set_variables before starting
;*    up papco. This variable contains a string of the names of the
;*    user-defined mouse actions in the following format:
;*    'user_slice_a/user_slice_b/user_slice_c'  etc. Seperate words
;*    Seperate words by underscores, slices types by backslash.
;*    These slice types are added to the Mouse Action menu. When chosen,
;*    and enabled by your plot_type, papco_cursor will attempt to call
;*    a function "user_slice_plot_type" to handle the slice.
;*
;*    Currently papco provides 5 "pre-written" slice types. Two of these
;*    function work as they are:
;*
;*         zooming
;*         haircursor 
;*
;*    The first enables the user create a rectangular area in the drawing
;*    widget by dragging the mouse. While dragging, data-coordinates are
;*    displayed in a small window. Afterwards, the user may zoom into this
;*    rectangle.
;*    The second displays a large hair cursor and prints out the
;*    x,y,z values of the mouse click both at the mouse click and
;*    at the top of the draw window.
;*
;*    The remaining three are functioning templates which don't do much
;*    by themselves, but can be easily extended by the user
;*
;*         drawing a slice of data
;*         accessing level zero application
;*         writing out data from a plot
;*
;*    When slicing, the user selects a time by clicking into the drawing-
;*    widget. Then, the data for this time are displayed in a small window,
;*    which contains a draw window for a user-defined plot. Possible uses
;*    are to draw a line spectrum for a color spectrogram.
;*
;*    When accessing level zero, another widget application can be called.
;*    This application only needs to know the time, which is passed
;*    across in the call. The idea is to use this slice option to call
;*    a "level zero" tool for the data, which can display individual detecors,
;*    distribution functions etc.
;*
;*    When writing out data, the mouse returns the panel and the x/y
;*    coordinates. The user can then open a file to write the data into.
;*    This is usefull for reading off a series of event times into a file.
;*
;*    Those procedures are contained:
;*
;*       FUNCTION papco_cursor_time,panelnr
;*	     ->
;*
;*       FUNCTION papco_Normal_ToData, panelNr, normal_position
;*	     -> convert normal coordinates of the plot-panel to data-
;*		coordinates of one plot.
;*
;*	 FUNCTION papco_Data_To_Normal, panelNr, data_position
;*	     -> convert data-coordinates of one plot to normal coordinates
;*
;*       FUNCTION papco_cursor_WhichPanel, normal_x, normal_y
;*	     -> return the number of the panelplot, the normal position
;*		'normal_x, normal_y' belongs to
;*
;*	 PRO papco_cursor_Reset, FULLY=FULLY
;*	     -> clear mouse-selection
;*
;*	 PRO papco_vert_slice_Event, Event
;*	     -> called, when an event occured in the slice-info-window
;*
;*	 PRO papco_vert_slice, panelNr, seconds, yValue, CLOSE=CLOSE
;*	     -> draw a slice
;*
;*	 PRO papco_cursor_UpdateData, YSCL=YSCL, TIME=TIME
;*	     -> update cursor-selection
;*
;*	 PRO papco_cursor_Rect_Event, event
;*	     -> called, when an event occured in the zoom-info-window
;*
;*	 PRO papco_cursor_Rect, panelNr, normal_x, normal_y, CLOSE=CLOSE
;*	     -> draw zoom-rectangle
;*
;*	 PRO papco_cursor_ShowSelection, panelNr, x, y, normal_x, normal_y, $
;*				         ERASE_ONLY=ERASE_ONLY
;*	     -> show current selection
;*
;*       PRO papco_cursor_SetPointer, panelNr, normal_x, normal_y
;*	     -> set the pointer-icon
;*
;*	 PRO papco_cursor, event
;*	     -> handles all events for the draw-widget
;*
;*       PRO papco_cursor_HairCursor
;*	     -> draws a large hair cursor and x,y,z value
;*
;*       FUNCTION papco_slice_value,slice_name
;*	     -> returns slice value for a slice name
;*
;* MODIFICATION HISTORY:
;*     august,september 1995, written by A.Keese
;*     may 1996, modified R. Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION
;*      papco_cursor_time,panelnr
;*
;* DESCRIPTION:
;*	Given the cursor time and Panel Nr. finds time of cursor.
;*      Reurns a verbose string of the time.
;*
;* INPUTS:
;*      user_time       User time value returned by cursor
;*	panelNr         Panel No of cursor
;*
;* KEYWORDS:
;* 	MJDT            if set, returns time in mjdt format.
;*
;* CALLING SEQUENCE:
;*	result=papco_cursor_time(time,panelnr)
;*
;* MODIFICATION HISTORY:
;*     written october 1997, Reiner Friedel
;******************************************************************************
FUNCTION papco_cursor_time,user_time, panelNr, MJDT=MJDT

COMMON mjdt, mjdt_start, mjdt_end
COMMON PLOT_COMPOSER, widgetData

;-- use the info in !X.CRANGE to scale start/end time to papco time.
user_mintime=double(widgetdata.plotsdrawn(panelNr).data_xrange(0))
user_maxtime=double(widgetdata.plotsdrawn(panelNr).data_xrange(1))
user_time_frac=(user_time-user_mintime) / (user_maxtime-user_mintime)

;convert mjdt_start, mjdt_end to TAI and find TAI of selected timerange
p_min={utc,mjd:mjdt_start.mjd,time:mjdt_start.t*1000}
p_min_tai=utc2tai(p_min)
p_max={utc,mjd:mjdt_end.mjd,  time:mjdt_end.t*1000}
p_max_tai=utc2tai(p_max)
papco_time_tai=p_min_tai+user_time_frac*(p_max_tai-p_min_tai)
  
IF keyword_set(MJDT) THEN BEGIN
    click_time_str=anytim2cal(papco_time_tai,FORM=11)
    year=strmid(click_time_str,0,4)
    month=strmid(click_time_str,5,2)
    day=strmid(click_time_str,8,2)
    click_time_str=month+'/'+day+'/'+year+strmid(click_time_str,10,9)
    result=convert_timeToSeconds(click_time_str,/T90,/MJDT)
    RETURN,result.value
ENDIF ELSE BEGIN
    click_time_str=anytim2cal(papco_time_tai,FORM=11)
    RETURN,click_time_str
ENDELSE

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Normal_ToData, panelNr, normal_position
;*
;* DESCRIPTION:
;*      Convert a normal coordinate to data-coordinates of a certain
;*      panel.
;*
;* INPUTS:
;*	panelNr	 	 an integer
;*			 the coordinates will be transformed using the
;*			 coordinate-system of widgetData.plotsDrawn(panelNr)
;*	normal_position  a two-dimensional float-array
;*			 normal_position(0)   : x-coordinate
;*			 normal_position(0)   : y-coordinate
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      a two dimensional array
;*	output(0)        x-data-coordinate
;*	output(1)        y-data-coordinate
;*
;* CALLING SEQUENCE:
;*	tmp=papco_Normal_To_Data( 0, [x_normal, y_normal])
;*	x_data=tmp(0)
;*	y_data=tmp(1)
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     modified october 1997, Reiner Friedel, to make USER mouse
;*     feateres configurable by user.
;******************************************************************************
FUNCTION papco_Normal_To_Data, panelNr, normal_position

COMMON PLOT_COMPOSER, widgetData

plot=widgetData.plotsDrawn(panelNr)
   IF plot.x_type EQ 0 THEN $	; linear x-axis
      x=(normal_position(0) - plot.x_s(0)) / plot.x_s(1) $
   ELSE $    	; logarithmic x-axis
      x=10^((normal_position(0) - plot.x_s(0)) / plot.x_s(1))
   IF plot.y_type EQ 0 THEN $	; linear y-axis
      y=(normal_position(1) - plot.y_s(0))  / plot.y_s(1) $
   ELSE $    	                ; logarithmic y-axis
      y=10^((normal_position(1) - plot.y_s(0)) / plot.y_s(1))

RETURN, [x, y]

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Data_ToNormal, panelNr, normal_position
;*
;* DESCRIPTION:
;*      Convert data coordinates of a certain panel to normal coordinates.
;*
;* INPUTS:
;*	panelNr	    an integer
;*		    the coordinates will be transformed using the
;*		    coordinate-system of widgetData.plotsDrawn(panelNr)
;*	position    a two-dimensional float-array
;*			 position(0)   : x-coordinate
;*			 position(1)   : y-coordinate
;*
;* KEYWORDS:
;* 	TO_DATA if set, input is data coords, output is normal
;*
;* OUTPUT:
;*      a two dimensional array
;*	output(0)        x-normal-coordinate
;*	output(1)        y-normal-coordinate
;* CALLING SEQUENCE:
;*	tmp=papco_Data_ToNormal(0, [x_data, y_data])
;*	x_normal=tmp(0)
;*	y_normal=tmp(1)
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_Data_To_Normal, panelNr, data_position

COMMON PLOT_COMPOSER, widgetData

   plot=widgetData.plotsDrawn(panelNr)
   normal_position=FLTARR(2)
   IF plot.x_type EQ 0 THEN $	; linear x-axis
      normal_position(0)=plot.x_s(0)+data_position(0)*plot.x_s(1) $
   ELSE $    	; logarithmic x-axis
      normal_position(0)=plot.x_s(0)+ALOG10(data_position(0))*plot.x_s(1)

   IF plot.y_type EQ 0 THEN $	; linear x-axis
      normal_position(1)=plot.y_s(0)+data_position(1)*plot.y_s(1) $
   ELSE $    	; logarithmic y-axis
      normal_position(1)=plot.y_s(0)+ALOG10(data_position(1))*plot.y_s(1)

   RETURN, normal_position
   
END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_cursor_WhichPanel, normal_x, normal_y
;*
;* DESCRIPTION:
;*      find out to which panel the normal coordinates normal_x and normal_y
;*	belong.
;*
;* INPUTS:
;*	normal_x 	a float
;*	normal_y	a float
;*			this are the normal coordinates of some point in the
;*			drawing-widget
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      an integer
;*	the number of the panel is returned. If the point does not lie in
;*	any panel, -1 is returned. if the cursor lies to the right or
;*      left of a panel, -panel-2 is returned. We return the undelying
;*      panel no. in the case of overplots.
;*
;* CALLING SEQUENCE:
;*	panelNr=papco_cursor_WhichPanel( normal_x, normal_y)
;*	IF panelNr EQ -1 THEN $
;*	   print, 'No panel selected' $
;*	ELSE $
;*	   print, 'selected panel-info:', widgetData.plotsDrawn(panelNr)
;*
;* MODIFICATION HISTORY:
;*     written August and September 1995, Andreas Keese
;*     modified October 1998, Reiner Friedel, to include indication of
;*     "right/left of panel" for use in calling the panel editor.
;******************************************************************************
FUNCTION papco_cursor_WhichPanel, normal_x, normal_y

  COMMON PLOT_COMPOSER, widgetData

  IF widgetData.numberOfPlotsDrawn LE 0 THEN RETURN, -1

  panelNr=-1 & i=0
  REPEAT BEGIN                  ;go through panels leaving out overplots!
    if not widgetData.plotinfos(i).OVERPLOT then begin
      xr=widgetData.plotsDrawn(i).normal_xrange
      yr=widgetData.plotsDrawn(i).normal_yrange
      IF normal_y GE yr(0) AND normal_y LE yr(1) THEN BEGIN
        panelNr=i*(-1)-2
        IF normal_x GE xr(0) AND normal_x LE xr(1) THEN BEGIN
          panelNr=i
        ENDIF
      ENDIF
    endif  
    i=i+1
  ENDREP UNTIL i GE widgetData.numberOfPlotsDrawn OR panelNr GE 0

  IF panelNr GE 0 THEN $
    IF widgetData.plotsDrawn(panelNr).changed THEN panelNr=-1

  RETURN, panelNr

END


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cursor_UpdateData, YSCL=YSCL, TIME=TIME
;*
;* DESCRIPTION:
;*      The ZOOOM-window is a small window, in which data about the selected
;*	rectangle is displayed.
;*	In this window, there are three update-buttons. If the user selects
;*	one, the time-range and/or the y-range of a plot is changed to the
;*	by mouse selected range.
;*	This procedure handles the pressing of such a button - the y-range
;*	and/or the time-range of a plot are changed.
;*
;*	These changes are not performed, if the orbit has changed since
;*	drawing the plots.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	YSCL	if specified, the y-range of the plot described in
;*		COMMON PAPCO_CURSOR_ECT, cursorWidgetData
;*		is changed.
;*      TIME	if specified, the time-range of all plots is changed to
;*		the current selection.
;*
;* CALLING SEQUENCE:
;*	papco_cursor_UpdateData, /YSCL, /TIME
;*	papco_cursor_UpdateData, /YSCL
;*	papco_cursor_UpdateData, /TIME
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     modified February 1997 by Reiner Friedel, to be independent of x-axis
;*                                                format and to use mjdt time
;******************************************************************************
PRO papco_cursor_UpdateData, YSCL=YSCL, TIME=TIME

   COMMON PLOT_COMPOSER, widgetData
   COMMON PAPCO_CURSOR_RECT, cursorWidgetData
   COMMON MJDT, mjdt_start, mjdt_end

   panel=widgetData.mouse_panelNr
   orbNr=widgetData.default_orbitInfo.number

   p1=papco_Normal_To_Data(widgetData.mouse_panelNr, $
                          widgetData.mouse_clicked_position)
   p2=papco_Normal_To_Data(widgetData.mouse_panelNr, $
                          widgetData.mouse_position)

;--- set the new time ---------------------------------------------------------
   IF KEYWORD_SET(TIME) THEN BEGIN
      IF orbNr NE widgetData.plotsDrawn(panel).orbit AND $
        widgetData.time_choice_action $
        THEN BEGIN
         dummy=messageBox(['You entered an orbit different from the', $
                          "plotted data's orbit " + $
                          strtrim(string(widgetData.plotsDrawn(panel).orbit),$
                                   2),$
			  'If you want to zoom, You have to reset the orbit'],$
                          ['Ok'], /CENTER, $
                          title='orbit changed...')
         RETURN
      ENDIF
      orbNr=widgetData.plotsDrawn(panel).orbit
      WIDGET_CONTROL, widgetData.ef_orbit, SET_VALUE=orbNr

;-- use the info in !X.CRANGE to scale start/end time to papco time.
      panel=widgetData.mouse_panelNr          ; the panel "clicked"
      user_mintime=double(widgetdata.plotsdrawn(panel).data_xrange(0))
      user_maxtime=double(widgetdata.plotsdrawn(panel).data_xrange(1))
      user_start_frac=(p1(0)-user_mintime) / (user_maxtime-user_mintime)
      user_end_frac=  (p2(0)-user_mintime) / (user_maxtime-user_mintime)

      ;convert mjdt_start, mjdt_end to TAI and find TAI of selected timerange
      p_min={utc,mjd:mjdt_start.mjd,time:mjdt_start.t*1000}
      p_min_tai=utc2tai(p_min)
      p_max={utc,mjd:mjdt_end.mjd,  time:mjdt_end.t*1000}
      p_max_tai=utc2tai(p_max)
      papco_start_tai=p_min_tai+user_start_frac*(p_max_tai-p_min_tai)
      papco_end_tai=  p_min_tai+user_end_frac  *(p_max_tai-p_min_tai)

      IF papco_start_tai gt  papco_end_tai THEN BEGIN
         dummy=papco_start_tai
         papco_start_tai=papco_end_tai
         papco_end_tai=dummy
      ENDIF

      ;convert TAI of selected timerange back to mjdt
      start_mjdt=tai2utc(papco_start_tai)
      widgetData.startTime.mjd=start_mjdt.mjd
      widgetData.startTime.t=  start_mjdt.time/1000
      end_mjdt=  tai2utc(papco_end_tai)
      widgetData.endTime.mjd= end_mjdt.mjd
      widgetData.endTime.t=end_mjdt.time/1000
      ;convert also to T90 for compatibility
      tstr=convert_secondstotime(widgetData.endTime, /t90,/mjdt)
      t90=convert_timetoseconds(tstr, /t90)
      widgetData.endTime.t90=t90.value
      tstr=convert_secondstotime(widgetData.startTime, /t90,/mjdt)
      t90=convert_timetoseconds(tstr, /t90)
      widgetData.startTime.t90=t90.value

      papco_REFRESH, /TIME

   ENDIF

;--- set the new y-range ------------------------------------------------------
   IF KEYWORD_SET(YSCL) THEN BEGIN
      widgetData.plotInfos(panel).manualYScaling=1
      widgetData.plotInfos(panel).YScaleMin=p1(1) < p2(1)
      widgetData.plotInfos(panel).YScaleMax=p1(1) > p2(1)
      papco_REFRESH, /LIST
   ENDIF
    
; now need to redraw plot!
   widgetData.need_to_draw=1

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cursor_Rect_Event, event
;*
;* DESCRIPTION:
;*      The ZOOOM-window is a small window, in which data about the selected
;*	rectangle is displayed.
;*	This procedure handles events for that window.
;*
;* INPUTS:
;*	event	 an XMANAGER-event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO papco_cursor_Rect_Event, event

COMMON PLOT_COMPOSER, widgetData
COMMON PAPCO_CURSOR_RECT, cursorWidgetData

WIDGET_CONTROL, Event.Id, GET_UVALUE = Ev

CASE Ev OF
    'pb_update_YSCL' : BEGIN
        papco_cursor_UpdateData, /YSCL
        WIDGET_CONTROL, cursorWidgetData.pb_draw, SENSITIVE=1
    END      

    'pb_update_time' : BEGIN
        papco_cursor_UpdateData, /TIME
        WIDGET_CONTROL, cursorWidgetData.pb_draw, SENSITIVE=1
    END      

    'pb_update_both' : BEGIN
        papco_cursor_UpdateData, /YSCL, /TIME
        WIDGET_CONTROL, cursorWidgetData.pb_draw, SENSITIVE=1
    END      

    'pb_help': xdisplayfile, papco_helpfile('papco_cursor_rect.help'), $
        title='papco Online-Help', group=event.top
      
    'pb_draw': BEGIN  ; draw. -> draw button in papco main window.
         papco_main_Event, 0, USER_EVENT='pb_draw'
         WIDGET_CONTROL, cursorWidgetData.pb_draw, SENSITIVE=0
     END

    'pb_cancel': BEGIN
        papco_cursor_ShowSelection, widgetData.mouse_panelNr, $
                                    0, 0, 0, 0, /ERASE_ONLY
        WIDGET_CONTROL, event.top, /DESTROY
    END
    ELSE : BEGIN
    END
ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cursor_Rect, panelNr, normal_x, normal_y, CLOSE=CLOSE
;*
;* DESCRIPTION:
;*	This procedure is called whenever the zoom-function is active and
;*	the mouse is dragged in the draw-widget.
;*	A zoom-window is opened - this is a small window, in which data
;*	about the selected rectangle is displayed.
;*      To be independent of the actual time axis format used by the plot,
;*      this routine uses the system variable !X.CRANGE and converts the
;*      cursor time returned to the papco internal time format.
;*
;*      N.B.!!: For this to work, the user plot routine needs to adhere to the
;*      papco philosophy and ALWAYS plot data for the time range given by
;*      COMMON mjdt, mjdt_start, mjdt_end
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the rectangular
;*			selection is done
;*	normal_x	a float
;*	normal_y	a float
;*			the normal coordinates of the last mouse-position
;*
;* KEYWORDS:
;* 	CLOSE		if set, the window is closed
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     modified February 1997 by Reiner Friedel, to be independent of x-axis
;*                                                format and to use mjdt time
;******************************************************************************
PRO papco_cursor_Rect, panelNr, CLOSE=CLOSE

COMMON PLOT_COMPOSER, widgetData
COMMON PAPCO_CURSOR_RECT, cursorWidgetData
COMMON MJDT, mjdt_start, mjdt_end

;-- close the zoom - window ? -------------------------------------------------
IF KEYWORD_SET(CLOSE) THEN BEGIN
    IF XREGISTERED('papco_cursor_Rect') THEN $
      IF N_ELEMENTS(cursorWidgetData) GT 0 THEN BEGIN
        IF cursorWidgetData.base NE 0 THEN $
          WIDGET_CONTROL, cursorWidgetData.base, /DESTROY
        cursorWidgetData.base=0
    ENDIF
    RETURN
ENDIF

;-- is the mouse still in the initially selected panel ? ----------------------
IF panelNr NE widgetData.mouse_panelNr THEN RETURN

;-- open the zoom-window ------------------------------------------------------
IF NOT XREGISTERED('papco_cursor_Rect') THEN BEGIN
     cursorRectBase=WIDGET_BASE(COLUMN=1, TITLE='Choose Area', $
		                GROUP=widgetData.main_base)
     lbl_bounds=WIDGET_TEXT(cursorRectBase, XSIZE=34, YSIZE=6)
     butt_base=WIDGET_BASE(cursorRectBase, ROW=1)
     butt_base1=WIDGET_BASE(butt_base, COLUMN=1)
     butt_base2=WIDGET_BASE(butt_base, COLUMN=1)
     butt_base3=WIDGET_BASE(butt_base, COLUMN=1)
     pb_updY=WIDGET_BUTTON(butt_base1, VALUE=strcenter('Update Y', 11), $
		      UVALUE='pb_update_YSCL')
     pb_updT=WIDGET_BUTTON(butt_base2, VALUE=strcenter('Update Time', 11), $
		      UVALUE='pb_update_time')
     pb_updB=WIDGET_BUTTON(butt_base3, VALUE=strcenter('Update Both', 11), $
		      UVALUE='pb_update_both')
     pb_help=WIDGET_BUTTON(butt_base1, VALUE=strcenter('Help', 11), $
		      UVALUE='pb_help')
     pb_draw=WIDGET_BUTTON(butt_base2, VALUE=strcenter('Draw', 11), $
		      UVALUE='pb_draw')
     pb_cancel=WIDGET_BUTTON(butt_base3, VALUE=strcenter('Close', 11), $
			      UVALUE='pb_cancel')
     cursorWidgetData={CURSOR_RECTANGLE, base:cursorRectBase, $
                       pb_draw:pb_draw, $
                       lbl_bounds:lbl_bounds}   
     WIDGET_CONTROL, cursorRectBase, /REALIZE
     XMANAGER, 'papco_cursor_Rect', cursorRectBase, /JUST_REG
ENDIF

;-- now, calculate the new coordinates and display them -----------------------
papco_get_data_box, starttime, endtime, startData, endData   
   
info=strarr(6)

;-- use the info in !X.CRANGE to scale start/end time to papco time.
   panel=widgetData.mouse_panelNr          ; the panel "clicked"
   user_mintime=double(widgetdata.plotsdrawn(panel).data_xrange(0))
   user_maxtime=double(widgetdata.plotsdrawn(panel).data_xrange(1))
   user_start_frac=(starttime-user_mintime) / (user_maxtime-user_mintime)
   user_end_frac=  (endtime-user_mintime)   / (user_maxtime-user_mintime)

   ;convert mjdt_start, mjdt_end to TAI and find TAI of selected timerange
   p_min={utc,mjd:mjdt_start.mjd,time:mjdt_start.t*1000}
   p_min_tai=utc2tai(p_min)
   p_max={utc,mjd:mjdt_end.mjd,  time:mjdt_end.t*1000}
   p_max_tai=utc2tai(p_max)
   papco_start_tai=p_min_tai+user_start_frac*(p_max_tai-p_min_tai)
   papco_end_tai=  p_min_tai+user_end_frac  *(p_max_tai-p_min_tai)

   ;convert TAI of selected timerange to verbose time
   start_string=anytim2cal(papco_start_tai)
   end_string  =anytim2cal(papco_end_tai)
   info(0)='Timerange start : ' + start_string
   info(1)='            end : ' + end_string
   info(3)='In plot '+strtrim(string(panelNr+1), 2) + ' you selected data'
   info(4)='           from : ' + strtrim(string(startData), 2)
   info(5)='             to : ' + strtrim(string(endData), 2)

IF XREGISTERED('papco_cursor_Rect') THEN BEGIN
    WIDGET_CONTROL, cursorWidgetData.lbl_bounds, SET_VALUE=info
    WIDGET_CONTROL, cursorWidgetData.pb_draw, SENSITIVE=0
ENDIF

END
 
;******************************************************************************
; routine to get data coordinates of box  made when dragging the
; mouse. The box is defined by the first clicked position and the
; rease click position.
;******************************************************************************
pro papco_get_data_box, starttime, endtime, startData, endData
  
COMMON PLOT_COMPOSER, widgetData
COMMON MOUSE_SELECT,old,click,pnlNr,normal_x,normal_y
  
pos1=papco_Normal_To_Data(widgetData.mouse_panelNr, $
                          widgetData.mouse_clicked_position)
pos2=papco_Normal_To_Data(widgetData.mouse_panelNr, [normal_x,normal_y])

IF pos1(0) LT pos2(0) THEN BEGIN
    starttime=pos1(0) & endtime=pos2(0)
ENDIF ELSE BEGIN
    starttime=pos2(0) & endtime=pos1(0)
ENDELSE

IF pos1(1) LT pos2(1) THEN BEGIN
    startData=pos1(1) & endData=pos2(1)
ENDIF ELSE BEGIN
    startData=pos2(1) & endData=pos1(1)
ENDELSE
    
end
  
;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cursor_SetPointer, panelNr, normal_x, normal_y
;*
;* DESCRIPTION:
;*	When the mouse is moved into a valid panel, the pointer changes to
;*	a hand-symbol. When it's not inside a valid panel, the pointer
;*	becomes an ugly symbol...
;*	When the mouse is dragged in a valid panel, the pointer is changed to
;*	a sizing-symbol.
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the rectangular
;*			selection is done
;*	normal_x	a float
;*	normal_y	a float
;*			the normal coordinates of the current mouse-position
;*
;* KEYWORDS:
;* 	none
;*
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO papco_cursor_SetPointer, panelNr, normal_x, normal_y

COMMON PLOT_COMPOSER, widgetData
COMMON PLOTTING, BUSY

  IF panelNr GE 0 THEN BEGIN
  ;-- for drag-actions, make the cursor a sizing-symbol -------------------
      IF NOT widgetData.mouse_doesSlice AND widgetData.mouse_clicked THEN BEGIN
          click=widgetData.mouse_clicked_position
          IF normal_x GT click(0) THEN BEGIN
              IF normal_y GT click(1) THEN $
                DEVICE, cursor_standard=136 $ ; sizing-symbol right-top
              ELSE $
                DEVICE, cursor_standard=14 ; sizing-symbol right-bottom
          ENDIF ELSE BEGIN
              IF normal_y GT click(1) THEN $
                DEVICE, cursor_standard=134 $ ; sizing-symbol left-top
              ELSE $
                DEVICE, cursor_standard=12 ; sizing-symbol left-bottom
          ENDELSE
                                ;-- IF the pointer is in a valid panel, make the cursor a hand-symbol ---
      ENDIF ELSE BEGIN
          DEVICE, CURSOR_STANDARD=58 ; hand-symbol
      ENDELSE
  ENDIF ELSE if panelNr ne -1 then begin
      if !version.os_family eq 'unix' then begin
          if normal_x lt 0.5 then $
            device, cursor_standard= 112 $
          else $
            device, cursor_standard= 110
      endif
  endif else BEGIN
      DEVICE, CURSOR_STANDARD=122 ; ugly-symbol
  ENDELSE

  END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_cursor, event
;*
;* DESCRIPTION:
;*	Whenever the mouse moves over the main papco-window drawing-widget,
;*	this procedure is called to handle that event.
;*	Here, the cursor-pointer is set according to the mouse-position, and
;*	slice-actions are processed according to the allowed slice-modes.
;*
;* INPUTS:
;*	event	 an XMANAGER-event
;*
;* KEYWORDS:
;* 	MOUSE_DOESSLICE   overwrites this value in widgetdata if
;*                        set. used for recusive calls to this rotuine
;*
;* CALLING SEQUENCE:
;*	called by papco_event
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     Torsten Heise modified September 1997,
;*     Reiner Friedel modified October 1997 to make slices configurable
;*     Reiner Friedel modified October 1998 to enable panel editor calling
;******************************************************************************
PRO papco_cursor, event, MOUSE_DOESSLICE=MOUSE_DOESSLICE

COMMON plot_composer, widgetdata
COMMON addins, plot_types
COMMON mouse_information, mousedata ;general mouse information
COMMON slice_calls, slice_calls
  
;-- prepare the current widget for output -------------------------------------
PAPCO_SET_PLOT, 'X'
WIDGET_CONTROL, widgetData.dw_draw,  GET_VALUE=drawIndex
WSET, drawIndex

;-- get the mouse-position in normal coordinates ------------------------------
tmp=CONVERT_COORD(event.x, event.y, /DEVICE, /TO_NORMAL)
x=tmp(0) & y=tmp(1)  &  panelNr=papco_cursor_WhichPanel(x,y)
  
;-- did the user release the mouse-button ? -----------------------------------
IF (event.release ne 0) AND (widgetData.mouse_clicked ne 0) then $ 
    widgetData.mouse_clicked=0

;-- set the mouse-pointer -----------------------------------------------------
papco_cursor_SetPointer, panelNr, x, y

;-- did the user press the button? Then initiate new selection,----------------
;   but only if he pressed it inside a valid panel.
IF (event.press ne 0) AND panelNr GT -1 THEN BEGIN
    papco_cursor_Reset
    widgetData.mouse_clicked=event.press
    widgetData.mouse_clicked_position=[x, y]
    widgetData.mouse_position=[0,0]
    widgetData.mouse_selectionDrawn=0
    widgetData.mouse_PanelNr=panelNr
ENDIF
  
;-- if the panel No is less then -1, call the panel editor for the
;   panel at position (panelNr+2)*(-1)
IF (event.press ne 0) AND panelNr LE -2 THEN BEGIN
    panelNr=(panelNr+2)*(-1)
    message,'Calling Panel editor for panel '+ varprt(panelNr), /cont
    papco_EditPlot, panelNr
    return
ENDIF

;-- Now, when the user drags the mouse, update the display --------------------
;   the display is only updated, IF the mouse-button is pressed and the
;   mouse is still inside of that panel in which the drag was started
IF widgetData.mouse_clicked ne 0 THEN BEGIN
    xx=x & yy=y
    mouse_Plot=widgetData.plotsDrawn(widgetData.mouse_panelNr)
    xx=xx > mouse_Plot.normal_xRange(0)
    xx=xx < mouse_Plot.normal_xRange(1)
    yy=yy > mouse_Plot.normal_yRange(0)
    yy=yy < mouse_Plot.normal_yRange(1)
    data_pos=papco_Normal_To_Data(widgetData.mouse_panelNr, [xx, yy])
    
; -- some slice types are tied to default mouse buttons -----------------------
;    The LEFT mouse button ALWAYS chooses the mouse action as set in
;    the main papco widget. Here we configure the other mouse buttons
;    too, to provide extra functionality. 
;    event.press; 1 (left button), 2 (middle button), 4 (right button)   
;    The default mouse action stays "on screen" when either the middle
;    or right mouse are pressed. When going back to the default action
;    (left button) then the middle or right action is erased.
    
    CASE widgetData.mouse_clicked of
      1:  BEGIN
        IF widgetData.mouse_doesSlice ne mouseData.mouseactiontyp then begin
            papco_cursor_ShowSelection, widgetData.mouse_panelNr, 0, 0, 0, 0, $
              /ERASE_ONLY
            papco_cursor_Rect, /CLOSE
            papco_cursor_HairCursor, /ERASE
        ENDIF   
        widgetData.mouse_doesSlice=mouseData.mouseactiontyp
      END  
      2:  BEGIN
          if mouseData.mouseactiontyp ne 1 then papco_cursor_Rect, /CLOSE
          widgetData.mouse_doesSlice= 5    ; middle button is always hair cursor
      END  
      4:  BEGIN
          if mouseData.mouseactiontyp ne 5 then papco_cursor_HairCursor, /ERASE
          widgetData.mouse_doesSlice= 1    ; right button is always zoom 
      END  
    ENDCASE  

;-- find mouse action ------------------- -------------------------------------
;   here the type of mouse action is checked according to the slice_type flag
;   set by the user. Depending on which bits he set, the action is enabled.
;
;   slice_type:   Selects which slices are supported and for which slice
;                 routines have been written.
;                 This is an integer logical with 16 switches. Currently
;                 there are 5 papco slices. You may extend this by
;                 further USER slices up to a total of 16 slices.
;                     0 = disable all slices
;                     1 = papco Zoom                 (2^0)
;                     2 = papco vertical Slice       (2^1)
;                     4 = papco Access level Zero    (2^2)
;                     8 = papco Write Paneldata      (2^3)
;                    16 = papco Hair Cursor          (2^4)
;                    32 = papco horizontal Slice     (2^5)  
;                    64 = papco multiple Slice       (2^6)
;                   128 = USER slice A or not used   (2^7)
;                   256 = USER slice B or not used   (2^8)
;                    "     "     "   "  "  "    "      "
;                 32768 = USER slice Z or not used  (2^15)
;
;   You can support more than one type of slice by arithmetically adding the
;   options, eg slice_type=1+4  support Zoom and Level Zero

    IF panelnr ge 0 THEN BEGIN
      
        ;depending on the slice Action, do same slice for more than one panel.
        ;make list of panels to do the slice action with. ignore overplots!
        case widgetData.slice_choice of
            1: do_panels=[panelNr]
            2: BEGIN       
                do_panels=[panelNr]
            END 
            3: BEGIN       
                do_panels=indgen(widgetData.numberOfPlots)
            END 
        ENDCASE  
        n_panels=n_elements(do_panels)
        slice_calls=strarr(n_panels)
      
        ;loop through all panels for which slice is required
        slice_count=0
        FOR kk=0,n_panels-1 DO BEGIN 
            panelNr=do_panels(kk)
            slice=fix(widgetData.plotsDrawn(panelNr).slice_type)
        
            papco_cursor_ShowSelection, widgetData.mouse_panelNr, $
                                        data_pos(0), data_pos(1), xx, yy
        
            IF keyword_set(MOUSE_DOESSLICE) THEN $
                 mouse_doesSlice=MOUSE_DOESSLICE  ELSE $
                 mouse_doesslice=widgetData.mouse_doesSlice
             
            CASE mouse_doesSlice OF ;first find standard papco mouse functions
                1:BEGIN  ; papco Zoom
                    IF (slice AND 1) EQ 1 THEN $        ;1 is zoom flag
                      papco_cursor_Rect, widgetData.mouse_panelNr
                END
                2:BEGIN  ; papco vertical Slice
                    IF (slice AND 2) EQ 2 THEN BEGIN    ;2 is vert slice flag
                    papco_vert_slice, panelNr, data_pos(0), data_pos(1), $
                      OVERPLOT = slice_count
              slice_calls(slice_count)='papco_vert_slice,'+ $
                string(panelNr)+','+ $
                string(data_pos(0))+','+string(data_pos(1))+ $
                ', OVERPLOT='+string(slice_count)
              slice_count=slice_count+1
            endif  
          END
          3:BEGIN  ; papco Access level Zero
            IF (slice and 4) EQ 4 THEN BEGIN   ; 4 is level zero flag
              currPlotInfo=widgetData.plotsDrawn(panelNr)
              result=strpos(plot_types,strchminus(currPlotInfo.panelKind),0)
              typeindex=where(result eq 0,count)
              IF count ne 0 THEN BEGIN
                seconds=data_pos(0) & yvalue=data_pos(1)
                levelzero_call='levelzero_'+strchminus(currPlotInfo.panelKind)
                levelzero_prmt='seconds, yvalue'
                callgood=execute(levelzero_call+','+levelzero_prmt)
              ENDIF
            ENDIF
          END
          4:BEGIN  ; papco Write Paneldata
            IF (slice and 8) EQ 8 THEN $          ; 8 is Write Paneldata flag
              papco_write_paneldata, widgetData.mouse_panelNr, $
                                     data_pos(0), data_pos(1)
          END
          5:BEGIN  ; papco Hair Cursor
            IF (slice and 16) EQ 16 THEN $      ; 16 is Hair Cursor flag
              papco_cursor_HairCursor, widgetData.mouse_panelNr, event, $
                                       data_pos(0), data_pos(1)
          END    
          6:BEGIN  ; papco horizontal Slice
            IF (slice and 32) EQ 32 THEN $     ; 32 is horiz slice flag
              papco_horiz_slice, widgetData.mouse_panelNr, $
                                      data_pos(0), data_pos(1)
          END  
            
          n_elements(widgetData.mouseAction)-1 : BEGIN ; multiple slices
            index=where(widgetData.default_multiple_slice ne 0,c)
            if c ne 0 then begin
              rsl_lines=0     
              for i=0,c-1 do $ ;recursively call papco_cursor for all slices
                papco_cursor, event, MOUSE_DOESSLICE=index(i)+1
            endif else message,'no multiple slices chose',/cont 
          END  
          ; Now call user defined mouse function. Name of the call is assembled
          ; from the user supplied slice name in env. var  USER_MOUSE_ACTIONS  
          ; Uses papco error trapping in calling user routines.
          ELSE: BEGIN
            slice_flag=2^(widgetData.mouse_doesSlice-1)
            IF (slice and slice_flag) EQ slice_flag THEN BEGIN
              uMa=widgetdata.mouseAction(widgetData.mouse_doesSlice).NAME
              uMa=strsubst(strtrim(uMa,2),' ','_')
              currPlotInfo=widgetData.plotsDrawn(panelNr)
              result=strpos(plot_types,strchminus(currPlotInfo.panelKind),0)
              typeindex=where(result eq 0,count)
              IF count ne 0 THEN BEGIN
                seconds=data_pos(0) & yvalue=data_pos(1)
                user_slice_call=uMa + '_' + strchminus(currPlotInfo.panelKind)
                proc_call=user_slice_call
                panelNr=widgetData.mouse_panelNr

                ;remember the slice call for printing!
                slice_calls(slice_count)=user_slice_call + ','+ $
                string(panelNr)+','+ $
                string(data_pos(0))+','+string(data_pos(1))
                slice_count=slice_count+1

                if widgetData.default_trap eq 1 then $
                  catch,error_status else error_status=0
                if error_status ne 0 then begin ;call error handler
                  papco_error_handler, $
                    error_status, proc_call, panelkind, [0,0,0], /NO_ERR_DRAW
                  error_status=0 & catch, /CANCEL ;disable error catching
                  return
                endif  
                CALL_PROCEDURE, proc_call, panelNr, seconds, yValue, $
                  READDATA=READDATA
                catch, /CANCEL    ;disable error catching 
              ENDIF
            ENDIF  
          END
        ENDCASE
      ENDFOR
      if slice_count gt 0 then slice_calls=slice_calls(0:slice_count-1)
    ENDIF
  ENDIF

END


;******************************************************************************
;* PROCEDURE:
;*      PRO  papco_cursor_ShowSelection, panelNr, x, y, normal_x, normal_y, $
;*				         ERASE_ONLY=ERASE_ONLY
;*
;* DESCRIPTION:
;*	Draw the selection the user has made dragging his mouse in the draw-
;*	widget. Before the new selection is drawn, the last selection shown
;*	is erased.
;*	In zoom-mode, the selection is a rectangle
;*	In slice-mode, it is a vertical line
;*      In access level zero, it is a vertical line
;*      In view ephemeris, it is a vertical line
;*      In write paneldata, it is a vertical line
;*
;* INPUTS:
;*	panelNr	 	an integer
;*			the number of the panel for which the rectangular
;*			selection is done
;*	normal_x	a float
;*	normal_y	a float
;*			  the normal coordinates of the last mouse-position
;*	x		a float
;*	y		a float
;*			  these is the location of the last mouse-position in
;*			  data-coordinates of the panel 'panelNr'
;*
;* KEYWORDS:
;* 	ERASE_ONLY	when set, the current selection is not shown but the
;*			last selection is hidden.
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     modified september 1997 Torsten Heise
;*     Reiner Friedel modified october 1997 to make show selection configurable
;******************************************************************************
PRO papco_cursor_ShowSelection, panelNr, x, y, normal_x, normal_y, $
				ERASE_ONLY=ERASE_ONLY
  
  COMMON PLOT_COMPOSER, widgetData
  COMMON MOUSE_SELECT,old,click,pnlNr,n_x,n_y
  n_x=normal_x & n_y=normal_y & pnlNr=panelNr

  IF (panelNr LT 0) AND NOT KEYWORD_SET(ERASE_ONLY) THEN RETURN
  
;-- get new and old mouse positions -------------------------------------------
  click=widgetData.mouse_clicked_position
  old  =widgetData.mouse_position  
  
;-- prepare the main drawing-widget for plotting ------------------------------
  IF widgetData.no_draw_window ne 0 THEN BEGIN
    PAPCO_SET_PLOT, 'X'
    WIDGET_CONTROL, widgetData.dw_draw, GET_VALUE=drawIndex
    WSET, drawIndex
  ENDIF ELSE RETURN

  DEVICE, SET_GRAPHICS=6        ; set XOR-Mode for graphics 
  
;-- make a list of show_selection types for use with the papco slice types ----
;   index of this array corresponds to widgetData.mouse_doesSlice-1  
  
  slice_show_selection = ['box', $
                          'vert_line', $
                          'vert_line', $
                          'vert_line', $
                          'hair_cursor', $
                          'horz_line' ]  
  
;-- make list for the show_selections associated with user slices ------------
;   add this list to slice_show_selection
;   user defined show_selections are defined in the env. var.USER_SHOW_ACTIONS
  
; check if USER_SHOW_ACTIONS is defined
  user_show_actions=getenv('USER_SHOW_ACTIONS')
  IF strlen(user_show_actions) eq 0 THEN BEGIN 
    ;if not defined, select 'vert_line' defaults for user slices
    user_start_idx = n_elements(slice_show_selection)+1
    user_end_idx   = n_elements(widgetdata.mouseAction)-1
    FOR i=user_start_idx, user_end_idx DO BEGIN
      slice_show_selection=[slice_show_selection,'vert_line']
    ENDFOR
  ENDIF ELSE BEGIN
    ;parse the user_show_actions string for the user show selections
    user_show_actions=str_sep(user_show_actions,'/')
    FOR i=0,n_elements(user_show_actions)-1 DO BEGIN
      slice_show_selection=[slice_show_selection,user_show_actions(i)]
    ENDFOR
  ENDELSE 
  
;-- add in default show selection for papco_multiple_slice  
   slice_show_selection = [slice_show_selection, 'vert_line']
  
;-- set the show selection procedure call -------------------------------------
  select_call='papco_cursor_showselection_' + $
              slice_show_selection(widgetData.mouse_doesSlice-1)
  
;-- erase the old selection ---------------------------------------------------
  IF widgetData.mouse_selectionDrawn THEN BEGIN
    select_prmt='/ERASE'
    callgood=execute(select_call+','+select_prmt)
  ENDIF
  
;-- show the new selection ----------------------------------------------------
  IF NOT KEYWORD_SET(ERASE_ONLY) THEN BEGIN
    widgetData.mouse_position=[normal_x, normal_y]
    widgetData.mouse_selectionDrawn=1    
    select_prmt='/DRAW'
    callgood=execute(select_call+','+select_prmt)
   ENDIF ELSE BEGIN
      widgetData.mouse_position=widgetData.mouse_clicked_position
   ENDELSE

   DEVICE, SET_GRAPHICS=3 ; set copy-mode for graphics

END   ;of papco_cursor_ShowSelection

;******************************************************************************
;* A bunch of single routines for each selection type currently
;* supported by PAPCO. 
;*
;* User may add his own routines externally!
;*
;* written February 1998, Reiner Friedel
;******************************************************************************
pro papco_cursor_showselection_vert_line, ERASE=ERASE, DRAW=DRAW
 
  COMMON PLOT_COMPOSER, widgetData
  COMMON MOUSE_SELECT,old,click,pnlNr,n_x,n_y 
  
;-- calculate the height (in normal coordinates) of a vertical line -----------
  IF pnlNr GE 0 THEN $     ;the vertical range  of this panel (normal coords)
    pYRg =widgetData.plotsDrawn(pnlNr).normal_yRange $
  ELSE  pYRg=[0,0]  
  
;-- draw / erase the vertical line --------------------------------------------
  if keyword_set(ERASE) THEN BEGIN
    PLOTS, [old(0), old(0)],[0,1],color=30,/NORMAL
    PLOTS, [old(0), old(0)],[pYRg(0),pYRg(1)],color=120,/NORMAL
  ENDIF
  
  IF keyword_set(DRAW) THEN BEGIN
    PLOTS, [n_x, n_x], [0,1], color=30, /NORMAL
    PLOTS, [n_x, n_x], [pYRg(0),pYRg(1)], color=120, /NORMAL
  ENDIF

end

pro papco_cursor_showselection_horz_line, ERASE=ERASE, DRAW=DRAW
 
  COMMON PLOT_COMPOSER, widgetData
  COMMON MOUSE_SELECT,old,click,pnlNr,n_x,n_y 
  
;-- calculate the width (in normal coordinates) of a horizontal line ----------
  IF pnlNr GE 0 THEN $      ;the horizontal range of this panel (normal coords)
    pXRg =widgetData.plotsDrawn(pnlNr).normal_xRange $
  ELSE pXRg=[0,0]
  
;-- draw / erase the vertical line --------------------------------------------
  IF keyword_set(ERASE) THEN BEGIN
    PLOTS, [0, 1], [old(1),old(1)], color=30, /NORMAL
    PLOTS, [pXRg(0),pXRg(1)], [old(1),old(1)], color=120, /NORMAL     
  ENDIF
  
  IF keyword_set(DRAW) THEN BEGIN
    PLOTS, [0,1], [n_y, n_y], color=30, /NORMAL
    PLOTS, [pXRg(0),pXRg(1)], [n_y, n_y], color=120, /NORMAL
  ENDIF

end

pro papco_cursor_showselection_hair_cursor, ERASE=ERASE, DRAW=DRAW
  
  papco_cursor_showselection_horz_line, ERASE=ERASE, DRAW=DRAW
  papco_cursor_showselection_vert_line, ERASE=ERASE, DRAW=DRAW
  
end

pro papco_cursor_showselection_box, ERASE=ERASE, DRAW=DRAW
  
  COMMON PLOT_COMPOSER, widgetData
  COMMON MOUSE_SELECT,old,click,pnlNr,n_x,n_y   
  
;-- draw / erase the box ------------------------------------------------------
  IF keyword_set(ERASE) THEN BEGIN
     POLYFILL, [click(0), click(0), old(0), old(0)], $
	       [click(1), old(1),   old(1), click(1)], color=30, /NORMAL
     PLOTS,    [click(0), click(0), old(0), old(0),   click(0)], $
	       [click(1), old(1),   old(1), click(1), click(1)], $
	       color=0, /NORMAL  
  ENDIF
  
  IF keyword_set(DRAW) THEN BEGIN
    POLYFILL, [click(0), click(0), n_x, n_x], $
	      [click(1), n_y, n_y, click(1)], color=30, /NORMAL
    PLOTS,    [click(0), click(0), n_x, n_x, click(0)], $
	      [click(1), n_y, n_y, click(1), click(1)], $
              color=0, /NORMAL
  ENDIF  
  
end ;of papco_cursor_showselection routines

;******************************************************************************
;* PROCEDURE:   PRO papco_cursor_SetMouseDataStruct
;*
;* DESCRIPTION: Initialisation of a structure, that contains mouse
;*              information (needed by papco_cursor_HairCursor)
;*
;*
;* INPUTS:      none
;*
;* OUTOUTS:     initialisation of common block MOUSE_INFORMATION
;*
;* KEYWORDS:    none
;*
;* MODIFICATION HISTORY:  written september 1997, Torsten Heise
;******************************************************************************
PRO papco_cursor_SetMouseDataStruct

COMMON MOUSE_INFORMATION, mouseData

tmp={MOUSE_INFORMATION, $
     BuildInMouseAction:0, $              ;all modules can handle
                           $              ;this mouse-action (like zoom)
     MouseActionTyp:0, $                  ;typ of mouse action (0=zoom)
     CurrentPosition:[0.0, 0.0], $        ;current Position of the mouse      
     CurrentPanelNr:-1, $                 ;Current PanelNr
     CurrentData:"", $
     LastPosition:[0.0, 0.0], $           ;Position of the last click
     LastPanelNr:-1, $                    ;Last PanelNr
     LastData:""}

mouseData=tmp

END   ;of papco_cursor_SetMouseDataStruct

;******************************************************************************
;* PROCEDURE:   PRO papco_cursor_HairCursor
;*
;* DESCRIPTION: draw a hair-cursor in the selected panel and show the
;*              x and y values.
;*
;* INPUTS:      panelNr - number of the selected panel
;*              event   - a cursor event
;*
;* KEYWORDS:    ERASE - delete the hair cursor from screen
;*              INXY - an input in event x,y for  the time to draw a
;*                     cursor. To be used by other calkling programs.
;*
;* MODIFICATION HISTORY:  written september 1997, Torsten Heise
;* modified February 2001 R. Friedel addad TAI
;******************************************************************************
PRO papco_cursor_HairCursor, panelNr, event, seconds, yvalue, $
                             ERASE=ERASE, INXY=INXY

COMMON PLOT_COMPOSER, widgetData
COMMON MOUSE_INFORMATION, mouseData
COMMON PAPCO_GRAPHICS, papco_drawn_window     ;current graphics output array
COMMON zscale, zscl			      ;man/auto zscaling
COMMON mjdt, mjdt_start, mjdt_end             ;papco time range of plot
COMMON papco_color_names  
  
;-- set oldmouse/data info ---------------------------------------------------
old=mouseData.LastPosition
oldData=mouseData.LastData+'% % % %'
result=str_sep(oldData,'%')
oldData_1=result(0) & oldData_2=result(1)

;-- set drawing params  -------------------------------------------------------
drawColor = red & xorColor = white & erasecolor = background
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
xOffset=chrs(0) & yOffset=chrs(1)*1.75

;-- erase old information on reset --------------------------------------------
IF keyword_set(ERASE) THEN BEGIN
    IF old(0) ne 0 and old(1) ne 0 THEN BEGIN
      DEVICE, SET_GRAPHICS=6        ; set XOR-Mode for graphics
      XYOUTS, old(0)+xOffset, old(1)+yOffset, oldData_1, $
        color=xorcolor, /NORMAL, charsize=1.5, FONT=5
      DEVICE, SET_GRAPHICS=3        ; set copy-mode for graphics
      XYOUTS, 0.5, 1-chrs(1)*1.5, oldData_2, $
        color=eraseColor, /NORMAL, charsize=1.5, FONT=5, ALIGN=0.5
      mouseData.LastPosition=[0.0]
    ENDIF
    RETURN
ENDIF

;-- is the mouse still in the initially selected panel ? ----------------------
IF panelNr NE widgetData.mouse_panelNr THEN RETURN

;-- compute data at mouse position --------------------------------------------
if keyword_set(INXY) then begin ;use input time
    newData=inxy 
    new=papco_Data_To_Normal(panelNr, newData)
    newDevice=convert_coord(new,/normal,/to_device)
    event={x:fix(newDevice(0)),y:fix(newDevice(1))}
endif else begin              ; calculate coords from cursor event data
    new=convert_coord([event.x, event.y],/device,/to_normal)
    newData=papco_Normal_To_Data(panelNr, [new(0), new(1)])
endelse

time_str=papco_cursor_time(seconds, panelNr)
time_mjdt=papco_cursor_time(seconds, panelNr,/mjdt)
yval_str = varprt(yvalue)

plot=widgetData.plotsDrawn(panelNr)
  
zrange=[plot.ZSCL_MIN, plot.ZSCL_MAX] & zlog=plot.zlinlog
zval=papco_get_zvalue(event.x, event.y, zrange, LOG=zlog)
zval_str = varprt(zval)

click_Data_1='Y: '+ yval_str + '  Z: '+zval_str+'!C' + time_str
click_Data_2='Y-val: '+ yval_str +'  '+ time_str + '  Z-val: '+zval_str

;-- show data at cursor position and at top of plot ---------------------------
;   show info at top in drawcolor, XOR info at cursor!

;-- erase old information -----------------------------------------------------
IF old(0) ne 0 and old(1) ne 0 THEN BEGIN
    DEVICE, SET_GRAPHICS=6        ; set XOR-Mode for graphics
    XYOUTS, old(0)+xOffset, old(1)+yOffset, oldData_1, $
      color=xorColor, /NORMAL, charsize=1.5, FONT=5
    DEVICE, SET_GRAPHICS=3        ; set copy-mode for graphics
    XYOUTS, 0.5, 1-chrs(1)*1.5, oldData_2, $
      color=eraseColor, /NORMAL, charsize=1.5, FONT=5, ALIGN=0.5
ENDIF 

;-- show new information ------------------------------------------------------
DEVICE, SET_GRAPHICS=6        ; set XOR-Mode for graphics
XYOUTS, new(0)+xOffset, new(1)+yOffset, click_Data_1, $
    color=xorcolor, /NORMAL, charsize =1.5, FONT=5
DEVICE, SET_GRAPHICS=3        ; set copy-mode for graphics
XYOUTS, 0.5, 1-chrs(1)*1.5, click_Data_2, $
    color=drawColor, /NORMAL, charsize =1.5, FONT=5, ALIGN=0.5
mouseData.LastPosition=[new(0), new(1)]
mouseData.LastPanelNr=panelNr
mouseData.LastData=click_Data_1+'%'+click_Data_2

END	;papco_cursor_HairCursor

;******************************************************************************
;* PROCEDURE:
;*      PRO Cursor_Reset
;*
;* DESCRIPTION:
;*	This resets the mouse-selection:
;*	If a zoom-rectangle or a vertical slicing-line was displayed, it is
;*	erased.
;*	If the keyword 'FULLY' is specified, the slice-window and the zoom-
;*	window are closed.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	FULLY	if set, the selection-windows are closed
;*
;* CALLING SEQUENCE:
;*	papco_cursor_Reset, /FULLY
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     march 1996, Haje Korth
;******************************************************************************
PRO papco_cursor_Reset, FULLY=FULLY

  COMMON PLOT_COMPOSER, widgetData
  
  papco_cursor_ShowSelection, widgetData.mouse_panelNr, 0, 0, 0, 0, $
    /ERASE_ONLY
  
  widgetData.mouse_clicked=0
  widgetData.mouse_clicked_position=[0,0]
  widgetData.mouse_position=[0,0]
  widgetData.mouse_PanelNr=-1
  
  return
  
  IF KEYWORD_SET(FULLY) THEN BEGIN ; close the papco cursor-windows...
    papco_vert_slice, /CLOSE
    papco_cursor_Rect, /CLOSE
    papco_write_paneldata, /CLOSE
    papco_cursor_HairCursor, /ERASE
  ENDIF
  
END

;******************************************************************************
;* FUNCTION
;*      papco_slice_value
;*
;* DESCRIPTION:
;*	Returns the slice value assigned to slice_name. If slice_name
;*      is not uniquely found, 0 is returned.
;*
;* INPUTS:
;*	slice_name    string of slice name
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	result=papco_slice_value(slice_name)
;*
;* MODIFICATION HISTORY:
;*     written october 1997, Reiner Friedel
;******************************************************************************
FUNCTION papco_slice_value,slice_name

  COMMON PLOT_COMPOSER, widgetData

  slice_name=strsubst(slice_name,'_',' ') 
  slice_name_list=widgetData.mouseAction.NAME
  result=strpos(slice_name_list,slice_name)
  index=where(result ne -1,count)

  IF count eq 1 THEN RETURN, 2^(index(0)-1) ELSE RETURN, 0



END
