;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_draw.pro
;*
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:
;*    Contained are procedures for drawing the plots that were previously
;*    created by papco. There are routines for drawing the plots to the
;*    screen, to the printer or to a file as well as routines for clearing
;*    the plot-widget.
;*
;* FILES BELONGING TO THE MAIN-WINDOW:
;*
;*      see header of papco.pro
;*
;*    Those procedures are contained:
;*
;*	 PRO papco_Clear_Drawing
;*	     -> erase the drawing-widget
;*
;*	 FUNCTION papco_printerDescription, printerInfo, $
;*		  PRINTER=PRINTER, FILE=FILE
;*           -> return verbose printer-description, substitute templates...
;*
;*	 FUNCTION papco_canYouPlot
;*	     -> check, if a plot can be done - if not, tell the user !
;*
;*	 PRO papco_DrawPlots
;*	     -> draw to screen
;*
;*	 PRO papco_PrintPlots, printerInfo, PRINTER=PRINTER, FILE=FILE
;*	     -> output plots to printer
;*
;*       FUNCTION papco_Draw_CheckResult, panelKind, panelVector, orbitNr, $
;*				 typeVector, channel
;*           -> check, if an error occured while reading data
;*
;*       PRO papco_Draw_ErrorPanel, panelVector, textArray
;*	     -> draw an empty plot.
;*
;*       PRO papco_Draw_IdentStamp
;*           -> draws an indentification stamp of papco version, time, product
;*
;*       PRO papco_DrawPlots_Draw, SCREEN=SCREEN
;*	     -> do the plotting.
;*
;* MODIFICATION HISTORY:
;*     august,september 1995, written by A.Keese
;*     february 1996, Haje Korth
;*     october 1996, Reiner Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Clear_Drawing
;*
;* DESCRIPTION:
;*	The drawing-widget, widgetData.dw_draw is erased
;*	Additionally, the variables ' controlling the plots-drawn-info
;*	and the display of the timerange are reset.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_Clear_Drawing
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
PRO papco_clear_drawing

COMMON PLOT_COMPOSER, widgetData
COMMON PAPCO_COLOR_NAMES

draw_win_name='papco_draw_window_' + $
    string(widgetData.no_draw_window,format="(i2.2)")

IF XREGISTERED(draw_win_name, /NOSHOW) THEN BEGIN
    PAPCO_SET_PLOT, 'X'
    WIDGET_CONTROL, widgetData.dw_draw,  GET_VALUE=drawIndex
    WSET, drawIndex
    papco_SetColorTable
    widgetData.timeRangeDrawn=0
    widgetData.numberOfPlotsDrawn=0
    widgetData.need_to_draw=1
    ;papco_Cursor_Reset, /FULLY
    erase, color=0
ENDIF

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_canYouPlot
;*
;* DESCRIPTION:
;*      check, if all data needed to plot is specified
;*      For plotting, those data are needed:
;*	    an orbit-number
;*	    at least on plot-description
;*
;*	If a plot can not be done, a message-box tells the user about this
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      a Boolean Integer (0 or 1)
;*	1	  If a plot can be done
;*	0	  if a plot can't be done
;*
;* CALLING SEQUENCE:
;*	IF papco_CanYouPLot() THEN $
;*	   ; do the plot
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_canYouPlot

  COMMON PLOT_COMPOSER, widgetData

;--- check, if plots are specified
  numberOfPlots=widgetData.numberOfPlots
  IF numberOfPlots EQ 0 THEN BEGIN
     dummy=messageBox(['No plot product specified', $
		       'There is nothing to do!'], $
		       ['Go back...'], /center, title='Plot-Product missing')
     RETURN, 0
  ENDIF


;--- check, if time is chosen
  WIDGET_CONTROL, widgetData.ef_orbit, GET_VALUE=orbitNr ; this returns a
							 ; string-array
  orbitNr=fix(orbitNr(0))
  IF (orbitNr EQ 0) AND ((widgetData.minTime.mjd EQ widgetData.maxTime.mjd) $
                    AND  (widgetData.minTime.t EQ widgetData.maxTime.t)) $
     THEN BEGIN
     dummy=messageBox(['You have composed '+strtrim(string(numberOfPlots), 2)+$
                       ' plots', $
                       'but defined no time-range'], $
                      ['Go back...'], title='Time-info missing')
    RETURN, 0
  ENDIF
  RETURN, 1

END

;******************************************************************************
; assumes, that a plot can be done - no error checking
PRO papco_DrawPlots

  COMMON PLOT_COMPOSER, widgetData
  COMMON MOUSE_INFORMATION, mouseData

  IF NOT papco_canYouPlot() THEN RETURN

  PAPCO_SET_PLOT, 'X'
  WIDGET_CONTROL, widgetData.dw_draw,  GET_VALUE=drawIndex
  WSET, drawIndex

  papco_SetColorTable
  papco_Set_SubTable,0  ;set default sub color table

  if widgetData.need_to_draw eq 1 then papco_clear_drawing
  mouseData.LastPosition=[0,0]  ;reset mouse position for new draw
  papco_DrawPlots_Draw, /SCREEN

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_PrintPlots, printerInfo, PRINTER=PRINTER, FILE=FILE
;*
;* DESCRIPTION:
;*	this procedure is called, when the panels shall be drawn to a
;*      printer or a file.
;*
;* INPUTS:
;*	printerInfo	a papco_PRINTER_INFO-structure instance
;*			this is the description of the printer or file
;*			to which the plots are to be made.
;*
;* KEYWORDS:
;* 	PRINTER		use the printer that is specified by printerInfo
;*	FILE		print to the file that is specified by printerInfo
;*			If printerInfo is a default-printer, you have to
;*			set one of these keywords.
;*      SLICE           indicates caller is slice and contains plot call
;*                      for slice
;*      ZBUFFER         put drawing in zbuffer, no file output. Used
;*                      for output data option.
;*
;* CALLING SEQUENCE:
;*	papco_PrintPlots, userSelectedPrinter
;*	papco_PrintPlots, defaultPrinterInfo, /PRINTER
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     february 1996, Haje Korth
;*     november 1997, Reiner Friedel
;******************************************************************************
PRO papco_PrintPlots, printerInfo, PRINTER=PRINTER, FILE=FILE, $
                      SLICE=SLICE, ZBUFFER=ZBUFFER, CLOSE_DEV=CLOSE_DEV

COMMON plot_composer, widgetdata
COMMON mjdt, mjdt_start, mjdt_end
COMMON papco_graphics, papco_drawn_window

IF KEYWORD_SET(ZBUFFER) THEN BEGIN
    PAPCO_SET_PLOT, 'Z'
    device, SET_RESOLUTION=widgetData.default_draw_size
    papco_DrawPlots_Draw
    device,/close
    PAPCO_SET_PLOT,'X'
    return
ENDIF

usePrinter=printerInfo.usePrinter
filetype=printerInfo.filetype
askForFileName=printerInfo.askForFileName

IF KEYWORD_SET(SLICE)   THEN slice=SLICE else slice='papco'
IF KEYWORD_SET(PRINTER) THEN usePrinter=1
IF KEYWORD_SET(FILE)    THEN usePrinter=0
IF KEYWORD_SET(FILE) AND KEYWORD_SET(PRINTER) THEN stop
if keyword_set(CLOSE_DEV) then close_dev=CLOSE_DEV else close_dev=[0,0]

IF NOT papco_canYouPlot() THEN RETURN

WIDGET_CONTROL, widgetData.ef_orbit, GET_VALUE=orbitNr
orbitNr=fix(orbitNr(0))

!P.CHARSIZE=widgetData.default_charsize &  !P.FONT=-1

fName=papco_getenv('PAPCO_HOME')+'papco_print_tmp.ps'

;if not first of several calls to print, initialize
if close_dev(0) ne 0 then goto, skip_init

;-- here we can either print ps. png or loop to do both. ----------------------
case filetype of
    0: begin
      do_type=[1,0]
      if widgetData.default_ps_eps then ext='.eps' else ext='.ps'
    end
    1: begin
      do_type=[0,1]
      ext='.png'
    end
    2: begin
      do_type=[1,1]
      if widgetData.default_ps_eps then ext='.eps' else ext='.ps'
      ext=[ext,'.png']
    end
endcase
n_ext=n_elements(ext)  &  filter=strarr(n_ext)
for i=0,n_ext-1 do filter(i)='*'+ext(i)

if useprinter then begin       ; send to printer, use temporary print file
    !P.CHARSIZE=widgetData.default_charsize
    !P.FONT=-1

    os_printer= strtrim(printerInfo.printerData[1,printerInfo.printer],2) $
      eq 'os_printer'

    if os_printer eq 0 then begin
        papco_set_plot, 'PS', /interpolate
    endif else begin
        set_plot, 'printer'
    end

    ;; set_device sets the page dimensions, etc
    set_device, fName, SLICE=SLICE
    papco_SetColorTable

    if slice eq 'papco' then begin  ; do papco draw loop or call slice draw
        papco_DrawPlots_Draw
    endif else begin
        slicedraw_call=slice
        callgood=execute(slicedraw_call)
        IF callgood NE 1 THEN BEGIN
            message, 'ERROR - problem with call to slice:', /cont
            print, '  ', slicedraw_call
        END
    endelse
    ; now, print the file
    map_toOut=INDGEN(N_ELEMENTS(printerInfo.printerData(0,*)))
    out=map_toOut(printerInfo.printer)
    device,/close

    if !d.name eq 'PRINTER' then begin
       device, /close_document
    endif else begin
       message,'Printing ps file:',/cont
       print,'  '+printerInfo.printerData(1,out)+' '+fName
       spawn, printerInfo.printerData(1,out)+' '+fName
    endelse
    message,'done.', /cont

  endif else begin              ; send to file, use filename or template

    if askForFileName then begin ; go get filename interactivly
      fName=dialog_pickfile(/FIX_FILTER, TITLE='Output Plot to File:', $
                            DIALOG_PARENT=widgetData.main_base, $
                            PATH=widgetData.outputFilePath, $
                            FILTER=filter, GET_PATH=pathSelected)
      widgetData.outputFilePath=pathSelected
      if fname eq '' then begin
        message,'no output name!', /cont  &  return
      endif
    endif else begin ; or substitute from file template
      fname=papco_PrinterDescription(printerInfo, /FILE)
    endelse
    papco_sepdir,fname,dir, file, ext
    fln=dir+file

    if do_type(0) then begin    ; do to postscript file
      !P.CHARSIZE=widgetData.default_charsize &  !P.FONT=-1
      widgetData.need_to_draw=1
      papco_set_plot, 'PS',/interpolate
      papco_SetColorTable
      if widgetData.default_ps_eps then ext='.eps' else ext='.ps'
      fname=fln+ext
      if close_dev(0) eq 0 then set_device,fName, SLICE=SLICE
      if slice eq 'papco' then begin  ; do papco draw loop or call slice draw
        papco_DrawPlots_Draw
      endif else begin
        slicedraw_call=slice
        callgood=execute(slicedraw_call)
      endelse
      message,'Written ps file to', /cont
      print,'  '+fname
      device,/close
    endif

    if do_type(1) then begin    ; do to png file
      !P.CHARSIZE=widgetData.default_charsize &  !P.FONT=-1
      widgetData.need_to_draw=1
      papco_set_plot, 'Z'
      papco_SetColorTable
      fname=fln+'.png'
      if close_dev(0) eq 0 then set_device,fName, SLICE=SLICE
      if slice eq 'papco' then begin  ; do papco draw loop or call slice draw
        papco_DrawPlots_Draw
      endif else begin
        slicedraw_call=slice
        callgood=execute(slicedraw_call)
      endelse
      image=tvrd()              ; read the image
      ;see if annotation graphics buffer needs to be added
     ;; if widgetData.annotateFileName ne '' then begin
     ;;   index=where(papco_drawn_window ne 0, c)
     ;;   if c ne 0 then image(index)=papco_drawn_window(index)
     ;; endif
      papco_write_png, fName, image
      device,/close
    endif

    PAPCO_SET_PLOT,'X'

  endelse

  skip_init:

  ;if last of several calls, then do the actual ouput and close devices
  if close_dev(0) ne close_dev(1) then goto, no_output

; restore original charsize (.ps might modify)
  !P.CHARSIZE=widgetData.default_charsize &  !P.FONT=-1

  no_output:

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Draw_CheckResult, panelKind, panelVector, orbitNr, $
;*					 typeVector, channel
;*
;* DESCRIPTION:
;*      Check, if the plot-data could be read successfully.
;*      The read-routines define a common-block,
;*	    COMMON get_error, get_err_no, get_err_msg
;*	that may be used to find out if an error occured. If an error
;*	occured, 'get_err_no NE 0', and then get_err_msg contains the
;*	error-description.
;*	If an error is found, an empty panel is plotted and 'get_err_msg'
;*	is displayed in the empty panel. If the error occurs in the bottom
;*      panel, also draw a time axis.
;*
;* INPUTS:
;*	panelKind	a string
;*			if an error occured, the plot-type is drawn to
;*			the empty panel.
;*	panelVector	a three-element int-array
;*			this panel-vector is used to find out, where to
;*			draw an empty panel.
;*	orbitNr		an integer
;*			the orbit - number of the plot.
;*	typeVector	a four-element int-array
;*			the plot's type-vector
;*	channel		an integer
;*			the channel-field for this plot from the
;*			plot-info-structure
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      0	 No error was found
;*	1	 An error occured. In this case, the empty panel has been
;*		 drawn.
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_Draw_CheckResult, panelKind, panelVector, orbitNr, $
				 typeVector, channel,  OUTPUT = OUTPUT

COMMON get_error, get_err_no, get_err_msg

IF get_err_no EQ 0 then ret = 1 ELSE BEGIN
    IF NOT keyword_set(OUTPUT) THEN $
      papco_Draw_ErrorPanel, panelVector, $
        ['Cannot draw panel for: '+panelKind, get_err_msg]
    ret = 0
ENDELSE

return, ret

END

;******************************************************************************
PRO papco_Draw_ErrorPanel, panelVector, textArray

COMMON plot_composer, widgetData
COMMON papco_color_names
COMMON coordinateSystems, plotted_x, plotted_y
COMMON mjdt, mjdt_start, mjdt_end    ;common time limit in mjdt
COMMON shift_label, down             ;common for x-axis label shifting

panelset, panelVector
xl=!p.position(0)  &  xr=!p.position(2)
yb=!p.position(1)  &  yt=!p.position(3)
plotted_x=!x       &  plotted_y=!y

plots, [xl, xr, xr, xl, xl], [yb, yb, yt, yt, yb], COLOR=1, /NORMAL
if widgetData.default_error_box then begin
    plots, [xl, xr], [yb, yt], COLOR=1, LINESTYLE=1, /NORMAL
    plots, [xl, xr], [yt, yb], COLOR=1, LINESTYLE=1, /NORMAL
    tmp=convert_coord(xl, yt, /NORMAL, /TO_DEVICE)
    text='!C'
    for i=0, N_ELEMENTS(textArray)-1 DO text=text+'  '+textArray(i)+'!C'
    xyouts, tmp(0), tmp(1), text, COLOR=red, /DEV, charsize=!P.CHARSIZE
ENDIF

xut1=mjdt_start.t
xut2=long(mjdt_end.mjd-mjdt_start.mjd)*86400l+mjdt_end.t
extra_plotPar={xrange:[xut1,xut2], color:1, xtickformat:'noticks', $
                yrange:[0,1], ystyle:5}

down=0
papco_draw_time_axis, panelVector, OVERPLOT=0, _extra=extra_plotPar


END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Draw_IdentStamp
;*
;* DESCRIPTION:
;*	This routine draws a identification string in the bottom left corner
;*	of the plot window to identify papco version, date and time of plot,
;*	and product (.prod) filename if specified
;*
;* INPUTS:
;*	none	 all necessary data are contained in COMMON PLOT_COMPOSER.
;*
;* KEYWORDS:
;*	none
;*
;* CALLING SEQUENCE:
;*	papco_Draw_IdentStamp
;*
;* MODIFICATION HISTORY:
;*     written september 1996, Reiner Friedel
;******************************************************************************
PRO papco_Draw_IdentStamp

  COMMON PLOT_COMPOSER, widgetData

  ver=papco_Version()
  now=strmid(systime(0),4,21)
  product=widgetData.saveFileName

  IF NOT(widgetData.default_draw_ident) then $
     xyouts,0.008,0.5,ver+' ('+ now + ') (' + product+')', $
         /normal,color=1,orientation=90,charsize=0.5*!P.CHARSIZE,alignment=0.5

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Draw_Border
;*
;* DESCRIPTION:
;*	This routine draws a border around the active papco plot area.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;*	none
;*
;* CALLING SEQUENCE:
;*	papco_Draw_Border
;*
;* MODIFICATION HISTORY:
;*     written October 1997, Reiner Friedel
;******************************************************************************
PRO papco_Draw_Border

  COMMON PLOT_COMPOSER, widgetData

  return
  !P.POSITION=[0.1,0.1,.9,.9]

  plot,findgen(10),/nodata,xstyle=5,ystyle=5,/NOERASE
  plots,0,0,/NORMAL
  plots,0,1,/NORMAL,/CONTINUE,THICK=2,COLOR=1
  plots,1,1,/NORMAL,/CONTINUE,THICK=2,COLOR=1
  plots,1,0,/NORMAL,/CONTINUE,THICK=2,COLOR=1
  plots,0,0,/NORMAL,/CONTINUE,THICK=2,COLOR=1

END

;******************************************************************************
;* FUNCTION:
;*      papco_check_plotinfos
;*
;* DESCRIPTION:
;*	check whether the entry in widgetData.old_plotinfos(plotNr) is the same
;*      as that in widgetData.plotinfos(plotNr). Returns true (1) if they
;*      are the same.
;*
;* INPUTS:
;*	plotNr  -  the number of the plot intry to be checked.
;*
;* KEYWORDS:
;*	none
;*
;* CALLING SEQUENCE:
;*	result=papco_check_plotinfos(plotNr)
;*
;* MODIFICATION HISTORY:
;*     written May 1998, Reiner Friedel
;******************************************************************************
function papco_check_plotinfos,plotNr

  COMMON PLOT_COMPOSER, widgetData

  if widgetData.need_to_draw then return,0

  old_plotinfo=widgetData.old_plotinfos(plotNr)
  new_plotinfo=widgetData.plotinfos(plotNr)

  old_tags=TAG_NAMES(old_plotinfo)

  check=1
  for i=0,n_elements(old_tags)-1 do begin
    for j=0,n_elements(old_plotinfo.(i))-1 do begin
      if old_plotinfo.(i)(j) ne new_plotinfo.(i)(j) then check=(check-1)>0
    endfor
  endfor

  return,check

end

;******************************************************************************
;* PROCEDURE:
;*      papco_clear_panel
;*
;* DESCRIPTION:
;*	Clears the position indicated by panelVector by filling with
;*      background color.
;*
;* INPUTS:
;*	panelVector  -  position of the panel
;*
;* KEYWORDS:
;*	RIGHT_SIDE   - only clear the right side label area
;*
;* CALLING SEQUENCE:
;*	result=papco_clear_panel,panelVector
;*
;* MODIFICATION HISTORY:
;*      written April 1998, Reiner Friedel
;******************************************************************************
PRO papco_clear_panel, panel, RIGHT_SIDE = RIGHT_SIDE

COMMON PLOT_COMPOSER, widgetData
COMMON papco_color_names

if !D.name eq 'PS' $
	or !d.name eq 'PRINTER' then col=white else col=background

IF keyword_set(RIGHT_SIDE) THEN BEGIN
    panelset, panel & pos=!p.position
    polyfill, [pos(2)+0.02,pos(2)+0.02,1,1], [pos(1),pos(3),pos(3),pos(1)], $
              /NORMAL, COLOR=col
    return
ENDIF

panelset, panel, /TOTAL & pos=!p.position
if panel(0) eq 0 then pos(1)=0
polyfill, [pos(0),pos(0),pos(2),pos(2)], [pos(1),pos(3),pos(3),pos(1)], $
          /NORMAL, COLOR=col

end

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_DrawPlots_Draw, SCREEN=SCREEN
;*
;* DESCRIPTION:
;*	This routine does the plotting.
;*	Before it is called, the plotting-device has to be set.
;*	The routine does not check, if all data needed to plot is specified.
;*	The latter has to be done before calling this procedure.
;*
;*	For each plot, the parameters for the reading/plotting-routines
;*	are set, then the data are read and afterwards the plots are done.
;*	If the output is done to the screen (must be indicated by the
;*	SCREEN-keyword), information about the plots are stored in the
;*	widgetData.plotsDrawn - array.
;*
;* INPUTS:
;*	none	 all necessary data are contained in COMMON PLOT_COMPOSER.
;*
;* KEYWORDS:
;* 	SCREEN	 used to tell this procedure, whether it plots to the display
;*		 or not.
;*
;* CALLING SEQUENCE:
;*	papco_DrawPlots_Draw
;*	papco_DrawPlots_Draw, /SCREEN
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     changed july 1997, Haje Korth, Thickness and Linestyle added
;******************************************************************************
PRO papco_DrawPlots_Draw, SCREEN=SCREEN

COMMON PLOT_COMPOSER, widgetData
COMMON ADDINS, plot_types
COMMON ISREAD, IsReadArr
COMMON get_error, get_err_no, get_err_msg

COMMON yscale, yscl
COMMON zscale, zscl
COMMON time, xut1,xut2
COMMON mjdt, mjdt_start, mjdt_end
COMMON colors, r_orig,g_orig,b_orig,r_curr,g_curr,b_curr
COMMON papco_color_names
COMMON coordinateSystems, plotted_x, plotted_y
COMMON papco_DRAW, drawWidgetData
COMMON papco_graphics, papco_drawn_window
COMMON right_side_1, rsl_arr, rsl_c

COMMON extra, extra_PlotInfo

;---reset right side label count to zero, used by papco_rs_lbl ----------------
rsl_c = 0

;--- make array of PAPCO fixed colors. ----------------------------------------
pcol=bytarr(17)
pcol=fixed_color_index

;--- Set verbose flag in  get_error common block. -----------------------------
default_verbose=widgetData.default_verbose

;--- what is the highest plot-panel-number ? ----------------------------------
numberOfPlots=widgetData.numberOfPlots
maxPlotNo=0
FOR i=0, numberOfPlots-1 DO $
    maxPlotNo=maxPlotNo > (widgetData.plotInfos(i).panelPosition+ $
			   widgetData.plotInfos(i).panelHeight)

;--- initialize y-scales and z-scales------------------------------------------
;    four element array for each plot: 0  0 disables, 1 enables user limits
;                                      1  low limit
;                                      2  high limit
;                                      3  0 linear, 1 log
  max_PanelPosition=max(widgetData.plotInfos.panelPosition)
  yscl=fltarr(max_PanelPosition+1,4)
  zscl=fltarr(max_PanelPosition+1,4)
  yscl(*,0)=0 & zscl(*,0)=0 ;default to auto, no user limits

;--- check the orbit ----------------------------------------------------------
  orbitErrorString='No orbit specified'
  if widgetData.time_choice_action eq 1 then begin
    WIDGET_CONTROL, widgetData.ef_orbit, GET_VALUE=orbitNr
    orbitNr=fix(orbitNr(0))

    IF widgetData.default_orbitInfo.number NE orbitNr THEN BEGIN
      IF orbitNr NE 0 THEN BEGIN ; if orbit has not been read yet, read it now
        IF NOT papco_SetOrbit(orbitNr) THEN BEGIN
          orbitErrorString='The orbit '+strtrim(string(orbitNr), 2) + $
            ' could not be found'
          message, orbitErrorString, /cont
        ENDIF
      ENDIF
    ENDIF
  endif else orbitNr=0

;--- reset the mouse selection as being not drawn
  widgetData.mouse_selectionDrawn=0

;--- Now, do the plotting -----------------------------------------------------

;---construct automatic array of IsRead variables, one for each plottype ------
;   only when data is read for first time, and initialize to zero.
;   otherwise the exisiting one from the COMMOMN ISREAD is used again
  if widgetData.needto_read_data eq 1 then begin
     no_plot_types=n_elements(plot_types)
     IsReadArr=intarr(no_plot_types)
  endif

;--- if plotting to the display, store the number of plots --------------------
  IF KEYWORD_SET(SCREEN) THEN $
     widgetData.numberOfPlotsDrawn=widgetData.numberOfPlots

;--- Process OVERPLOTS --------------------------------------------------------
;--- Next, the plots have to be sorted. If one plot is to be plotted
;    over another one, the latter has to use the y/z-scale-information
;    of the first one. For this reason, we have to search all plots
;    that are overplots and decide, which plot will give them the
;    yscale-information.
;--- Rules:
;    If the 'overplot'-flag is set for a plot, all plots with the same
;    panel-vector are searched.
; -> If there aren't any, the plot uses its own scaling
; -> if there are more than one:
;      -> if exactly one of those is a non-overplot type, it is
;         plotted first, and its scaling-info is used.
;      -> if non of them or more than one are of a non-overplot type,
;         we've got a little problem... In this case, a warning is displayed
;         and one of them used for scaling-information. (the first
;         one in 'widgetData.plotinfos'
;
;    NEW: 11/06/2000 Addition of slice plot feature. Slice plots also
;    are associated with previously plotted plots, so need to be
;    plotted last. Treat similar to overplots, but don't need to read
;    any scaling info.

get_YScalingFrom=indgen(numberOfPlots) & plot_sequence=indgen(numberOfPlots)
overPlots=where(widgetData.plotInfos(0:numberOfPlots-1).overplot EQ 1, count)
pos=strpos(widgetData.plotInfos(0:numberOfPlots-1).panelkind,'slice_thumbs')
slicePlots=where(pos ne -1, c1)

if count gt 0 then begin
    plot_Weight=indgen(numberOfPlots) ; the heavier, the later it's plotted
    startPositions=widgetData.plotInfos(0:numberOfPlots-1).panelPosition
    endPositions=startPositions+$
      widgetData.plotInfos(0:numberOfPlots-1).panelHeight

    for i=0, count-1 do begin
      thisPlot=overPlots(i)
      find=where(startPositions EQ startPositions(thisPlot) AND $
                 endPositions EQ endPositions(thisPlot) AND  $
                 (NOT widgetData.plotInfos.overPlot), eCount)

      if eCount EQ 0 then begin
        message, 'plot '+varprt(thisPlot)+' is an overplot', /cont
        message, 'but there is no plot to define its YRange!', /cont
      endif else begin ; now, make sure that the found plots are plotted
                       ; before this one's plotted
        if eCount GT 1 then begin
          message, 'plot '+varprt(thisPlot)+' is an overplot', /cont
          message, 'but the plot to define its YRange is not unique!', /cont
       endif
       get_YScalingFrom(thisPlot)=find(0) ; use the first plot found with
                                ; equal position as provider for the yscaling
      endelse
      plot_Weight(thisPlot)=plot_Weight(thisPlot)+numberOfPlots
    endfor
    plot_sequence=plot_sequence(sort(plot_Weight))
endif

;ensure that slice plots go last by adding max weight to them.
if c1 gt 0 then begin
    plot_Weight=plot_sequence ; the heavier, the later it's plotted
    for i=0, c1-1 do begin
      result=where(plot_sequence eq slicePlots(i))
      plot_Weight(result(0))=plot_Weight(result(0))+numberOfPlots
    endfor
    plot_sequence=plot_sequence(sort(plot_Weight))
endif

star_line = '***************************************'
star_line = star_line+star_line
dash_line = '---------------------------------------'
dash_line = dash_line+dash_line
print,''
print, star_line
message,'Start of plotting loop.', /cont

WIDGET_CONTROL, /HOURGLASS         ;set cursor to hourglass for plotting
dealloc_lun,/quiet & redraws=0

; save the current state (product) to file for "back" and "forward" button
IF NOT widgetData.default_states THEN papco_save_current_state

;-- Set the interrupt field in Draw Widget to deafult N (no)
drawWidgetData.monitor->setTaskSize, numberOfPlots


;-- delete all those panels which need deleting -------------------------------
;   i.e. have been edited, first and "remember" deleted entries: any unedited
;   panel, which overlaps one of the deleted ones, needs to be redrawn too!
;   keep an array of checked panels. Of the 2nd dimension:
;                                        0:     0 not edited, 1 edited
;                                        1:     panel pos if edited
;                                        1:     panel height if edited
panel_edit=bytarr(numberOfPlots,3)

FOR i=0, numberOfPlots-1 DO BEGIN

    ;check whether the entry in widgetData.plotsdrawn(plotNr) is the same as
    ;that in widgetData.plotinfos(plotNr)
    plotNr=plot_Sequence(i)
    if not papco_check_plotinfos(plotNr) then begin
      pos=widgetData.plotInfos(plotNr).panelPosition
      height=widgetData.plotInfos(plotNr).panelHeight
      panelVector=[pos, maxPlotNo, height]
      panel_edit(i,0)=1 & panel_edit(i,1)=pos & panel_edit(i,2)=height
      message,'Clearing panel ['+varprt(pos)+','+varprt(height)+']', /cont
      redraws=(redraws+1)<1
      papco_clear_panel,panelVector
    endif
ENDFOR

;-- Loop through all panels to be plotted--------------------------------------
;   For increased speed, work out which panels have been altered since
;   the last time they were drawn - and only replot those!
numbering_count=1
FOR i=0, numberOfPlots-1 DO BEGIN

    drawWidgetData.monitor->setProgress, i

    draw_error=0
;-- set the index in 'widgetData.plotInfos' of this plot
    plotNr=plot_Sequence(i)
    currPlotInfo=widgetData.plotInfos(plotNr)
    widgetData.current_plotNr=plotNr

;--- calculate the panel-vector
    pos=widgetData.plotInfos(plotNr).panelPosition
    height=widgetData.plotInfos(plotNr).panelHeight
    panelVector=[pos, maxPlotNo, height]
    cov=bytarr(maxPlotNo) & this_cov=cov
    this_cov(pos:pos+height-1)=1 ;set the panel occupation of this panel

;--- put out diagnostic on panel
    pVStr=string(panelVector,format="('(',i2.2,',',i2.2,',',i2.2,')')")
    if orbitNr ne 0 then $
      oStr=string(orbitNr,format="('    orbit: ',i4.4)") else $
      oStr= '    orbit: none'
    plotnr_str=string(plotnr+1,format="(i2.2)")
    panel_kind=currPlotInfo.panelkind

    print,'' & print,dash_line
    message,'Doing Plot No. '+plotnr_str+', '+panel_kind, /cont
    print,'                        panelVector: ', pVStr, oStr
    print,''

    drawWidgetData.monitor->setMessage, plotnr_str+': '+panel_kind

;-- check entries in  panel_edit array to see if panel needs redrawing.
;   It will need a redraw if a) the entry was edited
;                            b) the current panel overlaps a deleted
;                            one
;   Can be overuled by defaults_redraw = 0

    IF widgetData.default_redraw THEN BEGIN
        if not panel_edit(i,0) then begin ; = 0 if not edited
            has_overlap=0
            ;check overlaps with other edited panels
            for j=0,numberOfPlots-1 do begin
                if panel_edit(j,0) then begin
                   other_cov=cov
                   other_cov(panel_edit(j,1):panel_edit(j,1)+panel_edit(j,2)-1)=1
                   overlap=this_cov AND other_cov
                   index=where(overlap ne 0,c)
                   if c ne 0 then has_overlap=has_overlap+1
                ENDIF
            ENDFOR
            if not has_overlap then begin
                message,'Panel unchanged - no redraw needed', /cont
                goto,next_panel
            ENDIF
        ENDIF
    ENDIF

;-- check if the current panelkind is entered in the plot_types table
    panelkind=strchminus(strlowcase(currPlotInfo.panelKind))
    result=strpos(strlowcase(plot_types),panelkind,0)
    typeindex=where(result eq 0,count)
    if count eq 0 then begin
      papco_Draw_ErrorPanel, panelVector, $
        ["I can't plot "+currPlotInfo.panelKind, 'Module not loaded']
      message,'Module not loaded for '+currPlotInfo.panelKind, /cont
      goto,next_panel
    endif

;-- set up default graphics plot system variables. These assure a uniform
;   "look" of all panels, to create a uniform output. The user may of course
;   overwrite these defaults by setting the corrsponding keywords in his/her
;   own plot routine. The use of these defaults though is highly recommended.
    !P.CHARTHICK = widgetData.default_charthick
    !P.CHARSIZE = widgetData.default_charsize
    !P.FONT      = -1
    ;try new fonts for draw
    ;!P.FONT      = 1
    ;device, set_font = 'Helvetica', /TT_FONT
    !P.NOERASE   = 1
    !X.TICKS     = widgetData.default_XTICKS
    !X.STYLE     = widgetData.default_XSTYLE
    !X.MINOR     = widgetData.default_XMINOR
    !X.TICKV     = 0
    !Y.TICKS     = 0
    !Y.STYLE     = 1
    !Y.MINOR     = 0
    !Y.TICKV     = 0

;-- For many panels, the ticklengths become too small to "see". Double
;   the length every 4 panels - but only for x-axis!.
    ticklength_mult = (numberOfPlots / 4) + 1
    !X.TICKLEN   = widgetData.default_TICKLEN * ticklength_mult

;-- option to automatically calculate ytick numbers in order to avoid to
;   many ticks if panels get too narrow. Maximum is set to 8.
    IF currPlotInfo.manualyscaling EQ 1 THEN BEGIN ;manual y scaling selected
        !Y.TICKS=currPlotInfo.yscaleLb
    ENDIF ELSE BEGIN
        IF currPlotInfo.yLinLog EQ 1 THEN !Y.TICKS=0 ELSE BEGIN
            panelset,panelVector
            ypix2=convert_coord(0,!p.position(3),0,/normal,/to_device)
            ypix1=convert_coord(0,!p.position(1),0,/normal,/to_device)
            pix_height=ypix2(1)-ypix1(1)
            pix_cm_height = pix_height / !D.Y_PX_CM
            !Y.TICKS=( fix(pix_cm_height/0.5) ) < 8
        ENDELSE
    ENDELSE

;-- check status of cancel text field in draw_widget
    IF widgetdata.no_draw_window ne 0 then begin
        if ( drawWidgetData.monitor->isCancelled() ) then begin
            goto, end_plot_loop
        endif
    ENDIF

;-- reset internal color sub-table to default
    papco_Set_SubTable,0
;-- set up common start and end time for MJDT users
    mjdt_start=widgetData.startTime
    mjdt_end=widgetData.endTime
;--- set up common start and end time for old T90 users
    xut1=widgetData.startTime.t90
    xut2=widgetData.endTime.t90

    plotted_x = 'this has to be set by the plot-routine!'
    plotted_y = 'this has to be set by the plot-routine!'

;--- set the y-range and Z-range for this plot
;    the Y-scale-info specified by 'get_YScalingFrom' is used. If get_YScaling
;    points to another plot, that plot is assumed to have already been plotted
;    and its y-scale-info is used. Otherwise, the yscale-info from
;    currPlotInfo is used.
    if get_YScalingFrom(plotNr) NE plotNr then begin
      fromPlot=get_YScalingFrom(plotNr)
      yscl(pos,0)=1          ; enable use of user limits
      yscl(pos,1)=widgetData.plotsDrawn(fromPlot).yscl_min
      yscl(pos,2)=widgetData.plotsDrawn(fromPlot).yscl_max
;      yscl(pos,3)=widgetData.plotsDrawn(fromPlot).yLinLog
      message,'YScl for overplot: ' + $
         varprt(yscl(pos, 1))+'  '+varprt(yscl(pos, 2)), /cont
    endif else begin
      IF currPlotInfo.manualYScaling EQ 1 THEN BEGIN
        yscl(pos,0)=1       ; enable use of user limits
        yscl(pos,1)=currPlotInfo.yScaleMin
        yscl(pos,2)=currPlotInfo.yScaleMax
        message,'YScl: '+varprt(yscl(pos, 1))+'  '+varprt(yscl(pos, 2)), /cont
      endif
      yscl(pos,3)=currPlotInfo.yLinLog
      IF (currPlotInfo.manualZScaling /100) THEN BEGIN
        zscl(pos,0)=1       ; enable use of user limits
        zscl(pos,1)=currPlotInfo.zScaleMin
        zscl(pos,2)=currPlotInfo.zScaleMax
        message,'ZScl: '+varprt(zscl(pos, 1))+'  '+varprt(zscl(pos, 2)), /cont
      endif
      zscl(pos,3)=currPlotInfo.zLinLog
    endelse

    get_data_call='' ; will be stored in the 'plotsDrawn'-array
		     ; used by the slice-routines to read the data for a plot
    slice_type=-1    ; -1 means, that no plot was drawn !
         	     ; the slice-type will be stored in the plotsDrawn-array,
 	             ; and it is used by slice-routines to find out, what
 	             ; kind of slice is to be done for a panel.
    plottable=1

    IF orbitNr EQ 0 THEN BEGIN
      plottable=0
;--- call the routine which tells if plot is plottable by orbit / time
      panelkind=strchminus(currPlotInfo.panelKind)
      proc_call='plottable_'+panelkind
      if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
      if error_status ne 0 then begin ;call error handler
        papco_error_handler, $
          error_status, proc_call, panelkind, panelVector
        error_status=0 & catch, /CANCEL  ;disable error catching
        goto, next_panel
      endif
      CALL_PROCEDURE, proc_call, plottable
      catch, /CANCEL          ;disable error catching
    ENDIF

    if NOT plottable THEN BEGIN
      papco_Draw_ErrorPanel, panelVector, $
        ["I can't plot "+currPlotInfo.panelKind, $
         'without a valid orbit', orbitErrorString]
      goto, next_panel
    endif

;--- create the extra-structure containing data that will be passed to
;    the plots in the '_extra'-structure
    extra_PlotInfo={overPlot:currPlotInfo.overPlot, $
                    plots_AtThisPosition:1, $
                    subtable:currPlotInfo.channel, $
                    PSYM:currPlotInfo.psym, $
                    SYMSIZE:currPlotInfo.symsize, $
                    COLOR:pcol(currPlotInfo.color), $
                    THICK:currPlotInfo.thick, $
                    LINESTYLE:currPlotInfo.linestyle}

    ;increase line thickness for postscript, looks better
    if !D.name eq 'PS' then extra_plotInfo.THICK=extra_plotInfo.THICK*1.5

    ;find the number of plots at this position for exact
    ;overplots of previously plotted panels ONLY
    ov_count=0
    for kk=0,i do begin         ;i is current panel
      testplotNr=plot_Sequence(kk)
      if panelVector(0) eq widgetData.plotinfos(testplotNr).panelPosition and $
        panelVector(2) eq widgetData.plotinfos(testplotNr).panelHeight then $
        ov_count=ov_count+1
    endfor
    ov_count = ov_count-1
    IF ov_count NE 0 THEN message,'found '+strtrim(string(ov_count),2)+ $
            ' previous panels at this position', /cont
    extra_plotInfo.plots_AtThisPosition=ov_count

;--- call the respective plotting procedures. Makes the call to draw_plot_type
;    which is a user-supplied procedure in papcoadd_plot_type.
;    NB!! Add in here some error catching. MOST PAPCO errors occur in
;    user supplied routines - so here the error is trapped and reported,
;    so that PAPCO doesn't crash but plots thecerror in an "empty" panel. This
;    is needed for batch processing!
    FOR ii = 0, n_elements(plot_types)-1 DO BEGIN
        IF strlowcase(plot_types(ii)) EQ $
          strchminus(strlowcase(currPlotInfo.panelKind)) THEN break
    ENDFOR
    typeindex = ii
    IF typeindex NE n_elements(plot_types) THEN count = 1 ELSE count = 0

    IF count ne 0 then BEGIN
      ; set the keyword to redirect output to file if needed, and if
      ; supported by the current plot_type!

      if widgetData.output then begin
        canget=0
        panelkind=strchminus(currPlotInfo.panelKind)
        proc_call='canget_'+plot_types(typeindex(0))

        if widgetData.default_trap eq 1 then $
          catch, error_status else error_status=0
        if error_status ne 0 then begin  ;call error handler
          papco_error_handler, $
            error_status, proc_call, panelkind, panelVector, /NO_ERR_DRAW
          print,'  This error occurs when PAPCO is looking for the'
          print,'  "canget_" procedure in the module, which tells PAPCO'
          print,'  if the module supports the OUTPUT functionality.'
          print,'  Either there is an error in that routine, or that'
          print,'  routine does not exist. Either way there will be no'
          print,'  OUTPUT from this module.'
          print,''
          error_status=0 & catch, /CANCEL ;disable error catching
          goto, skip_canget
        endif

        CALL_PROCEDURE, proc_call, canget
        skip_canget:
        catch, /CANCEL          ;disable error catching

        IF canget THEN BEGIN    ;check if output feature enabled
            output=widgetdata.output
            IF widgetData.default_do_plots EQ 0 THEN output = 3
        ENDIF else output=0

      endif else output=0

      ; construct the draw call and do draw
      IsRead=IsReadArr(typeindex(0))
      panelkind=strchminus(currPlotInfo.panelKind)
      proc_call='draw_'+panelkind
      if widgetData.default_trap eq 1 then $
        catch, error_status else error_status=0
      if error_status ne 0 then begin    ;call error handler
        papco_error_handler, $
          error_status, proc_call, panelkind, panelVector
          draw_error=draw_error+1 ;set error flag to force redraw on next draw
        error_status=0 & catch, /CANCEL  ;disable error catching
        goto, next_panel
      ENDIF
      if output NE 0 then $
      CALL_PROCEDURE, proc_call, panelVector, currPlotInfo, orbitNr, $
        get_data_call, slice_type, Isread, OUTPUT=output else $
      CALL_PROCEDURE, proc_call, panelVector, currPlotInfo, orbitNr, $
        get_data_call, slice_type, Isread
      catch, /CANCEL            ;disable error catching
      IsReadArr(typeindex)=IsRead
    END

;--- Load the y/z scaling info into Plotinfo array if scaling was automatic
    IF currPlotInfo.manualYScaling EQ 0 THEN BEGIN
        yscl(pos,0)=0
        widgetData.plotInfos(plotNr).yScaleMin=yscl(pos,1)
        widgetData.plotInfos(plotNr).yScaleMax=yscl(pos,2)
    ENDIF
    IF NOT currPlotInfo.manualZScaling THEN BEGIN
        zscl(pos,0)=0       ; copy in automatic settings if set
        widgetData.plotInfos(plotNr).zScaleMin=zscl(pos,1)
        widgetData.plotInfos(plotNr).zScaleMax=zscl(pos,2)
    ENDIF

; finally, if the plot was done to the display, keep info about the drawn plot
; so that mouse-clicks can be traced. This data is used in papco_cursor.pro

    if (1) then begin ; jbf
      widgetData.plotsDrawn(plotNr).panelVector=panelVector
      ;the plotting-routines store info about the coordinate-system in
      ;plotted_x=!x and plotted_y=!y. This was necessary, because there
      ;may be multiple PLOT-commands in one plotting-routines (e.g. for
      ;drawing labels), but we need info about the date-coordinate-system.
      widgetData.plotsDrawn(plotNr).x_s=plotted_x.s
      widgetData.plotsDrawn(plotNr).data_xrange= plotted_x.crange
      widgetData.plotsDrawn(plotNr).data_yrange= plotted_y.crange
      widgetData.plotsDrawn(plotNr).x_type=plotted_x.type
      widgetData.plotsDrawn(plotNr).y_type=plotted_y.type
      widgetData.plotsDrawn(plotNr).y_s=plotted_y.s

      ;store the y/zscale-information. It will be used as y/z-range for slices.
      widgetData.plotsDrawn(plotNr).yscl_min=yscl(pos,1)
      widgetData.plotsDrawn(plotNr).yscl_max=yscl(pos,2)
      widgetData.plotsDrawn(plotNr).ylinlog=yscl(pos,3)
      widgetData.plotsDrawn(plotNr).zscl_min=zscl(pos,1)
      widgetData.plotsDrawn(plotNr).zscl_max=zscl(pos,2)
      widgetData.plotsDrawn(plotNr).zlinlog=zscl(pos,3)

      ;store, which area of the display was used by this plot.
      widgetData.plotsDrawn(plotNr).normal_xrange=$
        [!p.position(0),!p.position(2)]
      widgetData.plotsDrawn(plotNr).normal_yrange=$
        [!p.position(1),!p.position(3)]

      widgetData.plotsDrawn(plotNr).get_data_call=get_data_call
      widgetData.plotsDrawn(plotNr).slice_type=slice_type

      ;if the changed-flag EQ 1, you cannot select that orbit with the
      ;mouse-cursor. So, when the drawing/reading of data went wrong, we
      ;set that flag
      IF slice_type EQ -1 THEN widgetData.plotsDrawn(plotNr).changed=1 $
        ELSE widgetData.plotsDrawn(plotNr).changed=0

      ;have to remember some more data, as the user may change plots or orbit
      widgetData.plotsDrawn(plotNr).panelKind=currPlotInfo.panelKind
      widgetData.plotsDrawn(plotNr).orbit=orbitNr
      widgetData.plotsDrawn(plotNr).typeVector=currPlotInfo.typeVector
    ENDIF
    next_panel:
    ;do panel numbering if needed.
    if (widgetdata.default_numbering eq 0) AND $
      (not currPlotInfo.overplot) AND $
      (currPlotInfo.panelkind ne 'papco_text') then begin
      if widgetData.default_right_label eq 0 then begin ;outside numbering
        xpos=0.005  &  ypos=!p.position(1)+(!p.position(3)-!p.position(1))/2.0
      endif else begin          ;inside numbering
        xpos=!p.position(0)+0.01  &  ypos=!p.position(3)-0.015
      endelse
      number='!5['+varprt(numbering_count)+']!3'
      xyouts,xpos,ypos,number,color=red, charsize=!P.charsize*1.25,$
        /normal, align=0
      numbering_count=numbering_count+1
    endif
  ENDFOR
; end plotting loop -----------------------------------------------------------

  drawWidgetData.monitor->setProgress, i
  drawWidgetData.monitor->finished

print,dash_line & print,''
message,'End of plotting loop.', /cont
print,''

widgetData.needto_read_data=0 ;plot just done, all data read
papco_refresh,/NEED_READ

end_plot_loop:

; draw the ident string on plot
papco_Draw_IdentStamp

; draw the border
papco_Draw_border

; store the graphics window for access by cursor routine
if (papco_check_window_device()) AND redraws AND $
    widgetData.default_save_bitmap then BEGIN
    message,'saving draw window image bitmap...', /info
    papco_drawn_window=tvrd(TRUE=widgetdata.truecolor)
endif

; draw annotation if annotation filename exists!
if widgetData.annotateFileName ne '' then begin
  ; set colors that annotate can use from papco color table
    inidices=papco_get_Colorindices(3)
    fixed_color=indgen(16)+inidices(1)+1
    message,'Drawing annotation', /cont
    case !D.NAME of
      'PS' : begin
        papco_annotate, COLOR_INDICES=fixed_color, $
          LOAD_FILE=widgetData.annotateFileName, /EXIT, $
          	/inline, /batch
      end
      'PRINTER': begin
          papco_annotate, COLOR_INDICES=fixed_color, $
            LOAD_FILE=widgetData.annotateFileName, /EXIT, $
            /inline, /batch
      end
      'Z' : begin
          papco_annotate, COLOR_INDICES=fixed_color, $
            LOAD_FILE=widgetData.annotateFileName, /EXIT, $
            /inline, /batch
      end
      'X' : begin
        papco_annotate, COLOR_INDICES=fixed_color, $
          LOAD_FILE=widgetData.annotateFileName, /EXIT
      end
      'WIN': begin
        papco_annotate, COLOR_INDICES=fixed_color, $
          LOAD_FILE=widgetData.annotateFileName, /EXIT
	  end
    endcase
endif

; free all logical units assigned with get_lun that are no longer in use
dealloc_lun,/quiet

; "remember" the current state of plotinfo structures
widgetData.old_plotinfos=widgetData.plotinfos

; plot has just been drawn, a total redraw is not required
IF draw_error EQ 0 THEN widgetData.need_to_draw=0 ELSE $
  widgetData.need_to_draw=1

print, star_line
print, ''

END
