;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_draw_window.pro
;*
;* DESCRIPTION:
;*    This file contains routines for creating and maintaining papco Draw
;*    windows.
;*       - the draw-widget-construction-routines
;*       - the event-handlers.
;*
;* FILES BELONGING TO THE MAIN-WINDOW:
;*
;*      See the header in papco.pro
;*
;* COMMON-BLOCKS:
;*    One Common-Block is defined:
;* COMMON papco_DRAW, drawWidgetData
;*
;* STRUCTURES:
;*    The structures defined in this file are:
;*        drawWidgetData     -> a structure to hold the description of
;*				the current draw window
;*
;*    They are described in the file
;*	  $papco_BASE/papco_doc/papco_variables.text
;*
;*    Those procedures are contained:
;*       PROCEDURE papco_draw_window, name
;*       PROCEDURE papco_draw_window_Event, Event
;*
;* MODIFICATION HISTORY:
;*    october 1996, written by R.Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      papco_draw_window, name
;*
;* DESCRIPTION:
;*	This function is used whenever a new draw window is being created.
;*
;* INPUTS:
;*	name	the name under which the window is to be registered with
;*              the xmanager.
;*
;* KEYWORDS:
;* 	CLOSE   closes and destrys this wondow.
;*      DUMMY   makes an un-mapped dummy containing only the draw widget.
;*              used by the annotate procedure.
;*
;* OUTPUT:
;*      a long integer of the idl plot window number
;*	this number is used in directing the output of all idl plot routines
;*
;* CALLING SEQUENCE:
;*	draw_window_index=papco_draw_window()
;*
;* MODIFICATION HISTORY:
;*     October 1996, written by R.Friedel
;*     October 1998, modified by R. Friedel to include PRINT and OUTPUT
;******************************************************************************
PRO papco_draw_window, name, CLOSE=CLOSE, DUMMY=DUMMY

COMMON plot_composer, widgetdata
COMMON papco_draw, drawwidgetdata

;-- close the window ? --------------------------------------------------------
IF keyword_set(close) THEN BEGIN
    IF xregistered(name) THEN widget_control, drawwidgetdata.base, /DESTROY
    return 
ENDIF

;-- if the window does not exist, create it ----------------------------------
IF NOT XREGISTERED(name, /NOSHOW) THEN BEGIN

    ;check screensize. Limit Draw window so that it fits!
    ;limit either x or y to max - 50 pixels to allow borders.
    ;then, put action buttons either at bottom or side to allow
    ;for maximum draw window.
    ;or, if scrolling is set, allow default size, and add scroll bars
    ;if needed.
    screen=get_screen_size()
    action='side' ;default for action button placement

    needx_scroll=0 & needy_scroll=0 & scroll=0
    scr_xsize = widgetData.default_draw_size(0)
    scr_ysize = widgetData.default_draw_size(1)
    allow_scroll = widgetdata.default_scroll

    ;get x, y size requested by defaults, check for landscape/portrait
    if widgetData.default_layout eq 0 then begin
        xsize=widgetData.default_draw_size(0)
        ysize=widgetData.default_draw_size(1)
    endif else begin
        xsize=widgetData.default_draw_size(1)
        ysize=widgetData.default_draw_size(0)
    endelse

    if xsize gt screen(0)-20 then begin
        xsize=screen(0)-20  &  action='bottom'
        needx_scroll=1
    endif
    if ysize gt screen(1)-90 then begin
        ysize=screen(1)-90  &  action='side'
        needy_scroll=1
    endif

    if action eq 'side' then begin
        xsize=xsize < (screen(0)-125)
        base_column=2   &  bottom_row=0
        action_column=1 &  bottom_column=1
    endif else begin
        ysize=ysize < (screen(1)-90)
        base_column=1   &  bottom_row=1
        action_column=6 &  bottom_row=0
    endelse

    if needx_scroll and allow_scroll then begin
        scroll=1
        scr_xsize=screen(0)-30
        scr_ysize=ysize+90 < (screen(1)-30)
        xsize=widgetData.default_draw_size(0)
        ysize=widgetData.default_draw_size(1)
    endif
    if needy_scroll and allow_scroll then begin
        scroll=1
        scr_ysize=screen(1)-30
        scr_xsize=xsize+150 < (screen(0)-30)
        xsize=widgetData.default_draw_size(0)
        ysize=widgetData.default_draw_size(1)
    endif

    ;print,needx_scroll, needx_scroll, scr_xsize, scr_ysize, xsize, ysize
    if allow_scroll eq 1 then scroll=scroll else scroll=0

    if scroll then $
      base=WIDGET_BASE(COLUMN=base_column, GROUP=widgetData.main_base,$
                       title=name, SCROLL=scroll, $
                       SCR_XSIZE=scr_xsize, SCR_YSIZE=scr_ysize) $
    else $
      base=WIDGET_BASE(COLUMN=base_column, GROUP=widgetData.main_base,$
                       title=name)

    draw_base=WIDGET_BASE(base,COLUMN=1)

    dw_draw  =WIDGET_DRAW(draw_base, UVALUE='dw_draw', $
                          XSIZE=xsize, YSIZE=ysize)

    if KEYWORD_SET(DUMMY) then begin
        WIDGET_CONTROL, dw_Draw, MAP=0
        WIDGET_CONTROL, Draw_Base, /REALIZE
        XMANAGER, name, Draw_Base
        name=dw_Draw
        drawWidgetData.base=Draw_Base
        RETURN
    endif

    outst1='Product File: '+ $
      strright(widgetData.saveStatePath+widgetData.saveFileName,70)
    outst2='Annotate File: '+widgetData.annotateFileName
    lbl_Draw1=WIDGET_LABEL(Draw_Base, VALUE=strcenter(outst1,90))
    lbl_Draw2=WIDGET_LABEL(Draw_Base, VALUE=strcenter(outst2,90))

;--- set action base ----------------------------------------------------------
    bottom_base=WIDGET_BASE(base, Row=bottom_row, column=bottom_column,$
                            /ALIGN_CENTER)

;--- set action buttons -------------------------------------------------------
    Butt_Base=WIDGET_BASE(bottom_base, COLUMN=action_column)

    geometry=WIDGET_INFO(widgetData.pb_print, /GEOMETRY) ;set size for buttons
    buttonGeometry={ysize:geometry.ysize,     xsize:geometry.xsize,     $
                    yoffset:geometry.yoffset, xoffset:geometry.xoffset  }

    pb_draw=    WIDGET_BUTTON(Butt_Base, Value=strcenter('Draw', 10), $
                              UVALUE='pb_draw', _extra=buttonGeometry)

    pb_print=   WIDGET_BUTTON(Butt_Base, VALUE=strcenter('Print', 10), $
                              UVALUE='pb_print', _extra=buttonGeometry)

    pb_output=  WIDGET_BUTTON(Butt_Base,  VALUE=strcenter('Output', 10), $
                              UVALUE='pb_output', _extra=buttonGeometry)

    pb_help=    WIDGET_BUTTON(Butt_Base, Value=strcenter('Help', 10), $
			      UVALUE='pb_help', _extra=buttonGeometry)

    pb_previous=WIDGET_BUTTON(Butt_Base, Value=strcenter('<--', 10), $
                              UVALUE='pb_previous', _extra=buttonGeometry)

    pb_back=    WIDGET_BUTTON(Butt_Base,  VALUE=strcenter('Back', 10), $
                              UVALUE='pb_back', _extra=buttonGeometry)

    pb_next=    WIDGET_BUTTON(Butt_Base, Value=strcenter('-->', 10), $
                              UVALUE='pb_next', _extra=buttonGeometry)

    pb_forward= WIDGET_BUTTON(Butt_Base,  VALUE=strcenter('Forward', 10), $
                              UVALUE='pb_forward', _extra=buttonGeometry)

    pb_load=    WIDGET_BUTTON(Butt_Base,  VALUE=strcenter('Load',10), $
                              UVALUE='pb_load', _extra=buttonGeometry)

    pb_save=    WIDGET_BUTTON(Butt_Base,  VALUE=strcenter('Save',10), $
                              UVALUE='pb_save', _extra=buttonGeometry)

    pb_del_old= WIDGET_BUTTON(Butt_Base, VALUE=strcenter('Del Old', 10), $
                              UVALUE='pb_del_old', _extra=buttonGeometry)

    pb_del_all= WIDGET_BUTTON(Butt_Base, Value=strcenter('Del All', 10), $
			      UVALUE='pb_del_all', _extra=buttonGeometry)
    
;--- set draw break widget
;    --- ----------------------------------------------------
    break_base=WIDGET_BASE(bottom_base, COlumn=1, /base_ALIGN_LEFT)
    break_base_row= widget_base( break_base, /row )
    id_progress_bar= widget_draw( break_base_row, xsize=50, ysize=20 )
    id_cancel= widget_button( break_base_row, value='Cancel', uvalue='' )
    id_message= widget_label( break_base, xsize=100, ysize=20, value='ready to draw' )

    monitor= obj_new('papco_graph_monitor', $
                     idCancel=id_cancel, idDraw=id_progress_bar, $
                     idMessage=id_message, $
                     progress_Color=papco_get_color(100,100,100), background_Color=0 )
    

;;    pb_cancel_draw=CW_FIELD(break_base, COLUMN=1, $
;;                            TITLE='Cancel?', $
;;                            XSIZE=1, YSIZE=1, $
;;                            Value=strcenter('N', 1), $
;;			    UVALUE='pb_cancel_draw')

    drawWidgetData={papco_draw_window_a, $
                    base:Draw_base, $
                    product:widgetData.SAVEFILENAME, $
                    lbl_Draw1:lbl_Draw1, $
                    lbl_Draw2:lbl_Draw2, $
                    pb_del_all:pb_del_all, $
                    pb_draw:pb_draw, $
                    pb_help:pb_help, $
                    pb_del_old:pb_del_old, $
                    monitor:monitor }

;--- set position of widget and realize (top right corner) --------------------
    no_of_window=fix(strmid(name,strlen(name)-2,2))
    papco_set_plot,'X'
    device, get_screen_size=scrsize
    geo=WIDGET_INFO(base, /GEOMETRY)
    x= scrsize(0) - geo.xsize - 10* ( no_of_window-1 ) < 2000  ;; jbf's start menu is on the right of 2880pix
    y= 40+no_of_window*20
    WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y, /REALIZE

    XMANAGER, name, base, event_handler='papco_draw_window_event',/JUST_REG

    WIDGET_CONTROL, dw_Draw, /DRAW_BUTTON_EVENTS, /DRAW_MOTION_EVENTS
    widgetData.draw_window=!D.window
    widgetData.dw_Draw=dw_Draw
    widgetData.old_draw_id(widgetData.no_draw_window)=Base
ENDIF ELSE BEGIN
    outst1='Product File: '+widgetData.SAVEFILENAME
    WIDGET_CONTROL,drawWidgetData.lbl_Draw1,set_value=outst1
    outst2='Annotate File: '+widgetData.annotateFileName
    WIDGET_CONTROL,drawWidgetData.lbl_Draw2,set_value=outst2
ENDELSE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_draw_window_event, event
;*
;* DESCRIPTION:
;*      The main XMANAGER-event-handler.
;*	All events to the main-window are processed here
;*
;* INPUTS:
;*	event		an XManager-Event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	Called by XMANAGER only !
;*
;* MODIFICATION HISTORY:
;*     written october 1995, by R.Friedel
;******************************************************************************
pro papco_draw_window_event, event

COMMON Plot_Composer, Widgetdata
COMMON Papco_Draw, Drawwidgetdata

Widget_Control,Event.Id,Get_Uvalue=Ev

CASE Ev OF 

    'pb_draw' : Begin  ; Draw. -> Draw Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_draw'
    End

    'pb_previous': Begin ; Previous. -> Previous Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_previous'
    End

    'pb_next': Begin ; Next. -> Next Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_next'
    End

    'pb_print' : Begin ; Print. -> Print Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_print'
    End

    'pb_output': Begin ; Output. -> Output Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_output'
    End

    'pb_forward' : Begin ; Forward. -> Forward Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_forward'
    End

    'pb_back' : Begin ; Back. -> Back Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_back'
    End

    'pb_save' : BEGIN ; Save -> Save Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_save'
    end

   'pb_load' : BEGIN ; Save -> Load Button In Papco Main Window.
      Papco_Main_Event, 0, User_Event='pb_load'
    end

    'pb_del_old' : Begin        ; Delete Any Old Draw Windows
      For I=0, Widgetdata.No_Draw_Window-1 Do Begin
        Draw_Win_Name='papco_draw_window_' + String(I,Format="(I2.2)")
        If Xregistered(Draw_Win_Name) Then Begin
          message,'Destroying Draw Window: ' + Draw_Win_Name, /cont
          Widget_Control, Widgetdata.Old_Draw_Id(I), /Destroy
        Endif
      Endfor
    End

    'pb_del_all' : Begin
      For I=0, Widgetdata.No_Draw_Window Do Begin
        Draw_Win_Name='papco_draw_window_' + String(I,Format="(I2.2)")
        If Xregistered(Draw_Win_Name) Then Begin
          Message,'Destroying Draw Window: ' + Draw_Win_Name, /Cont
          Widget_Control, Widgetdata.Old_Draw_Id(I), /Destroy
        Endif
      Endfor
      Widgetdata.No_Draw_Window=0
      Widgetdata.Need_To_Draw=1
      Widget_Control, Widgetdata.pb_Draw_New, Sensitive=0
      Widget_Control, Widgetdata.pb_Annotate, Sensitive=0
      Widget_Control, Widgetdata.pb_Rm_Annot, Sensitive=0
      Papco_Input, /Enable
    End

    'pb_help': Begin
       Xdisplayfile, papco_helpfile('papco_draw_window.help'), $
         Title='Papco Online-Help', Group=Event.Top
    End

    'dw_draw':  Papco_Cursor, Event

    'pb_empty': Begin
    End

    Else : Begin
      Message, 'No Match Found: ' + Ev, /Cont
      Help, /Struct, Event
    End

ENDCASE 

END 
