;******************************************************************************
;******************************************************************************
;* FILE:         
;*    $papco_PATH/papco_editslots.pro
;* 
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:  
;*    Contained are procedures for changing the panel editor list.
;*
;*    Those procedures are contained:
;*
;*      pro papco_stripemptyslots
;*	   -> removes any gaps (empty slots) in the panel editor window,
;*            preserving any overplots!
;*
;*	pro papco_resetslots
;*         -> resets all panels to height one, top at bottom, ascending
;*
;*      pro papco_resortslots
;*         -> resorts the panels so that current bottom plot goes
;*            to bottom of list, ascending upwards
;*
;* KEYWORDS:
;*
;* MODIFICATION HISTORY:       
;*     june,1996, written by R. Friedel 
;******************************************************************************
;******************************************************************************
pro papco_stripemptyslots, NOREFRESH=NOREFRESH

  COMMON PLOT_COMPOSER, widgetData
  COMMON RefreshList, old_maxPlotNo
  
;find total number of plots    
    nplots=widgetData.numberOfPlots
;find total number of panels
    npanels=old_maxPlotNo
    empty_panel=bytarr(npanels)
        
    for i=0, npanels-1 do begin
      ;find out if there are any panels at this position
      n_at_thispos=0
      for j=0, nplots-1 do begin
        testrange=findgen(widgetdata.plotinfos(j).PANELHEIGHT) + $
                  widgetdata.plotinfos(j).PANELPOSITION
        index=where(testrange eq i, count)
        n_at_thispos=n_at_thispos+count
      endfor
      if n_at_thispos eq 0 then empty_panel(i)=1
      n_at_thispos=0
    endfor  
    
;now remove empty panels from list by re-writing  PANELPOSITION for
;each plot. 
    subtract=bytarr(nplots)
    for i=0, npanels-1 do begin
      if empty_panel(i) then begin
        index=where(widgetData.plotinfos(0:nplots-1).PANELPOSITION ge i,count)
        if count ne 0 then subtract(index)=subtract(index)+1 
      endif  
    endfor 
        
    if nplots ne 1 then $
    widgetData.plotinfos(0:nplots-1).PANELPOSITION= $
      widgetData.plotinfos(0:nplots-1).PANELPOSITION - subtract $
    else $
    widgetData.plotinfos(0).PANELPOSITION= $
      widgetData.plotinfos(0).PANELPOSITION - subtract(0)
    
    if not keyword_set(NOREFRESH) then papco_Refresh, LIST=2
  
end

;******************************************************************************
pro papco_resetslots

  COMMON PLOT_COMPOSER, widgetData


   FOR i=0, widgetData.numberOfPlots-1 DO BEGIN
     widgetData.plotInfos(i).overplot=0
     widgetData.plotInfos(i).PanelHeight=1
     widgetData.plotInfos(i).panelPosition=i
   END
   papco_Refresh, /LIST
   
   papco_stripemptyslots
   
end

;******************************************************************************
pro papco_resortslots, NOREFRESH=NOREFRESH

  COMMON PLOT_COMPOSER, widgetData
  
  ;find number of plots
  np=widgetData.numberOfPlots
  ;re-order the panels with the highest one first
  pos=widgetdata.plotinfos(0:np-1).panelposition
  index=reverse(sort(pos))
  widgetdata.plotinfos(0:np-1)=widgetdata.plotinfos(index)
  if not keyword_set(NOREFRESH) then papco_Refresh, /LIST
    
end

;*****************************************************************************
; this program removes all overlaps, preserving panels that are exact overlays.
pro papco_joinslots, NOREFRESH=NOREFRESH
  
  COMMON PLOT_COMPOSER, widgetData  
  
;find number of plots
  np=widgetData.numberOfPlots
  
;get overlaps array
  papco_FindOverplots, 0, overlaps, /FULL
  
;do a check loop first to detect conflicts.  
  for i=0, np-1 do begin
    sPlot=i
    cPlots=overlaps(sPlot).exact(0:overlaps(sPlot).exact_count-1)
    if n_elements(cPlots) gt 1 then begin ;panel has an exact overplot
      ;adjust only the panel that's the bottom one (overplot not set)
      index=where(widgetdata.plotinfos(cPlots).overplot eq 0, c)
      if c eq 0 then begin
        print,'% papco_joinslots: Conflict at panel',i+1
        print,'  All panels at this positions are overplots!'
        print,'  Choose one of the panels to be the "master panel"'
        print,'  before continuing.'
        return
      endif
      if c eq n_elements(cPlots) then begin
        print,'% papco_joinslots: Conflict at panel',i+1
        print,'  More than one panel defined at this position!'
        print,'  Set all but one of the panels at this position to be'
        print,'  overplots before continuing, or shift the panels.'
        return
      endif  
      if c gt 1 then begin
        print,'% papco_joinslots: Conflict at panel',i+1
        print,'  More than one panel defined to be the "master panel"!'
        print,'  Set all but one of the panels at this position to be"'
        print,'  overplots before continuing, or shift the panels.'
        return
      endif        
      sPlot=cPlots(index(0))
    endif    
  endfor      
    
;loop through all plots. Keep the current panel size, and adjust all
;others by incrementing their position until no more overlap
;exists. Check for exact overplots and do not adjust.
  for i=0, np-1 do begin
    sPlot=i
    cPlots=overlaps(sPlot).exact(0:overlaps(sPlot).exact_count-1)
    if n_elements(cPlots) gt 1 then begin ;panel has an exact overplot
      ;adjust only the panel that's the bottom one (overplot not set)
      index=where(widgetdata.plotinfos(cPlots).overplot eq 0, c)
      sPlot=cPlots(index(0))
    endif
    ;loop through all the panels apart from the current one and its overplots
    ;skip if panel is "anchored"
    pi=widgetdata.plotinfos(sPlot).panelposition
    hi=widgetdata.plotinfos(sPlot).panelheight
    for j=i, np-1 do begin
      if widgetData.anchored(j) then goto, next
      index=where(cPlots eq j,c) ; check if it's current or current's overplot
      if c eq 0 then begin
        pj=widgetdata.plotinfos(j).panelposition
        hj=widgetdata.plotinfos(j).panelheight
        if (pi+hi) gt pj then begin
          widgetdata.plotinfos(j).panelposition=(pi+hi)
        endif 
      endif  
      next:
    endfor
  endfor  
  
  ;papco_stripemptyslots, /NOREFRESH  
  if not keyword_set(NOREFRESH) then papco_Refresh, /LIST
    
end

;******************************************************************************
; this program moves the panel up by one.
;
pro papco_MoveUp
  
  COMMON PLOT_COMPOSER, widgetData
  
  sPlot=widgetData.selectedPlot
;don't do anything if the panel is "anchored"  
  if widgetData.anchored(sPlot) then return
  
  IF sPlot GE 0 THEN BEGIN
    widgetData.plotInfos(sPlot).panelPosition=$
      widgetData.plotInfos(sPlot).PanelPosition+1
    papco_Refresh, /LIST
  ENDIF
  widgetData.need_to_draw=1   
  
end  

;******************************************************************************
; this program moves the panel down by one.
;
pro papco_MoveDown
  
  COMMON PLOT_COMPOSER, widgetData
  
  sPlot=widgetData.selectedPlot
;don't do anything if the panel is "anchored"  
  if widgetData.anchored(sPlot) then return
  
  IF sPlot GE 0 THEN BEGIN   
    IF widgetData.plotInfos(sPlot).panelPosition EQ 0 THEN BEGIN
      FOR i = 0, widgetData.numberOfPlots-1 DO $
      IF i NE sPlot THEN widgetData.plotInfos(i).panelPosition=$
                         widgetData.plotInfos(i).panelPosition+1
      ENDIF ELSE widgetData.plotInfos(sPlot).panelPosition=$
                 (widgetData.plotInfos(sPlot).PanelPosition-1) > 0
    papco_Refresh, /LIST
  ENDIF
  widgetData.need_to_draw=1  
  
end

;******************************************************************************
; this program shrinks the size of other panels relative to the current
; one. This is done in two ways: 
;      a) if the other panel are larger than height "1", their height
;      is reduced.
;      b) if one of the other panel is of height "1", the current panel is
;      enlarged by "1".
;
; care is taken of overplots. ALL overplots that overlap at the
; current position are treated the same way as the panel at the
; current position. 
;
pro papco_ShrinkOthers
  
  COMMON PLOT_COMPOSER, widgetData
  
  sPlot=widgetData.selectedPlot
  nPlots=widgetData.numberOfPlots
  
  papco_FindOverplots, sPlot, cPlots   ;exact overlays are all "current" panels
  
  ;check if any of the other plots is already at height "1"
  h=widgetData.plotInfos(0:nPlots-1).panelHeight & h(cPlots)=0
  isone=where(h eq 1,count)
  if count gt 0 then begin
    papco_EnlargeMe
  endif else begin
    for i=0, nPlots-1 do begin
      ;skip if panel is "anchored"
      if widgetData.anchored(i) then goto, next1
      if count eq 0 then  begin
        pi=widgetData.plotInfos(i).panelPosition
        hi=widgetData.plotInfos(i).panelHeight
        if hi GT 1 then begin
          ; move all plots with 'panel-position GT pi' one position downwards,
          ; but don't move a plot, if plotInfos(i) and plotInfos(j) overlap
          for j=0, nPlots-1 do begin
            ;skip if panel is "anchored"
            if widgetData.anchored(i) then goto, next2
            pj=widgetData.plotInfos(j).panelPosition
            if pj gt pi AND pj ge pi+hi then $
              widgetData.plotInfos(j).panelPosition=(pj-1) > 0
            next2:
          endfor
          widgetData.plotInfos(i).panelHeight=hi-1
        endif
      endif
      next1:
    endfor
  endelse
  
  papco_Refresh, LIST=2
  widgetData.need_to_draw=1   
  
end

;******************************************************************************
; this program enlarges the current panel and all others at this
; position unless ALONE is set.
pro papco_EnlargeMe, NOJOIN=NOJOIN, ALONE=ALONE
  
  COMMON PLOT_COMPOSER, widgetData
    
  sPlot=widgetData.selectedPlot
  nPlots=widgetData.numberOfPlots
  
;don't do anything if the panel is "anchored"  
  if widgetData.anchored(sPlot) then return
  
  if keyword_set(ALONE) then cPlots=[sPlot] else $
  papco_FindOverplots, sPlot, cPlots   ;exact overlays are all "current" panels
  
;first,re-order the panels with the lowest one first
  pos=widgetdata.plotinfos(0:nPlots-1).panelposition
  index=sort(pos)
  widgetdata.plotinfos(0:nPlots-1)=widgetdata.plotinfos(index)
    
;now increase current plots and adjust all others  
  for i=0,n_elements(cPlots)-1 do begin
    new_idx=where(index eq cPlots(i))
    widgetdata.plotinfos(new_idx).panelHeight= $
      widgetdata.plotinfos(new_idx).panelHeight + 1
  endfor  
  if not keyword_set(NOJOIN) then papco_joinslots, /NOREFRESH
  
;resort the panels back to the order they were in
  b=fix(findgen(nPlots))
  widgetdata.plotinfos(index)=widgetdata.plotinfos(b)  
  papco_Refresh, /LIST  
    
end  
  
;******************************************************************************
; this program enlarges the size of other panels relative to the current
; one. This is done in two ways: 
;      a) if the current panel is larger than height "1", it's height
;      is reduced.
;      b) if the current panel is of height "1", all others are
;      enlarged by "1".
pro papco_EnlargeOthers
  
  COMMON PLOT_COMPOSER, widgetData
  
;don't do anything if the panel is "anchored"  
  
  sPlot=widgetData.selectedPlot
  nPlots=widgetData.numberOfPlots
  papco_FindOverplots, sPlot, cPlots   ;exact overlays are all "current" panels
  
  if widgetData.plotInfos(sPlot).panelHeight ne 1 then begin
    papco_ShrinkMe
  endif else begin
    for i=0, nPlots-1 do begin
      ;skip if panel is "anchored"
      if widgetData.anchored(i) then goto, next
      index=where(cPlots eq i,c)
      if c eq 0 then  begin
        pi=widgetData.plotInfos(i).panelPosition
        hi=widgetData.plotInfos(i).panelHeight
        ; move all plots with 'panel-position GT pi' one position upwards,
        ; but don't move a plot, if plotInfos(i) and plotInfos(j) overlap
        for j=0, widgetData.numberOfPlots-1 do begin
          pj=widgetData.plotInfos(j).panelPosition
          if pj gt pi AND pj ge pi+hi then $
            widgetData.plotInfos(j).panelPosition=pj+1
        endfor
        widgetData.plotInfos(i).panelHeight=hi+1
      endif
      next:  
    endfor
  endelse
  
  papco_Refresh, /LIST
  widgetData.need_to_draw=1         

end  

;******************************************************************************
pro papco_ShrinkMe, NOSTRIP=NOSTRIP, ALONE=ALONE
  
  COMMON PLOT_COMPOSER, widgetData
    
  sPlot=widgetData.selectedPlot
  nPlots=widgetData.numberOfPlots
  
;don't do anything if the panel is "anchored"  
  if widgetData.anchored(sPlot) then return
  
  if keyword_set(ALONE) then cPlots=[sPlot] else $
  papco_FindOverplots, sPlot, cPlots   ;exact overlays are all "current" panels
    
;first,re-order the panels with the lowest one first
  pos=widgetdata.plotinfos(0:nPlots-1).panelposition
  index=sort(pos)
  widgetdata.plotinfos(0:nPlots-1)=widgetdata.plotinfos(index)
  pos=widgetdata.plotinfos(0:nPlots-1).panelposition
    
;now decrease current plot and adjust all others  
  for i=0,n_elements(cPlots)-1 do begin
    new_idx=where(index eq cPlots(i))
    if widgetdata.plotinfos(new_idx).panelHeight ge 2 then $
      widgetdata.plotinfos(new_idx).panelHeight= $
        widgetdata.plotinfos(new_idx).panelHeight - 1
  endfor  
  if not keyword_set(NOSTRIP) then papco_stripemptyslots, /NOREFRESH
    
;resort the panels back to the order they were in
  b=fix(findgen(nPlots))
  widgetdata.plotinfos(index)=widgetdata.plotinfos(b)  
  papco_Refresh, LIST=2  
      
end  

;******************************************************************************
;this program returns an array of indices corresponding to the panels
;which overplot the current panel EXACTLY. If FULL is set, return the
;full overplot index array for all panels. 
;
pro papco_FindOverplots, sPlot, index , FULL=FULL
  
  COMMON PLOT_COMPOSER, widgetData
  
  nPlots=widgetData.numberOfPlots  
  
;--- calculate overlapping plots ----------------------------------------------
;     the 'overlaps'-array contains for each plot the plots it overlaps with
    overlapInfo={count:0, $
                 with:intarr(nPlots), $
                 exact_count:0, $
                 exact:intarr(nPlots)}
    overlaps=replicate(overlapInfo, nPlots)
    startPositions=widgetData.plotInfos(0:nPlots-1).panelPosition
    endPositions=startPositions+$
      widgetData.plotInfos(0:nPlots-1).panelHeight
    start_ind=sort(startPositions)

    for i=0, nPlots-1 do begin
      ; the next statement finds all plots, that overlap with
      ; widgetData.plotInfos(i)
      overl=where((startPositions(i) GE startPositions AND $
                   startPositions(i) LT endPositions) OR $
                  (endPositions(i) GT startPositions AND $
                   endPositions(i) LE endPositions) OR $
                  (startPositions GE startPositions(i) AND $
                   startPositions LT endPositions(i)) , count)
      ; the conditions checked in this where-statement are:
      ;    1: does this plot start inside the range of another ?
      ; OR 2: does this plot end inside the range of another ?
      ; OR 3: does another plot start inside the range of this plot ?
      ; To check, if another plot ends inside this plot is not
      ; necessary, as this is done in the first and second check.
      ; The third check is neccessary, if a plot's end AND start
      ; lie inside of this plot's
      overlaps(i).count=count           ;count is always GE 1, because the 
      overlaps(i).with(0:count-1)=overl ;plot i itself is contained  in 'overl'
      ; the next statements finds all plots that overlay EXACTLY.
      exact_overl=where((startPositions(i) EQ startPositions) AND $
                        (endPositions(i) EQ endPositions), count)   
      overlaps(i).exact_count=count
      overlaps(i).exact(0:count-1)=exact_overl  
    endfor  
    
    if keyword_set(FULL) then $
      index=overlaps $
      else index=overlaps(sPlot).exact(0:overlaps(sPlot).exact_count-1)
    
end  

  
