;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_files.pro
;*
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:
;*    Contained are procedures for loading and saving plot-infos and
;*    papco-defaults. Contained as well are routines for batch-printing.
;*
;*    Contains all file I/O routines that PAPCO uses.
;*
;* MODIFICATION HISTORY:
;*      august,september 1995, written by A.Keese
;*      february 1996, Haje Korth
;*      june 1996, R Friedel, all routines adapted to also handle VMS
;*      may 2002, R. Friedel, new papco "pickfile" routine.
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE(s):
;*      papco_pickfile, papco_pickfile_event, papco_pickfile_refresh
;*
;* DESCRIPTION:
;*      Wrapper to the idl routine DIALOG_PICKFILE that extends the
;*      functionailty of the file handling to allow creation/deletion of
;*      directories and creation/deletion of files, and links to a simple
;*      editor for text files, or links to a simple file explorer for idl
;*      saveset files.. Incorporates some PAPCO niceties
;*      for always forcing a given file extension.
;*
;* KEYWORDS: same as DIALOG_PICKFILE, see idl help
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      result=papco_pickfile(same keywords as for DIALOG_PICKFILE)
;******************************************************************************

;******************************************************************************
PRO papco_pickfile_event, event

common pickfile_data, pick_widgetdata

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF

    ELSE: BEGIN
        widgetData=papco_panel_Event(event, pick_widgetData)
    ENDELSE

ENDCASE

END

;******************************************************************************
PRO papco_pickfile_refresh, ALL=ALL

common pickfile_data, pick_widgetdata

IF keyword_set(ALL) THEN BEGIN

ENDIF

END

;******************************************************************************
function papco_pickfile, GROUP=GROUP, GET_PATH=pathSelected, _EXTRA = e

common plot_composer, widgetdata
common pickfile_data, pick_widgetdata

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=widgetdata.base
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'   ;label for action-button

IF keyword_set(e.title) THEN title = e.title ELSE title = 'PAPCO File Handler'

;since we use some build-in papco widgets, we needd a aPlotInfo structure
aPlotInfo=papco_getplotinfostruct()
additionalWData={canceled:1}

IF keyword_set(e.FILE) THEN aPlotInfo.filename = e.File


;--- create the base-window ---------------------------------------------------
pick_base=papco_WIDGET_BASE(TITLE=title, COLUMN=1, GROUP_LEADER=group)

widgets=papco_panel_AddFile_Widgets(pick_base, aPlotInfo, /MODIFY, _EXTRA = e)
additionalWData=create_struct(widgets, additionalWData)

;--- add some widgets that allow extra functionality on files------------------


;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(pick_base, Action, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)


;--- now, create the structure that keeps the panel-window data. You ----------
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'papco_pickfile.help', $
                   soptions:aplotinfo.soptions, $
                   ioptions:aplotinfo.ioptions}

pick_widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
;papco_pickfile_refresh, /ALL
papco_panel_Refresh, pick_widgetData,  /ADD_FILE
papco_Panel_Realize, pick_base, pick_widgetData

papco_xmanager, 'papco_pickfile', pick_base

pathSelected= pick_widgetData.extra.path

IF NOT pick_widgetData.canceled THEN return, pick_widgetData.filename ELSE $
  return, ''

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_saveState
;*
;* DESCRIPTION:
;*      Save the plot product to a file.
;*	This procedure lets the user pick a filename, to which the info
;*	are written, or writes to a user supplied fiel name.
;*
;*	The format of the written file is determined by the
;*	papco_getPlotInfoStruct. So, this routine should suffice even if
;*	you extend the plotInfo-structure.
;*
;*      This routine makes use of the general papco_write_struct function.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	FILENAME    if set, don't enquire for name but use filename
;*
;* CALLING SEQUENCE:
;*	papco_saveState
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     modified November 1995, R Friedel: to include filename in widgetData
;*                             add keyword FILE to pickfile
;*                             to "remember" previously used filename
;*              November 2000 to use papco_write_struct, R Friedel
;******************************************************************************
PRO papco_SaveState, FILENAME=FILENAME

common plot_composer, widgetData
common papco_draw, drawwidgetdata

if KEYWORD_SET(FILENAME) then begin          ;see if filename is set
    papco_sepdir, filename, pathSelected, fname
ENDIF else BEGIN                            ;let the user pick a filename
    if papco_finddir(widgetData.saveStatePath) then $
      path=widgetData.saveStatePath else path=papco_getenv('PAPCO_HOME')
    IF widgetData.saveFileName EQ 'none' THEN fname = '' ELSE $
      fname = widgetData.saveFileName

    filename=papco_pickfile(GROUP=widgetData.main_base, $
                            PATH=path, FILTER='*.papco', $
                            TITLE='Save the current Panelplot Product', $
                            GET_PATH=pathSelected, FILE=fname)

    widgetData.saveStatePath=pathSelected

    if filename ne '' then BEGIN
      widgetData.saveFileName=filename
      ;update product name in draw window if draw window exist
      if widgetdata.no_draw_window ge 1 then begin
        outst1='Product: '+pathSelected+filename
        widget_control, drawwidgetdata.lbl_Draw1, set_value=outst1
      ENDIF
    ENDIF else begin
      message,'No filename defined',/cont
      widgetData.saveFileName = 'none'
      return
    endelse
    fname = filename
    filename =  pathSelected+filename
ENDELSE

;write header for product file
message, '...'+strright(filename,56), /cont
stars='***************************************'
ON_IOERROR, io_Error
openw, u, filename, /get_lun
  printf,u,';'+stars+stars
  printf,u,';* DIR:  ' + pathSelected
  printf,u,';* FILE: ' + fname
  printf,u,';*'
  printf,u,';* DESCRIPTION:'
  printf,u,';*    Plot product definition for PAPCO'
  printf,u,';*    Lines starting with ";" are comments and are ignored'
  printf,u,';* '
  printf,u,';*    Written here are the contents of all the PAPCO'
  printf,u,';*    plotinfo structures for all panels in this product'
  printf,u,';* '
  printf,u,';*    Each variable is described in the preceeding line'
  printf,u,';*    starting with "->". The line contains'
  printf,u,';*'
  printf,u,';*    variable name -> variable type -> a b c d0---d8   '
  printf,u,';*'
  printf,u,';*    where a - TYPE integer'
  printf,u,';*          b - N_ELEMENTS integer'
  printf,u,';*          c - N_DIMENSIONS integer'
  printf,u,';*          d - DIMENSIONS of each array dimension.'
  printf,u,';*'
  printf,u,';* CREATED BY: papco_SaveState.pro at ' + systime(0)
  printf,u,';'+stars+stars
  printf,u,';'
  printf,u,'; Origin: '+ strmid(papco_Version(), 5, strlen(papco_Version())-10)
  printf,u,';'
  printf,u,'; Number of panels in product: '
  printf,u,varprt(widgetData.numberOfPlots)

  ;write all panel plotinfo's to the file
  for plotnr=0, widgetData.numberOfPlots-1 do begin
    printf,u,';'
    printf,u,';'+stars+stars
    printf,u,'; PlotInfo for panel '
    printf,u,varprt(plotnr)
    printf,u,';'
    struct=widgetData.plotInfos(plotnr)
    result=papco_write_struct(fName, 'PlotInfo', struct, /ARRAY)
    if result.error eq 0 then begin
      out_arr=result.TEXT  &  out_arr=[out_arr,'-> END STRUCT']
      for i=0,n_elements(out_arr)-1 do printf, u, out_arr(i)
    ENDIF else $
      message,'Error writing PlotInfo for panel '+ varprt(plotnr),/cont
  endfor

  ;add information about the time range of this product.
  printf,u,';'
  printf,u,'; Time Range Information and other'
  struct={startTime:widgetData.startTime, $
          endTime:widgetData.endTime, $
          minTime:widgetData.minTime, $
          maxTime:widgetData.maxTime, $
          time_choice_action:widgetData.time_choice_action, $
          default_orbitInfo:widgetData.default_orbitInfo, $
          annotateFileName:widgetData.annotateFileName}
  result=papco_write_struct(fName, 'OtherInfo', struct, /ARRAY)
  if result.error eq 0 then begin
    out_arr=result.TEXT  &  out_arr=[out_arr,'-> END STRUCT']
    for i=0,n_elements(out_arr)-1 do printf, u, out_arr(i)
  ENDIF else message,'Error writing OtherInfo for plot product' ,/cont

  printf,u,';'
  printf,u,'; End PAPCO Plot product definition'
close,u  &  free_lun, u
papco_refresh, /PRINTER
return

;the program jumps to this label, if an error occurs while writing
io_error:
IF keyword_set(u) THEN BEGIN & close,u  &  free_lun, u & ENDIF

dummy=messagebox(['I could not completely write your data...', $
                  !ERR_STRING], ['What a pity :-('], /center, $
                  title='Write-error')
END

;******************************************************************************
;* PROCEDURE:
;*      papco_LoadState
;*
;* DESCRIPTION:
;*      Load the plot product from a file.
;*	This procedure lets the user pick a filename, from which the data
;*	are read.
;*
;*	The format of the written file is determined by the
;*	papco_getPlotInfoStruct. So, this routine should suffice even if
;*	you've extended the plotInfo-structure.
;*
;*      This routine makes use of the general papco_read_struct function.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	FILENAME    if set, don't enquire for name but use filename
;*
;* CALLING SEQUENCE:
;*	papco_LoadState
;*
;* MODIFICATION HISTORY:
;*     written  August 1995, Andreas Keese
;*     modified February 1997, keyword PRODUCT added, Reiner Friedel
;*              June 1998, added tag downward compatibilty, Reiner Friedel
;*              November 2000 to use papco_read_struct, Reiner Friedel
;******************************************************************************
PRO papco_LoadState, FILENAME=FILENAME, STATE=STATE, VERBOSE=VERBOSE

COMMON plot_composer, widgetData
COMMON get_error, get_err_no, get_err_msg

;see if filename is set
if KEYWORD_SET(FILENAME) then BEGIN
    papco_sepdir, filename, pathSelected, fname, ext
    ;if there is no path, use default
    if pathSelected eq '' then pathSelected=widgetData.saveStatePath
    ;check for correct extension
    if ext eq '' then ext='.papco'
    fname=fname+ext
    ;make sure file exists
    result=findfile(pathSelected+fname, count=n)
    if n ne 1 then begin
      msg='File '+pathSelected+fname+' not found'
      get_err_msg=msg  &  get_err_no=1  &  message, msg, /cont
    ENDIF
    ;update product name variables- only if NOT a state filename
    if not keyword_set(STATE) then BEGIN
      widgetData.saveFileName=fName
      widgetData.saveStatePath=pathSelected
    ENDIF
ENDIF else BEGIN
    ;let the user pick a filename
    if papco_finddir(widgetData.saveStatePath) then $
      path=widgetData.saveStatePath else path=papco_getenv('PAPCO_HOME')
    IF widgetData.saveFileName EQ 'none' THEN fname = '' ELSE $
      fname = widgetData.saveFileName
    fName=papco_pickfile(GROUP=widgetData.main_base, $
                         PATH=path, FILTER='*.papco', $
                         TITLE='Load Panelplot Product', $
                         /MUST_EXIST, GET_PATH=pathSelected, FILE=fname)
    if fname ne '' then begin
      widgetData.saveFileName=fName
      widgetData.saveStatePath=pathSelected
      ;update product name in draw window if draw window exist
      if widgetdata.no_draw_window ge 1 then begin
        common papco_draw, drawwidgetdata
        outst1='Product File: '+ $
          strright(widgetData.saveStatePath+widgetData.saveFileName,70)
        WIDGET_CONTROL, drawwidgetdata.lbl_Draw1, set_value=outst1
      ENDIF
    ENDIF else begin
      message,'No filename defined',/cont  &  return
    endelse
ENDELSE

message,'Loading PAPCO product file',/cont
print,'  '+pathSelected+fname

;ON_IOERROR, io_Error  
dummy=''  &  dmy=';'
openr, u, pathSelected+fname, /get_lun

;skip header and get numberOfPlots
WHILE dmy EQ ';' DO BEGIN
    readf, u, dummy  &  dmy=strmid(dummy,0,1)
ENDWHILE
widgetData.numberOfPlots=fix(dummy)

FOR i=0, widgetData.numberOfPlots-1 DO BEGIN
    dummy=''  &  dmy=';'
    WHILE dmy EQ ';' DO BEGIN
        readf, u, dummy  &  dmy=strmid(dummy,0,1)
    ENDWHILE
    
    plotnr=fix(dummy)

    struct=widgetData.plotInfos(plotnr)
    ;now read the plotinfo structure - use papco_read_struct
    result=papco_read_struct('', 'PlotInfo', struct, unit=u)
    if result.error eq 0 then widgetData.plotInfos(plotnr)=result.struct $
      else message,'Error reading PlotInfo for panel '+ varprt(plotnr),/cont
ENDFOR

;read time range info and other info if it exists
if not eof(u) then readf, u, dummy
if not eof(u) then readf, u, dummy
  struct={startTime:widgetData.startTime, $
          endTime:widgetData.endTime, $
          minTime:widgetData.minTime, $
          maxTime:widgetData.maxTime, $
          time_choice_action:widgetData.time_choice_action, $
          default_orbitInfo:widgetData.default_orbitInfo, $
          annotateFileName:widgetData.annotateFileName}
  result=papco_read_struct('','OtherInfo',struct,unit=u)
  if result.error eq 0 then begin
    OtherInfo=result.struct
    widgetData.startTime=OtherInfo.startTime
    widgetData.endTime=OtherInfo.endTime
    widgetData.minTime=OtherInfo.minTime
    widgetData.maxTime=OtherInfo.maxTime
    widgetData.time_choice_action=OtherInfo.time_choice_action
    widgetData.default_orbitInfo=OtherInfo.default_orbitInfo
    widgetData.annotateFileName=OtherInfo.annotateFileName
ENDIF else message,'Error reading OtherInfo for product',/cont

close,u  &  free_lun, u
papco_Refresh, /ALL
papco_stripemptyslots
widgetData.needto_read_data=1	   ;new product loaded: re-read data
widgetData.need_to_draw=1
papco_Clear_Drawing

return

io_error:
  
close,u  &  free_lun, u
dummy=messagebox(['I could not read it...', !ERR_STRING], $
                 ['What a pity :-('], /center, title='Read-error')
papco_ResetState

END ;papco_LoadState


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_ResetState
;*
;* DESCRIPTION:
;*      ResetState resets papco - i.e. it clears all plots.
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_ResetState
;*
;* MODIFICATION HISTORY:
;*      written August 1995, Andreas Keese
;*      modified October 1998, Reiner Friedel
;******************************************************************************
PRO papco_ResetState

  COMMON PLOT_COMPOSER, widgetData
  COMMON papco_graphics, papco_drawn_window
  COMMON RefreshList, old_maxPlotNo

; fill up plotinfo aray with empty plotinfo structures
  dummy=papco_getPlotInfoStruct()
  np=n_elements(widgetData.plotInfos)
  for i=0,np-1 do widgetData.plotInfos(i)=dummy
  widgetData.numberOfPlots=0
  widgetData.annotateFileName=''
  widgetData.anchored(*)=0
  old_maxPlotNo=1

  if n_elements(papco_drawn_window) gt 0 then papco_drawn_window(*)=0

; reset sliders to max times again
  widgetData.startTime=widgetData.minTime
  widgetData.endTime=widgetData.maxTime

  papco_Clear_Drawing
  papco_Refresh, /ALL

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_readPrinterConfig
;*
;* DESCRIPTION:
;*      At program-start, this procedure reads the available printer
;*      from 'papco_user/papco_printer_config.dat'
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	papco_readPrinterConfig, printerData
;*
;* MODIFICATION HISTORY:
;*      written February 1996, Haje Korth
;*      updated June 2001, Reiner friedel
;******************************************************************************
PRO papco_readPrinterConfig, printerData

fname=papco_getenv('PAPCO_HOME')+ 'papco_printer_config.dat'
message, 'Reading printer configuration file', /cont
papco_prt, '  ',fname

printerData = STRARR(2,20) & count = 0  &  dummy=''

OPENR, inputUnit, fname, ERROR=err, /GET_LUN
IF err EQ 0 THEN BEGIN
    ON_IOERROR, io_error
    WHILE NOT EOF(inputUnit) DO BEGIN
      READF, inputUnit, dummy
      ;take out anting following a ";" since it's a comment
      pos = strpos(dummy, ';')
      IF pos NE -1 THEN dummy = strmid(dummy, 0, pos)
      printerString=STR_SEP(dummy,',')
      test=size(printerstring) ;see if there's anything in the line
      IF test(1) eq 2 then begin
        printerData(0, count) = STRTRIM(printerString(0),2)
        printerData(1, count) = STRTRIM(printerString(1),2)
        message, printerData(0, count) +'; ' + printerData(1, count), /cont
        count = count+1
        IF count GE  20 THEN BEGIN
          message, 'Reached maximum of '+ varprt(count)+' printers', /cont
          break
        ENDIF
      ENDIF
    ENDWHILE
    CLOSE, inputUnit & FREE_LUN, inputUnit
ENDIF ELSE BEGIN
    message, 'Found no printer config file!', /cont
ENDELSE
RETURN

io_error:
CLOSE, inputUnit & FREE_LUN, inputUnit
papco_prt, 'Error while reading printer configuration...'

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_readDefaultSettings, FONT_ONLY=FONT_ONLY
;*
;* DESCRIPTION:
;*      At program-start, this procedure reads the default-settings.
;*	These are the default printer and the default write/read-paths.
;*	All defaults are written at program-end to the file
;*      'papco_user/papco_default.dat' by papco_writeDefaultSettings
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	TAGID   	if set, return TAGID only
;*
;* CALLING SEQUENCE:
;*	papco_readDefaultSettings, TAGID=TAGID
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     modified october 1996, Reiner Friedel
;*     generalized Februaty 2000, Reiner Friedel, for easy expandability.
;******************************************************************************
PRO papco_readDefaultSettings, TAGID=TAGID

  common plot_composer, widgetdata

  fln=papco_getenv('PAPCO_CONFIG')+'papco_default.dat'

  if keyword_set(TAGID) then begin
    read_tag_id=TAGID.id
    struct_name='TAGID.struct'
    struct=TAGID.struct
  ENDIF else begin
    read_tag_id=''
    struct_name='widgetdata'
    struct=widgetdata
  endelse

  result=papco_read_struct(fln, struct_name, struct, TAGID=read_tag_id)

  if result.error ne 0 then begin
    message,result.text, /cont
    message,'using built-in defaults',/cont
    if keyword_set(TAGID) then begin
      case TAGID.id of
        'DEFAULT_FONT': TAGID.struct.DEFAULT_FONT='6x13'
        'DEFAULT_TRAP': TAGID.struct.DEFAULT_TRAP=1
      endcase
    ENDIF
    return
  ENDIF

  if keyword_set(TAGID) then begin
    TAGID.struct=result.struct
    message, 'read '+TAGID.ID, /cont
    return
  ENDIF

  widgetdata=result.struct

  ;set other PAPCO plotting  defaults
  !P.CHARSIZE  = widgetData.default_charsize
  !P.CHARTHICK = widgetData.default_charthick
  !P.FONT      = -1
  !P.NOERASE   = 1
  !P.TICKLEN   = widgetData.default_TICKLEN
  !X.TICKS     = widgetData.default_XTICKS
  !X.STYLE     = widgetData.default_XSTYLE
  !Y.STYLE     = 1
  !X.MINOR     = widgetData.default_XMINOR

  message, 'read all', /cont
  return

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_writeDefaultSettings
;*
;* DESCRIPTION:
;*      At program-end, this procedure writes all the default-settings
;*      and other "remembered" settings.
;*	Defaults file:  $PAPCO_HOME/papco_default.dat
;*      At program-start, they are read by papco_readDefaultSettings
;*
;*      All common paramters for PAPCO are stored in the structure and
;*      common block:
;*
;*        common plot_composer, widgetdata
;*
;*      Here we store all tags of the structure whose name contain
;*      any of the the following strings:string "default" in a text file.
;*
;*         default - all the varibales from the papco defaults widget
;*         keep    - any variables that need to be "remembered" for
;*                   the next PAPCO session.
;*
;*      We write to file the name of the variable, followed by the IDL_SIZE
;*      structure for that variable, followed by the variable contents.
;*
;* INPUTS:   none
;*
;* KEYWORDS: none
;*
;* CALLING SEQUENCE:
;*	papco_writeDefaultSettings
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     modified october 1996, Reiner Friedel
;*     generalized Februaty 2000, Reiner Friedel, for easy expandability.
;******************************************************************************
PRO papco_writeDefaultSettings

COMMON plot_composer, widgetdata

widgetdata.default_useprinter=widgetdata.printerinfo.useprinter
widgetdata.default_printer=widgetdata.printerinfo.printer
widgetdata.default_askforfilename=widgetdata.printerinfo.askforfilename
widgetdata.default_filetype=widgetdata.printerinfo.filetype
widgetdata.default_filename_tmpl=widgetdata.printerinfo.filename_tmpl

name = 'papco_default.dat'
fln=papco_getenv('PAPCO_CONFIG')+name
struct_name='widgetdata'
save_id_arr=['*default','*Path','mouse_doesSlice','time_choice_action', $
             'StartTime','EndTime']

message, '', /cont
print, '  ', fln

;write header for defaults file
openw, u, fln, error=err, /get_lun
IF err EQ 0 then begin
    printf,u,';*****************************************' + $
             '************************************'
    printf,u,';* FILE:'
    printf,u,';*    $PAPCO_HOME/'+name
    printf,u,';*'
    printf,u,';* DESCRIPTION:'
    printf,u,';*    Default settings for PAPCO'
    printf,u,';*    Lines starting with ";" are comments and are ignored'
    printf,u,';* '
    printf,u,';*    Written here are some of the the contents '
    printf,u,';*    of the PAPCO structure "'+struct_name+'"'
    printf,u,';* '
    printf,u,';*    Each variable is described in the preceeding line'
    printf,u,';*    starting with "->". The line contains'
    printf,u,';*'
    printf,u,';*    variable name -> variable type -> a b c d0---d8   '
    printf,u,';*'
    printf,u,';*    where a - TYPE integer'
    printf,u,';*          b - N_ELEMENTS integer'
    printf,u,';*          c - N_DIMENSIONS integer'
    printf,u,';*          d - DIMENSIONS of each array dimension.'
    printf,u,';*'
    printf,u,';* CALLED BY: papco'
    printf,u,';*'
    printf,u,';* CREATED BY: papco_writeDefaultSettings.pro at ' + systime(0)
    printf,u,';*****************************************' + $
             '************************************'
    printf,u,''
ENDIF
close,u  &  free_lun, u

;write structure contents
FOR i=0,n_elements(save_id_arr)-1 DO $
    result=papco_write_struct(fln, struct_name, widgetdata, $
                              TAGID=save_id_arr(i), /APPEND)

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_batch, fName, PREVIEW=PREVIEW
;*
;* DESCRIPTION:
;*      This procedure does the batch-plotting of panel-plots.
;*	When doing a batch-plot, the currently entered panel-plots are drawn
;*	for a list containing orbits and/or time-ranges.
;*	This list is read from a file. The file has this format:
;*	-> empty lines or lines starting with '#' are ignored
;*	-> it contains one orbit and/or time-description per line
;*	-> a line has this format (in BNF) :
;*	      line=orbit [, time-range] | time-range
;*	      orbit = aNumber
;*	      time-range= t90-time "-" t90-time
;*	      t90-time=date [" " time]
;*	      date = doy"/"year | day_om"."month"."year | month"/"day_om"/"year
;*	      (doy=aNumber, year=aNumber, day_om=aNumber, month=aNumber)
;*	      time = hour":"minutes[":"seconds]
;*	      (time=aNumber, minutes=aNumber, seconds=aNumber)
;*
;* INPUTS:
;*      fName	 the file-name including the path
;*
;* KEYWORDS:
;*      PREVIEW  if specified, the output goes to the screen. If not specified,
;*		 the output goes to a file whose name is derived from the
;*		 standard-filename-template.
;*      QUIET    if set, this routine puts out no x-windows
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
PRO papco_batch, fName, PREVIEW=preview, QUIET=quiet

COMMON PLOT_COMPOSER, widgetData

;--make sure there is a plot product defined-----------------------------------
if widgetData.numberOfPlots EQ 0 then begin
    dummy=messageBox(["You can't use the batch-mode until you have", $
                      "added some plots."], ['O.K.'], $
                       /center, title="I can't plot !")
    return
ENDIF

if keyword_Set(preview) then  batch_preview=preview else batch_preview=0

;--open a window displaying the batch-progress --------------------------------
;  this window contains a Cancel-Button, too.
if not keyword_set(QUIET) then begin
    case batch_preview of
      0:begin
        info_str='The generated graphics files will be:'
        output_template=widgetData.printerInfo.fileName_tmpl
      end
      1:begin
        output_template=''
        info_str='The output will be to screen'
      end
      2:BEGIN
        output_template=widgetData.printerInfo.FILENAME_TMPL
        papco_sepdir,output_template,dir, file, ext
        output_template=file+ '.'+$
        strlowcase(widgetData.default_output_format)
        info_str='The generated data output files will be:'
      end
    endcase
    ask=messageBox(['I will use this batch-control-file:', fName, $
                    info_str, output_template], $
                   ['Do it', 'Cancel'], title='Execute batch-file ?')
    if ask EQ 1 then return
    progress_base=widget_base(GROUP_LEADER=widgetData.main_base, $
                               COLUMN=1, title='BATCH - mode')
    lbl_info1=widget_label(progress_base, value='Executing batch-file:')
    lbl_info=widget_label(progress_base, value='  '+fName)
    lbl_dummy=widget_label(progress_base, value='')
    lbl_done=widget_label(progress_base, value= $
                           strcenter('   ',80))
    lbl_current1=widget_label(progress_base, value='current plot:')
    lbl_current=widget_label(progress_base, value= $
                           strcenter(' ',100))
    lbl_dummy=widget_label(progress_base, value='')
    lbl_note=widget_label(progress_base, $
                 value=strcenter("Note: Cancel won't be processed until "+$
                                  "completing the current plot",70))
    pb_cancel=widget_button(progress_base,value='Cancel', uvalue='pb_cancel')
    geo=WIDGET_INFO(progress_base, /GEOMETRY)
    x=600 - geo.xsize/2
    y=300 - geo.ysize/2
    widget_control, progress_base, XOFFSET=x, YOFFSET=y
    widget_control, progress_base, /realize
ENDIF

;--for printing, the variables of widgetData concerning
;  time-information will be modified - keep them in other variables.
keep_orbitInfo=widgetData.default_orbitInfo
keep_endTime=widgetData.endTime
keep_maxTime=widgetData.maxTime
keep_startTime=widgetData.startTime
keep_minTime=widgetData.minTime
keep_time_choice_action=widgetData.time_choice_action

;--loop through batch file lines. Use NEW routine papco_read_batch-------------
;  to read all batch times at once and then close file again.
papco_read_batch, fName, time_str_array, /ALL

n_plots=1  &  n_lines=n_elements(time_str_array)

;-- The output filename will be based on the print template -------------------
orig_fName=widgetdata.printerinfo.filename_tmpl

;--process all lines of time_str_array ----------------------------------------
for i=0, n_lines-1 do begin
    close,/all
    dummy=''
    line=time_str_array(i)

    if strleft(line, 1) eq '#' OR line eq '' then goto, next_line
    formatError='' & orb_str='' & time_str=''

;-- check for any comments in each line and write them to the default
;   "batch.paneltext" file
    p_secol=strpos(line, ';')
    if p_secol gt -1 then begin
      comment=strmid(line, p_secol+1, strlen(line))
      message,'Batch line comment:' + comment, /cont
      fln=papco_getenv('PAPCO_HOME') + $
        'papco_products/panel_text/batch.panel_text'
      openw,unit,fln,/get_lun
        printf,unit, comment
      close,unit & free_lun,unit
      line=strtrim(strmid(line, 0, p_secol),2)
      line_str=line
    ENDIF

;-- check if we need to add a running number to the file name template
    fName=orig_fName
    substituteHash=string(n_plots,format="(i5.5)")
    pos=strpos(fName, '#')    ; substitute '#'
    IF pos GT -1 THEN fName=strleft(fName, pos)+ $
      substituteHash + strfrom(fName, pos+1)
    ;widgetData.saveFileName=fname
    message,'Output filename: '+ fname, /cont

    if not keyword_set(QUIET) then widget_control, lbl_current, set_value=line

    p_comma=strpos(line,',')
    p_blank=strpos(line,' ')
    p_minus=strpos(line,'-')

;--check, if the line contains an orbit - in this case, the line either
;  contains a ',' or it contains neither blanks nor '-'
    if p_comma gt -1 or (p_blank eq -1 and p_minus eq -1) then begin
      if p_comma eq -1 then begin ; -- line contains an orbit only---
        if not isDigits(line) then $
          formatError="line contains only one word that is no orbit" $
        else orb_str=strtrim(line, 2)
      ENDIF else begin ;--the line contains orbit and time-description---
        orb_str=strtrim(strleft(line, p_comma), 2)
        time_str=strtrim(strfrom(line, p_comma+1), 2)
        if not isDigits(orb_str) then $
          formatError="the part left of the ',' is no number"
      endelse
    ENDIF else time_str=strtrim(line, 2) ; the line contains no orbit

    ;--now, time_str is empty or contains the plot-times
    ;       orb_str is empty or contains a number: parse these strings
    orbit=-1
    if orb_str ne '' and formatError EQ '' then orbit=fix(orb_str)

      timeStart={MJDT, mjd:0l, t:0.0}
      timeEnd={MJDT, mjd:0l, t:0.0}
      if time_str ne '' and formatError EQ '' then begin
        p_minus=strpos(time_str, '-')
        if p_minus lt 0 then $
          formatError='incorrect time-format' $
        else begin
          t1_str=strtrim(strleft(time_str, p_minus), 2)
          t2_str=strtrim(strfrom(time_str, p_minus+1), 2)
          tmp1=convert_timeToSeconds(t1_str, /t90, /mjdt)
          tmp2=convert_timeToSeconds(t2_str, /t90, /mjdt)
          ; Next two lines from J. Faden
          tmp1t90=convert_timeToSeconds(t1_str, /t90 )
          tmp2t90=convert_timeToSeconds(t2_str, /t90 )
          if tmp2.err then formatError=tmp2.errText
          if tmp1.err then formatError=tmp1.errText
          if formatError(0) EQ '' then begin
            timeStart=tmp1.value
            timeEnd=tmp2.value
            ; Next two lines from J. Faden
            xut1= tmp1t90.value
            xut2= tmp2t90.value
          ENDIF
        endelse
      ENDIF

      if formatError NE '' then begin
        ;-- if no syntax-error was found, draw the plots
        message, 'SYNTAX-ERROR in line ' + strtrim(string(i+1), 2),/cont
        message, '                       <'+line+'>',/cont
        message, '                --->>>'+formatError,/cont
        print, ''
      ENDIF else begin
        ;-- set the time-variables to the infos contained in 'line'
        print, ''  &  message, 'line '+ varprt(i+1),/cont
        ; update display in the progress-indicator
        if not keyword_set(QUIET) then begin
          widget_control, lbl_done, set_value='doing plot '+ varprt(n_plots)+ $
            ' (line '+varprt(i+1) + ' of ' + varprt(n_lines) + ')'
        ENDIF

           ;if orb_str is of length 8, it contains a date
           IF strlen(orb_str) EQ 8 THEN BEGIN
               tday_t90 = CONVERT_timeToSeconds(orb_str, /T90)
               tday_mjdt= CONVERT_timeToSeconds(orb_str, /T90, /MJDT)
               widgetdata.minTime.t90=tDay_t90.value
               widgetdata.maxTime.t90=tDay_t90.value + 86400
               xut1=widgetdata.minTime.t90
               xut2=widgetdata.minTime.t90
               widgetdata.minTime.mjd=tday_mjdt.value.mjd
               widgetdata.maxTime.mjd=tday_mjdt.value.mjd+1
               widgetdata.minTime.t=0.0
               widgetdata.maxTime.t=0.0
               timeStart = widgetdata.minTime
               timeEnd = widgetdata.maxTime
               widgetData.startTime.mjd=timeStart.mjd
               widgetData.startTime.t=timeStart.t
               widgetData.endTime.mjd=timeEnd.mjd
               widgetData.endTime.t=timeEnd.t
               time_str = 'time by day'
               orb_str = ''
           ENDIF

        if orb_str ne '' then begin
          dummy=papco_setOrbit(orbit)
          ;-- modify the timerange according to widgetData.default_adjust_time
          if widgetData.default_adjust_time ne 0 then papco_time_adjust
          widgetData.time_choice_action=1
        ENDIF else  widgetData.time_choice_action=0

        if time_str ne '' then begin
          widget_control, widgetData.ef_orbit, set_value=orbit
          widgetData.default_orbitInfo=papco_GetOrbitInfoStruct()
          widgetData.endTime.mjd=timeEnd.mjd
          widgetData.endTime.t=timeEnd.t
          widgetData.maxTime.mjd=timeEnd.mjd
          widgetData.maxTime.t=timeEnd.t
          widgetData.startTime.mjd=timeStart.mjd
          widgetData.startTime.t=timeStart.t
          widgetData.minTime.mjd=timeStart.mjd
          widgetData.minTime.t=timeStart.t
          ; Next two lines from J. Faden
          widgetData.startTime.t90= xut1
          widgetData.endTime.t90= xut2
          ;-- modify the timerange according to widgetData.default_adjust_time
          if widgetData.default_adjust_time ne 0 then papco_time_adjust
        ENDIF

        if time_str ne '' then message, '....time: '+$
          convert_secondsToTime(timeStart, /t90, /mjdt) + '/' + $
          convert_secondsToTime(timeEnd, /t90, /mjdt), /cont

        n_plots=n_plots+1
        widgetData.need_to_draw=1
        widgetData.needto_read_data=1	;new time range: re-read data

        case batch_preview of
          1: begin ;---- output to screen ----------------------------------
            draw_win_name='papco_draw_window_' + $
               string(widgetdata.no_draw_window,format="(i2.2)")
            papco_draw_window,draw_win_name    ;create or show draw window
            !P.CHARSIZE=widgetData.default_charsize
            papco_drawPlots

            dummy=messagebox(['Plot:', time_str], ['next', 'cancel'], $
                               xpos=10, ypos=10)

            if dummy EQ 1 then goto, endBatchMode
            widgetData.needto_read_data=1	;new time range: re-read data
          end
          2: begin ;---- output to data file -------------------------------
            papco_PrintPlots, widgetData.printerInfo, /zbuffer
          end
          0: begin ;---- output to printer ---------------------------------
            if not keyword_set(QUIET) then $
              widget_control, lbl_current, set_value= $
              papco_PrinterDescription(widgetData.printerInfo, /FILE)
            message,'Plotting file : '+ $
              papco_PrinterDescription(widgetData.printerInfo, /FILE), /cont
            papco_PrintPlots, widgetData.printerInfo, /FILE
          end
        endcase

        if not keyword_set(QUIET) then begin
          event=widget_event(progress_base, /NOWAIT)
          if event.id EQ pb_cancel then begin
            ask=messageBox(['Cancel-Batch mode ?'], ['Yes', 'No'])
            if ask eq 0 then goto, endBatchMode
          ENDIF
        ENDIF
      endelse                 ; of formatError EQ ''
      widgetdata.printerinfo.filename_tmpl=orig_fName
      next_line:
endfor                   ; of loop going through time_str_array

; the program jumps to this label if the user interrupts further
; printing by pressing the 'cancel'-button
endBatchMode:

; restore the old time-infos
widgetData.saveFileName=orig_fName
widgetData.default_orbitInfo=keep_orbitInfo
widgetData.endTime=keep_endTime
widgetData.maxTime=keep_maxTime
widgetData.startTime=keep_startTime
widgetData.minTime=keep_minTime
widgetData.time_choice_action=keep_time_choice_action
if batch_preview then papco_Clear_Drawing
if not keyword_set(QUIET) then widget_control, progress_base, /DESTROY
if not keyword_set(QUIET) then papco_Refresh, /ALL
return

END

;******************************************************************************
;* PROCEDURE:
;*   PRO papco_write_batch, fileName, batch_times
;*
;* DESCRIPTION:
;*   wirites the contents of batch_times to a papco bacth file.
;*
;* MODIFICATION HISTORY:
;*   January 2001, written by R. Friedel
;******************************************************************************
PRO papco_write_batch, fileName, batch_times

;--parse filename to see if it contains a path, correct extension -------------
  papco_sepdir, fileName, path, file, ext
  ;if there is no path, use default
  if path eq '' then path=papco_getenv('PAPCO_HOME')+'papco_products/batch/'
  ext='.papco_batch'  ;force correct extension
  fName=path+file+ext

  message,fName,/cont

  openw, u, fName, /GET_LUN
    for i=0,n_elements(batch_times)-1 do printf, u, batch_times(i)
  close,u  &  free_lun,u

end

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_read_batch, fName
;*
;* DESCRIPTION:
;*      This procedure reads an entry from a batch file and sets the time
;*      accordingly. The entry number is set by widgetData.batch_count
;*	The batch file has this format:
;*	-> empty lines or lines starting with '#' are ignored
;*	-> it contains one orbit and/or time-description per line
;*	-> a line has this format (in BNF) :
;*	      line=orbit [, time-range] | time-range
;*	      orbit = aNumber
;*	      time-range= t90-time "-" t90-time
;*	      t90-time=date [" " time]
;*	      date = doy"/"year | day_om"."month"."year | month"/"day_om"/"year
;*	      (doy=aNumber, year=aNumber, day_om=aNumber, month=aNumber)
;*	      time = hour":"minutes[":"seconds]
;*	      (time=aNumber, minutes=aNumber, seconds=aNumber)
;*
;* INPUTS:
;*      fName	 the file-name including the path
;*
;* KEYWORDS:
;*      ALL      read all the times in file into time_str
;*
;* MODIFICATION HISTORY:
;*      April 1999, written by R. Friedel
;******************************************************************************
PRO papco_read_batch, fileName, time_str, ALL=ALL

COMMON PLOT_COMPOSER, widgetData
COMMON get_error, get_err_no, get_err_msg

;--parse filename to see if it contains a path, correct extension -------------
papco_sepdir, fileName, path, file, ext
;if there is no path, use default
if path eq '' then path=papco_getenv('PAPCO_HOME')+'papco_products/batch/'
ext='.papco_batch'  ;force correct extension
fName=path+file+ext
;make sure file exists
result=findfile(fName, count=n)
if n ne 1 then begin
    msg='File '+fName+' not found'
    get_err_msg=msg  &  get_err_no=1  &  message, msg, /cont
    return
ENDIF

;--open the batch file for reading---------------------------------------------
openr, inputUnit, fName, ERROR=err, /GET_LUN
on_ioerror, ioError

;--for printing, the variables of widgetData concerning
;  time-information will be modified - keep them in other variables.
keep_orbitInfo=widgetData.default_orbitInfo
keep_endTime=widgetData.endTime
keep_maxTime=widgetData.maxTime
keep_startTime=widgetData.startTime
keep_minTime=widgetData.minTime
keep_time_choice_action=widgetData.time_choice_action

;-- process all lines of the input-file until batch_count is found -----------
lineCount=-1
while not eof(inputUnit) do begin
    dummy=''
    readf, inputUnit, dummy
    line=strtrim(dummy, 2)
    if strleft(line, 1) NE '#' AND line NE '' then BEGIN ;line not a comment
        formatError='' & orb_str='' & time_str=''
        lineCount=lineCount+1

        if keyword_set(ALL) then begin
            if n_elements(time_str_arr) eq 0 then time_str_arr=line else $
              time_str_arr=[time_str_arr,line]
            goto, next_line
        ENDIF

        IF lineCount LT widgetdata.batch_count THEN GOTO, next_line

;-- check for any comments in each line and write them to the default
;   "batch.paneltext" file
      p_secol=strpos(line, ';')
      if p_secol gt -1 then begin
        comment=strmid(line, p_secol+1, strlen(line))
        fln=papco_getenv('PAPCO_HOME') + $
          'papco_products/panel_text/batch.panel_text'
        openw,unit,fln,/get_lun
        printf,unit, comment
        close,unit & free_lun,unit
        line=strtrim(strmid(line, 0, p_secol),2)
      ENDIF

;-- check if we need to add a running number to the papco_product name
      fName=widgetData.saveFileName
      orig_fName=fName
      substituteHash=string(lineCount,format="(i4.4)")
      pos=strpos(fName, '#')    ; substitute '#'
      IF pos GT -1 THEN fName=strleft(fName, pos)+ $
        substituteHash + strfrom(fName, pos+1)
      widgetData.saveFileName=fname

      p_comma=strpos(line, ',')
      p_blank=strpos(line, ' ')
      p_minus=strpos(line, '-')

;--check, if the line contains an orbit - in this case, the line either
;  contains a ',' or it contains neither blanks nor '-'
     if p_comma gt -1 or (p_blank eq -1 and p_minus eq -1) then begin
       if p_comma eq -1 then begin ; -- line contains an orbit only
         if not isDigits(line) then begin
           formatError="line contains only one word that is no orbit"
         ENDIF else begin
          orb_str=strtrim(line, 2)
          endelse
       ENDIF else begin ;--the line contains orbit and time-description---
         orb_str=strtrim(strleft(line, p_comma), 2)
         time_str=strtrim(strfrom(line, p_comma+1), 2)
         if not isDigits(orb_str) then $
           formatError="the part left of the ',' is no number"
         endelse
       ENDIF else begin ; the line contains no orbit
         time_str=strtrim(line, 2)
       endelse

       ;--now, time_str is empty or contains the plot-times
       ;       orb_str is empty or contains a number: parse these strings
       orbit=-1
       if orb_str ne '' and formatError EQ '' then orbit=fix(orb_str)

       timeStart={MJDT, mjd:0l, t:0.0}
       timeEnd={MJDT, mjd:0l, t:0.0}
       if time_str ne '' and formatError EQ '' then begin
            p_minus=strpos(time_str, '-')
            if p_minus lt 0 then $
              formatError='incorrect time-format' $
            else begin
               t1_str=strtrim(strleft(time_str, p_minus), 2)
               t2_str=strtrim(strfrom(time_str, p_minus+1), 2)
               tmp1=convert_timeToSeconds(t1_str, /t90, /mjdt)
               tmp2=convert_timeToSeconds(t2_str, /t90, /mjdt)
               ; Next two lines from J. Faden
               tmp1t90=convert_timeToSeconds(t1_str, /t90 )
               tmp2t90=convert_timeToSeconds(t2_str, /t90 )
               if tmp2.err then formatError=tmp2.errText
               if tmp1.err then formatError=tmp1.errText
               if formatError(0) EQ '' then begin
                  timeStart=tmp1.value
                  timeEnd=tmp2.value
                  ; Next two lines from J. Faden
                  xut1= tmp1t90.value
                  xut2= tmp2t90.value
               ENDIF
            endelse
       ENDIF

        ; if no syntax-error was found, set the time!
       if formatError NE '' then begin
         print, '% papco_batch: SYNTAX-ERROR in line ' + $
                                strtrim(string(lineCount), 2)
         print, '% papco_batch:                       <'+line+'>'
         print, '% papco_batch:                --->>>'+formatError
         print, '% papco_batch: '
       ENDIF else begin
          ;-- set the time-variables to the infos contained in 'line'
          ;if orb_str is of length 8, it contains a date
           IF strlen(orb_str) EQ 8 THEN BEGIN
               tday_t90 = CONVERT_timeToSeconds(orb_str, /T90)
               tday_mjdt= CONVERT_timeToSeconds(orb_str, /T90, /MJDT)
               widgetdata.minTime.t90=tDay_t90.value
               widgetdata.maxTime.t90=tDay_t90.value + 86400
               widgetdata.minTime.mjd=tday_mjdt.value.mjd
               widgetdata.maxTime.mjd=tday_mjdt.value.mjd+1
               widgetdata.minTime.t=0.0
               widgetdata.maxTime.t=0.0
               timeStart = widgetdata.minTime
               timeEnd = widgetdata.maxTime
               widgetData.startTime.mjd=timeStart.mjd
               widgetData.startTime.t=timeStart.t
               widgetData.endTime.mjd=timeEnd.mjd
               widgetData.endTime.t=timeEnd.t
               time_str = 'time by day'
               orb_str = ''
               goto, normal_out
           ENDIF

          if orb_str ne '' then begin
            dummy=papco_setOrbit(orbit)
            ;-- modify the timerange according to widgetData.adjust_time
            if widgetData.default_adjust_time ne 0 then papco_time_adjust
          ENDIF

          if time_str ne '' then begin
            widget_control, widgetData.ef_orbit, set_value=orbit
            widgetData.default_orbitInfo=papco_GetOrbitInfoStruct()
            widgetData.endTime.mjd=timeEnd.mjd
            widgetData.endTime.t=timeEnd.t
            widgetData.maxTime.mjd=timeEnd.mjd
            widgetData.maxTime.t=timeEnd.t
            widgetData.startTime.mjd=timeStart.mjd
            widgetData.startTime.t=timeStart.t
            widgetData.minTime.mjd=timeStart.mjd
            widgetData.minTime.t=timeStart.t
            ; Next two lines from J. Faden
            widgetData.startTime.t90= xut1
            widgetData.endTime.t90= xut2
            ;-- modify the timerange widgetData.default_adjust_time
            if widgetData.default_adjust_time ne 0 then papco_time_adjust
          ENDIF
        endelse                 ; of formatError EQ ''
        widgetData.saveFileName=orig_fName
        if lineCount eq widgetdata.batch_count then goto, normal_out
    ENDIF                       ; of 'line empty or comment ?'
    next_line:
  endwhile ; of while not eof(inputUnit)

normal_out:

  if keyword_set(ALL) then begin
    time_str=time_str_arr
    widgetData.default_orbitInfo=keep_orbitInfo
    widgetData.endTime=keep_endTime
    widgetData.maxTime=keep_maxTime
    widgetData.startTime=keep_startTime
    widgetData.minTime=keep_minTime
    widgetData.time_choice_action=keep_time_choice_action
    return
  ENDIF

  widgetData.need_to_draw=1
  widgetData.needto_read_data=1	;new time range: re-read data

  line_str='line ' + varprt(lineCount)+': '
  if time_str ne '' then out_str=convert_secondsToTime(timeStart,/t90,/mjdt)+$
    ' - ' + convert_secondsToTime(timeEnd,/t90,/mjdt) $
  else out_str='No valid time'
  message,line_str+out_str, /cont

  papco_refresh, /TIME
  close, inputUnit  &  close, /all  & free_lun, inputUnit

  times=str_sep(time_str,' - ')
  if n_elements(times) gt 1 then time_str=times(0)+'-'+times(1)
  return

; this label is called, if an io-error occured while reading
; from the fName-file
ioError:
  close, inputUnit
  free_lun, inputUnit
  dummy=messagebox(['An IO-Error occured in line '+ $
                     strtrim(string(lineCount),2),$
                    'of '+fName, $
                    '', !err_String], $
                   [':-('], title='IO - error in batch-file')

  widgetData.default_orbitInfo=keep_orbitInfo
  widgetData.endTime=keep_endTime
  widgetData.maxTime=keep_maxTime
  widgetData.startTime=keep_startTime
  widgetData.minTime=keep_minTime
  widgetData.time_choice_action=keep_time_choice_action

  time_str=''
  papco_Refresh, /ALL
  return

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_edit_batch_event_event, event
;*
;* DESCRIPTION:
;*	This processes events from the XMANAGER.
;*	As this example does not extend the basic functionality of plot-
;*	editors, it directly calls the default-handler, papco_PANEL_Event
;*
;* INPUTS:
;*	event		an XManager event
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*      June 1998, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_batch_event, event

COMMON edit_batch_data, editwidgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

;test for any of the time entries to be correct format.
if strmid(uval,0,11) eq 'ef_series_t' then begin
    this_mjdt=CONVERT_timeToSeconds(event.value(0), /T90, /MJDT)
    if this_mjdt.err ne 0 then begin
      dummy=messageBox(this_mjdt.errtext, ['Ok'], title='Wrong time format')
      return
    ENDIF
    result=convert_secondsToTime(this_mjdt.value, /T90, /MJDT)
ENDIF

case uval of
    'ef_series_t1': begin
      editwidgetData.soptions(0)=result(0)
      widget_control, editwidgetData.ef_series_t1, SET_VALUE=result(0)
      papco_edit_batch_refresh, /LIST
    end
    'ef_series_t2': begin
      editwidgetData.soptions(1)=result(0)
      widget_control, editwidgetData.ef_series_t2, SET_VALUE=result(0)
      papco_edit_batch_refresh, /LIST
    end
    'ef_series_dl': begin
      editwidgetData.soptions(2)=event.value(0)
      papco_edit_batch_refresh, /DELTA, /LIST
    end
    'ef_series_d2': begin
      editwidgetData.soptions(3)=event.value(0)
      papco_edit_batch_refresh, /DURATION, /LIST
    end
    'ef_series_o1': begin
      editwidgetData.ioptions(0)=event.value(0)
      papco_edit_batch_refresh, /ORBIT
    end
    'ef_series_o2': begin
      editwidgetData.ioptions(1)=event.value(0)
      papco_edit_batch_refresh, /ORBIT
    end
    'spb_save_series': begin
      message,'Writing user file',/cont
      print,'  ',editwidgetData.PATH+editwidgetData.FILENAME
      widget_control,editwidgetData.auto_series_lst, GET_VALUE=times_arr
      openw,unit,editwidgetData.PATH+editwidgetData.FILENAME, /get_lun
      for i=0,n_elements(times_arr)-1 do printf,unit,times_arr(i)
      close,unit  &  free_lun,unit
      editwidgetData.status=1
      papco_panel_Refresh, editwidgetData,  /ADD_FILE
    end

    else: begin
      widgetData=papco_panel_Event(event, editwidgetData)
      if xregistered('papco_edit_batch') then $
        if editwidgetData.status then $
        widget_control,editwidgetData.spb_save_series,sensitive=1 else $
        widget_control,editwidgetData.spb_save_series,sensitive=0
    end
ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_edit_batch_event_refresh
;*
;* DESCRIPTION:
;*	This processes refreshes papco_edit_batch widget
;*
;* INPUTS:
;*	none
;*
;* KEYWORDS:
;* 	refresh actions to be taken
;*
;* CALLING SEQUENCE:
;*	called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*      September 2000, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_batch_refresh, ALL=ALL, LIST=LIST, DELTA=DELTA, ORBIT=ORBIT, $
                              DURATION = DURATION

COMMON edit_batch_data, editwidgetData

if keyword_set(ALL) then begin
    DELTA=1 & DURATION = 1
    LIST=1
    ORBIT=1
END

IF keyword_set(DURATION) THEN BEGIN
    in_str=editwidgetData.soptions(3)
    case in_str of
      'day'  : begin
        days=1  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'week': begin
        days=7  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'month': begin  ;special case
        dec_d=-1.0d
        dl_str='month'
      end
      'year' : begin  ;special case
        dec_d=-2.0d
        dl_str='year'
      end

      else: begin               ;valid fomat is ddd/hh:mm:ss
        dl=in_str
        ;look for ddd string - either before /, or length 3
        p=strpos(dl,'/')
        if p eq -1 then begin
          if strlen(dl) eq 3 then days=fix(dl) else days=0
        ENDIF else begin
          days=fix(strmid(dl,0,3))
          dl=strmid(dl,p+1,strlen(dl)-p-1)
        endelse
        ;parse time string
        t_arr=str_sep(dl,':')
        if t_arr(0) ne '' then hr=fix(t_arr(0)) else hr=0
        if n_elements(t_arr) gt 1 then mn=fix(t_arr(1)) else mn=0
        if n_elements(t_arr) gt 2 then s=fix(t_arr(2)) else s=0.
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      endelse
    endcase
    editwidgetData.soptions(3)=dl_str
    widget_control, editwidgetData.ef_series_d2, SET_VALUE=dl_str
    editwidgetData.delta2=dec_d
ENDIF

if keyword_set(DELTA) then begin
    in_str=editwidgetData.soptions(2)
    case in_str of
      'day'  : begin
        days=1  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'week': begin
        days=7  & hr=0  &  mn=0  & s=0
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      end
      'month': begin  ;special case
        dec_d=-1.0d
        dl_str='month'
      end
      'year' : begin  ;special case
        dec_d=-2.0d
        dl_str='year'
      end

      else: begin               ;valid fomat is ddd/hh:mm:ss
        dl=in_str
        ;look for ddd string - either before /, or length 3
        p=strpos(dl,'/')
        if p eq -1 then begin
          if strlen(dl) eq 3 then days=fix(dl) else days=0
        ENDIF else begin
          days=fix(strmid(dl,0,3))
          dl=strmid(dl,p+1,strlen(dl)-p-1)
        endelse
        ;parse time string
        t_arr=str_sep(dl,':')
        if t_arr(0) ne '' then hr=fix(t_arr(0)) else hr=0
        if n_elements(t_arr) gt 1 then mn=fix(t_arr(1)) else mn=0
        if n_elements(t_arr) gt 2 then s=fix(t_arr(2)) else s=0.
        dec_d=double(days)+(hr*3600.0d +mn*60.0d +double(s))/86400.0d
        dl_str=string(days,hr,mn,s,format="(i3.3,'/',i2.2,':',i2.2,':',i2.2)")
      endelse
    endcase
    editwidgetData.soptions(2)=dl_str
    widget_control, editwidgetData.ef_series_dl, SET_VALUE=dl_str
    editwidgetData.delta=dec_d
ENDIF

IF keyword_set(LIST) THEN BEGIN

    if editwidgetData.soptions(0) ne '' and $
       editwidgetData.soptions(1) ne '' and $
       editwidgetData.soptions(2) ne '' then begin

      widget_control,HOURGLASS=1

      res=CONVERT_timeToSeconds(editwidgetData.soptions(0), /T90, /MJDT)
      st_mjdt=res.value
      res=CONVERT_timeToSeconds(editwidgetData.soptions(1), /T90, /MJDT)
      en_mjdt=res.value

      st=double(st_mjdt.mjd)+double(st_mjdt.t / 86400.0d)
      en=double(en_mjdt.mjd)+double(en_mjdt.t / 86400.0d)
      dl=editwidgetData.delta
      d2=editwidgetData.delta2

      if dl eq 0 then return

      if dl lt 0.0 then begin   ;special cases of steps
        n_times=(en-st)/30 +2  &  times_arr=strarr(n_times)  & line=0
        while st lt en do begin
          t1_str=convert_secondsToTime(st_mjdt,/T90, /MJDT)
          convert_t90_to_Date, st_mjdt, year, day_oy, hour, minute, second, $
            month, cmonth, day_om, /MJDT

          if dl eq -1.0 then begin
            month=month+1
            if month gt 12 then begin
              month=1l  &  year=year+1
            ENDIF
          ENDIF
          if dl eq -2.0 then year=year+1

          res=convert_date_to_t90(day=day_om, month=month, year=year, $
                                  hour=hour, minutes=minute, $
                                  seconds=second, /MJDT)
          t2_str=convert_secondsToTime(res, /T90, /MJDT)
          times_arr(line)=t1_str+' - '+t2_str
          st=double(res.mjd)+double(res.t / 86400.0d)
          st_mjdt=res  &  line=line+1
        endwhile
        goto, done_list
      ENDIF

      ;make string array of times
      n_times=(en-st)/dl + 1  &  times_arr=strarr(n_times)  & line=0
      while st lt en do begin
        t1=st
        IF d2 NE 0 THEN t2=st+d2 ELSE t2 = st+dl
        t1_str=convert_secondsToTime({mjd:long(t1),t:(t1-long(t1))*86400.0}, $
                                     /T90, /MJDT)
        t2_str=convert_secondsToTime({mjd:long(t2),t:(t2-long(t2))*86400.0}, $
                                     /T90, /MJDT)
        times_arr(line)=t1_str+' - '+t2_str
        st=t1+dl  &  line=line+1
      endwhile

      done_list:
      times_arr=times_arr(0:line-1)
      widget_control, editwidgetData.auto_series_lst, SET_VALUE=times_arr
      widget_control,HOURGLASS=0
    ENDIF
ENDIF

if keyword_set(ORBIT) then begin
    st_orbit=editwidgetData.ioptions(0)
    en_orbit=editwidgetData.ioptions(1)
    if en_orbit gt st_orbit then begin
      widget_control,HOURGLASS=1
      orbit_arr=strarr(en_orbit-st_orbit+1)
      for i=0, n_elements(orbit_arr)-1 do $
        orbit_arr(i)=string(st_orbit+i,format="(i5.5)")
      widget_control, editwidgetData.auto_series_lst, SET_VALUE=orbit_arr
      widget_control,HOURGLASS=0
    ENDIF

ENDIF

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_edit_Batch
;*
;* DESCRIPTION:
;*      Allows interactive editing of batch files. Makes use of a lot
;*      of features from panel editors!
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      FILENAME  set to a variable returning the chosen Batch File
;*      BATCH_TIMES set to a variable containing an array of batch times
;*
;* MODIFICATION HISTORY:
;*      June 1998, written by Reiner Friedel
;******************************************************************************
PRO papco_edit_batch, FILENAME=FILENAME, BATCH_TIMES=BATCH_TIMES, GROUP=GROUP

COMMON plot_composer, widgetdata
COMMON edit_batch_data, editwidgetData

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=widgetdata.base

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

aPlotInfo=papco_getplotinfostruct()
additionalWData={canceled:1}

dirpath=widgetData.BATCHFILEPATH
papco_sepdir, widgetData.BATCHFILENAME, dir, file
aPlotInfo.filename=file
aPlotInfo.soptions(0:2)=widgetData.BATCHSERIES

filter='*.papco_batch'
if keyword_set(FILENAME) then title='PAPCO Select Batch File' else $
                              title='PAPCO Edit Batch File'

;--- create the base-window ---------------------------------------------------
panel_base=papco_WIDGET_BASE(title=title, column=1, GROUP_LEADER=group)

;--- some panels might need to read/input to text files (event, text ----------
;    module). So a generic papco widget is supplied for this.
widgets=papco_panel_AddFile_Widgets(panel_base, aPlotInfo, $
                                    DIRPATH=dirpath, FILTER=filter)
additionalWData=create_struct(widgets, additionalWData)

;--- Add some widgets that allow automatic generation of time series ----------
auto_series_base=widget_base(panel_base,column=2,/frame)

auto_series_info=widget_base(auto_series_base,column=1, /ALIGN_CENTER)
ef_series_t1=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Start Date:', $
                        VALUE=aPlotInfo.soptions(0), $
                        UVALUE= 'ef_series_t1')
ef_series_t2=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series End Date:  ', $
                        VALUE=aPlotInfo.soptions(1), $
                        UVALUE= 'ef_series_t2')
ef_series_dl=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Time Step: ', $
                        VALUE=aPlotInfo.soptions(2), $
                        UVALUE= 'ef_series_dl')
ef_series_d2=CW_FIELD(auto_series_info,/STRING,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Time Duration: ', $
                        VALUE=aPlotInfo.soptions(3), $
                        UVALUE= 'ef_series_d2')

ef_series_o1=CW_FIELD(auto_series_info,/INTEGER,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series Start Orbit:  ', $
                        VALUE=aPlotInfo.ioptions(0), $
                        UVALUE= 'ef_series_o1')
ef_series_o2=CW_FIELD(auto_series_info,/INTEGER,/RETURN_EVENTS,XSIZE=19,$
                        /COLUMN, TITLE='Series End Orbit: ', $
                        VALUE=aPlotInfo.ioptions(1), $
                        UVALUE= 'ef_series_o2')

;--- Add file action buttons --------------------------------------------------
file_action_base=widget_base(auto_series_info,/ALIGN_BOTTOM)
spb_save_series=widget_button(file_action_base, $
                                VALUE=' Save Series to File ', $
                                UVALUE='spb_save_series')
if aPlotInfo.filename eq '' then widget_control,spb_save_series,sensitive=0

if keyword_set(BATCH_TIMES) then batch_times=BATCH_TIMES else batch_times=''
auto_series_lst=widget_text(auto_series_base, scr_xsize=290, $
                            value=batch_times, $
                            scr_ysize=380, /SCROLL)

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You ----------
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'papco_edit_batch.help', $
                 soptions:aplotinfo.soptions, $
                 ioptions:aplotinfo.ioptions, $
                 delta:0d, delta2:0d, $
                 ef_series_t1:ef_series_t1, $
                 ef_series_t2:ef_series_t2, $
                 ef_series_dl:ef_series_dl, $
                 ef_series_d2:ef_series_d2, $
                 ef_series_o1:ef_series_o1, $
                 ef_series_o2:ef_series_o2, $
                 spb_save_series:spb_save_series, $
                 auto_series_lst:auto_series_lst}

editwidgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

;--- realize the window -------------------------------------------------------
papco_edit_batch_refresh, /ALL
papco_panel_Refresh, editwidgetData,  /ADD_FILE
papco_Panel_Realize, panel_base, editwidgetData

papco_xmanager, 'papco_edit_batch', panel_base

if editwidgetData.CANCELED eq 0 then begin
    widgetData.BATCHFILEPATH=editwidgetData.PATH
    widgetData.BATCHFILENAME=editwidgetData.FILENAME
    widgetData.BATCHSERIES=editwidgetData.soptions(0:2)
ENDIF

END

;******************************************************************************
PRO papco_select_from_batch, fName, GROUP=group

COMMON plot_composer, widgetData
COMMON select_from_batch_data, thiswidgetData

WIDGET_CONTROL, HOURGLASS=1

; if GROUP is not specified, the group-leader will be the desktop
IF NOT KEYWORD_SET(GROUP) THEN GROUP=0

; set the label for the action-button
IF NOT KEYWORD_SET(ACTION) THEN ACTION='Done'

aPlotInfo=papco_getplotinfostruct()
additionalWData={canceled:1}

;--- create the base-window ---------------------------------------------------
panel_base=WIDGET_BASE(title='PAPCO Batch: Select Entry', $
                             column=1, GROUP_LEADER=group)

batch_lbl=WIDGET_LABEL(panel_base, VALUE='File: ...'+strright(fName,45))

;--- read all the times in batch file -----------------------------------------
message,'Reading batch File: ...'+strright(fName,60),/cont
papco_read_batch, fName, fln_time_str, /ALL
time_str = fln_time_str
for i=0, n_elements(fln_time_str)-1 do $
    time_str(i)=strformat(varprt(i),5,/left)+':  '+time_str(i)

;read exisiting time label (last chosen one) and initialise widget
;list at that position.
widget_control, widgetdata.label_batch_time, GET_VALUE = this_time_str
result = strsplit(this_time_str, '-', /EXTRACT)
this_time_str = result(0) + ' - '+result(1)
result = strpos(fln_time_str, this_time_str)
idx = where(result NE -1, c)
IF c EQ 0 THEN list_idx = 0 ELSE list_idx = idx(0)

batch_list=WIDGET_LIST(panel_base, /frame, $
                       SCR_XSIZE=450, SCR_YSIZE=600, $
                       VALUE=time_str, $
                       UVALUE='batch_list')

widget_control, batch_list, SET_LIST_SELECT = list_idx
widget_control, batch_list, SET_LIST_TOP = (list_idx-15) > 0

;--- finally, the Actionbuttons (help, cancel, done) are added to the ---------
;    editor window.
widgets=papco_panel_AddActionButtons(panel_base, Action, aPlotInfo)
additionalWData=create_struct(widgets, additionalWData)

;--- now, create the structure that keeps the panel-window data. You ----------
;    always need to specify the help-file, and your structure must
;    always contain the 'additionalWData'.
thisWidgetsData={helpfile:'papco_edit_batch.help',$
                 panel_base:panel_base}
thiswidgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)

WIDGET_CONTROL, panel_base, HOURGLASS=1

;--- realize the window -------------------------------------------------------
papco_Panel_Realize, panel_base, thiswidgetData

; As this is a function that will return all changes made by the
; user, the xmanager has to be called with the keyword MODAL. If we
; wouldn't specify /MODAL, no changes would be reportet but the
; data contained in aPlotInfo would be returned.
xmanager, 'papco_select_from_batch', panel_base

END

;******************************************************************************
PRO papco_select_from_batch_event, event

COMMON select_from_batch_data, thiswidgetData
COMMON PLOT_COMPOSER, widgetData

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF

    'batch_list': BEGIN 
        widgetData.batch_count=event.index
        fln=widgetData.BATCHFILEPATH+widgetData.BATCHFILENAME
        papco_read_batch, fln, time_str
        widget_control,widgetdata.label_batch_time, SET_VALUE=time_str
        papco_Refresh, /new_timeRange
    END 

    ELSE: BEGIN 
        thiswidgetData=papco_panel_Event(event, thiswidgetData)
    END 

ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      papco_save_compiled
;*
;* DESCRIPTION:
;*      saves the current status of all compiled routines for quick startup
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_save_compiled
;******************************************************************************
pro papco_save_compiled

   dir_path=papco_getenv('PAPCO_HOME')
   rfln='papco_routines.save'
   save,filename=dir_path+rfln,/routines,/verbose
   message,'Written ' + dir_path+rfln, /cont

end

;******************************************************************************
;* PROCEDURE:
;*      papco_save_current_state
;*
;* DESCRIPTION:
;*      saves the previous state of the panel editor to file. Panel Editor
;*      state files are the same as papco product files.
;*      Naming convention: state_00001.papco, etc, stored in the users
;*      papco_user/papco_products directory.
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_save_current_state
;******************************************************************************
pro papco_save_current_state

  COMMON product_history, current, last

  last=last+1  &   current=last

; make the new state file filename and save
  path=papco_addpath(papco_getenv('PAPCO_CONFIG'),'papco_state')
;test for directory - make if it does not exisit!
  if not papco_finddir(path) then papco_create_dir,path, /NOASK
  fname=papco_fname(path+'state_'+string(last,format="(i5.5)")+'.papco')
  papco_saveState, FILENAME=fname

;for vms, purge the directory of old state file version numbers!
  os=!version.os_family
  if os eq 'VMS' then spawn,'purge '+path+'*.*'

end

;******************************************************************************
;* PROCEDURE:
;*      papco_load_previous_state
;*
;* DESCRIPTION:
;*      loads up the previous state of the panel editor from
;*      file. Panel Editor state files are the same as papco product files.
;*      naming convention: state_00001.papco, etc, stored in the users
;*      papco_user/papco_products directory.
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_load_previous_state
;******************************************************************************
pro papco_load_previous_state

  COMMON product_history, current, last
  COMMON PLOT_COMPOSER, widgetData

  IF widgetData.default_states THEN BEGIN
    message, 'Disabled by defaults', /cont
    return
  ENDIF

  current=current-1

  if current eq 0 then begin
    message,'No further previous state saved', /cont
    current=1  &  return
  ENDIF

; make the new state file filename and load
  path=papco_addpath(papco_getenv('PAPCO_CONFIG'),'papco_state')
;test for directory - make if it does not exisit!
  if not papco_finddir(path) then papco_create_dir,path
  fname=papco_fname(path+'state_'+string(current,format="(i5.5)")+'.papco')
  papco_Loadstate, FILENAME=fname, /STATE, VERBOSE=0

end

;******************************************************************************
;* PROCEDURE:
;*      papco_load_next_state
;*
;* DESCRIPTION:
;*      loads up the previous state of the panel editor from
;*      file. Panel Editor state files are the same as papco product files.
;*      naming convention: state_00001.papco, etc, stored in the users
;*      papco_user/papco_products directory.
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_load_next_state
;******************************************************************************
pro papco_load_next_state

  COMMON product_history, current, last
  COMMON PLOT_COMPOSER, widgetData

  IF widgetData.default_states THEN BEGIN
    message, 'Disabled by defaults', /cont
    return
  ENDIF

  current=current+1

  if current gt last then begin
    message,'No further next state saved', /cont
    current=last  &  return
  ENDIF

; make the new state file filename and load
  path=papco_addpath(papco_getenv('PAPCO_CONFIG'),'papco_state')
;test for directory - make if it does not exisit!
  if not papco_finddir(path) then papco_create_dir,path
  fname=papco_fname(path+'state_'+string(current,format="(i5.5)")+'.papco')
  papco_Loadstate, FILENAME=fname, /STATE, VERBOSE=0

end

;******************************************************************************
;* PROCEDURE:
;*      papco_load_last_state
;*
;* DESCRIPTION:
;*      loads up the last state of the panel editor from file. Used to
;*      recover from a crashed session. A last file should only exist
;*      if papco was exited abnormnally.
;*      Panel Editor state files are the same as papco product files.
;*      naming convention: state_00001.papco, etc, stored in the users
;*      papco_user/papco_products directory.
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      none
;*
;* CALLING SEQUENCE:
;*      papco_load_last_state
;******************************************************************************
pro papco_load_last_state

COMMON product_history, current, last
COMMON PLOT_COMPOSER, widgetData

IF widgetData.default_states THEN BEGIN
    message, 'Disabled by defaults', /cont
    return
ENDIF

path=papco_addpath(papco_getenv('PAPCO_CONFIG'),'papco_state')
; test for directory - make if it does not exisit!
if not papco_finddir(path) then papco_create_dir,path
; get list of exisiting state files and work out last state file number.
files=findfile(path,count=n)
pos = strpos(files, 'state_')
idx = where(pos NE -1, n)
if n lt 1 then begin
    message,'No last state saved', /cont
    return
ENDIF else begin                ;read file
    files = files(idx)
    last=fix(strmid(files(n-1),strlen(path)+6,5))
    current=last
    print,''
    message,'Reading papco state file from previous crashed session.', /cont
    papco_Loadstate, FILENAME=path+files(n-1), /STATE, VERBOSE=0
    print,''
endelse

END


PRO papco_set_default_widget_font_event, event

COMMON PLOT_COMPOSER, widgetData
COMMON set_default_widget_font_data, fontwidgetData

WIDGET_CONTROL, GET_UVALUE=uval, event.id

case uval of
    'lst_font': begin
       fontwidgetData.default_font=fontwidgetData.fnames(event.index)
    end
    'spb_done':begin
        WIDGET_CONTROL, fontwidgetData.ef_size, GET_VALUE = x
        WIDGET_CONTROL, fontwidgetData.base, /DESTROY
        new_font=fontwidgetData.default_font+'*'+$
                 fontwidgetData.weight_names(fontwidgetData.weight)+'*'+$
                 fontwidgetData.quality_names(fontwidgetData.quality)+'*'+$
                 fontwidgetData.pitch_names(fontwidgetData.pitch)+'*'+$
                 varprt(x)
        print, new_font
        widgetData.default_font=new_font
    end
    'spb_weight': begin
        fontwidgetData.weight = event.value
    end
    'spb_quality': begin
        fontwidgetData.quality = event.value
    end
    'spb_pitch': begin
        fontwidgetData.pitch = event.value
    end
ENDCASE



end

;******************************************************************************
;* procedure:    papco_set_default_widget_font
;*
;* description:  creates a widget for the user to choose a default font
;*               for all widgets.
;*
;* inputs:       none
;*
;* output:       none
;*
;* keywords:     none
;*
;* author:       R. Friedel, May 2003
;*
;* changes:      -
;******************************************************************************

pro papco_set_default_widget_font

COMMON PLOT_COMPOSER, widgetData
COMMON set_default_widget_font_data, fontwidgetData

CASE STRUPCASE(!version.os_family) OF
    'UNIX': BEGIN
        result=xfont(group=widgetdata.main_base)
        if result ne '' then begin
            widgetData.default_font=result
            result=messageBox(['New Font is set.', $
                         'To take effect, exit and restart PAPCO'], ['OK'], $
                          title='New Font')
            PAPCO_writeDefaultSettings
        ENDIF
    END
    'VMS' : BEGIN
        result=xfont(group=widgetdata.main_base)
        if result ne '' then begin
            widgetData.default_font=result
            result=messageBox(['New Font is set.', $
                         'To take effect, exit and restart PAPCO'], ['OK'], $
                          title='New Font')
            PAPCO_writeDefaultSettings
        ENDIF
    END
    'WINDOWS': BEGIN
        ;get list of font names
        DEVICE, GET_FONTNAMES=fnames, SET_FONT='*'
        weight_names  = ['THIN','LIGHT','BOLD','HEAVY']
        quality_names = ['DRAFT', 'PROOF']
        pitch_names   = ['FIXED', 'VARIABLE']

        r=STRSPLIT(widgetData.default_font,'*',/EXTRACT)
        font=r(0)

        idx=where(weight_names eq r(1),c)
        if c ne 0 then weight=idx(0) else weight=1

        idx=where(quality_names eq r(2),c)
        if c ne 0 then quality=idx(0) else quality=1

        idx=where(pitch_names eq r(3),c)
        if c ne 0 then pitch=idx(0) else pitch=0

        pt_size=fix(r(4))

        idx=where(fnames eq font,c)
        if c eq 0 then fname_idx=0 else fname_idx=idx(0)
        default_font=fnames(fname_idx)


        ;--- create the base-window -------------------------------------------
        font_base=papco_WIDGET_BASE(title='Select Widget Default Font',$
                                    column=1, GROUP_LEADER= widgetData.base)

        font_list=WIDGET_LIST(font_base, xsize=40, ysize= 10, value= fnames, $
                              uvalue='lst_font')
        WIDGET_CONTROL,font_list, SET_LIST_SELECT=fname_idx

        weight_base = CW_BGROUP(font_base, weight_names, $
                                /FRAME, ROW=1, /EXCLUSIVE, $
                                LABEL_TOP='Select Font Weight',$
                                UVALUE='spb_weight', SET_VALUE=weight)

        quality_base = CW_BGROUP(font_base, quality_names, $
                               /FRAME, ROW=1, /EXCLUSIVE, $
                               LABEL_TOP='Select Font quality',$
                               UVALUE='spb_quality', SET_VALUE=quality)

        pitch_base = CW_BGROUP(font_base, pitch_names, $
                                /FRAME, ROW=1, /EXCLUSIVE, $
                                LABEL_TOP='Select Font pitch',$
                                UVALUE='spb_pitch', SET_VALUE=pitch)

        ef_size = CW_FIELD(font_base, /INTEGER, /ROW, /FRAME, $
                           TITLE='Font Size (points):',VALUE=pt_size)

        spb_done=WIDGET_BUTTON(font_base, value='Done',uvalue='spb_done')

        fontwidgetData={base:font_base, fnames:fnames,$
                        default_font:default_font, $
                        weight:weight, quality:quality,pitch:pitch,$
                        weight_names:weight_names,$
                        quality_names:quality_names,$
                        pitch_names:pitch_names,$
                        ef_size:ef_size}

        ;--- realize the window -----------------------------------------------
        widget_control, font_base, /REALIZE

        papco_xmanager, 'papco_set_default_widget_font', font_base

        PAPCO_writeDefaultSettings

        r = DIALOG_MESSAGE(['New Font is set:'+ widgetData.default_font, $
                            'To take effect, exit and restart PAPCO'], $
                            /INFORMATION, TITLE='New Font')

    END
    ELSE  : message,'PAPCO does not support '+ strupcase(!version.os_family)
ENDCASE

END

