;******************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_get_remote_data.pro
;*
;*    This file belongs to the papco-project and provides a papco
;*    standard utility for papco modules.
;*
;* DESCRIPTION:
;*    Contained are procedures accessing and downloading data files
;*    via ftp using the UNIX GNU freeware "wget".
;*    It is up to the module writer to use and access the routines contained
;*    here. The idea is that if a data file cannot be found on the
;*    user's local system, then these routines are used to "get" data from
;*    a remote site.
;*
;*    Those procedures are contained:
;*
;*      FUNCTION papco_get_remote_host_name
;*	     -> extracts user and password from host name string
;*
;*      PRO papco_get_remote_data
;*	     -> downloads data from a remote system using wget
;*
;*      FUNCTION process_active
;*	     -> checks if a specific process is running
;*
;*      FUNCTION download_status_get_FileName
;*	     -> get name of file in transfer by parsing the wget logfile.
;*
;*      FUNCTION download_status_get_FileLength
;*	     -> get length of file in transfer by parsing the wget logfile.
;*
;*      FUNCTION download_status_get_BytesRead
;*	     -> get bytes transferred of file by parsing the wget logfile.
;*
;*      PRO limit_download
;*	     -> limits the number of wget prosesses that can run at a time
;*
;*      FUNCTION download_status
;*	     -> displays the download status by parsing the wget logfile.
;*
;*      PRO papco_get_remote_data_event
;*	     -> Event handler for papco_get_remote_data
;*
;*      PRO papco_get_remote_file_list_event
;*	     -> Event handler for papco_get_remote_file_list
;*
;*      PRO papco_get_remote_file_list
;*	     -> download the file list of a remote  directory using wget
;*
;*      PRO papco_get_remote_account_info
;*	     -> checks the url for user&pwd  and gets these interactively
;*
;*      PRO papco_get_remote_account_info_event
;*           -> Event handler for papco_get_remote_account_info
;*
;* MODIFICATION HISTORY:
;*     written september 1997, Torsten Heise
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION process_active
;*
;* DESCRIPTION:
;*      checks, if a special process is running
;*
;* INPUTS:
;*      c_command   string which contains the execution command of the
;*                  process
;*      c_url       string contains the url currently being accessed.
;*                  this is needed to esablish the uniqueness of the
;*                  get command being executed.
;*
;* OUTPUT: Function returns:
;*      1   if process containing c_command and c_url exist
;*      0   otherwise
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written September 1997, Torsten Heise
;******************************************************************************
FUNCTION process_active, c_command, c_url, pid

if !version.os_family eq 'Windows' then begin
   return, 1
endif

command = 'wget'

str = 'ps -e | grep '+command+' | grep -v grep'

spawn, str, result

count=0
for i=0, n_elements(result)-1 do BEGIN
    pos=strpos(result(i),command)
    if pos ne -1 THEN BEGIN
        s= str_sep( result[i], ' ' )
        r= where( s ne '' )
        s= s[r]

        thisid= long(s[0])

        if thisid eq pid then count=count+1
;        count= count+1

    ENDIF
ENDFOR

IF count EQ 0 THEN return,0 ELSE return, 1

END


;******************************************************************************
;* PROCEDURE:
;*      FUNCTION papco_get_remote_host_name
;*
;* DESCRIPTION:
;*      this function returns from a given hostname-string the part of
;*      the string containing only the hostname without username and password
;*
;* INPUTS:
;*      c_host - name of the host
;*           ( eg ftp://username:password@leadbelly.lanl.gov/~\lanl_geo/ )
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written september 1997, Torsten Heise
;******************************************************************************
FUNCTION papco_get_remote_host_name, c_host

  host=c_host
  pos=STRPOS(host,'@')
  IF pos GE 0 THEN result=STRMID(host,pos+1,STRLEN(host)-pos) $
  ELSE result = host

  RETURN, result

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION download_status_get_FileName
;*
;* DESCRIPTION:
;*      Get the name of the file currently transfered by analyzing the
;*      wget logfile.
;*
;* INPUTS:
;*      log_file            name of log file
;*      line_offset         start at this line with analyzing the wget logfile.
;*      new_line_offset     inspected the wget logfile until this
;*                          position, set bye this function
;* OUTPUT:
;*      new_line_offset
;*      function returns the name of the currently transfered file
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written july 1997,Haje Korth
;*     rewritten september 1997, Torsten Heise
;******************************************************************************
FUNCTION download_status_get_FileName, log_file, line_offset, new_line_offset

COMMON get_error, get_err_no, get_err_msg

  get_err_no = 1
  get_err_msg = '% download_status_get_FileName: Can`t read logfile'
  new_line_offset=-1
  File=''

  dummystr=''
  FileId='           => '

  tries= 5
again:
  OPENR, unit, log_file, /GET_LUN, ERROR=err
  IF err NE 0 and tries gt 0 THEN begin
      wait, 0.2
      tries= tries-1
      GOTO, again
  endif
  
; skip the first lines
  FOR i=1, line_offset DO begin
    IF NOT EOF(unit) THEN begin
      READF, unit, dummystr
    endIF ELSE begin
      GOTO, finito
    endELSE
  endFOR

; analysing the file
  WHILE NOT EOF(unit) DO begin
    i=i+1
    READF, unit, dummystr
    id=STRMID(dummystr,0,STRLEN(FileId))
;   get the FileName
    IF id EQ FileId THEN begin
      File=STRTRIM(dummystr,1)
      File=STR_SEP(File,' ')
      File=STRTRIM(File(1),2)
      File=STRMID(File,1,STRLEN(File)-2)
      new_line_offset=i
      get_err_no = 0
      get_err_msg = '% download_statut_get_FileName: File found: ' + File
      GOTO, finito
    endIF
  endWHILE

finito:
  CLOSE, unit
  FREE_LUN, unit
  RETURN, File

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION download_status_get_FileLength
;*
;* DESCRIPTION:
;*      Get the length of the file currently transfered by analyzing
;*      the wget logfile.
;*
;* INPUTS:
;*      log_file            name of log file
;*      line_offset         start at this line with analyzing the wget logfile.
;*      new_line_offset     inspected the wget logfile until this
;*
;* OUTPUT:
;*      new_line_offset
;*      function returns the length of the currently transfered file
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*     rewritten september 1997, Torsten Heise
;******************************************************************************
FUNCTION download_status_get_FileLength, log_file, line_offset, new_line_offset

COMMON get_error, get_err_no, get_err_msg

  get_err_no = 1
  get_err_msg = '% download_statut_get_FileLength: Can`t read logfile'
  new_line_offset=-1
  FileLength=''

  dummystr=''
  FileLengthId='Length: '

again:
  OPENR, unit, log_file, /GET_LUN, ERROR=err
  IF err NE 0 THEN GOTO, again

; skip the first lines
  FOR i=1, line_offset DO begin
    IF NOT EOF(unit) THEN begin
      READF, unit, dummystr
    endIF ELSE begin
      GOTO, finito
    endELSE
  endFOR

; analysing the file
  WHILE NOT EOF(unit) DO begin
    i=i+1
    READF, unit, dummystr
    id=STRMID(dummystr,0,STRLEN(FileLengthId))
;   get the FileLength
    IF id EQ FileLengthId THEN begin
      FileLength=STRTRIM(dummystr,1)
      FileLength=STR_SEP(FileLength,' ')
      FileLength=STRTRIM(FileLength(1),2)
      Length=''
      FOR i=0,STRLEN(FileLength)-1 DO begin
        char=STRMID(FileLength,i,1)
        IF char NE ',' THEN Length=Length+char
      endFOR
      FileLength=Length
      FileLength=FLOAT(FileLength)
      new_line_offset=i
      get_err_no = 0
      get_err_msg = '% download_statut_get_FileLength: FileLength found: '+ $
        STRING(FileLength)
      GOTO, finito
    endIF
  endWHILE

finito:
  CLOSE, unit
  FREE_LUN, unit
  RETURN, FileLength

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION download_status_get_BytesRead
;*
;* DESCRIPTION:
;*      Get the number of read bytes of the file currently transfered
;*      by analyzing the wget logfile.
;*
;* INPUTS:
;*      log_file            name of log file
;*      line_offset         start at this line with analyzing the wget logfile.
;*      new_line_offset     set bye this function
;*      FileRead            set bye this function
;* OUTPUT:
;*      new_line_offset     inspected the wget logfile until this
;*                          position
;*      FileRead            1 if the file read completly
;*      function returns the name of the currently transfered file
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written july 1997, Haje Korth
;*     rewritten september 1997, Torsten Heise
;******************************************************************************
FUNCTION download_status_get_BytesRead, log_file, line_offset, $
                                        new_line_offset, FileRead

COMMON get_error, get_err_no, get_err_msg

  get_err_no = 0
  get_err_msg = '% get_BytesRead: o.k.'
  new_line_offset=-1
  Bytes=''
  BytesRead=0.0

  dummystr=''

  BytesReadId='.....'

again:
  OPENR, unit, log_file, /GET_LUN, ERROR=err
  IF err NE 0 THEN GOTO, again

; skip the first lines
  FOR i=1, line_offset DO begin
    IF NOT EOF(unit) THEN begin
      READF, unit, dummystr
    endIF ELSE begin
      get_err_no = 1
      get_err_msg = '% get_BytesRead: Can`t read logfile'
      GOTO, finito
    endELSE
  endFOR

; analysing the file
  WHILE NOT EOF(unit) DO begin
    i=i+1
    READF, unit, dummystr
    idx= strpos( dummystr, BytesReadId )
;   compute the bytes read
    IF idx ne -1 THEN begin
      Bytes = STRTRIM(dummystr,1)
      Bytes = STR_SEP(Bytes,' ')
      FOR j=1, N_ELEMENTS(Bytes)-1 DO begin
          if ( strmid( bytes[j],0,3 ) eq '...' ) then $
            BytesRead = BytesRead + 1024 * STRLEN(Bytes(j))
      endFOR
    endIF ELSE begin
      IF STRPOS(dummystr, "saved") GT -1 THEN begin
        new_line_offset=i
        FileRead = 1
        GOTO, finito
      endIF
    endELSE
  endWHILE

finito:
  CLOSE, unit
  FREE_LUN, unit
  RETURN, FLOAT(BytesRead)

END

;******************************************************************************
;* PROCEDURE:
;*      PRO limit_download
;*
;* DESCRIPTION:
;*      Checks if maximum number of allowed downloads specified in
;*      the total_allowed variabled is exceeded.
;*
;* INPUTS:
;*      none
;* OUTPUTS:
;*      allowed     set if download is possible.
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
PRO limit_download, allowed

  total_allowed=1

  command = 'wget'
  str = "ps -e  | grep '" + command + "' | grep -v grep"

  SPAWN, str, wget_processes

  wget_processes=N_ELEMENTS(wget_processes)
  IF wget_processes LT total_allowed THEN allowed=1 ELSE allowed=0

  allowed=1

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION download_status
;*
;* DESCRIPTION:
;*      Displays the download status by analyzing the wget logfile.
;*
;* INPUTS:
;*      c_tmp_dir          where to put the files
;*      c_log_file         name of log file
;*      c_command          string with full wget command
;*
;* OUTPUT:
;*      function returns number of saved files
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*     modified september 1997 Torsten Heise
;*         diveded routine in four functions:
;*             download_status,
;*             download_status_get_FileName,
;*             download_status_get_FileLength and
;*             download_status_get_BytesRead
;*         this was necessary to get more than one file at the
;*         same time ( e.g. with the wget command: wget hydra.mpae.gwdg.de/*.*)
;******************************************************************************
FUNCTION download_status, c_url, c_tmp_dir, c_logfile, c_command, pid, cancel=cancel

COMMON get_error, get_err_no, get_err_msg

; set error defaults
get_err_no = 0 & get_err_msg = '% download_status: o.k.'

; make copy of arguments
url = c_url
tmp_dir = c_tmp_dir
logfile = c_logfile
command = c_command

; see if process is still active - on local connections there might be
;                                  no time to report any status!
if papco_process_active(pid) then begin
    ;check if log file exisits!
    result=findfile(tmp_dir+logfile,count=c)
    if c eq 1 then begin
      SavedFiles = 1
      goto, finito
    endif
endif

; initialisation of variables
urlString=papco_get_remote_host_name(url)
line_offset=0
File=''
FileLength=0
BytesRead=0
maxwidth=80
starttime=SYSTIME(1)
new_line_offset1 = 0
new_line_offset2 = 0
new_line_offset3 = 0
FileRead = 0
percent = 0
percentStr = ''
SavedFiles= 0
TotalBytes = 0.0

infolen=INTARR(5)
infotext=STRARR(5)
infotext(0)='Copying    ' + File
infotext(1)='from       ' + urlString
infotext(2)='to         ' + tmp_dir
infotext(3)='Bytes read:   0% of ' + STRING(FileLength) + ' Bytes '
infotext(4)=''
FOR i=0,N_ELEMENTS(infotext)-1 DO infolen(i)=STRLEN(infotext(i))
FOR i=0,N_ELEMENTS(infotext)-1 DO BEGIN
    space=''
    FOR j=0,maxwidth-infolen(i) Do space=space+' '
    infotext(i)=infotext(i)+space
ENDFOR

; create widget
base=WIDGET_BASE(/COLUMN, TITLE='Download Status')
info=WIDGET_TEXT(base, VALUE=infotext, XSIZE=maxwidth, $
                 YSIZE=N_ELEMENTS(infotext))

progress_base= widget_base( base, /row )
id_draw= widget_draw( progress_base, xsize=300, ysize=8 )
x= widget_label(progress_base, value=' ')
id_cancel= widget_button( progress_base, value='Cancel' )


geo=WIDGET_INFO(base, /GEO)
x=600-geo.xsize/2 & y=300-geo.ysize/2

oldw= !d.window

;sm note: under Mac OS X, if you specify an offset before a pop-up widget
;is realized, the buttons on it don't generate events.
WIDGET_CONTROL, base, /REALIZE, MAP=0              ;sm
WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y, MAP=1  ;sm

wset, oldw

monitor= obj_new('papco_graph_monitor', idDraw=id_draw, idCancel=id_cancel )

cancel=0
notdone= 1
; update widget information
WHILE notdone DO begin
    wait, 0.5                   ;don`t update display too fast
    ;get Information for the status-widget

    if monitor->isCancelled() then begin
       cancel=1
       papco_process_kill, pid
       break;
    endif

    File       = download_status_get_FileName(logfile, line_offset, $
                                              new_line_offset1)
    FileLength = download_status_get_FileLength(logfile, line_offset, $
                                                new_line_offset2)
    BytesRead  = download_status_get_BytesRead(logfile, line_offset, $
                                               new_line_offset3, FileRead )

    monitor->setTaskSize, FileLength
    monitor->setProgress, BytesRead

    IF get_err_no EQ 0 THEN begin
        IF FileLength GT 0.0 THEN percent = BytesRead/FileLength*100 $
        ELSE percent = 0.0
        percent = percent <  100
        percentStr = STRTRIM(STRING(FIX(percent)),2)
        infotext(0)   = 'Copying    ' + File
        infotext(1)   = 'from       ' + urlString
        infotext(2)   = 'to         ' + tmp_dir
        ; if a wildcard like * is used to get files, first a file named
        ; .listing is created, containing information about the remote_dir
        ; for that reason you must distinguish between two cases:
        ; a) you get the .listing file
        ; b) you get another file
        IF STRPOS(File, '.listing') NE -1 THEN begin
            infotext(3) = 'Bytes read: '+STRING(BytesRead) + ' of ' + File
            infotext(4) = ''
        ENDIF  ELSE BEGIN
            infotext(3) = 'Bytes read: '+percentStr + '% of ' + $
              STRING(FileLength) + ' Bytes '
            IF percent NE 0.0 THEN begin
                elapsedtime=SYSTIME(1)-starttime
                rate=BytesRead/elapsedtime
                minutes=( (FileLength-BytesRead)/(rate) ) / 60
                seconds=((minutes-fix(minutes) ) * 60) >  0.0
                seconds=STRTRIM(STRING(fix(seconds),format="(i2.2)"),2)
                minutes=STRTRIM(STRING(fix(minutes)),2)
                infotext(4)='Estimated Download Time Left: '+ $
                  minutes+':'+seconds+' min.  ' + $
                  'rate: '+ strtrim( string(BytesRead/1024/elapsedtime, $
                                            format='(f9.2)'), 2 )+'kB/s'
            ENDIF
        ENDELSE
        FOR i=0,N_ELEMENTS(infotext)-1 DO infolen(i)=STRLEN(infotext(i))
        maxwidth=MAX(infolen)
        FOR i=0,N_ELEMENTS(infotext)-1 DO begin
            space=''
            FOR j=0,maxwidth-infolen(i) Do space=space+' '
            infotext(i)=infotext(i)+space
        ENDFOR
        WIDGET_CONTROL, info, SET_VALUE=infotext
    endIF ELSE BEGIN
        message, get_err_msg, /cont
        get_err_no = 0
        get_err_msg = '% download_status: o.k.'
    ENDELSE
    ; if a file is saved, show information about the next file
    IF FileRead THEN BEGIN
        message, file +' saved',  /cont
        TotalBytes = TotalBytes + BytesRead
        FileRead = 0

        line_offset = new_line_offset3+1
        SavedFiles = SavedFiles+1
        File = '' & FileLength = 0.0 & BytesRead = 0.0
    ENDIF

    notdone= papco_process_active(pid)

ENDWHILE

;;;check if log file exists!
;;result=findfile(tmp_dir+logfile,count=c)
;;if c eq 1 then SavedFiles = SavedFiles+1

WIDGET_CONTROL, base, /DESTROY

finito:

RETURN, SavedFiles

END


;******************************************************************************
;* PROCEDURE:
;*      PRO papco_get_remote_file_list_event
;*
;* DESCRIPTION:
;*      Event handler for papco_get_remote_file_list
;*
;* INPUTS:
;*      event    Event
;*
;* KEYWORDS:
;* 	BIN - if this keyword is set write binary file else write text
;             file (only use bin if all files have same format )
;*
;* MODIFICATION HISTORY:
;*     september 1997, Torsten Heise
;******************************************************************************
PRO papco_get_remote_file_list_event, event

  COMMON remote_file_list, remoteWidgetData

  WIDGET_CONTROL, event.id, GET_UVAL=uval

  CASE uval OF

    'pb_cancel': begin
      WIDGET_CONTROL, event.top, /DESTROY
    end

    'pb_ok': begin
      url=remoteWidgetData.url
      file=remoteWidgetData.file
      dir=remoteWidgetData.dir

      IF strmid(url,strlen(url)-1,1) NE '/' THEN url=url+'/'
      IF strmid(dir,strlen(dir)-1,1) NE '/' THEN dir=dir+'/'
      env_var='PAPCO_DOWNLOAD'
      tmp_dir=papco_getenv(env_var)
      IF strlen(tmp_dir) EQ 0 THEN begin
        get_err_no=1
        get_err_msg='Environmental variable '+env_var+' not set'
        message, get_err_msg, /cont
        WIDGET_CONTROL, event.top, /DESTROY
        RETURN
      endif
      IF strmid(tmp_dir,strlen(tmp_dir)-1,1) NE '/' THEN tmp_dir=tmp_dir+'/'
      wget_exe='wget'

;     check if you can start another wget process
      limit_download, allowed
      IF NOT allowed THEN begin
        WIDGET_CONTROL, event.top, /DESTROY
        info=WIDGET_MESSAGE(['Maximum number of allowed', $
                             'download processes exceeded.'])
        RETURN
      endIF

;     set some variables and start wget. To get a list of files only,
;     call wget with the directory only (don't specify a file). wget
;     downloads a file index.html, containing the directory list.

      SPAWN, "pwd", pwd
      CD, tmp_dir
      result=FINDFILE("index.html")
      IF result(0) NE '' THEN papco_file_delete, "index.html", /QUIET
      logfile="index_html.log"
      exestr="xterm -iconic -e "+wget_exe+" "+url + ' -o ' + logfile + ' &'
      command = wget_exe + ' ' + url

      SPAWN, exestr, result

;     show status of current download process
      SavedFiles = download_status(url, tmp_dir, logfile, command)
      papco_file_delete, logfile, /QUIET

;     the file index.html is downloaded
      IF (SavedFiles GT 0)  THEN begin
        message, 'index.html downloaded', /cont

        SPAWN, "cat index.html | grep 'File'",files
        papco_file_delete, "index.html", /QUIET

;       get filenames from index.html (index.html is a html document)
        FOR i=0, N_ELEMENTS(files)-1 DO begin
          dummy=STR_SEP(files(i),'>')
          dummy=dummy(1)
          dummy=STRMID(dummy,0,STRLEN(dummy)-3)
          files(i)=dummy
        endFOR

;       write a file containing the filenames from the remote host
        message, 'writing file: '+file, /cont
        PRINT, '  in dir: ',dir
        file=dir+file
        OPENW, unit, file, /GET_LUN, ERROR=err
        IF err NE 0 THEN begin
          get_err_nr=err
          get_err_msg=!ERR_STRING
          print,'% papco_get_remote_file_list: ',get_err_msg
          WIDGET_CONTROL, event.top, /DESTROY
          RETURN
        endIF
        FOR i=0, N_ELEMENTS(files)-1 DO begin
          PRINTF, unit, files(i)
        endFOR
        CLOSE, unit
        FREE_LUN, unit

        SPAWN, "chmod g+rw "+file

      endIF ELSE begin
        info=WIDGET_MESSAGE(["% papco_get_remote_file_list: can't get file list."])
      endELSE

      CD,pwd(0)

      WIDGET_CONTROL, event.top, /DESTROY

    end
    ELSE: PRINT, '%papco_get_remote_file_list: Event not handled.'
  endCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_get_remote_file_list
;*
;* DESCRIPTION:
;*      this procedure download the file list of the directory given
;*      by 'url' using gnu wget
;*
;* INPUTS:
;*      c_url  - the url of a remote host including username, password,
;*               hostname, path, (eg
;*               user:passwd@leadbelly.lanl.gov/~\lanl_geo/)
;*      c_dir  - where the file list shall be saved
;*      c_file - name of the file list
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     september 1997, Torsten Heise
;******************************************************************************
PRO papco_get_remote_file_list, url, dir, file

COMMON remote_file_list, remoteWidgetData
COMMON get_error, get_err_no, get_err_msg

remoteWidgetData={url:url, file:file, dir:dir}

base=WIDGET_BASE(/COLUMN, TITLE='get file list')
txt_msg=WIDGET_TEXT(base, VALUE=strleft('Get new list via FTP from ',50) + $
                    url, XSIZE=50, YSIZE=3, /WRAP)
pb_base=WIDGET_BASE(base, /ROW)
pb_ok=WIDGET_BUTTON(pb_base, VALUE='     OK     ', UVALUE='pb_ok')
pb_cancel=WIDGET_BUTTON(pb_base, VALUE='   Cancel   ', UVALUE='pb_cancel')

geo=WIDGET_INFO(base, /GEO)
x=600-geo.xsize/2 & y=300-geo.ysize/2

;sm note: under Mac OS X, if you specify an offset before a pop-up widget
;is realized, the buttons on it don't generate events.
WIDGET_CONTROL, base, /REALIZE, MAP=0              ;sm
WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y, MAP=1  ;sm

XMANAGER, 'papco_get_remote_file_list', base, /MODAL

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_get_remote_account_info
;*
;* DESCRIPTION:
;*      checks the url for "user:password" and gets these interactively.
;*
;* INPUTS:
;*      url   the WGET url string
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     October 1997, Reiner Friedel
;******************************************************************************
PRO papco_get_remote_account_info, url

COMMON remote_account_info, account_info_WidgetData

;check the url for "pasword" and "user", if there then  get interactively!
res=strpos(url,'user')
frontpart=strmid(url,0,res)
IF res NE -1 THEN BEGIN
    base=WIDGET_BASE(/COLUMN, TITLE='Enter login information')
    pos=strpos(url,'@')
    backpart=strmid(url,pos,strlen(url)-1)
    host=strmid(url,pos+1,strlen(url))
    txt_msg=WIDGET_TEXT(base, VALUE='Host: '+host)
    user_fld=CW_field(base,/ROW,/STRING,TITLE='User:   ',UVALUE='user_fld')
    pswd_fld=CW_field(base,/ROW,/STRING,/ALL_EVENTS, $
                     TITLE='Password:', UVALUE='pswd_fld')
    pb_base=WIDGET_BASE(base, /ROW)
    pb_enter= WIDGET_BUTTON(pb_base, VALUE='    Enter    ', $
                            UVALUE='pb_enter')
    pb_cancel=WIDGET_BUTTON(pb_base, VALUE='    Cancel   ', $
                            UVALUE='pb_cancel')
    pb_clear= WIDGET_BUTTON(pb_base, VALUE='    Clear   ', $
                            UVALUE='pb_clear')

    account_info_WidgetData={user:'',$
                             pswd:'',$
                             user_fld:user_fld, $
                             pswd_fld:pswd_fld}

    WIDGET_CONTROL, base, /REALIZE
    XMANAGER, 'papco_get_remote_account_info', base, /MODAL

    url=frontpart + $
      account_info_WidgetData.user+':'+account_info_WidgetData.pswd + $
      backpart
ENDIF ELSE BEGIN
   ;check the url for "@" - if absent, this indicated anoymous fto access
   res=strpos(url,'@')
   IF res EQ -1 THEN message, 'Using anonymous ftp access...', /cont $
     ELSE message,'Have user info in defaults...', /cont
ENDELSE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_get_remote_account_info_event
;*
;* DESCRIPTION:
;*      Event handler for papco_get_remote_account_info
;*
;* INPUTS:
;*      event    Event
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     October 1997, Reiner Friedel
;******************************************************************************
PRO papco_get_remote_account_info_event, event

  COMMON remote_account_info, account_info_WidgetData

  WIDGET_CONTROL, event.id, GET_UVAL=uval

  CASE uval OF

    'pb_cancel': begin
      WIDGET_CONTROL, event.top, /DESTROY
    end

    'pb_enter': begin
      WIDGET_CONTROL,account_info_WidgetData.user_fld,GET_VALUE=getuser
      account_info_WidgetData.user=getuser(0)
      WIDGET_CONTROL, event.top, /DESTROY
    end

    'pb_clear': begin
      account_info_WidgetData.pswd=''
      WIDGET_CONTROL,account_info_WidgetData.pswd_fld,SET_VALUE=''
    end

    'pswd_fld': begin
        if ( event.update ) then begin ;; CR save as Enter button
            WIDGET_CONTROL,account_info_WidgetData.user_fld,GET_VALUE=getuser
            account_info_WidgetData.user=getuser(0)
            WIDGET_CONTROL, event.top, /DESTROY
        endif else begin
            WIDGET_CONTROL,account_info_WidgetData.pswd_fld,GET_VALUE=getpswd
            len=strlen(getpswd(0))
            if ( len lt strlen(account_info_WidgetData.pswd) ) then begin  ;; DELETE
                account_info_WidgetData.pswd= strmid( account_info_WidgetData.pswd,0,len )
                widget_control, account_info_WidgetData.pswd_fld, set_text_select=len
            endif else begin
                newchar=strmid(getpswd(0),0,1)
                account_info_WidgetData.pswd=account_info_WidgetData.pswd+newchar
                hide_user='' & star='*'
                for i=0,strlen(account_info_WidgetData.pswd)-1 do $
                  hide_user=hide_user+star
                WIDGET_CONTROL,account_info_WidgetData.pswd_fld,SET_VALUE=hide_user
                widget_control, account_info_WidgetData.pswd_fld, set_text_select=strlen(hide_user)
            endelse
        endelse
    end

    ELSE: message, 'Event not handled.', /cont

  endCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_get_remote_data
;*
;* DESCRIPTION:
;*      Routine downloads data from a remote system using the WGET
;*      program available from GNU Software.
;*
;* INPUTS:
;*      url            internet address inclusive path, user, passwrd. Syntax:
;*                     user:pwd@host/path     e.g.
;*                     friedel:*****@hydra.mpae.gwdg.de//pub/
;*                     if no user and pwd is given anoymous ftp is used.
;*                     see the wget man pages for further info.
;*      filename       filename of data file, e.g. test.dat
;*      download_dir   download directory
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*	   revised to use file_delete instead of papco_spawn, Nov 2003, J L Roeder
;******************************************************************************
PRO papco_get_remote_data, c_url, c_file, c_download_dir

COMMON remote, remoteWidgetData
COMMON plot_composer, widgetData
COMMON get_error, get_err_no, get_err_msg

get_err_no=0 & get_err_msg='Remote file succesfully transferred'

; check if wget has been enabled by defaults.
if n_elements(widgetData) eq 1 then begin
    if widgetData.default_wget EQ 0 then begin
        get_err_no=2
        get_err_msg='wGet disabled by Defaults'
        message,get_err_msg,/cont
        return
    endif
endif

; check if wGet available on current system.
if getenv('PAPCO_WGET') eq '' then begin
    get_err_no=2
    get_err_msg='wGet not available on your system'
    message,get_err_msg,/cont
    message,'wGet will only work on UNIX, and if you have it installed.',/cont
    return
endif

url = c_url
IF strmid(url,strlen(url)-1,1) NE '/' THEN url=url+'/'
papco_sepdir, c_file, path, file, ext
IF path NE '' THEN BEGIN
    url = url + path
    c_download_dir = c_download_dir+ path
ENDIF

file = file+ext
download_dir = c_download_dir
IF strmid(download_dir,strlen(download_dir)-1,1) NE papco_get_pathsep() THEN $
download_dir=download_dir+papco_get_pathsep()
if not papco_check_data_env('PAPCO_DOWNLOAD', PATH=tmp_dir) then return

IF strmid(tmp_dir,strlen(tmp_dir)-1,1) NE papco_get_pathsep() THEN $
	tmp_dir=tmp_dir+papco_get_pathsep()

;make a query widget for data get if needed.
if ( n_elements(widgetData) eq 1 ) then begin
    IF widgetData.default_ask_wget EQ 0 THEN $
      result = messageBox(['The data file '+ file, $
                           ' you have requested is not available locally on',  $
                           download_dir, $
                           'Do you want to retrieve the data via FTP from ', url], $
                          ['Retrieve', 'Cancel'], $
                          TITLE='No Data File', /CENTER) $      
    ELSE result = 0
endif else result=0

IF result EQ 0 THEN BEGIN       ;get data remotely
    limit_download, allowed
    IF NOT allowed THEN BEGIN
        info=WIDGET_MESSAGE(['Maximum number of allowed', $
                             'download processes exceeded.']) & RETURN
    ENDIF
    message,'Attempting to get data from URL / file', /cont
    print,'  '+ url & print,'  '+ file
    logfile=tmp_dir+'papco_wget.log'

    papco_get_remote_account_info, url

    CD, tmp_dir,  current = cur_dir
    papco_file_delete, /allow_nonexistent, file
    command = 'wget ' + url + file
    papco_wget, url+file, logfile, pid

    wait, 0.1

    SavedFiles = download_status(url, tmp_dir, logfile, command, pid, cancel=cancel )
    papco_FILE_DELETE, logfile, /QUIET

    if ( cancel ) then begin
       get_err_no=6 & get_err_msg='Remote file get cancelled'
       return
    endif

    tries= 2
    FileExist= findfile(file)
    while ( tries ge 0 and fileExist[0] eq '' ) do begin
        message, 'didn''t find it, trying again', /cont
        wait, 0.5 ;; sometimes takes a second to show up--this might be due to samba server
        FileExist=findfile(file)
        tries= tries-1
    endwhile
    IF  (SavedFiles GT 0)  THEN BEGIN
        get_err_no=0
        get_err_msg='Remote GET successful'
        message, 'Move '+file+' from '+tmp_dir+ ' to '+download_dir, /cont
        papco_chmod, 'g+rw', tmp_dir+file
        papco_create_dir,download_dir
        papco_file_move, tmp_dir+file, download_dir
        message, 'File: '+file+  ' downloaded.' ,/cont
    endif else if ( fileexist[0] ne '' and savedfiles gt 0 ) then begin
        get_err_no=4
        get_err_msg='file found but wget unsuccessful, probably a partial file'
        message, get_err_msg, /cont
    ENDIF ELSE BEGIN
        get_err_no=3
        get_err_msg='File: '+file+ ' not found on remote system.'
        message, get_err_msg, /cont
    ENDELSE
    CD, cur_dir
ENDIF

IF result EQ 1 THEN BEGIN  ;cancel
    get_err_no=6 & get_err_msg='Remote file get cancelled'
ENDIF

END
