;******************************************************************************
;* PROCEDURE:
;*      PRO papco_module_Configure
;*
;* DESCRIPTION:
;*      Displays the configuration widget for the environmental
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      Writes configuration file containing environmental variables used
;*      by the module
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*     modified October 1997, Reiner Friedel, to work under VMS
;******************************************************************************
pro papco_module_configure, module

common configure_common, configWidgetData, config_data
common PLOT_COMPOSER, widgetData
common get_error, get_err_no, get_err_msg

; read config info
  r_module_config, strlowcase(module), config_data, msg
  if get_err_no ne 0 then begin
    r_module_default, strlowcase(module), config_data, msg
    if get_err_no ne 0 then return
  endif

  ;load in actually set values for env vartiables
  FOR i = 0, n_elements(config_data)-1 DO $
    config_data(i).PATH = papco_getenv(config_data(i).ENV_VAR,  /NOFS)

  base= papco_widget_base(title=module+' Configure' ,/column, /modal, $
                          group_leader=widgetData.base)
  lbl_envvar=widget_label(base, value='Current Environment Variables')
  lst_envvar= widget_list(base,  value=config_data.env_var+ $
                          ' = '+config_data.path, $
                          uvalue='lst_envvar', $
                          xsize=80, ysize=10, /frame)

  base1= widget_base(base, /row )
  txt_envvar= cw_field( base1, value= '', $
                        title= 'Environment Variable:', $
                        uvalue= 'txt_envvar', $
                        /string, /frame, /column, xsize=80 )
  pb_browse=widget_button(base1, value='Browse',  uvalue='pb_browse')
  pb_set=   widget_button(base1, value=' Set  ',  uvalue='pb_set')

  action_base= widget_base( base, /row, /frame )
  pb_done=   widget_button(action_base, value='Done',    uvalue='pb_done')
  pb_save=   widget_button(action_base, value='Save',    uvalue='pb_save')
  pb_cancel= widget_button(action_base, value='Cancel',  uvalue='pb_cancel')
  pb_revert= widget_button(action_base, value='Revert',  uvalue='pb_revert')
  pb_default=widget_button(action_base, value='Default', uvalue='pb_default')
  pb_help=widget_button(action_base, value='Configure Help', uvalue='pb_help')
  pb_mhelp=widget_button(action_base, value='Module Help', uvalue='pb_mhelp')

  configWidgetData={module_config, $
                    lst_envvar:lst_envvar, txt_envvar:txt_envvar, $
                    pb_browse:pb_browse, pb_set:pb_set, $
                    pb_done:pb_done, pb_save:pb_save, $
                    pb_cancel:pb_cancel, pb_revert:pb_revert, $
                    pb_help:pb_help, pb_mhelp:pb_mhelp, $
                    module:module, index:0}

  geo= widget_info( base, /geo )
  x= 600- geo.xsize/2 & y= 300- geo.ysize/2

  widget_control, configWidgetData.lst_envvar, set_list_select=0
  widget_control, configWidgetData.txt_envvar, set_value=config_data(0).path
  widget_control, base, xoffset=x, yoffset=y
  widget_control, base, /realize

  papco_xmanager, 'papco_module_configure', base
  return
end

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_module_Configure_event
;*
;* DESCRIPTION:
;*      Event handler for papco_module_Configure
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;* 	none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
PRO papco_module_configure_event, event

COMMON configure_common, configWidgetData, config_data
COMMON get_error, get_err_no, get_err_msg

widget_control, event.id, get_uval= uval

CASE uval OF
    'lst_envvar': begin
      configWidgetData.index=event.index
      val=config_data(configWidgetData.index).path
      widget_control, configWidgetData.txt_envvar, set_value=val
    end
    'pb_browse': BEGIN
        widget_control, configWidgetData.txt_envvar, get_value=data
        used_dlnk = 0

        ;data may contain env var $PAPCO_DLNK. Expand.
        test = strmid(data(0), 0, 11)
        IF test EQ '$PAPCO_DLNK' THEN BEGIN
            used_dlnk = 1
            part_path = strmid(data(0), 12, strlen(data(0))-12)
            dlnk_path =  papco_getenv('PAPCO_DLNK')
            IF dlnk_path NE '' THEN data(0) = dlnk_path+part_path ELSE BEGIN
                get_err_no=3
                get_err_msg='Environmental variable PAPCO_DLNK not set'
                message, get_err_msg, /cont
                return
            ENDELSE
        ENDIF

        IF NOT papco_finddir(data(0)) THEN BEGIN
        dummy=messageBox('Directory '+data(0)+' not found', ['OK'], $
                          /CENTER, TITLE='Error')
        return
        endif

        r= pickfile( path=data(0), get_path=new )

        IF used_dlnk THEN BEGIN
            test = strmid(new, 0, strlen(dlnk_path))
            IF test EQ dlnk_path THEN BEGIN
                part_path = $
                  strmid(new, strlen(dlnk_path), strlen(new)-strlen(dlnk_path))
                new = '$PAPCO_DLNK'+ papco_get_dirsep()+part_path		; JLR 11/24/03
            ENDIF
        ENDIF

        widget_control, configWidgetData.txt_envvar, set_value=new
    end
    'pb_set': begin
      widget_control, configWidgetData.txt_envvar, get_value=path
      path=path(0)
      config_data(configWidgetData.index).path=path
      widget_control, ConfigWidgetData.lst_envvar, $
        set_value=config_data.env_var+' = '+config_data.path
    end
    'pb_done': begin
      for i=0,n_elements(config_data)-1 do $
        papco_setenv,config_data(i).env_var+'='+config_data(i).path
      widget_control, event.top, /destroy
    end
    'pb_save': begin
      s_module_config, configWidgetData.module, config_data
      for i=0,n_elements(config_data)-1 do $
        papco_setenv,config_data(i).env_var+'='+config_data(i).path
      widget_control, event.top, /destroy
    end
    'pb_cancel': begin
      widget_control, event.top, /destroy
    end
    'pb_revert': begin
      r_module_config, configWidgetData.module, config_data, msg
      if get_err_no ne 0 then begin
        message,'No user defaults file', /cont
        return
      endif
      widget_control, configWidgetData.lst_envvar, $
        set_value=config_data.env_var+' = '+config_data.path
    end
    'pb_default': begin
      r_module_default, configWidgetData.module, config_data, msg
      widget_control, configWidgetData.lst_envvar, $
        set_value=config_data.env_var+' = '+config_data.path
    end
    'pb_help': begin
      path=papco_getenv('PAPCO_PATH')
      fln=path+'/help/papco_module_configure.help'
      xdisplayfile, fln, title='Configure Help',group=event.top, /MODAL
    end
    'pb_mhelp': BEGIN
         name = configWidgetData.module
         done = 'Click to return to '+name+' config widget'
         xdisplayfile, papco_module_helpfile(name+'_panel.help'), $
                       title=name+'_panel.help'+' - Online Help',$
                       group=event.top, height = 48, /modal, $
                       DONE_BUTTON = done
    END
    else: message, 'module_configure: internal error.',/cont

ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO r_module_config, module, config_data, msg
;*
;* DESCRIPTION:
;*      Reads configuration file
;*
;* INPUTS:
;*      module   module name
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      config_data   configuration data for the module
;*      msg           status message
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*     check of path separtors, revised Nov 2003, Jim Roeder
;******************************************************************************
PRO r_module_config, module, config_data, msg

COMMON get_error, get_err_no, get_err_msg

if not (papco_check_data_env('PAPCO_HOME', path=PATH)) then return

fname = papco_addpath(path, 'papco_modules_defaults')
fname = fname + strlowcase(module) + '.config'

openr, unit, fname, /get_lun, ERROR=err
IF err NE 0 THEN BEGIN
    get_err_no=2 & get_err_msg=!ERR_STRING & RETURN
ENDIF ELSE get_err_no=0

dummy_str='' & dummy_struct={CONFIG, env_var:'', path:''} & i = 0

WHILE NOT eof(unit) DO BEGIN
    readf,unit,dummy_str
    dummy_str=strtrim(dummy_str,2)
    IF dummy_str NE '' THEN BEGIN
        dummy_arr=str_sep(dummy_str,' ')
        IF  n_elements(dummy_arr) LT 2 THEN dummy_arr=[dummy_arr,'']
        reads, dummy_arr, dummy_struct
        IF i EQ 0 THEN config_data = [dummy_struct] ELSE $
                       config_data = [config_data, dummy_struct]
        i = i+1
    ENDIF
ENDWHILE

;	Assure that directory separators conform to operating system
nstrucs = N_ELEMENTS( config_data)
IF nstrucs gt 0L THEN BEGIN
	FOR istruc = 0L, nstrucs - 1L DO BEGIN

		path = config_data( istruc).path
		if strpos( strlowcase( path), 'ftp') eq -1L then $
			papco_opsys_path, path
		config_data( istruc).path = path

	ENDFOR
ENDIF

close, unit & free_lun, unit

msg = module+' settings have been read.'

END

;******************************************************************************
;* PROCEDURE:
;*      PRO r_module_default, fln, config_data, msg
;*
;* DESCRIPTION:
;*      Read default configuration from module path.
;*
;* INPUTS:
;*      fln      file name for default file
;*      module   module name
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      config_data   default configuration data for the module
;*      msg           status message
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;*     check of path separtors, revised Nov 2003, Jim Roeder
;******************************************************************************
PRO r_module_default, module, config_data, msg

COMMON MODULE_PATHS, paths    ;contains the paths found by find_modules
COMMON get_error, get_err_no, get_err_msg

result=strpos(strlowcase(paths),strlowcase(module))
index=where(result ne -1)
path=paths(index(0))

fname=papco_addpath(path,'plot_read')
fname=fname+'defaults.config'
dummy_str=''
dummy_struct={CONFIG, env_var:'', path:''}
config_data=replicate(dummy_struct,50)
i=-1

openr, unit, fname, /get_lun, ERROR=err
IF err NE 0 THEN BEGIN
    msg=path+' defaults not available. Environment variables set?'
    get_err_no=2
    get_err_msg=!ERR_STRING
    RETURN
ENDIF ELSE get_err_no=0

while not eof(unit) do begin
    readf,unit,dummy_str
    dummy_str=strtrim(dummy_str,2)
    if dummy_str ne '' then begin
      i=i+1
      dummy_arr=str_sep(dummy_str,' ')
      reads,dummy_arr,dummy_struct
      config_data(i)=dummy_struct
    endif
ENDWHILE

close, unit & free_lun, unit

config_data=temporary(config_data(0:i))

;	Assure that directory separators conform to operating system
nstrucs = N_ELEMENTS( config_data)
IF nstrucs gt 0L THEN BEGIN
	FOR istruc = 0L, nstrucs - 1L DO BEGIN

		path = config_data( istruc).path
		if strpos( strlowcase( path), 'ftp') eq -1L then $
			papco_opsys_path, path
		config_data( istruc).path = path

	ENDFOR
ENDIF

msg=module+' defaults have been read.'

end

;******************************************************************************
;* PROCEDURE:
;*      PRO s_module_config, module, config_data
;*
;* DESCRIPTION:
;*      Writes configuration file.
;*
;* INPUTS:
;*      module        module name
;*      config_data   configuration data to write
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      none
;*
;* MODIFICATION HISTORY:
;*     written July 1997, Haje Korth
;******************************************************************************
PRO s_module_config,module,config_data

COMMON get_error, get_err_no, get_err_msg

  path=papco_getenv('PAPCO_HOME')
  if strlen(path) eq 0 then begin
    get_err_no=1
    get_err_msg='Environmental variable '+data_dir+' not set'
    print,get_err_msg
    return
  endif

  fname=papco_addpath(path,'papco_modules_defaults')
  fname=fname+strlowcase(module)+'.config'
  message,'Saving: ', /cont
  print,'  ',fname

  num=n_elements(config_data)

; write the config file to the user's papco_modules_defaults
; directory. Check if that is possible first!
  papco_file_status, fname, write=canwrite
  if canwrite then begin
    openw, unit, fname, /get_lun, ERROR=err
    IF err NE 0 THEN BEGIN
      message,'Error writing '+ fname, /cont
      get_err_no=2
      get_err_msg=!ERR_STRING
      RETURN
    ENDIF
  endif else begin
    message,"can't write "+ fname, /cont
    RETURN
  endelse

  for i=0,num-1 do printf,unit,config_data(i).env_var+' '+config_data(i).path

  close, unit
  free_lun, unit

END


