;*****************************************************************************
;******************************************************************************
;* FILE:
;*    $papco_PATH/papco_opsys.pro
;*
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:
;*    implements operating-system dependant tasks. Mainly file I/O,
;*    operating system spawns, formatting of directory/filenames.
;*    The idea here is that in papco the operating system dependency
;*    becomes transparent, all such dependency is resolved here.
;*
;*    Those procedures are contained:
;*
;*       FUNCTION papco_lastdir, path
;*       -> finds the last directory of a path
;*
;*       FUNCTION papco_checkdirname, dir
;*       -> checks if dir path has vaild ending
;*
;*       FUNCTION papco_getenv
;*       -> get a pathname from an environent variable
;*
;*       FUNCTION papco_hasdir, fname
;*       -> checks if fname contains a directory string for
;*               the given operating system.
;*
;*       PRO papco_sepdir, fname, path, file
;*       -> seperates a filename string into directory and filename
;*
;*       FUNCTION papco_addpath, path, new
;*       -> adds a path to an exisiting path. New is in UNIX format.
;*
;*       FUNCTION papco_fname, fname
;*       -> formats the filename or directory string correctly for
;*               the given operating system. For VMS, removes version #
;*
;*       FUNCTION papco_setenv
;*       -> sets an environmental variable. Expects the input
;*          string to be in the UNIX format: 'ENV_VAR_NAME=env_var_value'
;*
;*       PRO papco_spawn, cmdline, result
;*       -> executes operating system commands. Only works for the
;*          list of commands "known" here. The cmdline passed in is
;*          parsed for a set of "known" commands, and the correct
;*          command for the os is inserted. The arguments are NOT checked.
;*
;*       FUNCTION  papco_cleanup
;*       -> Called on exiting the PAPCO main window. Used to delete
;*               temporary files etc
;*
;*       FUNCTION papco_add_module_path, path_name
;*       -> returns the entry needed in the papco_include_modules.pro
;*               file to extend the idl search path by the path_name
;*
;*       FUNCTION papco_get_dirlist, path_name
;*       -> returns list of directories found in path_name
;*
;*       FUNCTION papco_get_dirsep
;*       -> returns the character used in !PATH to seperate directories.
;*
;*       FUNCTION papco_set_plot
;*       -> sets the plot device to the appropriate device for
;*               the operating system.
;*
;*       FUNCTION papco_get_module_names
;*       -> given a list of mosule dirs, returns the module names.
;*
;*       FUNCTION papco_check_window_device
;*       -> checks if current device is set to a window device
;*
;*
;* MODIFICATION HISTORY:
;*    October 1997, written by R. Friedel
;******************************************************************************
;******************************************************************************


;******************************************************************************
;* function:     papco_lastdir, path
;*
;* description:  finds the last directory of a path
;*
;* inputs:       path: the name of the dir to be checked
;*
;* output:       last dir name
;*
;* author:       R. Friedel, February 1998
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_lastdir, path

CASE strupcase(!version.os_family) OF
    'VMS' : BEGIN
         pos = rstrpos(path,']')
         po2 = rstrpos(path,'.')
         return,strmid(path, po2+1,pos-po2-1)
    END
    'UNIX': BEGIN
         new=strtrim(path,2)
         dirs=str_sep(path,'/')
         return,dirs(n_elements(dirs)-2)
    END
    'WINDOWS': BEGIN
         pos = rstrpos(path,'\') ;'
         return,strmid(path,pos+1,strlen(path))
    END

    ELSE: message,'does not support '+ strupcase(!version.os_family)

ENDCASE

END

;******************************************************************************
;* function:     papco_checkdirname, dir
;*
;* description:  checks if dir path has vaild ending
;*
;* inputs:       dir : the name of the dir to be checked
;*
;* output:       valid dir
;*
;* author:       R. Friedel, November 1997
;*
;* changes:      Jan 1999. modified to handle multiple directories
;******************************************************************************
FUNCTION papco_checkdirname, dir

CASE strupcase(!version.os_family) OF
    'VMS' : BEGIN
       IF dir NE '' THEN  BEGIN
         endch=strright(dir, 1)
         if endch eq '.' then dir=strmid(dir,0,strlen(dir)-1)+']' else $
           if endch ne ']' then dir=dir+']'
       ENDIF
    END
    'UNIX': BEGIN
      paths=str_sep(dir,',')
      for i=0,n_elements(paths)-1 do begin
        dir=paths(i)
        IF dir NE '' THEN  IF strright(dir, 1) NE '/' THEN dir=dir+'/'
        paths(i)=dir
      endfor
      if n_elements(paths) eq 1 then dir=paths(0) else dir=paths
    END
    'WINDOWS': IF dir NE '' THEN  IF strright(dir, 1) NE '\' THEN $
                  dir=dir+'\'

     ELSE: message, 'does not support ' + strupcase(!version.os_family)

ENDCASE

return,dir

END

;******************************************************************************
;* function:     papco_getenv, envname
;*
;* description:  get a pathname from an environent variable
;*
;* inputs:       envname : the name of the environment variable
;*
;* output:       the content of the environment variable named 'envname'
;*               the returned value ends with a slash
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_getenv, envname,  NOFS = NOFS

epath=getenv(envname)
IF NOT keyword_set(NOFS) THEN epath = papco_checkdirname(epath)
return,  epath

END

;******************************************************************************
;* function:     papco_hasdir, fname
;*
;* description:  checks if fname contains a directory string for
;*               the given operating system.
;*
;* inputs:       fname : input string
;*
;* output:       1 if it contains a dir, 0 if not
;*
;* keywords:     none
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_hasdir, fname

CASE strupcase(!version.os_family) OF
    'VMS' : BEGIN
      res=strpos(fname,']')
      if res eq -1 then return,0 else return,1
    END
    'UNIX': BEGIN
      res=strpos(fname,'/')
      if res eq -1 then return,0 else return,1
    END
    'WINDOWS': BEGIN
      res=strpos(fname,'\') ;'
      if res eq -1 then return,0 else return,1
    END
    ELSE  : BEGIN
      message,'papco does not yet support '+ strupcase(!version.os_family)
      stop
    END
ENDCASE

END

;******************************************************************************
;* Procedure:    papco_sepdir, fname, path, file, ext
;*
;* description:  seperates a filename string into directory and filename
;*
;* inputs:       fname : input filename
;*
;* output:       path, file, extension (strings)
;*
;* keywords:     none
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      R. Friedel, updated December 2000 to include extension
;******************************************************************************
PRO papco_sepdir, fln, dir, file, ext

  dir='' & file=''  &  ext='' & fname=fln

  if n_params() eq 4 then begin ; split out extemsion first!
    pos=strpos(fname,'.')
    if pos ne -1 then begin
      ext=strmid(fname,pos,strlen(fname)-pos)
      fname=strmid(fname,0,pos)
    endif else begin
      ext=''
      fname=fname
    endelse
  endif

  case strupcase(!version.os_family) of
    'VMS' : BEGIN
      pos=strpos(fname,']',/REVERSE_SEARCH)
      if pos eq -1 then file=fname $
      else begin
        dir=strmid(fname,0,pos+1)
        file=strmid(fname,pos+1,strlen(fname))
      endelse
    END
    'UNIX': BEGIN
      pos=strpos(fname,'/',/REVERSE_SEARCH)
      if pos eq -1 then file=fname $
      else begin
        dir=strmid(fname,0,pos+1)
        file=strmid(fname,pos+1,strlen(fname))
      endelse
    END
    'WINDOWS': BEGIN
      pos=strpos(fname,'\',/REVERSE_SEARCH) ;'
      if pos eq -1 then file=fname $
      else begin
        dir=strmid(fname,0,pos+1)
        file=strmid(fname,pos+1,strlen(fname))
      endelse
    END
    ELSE  : BEGIN
      message,'papco does not yet support '+strupcase(!version.os_family), /cont
      stop
    END
  endcase

END

;********************************************************************************
;* FUNCTION:     papco_addpath, path, new
;*
;* DESCRIPTION:  an operating system independent routine to add a path to
;*               an exisiting path. New is in UNIX format.
;*
;* INPUTS:       path       string which contains the original path
;*               new        string which contains path to be added
;*
;* OUTPUT:       Function returns the new path
;*
;* KEYWORDS:     none
;*
;* HISTORY:      written Octoberr 1997, Reiner Friedel
;********************************************************************************
function papco_addpath, path, addpath

  os=!version.os_family
  new=addpath

  case strupcase(os) of

    'UNIX':  begin
      path=papco_checkdirname(path)
      return,path+papco_checkdirname(new)
    end
    'WINDOWS': begin
      path=papco_checkdirname(path)
      new=papco_checkdirname(new)
      return,path+new
    end
    'VMS' : BEGIN
      pos = rstrpos(path,']')
      if pos ne -1 then path=strmid(path,0,pos)
      pos = rstrpos(new,'[')
      if pos ne -1 then new=strmid(new,pos+1,strlen(new))
      pos = rstrpos(new,']')
      if pos ne -1 then new=strmid(new,0,pos)
      newpath=path+'.'+new+']'
      return,newpath

      ;pos=rstrpos(new,'/')
      ;if pos ne -1 then strput,new,']',pos else return,path+new
      ;pos=strpos(path,']')
      ;if pos ne -1 then strput,path,'.',pos
      ;pos=strpos(new,'/')
      ;if pos ne -1 then new=strmid(new,1,strlen(new))
      ;new=strsubst(new,'/','.')
      ;return,path+new
    END

    ELSE  : BEGIN
      message, 'papco does not yet support ' + $
               strupcase(!version.os_family)
      stop
    END
  endcase

END

;******************************************************************************
;* Function:     papco_fname, fname
;*
;* description:  formats the filename or directory string correctly for
;*               the given operating system. For VMS, removes version #
;*
;* inputs:       fname : name and value of the env. var. in above format
;*
;* output:       the correct fname string
;*
;* keywords:     none
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_fname, fname

  os=strupcase(!version.os_family)
  os_in=''

; try to guess the format the input sting is in. "/" indicates UNIX,
; "[" indicates VMS, "\" indicates W95
  result=strpos(fname,'/')
  if result ne -1 then os_in='UNIX'
  result=strpos(fname,'[')
  if result ne -1 then os_in='VMS'
  result=strpos(fname,'\')
  if result ne -1 then os_in='WINDOWS'
  if os_in eq '' then return,fname ; if only filename then os don't matter

  CASE os OF

    'VMS' : BEGIN
      CASE os_in OF
        'VMS' : BEGIN
          ;if file has no extension, at least add a '.' dot for VMS
          papco_sepdir, fname, path, file
          if file ne '' then begin
            pos=rstrpos(file,'.')
            if pos eq -1 then fname=path+file + '.'
          endif
        END
        'UNIX': BEGIN
          pos=strpos(fname,'/')
          if pos eq 0 then strput,fname,'$',pos else fname='$'+fname
          pos=strpos(fname,'/')
          if pos ne -1 then $
            fname=strmid(fname,0,pos)+':['+strmid(fname,pos+1,strlen(fname)) $
          else fname=fname+':['
          pos=rstrpos(fname,'/')
          if pos ne -1 then strput,fname,']',pos else fname=fname+']'
          fname=strsubst(fname,'/','.')
        END
        'WINDOWS': BEGIN
          pos=strpos(fname,'\')
          if pos eq 0 then strput,fname,'$',pos else fname='$'+fname
          pos=strpos(fname,'\')
          if pos ne -1 then $
            fname=strmid(fname,0,pos)+':['+strmid(fname,pos+1,strlen(fname)) $
          else fname=fname+':['
          pos=rstrpos(fname,'\')
          if pos ne -1 then strput,fname,']',pos else fname=fname+']'
          fname=strsubst(fname,'\','.')
        END
      ENDCASE
      ; check for version numbers, delete
      pos=rstrpos(fname,';')
      if pos ne -1 then fname=strmid(fname,0,pos)
      return,fname
    END

    'UNIX': BEGIN
      CASE os_in OF
        'VMS' : BEGIN ;  "[', "]', "."   -> "/" , and ":", ";1" delete
          pos=strpos(fname,']')
          dir=strmid(fname,0,pos+1)
          file=strmid(fname,pos+1,strlen(fname))
          dir=strsubst(dir,'[','/')
          dir=strsubst(dir,']','/')
          dir=strsubst(dir,'.','/')
          dir=strsubst(dir,':',' ')
          dir=strcompress(dir,/REMOVE_ALL)
          fname=dir+file
        END
        'UNIX': BEGIN
        END
        'WINDOWS': BEGIN
         fname=strsubst(fname,'\','/')
        END
      ENDCASE
      ; check for version numbers, delete
      pos=rstrpos(fname,';')
      if pos ne -1 then fname=strmid(fname,0,pos)
      return,fname
    END

    'WINDOWS': BEGIN
      CASE os_in OF
        'VMS' : BEGIN ;  "[', "]', "."   -> "/" , and ":", ";1" delete
          pos=strpos(fname,']')
          dir=strmid(fname,0,pos+1)
          file=strmid(fname,pos+1,strlen(fname))
          dir=strsubst(dir,'[','\')
          dir=strsubst(dir,']','\')
          dir=strsubst(dir,'.','\')
          dir=strsubst(dir,':',' ')
          dir=strcompress(dir,/REMOVE_ALL)
          fname=dir+file
        END
        'UNIX': BEGIN
          fname=strsubst(fname,'\','/')
        END
        'WINDOWS': BEGIN
        END
      ENDCASE
      ; check for version numbers, delete
      pos=rstrpos(fname,';')
      if pos ne -1 then fname=strmid(fname,0,pos)
      return,fname
    END

    ELSE  : BEGIN
      message, 'papco does not yet support ' + os
      stop
    END

  ENDCASE

END

;******************************************************************************
;* Procedure:    papco_setenv
;*
;* description:  sets an environmental variable. Expects the input
;*               string to be in the UNIX format:
;*               'ENV_VAR_NAME=env_var_value'
;*
;* inputs:       setstr : name and value of the env. var. in above format
;*
;* output:       none
;*
;* keywords:     FS    - file stystem variable, check for correct naming.
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
PRO papco_setenv, setstr, FS=FS

  os=strupcase(!version.os_family)

; check for valid input string
  result=str_sep(setstr,'=')
  if n_elements(result) eq 2 then begin
    env_var_name=result(0)
    env_var_value=result(1)
  endif else begin
    print,'% papco_papco_papco_setenv: Wrong format. Env. Variable not set.'
    return
  endelse

; format the env_var_value directory correctly for the operating system.
  if keyword_set(FS) then env_var_value=papco_fname(env_var_value)

  CASE os OF
    'VMS' : BEGIN
     setlog, env_var_name, env_var_value
    END
    'UNIX': BEGIN
      ;check for $HOME in path and expand.
      p=strpos(env_var_value,'$HOME')
      if p ne -1 then begin
        home=getenv('HOME')
        env_var_value=strmid(env_var_value,0,p)+home+ $
                      strmid(env_var_value,p+5,strlen(env_var_value))
      endif
      setenv, env_var_name + '=' + env_var_value
    END
    'WINDOWS': begin
      setenv, env_var_name + '=' + env_var_value
    end
    ELSE  : BEGIN
      message,'papco does not yet support '+ os,  /cont
      stop
    END
  ENDCASE

END

;******************************************************************************
;* Procedure:    papco_unzip, cmdline, result
;*
;* description:  Executes operating system command for unzipping files
;*               Uses unzip on UNIX / MAC OS10
;*               Uses the command line version of WINZIP 8.1 on Windows WZUNZIP
;*               No support for other MAC OS or VMS yet.
;*               Action: unzips files PRESERVING origimal zipped file.
;*
;*
;* inputs:       cmdline : command line string, needed
;*               result  : if passed in, pass out output of spawn in it.
;*
;* output:       result
;*
;* keywords:     VERBOSE - print cmdline before executing
;*
;* author:       R. Friedel, July 2003
;*
;* changes:      -
;******************************************************************************
PRO papco_unzip, zip_source, unzipfile, result, VERBOSE=VERBOSE

os = strupcase(!version.os_family)

cmd = getenv('PAPCO_UNZIP')
IF cmd EQ '' THEN BEGIN
    message, 'PAPCO_UNZIP NOT set,  cannot do unzip', /cont
    return
ENDIF

papco_sepdir, unzipfile, path, fln, ext

CASE os OF
    'UNIX': BEGIN
        cmdline = cmd+' -o '+zip_source+' '+path+' '+fln+ext
        IF keyword_set(VERBOSE) THEN message, cmdline, /cont
        spawn, cmdline, result
    END
    'WINDOWS': BEGIN
        cmd = '"'+cmd+'"'
        cmdline = cmd+' -o '+zip_source+' '+path+' '+fln+ext
        IF keyword_set(VERBOSE) THEN message, cmdline, /cont
        spawn, cmdline, result, /LOG_OUTPUT
    END
    ELSE  : BEGIN
      message, 'papco does not yet support ' + os
      stop
    END
ENDCASE

END

;******************************************************************************
;* Procedure:    papco_gunzip, cmdline, result
;*
;* description:  Executes operating system command for gunzipping files
;*               Uses gzip -d on UNIX / MAC OS10
;*               No support for other MAC OS, Windows, or VMS yet.
;*               Action: gunzips files PRESERVING original gzipped file.
;*
;*
;* inputs:       cmdline : command line string, needed
;*               result  : if passed in, pass out output of spawn in it.
;*
;* output:       result
;*
;* keywords:     VERBOSE - print cmdline before executing
;*
;* author:       J. Niehof February 2003 (from papco_unzip R. Friedel)
;*
;* changes:      J. Faden September 2004
;*                 use idl's open,/compress to implement gunzip for non-unix
;******************************************************************************
PRO papco_gunzip, zip_source, unzipfile, result, VERBOSE=VERBOSE

os = strupcase(!version.os_family)

CASE os OF
    'UNIX': BEGIN
        cmdline = 'gzip -dc '+zip_source+' > '+unzipfile
        IF keyword_set(VERBOSE) THEN message, cmdline, /cont
        spawn, cmdline, result
    END
    ELSE  : BEGIN
        ;; this is untested on a vax, and documentation of compress
        ;; indicates there might be problems with this

        openr, in, /compress, zip_source, /get_lun

        openw, out, unzipfile, /get_lun
        buffsize= 1024
        buffer = bytarr(buffsize)

        on_ioerror, almostDone  ;; causes seg fault on linux if invalid gz
        bytesRead=0L
        while ( 1 ) do begin
            readu, in, buffer
            writeu, out, buffer
        endwhile

        almostDone:
        ;  transfer_count doesn't set the bytes read on ioerror, but
        ;  does read the bytes.  Kludge to get bytesRead by looking
        ;  for last non-zero byte

        r= where( buffer ne 0 )
        if r[0] eq -1 then lastbyte= buffsize-1
        lastByte= r[n_elements(r)-1]

        buff= bytarr( lastByte+1 )
        buff[*]= buffer[0:lastByte]
        writeu, out, buff

        close, in
        free_lun, in
        close, out

        free_lun, out
    END
ENDCASE

END

;******************************************************************************
;* Procedure:    papco_spawn, cmdline, result
;*
;* description:  executes operating system commands. Only works for the
;*               list of commands "known" here. The cmdline passed in is
;*               parsed for a set of "known" commands, and the correct
;*               command for the os is inserted. The arguments are NOT
;*               checked.
;*
;* inputs:       cmdline : command line string, needed
;*               result  : if passed in, pass out output of spawn in it.
;*
;* output:       none
;*
;* keywords:     none
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
PRO papco_spawn, cmdline, result,  VERBOSE = VERBOSE

; list of known commands are the UNIX commands which are used in PAPCO:
know_cmd=['rm',        'mv',           'cp',   'ls'   ]
UNIX_cmd=['/bin/rm -f','/bin/mv -f',   'cp'  , 'ls -l']
VMS_cmd =['del',       'rename',       'copy', 'dir'  ]
W95_cmd =['del',       'move',         'copy', 'dir'  ]

os=strupcase(!version.os_family)

CASE os OF
    'VMS': BEGIN
      for i=0,n_elements(know_cmd)-1 do begin
        cmd=strmid(cmdline,0,strlen(know_cmd(i)))
        arg=strmid(cmdline,strlen(know_cmd(i)),strlen(cmdline))
        if cmd eq know_cmd(i) then begin
          cmd=VMS_cmd(i)
          if cmd eq 'del' then arg=arg+';*' ; add version wildcard for VMS del
          cmdline=cmd+arg
        endif
      endfor
    END
    'UNIX': BEGIN
      for i=0,n_elements(know_cmd)-1 do begin
        cmd=strmid(cmdline,0,strlen(know_cmd(i)))
        arg=strmid(cmdline,strlen(know_cmd(i)),strlen(cmdline))
        if cmd eq know_cmd(i) then begin
          cmd=UNIX_cmd(i)
          cmdline=cmd+arg
        endif
      endfor
    END
    'WINDOWS': BEGIN
      for i=0,n_elements(know_cmd)-1 do begin
        cmd=strmid(cmdline,0,strlen(know_cmd(i)))
        arg=strmid(cmdline,strlen(know_cmd(i)),strlen(cmdline))
        if cmd eq know_cmd(i) then begin
          cmd=W95_cmd(i)
          cmdline=cmd+arg
        endif
      endfor
    END
    ELSE  : BEGIN
      message, 'papco does not yet support ' + os
      stop
    END
ENDCASE

;if result was passed in, use it
IF keyword_set(result) then spawn, cmdline, result else $
	spawn, cmdline

END


;******************************************************************************
;* function:     papco_add_module_path, path_name
;*
;* description:  returns the entry needed in the papco_include_modules.pro
;*               file to extend the idl search path by the path_name
;*
;* inputs:       path_name
;*
;* output:       the extend path string
;*
;* keywords:     none.
;*
;* author:       R. Friedel, November 1997
;*
;* changes:      Jeremy Faden, September 2004
;*               use papco_add_to_path, which only adds to the path if the
;*                 directory is not on the path already.  This allows the
;*                 use (hopefully) of IDL's .reset command. .reset does not
;*                 clear the path.
;******************************************************************************
FUNCTION papco_add_module_path, path_name

  return, "papco_add_to_path, /expand, '"+path_name+"'"

 ;; case strupcase(!version.os_family) of

 ;;   'UNIX':  begin
 ;;     sep=':'  & slash= '/' & tail=''
 ;;     outstr="!path = !path + '" + sep + "' + expand_path('+"
 ;;     len=strlen(path_name)
 ;;     if strmid(path_name,len-1,len) eq '/' then $
 ;;       path_name=strmid(path_name,0,len-1)
 ;;   end
;;
 ;;   'WINDOWS':  begin
 ;;     sep=';'  & slash= '\' ;'
 ;;     tail=''
 ;;     outstr="!path = !path + '" + sep + "' + expand_path('+"
 ;;   end

 ;;  'VMS' : BEGIN
 ;;     sep=','  & slash='.' & tail=''
 ;;     outstr="!path = !path + ',' + expand_path('+"
 ;;   END

 ;;   ELSE  : BEGIN
 ;;     message, 'papco does not yet support ' + $
 ;;              strupcase(!version.os_family)
 ;;     stop
 ;;   END

 ;; endcase

 ;; return,outstr + path_name + tail + "')"

END

;******************************************************************************
;* function:     papco_get_dirlist, path_name
;*
;* description:  returns list of directories found in path_name
;*
;* inputs:       path_name
;*
;* output:       the array of directory names
;*
;* keywords:     none.
;*
;* author:       R. Friedel, November 1997
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_get_dirlist, path_name

  case strupcase(!version.os_family) of
    'VMS' : BEGIN
       len=strlen(path_name)
       dirlist=FINDFILE(path_name+'*.DIR')
       pos=strpos(dirlist,'.DIR')
       for i=0,n_elements(dirlist)-1 do $
         dirlist(i)=strmid(dirlist(i),len,pos(i)-len)
    END
    'UNIX': BEGIN
      dirlist=papco_dirfilter(FINDFILE(path_name), path_name)
     END
    'WINDOWS': BEGIN
       CD,path_name, CURRENT = old_path
       dirlist=FINDFILE('*')
       CD, old_path
       jbsl = strpos( dirlist, '\')
       ibsl = where( jbsl ne -1L, n)
       if ibsl(0) ne -1L then begin
       		dirlist = dirlist( ibsl)
			jbsl = jbsl( ibsl)
  	   		dirlist1 = strarr( n)
  	   		for i=0, n - 1 do dirlist1( i) = strmid( dirlist(i), 0, jbsl(i))
  	   		dirlist = dirlist1
       endif else dirlist = ''
       jdot = where( (dirlist ne '.') and (dirlist ne '..'))
       if jdot(0) ne -1L then dirlist = dirlist( jdot) else dirlist = ''

     END
     ELSE : BEGIN
       print,'% papco_configure: papco does not yet operate under ', $
       strupcase(!version.os_family)
       stop
     END
  endcase

  return,dirlist

END


;******************************************************************************
;* procedure:    papco_set_plot
;*
;* description:  sets the plot device to the appropriate device for
;*               the operating system.
;*
;* inputs:       device, a string like 'X' or 'PS'
;*
;* output:       none.
;*
;* keywords:     none.
;*
;* author:       Jeremy Faden
;*
;* changes:      -
;******************************************************************************
PRO papco_set_plot, device, _extra=e

  device= strupcase( device )

  window_devices=[ 'X', 'WIN' ] ; known window devices

  if ( where( window_devices eq device ))(0) ne -1 then begin ; is window
    case strupcase(!version.os_family) of
      'UNIX': device= 'X'
      'VMS': device= 'X'
      'WINDOWS': device= 'WIN'
    else: begin
      message, 'Operating System not supported yet.', /cont
      stop
    end
    endcase
    set_plot, device, _extra=e
    device, retain=2
    return
  endif

  if !d.n_colors gt 256 then device, decomposed=0
  set_plot, device, _extra=e

END


;******************************************************************************
;* function:     papco_get_module_names
;*
;* description:  given a list of module dirs, returns the module names.
;*
;* inputs:       none
;*
;* output:       list of module names
;*
;* keywords:     none.
;*
;* author:       R. Friedel, November 1997
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_get_module_names, paths

list=paths

CASE  strupcase(!version.os_family) OF

    'UNIX':  begin
      names=strarr(n_elements(list))
      for i=0,n_elements(list)-1 do begin
        pos=strpos(list(i),'/')
        if pos(0) ne -1 then begin
          dirparts=str_sep(list(i),'/')
          name=dirparts(n_elements(dirparts)-2)
          names(i)=name
        endif else  names(i)= list(i)
      endfor
      return, names
    end

    'VMS' : BEGIN
      return, list
    END

    'WINDOWS': BEGIN
        for i=0,n_elements(list)-1 do begin
            dirparts=str_sep(list(i),'\') ;'
            name=dirparts(n_elements(dirparts)-2)
            list(i)=name
        endfor
        index=where((list ne '.') and (list ne '..'))
        list=list(index)
        ;sort alphabetically
        idx=sort(list)
        list=list(idx)
        return,list
    END

    ELSE: message, 'does not support ' + strupcase(!version.os_family)

ENDCASE

END

;******************************************************************************
;* function:     papco_helpfile
;*
;* description:  uses the environment-variable 'PAPCO_PATH' to identify
;*                                                        the directory
;*
;* inputs:       the filename without directory
;*
;* output:	 the complete filename
;*
;* author:       R. Friedel, March 1998
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_helpfile, filename

  papco_path=papco_getenv('PAPCO_PATH')
  help_path=papco_addpath(papco_path, 'help')

  RETURN, help_path+filename

END

;******************************************************************************
;* function:     papco_check_window_device
;*
;* description:
;*
;* inputs:
;*
;* output:
;*
;* author:       R. Friedel, June 1998
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_check_window_device

if (!D.NAME eq 'X') or (!D.NAME eq 'WIN') or (!D.NAME eq 'Z') $
    then return,1 else return,0

END

;******************************************************************************
;* function:     papco_papco_have_utility
;*
;* description:  checks if a given system utility program is
;*               available.
;*
;* inputs:       exec_name - string, name of the command
;*
;* output:	 1 if it exisits, 0 if not
;*
;* author:       R. Friedel, October 1998
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_have_utility, exec_name

CASE strupcase(!version.os_family) OF

    'UNIX':  begin
        spawn,'which '+exec_name, result
        pos=strpos(result(0),'Command not found')
        if pos ne -1 then return, 0 else return,1
    END

    ELSE: message,'papco does not yet support ' + strupcase(!version.os_family)

ENDCASE

END

;******************************************************************************
;* function:     papco_unilib_exe
;*
;* description:  returns the correct path to the UNILIB .so file for
;*               the current operating system.
;*
;* inputs:       none
;*
;* output:       path string
;*
;* keywords:     none
;*
;* author:       R. Friedel, June 2003
;*
;* changes:      -
;******************************************************************************
FUNCTION papco_get_unilib_exe

unilib_path_root = papco_getenv('PAPCO_LIB')+'unilib/'

CASE STRUPCASE(!version.os_family) OF
    'UNIX': BEGIN
        CASE STRUPCASE(!version.os) OF
            'LINUX': unilib_exe = 'linux/'
            'SUNOS': unilib_exe = 'solaris/'
            ELSE: BEGIN
                message, 'PAPCO does not support UNILIB for ' + $
                  strupcase(!version.os_family)+' -> '+ $
                  strupcase(!version.OS), /info
                return, ''
            END
        ENDCASE
    END

    ELSE: BEGIN
        message, 'PAPCO does not support UNILIB for ' + $
          strupcase(!version.os_family), /info
        return, ''
    END
ENDCASE

version = float(strmid(!VERSION.RELEASE, 0, 3))
IF version GT 5.4 THEN idl_ver_str = 'idl_5_5' ELSE idl_ver_str = 'idl_5_4'

fln = unilib_path_root+unilib_exe+'unilib_2_06_'+idl_ver_str+'.so'

;see if the file requested exisits
res = findfile(fln, count = c)

IF c EQ 1 THEN return, fln ELSE BEGIN
    message, 'Unilib library not found:', /cont
    print, '  ', fln
    return, ''
ENDELSE

END

;******************************************************************************
;* Procedure:    papco_opsys_path, path
;*
;* description:  Checks path string for conformity to operating system rules
;*               Changes all instances of directory separator character
;*               No support for other MAC OS or VMS .
;*               Action: unzips files PRESERVING origimal zipped file.
;*
;*
;* inputs:       path :    String containing paths, ie '/n/radbelt/crres'
;*
;* output:       path is modified in place
;*
;* keywords:     None.
;*
;* author:       J. Roeder, November 2003
;*
;* changes:      -
;******************************************************************************
PRO papco_opsys_path, path


CASE !version.os_family of

	'Windows':	BEGIN

		good_sep = '\'
		bad_sep = '/'
		END

	'Unix':		BEGIN

		good_sep = '\'
		bad_sep = '/'
		END

	ELSE: return

ENDCASE


done = 0
WHILE NOT done do begin

	bad = strpos( path, bad_sep)
	if bad ne -1L then strput, path, good_sep, bad else done = 1

ENDWHILE


END

;******************************************************************************
;* Procedure:    papco_wget
;*
;* description:  Downloads data file from remote system using Wget utility.
;*
;*
;* inputs:       url ie ftp://user.password@nis-ftp.lanl.gov/omni/omni2_1997.idl
;*               logfile is complete path and name of log file
;*
;* output:       pid is process identifier for wget subprocess
;*
;* keywords:     None.
;*
;* author:       J. Roeder, November 2003
;*
;* changes:      Jeremy Faden, September 2004.  pid is the pid of the
;*                  wget process, not the unix shell
;******************************************************************************
PRO papco_wget, url, logfile, pid

wget_exe='wget'

CASE strupcase(!version.os_family) OF

    'UNIX':	BEGIN

        exe_str= wget_exe+" "+ url+" -o "+logfile+" &"
        spawn, exe_str, pid=pid, result
        if stregex( result[0], '\[[0-9]\] [0-9]*' ) ne -1 then begin
            s= str_sep( result[0], ' ' )
            pid= long(s[1])
        end
    END

    'WINDOWS': BEGIN

        exe_str= wget_exe+' '+ url+' -o '+logfile
        spawn, exe_str, pid=pid, /NOWAIT, /HIDE

    END

ENDCASE

END


;******************************************************************************
;* Procedure:    papco_chmod
;*
;* description:  Changes file permissions only for UNIX.
;*
;*
;* inputs:       String containing requested permission changes, eg 'g+rw'
;*
;* output:       None.
;*
;* keywords:     None.
;*
;* author:       J. Roeder, November 2003
;*
;* changes:      -
;******************************************************************************
PRO papco_chmod, perm, file

CASE !version.os_family OF

	'Unix':	BEGIN

        spawn, 'chmod ' + perm, + ' ' + tmp_dir+file

		END

	ELSE:

ENDCASE

END

;******************************************************************************
;* Procedure:    papco_cleanup
;*
;* description:  Called on exiting the PAPCO main window. Used to delete
;*               temporary files etc
;*
;* inputs:       none
;*
;* output:       none
;*
;* keywords:     none.
;*
;* author:       R. Friedel, October 1997
;*
;* changes:      -
;******************************************************************************
PRO papco_cleanup

path=papco_getenv('PAPCO_HOME')  ;startup directory
path=papco_addpath(papco_getenv('PAPCO_CONFIG'),'papco_state')

; the default-settings (printer, paths, ...) are written
papco_writeDefaultSettings

; delete all papco state files
result = papco_file_search(path+'/state_*.papco',count=n)
if n ne 0 then BEGIN
    papco_file_delete, result, /verb
    message,'Removed '+varprt(n)+' temporary papco state files',/cont
ENDIF

; remove temporary print-file if it exists
result =  papco_file_search(path+'papco_print_tmp.ps',count=n)
if n ne 0 then BEGIN
    papco_file_delete, result, /verb
    message,'Removed temporary print file',/cont
endif

; purge old version numbers of files created by PAPCO in PAPCO_HOME for VMS
if !version.os_family eq 'vms'  then spawn,'purge '+path+'*.*'

; clean up any "leftovers" in the download directory
result =  papco_file_search(path+'papco_download_temp/*.*',count = n)
if n ne 0 then papco_file_delete, result, /verb

END

function papco_get_tempfile
  if !version.os_family eq 'unix' then begin
      f= findfile( '/tmp/papco.???.tmp', count=count )
      return, '/tmp/papco.'+string(count,format='(i3.3)')+'.tmp'
  endif else begin
      f= findfile( getenv('TEMP')+'\papco.???.tmp', count=count )
      return, getenv('TEMP')+'\papco.'+string(count,format='(i3.3)')+'.tmp'
  endelse
end

function papco_is_windows_xp
   if !version.os_family eq 'Windows' then begin
      spawn, 'tasklist', result
      return, n_elements(result) gt 1
   endif else begin
      return, 0
   endelse
end

pro papco_process_kill, pid
   if !version.os_family eq 'unix' then begin
   	  cmd= 'kill '+strtrim(pid,2)
   	  spawn, cmd

   endif else if !version.os_family eq 'Windows' then begin
      if ( papco_is_windows_xp() eq 1 ) then begin
		spawn, 'taskkill /pid '+strtrim(pid,2), result, /hide
      endif else begin
	  	message, 'not supported on windows versions less than XP'
      endelse

   endif else begin
      message, 'OS not supported'

   endelse
end


function papco_process_active, pid
   ;; returns 1 if a process is running, 0 if not running
  if !version.os_family eq 'unix' then begin
      cmd= 'ps -ef | grep '+strtrim(pid,2)+' | grep -v grep'
      spawn, cmd, result
      count=0
      for i=0, n_elements(result)-1 do BEGIN
          pos=strpos(result(i),strtrim(pid,2))
          if pos ne -1 THEN BEGIN
              s= str_sep( strcompress(result[i]), ' ' )
              thisid= long(s[1])

              if thisid eq pid then count=count+1
          endif
      endfor

      return, count gt 0

  endif else if !version.os_family eq 'Windows' then begin
      spawn, 'tasklist ', result, /hide
      haveCommand=0
      haveProcess=0
      for i=0,n_elements(result)-1 do begin
          if strtrim(result[i],2) eq '' then continue
          if ( strmid( result[i],0,5 ) eq 'Image' ) then begin
              haveCommand=1
          endif else if ( strmid(result[i],0,5) eq '=====' ) then begin
          endif else begin
              pid1= long( strmid( result[i], 26, 6 ) )
              if pid1 eq pid then haveProcess=1
          endelse
      endfor
      if ( not haveCommand ) then message, 'OS not supported'  ;  win98, NT, etc
                                                          ; do not have this command
      return, haveProcess

  endif else begin
      message, 'OS not supported'

  endelse
end

function papco_dirsep
  if !version.os_family eq 'unix' then begin
   	  return, '/'
   endif else if !version.os_family eq 'Windows' then begin
      return, '\'
   endif else begin
      message, 'OS not supported'
   endelse
end

function papco_pathsep
  if !version.os_family eq 'unix' then begin
   	  return, ':'
   endif else if !version.os_family eq 'Windows' then begin
      return, ';'
   endif else begin
      message, 'OS not supported'
   endelse
end