;******************************************************************************
;******************************************************************************
;* FILE:	 
;*      papco_orbitinfo.pro
;* 
;* DESCRIPTION:  
;*      this file contains functions and structures for keeping information 
;*      about orbits. All info are stored in an array containing instances
;*      of the ORBIT_INFO-structure.
;* 
;* STRUCTURES:
;*     {ORBIT_INFO, $     
;*	number:0, 	$	; this is the orbit-number
;*	day:0, 		$ 	; the  day of the orbit
;*	year:0, 	$	; its year
;*	xut1:0.0, 	$	; the start of the orbit in seconds
;*	xut2:0.0, 	$	; its end
;*	xdata:0, 	$	; the number of time-data
;*	ydata:0 } 		; the number of energy-data
;*
;* COMMON-BLOCKS:   
;*      COMMON PAPCO_ORBIT_INFO, readOrbits, orbitDescriptions, satellite_name
;*	       this common-block is used to cache orbit-infos.
;*
;*	readOrbits	an array of 'POLAR_ORBIT_INFO'-structures
;*			when calling 'papco_readOrbitInfo', the result is
;*			cached in this variable
;*	orbitDescriptions an array of strings
;*			when calling 'papco_orbitDescriptionList', the 
;*			result is cached here.
;*
;* OVERVIEW:
;*
;*     FUNCTION papco_getOrbitInfoStruct
;*              -> get an orbit-template
;*
;*     FUNCTION papco_orbitDescriptionList, orbitInfo, CACHE_THIS=CACHE_THIS
;*              -> list of verbose orbit-descriptions
;*
;*     FUNCTION papco_writeOrbitInfo, SCraft, orbitInfo
;*              -> store orbit-infos in 'orbit.infos'
;*     
;*     FUNCTION papco_readOrbitInfo, SCraft, orbitInfo
;*              -> read the infos created by papco_createOrbitInfo
;*
;*     FUNCTION papco_getOrbitInfoFor, SCraft, anOrbitNr
;*              -> get the information for one orbit
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel 
;*     july 1997, Haje Korth
;******************************************************************************
;******************************************************************************
 
;******************************************************************************
;* FUNCTION:     
;*      papco_getOrbitInfoStruct
;* 
;* DESCRIPTION:  
;*      this function is used to create structures of the ORBIT_INFO-
;*      type. All procedures that use this type should create instances of
;*      the structure by using this function to enable maintaining.
;* 
;* INPUTS:       
;*      none
;* 
;* OUTPUT:	 
;*      a Variable of the type ORBIT_INFO
;*      for a description of its structure, see top of this file
;*
;* CALLING SEQUENCE:
;*      template=papco_getOrbitInfoStruct()
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel 
;******************************************************************************
FUNCTION papco_getOrbitInfoStruct

   return, {ORBIT_INFO, $     
  	   number:0, 	$	; this is the orbit-number
	   day:0, 	$ 	; the orbit's day
	   year:0, 	$	; its year
	   xut1:0l, 	$	; the start of the orbit in seconds
	   xut2:0l, 	$	; its end
	   xdata:0, 	$	; the number of time-data
     	   ydata:0 } 		; the number of energy-data

END; (papco_getOrbitInfoStruct)

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION papco_orbitDescriptionList, orbitInfo, CACHE_THIS=CACHE_THIS
;* 
;* DESCRIPTION:  
;*      This function is mainly used by papco_selectOrbitinfo.pro. It creates 
;*      a string array containing descriptions for all orbits in the 
;*	parameter 'orbitInfo'.
;*	This list is the list that's displayed in the list-window of the 
;*	orbit-selection-widget.
;*	Once the description-list is created, it is cached in the common-
;*	variable 'orbitDescriptions', if the keyword 'CACHE_THIS' was set.
;*
;* INPUTS:       
;*      orbitInfo	the array of orbit-descriptions for which infos
;*			shall be generated
;*
;* KEYWORDS:
;* 	CACHE_THIS	if set, the generated descriptions are cached in 
;*			'orbitDescriptions'.
;*
;* OUTPUT:	 
;*      a string array	containing verbose descriptions for each orbitInfo
;*			in the array 'orbitInfo'
;*
;* CALLING SEQUENCE:
;*      -
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel 
;******************************************************************************
FUNCTION papco_orbitDescriptionList, orbitInfo, CACHE_THIS=CACHE_THIS
  
  COMMON PAPCO_ORBIT_INFO, readOrbits, orbitDescriptions, satellite_name

  IF KEYWORD_SET(CACHE_THIS) THEN $
    IF N_ELEMENTS(SIZE(orbitDescriptions)) GT 3 THEN $ 
	return, orbitDescriptions
  
  list = strarr(N_ELEMENTS(orbitInfo))
  FOR i=0, N_ELEMENTS(orbitInfo)-1 DO BEGIN 
    timeFrom=convert_secondsToTime(orbitInfo(i).xut1)
    timeTo  =convert_secondsToTime(orbitInfo(i).xut2)
    date    =convert_dayToDate(orbitInfo(i).day, orbitInfo(i).year)
    list(i)=STRING($
        orbitInfo(i).number, date, orbitInfo(i).day, orbitInfo(i).year, $
        timeFrom, timeTo, $
        orbitInfo(i).xdata, orbitInfo(i).ydata, $
        FORMAT = '(I4,": ", A,"  (",I3,"/",I4,")",' + $
                 '"  ", A, "-",A, TR10,'+ $
                 '"xData:", I5, TR8, "yData:", I3)')
  ENDFOR

  IF KEYWORD_SET(BUFFER_LIST) THEN orbitDescriptions=list
  return, list
  
END

;******************************************************************************
;* FUNCTION:     
;*      papco_writeOrbitInfo
;* 
;* DESCRIPTION:  
;*      the orbit-infos are written.
;* 
;* INPUTS:       
;*      SCraft:      Name of space craft
;*      orbitInfo :  an array containing instances of the 
;*                   ORBIT_INFO-structure
;* 
;* OUTPUT:       
;*      a string. If writing was succesful, the string is 'OK'. 
;*      If not, it contains the error message
;*
;* CALLING SEQUENCE:
;*	result=papco_writeOrbitInfo(SCraft, anOrbitInfoArray)
;*	IF result NE 'OK' THEN BEGIN 
;*         print, 'error:' + result  
;*	   return
;*      ENDIF
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel
;*     july 1997, Haje Korth 
;******************************************************************************
FUNCTION papco_writeOrbitInfo, SCraft, orbitInfo

   papco_util_path=papco_getenv(SCraft+'_ORBITS')
   IF papco_util_path EQ '' THEN BEGIN
      print, '-------------------------------------------------------'
      print, 'you have to set the environment-variable '+SCraft+'_ORBITS,'
      print, 'if you want to use this data.'
      print, 'for now, I am writing the data into your home-'
      print, 'directory'
      print, '-------------------------------------------------------'
      papco_util_path='~/'
   ENDIF

   print, '...now writing orbitInfo to the file ' + $
	  papco_util_path+Scraft+'_orbits.info'

   anzahl=N_ELEMENTS(orbitInfo)

   GET_LUN, outputUnit
   openw, outputUnit, papco_util_path+SCraft+'_orbits.info', ERROR=err
   IF (err NE 0) THEN RETURN, !ERR_STRING
         
   printf, outputUnit, '--- FILE CONTAINS ORBIT INFOS ---'
   printf, outputUnit, anzahl
   FOR i=0, anzahl-1 DO BEGIN
      PRINTF, outputUnit, $
              orbitInfo(i).number, orbitInfo(i).day, orbitInfo(i).year, $
              orbitInfo(i).xut1,   orbitInfo(i).xut2, $
              orbitInfo(i).xdata,  orbitInfo(i).ydata
   ENDFOR; (i=0, anzahl-1)
   close, outputUnit
   free_lun, outputUnit
   RETURN, 'OK'
   
END ;(papco_writeOrbitInfo)

;******************************************************************************
;* FUNCTION:     
;*      papco_readOrbitInfo
;* 
;* DESCRIPTION:  
;*      read an Array of orbit-infos.
;*      The read orbits are cached in the variable 'readOrbits'.
;*
;* INPUTS:       
;*      SCraft          : name of space_craft
;*      orbitInfo	: the Variable, in which the orbits are to be read
;* 
;* OUTPUT:       
;*      if reading was succesful, the string 'OK' is returned
;*		           and the parameter orbitInfo will 
;*			   contain an array of orbitInfos
;*      if reading failed, a string containing the error-message
;*	                   is returned, and the parameter
;*    			   orbitInfo will have undefined
;*			   value.
;* 
;* CALLING SEQUENCE:
;*      result=papco_readOrbitInfo(SCraft, orbits)
;*	IF result NE 'OK' THEN BEGIN 
;*         print, 'error:' + result  
;*	   return
;*       ENDIF ELSE $
;*         doWhateverYouWantWith, orbits
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel 
;*     july 1997, Haje Korth
;******************************************************************************
FUNCTION papco_readOrbitInfo, SCraft, orbitInfo

COMMON PAPCO_ORBIT_INFO, readOrbits, orbitDescriptions, satellite_name
  
orbits_path=papco_getenv(SCraft+'_ORBITS')
IF orbits_path EQ '' THEN BEGIN
    message,'environment-variable '+ SCraft+'_ORBITS not set', /cont
    print, ' Trying to read orbit-infos from current directory'
    orbits_path=''
ENDIF
  
message, orbits_path+SCraft+'_orbits.info', /cont
GET_LUN, inputUnit
openr, inputUnit, orbits_path+SCraft+'_orbits.info', ERROR=err
IF (err NE 0) THEN BEGIN
    message,orbits_path+SCraft+'_orbits.info file not found', /cont
    RETURN, !ERR_STRING
ENDIF
 
dummy=''			; the first line of the file contains
readf, inputUnit, dummy         ; a file-description only
anzahl=0
READF, inputUnit, anzahl        ; number of orbit-infos.
tmp=papco_getOrbitInfoStruct()
readOrbits=REPLICATE(tmp, anzahl)
READF, inputUnit, readOrbits
close, inputUnit  &  free_lun, inputUnit

orbitInfo=readOrbits
IF orbitInfo(0).year LT 1900 THEN orbitInfo(*).year=orbitInfo(*).year+1900
  
return, 'OK'
   
END ;(read_orbitInfo)


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION papco_getOrbitInfoFor, SCraft, anOrbitNr
;* 
;* DESCRIPTION:  
;*      If possible, get the orbit-information for spacecraft 'SCraft'
;*      and the orbit with number 'anOrbitNr'
;*	To do this, the orbit-informations are fetched using 
;*	'papco_readOrbitInfo'. If the hereby returned list contains an orbit
;*	with number 'anOrbitNr', it is returned.
;*
;*	If it does not contain such an orbit, a string containing an 
;*	error-text is returned
;*
;* INPUTS:       
;*      SCraft          - name of space craft
;*      anOrbitNr	- an integer from 0..9999
;*			  the number of the orbit 
;*
;* OUTPUT:	 
;*      an instance of the expanded PAPCO_ORBIT_INFO-structure 
;*	or a string containing an error-description
;* 
;* CALLING SEQUENCE:
;*      a=papco_getOrbitInfoFor, 'POLAR', 100
;*	if n_elements(size(a)) LE 3 then print, 'error'+a
;*
;* MODIFICATION HISTORY:       
;*     october 1996, written by R. Friedel 
;*     july 1997, Haje Korth
;*     May 1999, R. Friedel, to also return orbit time in MFDT format
;******************************************************************************
FUNCTION papco_getOrbitInfoFor, SCraft, anOrbitNr

COMMON PAPCO_ORBIT_INFO, readOrbits, orbitDescriptions, satellite_name

COMMON get_error, get_err_no, get_err_msg
COMMON orbinfo, iorb,iday,iyear
COMMON time, xut1,xut2
COMMON mjdt, mjdt_start, mjdt_end
COMMON PLOT_COMPOSER, widgetData

IF anOrbitNr LT 0 OR anOrbitNr GT 9999 THEN $
    return, 'The orbit has to be an integer in the interval [0,9999]'
   
  ; check if new orbitdata needs to be read
  if not keyword_set(satellite_name) then satellite_name=''
  if SCraft ne satellite_name then begin
    dummy=papco_ReadOrbitInfo(SCraft, anOtherDummy)
    satellite_name=SCraft
  endif
  
  IF N_ELEMENTS(SIZE(readOrbits)) GT 3 THEN BEGIN
    index=where(readOrbits.number eq anOrbitNr,c)
    if c eq 1 then begin        ;found one orbit, O.K.
      RETURN,readOrbits(index(0))
    endif
  ENDIF 

  ; The orbits are not yet cached or the requested orbit was not found in the 
  ; cached data. The last can happen, if the orbit-descriptions have not 
  ; recently been updated. 
  dummy=messageBox(['The requested orbit, '+ strtrim(string(anOrbitNr), 2), $
                    ' could not be located !'], $
                   ['What a pity :->'], $
                   title='Orbit information not located', /center)
   
  return, papco_getOrbitInfoStruct()
   
END

;******************************************************************************
;******************************************************************************
;* PROCEDURES:	 
;*      papco_selectOrbits_*
;* 
;* DESCRIPTION:  
;*	A window containg a list of known orbits is opened. When the user
;*	selects one, it is returned to the calling program.	
;*	The displayed orbits may be restricted to a certain range.
;*
;* USED MODULES:
;*	papco_orbitinfo.pro	 for managing the orbit-infos
;*	messagebox.pro
;*	papco_util.pro		 
;*
;* COMMON-BLOCKS:
;*      COMMON papco_SELECTORBITINFO, SC, selectedOrbit
;*			this one is used to store the orbit the user
;*			selected last. No other program has to use it,
;*			as its value is returned by 'papco_selectOrbit()'
;*
;* OVERVIEW:
;*      a=papco_selectOrbit()    opens the main-window and lets the user select
;*				 an orbit
;* 
;* MODIFICATION HISTORY:       
;*      March 1997, written by Reiner Friedel 
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      papco_selectedOrbits_refresh
;* 
;* DESCRIPTION:  
;*      updates the display using the data found in 'infoStruct'
;* 
;* INPUTS:       
;*      none    All parameters passed in common block papco_selectorbit
;*
;* OUTPUT:	 
;*      none
;*
;* CALLING SEQUENCE:
;*	papco_selectOrbits_refresh
;*
;* MODIFICATION HISTORY:       
;*     March 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_selectOrbits_refresh, ALL=ALL, LIST=LIST, SATNAME=SATNAME, $
                                FROMTO=FROMTO, PL_WAIT=PL_WAIT

  COMMON papco_sel_orbit, SC, selectedOrbit, orbitInfo, orbitTexte, $
                          w_data
  
  if keyword_set(ALL) then begin
    LIST=1
    SATNAME=1
    FROMTO=1
  endif  
  
  if keyword_set(PL_WAIT) then begin
    listheight=30
    ListVal = strarr(listheight)
    ListVal(listheight/2)=  '         please wait....'
    ListVal(listheight/2+1)="         while I'm reading data"    
    WIDGET_CONTROL, w_data.wid_lb_orbitInfo, SET_VALUE=ListVal
  endif  
 
  if keyword_set(FROMTO) then begin
    WIDGET_CONTROL, w_data.wid_ef_from, $
 	SET_VALUE=convert_dayToDate(w_data.minView.day,$
                                    w_data.minView.Year)
    WIDGET_CONTROL, w_data.wid_ef_to, $
        SET_VALUE=convert_dayToDate(w_data.maxView.day, $
                                    w_data.maxView.Year)
  endif
  
  if keyword_set(LIST) then begin
    WIDGET_CONTROL, w_data.wid_lb_orbitInfo, $
      SET_VALUE=orbitTexte(w_data.viewIndexFrom:w_data.viewIndexTo)
  endif
  
  if keyword_set(SATNAME) then begin
    WIDGET_CONTROL,w_data.sat_lbl, $
      SET_VALUE='List of orbit start/end times for: '+SC
  endif   
  
END

;******************************************************************************
;* PROCEDURE:     
;*      papco_selectOrbits_calcBounds, infoStruct, newDay, newYear, $
;*				       NEWMAX, NEWMIN
;* 
;* DESCRIPTION:  
;*      when the user enters a new upper or lower bound for the data to be 
;*	displayed, this procedure is used to calculate the corresponding 
;*	index-range in the list  of orbit-descriptions. Furthermore, the bounds
;*	are checked: the bound won't be set before  the day of the first known 
;*	orbit or after the day of the last known index. The lower bound won't 
;*	be higher than the upper bound.
;* 
;* INPUTS:       
;*      infoStruct	this is a structure containing the data to be shown on 
;*			the screen. It is the same data that is kept in the
;*			first widget of the Window. For its description see
;*			the function 'papco_selectOrbit() at the end of this 
;*			file.
;*      newDay		An integer: the newDay of the new bound.
;*	newYear		An integer: its year. If year < 100 then 1900 is added.
;*
;* KEYWORDS:
;* 	NEWMAX		if set, the upper bound is calculated
;*	NEWMIN		if set, the lower bound is calculated
;*			
;* OUTPUT:	 
;*      -
;*
;* CALLING SEQUENCE:
;*      for displaying data from the 201st day of 1990:
;*	  papco_selectOrbits_calcBounds, infoStruct, 201, 1990, /NEWMIN
;*
;*      for displaying data to the 201st day of 1991:
;*	  papco_selectOrbits_calcBounds, infoStruct, 201, 1991, /NEWMAX
;*
;* MODIFICATION HISTORY:       
;*     March 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_selectOrbits_calcBounds, newDay, newYear, $
                                 NEWMAX=kw_newmax, NEWMIN=kw_newmin

  COMMON papco_sel_orbit, SC, selectedOrbit, orbitInfo, orbitTexte, $
    widgetdata
  
  newMin={day:widgetdata.minView.Day, year:widgetdata.minView.Year}
  newMax={day:widgetdata.maxView.Day, year:widgetdata.maxView.Year}

  orbitAnzahl=N_ELEMENTS(orbitInfo)
; check, if the new day is after the last known orbit
  mDay=orbitinfo(orbitAnzahl-1).day
  mYear=orbitinfo(orbitAnzahl-1).year
  IF (newYear GT mYear) OR (newYear EQ mYEAR AND newDay GT mDay) THEN BEGIN
   newDay=mDay
   newYear=mYear
  ENDIF

; check, if the new Day is earlier than the first known orbit
  mDay=orbitinfo(0).day  &  mYear=orbitinfo(0).year
  IF (newYear LT mYear) OR (newYear EQ mYEAR AND newDay LT mDay) THEN BEGIN
    newDay=mDay  &  newYear=mYear
  ENDIF
     
; now, make sure, that newMin LE newMax
  IF KEYWORD_SET(kw_newmax) THEN BEGIN
    newMax.day=newDay  &   newMax.Year=newYear
    IF (newMin.year GT newMax.Year) OR (newMin.year EQ newMax.year $
           AND newMin.day GT newMax.day) THEN newMin=newMax
  ENDIF ELSE BEGIN
    newMin.day=newDay  & newMin.year=newYear
    IF (newMax.year LT newMin.Year) OR (newMax.year EQ newMin.year $
          AND newMax.day LT newMin.day) THEN newMax=newMin
  ENDELSE

; newMin contains the new first day, and newMax contains the last day  to show.
; seek the indices of these days in orbitInfo and store them in the widgetdata.
  nminIndex=0
  WHILE (orbitinfo(nminIndex).year LT newMin.Year) OR $
        (orbitinfo(nminIndex).year EQ newMin.Year AND $
        orbitinfo(nminIndex).day LT newMin.Day) $
  DO nminIndex=nminIndex+1

  nmaxIndex=orbitAnzahl-1
  WHILE (orbitinfo(nmaxIndex).year GT newMax.Year) OR $
          (orbitinfo(nmaxIndex).year EQ newMax.Year AND $
           orbitinfo(nmaxIndex).day GT newMax.Day) $
  DO nmaxIndex=nmaxIndex-1

  widgetdata.viewIndexFrom=nminIndex
  widgetdata.viewIndexTo=nmaxIndex    
  widgetdata.minView.day=newMin.day
  widgetdata.minView.Year=newMin.year
  widgetdata.maxView.day=newMax.day
  widgetdata.maxView.Year=newMax.Year
  
END


;******************************************************************************
;* PROCEDURE:     
;*      papco_selectOrbits_Event, Event
;* 
;* DESCRIPTION:  
;*      this routine handles Events 
;* 
;* INPUTS:       
;*      Event	 the event to be handled
;* 
;* OUTPUT:	 
;*      -
;*
;* CALLING SEQUENCE:
;*	Called by XMANAGER only !
;*
;* MODIFICATION HISTORY:       
;*     March 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_selectOrbits_Event, Event

  COMMON papco_sel_orbit, SC, selectedOrbit, orbitInfo, orbitTexte, $
                          widgetdata
  WIDGET_CONTROL, Event.Id, GET_UVALUE=uval
  
  CASE uval OF 

    'lb_orbitInfo': BEGIN
       selectedOrbit=orbitInfo(event.index+widgetData.viewIndexFrom)
       IF event.clicks GT 1 THEN $
           CALL_PROCEDURE, widgetData.event_Function, selectedOrbit
                                ; pass the orbit to the caller	
    END

    'ef_from': BEGIN
       newVal=event.value
       conv=convert_dateToDay(newVal) 
       IF conv.error NE 0 THEN BEGIN
	  dummy=MessageBox(['Enter a date in one of these formats:', $
                            'day.month.year', $
			    'day/month/year', $
                            'where month may be a number or a'+ $
                            'month-name of length 3',$
			    '', $
                            'e.g.: 1.5.1995,1/5/1995,1.may.1995,1/may/1995'], $
			    ['Ok'], /Center, Title='Date format error')
       ENDIF ELSE BEGIN
          papco_selectOrbits_calcBounds, conv.day, conv.year, /NEWMIN
	  papco_selectOrbits_refresh, /LIST, /FROMTO
       ENDELSE
     END

    'ef_to': BEGIN
      newVal=event.value
      conv=convert_dateToDay(newVal) 
      IF conv.error NE 0 THEN BEGIN
	dummy=MessageBox(['Enter a date in one of these formats:', $
                          'day.month.year', $
			     'day/month/year', $
                             'where month may be a number or a'+ $
                             'month-name of length 3',$
			     '', $
                             'e.g.: 1.5.1995,1/5/1995,1.may.1995,1/may/1995'],$
			     ['Ok'], /Center, Title='Date format error')
      ENDIF ELSE BEGIN
        papco_selectOrbits_calcBounds, conv.day, conv.year, /NEWMAX
        papco_selectOrbits_refresh, /LIST, /FROMTO
      ENDELSE
    END

    'pb_reset': BEGIN
        anzahl=N_ELEMENTS(orbitInfo)
        widgetData.maxView.day=orbitInfo(anzahl-1).day
	widgetData.maxView.year=orbitInfo(anzahl-1).year
	widgetData.viewIndexFrom=0
	widgetData.viewIndexTo=anzahl-1
	papco_selectOrbits_refresh, /LIST, /FROMTO
    END

    'pb_refresh': BEGIN  
        IF MessageBox(['The orbit-infos will be re-read. For this,', $
                       'each orbit is read. Afterwards, the orbit-', $
                       'infos are written to $'+SC+'_EPH/'+SC+'_orbits.info', $
		       '', $
                       '***** THIS STEP MAY TAKE A LONG TIME *****', $
                       '', $
                       'During execution, you may interrupt using',$
                       "<CTRL-C>. Your old orbits.info won't be damaged by", $
                       'interrupting.',$
		       '', $
                       'Output goes to the IDL-Transcript while refreshing'],$
                      ['Refresh', 'Help', 'Cancel'], $
                       /center, title='Refresh orbit-infos') EQ 0 $
        THEN BEGIN
	    IF MessageBox(['You have been warned'], ['Refresh', 'Cancel'], $
		       /center, title='Refresh orbit-infos') EQ 0 THEN BEGIN
	       WIDGET_CONTROL, Event.top, HOURGLASS=1
	       CALL_PROCEDURE, SC+'_createOrbitInfo'
	       WIDGET_CONTROL, Event.top, HOURGLASS=0	
            ENDIF
	ENDIF
    END

    'pb_choose' : BEGIN ; pass the orbit to the caller of selectOrbit(..)
	CALL_PROCEDURE, widgetData.event_Function, selectedOrbit 
    END
  
    'pb_cancel': BEGIN
       WIDGET_CONTROL, Event.top, /DESTROY
       RETURN
    END

    'pb_help' : BEGIN
    END

    ELSE: Message,'Unknown button pressed', /cont
  ENDCASE

END

;******************************************************************************
;* FUNCTION:     
;*      papco_selectOrbit, group
;* 
;* DESCRIPTION:  
;*	Display a window containing a list of orbit descriptions. When the user
;* 	selects one, its orbitInfo is returned.
;*
;* INPUTS:       
;*	none
;* 
;* KEYWORDS:
;*      group	the widget-group of the window
;* 
;* OUTPUT:	 
;*      if the user exits using the 'CHOOSE'-button:
;*         a POLAR_ORBIT_INFO-structure. Its fields are explained in the file 
;*	   'papco_orbitinfo.pro'
;*	if the user exits using the 'CANCEL'-button:
;*	   -1 
;*
;* CALLING SEQUENCE:
;*	orbInfo=papco_selectOrbit()
;*
;* MODIFICATION HISTORY:       
;*     March 1997, written by Reiner Friedel
;******************************************************************************
PRO papco_selectOrbit, SCraft, Event_Function, GROUP=Group

  COMMON papco_sel_orbit, SC, selectedOrbit, orbitInfo, orbitTexte, $
                          widgetdata
  COMMON papco_orbit_info, readOrbits, orbitDescriptions, satellite_name
    
  ; check if new orbitdata needs to be read - change of satellite
  if not keyword_set(satellite_name) then satellite_name=''
  if SCraft ne satellite_name then begin
    read_data=1 
    satellite_name=SCraft 
  endif else read_data=0
  SC=satellite_name
  
  ; check if any orbitdata has been read - first time around
  if n_elements(orbitDescriptions) eq 0 then read_data=1

  ;prevents multiple select-orbit-windows and force existing windows to front
  ;and check if a re-read is required (change of satellite name) 
  IF XREGISTERED('papco_selectOrbits') THEN BEGIN
    ; check if new orbitdata needs to be read
    if read_data then begin
      WIDGET_CONTROL, widgetdata.base, HOURGLASS=1
      papco_selectOrbits_refresh, /PL_WAIT
      result=papco_readOrbitInfo(SCraft, orbitInfo)
      IF result NE 'OK' THEN BEGIN
        dummy=messagebox(["I can't read the list of orbit-descriptions :(", $
                         'This error occured, while I tried to read them:', $
		         '', result], ['What a pity :('], /center, $
		         title='Error reading descriptions')
        WIDGET_CONTROL, base, HOURGLASS=0
        return
      ENDIF  
      orbitTexte = papco_orbitDescriptionList(orbitInfo, /CACHE_THIS)  
      n=N_ELEMENTS(orbitInfo)
      widgetdata.minView.day=orbitInfo(0).day
      widgetdata.minView.year=orbitInfo(0).year
      widgetdata.maxView.day=orbitInfo(n-1).day
      widgetdata.maxView.year=orbitInfo(n-1).year
      widgetdata.viewIndexFrom=0
      widgetdata.viewIndexTo=n-1
      papco_selectOrbits_refresh, /LIST, /SATNAME, /FROMTO
      WIDGET_CONTROL, widgetdata.base, HOURGLASS=0
      return
    endif else return  
  ENDIF  
                            
  IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0

  junk   = { CW_PDMENU_S, flags:0, name:'' }

  base = WIDGET_BASE(GROUP_LEADER=Group, COLUMN=1, MAP=1, $
                     UVALUE='selectOrbits base', $
                     TITLE='Select an orbit')
  
; --- create a label for the satellite name -----------------------------------
  sat_lbl=widget_label(base, /frame, $
                       value='List of orbit start/end times for: '+Scraft)
  header_lbl=widget_label(base, /align_left, $
    value='ORBIT    DATE     (DOY/YEAR)     START      END     ')
  
; --- create a listbox --------------------------------------------------------
  listheight=30
  ListVal = strarr(listheight)
  ListVal(listheight/2)=  '         please wait....'
  ListVal(listheight/2+1)="         while I'm reading data"
  lb_orbitInfo = WIDGET_LIST(base,VALUE=ListVal, $
                             UVALUE='lb_orbitInfo', XSIZE=50, YSIZE=listheight)

; --- create fields for controlling the range of displayed orbits -------------
  base2= WIDGET_BASE(base, /row, /frame)
  base3= WIDGET_BASE(base2, /column)
  base4= WIDGET_BASE(base2, /column)
  ef_from = CW_FIELD(base3,VALUE=[''], ROW=1, STRING=1, $
                     RETURN_EVENTS=1, $
                     TITLE='show from date: ', $
                     UVALUE='ef_from')

  ef_to   = CW_FIELD(base3,VALUE=[''], ROW=1, STRING=1, $
                     RETURN_EVENTS=1, $
                     TITLE='       to date: ', $
                     UVALUE='ef_to')

  pb_reset=WIDGET_BUTTON(base4, /ALIGN_CENTER, $
                         VALUE=strcenter('All Orbits', 12), $
                         UVALUE='pb_reset' )
  pb_reread=WIDGET_BUTTON(base4, /ALIGN_CENTER, $
                          VALUE=strcenter('Refresh', 12), $
                          UVALUE='pb_refresh')

; create the buttons at the bottom of the window.
  base6= WIDGET_BASE(base, /row, /frame)
  Buttons = ['Choose', 'Help', 'Cancel' ]
  ButtonNames = ['pb_choose', 'pb_help', 'pb_cancel']
  FOR i=0, N_ELEMENTS(Buttons)-1 DO BEGIN
     IF i GT 0 THEN $
        dummy=WIDGET_LABEL(base6, VALUE='      ')

     dummy=WIDGET_BUTTON(base6, $
              VALUE=strcenter(Buttons(i),12), $
              UVALUE=ButtonNames(i))
  ENDFOR

  WIDGET_CONTROL, base, /REALIZE

; read the orbit-infos.
  if read_data then begin
    WIDGET_CONTROL, base, HOURGLASS=1
    result=papco_readOrbitInfo(SCraft, orbitInfo)
    IF result NE 'OK' THEN BEGIN
      dummy=messagebox(["I can't read the list of orbit-descriptions :(", $
                        'This error occured, while I tried to read them:', $
                        '', result], ['What a pity :('], /center, $	
                        title='Error reading descriptions')
      WIDGET_CONTROL, base, /DESTROY
      return
    Endif
  endif 
  orbitTexte = papco_orbitDescriptionList(orbitInfo, /CACHE_THIS)  
  anzahl=N_ELEMENTS(orbitInfo)

; the event-handler must access some of the read variables. In order to do 
; this, a structure containing the needes variables is created and stored in
; the UVALUE of the first widget of the window. The first widget is $
; 'valueKeeper', a WIDGET_BASE without childs.
  widgetdata={base:base, $
              sat_lbl:sat_lbl, $
              minView:{day:orbitInfo(0).day, year:orbitInfo(0).year}, $
              maxView:{day:orbitInfo(anzahl-1).day, $
                       year:orbitInfo(anzahl-1).year}, $
              event_Function:event_Function, $
              viewIndexFrom:0, $
	      viewIndexTo:anzahl-1, $
              wid_base:base,$
	      wid_lb_orbitInfo:lb_orbitInfo, $
              wid_ef_From:ef_from, $
              wid_ef_To:ef_to }

  papco_selectOrbits_refresh, /ALL

  WIDGET_CONTROL, base, HOURGLASS=0
	
  selectedOrbit=papco_getOrbitInfoStruct()

  XMANAGER, 'papco_selectOrbits', base

END

;******************************************************************************
;* FUNCTION:     
;*      papco_getOrbitList
;* 
;* DESCRIPTION:  
;*	Given the times set by papco, and the satellite mission, this function
;* 	uses the orbit times file to return a list of orbits spanning the
;*      required time interval.
;*
;* INPUTS:       
;*	none
;* 
;* KEYWORDS:
;*      group	the widget-group of the window
;* 
;* OUTPUT:	 
;*      if the user exits using the 'CHOOSE'-button:
;*         a POLAR_ORBIT_INFO-structure. Its fields are explained in the file 
;*	   'papco_orbitinfo.pro'
;*	if the user exits using the 'CANCEL'-button:
;*	   -1 
;*
;* CALLING SEQUENCE:
;*	orbInfo=papco_selectOrbit()
;*
;* MODIFICATION HISTORY:       
;*     March 1997, written by Reiner Friedel
;******************************************************************************
FUNCTION papco_getOrbitList, SCraft
  
  COMMON mjdt, mjdt_start, mjdt_end
  
  orbit_list=[0]
  old=[mjdt_start, mjdt_end]
  
  set_start=double(mjdt_start.mjd)*86400.0d +double(mjdt_start.t)
  set_end  =double(mjdt_end.mjd)*86400.0d  +double(mjdt_end.t)
  
  result=papco_readOrbitInfo(SCraft, orbitInfo)
  if result ne 'OK' then goto, out
  
  ;search though orbit infos for orbits covering timerange
  for i=0, n_elements(orbitInfo)-1 do begin
    result=papco_setorbit(orbitInfo(i).NUMBER, SATELLITE=SCraft, /NOWIDGET)
    this_start=double(mjdt_start.mjd)*86400.0d +double(mjdt_start.t)
    this_end=double(mjdt_end.mjd)*86400.0d  +double(mjdt_end.t)
    if this_end lt set_start then goto, next
    if this_start gt set_end then goto, out
    orbit_list=[orbit_list,orbitInfo(i).NUMBER]
    print,orbitInfo(i).NUMBER,set_start,this_start,set_end,this_end, $
      format="(i4.4,4(f16.1))"
    next:
  endfor  
   
  out:
  mjdt_start=old(0)  &  mjdt_end=old(1)  
  n=n_elements(orbit_list)
  if n gt 1 then orbit_list=orbit_list(1:n-1)
  
  return,orbit_list
  
end  
