;******************************************************************************
;******************************************************************************
;* FILE:
;*    $PAPCO_PATH/papco_output.pro
;*
;*    This file belongs to the core of the PAPCO project. 
;*
;* DESCRIPTION:
;*    This file provides generic routines for PAPCO modules to make their
;*    data avauilable. This data can either be made available in memory
;*    or in a variety of output file formats.   
;*    These routines get called by the user-written data plot routine. 
;*         
;*    Those procedures are contained:
;*
;*      PROCEDURE papco_write_data, time, data
;*	     -> writes the calling modules' panel plot data to file.
;*
;*       FUNCTION papco_unformat_idlstring, idlstr
;*	     -> Removes any "!" format strings from a given string
;*
;*
;* MODIFICATION HISTORY:
;*    January 1998, written by R. Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION
;*      papco_unformat_idlstring
;*
;* DESCRIPTION:
;*	Removes any "!" format strings from a given string and
;*      converts it to readable straight text only. 
;*
;* INPUTS:
;*      idlstr          the idl formatted string
;*
;* OUTPUTS:
;*      textstr         the unformatted text only string
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*	called from papco_write_data
;*      textstr=papco_unformat_idlstring(idlstr)
;*
;* MODIFICATION HISTORY:
;*     written May 1998, Reiner Friedel
;******************************************************************************
function papco_unformat_idlstring, idlstr
  
  textstr=idlstr    
  pos=strpos(idlstr,'!')
  while pos ne -1 do begin
    if strupcase(strmid(textstr,pos+1,1)) eq 'C' then $
      textstr=strmid(textstr,0,pos)+' '+strmid(textstr,pos+2,strlen(textstr)) $
    else $
      textstr=strmid(textstr,0,pos)+strmid(textstr,pos+2,strlen(textstr))
    pos=strpos(textstr,'!')
  endwhile  
  return,textstr
  
end

;******************************************************************************
;* PROCEDURE
;*      papco_write_data, time, data
;*
;* DESCRIPTION:
;*	writes the calling modules' panel plot data to file. The
;*	filename is constructed using the time range of the plot and the
;*	product name if set. 
;*
;* INPUTS:
;*      time            the time array of the data (dimension n)
;*	data            the data array (dimension [n*ch] or [ch])
;*                        where n is the number of points of the time array,
;*                        and ch is the number of data channels. 
;*
;* OUTPUTS:
;*      none
;*
;* KEYWORDS:
;* 	DESCRIPTION   a string or string array describing the data
;*      TIME_LABEL    a string decribing the x-axis format if it is time
;*                    if empty or unknown format, use time as is
;*                    known formats:  'TAI'   -   int. atomic time
;*                                    'DMJD'  -   dec. mod. Julian day
;*                                    'DY'    -   decimal year
;*                                    'T90'   -   sec since 1/1/1990 
;*                                    'SSD'   -   sec from start day
;*      SLICE         a string containg the time of the data for slice
;*      CHANNELS      an (ch,2) array giving the channel start/end
;*      X_LABEL       a string with the x-axis label
;*      Y_LABEL       a string with the y-axis label
;*      Z_LABEL       a string with the z-axis label
;*      RS_LABEL      a string with the righ-side label of the plot
;*      NODATA        contains the no data flag value.
;*      ONECOUNT      an (ch) array giving the one count level
;*      FORMAT        a string for the file output format:
;*                       'TXT'   - ASCII flat text file
;*                       'CDF'   - common data format (binary)
;*                       'IDL'   - IDL save/restore set (binary)
;*                       'MEM'   - put data in common block for
;*                                 scratch pad
;*                       'PUM'   - PAPCO utility module / S. Bourdarie
;*      FILE          if setm use this string as filename. 
;*
;* CALLING SEQUENCE:
;*	called from the user modules' plot routine. 
;*
;* MODIFICATION HISTORY:
;*     written May 1998, Reiner Friedel
;******************************************************************************
PRO papco_write_data, time, data, $
                      DESCRIPTION=DESCRIPTION, TIME_LABEL=TIME_LABEL, $
                      CHANNELS=CHANNELS, TRANGE=TRANGE, $
                      X_LABEL=X_LABEL, Y_LABEL=Y_LABEL, Z_LABEL=Z_LABEL, $
                      RS_LABEL=RS_LABEL, NODATA=NODATA, FORMAT=FORMAT, $
                      FILE=FILE, ONECOUNT=ONECOUNT, SLICE=SLICE
  
COMMON plot_composer, widgetData
COMMON mjdt, mjdt_start, mjdt_end     ;common time limit in mjdt
  
ver=papco_Version()
now=strmid(systime(0),4,21)
currPlotInfo=widgetData.plotInfos(widgetData.current_plotNr)
  
;select correct file formay
IF keyword_set(FORMAT) THEN format=FORMAT ELSE $
    format=widgetData.default_output_format
  
;make a filename based on the time of the plot.
IF format NE 'MEM' THEN BEGIN 
    fname=papco_PrinterDescription(widgetData.printerInfo, /FILE)

    p=strpos(fname,'.')
    if p ne -1 then fname=strmid(fname,0,p)
    if fname eq 'file: <<dialog>>' then fname='papco'
  
    if keyword_set(FILE) then begin
      papco_sepdir, fname, dir, fln
      fname=dir+file
    end  
      
    papco_sepdir,widgetdata.SAVEFILENAME,path,prod_name
    if prod_name eq '' then prod_name='none'
    panelkind=currPlotInfo.PANELKIND
    plotNr=widgetData.current_plotNr
ENDIF 
  
IF keyword_set(SLICE) THEN BEGIN 
    time_stamp_str=SLICE
ENDIF ELSE BEGIN  
    convert_t90_to_Date, widgetData.startTime, s_year, s_day_oy, s_hour, $
                         s_min, s_sec, s_month, s_cmonth, s_day_om, /mjdt
    timeStart=string(s_year,s_month,s_day_om,s_hour,s_min,s_sec, $
                     format='(i4.4,i2.2,i2.2,":",i2.2,":",i2.2,":",i2.2)')
    convert_t90_to_Date, widgetData.endTime, e_year, e_day_oy, e_hour, $
                         e_min, e_sec, e_month, e_cmonth, e_day_om, /mjdt
    timeEnd=string(e_year,e_month,e_day_om,e_hour,e_min,e_sec, $
                   format='(i4.4,i2.2,i2.2,":",i2.2,":",i2.2,":",i2.2)')  
    time_stamp_str='Time period:        '+timeStart+' to '+timeEnd
ENDELSE

CASE format OF 
 
    'TXT': BEGIN
        ext='.txt'
        ;open file for writing. if file exists, append. 
        result=papco_findfile(fname+ext,count=c)
        if plotNr eq 0 then c=0
        if c eq 0 then begin 
            message, 'Opening file ',/CONT
            print, '  ',fname+ext
            openw, u, fname+ext, /get_lun
            printf,u,'Data output from:   '+ver
            printf,u,'File produced:      ',now  
            printf,u,'PAPCO product file: ',prod_name
            printf,u,time_stamp_str
            printf,u,''
        endif else begin
            message, 'Appending file ',/CONT
            print, '  ',fname+ext
            openw, u, fname+ext, /get_lun, /append
        endelse  

        r=size(time,/str) 
        ndat = r.N_ELEMENTS/r.N_DIMENSIONS    

        if keyword_set(SLICE) then begin
            ;output is from slice
            printf,u,'Output data from PAPCO slice'
            printf,u,''
        endif else begin  
           ;output headers for the current panel
            printf,u,'Begin data from Panel type: '+panelkind
            printf,u,''
            printf,u,'Number of data intervals: ',ndat
            printf,u,''
        ENDELSE 
      
        ;output optional information from keywords
        if KEYWORD_SET(DESCRIPTION) then begin
            printf,u,'DESCRIPTION: ',DESCRIPTION
            printf,u,''
        ENDIF 
      if KEYWORD_SET(TIME_LABEL) then begin
        printf,u,'TIME_LABEL: ',TIME_LABEL
        printf,u,''
      endif
      if KEYWORD_SET(CHANNELS) then begin
        printf,u,'CHANNELS: '
        res=size(channels) & n=n_elements(res)
        
        if res(n-2) eq 7 then begin ;string array
          for i=0,res(1)-1 do begin
            printf,u, strcenter(varprt(i),10) + channels(i)
          endfor
        endif else begin ;number array
          out_str=strcenter('No.',10)+strcenter('Begin',12)+strcenter('End',12)
          if KEYWORD_SET(ONECOUNT) then $
            out_str=out_str+strcenter('One Count',12)
          printf,u,out_str
          for i=0,res(1)-1 do begin
            out_str=strcenter(varprt(i),10) + $
              strcenter(string(channels(i,0),format="(g10.4)"),12) + $
              strcenter(string(channels(i,1),format="(g10.4)"),12)
            if KEYWORD_SET(ONECOUNT) then out_str=out_str+ $
              strcenter(string(ONECOUNT(i), format="(g10.4)"),12)
            printf,u, out_str
          endfor  
        endelse
        printf,u,''
      endif
      if KEYWORD_SET(X_LABEL) then begin
        printf,u,'X_LABEL  (IDL format): ',X_LABEL
        printf,u,'          (text only): ',papco_unformat_idlstring(X_LABEL)
        printf,u,''
      endif  
      if KEYWORD_SET(Y_LABEL) then begin
        printf,u,'Y_LABEL  (IDL format): ',Y_LABEL
        printf,u,'          (text only): ',papco_unformat_idlstring(Y_LABEL)
        printf,u,''
      endif  
      if KEYWORD_SET(Z_LABEL) then begin
        printf,u,'Z_LABEL  (IDL format): ',Z_LABEL
        printf,u,'          (text only): ',papco_unformat_idlstring(Z_LABEL)
        printf,u,''
      endif
      if KEYWORD_SET(RS_LABEL) then begin
        printf,u,'RS_LABEL (IDL format): ',RS_LABEL
        printf,u,'          (text only): ',papco_unformat_idlstring(RS_LABEL)
        printf,u,''
      endif
      if KEYWORD_SET(NODATA) then begin
        printf,u,'NODATA flag          : ',string(NODATA,format="(e11.4)")
        printf,u,''
      endif
      ;output data for current panel  
      printf,u,'Plot data:'
      printf,u,''
      ;check size of data array is O.K.
      res=size(data)
      if n_elements(res) gt 5 then begin
        message,'dimension of data array > 2 - no output', /cont
        printf,u,'dimension of data array > 2 - no output'
        return
      endif
      if n_elements(res) eq 5 then begin
        sl_data=fltarr(res(2))
      endif
      if n_elements(res) eq 4 then begin
        sl_data=fltarr(1)
      endif
      if n_elements(res) le 3 then begin
        message,'data array not defined - no output', /cont
        printf,u,'data array not defined - no output'
        return
      endif  
      ;write header for columns  
      column_id_str='(x)'
      if keyword_set(X_LABEL) then column_id_str='[(x) '+X_LABEL+']'
      IF keyword_set(TIME_LABEL) THEN BEGIN
          CASE TIME_LABEL OF
              'TAI' : BEGIN
                  column_id_str = 'Verbose Time                 TAI[s]'
                  column_id_str=strFormat(column_id_str,42,/LEFT)
              END
              ELSE: BEGIN
                  IF r.N_DIMENSIONS EQ 1 THEN column_id_str='[(x) time]' $
                  ELSE column_id_str='[(x) tsart - tend]'
                  column_id_str=strFormat(column_id_str,25,/LEFT)
              END    
          ENDCASE
      ENDIF    

      if n_elements(sl_data) eq 1 then begin
        if keyword_set(Y_LABEL) then y_str='[(y) '+Y_LABEL+']' else y_str='(y)'
        y_str=strcenter(y_str,12)
        column_id_str=column_id_str+' '+y_str
      endif  else $  
        for i=0l, n_elements(sl_data)-1 do column_id_str=column_id_str + ' '+$
          strcenter('Ch '+strtrim(string(i,format="(i4)"),2),12)
      printf,u,column_id_str
      ;write out data
      for i=0l,ndat-1 do begin
        sl_data(*)=data(i,*)
        ;make time string - if a known format, convert to verbose
        ;and also original time format
        case TIME_LABEL of
            'TAI' : BEGIN
                IF r.N_DIMENSIONS EQ 1 THEN BEGIN
                    verb_time_str=strFormat(tai2utc(time(i),/ECS),25,/LEFT)
                    ;print,  tai2utc(time(i),/ECS)
                    time_str = string(time(i), format = "(f18.3)")
                ENDIF ELSE BEGIN
                    tai = (time(i, 0)+time(i, 0))/2.0
                    verb_time_str=strFormat(tai2utc(tai,/ECS),25,/LEFT)
                    time_str = string(tai, format = "(f18.3)")
                ENDELSE     
                data_str = verb_time_str +time_str
            END    
            ELSE : BEGIN
                IF r.N_DIMENSIONS EQ 1 THEN $
                  data_str=strcenter(varprt(time(i)),25) $
                ELSE data_str = $
                  strcenter(varprt(time(i, 0))+' - '+varprt(time(i, 1)),25)
            END     
        ENDCASE 
        for j=0l,n_elements(sl_data)-1 do data_str=data_str + ' ' + $
          strcenter(strtrim(string(sl_data(j),format="(g10.4)"),2),12)
        printf,u,data_str
      endfor
      if keyword_set(SLICE) then begin
        ;output is from slice
        printf,u,''
        printf,u,'End data from PAPCO slice'
        printf,u,''
      endif else begin   
        printf,u,''
        printf,u,'End data from Panel type:  '+panelkind
        printf,u,''
      endelse  
      close,u & free_lun,u
    END                         ;case TXT
    
    'IDL': BEGIN
      ext='.idl'
      ; Insert '_x' at end of file name, where x is the file number
      ; add the current panelkind to the filename too
      fname=fname+'_'+panelkind+string(plotNr, format="('_',i2.2)")
      
      message, 'Writing file ',/CONT
      print, '  ',fname+ext
      ;make header information structure from keywords if set
      if KEYWORD_SET(CHANNELS) then res1=size(channels) else res1=[0,1]
      
      input_header={DESCRIPTION:'', PANELKIND:currPlotInfo.PANELKIND, $
                    TIME_LABEL:'', $
                    CHANNELS:fltarr(res1(1),2), $
                    ONECOUNT:fltarr(res1(1)), $
                    Y_LABEL:'', Z_LABEL:'', $
                    RS_LABEL:'', NODATA:0.0}
      if KEYWORD_SET(DESCRIPTION) then input_header.DESCRIPTION=DESCRIPTION
      if KEYWORD_SET(TIME_LABEL) then input_header.TIME_LABEL=TIME_LABEL
      if KEYWORD_SET(CHANNELS) then input_header.CHANNELS=CHANNELS
      if KEYWORD_SET(ONECOUNT) then input_header.ONECOUNT=ONECOUNT
      if KEYWORD_SET(LABEL) then input_header.LABEL=LABEL
      if KEYWORD_SET(Z_LABEL) then input_header.Z_LABEL=Z_LABEL
      if KEYWORD_SET(Y_LABEL) then input_header.Y_LABEL=Y_LABEL
      if KEYWORD_SET(RS_LABEL) then input_header.RS_LABEL=RS_LABEL
      if KEYWORD_SET(NODATA) then input_header.NODATA=NODATA
      ;make data structure. Standard form: array of time ordered structures.
      ndata = n_elements(time(*, 0))
      dummy = {time:transpose(time(0, *)), data:transpose(data(0, *))}
      input_data = replicate(dummy, ndata)
      res = size(time)
      IF res(0) EQ 1 THEN input_data.time = transpose(time) $
        ELSE input_data.time = transpose(time)
      FOR i = 0l, ndata-1 DO input_data(i).data = transpose(data(i,*))
      ;save all the info
      save, filename=fname+ext, input_header, input_data, /verb, /compress
    END  

    'MEM' : BEGIN  ;store data in memory structure array for scratch pad.
      common scratch_data, scratch
      ;need to put all data onto same time scale - this can be set by user
      common scratch_pad_panel_data, panelData
      ;t_res=panelData.t_res
      t_res=3600; set to one hour for omni test for now!
      
      ;cannot assume we know time format of the time array passed in here.
      ;but we know start and end as set by papco, AND start /end set by the 
      ;plot routine's calling routine...

      ;make time array in TAI format. We already know papco start /end times..
      ;then convert data time to TAI in resolution required...
      tai_s=utc2tai({YEAR:s_year, MONTH:s_month, DAY:s_day_om, HOUR:s_hour, $
                      MINUTE:s_min, SECOND:s_sec, MILLISECOND:0.0})
      tai_e=utc2tai({YEAR:e_year, MONTH:e_month, DAY:e_day_om, HOUR:e_hour, $
                      MINUTE:e_min, SECOND:e_sec, MILLISECOND:0.0})

    END
    
    'PUM' : BEGIN
      ;convert to PUM time format if needed: decimal year!
      result=str_sep(time_label,' ')
      ;get the right case for hydra moments (others?) -JTN-
      if result(0) EQ 'Seconds' THEN result(0)='SSD'

      case  result(0) of
          'TAI': BEGIN
              t1 = tai2utc(time(0), /external)
              t2 = tai2utc(time(n_elements(time)-1), /external)
              IF t1.year EQ t2.year THEN BEGIN
                  message, 'All in one year', /cont
                  st_sec = UTC2TAI({year:t1.year, MONTH:1, DAY:1, $
                                    HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
                  en_sec = UTC2TAI({year:t1.year+1, MONTH:1, DAY:1, $
                                    HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
                  year_sec = en_sec-st_sec
                  time = t1.year+(time - st_sec) / year_sec
              ENDIF ELSE BEGIN 
                FOR i = 0l, n_elements(time)-1 DO BEGIN
                  ;get the year -> seconds in that year
                  res = tai2utc(time(i), /external)                
                  st_sec = UTC2TAI({year:res.year, MONTH:1, DAY:1, $
                                    HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
                  en_sec = UTC2TAI({year:res.year+1, MONTH:1, DAY:1, $
                                    HOUR:0, MINUTE:0, SECOND:0, MILLISECOND:0})
                  year_sec = en_sec-st_sec
                  time(i) = res.year+(time(i) - st_sec) / year_sec
                ENDFOR 
              ENDELSE 
          end
          'DMJD': begin
          end
          'DY': begin     ; this is the one we want - do nothing
          end
          'T90' : begin
          end  
          'SSD' : begin   ;seconds since the start of the start day of plot
           ;get start date from mjdt_start
           convert_t90_to_Date, mjdt_start,year,day_oy, hour, minute, second, $
             month, cmonth, day_om, /MJDT
           ;find # of days in this year
           dayst=convert_date_to_t90(day=31, month=12, year=year-1,/MJDT)
           dayen=convert_date_to_t90(day=31, month=12, year=year,/MJDT)
           ydays=dayen.mjd-dayst.mjd
           ;get startday in seconds since beginning of year
           ;First day of year starts at zero seconds -JTN-
           start_sec=(day_oy-1)*86400.0d
           ;time array in seconds since beginning of year
           new_time=time+start_sec
          ;calculate fraction of year
           new_time=new_time / (ydays *86400.0d)
           ;make fractional year
           new_time=double(year)+new_time
           time=new_time
          end  
          else: begin
          end
      endcase
      
      ext='.map'
      ;open file for writing. if file exists, append. 
      result=papco_findfile(fname+ext,count=c)
      if plotNr eq 0 then c=0
      if c eq 0 then begin 
        message, 'Opening file ',/CONT
        print, '  ',fname+ext
        openw, unit, fname+ext, /get_lun
      endif else begin
        message, 'Appending file ',/CONT
        print, '  ',fname+ext
        openw, unit, fname+ext, /get_lun, /append
      endelse 
  
      ;output optional information from keywords
      if KEYWORD_SET(RS_LABEL) then begin
        printf,unit,papco_unformat_idlstring(RS_LABEL)
      endif else printf,unit,' '
      printf,unit,' '
      printf,unit,' '
      printf,unit,' '
      printf,unit,' '
      if KEYWORD_SET(Z_LABEL) then begin
        printf,unit,Z_LABEL
      endif else printf,unit,' '
      if KEYWORD_SET(TIME_LABEL) then begin
        printf,unit,TIME_LABEL
      endif else printf,unit,' '
      if KEYWORD_SET(Y_LABEL) then begin
        printf,unit,Y_LABEL
      endif else printf,unit,' '

      if KEYWORD_SET(CHANNELS) then begin
         res=size(channels)
         printf,unit,n_elements(time),res(1)+1
         if KEYWORD_SET(NODATA) then begin
            printf,unit,'No data flag : ',nodata
         endif else printf,unit,' '
         for i=0,res(1)-1 do printf,unit,channels(i,0)
         printf,unit,channels(res(1)-1,1)
      ENDIF

      ;Uncommented and corrected (add 'end' after last y) -JTN-
      if KEYWORD_SET(DESCRIPTION) then begin
         res=n_elements(DESCRIPTION)
         printf,unit,n_elements(time),res+1
         if KEYWORD_SET(NODATA) then begin
            printf,unit,'No data flag : ',nodata
         endif else printf,unit,' '
         for i=0,res-1 do printf,unit,description(i)
         printf,unit,'end'
      endif

      ;output data for current panel  
      ;check size of data array is O.K.
      res=size(data)
      if n_elements(res) gt 5 then begin
        message, 'dimension of data array > 2 - no output', /cont
        printf,unit,'dimension of data array > 2 - no output'
        return
      endif
      if n_elements(res) eq 5 then begin
        slice=fltarr(res(2))
      endif
      if n_elements(res) eq 4 then begin
        slice=fltarr(1)
      endif
      if n_elements(res) le 3 then begin
        print,'% papco_write_data: data array not defined - no output'
        printf,unit,'data array not defined - no output'
        return
      endif  
      ;write out data
      for i=0l,n_elements(time)-1l do begin
        slice(*)=data(i,*)
        data_str=string(time(i),format='(f13.8)')
        for j=0l,n_elements(slice)-1 do data_str=data_str + $
          string(slice(j),format="('  ',e11.4)")
        printf,unit,data_str
      endfor
      close,unit & free_lun,unit      
    END
    
    
    ELSE : BEGIN
      message,'Output format '+format+' not yet supported', /cont
      return
    END                         ;case else
    
  ENDCASE
    
END

;******************************************************************************
;* Procedure:
;*      r_papco_output
;*
;* DESCRIPTION:
;*      reads files produced by the PAPCO output function
;*
;* INPUTS:
;*      filename       full path of the papco output file
;*       
;* KEYWORDS:
;*      FILE_FORMAT    a string of either 'ASCII', 'IDL', 'CDF'
;*                     default is 'IDL', 'CDF' is not implemented yet
;*      VERBOSE        if set, be verbose. 
;*
;* OUTPUT:
;*      none
;*      data read is returned in a common block papco_output
;*
;* CALLING SEQUENCE:
;*      r_papco_output, fln
;*
;* MODIFICATION HISTORY:
;*      written May 1999, R. Friedel
;******************************************************************************
PRO r_papco_output, filename, FILE_FORMAT=FILE_FORMAT, VERBOSE=VERBOSE
  
COMMON papco_output, input_header, input_data
COMMON get_error, get_err_no, get_err_msg
  
;data read is in structures input_header, input_data. If the file contains
;more than one data type, and array of input_header, input_data structures
;is returned.
  
valid_formats=['TXT','IDL','CDF']
if keyword_set(FILE_FORMAT) then file_format=strupcase(FILE_FORMAT) $
else file_format='IDL'
index=where(valid_formats eq file_format,c)
if c eq 0 then begin
    message,file_format +' is not a valid file format', /CONT
    return
endif
  
if keyword_set(VERBOSE) then verbose=VERBOSE else verbose=0
  
case file_format of
    'TXT': begin
    end
    'IDL': begin
      message, filename, /CONT
      ON_IOERROR, report_error
      restore, filename, VERBOSE=VERBOSE
    end
    'CDF': begin
    end
ENDCASE
 
return

report_error:
get_err_no=1
get_err_msg='!C '+!ERR_STRING
message, get_err_msg, /cont
 
END
