;******************************************************************************
;******************************************************************************
;* FILE:
;*    $PAPCO_PATH/papco_panel.pro
;*
;*    This file belongs to the papco-project. See papco.pro for further
;*    information.
;*
;* DESCRIPTION:
;*    This file contains the default behaviour of panelplot-editors.
;*    Provided are widgets for selecting the panelplot height, for selecting
;*    its Y-Range and Z-Range, for selecting the plot-color and
;*    plot-symbols and color bars. The action-buttons
;*    'help', 'config', 'cancel', 'done' are provided as well.
;*    This file furthermore contains the event-handling-routines for
;*    those widgets.
;*
;*    Imagine this file as the default panelplot-editor and the other
;*    panelplot-editors as extensions of this one. If you want to add a kind
;*    of panelplot, your widgets should use this file's procedures.
;*
;* HINT:
;*    In any case : read the file $papco_modules/plot_type_panel.pro
;*                  this is a well-documented example on how to use
;*                  papco_panel.pro. If you want to build your own editor,
;*                  copy that file to another name and edit it.
;*                  Without having read $papco_MODULES/plot_type_panel.pro
;*                  it will be hard for you to understand this file.
;*
;* FILES YOU MAY WANT TO READ
;*    papco_doc/papco_conventions.text  which conventions were used
;*			                when developing ?
;*    papco_doc/papco_variables.text	what structures and global
;*					variables are used by
;*					which file ?
;*
;* USED MODULES:
;*    For a list of files, that are needed for papco to run, read the
;*    file papco_init.pro
;*
;* COMMON-BLOCKS:
;*    No Common-Block is created
;*
;* STRUCTURES:
;*    Though this module creates no concrete structure, it makes use of fields
;*    of other structures. The routines for adding widgets return structures
;*    which contain all fields needed by them. Your calling program has to
;*    collect those fields (by using the IDL-command 'CREATE_STRUCT') in
;*    another structure.
;*
;*    Though no concrete structure is defined, I like to think of those fields
;*    as defining a virtual structure, that the other editors have to expand
;*    (or, if you like OOP, that they have to inherit). I will call this
;*    virtual structure will papco_panel_EDITOR, and it might be
;*    (Keep in mind, that you will nowhere find this structure - its just a
;*    virtual structure !)
;*
;*       {papco_panel_editor,
;*	      panelHeight:1, $
;*	      panelKind:'some string', $
;*	      helpfile:'a_helpfile_name.txt', $
;*	      manualYScaling:0, $
;*	      yScaleMin:0.0, $
;*	      yScaleMax:0.0, $
;*	      ef_yscaleFrom:aWidget, $
;*	      ef_yscaleTo:aWidget, $
;*	      canceled:1}
;*
;*    If your editor contains a plotstyle-selection, those fields will be
;*    additionally contained:
;*
;*      {pb_selectSymbol:aWidget, $
;*            pb_selectColor:aWidget, $
;*            ps_symbol_base:aWidget, $
;*            ps_color_base:aWidget, $
;*            sl_plotStyle:aWidget, $
;*            dw_plotColor:aWidget, $
;*            dw_plotStyle_preview:aWidget, $
;*            color:1, $
;*            plotStyle:1, $
;*            userstyle:0, $
;*            is_scatterPlot:0, $ ; EQ 0 <=> not a scatter-plot
;*            ps_numberOfColors:16, $
;*            ps_selectColor:0, $ ; EQ 0 <=> the color-selector is visible
;*            ps_isScatterPlot:0}
;*
;*    All the functions in $papco_PATH/papco.pro that edit plots or add new
;*    ones, expect the opened editor-windows to return a structure containing
;*    at last some of those fields. Those fields might make up a structure,
;*    but as no program defines the structure-type papco_panel_EDITOR_RETURN,
;*    this poor structure too is not alive but only virtual.
;*
;*    The return-values of plot-editors expand this structure (I like to
;*    think, that they inherit from that structure). They do this by creating
;*    a structure of their special tag-names and expand this by using the
;*    papco_panel_returnData-function provided in papco_panel.pro
;*
;*   	 {papco_panel_EDITOR_RETURN, $
;*	      canceled:1, $
;*	      panelHeight:1, $
;*	      channel:0, $
;*	      typeVector:[0,0,0,0], $
;*	      manualYScaling:0 $
;*	      yScaleMin:1., $
;*	      yScaleMax:1., $
;*	      panelKind:'some string'}
;*
;*    If your editor contains a plotstyle-selection, those fields will be
;*    additionally contained:
;*
;*       {plotStyle:widgetData.plotStyle, $
;*            color:widgetData.color})
;*
;*    For a more complete description see $papco_PATH/papco_variables.text
;*
;* OVERVIEW:
;*    For a better overview, read the file
;*       $papco_MODULE/plot_type_panel.pro
;*
;*    When you create the widgets of a panelplot, you can use the
;*    procedures provided in this file to add default-widgets to your base:
;*
;*       FUNCTION get_plotStyle_value, plotinfo
;*           -> get slider value plot symbol widget
;*
;*       PRO get_user_symbol, userstyle, userx, usery
;*           -> define user symbol
;*
;*       PRO papco_ColorThickness_Coder, widgetData, color, thickness,
;*           -> encode/decode thickness and color
;*
;*       PRO papco_Style_Coder, widgetData, plotstyle, linestyle,
;*           -> encode/decode symbol and linestyle
;*
;*       FUNCTION papco_panel_AddHeightWidgets, aBase, plotinfo
;*	     -> create widgets for selecting plot-height
;*
;*       FUNCTION papco_panel_returnData, widgetData
;*           -> collect the data to be returned
;*
;*       FUNCTION papco_panel_add_PlotStyleButtons, aBase, plotinfo, isScatter
;*	     -> add plot-style buttons
;*
;*       FUNCTION papco_panel_Is_ScatterPlot, widgetData, pType
;*	     -> define, it this panel edits a scatter-plot
;*
;*       FUNCTION papco_panel_AddActionButtons, aBase, action, plotinfo
;*	     -> add action-buttons
;*
;*       FUNCTION papco_panel_CreateWidgetData, struct1, struct2
;*	     -> merge 'struct1' and 'struct2' to widgetData
;*
;*       PRO papco_panel_Realize, aBase, widgetData
;*	     -> realize the window (show it)
;*
;*       FUNCTION papco_panel_AddYScale_Widgets, aBase, plotinfo
;*	     -> add yscale - widgets
;;*
;*       FUNCTION papco_panel_AddZScale_Widgets, aBase, plotinfo
;*	     -> add zscale - widgets
;*
;*       PRO papco_ZScale_Refresh, event, widgetData, BASE=BASE
;*	     -> update display of zscale-widgets
;*
;*       FUNCTION papco_panel_Addcolorbar_Widgets, aBase, plotinfo
;*	     -> add colorbar -widget
;*
;*       PRO papco_Addcolorbar_Refresh, event, widgetData, BASE=BASE
;*	     -> update display of Addcolorbar-widgets
;*
;*       PRO papco_panel_Refresh, widgetData, $
;*                         ALL=ALL, $
;*                         SELECT_PLOT_STYLE=SELECT_PLOT_STYLE, $
;*                         SCATTER_TYPE=SCATTER_TYPE, PLOT_STYLE=PLOT_STYLE
;*	     -> update display of window
;*
;*       FUNCTION papco_panel_Event, event, widgetData
;*           -> interpret events generated by those widgets
;*
;* MODIFICATION HISTORY:
;*     august 1995, written by A.Keese
;*     march 1996, Haje Korth
;*     october 1997, Reiner Friedel
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION get_plotStyle_value, plotinfo
;*
;* DESCRIPTION:
;*	Called by function papco_panel_add_PlotStyleButtons. This function
;*	sets the correct value for the plot style slider widget. it is
;*	neccessary because the all user defined symbols use the plot
;*	style psym=8
;*
;* INPUTS:
;*
;*      plotinfo  a PLOTINFO - structure
;*		   the info-structure-instance edited in this widget
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*
;*      returns the slider value in plotinfo.plotStyle
;*
;*
;* CALLING SEQUENCE:
;*     value=get_plotStyle_value(plotinfo)
;*
;* MODIFICATION HISTORY:
;*     written march 1996, Haje Korth
;******************************************************************************
function get_plotStyle_value,plotinfo

  style=plotinfo.plotStyle

  if abs(style) eq 8 then begin
    case plotinfo.userstyle(2) of
      1:if style eq abs(style) then style=8 else style=-8
      2:if style eq abs(style) then style=9 else style=-9
      else: style=style
    endcase
  endif
  plotinfo.plotStyle=style

  return,plotinfo.plotStyle

end

;******************************************************************************
;* PROCEDURE:
;*      PRO get_user_symbol, userstyle, userx, usery
;*
;* DESCRIPTION:
;*	The procedure contains the user defined plot symbols. For adding
;*      new plot symbols see file papco_EXTENDING.TEXT
;*
;* INPUTS:
;*
;*      userstyle - number of user defined symbol
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*
;*      returns the vectors userx, usery of the user defined symbol
;*
;*
;* CALLING SEQUENCE:
;*     get_user_symbol, userstyle, userx, usery
;*
;* MODIFICATION HISTORY:
;*     written march 1996, Haje Korth
;******************************************************************************
PRO get_user_symbol,userstyle,userx,usery

  CASE userstyle OF
    1: BEGIN
         userx=[-0.5,0.0,0.5]
         usery=[0.0,1.0,0.0]
       END
    2: BEGIN
         userx=[-0.3,0.0,0.3]
         usery=[0.0,3.0,0.0]
       END
    ELSE: PRINT,'User symbol not defined in procedure: GET_USER_SYMBOL'
  ENDCASE

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION papco_panel_AddHeightWidgets, aBase, height
;*
;* DESCRIPTION:
;*	Add a slider and label  to the WIDGET_BASE 'aBase' which the user
;*	may use to select the height of that panelplot (that is, the 3rd
;*	element of the panelvector).
;*
;* INPUTS:
;*      aBase	 	a WIDGET_BASE
;*			the base, the widgets are to be added to
;*	height		a positive integer
;*			the current height of this panelplot.
;*
;* KEYWORDS:
;* 	VERTICAL        makes vertical slider if set. Default is horizontal.
;*
;* OUTPUT:
;*      returns a structure
;*          {lbl_height:aWidget, $
;*            sl_height:aWidget, $
;*            panelHeight:aNumber}
;*      This fields of this structure have to be added to the widgetData of
;*      the panel-editor.
;*
;* CALLING SEQUENCE:
;*     widgets=papco_panel_AddHeightWidgets(panel_base, plotinfo)
;*     additionalWData=create_struct(widgets, additionalWData)
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_panel_AddHeightWidgets, aBase, plotinfo, VERTICAL=VERTICAL

  if keyword_set(VERTICAL) then vertical=VERTICAL else vertical=0

  if vertical then begin
    xsize=20  &  ysize=460  & row=0  &  column=1
  endif else begin
    xsize=460  &  ysize=20  & row=1  &  column=0
  endelse

  height_base=widget_base(abase, row=row, column=column, /frame, /align_center)

  sl_height=WIDGET_SLIDER(height_base, xsize=xsize, ysize=ysize, $
                          VERTICAL=VERTICAL, /DRAG, $
                          minimum=1, maximum=20 > plotinfo.panelHeight , $
                          value=plotinfo.panelHeight > 1, uvalue='sl_height')
  sl_label=WIDGET_LABEL(height_base, xsize=20, $
                        value=varprt(plotinfo.panelHeight > 1), $
                        uvalue='sl_label')

  return, {sl_height:sl_height, $
           sl_label:sl_label, $
           panelHeight:plotinfo.panelHeight}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_returnData, widgetData
;*
;* DESCRIPTION:
;*      Creates a structure containing all tags in widgetData that should
;*      be returned from a panel-editor, depending on the widgets
;*      chosen by the user!
;*
;* INPUTS:
;*      widgetData        a structure containing the editor's widget-data.
;*
;* OUTPUT:
;*      a structure with the neccessary tag-fields
;*
;* CALLING SEQUENCE:
;*      returnData={typeVector:plotinfo.typeVector}
;*      additionalRData=papco_panel_ReturnData(widgetData)
;*      return, create_struct(ReturnData, additionalRData)
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
FUNCTION papco_panel_returnData, widgetData

; return tags that always get returned
ret={canceled:widgetData.canceled, $
     panelHeight:widgetData.panelHeight, $
     panelKind:widgetData.panelKind}

; only return certain TAGS if they have been defined!
names=TAG_NAMES(widgetData)

dummy=where(names EQ 'COLOR',  c)
if c GT 0 then ret=create_struct(ret, {color:widgetData.color})
dummy=where(names EQ 'PSYM',  c)
if c GT 0 then ret=create_struct(ret, {psym:widgetData.psym})
dummy=where(names EQ 'THICK',  c)
if c GT 0 then ret=create_struct(ret, {thick:widgetData.thick})
dummy=where(names EQ 'SYMSIZE',  c)
if c GT 0 then ret=create_struct(ret, {symsize:widgetData.symsize})
dummy=where(names EQ 'LINESTYLE',  c)
if c GT 0 then ret=create_struct(ret, {linestyle:widgetData.linestyle})

dummy=where(names EQ 'YSCALEINPUT_BASE', c)
if c GT 0 then $
  ret=create_struct(ret, {manualYScaling:widgetData.manualYScaling, $
                          yscaleLb:widgetData.yscaleLb, $
                          yScaleMin:widgetData.yScaleMin, $
                          yScaleMax:widgetData.yScaleMax} )

dummy=where(names EQ 'ZSCALEMIN', c)
if c GT 0 then $
  ret=create_struct(ret, {manualZScaling:widgetData.manualZScaling, $
                          zScaleMin:widgetData.zScaleMin, $
                          zScaleMax:widgetData.zScaleMax} )

dummy=where(names EQ 'CHANNEL', c)
if c GT 0 then $
  ret=create_struct(ret, {channel:widgetData.channel} )

dummy=where(names EQ 'PLOTSTYLE', c)
if c GT 0 then ret=create_struct(ret, {plotStyle:widgetData.plotStyle})

dummy=where(names EQ 'YLINLOG', c)
if c GT 0 then ret=create_struct(ret, {yLinLog:widgetData.yLinLog} )

dummy=where(names EQ 'ZLINLOG', c)
if c GT 0 then ret=create_struct(ret, {zLinLog:widgetData.zLinLog} )

dummy=where(names EQ 'FILENAME', c)
if c GT 0 THEN ret=create_struct(ret, {filename:widgetData.filename} )

dummy=where(names EQ 'DATAUNITS', c)
if c GT 0 then ret=create_struct(ret, {dataUnits:widgetdata.dataUnits} )

dummy=where(names EQ 'DESCR_ENTRY', c)
if c GT 0 then ret=create_struct(ret, {descr_type:widgetdata.descr_type, $
                                       descr_str:widgetdata.descr_str, $
                                       descr_extend:widgetdata.descr_extend, $
                                       descr:widgetdata.descr, $
                                       ylbl_str:widgetdata.ylbl_str, $
                                       ylbl_extend:widgetdata.ylbl_extend, $
                                       ylbl:widgetdata.ylbl, $
                                       zlbl_str:widgetdata.zlbl_str, $
                                       zlbl_extend:widgetdata.zlbl_extend, $
                                       zlbl:widgetdata.zlbl, $
                                       nogap:widgetdata.nogap} )

dummy=where(names EQ 'EXTERNAL_MODEL', c)
if c GT 0 then $
  ret=create_struct(ret, {internal_model:widgetdata.internal_model, $
                          external_model:widgetdata.external_model, $
                          mag_coords:widgetdata.mag_coords, $
                          mag_dyn:widgetdata.mag_dyn} )

return, ret

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_add_PlotStyleButtons, aBase, plotinfo, isScatter
;*
;* DESCRIPTION:
;*      Add the plotstyle - buttons to the panel-editor.
;*
;* INPUTS:
;*      aBase      a Widget
;*		   the base the plotstyle-widgets will be added to
;*      plotinfo  a PLOTINFO - structure
;*		   the info-structure-instance edited in this widget
;*
;*      isScatter  a boolean
;*		   if 0, the initial setting for PSYM will be smaller 0
;*			 which will produce a plot with connected points.
;*		   if 1, the initial setting for symbo-type will be gt 0,
;*			 which will produce a plot with isolated points.
;*
;* KEYWORDS:
;* 	VERTICAL        arranges widget vertical
;*
;* OUTPUT:
;*      a structure with the neccessary tag-fields
;*
;* CALLING SEQUENCE:
;*      widgets=papco_panel_add_PlotStyleButtons(panel_base, plotinfo, 0)
;*      additionalWData=create_struct(widgets, additionalWData)
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;*      march 1996, Haje Korth, userstyle added
;*      july 1997, Haje Korth, Thickness and Linestyle added
;*      december 1999, Reiner Friedel, symsize added
;******************************************************************************
FUNCTION papco_panel_Add_PlotStyleButtons, aBase, plotinfo, isScatter, $
                                           VERTICAL=VERTICAL

IF keyword_set(VERTICAL) THEN BEGIN
    xsize = 225 & yfac = 2
    ps_base=WIDGET_BASE(aBase, COLUMN=1, /FRAME, /ALIGN_CENTER)
ENDIF  ELSE BEGIN
    xsize = 225 & yfac = 1
    ps_base=WIDGET_BASE(aBase, ROW=1, /FRAME, /ALIGN_CENTER)
ENDELSE

thick     = plotinfo.THICK
linestyle = plotinfo.LINESTYLE
color     = plotinfo.COLOR
symsize   = plotinfo.SYMSIZE
psym      = plotinfo.PSYM

ps_select_Base=widget_base(ps_base, column=1)

style_names = ['Symbol', 'Color', 'Line Thick', 'Line Style', 'Symbol Size']
ps_type_base = CW_BGROUP(ps_select_Base, style_names, ROW= 2, /EXCLUSIVE,  $
                         UVALUE='spb_ps_type',  SET_VALUE = 0)

geom=widget_Info(ps_type_base, /GEOM)

mbase=widget_base(ps_select_Base)
ps_psym_base    = WIDGET_BASE(mbase, MAP=1)
ps_color_base   = WIDGET_BASE(mbase, MAP=0)
ps_thick_base   = WIDGET_BASE(mbase, MAP=0)
ps_lstyle_base  = WIDGET_BASE(mbase, MAP=0)
ps_symsize_base = WIDGET_BASE(mbase, MAP=0)

dw_plotColor = widget_draw(ps_color_base, xsize=xsize, ysize=15, $
                               UVALUE='dw_plotColor')

sl_ps_psym   = widget_slider(ps_psym_base, minimum=-9, maximum=10, $
                               value=(psym > (-9)) < 10, $
                               xsize=geom.xsize, /DRAG, /SUPPRESS_VALUE, $
                               UVALUE='sl_ps_psym')

sl_ps_thick  = widget_slider(ps_thick_base, minimum=1, maximum=15, $
                               value=(thick > 1) < 15, $
                               xsize=geom.xsize, /DRAG, /SUPPRESS_VALUE, $
                               UVALUE='sl_ps_thick')

sl_ps_lstyle = widget_slider(ps_lstyle_base, minimum=0, maximum=5, $
                               value=(linestyle > 0) < 5, $
                               xsize=geom.xsize, /DRAG, /SUPPRESS_VALUE, $
                               UVALUE='sl_ps_lstyle')

sl_ps_symsize= widget_slider(ps_symsize_base, minimum=1, maximum=5, $
                               value=(symsize > 1) < 5, $
                               xsize=geom.xsize, /DRAG, /SUPPRESS_VALUE, $
                               UVALUE='sl_ps_symsize')

widget_control, dw_plotColor, /DRAW_BUTTON

geom=widget_Info(ps_select_base, /geom)

dw_ps_prev=WIDGET_DRAW(ps_base, XSIZE=geom.xsize, YSIZE=geom.ysize/yfac)

return, {ps_psym_base:ps_psym_base, $
         ps_color_base:ps_color_base, $
         ps_thick_base:ps_thick_base, $
         ps_lstyle_base:ps_lstyle_base, $
         ps_symsize_base:ps_symsize_base, $
         userstyle:0, $
         dw_plotColor:dw_plotColor, $
         dw_ps_prev:dw_ps_prev, $
         color:plotinfo.color, $
         psym:plotinfo.psym, $
         thick:plotinfo.thick, $
         linestyle:plotinfo.linestyle, $
         symsize:plotinfo.symsize, $
         plotStyle:plotinfo.plotStyle, $
         is_scatterPlot:isScatter, $ ; EQ 0 <=> not a scatter-plot
         ps_numberOfColors:17, $
         ps_selectColor:0, $ ; EQ 0 <=> the color-selector is visible
         ps_isScatterPlot:0}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_Is_ScatterPlot, widgetData, pType
;*
;* DESCRIPTION:
;*      set the flag in 'widgetData' that indicates, if the currently edited
;*	plot is a scatter-plot or not. This routine is provided so that you
;*	don't have to know the name of that tag from the outside of
;*	papco_panel.pro
;*
;* INPUTS:
;*      widgetData	the data of the current plot-editor
;*	pType		a boolean
;*			0 -> is no scatter-plot
;*			1 -> is a scatter-plot
;*
;* OUTPUT:
;*      widgetData is returned
;*
;* CALLING SEQUENCE:
;*      widgetData=papco_panel_Is_ScatterPlot(widgetData, 0)
;*	; or :
;*      widgetData=papco_panel_Is_ScatterPlot(widgetData, 1)
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
FUNCTION papco_panel_Is_ScatterPlot, widgetData, pType

   widgetData.is_ScatterPlot=pType
   papco_panel_Refresh, widgetData, /SCATTER_TYPE
   return, widgetData

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddActionButtons, aBase, action
;*
;* DESCRIPTION:
;*      Add the default-buttons to the panelplot-editor 'aBase'. These
;*      are the help-button, the cancel-button and the action-button.
;*	The action-button's label is the input-variable 'action'.
;*
;* INPUTS:
;*      aBase	 	a WIDGET_BASE
;*			the base, the widgets are to be added to
;*	action		a string
;*			this is the label of the 'done'-button
;*
;* KEYWORDS:
;* 	VERTICAL        sets vertical arrangement of action buttons
;*
;* OUTPUT:
;*      a structure with the neccessary tag-fields
;*
;* CALLING SEQUENCE:
;*      widgets=papco_panel_AddActionButtons(panel_base, Action, plotinfo)
;*      additionalWData=create_struct(widgets, additionalWData)
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_panel_AddActionButtons, aBase, action, plotinfo, $
                                       CONFIG=CONFIG, $
                                       DATACACHE=DATACACHE, $
                                       VERTICAL=VERTICAL, $
                                       ROW=ROW, COLUMN=COLUMN

  if keyword_set(ROW) then begin
    row=ROW  &  column=0
  endif else row=1
  if keyword_set(COLUMN) then begin
    column=COLUMN  & row=0
  endif else column=0
  if keyword_set(VERTICAL) then begin
    temp=row  & row=column & column=temp
  endif

  ;make widget base for action buttons
  action_base=widget_base(aBase, COLUMN=column, ROW=row, /frame, /align_center)

  pb_help   = widget_button(action_base, uvalue='pb_help',   value=' Help ')
  pb_config = widget_button(action_base, uvalue='pb_config', value='Config')

  if n_elements(datacache) eq 1 then begin
      pb_datacache= widget_button( action_base, uvalue='pb_datacache', value='DataCache' )
  endif else begin
  	  pb_datacache= 0
  	  datacache= obj_new()
  end

  pb_cancel = widget_button(action_base, uvalue='pb_cancel', value='Cancel')
  pb_done_and_cont=0

  if not keyword_set(CONFIG) then widget_control, pb_config, sensitive=0

  if strlowcase(strleft(strtrim(action,2),3)) eq 'add' then begin
    pb_done_and_cont=widget_button(action_base, $
      uvalue='pb_done_and_cont', value=action+' & continue')
    pb_done =widget_button(action_base, $
      uvalue='pb_done',          value=Action+' & close')
  endif else $
     pb_done =widget_button(action_base, $
      uvalue='pb_done',          value=Action)

  return, {pb_config:pb_config, pb_help:pb_help, $
           pb_done_and_cont:pb_done_and_cont, pb_done:pb_done, $
           pb_datacache:pb_datacache, $
           datacache:datacache, $
           pb_cancel:pb_cancel, panelKind:plotinfo.panelKind}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_CreateWidgetData, struct1, struct2
;*
;* DESCRIPTION:
;*      The structures struct1 and struct2 are merged via the IDL-command
;*	'create_struct'. Additionally: If they have a tag in common,
;*      the value from struct2 is copied to struct1
;*
;*	This routine  is used to create the widgetData before realizing the
;*	editor-window. It is provided, because we don't know if in
;*	future-versions of papco_panel.pro other variables have to be added...
;*
;* INPUTS:
;*      struct1, struct2	two structures
;*
;* OUTPUT:
;*      a structure containing all tags from struct1 and struct2
;*
;* CALLING SEQUENCE:
;*    thisWidgetsData={helpfile:'papco_panel_template.help'}
;*    widgetData=papco_panel_CreateWidgetData(thisWidgetsData, additionalWData)
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
FUNCTION papco_panel_CreateWidgetData, struct1, struct2,  VERBOSE = VERBOSE

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0

t1=tag_names(struct1) & t2=tag_names(struct2) & struct = struct1

FOR i = 0, n_elements(t2) -1 DO BEGIN
    idx = where(t1 EQ t2(i), c)
    IF c EQ 0 THEN BEGIN
        IF verbose THEN message, 'Adding tag '+t2(i), /cont
        struct = create_struct(struct, t2(i), struct2.(i))
    ENDIF ELSE BEGIN
        IF verbose THEN message, 'Copying tag '+t2(i), /cont
        struct.(idx(0)) = struct2.(i)
    ENDELSE
ENDFOR

return, struct

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_panel_Realize, aBase, widgetData
;*
;* DESCRIPTION:
;*      realize the editor - window.
;*	The editor-window is displayed on the screen - the plot-preview is
;*	drawn in the drawing-widget
;*
;* INPUTS:
;*      aBase	 	-> the base-widget
;*	widgetData	-> the widgetData-structure
;*
;* KEYWORDS:
;*      _extra keywords go to widget_control.
;*
;* CALLING SEQUENCE:
;*      papco_panel_Realize, panel_base, widgetData
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
PRO papco_panel_Realize, aBase, widgetData, _extra=e

COMMON papco_color_names

pcol=fixed_color_index

;now, we'll set the position of the base:
geom = widget_info(aBase, /GEOMETRY)
x = 100-geom.xsize/2 > 10  &   y = 400-geom.ysize/2 > 10

widget_control, aBase, /REALIZE
widget_control, aBase, XOFFSET = x, YOFFSET = y

names=TAG_NAMES(widgetData)

; if required, draw the line color bar for the line color chooser
dummy=where(names EQ 'PS_COLOR_BASE', count)
IF count GT 0 then BEGIN
    widget_control, widgetData.dw_plotColor, get_value=windex
    wset, windex
    geom=widget_info(widgetData.dw_plotColor, /geometry)
    n=widgetData.ps_numberOfColors
    width=float(geom.xsize) / float(n)
    height=geom.ysize
    for i=0, 15 do polyfill, [width*i, width*i, width*(i+1), width*(i+1)], $
                             [0, height, height, 0], color=pcol(i), /device
    device, CURSOR_STANDARD=58 ; hand-symbol
ENDIF

papco_panel_Refresh, widgetData, /ALL

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddYScale_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add exclusive-buttons for selecting whether this plot shall use
;*	manual or automatic y-scaling and add entry fields for entering
;*	the lower and upper bound of the y-axis
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the papco_PLOTINFO-structure:
;*			     manualYScaling
;*			     yScaleMin
;*			     yScaleMax
;*			It is used to set the initial values for these
;*			widgets
;*
;* KEYWORDS:
;* 	VERTICAL        arranges widget vertical
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddYScale_Widgets(panelBase, plotinfo)
;*	widgetData.ef_yScaleFrom=tmp.ef_yScaleFrom
;*	widgetData.ef_yScaleTo  =tmp.ef_yScaleTo
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_panel_AddYScale_Widgets, base, plotinfo, VERTICAL=VERTICAL

IF keyword_set(VERTICAL) THEN $
  yscale_Base = WIDGET_BASE(base, COLUMN=1, /FRAME, /ALIGN_CENTER) ELSE $
  yscale_Base = WIDGET_BASE(base, ROW=1, /FRAME, /ALIGN_CENTER)

yscale_lbl = WIDGET_LABEL(yscale_Base, VALUE = 'Y-scaling:')

mpb_yautman = CW_BGROUP(yscale_Base, ['Aut', 'Man'], $
                       ROW=1, /EXCLUSIVE, UVALUE = 'mpb_yautman', $
                       SET_VALUE = PlotInfo.manualYScaling)

mpb_ylinlog = CW_BGROUP(yscale_Base, ['Lin', 'Log'], $
                       ROW=1, /EXCLUSIVE, UVALUE = 'mpb_ylinlog', $
                       SET_VALUE = PlotInfo.yLinLog)

IF keyword_set(VERTICAL) THEN $
  yscaleInput_Base=WIDGET_BASE(yscale_Base, COLUMN=1, /ALIGN_CENTER)ELSE $
  yscaleInput_Base=WIDGET_BASE(yscale_Base, ROW=1, /ALIGN_CENTER)

ef_yScaleFrom=CW_FIELD(yscaleInput_Base, TITLE='L:', XSIZE=10, $
		       VALUE=PlotInfo.yScaleMin, /FLOATING)
ef_yScaleTo=CW_FIELD(yscaleInput_Base, TITLE='H:', XSIZE=10, $
                       VALUE=PlotInfo.yScaleMax, /FLOATING)

ef_yscaleLb=CW_FIELD(yscale_Base, TITLE='Label #:', XSIZE=2, $
                       VALUE=PlotInfo.yscaleLb, /INTEGER)

RETURN, {yscaleInput_Base:yscaleInput_Base, $
         ef_yScaleFrom:ef_yScaleFrom, ef_yScaleTo:ef_yScaleTo, $
         ef_yscaleLb:ef_yscaleLb, $
         manualYScaling:PlotInfo.manualYScaling, $
         yscaleLb:PlotInfo.yscaleLb, $
         yLinLog:PlotInfo.yLinLog, mpb_ylinlog:mpb_ylinlog, $
         yScaleMin:PlotInfo.yScaleMin, yScaleMax:PlotInfo.yScaleMax}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddZScale_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add exclusive-buttons for selecting whether this plot shall use
;*	manual or automatic y-scaling and add entry fields for entering
;*	the lower and upper bound of the y-axis
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the papco_PLOTINFO-structure:
;*			     manualYScaling
;*			     zScaleMin
;*			     zScaleMax
;*			It is used to set the initial values for these
;*			widgets
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddZScale_Widgets(panelBase, plotinfo)
;*	widgetData.ef_zScaleFrom=tmp.ef_zScaleFrom
;*	widgetData.ef_zScaleTo  =tmp.ef_zScaleTo
;*
;* MODIFICATION HISTORY:
;*     written february 1997, Reiner Friedel
;***************************************************************************
FUNCTION papco_panel_AddZScale_Widgets, aBase, plotinfo, VERTICAL=VERTICAL

zscale_Base=WIDGET_BASE(aBase, COLUMN=1, /FRAME, /align_center)
zscaleChoice=WIDGET_BASE(zscale_Base, ROW=1,/ALIGN_CENTER)

if keyword_set(VERTICAL) then zscaleChoice=zscale_Base

zscaleLabel=WIDGET_LABEL(zscaleChoice,VALUE='Z-scaling:')
zscaleUse_Base=WIDGET_BASE(zscaleChoice, ROW=1, /EXCLUSIVE)
pb_zScaleAuto=WIDGET_BUTTON(zscaleUse_Base, VALUE='Aut',$
			      UVALUE='pb_zScaleAuto')
pb_zScaleManual=WIDGET_BUTTON(zscaleUse_Base, VALUE='Man',$
			        UVALUE='pb_zScaleManual')
linlog_Base=WIDGET_BASE(zscaleChoice, ROW=1, /EXCLUSIVE)
pb_zLin=WIDGET_BUTTON(linlog_Base, VALUE='Lin', UVALUE='pb_zLin')
pb_zLog=WIDGET_BUTTON(linlog_Base, VALUE='Log', UVALUE='pb_zLog')

zscaleInput_Base=WIDGET_BASE(zscale_Base, ROW=1, /ALIGN_CENTER)
if keyword_set(VERTICAL) then zscaleInput_Base=zscale_Base

ef_zScaleFrom=CW_FIELD(zscaleInput_Base, TITLE='Low: ', XSIZE=8, $
		         VALUE=plotinfo.zScaleMin, /FLOATING)
ef_zScaleTo=CW_FIELD(zscaleInput_Base, TITLE='High:', XSIZE=8, $
	               VALUE=plotinfo.zScaleMax, /FLOATING)

IF plotinfo.manualzscaling GE 100 THEN extend = plotinfo.manualzscaling-100 $
    ELSE extend = plotinfo.manualzscaling

ef_zscaleLb=CW_FIELD(zscaleInput_Base, TITLE='Extend:', XSIZE=2, $
                       VALUE=extend, /INTEGER)

; now, set the initial values for the exclusive-buttons
IF (plotinfo.manualZScaling / 100) THEN BEGIN
    WIDGET_CONTROL, pb_zScaleManual, /SET_BUTTON
    WIDGET_CONTROL, ef_zScaleFrom, SENSITIVE=1
    WIDGET_CONTROL, ef_zScaleTo, SENSITIVE=1
ENDIF ELSE BEGIN
    WIDGET_CONTROL, pb_zScaleAuto, /SET_BUTTON
    WIDGET_CONTROL, ef_zScaleFrom, SENSITIVE=0
    WIDGET_CONTROL, ef_zScaleTo, SENSITIVE=0
ENDELSE

; set initial values for lin/log
IF plotinfo.zLinLog THEN WIDGET_CONTROL, pb_zLog, /SET_BUTTON $
    ELSE WIDGET_CONTROL, pb_zLin, /SET_BUTTON

RETURN, {ef_zscaleLb:ef_zscaleLb, $
         ef_zScaleFrom:ef_zScaleFrom, $
         ef_zScaleTo:ef_zScaleTo, $
         pb_zScaleAuto:pb_zScaleAuto, $
         pb_zScaleManual:pb_zScaleManual, $
         pb_zLog:pb_zLog, $
         pb_zLin:pb_zLin, $
         manualZScaling:plotinfo.manualZScaling, $
         zScaleMin:plotinfo.zScaleMin, $
         zScaleMax:plotinfo.zScaleMax, $
         zLinLog:plotinfo.zLinLog}

END

;***************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_Addcolorbar_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add exclusive-buttons for selecting the papco internal Color Bars
;*      A choice of four color bars is built in. These are sub-colorbars
;*      dividing the main color bar into four. Color Bar 0 (bottom of table)
;*      is default (GSFC Rainbow).
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the PAPCO_PLOTINFO-structure:
;*			     channel
;*			It is used to set the initial values for these
;*			widgets
;*
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_Addcolorbar_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written february 1997, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_Addcolorbar_Widgets, aBase, plotinfo, VERTICAL=VERTICAL

COMMON plot_composer, widgetData
COMMON papco_color
COMMON papco_color_names

if keyword_set(VERTICAL) then begin
    column=1  &  xsize=225
endif else begin
    column=2  &  xsize=300
endelse

; check if the caller is the scratch pad. If so, don't use this widget.
if widgetData.composer_type eq 'scratch_pad' then begin
    spb_color=0 &  channel=0 & dw_colorbar0=0
    dw_colorbar0=0 & dw_colorbar1=0 & dw_colorbar2=0 & dw_colorbar3=0
    return,{junk:0}
    goto,out
endif

color_outer_base=WIDGET_BASE(aBase, COLUMN=1, /FRAME, /align_center)

colorName_Base=WIDGET_BASE(color_outer_base, COLUMN=column,/EXCLUSIVE)

color_names=['(def)', '(u 1)', '(u 2)', '(u 3)']
for i=0,3 do color_names(i)=strFormat(color_bar_names(i),22,/left) + $
                              strFormat(color_names(i),6,/right)

spb_color=lonarr(N_ELEMENTS(color_names))
for i=0, N_ELEMENTS(color_names)-1 DO $
    spb_color(i)=WIDGET_BUTTON(ColorName_Base, VALUE=color_names(i), $
	  	               UVALUE='spb_color'+strtrim(string(i), 2))

;make base for colorbar - drawn by refresh
color_bar_base=WIDGET_BASE(color_outer_base, COLUMN=1,/align_center)
label_base=WIDGET_LABEL(color_bar_base, VALUE='Current Color bar: ')
dw_colorbar=WIDGET_DRAW(color_bar_base, xsize=xsize, ysize=20, $
                        UVALUE='dw_colorbar00', /align_center)
out:

RETURN, {spb_color:spb_color, $
         channel:plotinfo.channel, $
         dw_colorbar:dw_colorbar}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddFile_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add a widget for opening and editing simple text files. Includes
;*      a browse function, and crerates files if they do not exist.
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the papco_PLOTINFO-structure:
;*			     filename
;*			It is used to set the initial values for this
;*			widget.
;*
;* KEYWORDS:
;*      DIRPATH  contains the "base" directory for the files
;*      FILTER   contains a file filter, e.g. '.dat'
;*      NOEDIT   indicates files are not ascii and should just be
;*               saved or restored.
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddFile_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written October 1997, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddFile_Widgets, aBase, plotinfo, $
                                      NOEDIT = NOEDIT, MODIFY = MODIFY, $
                                      DIRPATH = DIRPATH, FILTER = FILTER, $
                                      _EXTRA = e

IF keyword_set(e) THEN BEGIN
    tags = TAG_NAMES(e)
    pos = strpos(tags, 'PATH') & index = where(pos NE -1, c)
    IF c EQ 0 THEN e = create_struct(e, 'PATH', papco_getenv('PAPCO_HOME'))
    pos = strpos(tags, 'FILTER') & index = where(pos NE -1, c)
    IF c EQ 0 THEN e = create_struct(e, 'FILTER', '')
ENDIF ELSE e = {PATH:'', FILTER:''}

IF keyword_set(DIRPATH) THEN e.path = DIRPATH
IF keyword_set(FILTER) THEN e.filter = FILTER

filename=plotinfo.filename

IF filename EQ '' THEN BEGIN
    status=4
ENDIF ELSE BEGIN
    result=FINDFILE(e.path+filename, count=c)
    IF c GT 0 THEN status=1 ELSE status=0
ENDELSE

; create file choosing/editing/save/read widget.
file_base=WIDGET_BASE(abase, COLUMN=1, /FRAME, /align_center)

direc_base=WIDGET_BASE(file_base, ROW=1)
direc_label=WIDGET_LABEL(direc_base, VALUE=strformat('Directory:', 10, /LEF))
direc_text=WIDGET_TEXT(direc_base, XSIZE=60, YSIZE=1, $
                                VALUE=e.path, UVALUE='file_name', EDITABLE=1)

name_base=WIDGET_BASE(file_base, ROW=1)
name_label=WIDGET_LABEL(name_base, VALUE=strformat('Filename:', 10, /LEF))
name_text=WIDGET_TEXT(name_base, XSIZE=60, YSIZE=1, $
                      VALUE=filename, UVALUE='file_name', EDITABLE=1)

status_base=WIDGET_BASE(file_base, ROW=1)
status_label=WIDGET_LABEL(status_base, VALUE=strformat('Status:', 10, /LEF))
status_text=WIDGET_TEXT(status_base, XSIZE=20, YSIZE=1)

;--- create two bases for the buttons - one for ascii, one for binary files
action_outer_base=WIDGET_BASE(status_base,ROW=1)
action_base=WIDGET_BASE(action_outer_base)

ascii_base=WIDGET_BASE(action_base, ROW=1)
status_button=WIDGET_BUTTON(ascii_base, VALUE=strcenter('Create',10), $
                              UVALUE='status_button')

binary_base=WIDGET_BASE(action_base, ROW=1)
save_button=WIDGET_BUTTON(binary_base, VALUE=strcenter('Save',10), $
                              UVALUE='save_button')

load_button=WIDGET_BUTTON(binary_base, VALUE=strcenter('Load',10), $
                              UVALUE='load_button')

file_browse=WIDGET_BUTTON(status_base, VALUE=strcenter('Browse',10), $
                            UVALUE='file_browse')

;widgets for deleting / creating directories, deleting files
file_action_base = WIDGET_BASE(action_base, ROW = 1)
fil_delete_spb = WIDGET_BUTTON(file_action_base, $
                                  VALUE = strcenter('Del File', 8), $
                                  UVALUE='fil_delete_spb')
dir_delete_spb = WIDGET_BUTTON(file_action_base, $
                                  VALUE = strcenter('Del Dir', 8), $
                                  UVALUE='dir_delete_spb')
dir_create_spb = WIDGET_BUTTON(file_action_base, $
                                  VALUE = strcenter('New Dir', 8), $
                                  UVALUE='dir_create_spb')

if keyword_set(NOEDIT) then begin
    WIDGET_CONTROL, file_action_base, MAP=0
    WIDGET_CONTROL, ascii_base, MAP=0
    WIDGET_CONTROL, binary_base, MAP=1
endif else begin
    WIDGET_CONTROL, file_action_base, MAP=0
    WIDGET_CONTROL, ascii_base, MAP=1
    WIDGET_CONTROL, binary_base, MAP=0
endelse

if keyword_set(MODIFY) then begin
    WIDGET_CONTROL, file_action_base, MAP=1
    WIDGET_CONTROL, ascii_base, MAP=0
    WIDGET_CONTROL, binary_base, MAP=0
endif else begin
    WIDGET_CONTROL, file_action_base, MAP=0
    WIDGET_CONTROL, ascii_base, MAP=1
    WIDGET_CONTROL, binary_base, MAP=0
endelse

WIDGET_CONTROL, direc_text, SET_TEXT_SELECT=[strlen(e.path)]
WIDGET_CONTROL, name_text, SET_TEXT_SELECT=[strlen(filename)]

RETURN, {base:file_base, extra:e, $
         direc_text:direc_text, $
         name_text:name_text, $
         filter:e.filter, $
         path:e.path, $
         filename:filename, $
         file_browse:file_browse, $
         status:status, $
         status_text:status_text, $
         status_button:status_button,$
         save_button:save_button,$
         load_button:load_button}

end

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_panel_AddFile_EditFile
;*
;* DESCRIPTION:
;*      create or edit the user defined file
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;* 	none
;*
;* CALLING SEQUENCE:
;*      papco_panel_AddFile_EditFile
;*
;* MODIFICATION HISTORY:
;*      October 1997, Reiner Friedel
;******************************************************************************
PRO papco_panel_AddFile_EditFile, widgetData

COMMON papco_panel_AddFile_EditFile, widgetData2, edit_text, entries

widgetData2=widgetData
entries=STRARR(1) & tmpentry='' & i=-1
fname=papco_fname(widgetData.path+widgetData.filename)

IF widgetData.status EQ 0 THEN BEGIN
    ;check if a valid writable file was chosen
    on_ioerror, report
    openw,f_unit, fname, /GET_LUN
    close,f_unit & FREE_LUN, f_unit
    action='Create '
    goto,ok
    report:
       dummy=messageBox('Not able to create '+widgetData.path+ $
                        widgetData.filename,$
                        ['OK'], /CENTER, $
                        TITLE='Error opening file')
       widgetData.status=0
       return
    ok:
ENDIF ELSE BEGIN
    action='Edit '
    OPENR, f_unit, fname, /GET_LUN
    WHILE NOT EOF(f_unit) DO BEGIN
      i=i+1
      READF, f_unit, tmpentry
      if i eq 0 then  entries(i)=tmpentry else entries=[entries,tmpentry]
    ENDWHILE
    CLOSE, f_unit &  FREE_LUN, f_unit
ENDELSE

widgetData.status=1
edit_base=PAPCO_WIDGET_BASE(TITLE=action+widgetData.filename, $
                            /COLUMN, /MODAL, $
                            GROUP_LEADER=widgetData.base)
edit_text=WIDGET_TEXT(edit_base, XSIZE=100, YSIZE=25, VALUE=entries, $
                      UVALUE='edit_text', /EDITABLE,  /ALL_EVENTS, /SCROLL)
edit_done=WIDGET_BUTTON(edit_base, VALUE='Done', UVALUE='edit_done')

WIDGET_CONTROL,edit_base,/REALIZE

papco_xmanager, "papco_panel_AddFile_EditFile", edit_base, $
                GROUP_LEADER=widgetData.base

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_panel_AddFile_EditFile_event
;*
;* DESCRIPTION:
;*      event handler called bei XMANAGER
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*
;* CALLING SEQUENCE:
;*      have a look at PRO events_panel_event
;*
;* MODIFICATION HISTORY:
;*      March 1996, Haje Korth
;******************************************************************************
PRO papco_panel_AddFile_EditFile_event, event

COMMON papco_panel_AddFile_EditFile, widgetData, edit_text, entries

WIDGET_CONTROL,GET_UVALUE=uval, event.id

CASE uval OF
    'edit_text' : BEGIN
      WIDGET_CONTROL, edit_text, GET_VALUE=entries
    END
    'edit_done' : BEGIN
      valid_entries=WHERE(entries NE '',count)
      IF count GT 0 THEN BEGIN
        message,'Writing user file '+widgetData.path+widgetData.filename,/cont
        fname=papco_fname(widgetData.path+widgetData.filename)
        OPENW, f_unit, fname, /GET_LUN
        FOR c=0,N_ELEMENTS(valid_entries)-1 DO $
           PRINTF, f_unit, entries(valid_entries(c))
        CLOSE, f_unit & FREE_LUN, f_unit
      ENDIF
      WIDGET_CONTROL, event.top, /DESTROY
    END
ENDCASE

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddUnits_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add a widget for selecting flux/conts/ energyflux etc units
;*      for particle spectra.
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the papco_PLOTINFO-structure:
;*			     filename
;*			It is used to set the initial values for this
;*			widget.
;*
;* KEYWORDS:
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddUnits_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written February 2000, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddUnits_Widgets, aBase, plotinfo

; create base for unit choice buttons.
unit_base=WIDGET_BASE(abase, COLUMN=1, /FRAME, /align_center)
names=['Counts  [cts]', 'Counts per second  [cts]s-1', $
       'Flux [cts]s-1 cm-2 sr-1', 'Diff. Flux [cts]s-1 cm-2 sr-1 keV-1', $
       'PSD ', 'PSD @ const. mu']
FOR i = 0,  n_elements(names)-1 DO names(i) = strformat(names(i), 30)
mpb_unit=CW_BGROUP(unit_base, names, ROW=3, /EXCLUSIVE, $
                   LABEL_LEFT=strFormat('Data Units:', 12, /LEFT), $
                   UVALUE='mpb_unit', IDS=unit_ids, $
                   SET_VALUE = plotinfo.dataUnits)

return,{dataUnits:plotinfo.dataUnits, mpb_unit:mpb_unit}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddVar_Ave_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add widgets for selecting averaging variables.
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*
;* KEYWORDS:
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddVar_Ave_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written March 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddVar_Ave_Widgets, aBase, name, $
                                         st, en, v1, v2, vr, frmt, $
                                         STRUCT_ONLY = STRUCT_ONLY
name = strupcase(name)

dummy =  create_struct($
  name+'_AVERAGE_FROM_VAL', 0.0, $
  name+'_AVERAGE_FROM_VAL_INFO', 'Start value for variable range', $
  name+'_AVERAGE_TO_VAL', 0.0, $
  name+'_AVERAGE_TO_VAL_INFO', 'End value for variable range', $
  ;ame+'_AVERAGE_RANGE_FRMT',"("+frmt+",'<"+name+"<',"+frmt+")", $
  name+'_AVERAGE_RANGE_FRMT',"("+"'"+name+": ',"+frmt+",'-',"+frmt+")", $
  name+'_AVERAGE_RANGE_FRMT_INFO', 'Verbose string of range chosen' )

IF keyword_set(STRUCT_ONLY) THEN return, dummy

base = WIDGET_BASE(aBase, ROW = 1, /ALIGN_CENTER, /FRAME)

;set variable range and resolution
st = st < v2 > v1 & en = en < v2 > v1

from_base = WIDGET_BASE(base, COLUMN = 1, /ALIGN_CENTER)
from_lbl= WIDGET_LABEL(from_base, /ALIGN_CENTER, UVALUE='FROM_AVERAGE_LBL',$
                       VALUE=strformat('Low',35, /CENTER))
from = WIDGET_SLIDER(from_base, /DRAG, /SUPPRESS_VALUE, xsize=200, $
         VALUE=(st-v1)/vr, MINIMUM = 0, MAXIMUM = (v2-v1)/vr, $
         UVALUE = {uvalue:name+'_AVERAGE_FROM', v1:v1, v2:v2, vr:vr})

to_base = WIDGET_BASE(base, COLUMN = 1, /ALIGN_CENTER)
to_lbl  = WIDGET_LABEL(to_base, /align_center, UVALUE='TO_AVERAGE_LBL',$
                       VALUE=strformat('High',35, /CENTER))
to   = WIDGET_SLIDER(to_base, /DRAG, /SUPPRESS_VALUE, xsize=200, $
         VALUE=(en-v1)/vr, MINIMUM=0, MAXIMUM = (v2-v1)/vr, $
         UVALUE={uvalue:name+'_AVERAGE_TO', v1:v1, v2:v2, vr:vr})

return, create_struct(name+'_AVERAGE_BASE', base, $
                      name+'_AVERAGE_FROM', from, $
                      name+'_AVERAGE_TO', to, $
                      name+'_AVERAGE_FROM_LBL', from_lbl, $
                      name+'_AVERAGE_TO_LBL', to_lbl, $
                      name+'_AVERAGE_FRMT', frmt, $
                      name+'_AVERAGE_FROM_VAL', st, $
                      name+'_AVERAGE_TO_VAL', en)

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddVar_Sel_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add a widget for selecting flux/conts/ energyflux etc units
;*      for particle spectra.
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*
;* KEYWORDS:
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddVar_Sel_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written March 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddVar_Sel_Widgets, aBase, name, $
                                         select, st, en, v1, v2, vr, frmt, $
                                         STRUCT_ONLY = STRUCT_ONLY
name = strupcase(name)

dummy =  create_struct($
  name+'_SELECT_RANGE', 0, $
  name+'_SELECT_RANGE_INFO', '0-turns selection off, 1-turns selection on', $
  name+'_SELECT_FROM_VAL', 0.0, $
  name+'_SELECT_FROM_VAL_INFO', 'Start value for variable range', $
  name+'_SELECT_TO_VAL', 0.0, $
  name+'_SELECT_TO_VAL_INFO', 'End value for variable range', $
  name+'_SELECT_RANGE_FRMT',"("+frmt+",'<"+name+"<',"+frmt+")", $
  name+'_SELECT_RANGE_FRMT_INFO', 'Verbose string of range chosen' )

IF keyword_set(STRUCT_ONLY) THEN return, dummy

base=WIDGET_BASE(aBase, ROW = 1, /ALIGN_CENTER, /FRAME)
rangeNames=['Off','On']
dummy = WIDGET_BASE(Base, ROW = 1, /ALIGN_CENTER)
range_base=CW_BGROUP(dummy, rangeNames, /COLUMN, /EXCLUSIVE, $
                     UVALUE=name+'_SELECT_RANGE', SET_VALUE = select < 1)

;set variable range and resolution
st = st < v2 > v1 & en = en < v2 > v1

set_base = WIDGET_BASE(base, COLUMN = 1, /ALIGN_CENTER)


from_lbl= WIDGET_LABEL(set_base, /ALIGN_CENTER, UVALUE='FROM_SELECT_LBL',$
                       VALUE=strformat('Low',35, /CENTER))
from = WIDGET_SLIDER(set_Base, /DRAG, /SUPPRESS_VALUE, xsize=350, $
         VALUE=(st-v1)/vr, MINIMUM = 0, MAXIMUM = (v2-v1)/vr, $
         UVALUE = {uvalue:name+'_SELECT_FROM', v1:v1, v2:v2, vr:vr})

to_lbl  = WIDGET_LABEL(set_base, /align_center, UVALUE='TO_SELECT_LBL',$
                       VALUE=strformat('High',35, /CENTER))
to   = WIDGET_SLIDER(set_Base, /DRAG, /SUPPRESS_VALUE, xsize=350, $
         VALUE=(en-v1)/vr, MINIMUM=0, MAXIMUM = (v2-v1)/vr, $
         UVALUE={uvalue:name+'_SELECT_TO', v1:v1, v2:v2, vr:vr})

return, create_struct(name+'_SELECT_BASE', base, $
                      name+'_SELECT_SET_BASE', set_base, $
                      name+'_SELECT_FROM', from, $
                      name+'_SELECT_TO', to, $
                      name+'_SELECT_FROM_LBL', from_lbl, $
                      name+'_SELECT_TO_LBL', to_lbl, $
                      name+'_SELECT_RANGE', select, $
                      name+'_SELECT_FRMT', frmt, $
                      name+'_SELECT_FROM_VAL', st, $
                      name+'_SELECT_TO_VAL', en)

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddDescr_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add a widget for specifying own plot description label at right of
;*      panel.
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least those fields from
;*			from the papco_PLOTINFO-structure:
;*			     soptions
;*			soptions(15) and iotions(15) is used to set the initial
;*			values for this widget.
;*
;* KEYWORDS:
;* 	VERTICAL        makes vertical slider if set. Default is horizontal.
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_AddUnits_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written August 2000, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddDescr_Widgets, aBase, plotinfo, VERTICAL=VERTICAL, $
                                       ZLBL = ZLBL

IF keyword_set(VERTICAL) THEN $
  descr_base=WIDGET_BASE(abase, COLUMN=1, /FRAME, /ALIGN_CENTER) $
ELSE $
  descr_base=WIDGET_BASE(abase, ROW=1, /FRAME, /ALIGN_CENTER)

descr_names = ['RS Panel', 'LS Panel', 'Colorbar']
spb_descr = papco_CW_BGROUP(descr_base, descr_names, ROW = 1, $
  /EXCLUSIVE, LABEL_TOP='Description:', $
  UVALUE='spb_descr', SET_VALUE = plotinfo.descr_type)

names=['Auto','Manual', 'Inhibit']

;make toggle base for all setable descriptions
descr_out_base=WIDGET_BASE(descr_base, column=1, /base_align_center)
descr_top_base=WIDGET_BASE(descr_out_base, /base_align_center)

rs_base = WIDGET_BASE(descr_top_base, COLUMN=1, /ALIGN_CENTER)
mpb_descr=CW_BGROUP(rs_base, names,ROW=1, /EXCLUSIVE, $
                    UVALUE='mpb_descr', SET_VALUE=plotinfo.descr)
descr_str=str_sep(plotinfo.descr_str,'!C')
descr_entry=WIDGET_TEXT(rs_base, value=descr_str, SCR_YSIZE=80,$
                    YSIZE=30, XSIZE=32, /SCROLL, /EDITABLE, $
                    UVALUE='descr_entry')

ls_base = WIDGET_BASE(descr_top_base, COLUMN=1, /ALIGN_CENTER)
mpb_ylbl=CW_BGROUP(ls_base, names,ROW=1, /EXCLUSIVE, $
                    UVALUE='mpb_ylbl', SET_VALUE=plotinfo.ylbl)
ylbl_str=str_sep(plotinfo.ylbl_str,'!C')
ylbl_entry=WIDGET_TEXT(ls_base, value=ylbl_str, SCR_YSIZE=80,$
                    YSIZE=30, XSIZE=32, /SCROLL, /EDITABLE, $
                    UVALUE='ylbl_entry')

z_base = WIDGET_BASE(descr_top_base, COLUMN=1, /ALIGN_CENTER)
mpb_zlbl=CW_BGROUP(z_base, names,ROW=1, /EXCLUSIVE, $
                   UVALUE='mpb_zlbl', SET_VALUE=plotinfo.zlbl)
zlbl_str=str_sep(plotinfo.zlbl_str,'!C')
zlbl_entry=WIDGET_TEXT(z_base, value=zlbl_str, SCR_YSIZE=80,$
                   YSIZE=30, XSIZE=32, /SCROLL, /EDITABLE, $
                   UVALUE='zlbl_entry')
;options base
opt_base = WIDGET_BASE(descr_base, ROW = 1)
ef_ext_desc = CW_FIELD(opt_base, TITLE = 'RSH:', XSIZE = 2, $
                    VALUE = plotinfo.descr_extend, /INTEGER, $
                    UVALUE = 'ef_ext_desc',  /RETURN_EVENTS)
ef_ext_ylbl = CW_FIELD(opt_base, TITLE = 'LSH:', XSIZE = 2, $
                    VALUE = plotinfo.ylbl_extend, /INTEGER, $
                    UVALUE = 'ef_ext_ylbl',  /RETURN_EVENTS)
toggle_base=WIDGET_BASE(opt_base, /NONEXCLUSIVE, COLUMN=1)
spb_nogap = WIDGET_BUTTON(toggle_base, UVALUE = 'spb_nogap', $
                    VALUE='No Gap')
WIDGET_CONTROL, spb_nogap, SET_BUTTON = plotinfo.nogap


return,{descr_type:plotinfo.descr_type, $
        descr_entry:descr_entry, descr_extend:plotinfo.descr_extend, $
        descr:plotinfo.descr, descr_str:plotinfo.descr_str, $
        ylbl_entry:ylbl_entry, ylbl_extend:plotinfo.ylbl_extend, $
        ylbl:plotinfo.ylbl, ylbl_str:plotinfo.ylbl_str, $
        zlbl_entry:zlbl_entry, zlbl_extend:plotinfo.zlbl_extend, $
        zlbl:plotinfo.zlbl, zlbl_str:plotinfo.zlbl_str, $
        nogap:plotinfo.nogap, $
        ef_ext_desc:ef_ext_desc, ef_ext_ylbl:ef_ext_ylbl, $
        rs_base:rs_base, ls_base:ls_base, z_base:z_base}

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_AddBatch_Widgets, aBase, plotinfo
;*
;* MODIFICATION HISTORY:
;*      written January 2001, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_AddBatch_Widgets, aBase, plotinfo, REDO = REDO

  batch_base=WIDGET_BASE(abase, ROW=2, /FRAME, /align_center)

;-- some options for the batch make -------------------------------------------
  IF keyword_set(REDO) THEN option_names = redo $
    ELSE option_names = ['Level 0', 'Level 1', 'Level2']
  ba_option_base = CW_BGROUP(batch_base, option_names, COLUMN = 6, $
                             /EXCLUSIVE,  $
                             UVALUE='spb_ba_option',  SET_VALUE = 0)

  ba_action_base = WIDGET_BASE(batch_base, ROW = 1, /align_center)

;-- entry field to restrict dates "from" --------------------------------------
  ef_ba_from = CW_FIELD(ba_action_base, /STRING, TITLE = 'From Date:', $
                        UVALUE='ef_ba_from', /RETURN_EVENTS,  $
                        VALUE = '', XSIZE=8)

;-- some single toggle options ------------------------------------------------
  ba_toggle_base=WIDGET_BASE(ba_action_base, /FRAME, ROW = 1, /NONEXCLUSIVE)
  spb_ba_force = WIDGET_BUTTON(ba_toggle_base, UVALUE='spb_ba_force', $
                               VALUE='Force')

;-- the batch action button ---------------------------------------------------
  spb_ba_do=widget_button(ba_action_base, Value='Make Batch', $
                          uvalue='spb_dobatch')

  proc_call=plotinfo.PANELKIND+'_make_batch'

  return,{make_batch_call:proc_call, ba_force:0, ba_option:0, ba_from:''}

END


;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_panel_Add_MagMod_Widgets, aBase, plotinfo
;*
;* DESCRIPTION:
;*      Add a widget for specifying ONERA lib magnetic field models to use.
;*      Typically used by ephemeris modules
;*
;* INPUTS:
;*	aBase	 	a WIDGET_BASE (an integer)
;*			the base, you want the widgets to add to
;*      plotinfo	a structure containing a least these fields:
;*
;*
;*			     soptions
;*			soptions(15) and iotions(15) is used to set the initial
;*			values for this widget.
;*
;* KEYWORDS:
;*
;*      STRUCT_ONLY - return the structure made for this widget only.
;*
;* OUTPUT:
;*      a structure containing the widget-ids of the added entry-fields
;*
;* CALLING SEQUENCE:
;*	tmp=papco_panel_Add_MagMod_Widgets(panelBase, plotinfo)
;*
;* MODIFICATION HISTORY:
;*      written May 2003, Reiner Friedel
;******************************************************************************
FUNCTION papco_panel_Add_MagMod_Widgets, aBase, plotinfo, $
        TICKS = TICKS, MODEL_ONLY =  MODEL_ONLY, STRUCT_ONLY = STRUCT_ONLY, $
        LABEL = LABEL

IF keyword_set(TICKS) THEN ticks = 1 ELSE ticks = 0

;if LABEL is set, a current plotinfo struct is passed in, and a
;descriptor label of what is set is required out.
IF keyword_set(LABEL) THEN BEGIN
    int_list = papco_make_choice_names(plotinfo.INTERNAL_MODEL_INFO)
    ext_list = papco_make_choice_names(plotinfo.EXTERNAL_MODEL_INFO)
    mag_coord_names = papco_make_choice_names(plotinfo.MAG_COORDS_INFO)
    mag_pitch_names = papco_make_choice_names(plotinfo.PITCH_PROC_INFO)
    label = int_list(plotinfo.INTERNAL_MODEL)+' - '+ $
            ext_list(plotinfo.EXTERNAL_MODEL)+' - '+ $
            mag_coord_names(plotinfo.MAG_COORDS)
    IF plotinfo.MAG_COORDS EQ 1 THEN $ ; L*, add pitch angle info
      label = label + ' ('+ mag_pitch_names(plotinfo.PITCH_PLOT)+' deg)'
    return, label
ENDIF

;test if plotinfo is a structure, and the required structure tags are
;available, if not, add them.

dummy = {EXTERNAL_MODEL:5, $ ;default Olson Pgitzer 77
         EXTERNAL_MODEL_INFO:'External; 0: none, ' + $
                             '1: Mead&Fairfield [1975], ' +  $
                             '2: Tsyganenko short [1987], ' + $
                             '3: Tsyganenko long [1987], ' +  $
                             '4: Tsyganenko [1989c], ' +  $
                             '5: Olson&Pfitzer quiet [1977], ' +  $
                             '6: Olson&Pfitzer dyn. [1988], ' + $
                             '7: Tsyganenko [1996], ' + $
                             '8: Ostapenko&Maltsev [1997], ' + $
                             '9: Tsyganenko [2001], ' + $
                             '10: Tsyganenko [2001] storm', $
         INTERNAL_MODEL:0, $
         INTERNAL_MODEL_INFO:'Internal; ' + $
                             '0: IGRF [1945-2005]', $
         MAG_COORDS:0,     $
         MAG_COORDS_INFO:'Mag Coords; 0: L, 1: L*, 2: MLT, 3: MLAT,' +  $
                         '4: Bx, 5: By, 6: Bz, 7: Btot, 8: Btot/Bmin', $
         MAG_COORDS_UNITS:'0: L, 1: L*, 2: hr., 3: deg.,' +  $
                          '4: nT, 5: nT, 6: nT, 7: nT, 8: no dim.', $
         L_CALC:0, $
         L_CALC_INFO:'0: McIlwain L calc, 1: McIlwain L + Roederer L* calc', $
         MAG_DYN:0, MAG_DYN_INFO:'0: static default, 1: Dynamic inputs', $
         MAG_PROC:0, $
         MAG_PROC_INFO:   '0: Current+Previous, 1: Current, ' + $
                         '2: Force Current, 3: Force Current+Previous' , $
         PITCH_PROC:1, $
         PITCH_PROC_INFO: 'PA to use (bit toiggle ^2); 0: 90, 1: 80, 2: 70, '+$
                          '3: 60, 4: 50, 5: 40, 6: 30, 7: 20, 8: 10', $
         PITCH_PLOT:0, $
         PITCH_PLOT_INFO: 'L(PA) to plot; 0: 90, 1: 80, 2: 70, 3: 60, '+ $
                          '4: 50,  5: 40, 6: 30, 7: 20, 8: 10'}

IF keyword_set(STRUCT_ONLY) THEN return, dummy

in_struct = plotinfo
r = size(in_struct, /st)
IF r.TYPE_NAME	NE 'STRUCT' THEN in_struct = dummy ELSE BEGIN
    STRUCT_ASSIGN,  in_struct, dummy & in_struct = dummy
ENDELSE

title = 'used as keyword variable, needs to be initialized'

mag_model_outer = WIDGET_BASE(aBase, COLUMN=1, /ALIGN_CENTER, /FRAME)

;--- choices for mag coord processing control ---------------------------------
mag_proc_names = papco_make_choice_names(dummy.MAG_PROC_INFO)
mag_proc_base = papco_CW_BGROUP(mag_model_outer, mag_proc_names, $
                         BUTTON_IDS = mag_proc_ids, $
                         ROW=1, /EXCLUSIVE, $
                         UVALUE='spb_mag_proc', $
                         SET_VALUE=in_struct.MAG_PROC)

mag_model_base = WIDGET_BASE(mag_model_outer, ROW=1, /ALIGN_CENTER)
model_choice_base = WIDGET_BASE(mag_model_base, COLUMN=1, /ALIGN_CENTER)

;--- provide choices for Internal Mag Field -----------------------------------
int_list = papco_make_choice_names(dummy.INTERNAL_MODEL_INFO, TITLE = title)
spb_int = WIDGET_DROPLIST(model_choice_base, VALUE = int_list, $
                          TITLE = title, UVALUE='spb_int')
widget_control, spb_int, SET_DROPLIST_SELECT = in_struct.INTERNAL_MODEL

;--- provide choices for External Mag Field -----------------------------------
ext_list = papco_make_choice_names(dummy.EXTERNAL_MODEL_INFO, TITLE = title)
spb_ext = WIDGET_DROPLIST(model_choice_base, VALUE = ext_list, $
                          TITLE = title, UVALUE='spb_ext')
widget_control, spb_ext, SET_DROPLIST_SELECT = in_struct.EXTERNAL_MODEL

toggle_base=WIDGET_BASE(mag_model_base, COLUMN = 1)

;--- provide choices for L*, since it takes time to calculate
l_names = ['L', 'L+L*']
lstar_base = CW_BGROUP(toggle_base, l_names, ROW = 1, /EXCLUSIVE, $
               UVALUE='spb_lvalues', SET_VALUE = in_struct.L_CALC)

;--- toggle button for dynamic magnetic field ---------------------------------
dyn_base = WIDGET_BASE(toggle_base, COLUMN = 1, /NONEXCLUSIVE)
spb_dynamic = WIDGET_BUTTON(dyn_base, UVALUE='spb_dynamic', $
                            VALUE='Dynamic Inputs')
WIDGET_CONTROL, spb_dynamic, SET_BUTTON=in_struct.mag_dyn

;--- pitch angle for shell splitting control ----------------------------------
mag_pitch_names = papco_make_choice_names(dummy.PITCH_PROC_INFO)
n = n_elements(mag_pitch_names) & set = bytarr(n)
FOR i = 0, n-1 DO $
    IF (in_struct.PITCH_PROC AND 2^i) EQ 2^i THEN set(i) = 1 ELSE set(i) = 0
mag_pitch_base = CW_BGROUP(mag_model_outer, mag_pitch_names, ROW = 1, $
                   label_left = 'L(PA) to calc:', $
                   /NONEXCLUSIVE, UVALUE='spb_mag_pitch', SET_VALUE = set)

IF keyword_set(MODEL_ONLY) THEN BEGIN
    coord_base = -1 & mag_coord_ids = -1 & pitch_plot_base = 1 & GOTO, out
ENDIF

coord_base = WIDGET_BASE(mag_model_outer, COLUMN = 1, /ALIGN_CENTER)

;--- provide choices for Magnetic coordinates --------------------------------
mag_coord_names = papco_make_choice_names(dummy.MAG_COORDS_INFO, TITLE = title)

;--- pitch angle for plotting -------------- ----------------------------------
pitch_plot_base = CW_BGROUP(coord_base, mag_pitch_names, ROW = 1, $
                   label_left = 'L(PA) to plot:', /EXCLUSIVE,  $
                   UVALUE='spb_pitch_plot', SET_VALUE = in_struct.PITCH_PLOT)
IF ticks EQ 1 THEN BEGIN ;make multiple press buttons...
    n = n_elements(mag_coord_names) & set = bytarr(n)
    FOR i = 0, n-1 DO $
      IF (in_struct.mag_coords and 2^i) eq 2^i then set(i)=1 else set(i)=0
    mag_coord_base=papco_CW_BGROUP(coord_base, mag_coord_names, $
                         BUTTON_IDS = mag_coord_ids, $
                         ROW=1, /NONEXCLUSIVE, UVALUE='mpb_mag_coord', $
                         LABEL_LEFT = title, SET_VALUE=set)

ENDIF ELSE begin
    mag_coord_base=papco_CW_BGROUP(coord_base, mag_coord_names, $
                         BUTTON_IDS = mag_coord_ids, $
                         ROW=1, /EXCLUSIVE, UVALUE='spb_mag_coord', $
                         LABEL_LEFT = title, SET_VALUE=in_struct.MAG_COORDS)
ENDELSE


IF keyword_set(MODEL_ONLY) THEN BEGIN
    coord_base = -1 & mag_coord_ids = -1 & GOTO, out
ENDIF

out:

return, create_struct(in_struct, $
                      'pitch_plot_base', pitch_plot_base, $
                      'mag_pitch_base', mag_pitch_base, $
                      'mag_coord_ids', mag_coord_ids, $
                      'ticks', ticks, $
                      'mag_model_outer_base', mag_model_outer, $
                      'coord_base', coord_base)

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_panel_Refresh, widgetData,  ALL=ALL, $
;*                         SELECT_PLOT_STYLE=SELECT_PLOT_STYLE, $
;*                         SCATTER_TYPE=SCATTER_TYPE, PLOT_STYLE=PLOT_STYLE
;*
;*
;* DESCRIPTION:
;*      update the paneleditor-display
;*	This procedure is called on some of the events handled in this file
;*
;* INPUTS:
;*      widgetData	the windows' data
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*      called from papco_panel_Event
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;*      march 1996, Haje Korth, user symbols added
;*      july 1997, Haje Korth, Thick and Linestyle added
;*      october 1997, Reiner Friedel, Add File added
;*      March, 2003, reiner Freidel, Add Var_Sel added
;******************************************************************************
PRO papco_panel_Refresh, widgetData,  ALL=ALL, ADD_FILE=ADD_FILE, $
                         SELECT_PLOT_STYLE=SELECT_PLOT_STYLE, $
                         SCATTER_TYPE=SCATTER_TYPE, PLOT_STYLE=PLOT_STYLE, $
                         COLBAR_STYLE=COLBAR_STYLE, $
                         YSCALE=YSCALE, ZSCALE=ZSCALE, DESCR=DESCR,  $
                         UNITS = UNITS, VAR_SEL = VAR_SEL, VAR_AVE = VAR_AVE, $
                         MAG_COORD = MAG_COORD

COMMON papco_color_names

IF KEYWORD_SET(ALL) THEN BEGIN
    PLOT_STYLE=1  &  SELECT_PLOT_STYLE=1  &  COLBAR_STYLE=1
    ADD_FILE=1  &  YSCALE=1  &  ZSCALE=1  &  DESCR=1 &  UNITS = 1
    VAR_SEL = 'ALL' & VAR_AVE= 'ALL' & MAG_COORD = 1
ENDIF

names=TAG_NAMES(widgetData)

dummy=where(names EQ 'EXTERNAL_MODEL', c)   &  has_Mag_Coord=c
dummy=where(names EQ 'SPB_COLOR', c)        &  has_ColBarEditor=c
dummy=where(names EQ 'PS_COLOR_BASE', c)    &  has_PlotStyleEditor=c
dummy=where(names EQ 'STATUS', c)           &  has_add_file=c
dummy=where(names EQ 'YSCALEINPUT_BASE', c) &  has_yscale=c
dummy=where(names EQ 'ZLINLOG', c)          &  has_zscale=c
dummy=where(names EQ 'DESCR', c)            &  has_descr=c
dummy=where(names EQ 'UNITS', c)            &  has_units=c

;check for any Var Sel widgets. They names contain "_SELECT_"
pos = strpos(names, '_SELECT_RANGE') & var_idx = where(pos NE -1, c)
IF c NE 0 THEN BEGIN
    var_names = strarr(c)
    FOR i=0,c-1 DO var_names(i) = strmid(names(var_idx(i)),0,pos(var_idx(i)))
    has_var_sel = 1
ENDIF ELSE has_var_sel = 0

;check for any Var Ave widgets. They names contain "_AVERAGE_"
pos = strpos(names, '_AVERAGE_RANGE') & ave_idx = where(pos NE -1, c)
IF c NE 0 THEN BEGIN
    ave_names = strarr(c)
    FOR i=0,c-1 DO ave_names(i) = strmid(names(ave_idx(i)),0,pos(ave_idx(i)))
    has_var_ave = 1
ENDIF ELSE has_var_ave = 0

;-- display the colorbar-style ------------------------------------------------
  if keyword_set(COLBAR_STYLE) AND has_ColBarEditor then begin
     c_index=widgetData.channel

     widget_control, widgetData.spb_color(c_index), /SET_BUTTON

     color_names=['(def)', '(u 1)', '(u 2)', '(u 3)' ]
     for i=0,3 do begin
       color_names(i)=strFormat(color_bar_names(i),22,/left) + $
       strFormat(color_names(i),6,/right)
       widget_control, widgetData.spb_color(i), $
         SET_VALUE=color_names(i)
     endfor

     widget_control, widgetData.dw_colorbar, get_value=windex
     wset, windex  & erase

     geom=widget_info(widgetData.dw_colorbar, /geometry)
     range=papco_get_Colorindices(c_index)
     n=range(1)-range(0)
     width=float(geom.xsize) / float(n)
     height=geom.ysize
     for i=0, n-1 do polyfill,[width*i, width*i, width*(i+1), width*(i+1)], $
                              [0, height, height, 0], color=range(0)+i, /device
   endif

;-- select correct slider for plot style widget -------------------------------
   if keyword_set(SELECT_PLOT_STYLE) AND has_PlotStyleEditor then BEGIN
       index = intarr(5)
       index(widgetData.ps_selectColor) = 1
       bases = ['ps_psym_base',   'ps_color_base', 'ps_thick_base', $
                'ps_lstyle_base', 'ps_symsize_base']
       FOR i = 0, 4 DO BEGIN
           exe_str = "widget_control, widgetData." + bases(i) + $
             " ,map=" + varprt(index(i))
           result = execute(exe_str)
       ENDFOR
   endif

;-- display the plot-style ----------------------------------------------------
   IF (KEYWORD_SET(PLOT_STYLE) Or KEYWORD_SET(SCATTER_TYPE)) $
       AND  has_PlotStyleEditor $
   THEN BEGIN
      PAPCO_SET_PLOT, 'X'
      WIDGET_CONTROL, widgetData.dw_ps_prev,  GET_VALUE=drawIndex
      WSET, drawIndex

      lstyle  = widgetData.LINESTYLE
      psym    = widgetData.PSYM
      thick   = widgetData.THICK
      color   = widgetData.COLOR
      symsize = widgetData.SYMSIZE

      case abs(psym) of
        8: begin
             if psym eq abs(psym) then psym=8 else psym=-8
             widgetData.userstyle=1
           end
        9: begin
             if psym eq abs(psym) then psym=8 else psym=-8
             widgetData.userstyle=2
           end
;      10: reserved by histogram mode
       else: widgetData.userstyle=0
      endcase

      xdata=(randomu(seed,10)-0.5)*2  &  ydata=(randomu(seed,10)-0.5)*2
      xr=[-1.2,1.2]

      if NOT widgetData.is_ScatterPlot then BEGIN
          xdata = findgen(10)*2*!PI/9  &  ydata = sin(xdata)
          xr=[0, 2*!PI]  &  yr = [-1.2, 1.2]
      endif

      if widgetData.userstyle ne 0 then begin
        get_user_symbol, widgetData.userstyle, userx, usery
        usersym, userx, usery, /fill
      endif

      erase & plot, xdata, ydata, xstyle=5, ystyle=5, symsize=symsize, $
        xrange=xr, xmargin=[0,0], yrange=yr, ymargin=[0,0], $
        yticks=0, xticks=0, $
        position=[0.1,0.1,0.9,0.9], psym=psym, thick=thick, linestyle=lstyle, $
        color=fixed_color_index(color)

   ENDIF

;-- update Add File widget ----------------------------------------------------
   if keyword_set(ADD_FILE) and has_add_file then begin

     case widgetData.status of
       0: BEGIN
           status_text = 'File NOT Found!'
           status_button = 'Create'
           WIDGET_CONTROL, widgetData.status_button, SENSITIVE=1
           WIDGET_CONTROL, widgetData.name_text, SENSITIVE=1
           WIDGET_CONTROL, widgetData.load_button, SENSITIVE=0
           WIDGET_CONTROL, widgetData.save_button, SENSITIVE=1
       end
       1: begin
           status_text = 'File Found!'
           status_button = 'Edit'
           WIDGET_CONTROL, widgetData.status_button, SENSITIVE=1
           WIDGET_CONTROL, widgetData.name_text, SENSITIVE=1
           WIDGET_CONTROL, widgetData.save_button, SENSITIVE=1
           WIDGET_CONTROL, widgetData.load_button, SENSITIVE=1
       end
       2: begin
           status_text = 'Directory NOT Found!'
           status_button = '----'
           WIDGET_CONTROL, widgetData.status_button, SENSITIVE=0
           WIDGET_CONTROL, widgetData.name_text, SENSITIVE=0
       end
       3: BEGIN
           status_text = 'Directory Found!'
           status_button = '----'
           WIDGET_CONTROL, widgetData.status_button, SENSITIVE=0
           WIDGET_CONTROL, widgetData.name_text, SENSITIVE=1
       end
       4: BEGIN                 ;no file name entered
           status_text = 'No File entered'
           status_button = '----'
           WIDGET_CONTROL, widgetData.status_button, SENSITIVE=0
           WIDGET_CONTROL, widgetData.name_text, SENSITIVE=1
       END
     endcase
     WIDGET_CONTROL, widgetData.status_text, SET_VALUE = status_text
     WIDGET_CONTROL, widgetData.status_button, SET_VALUE = status_button

   endif

;-- update Y-scale widget -----------------------------------------------------
IF keyword_set(YSCALE) AND has_yscale THEN BEGIN

    IF widgetData.manualYScaling EQ 1 THEN BEGIN
        WIDGET_CONTROL, widgetData.yscaleInput_Base, SENSITIVE=1
    ENDIF ELSE BEGIN
        WIDGET_CONTROL, widgetData.yscaleInput_Base, SENSITIVE=0
    ENDELSE

    ;get manual entry settings
    WIDGET_CONTROL, widgetData.ef_yscaleFrom, GET_VALUE=from
    WIDGET_CONTROL, widgetData.ef_yscaleTo,   GET_VALUE=to

    ; check limits, and if one zero or negative set linear scaling
    IF (from LE 0) OR (to LE 0) THEN widgetData.yLinLog = 0
    WIDGET_CONTROL, widgetData.mpb_ylinlog, SET_VALUE = widgetData.ylinlog

ENDIF

;-- update Z-scale widget -----------------------------------------------------
   if keyword_set(ZSCALE) and has_zscale then begin
     ;check if ef_yScaleLb exists
     tags = tag_names(widgetData) & pos = strpos(tags, 'EF_ZSCALELB')
     idx = where(pos NE -1, c)

     IF (widgetData.manualZScaling / 100) THEN BEGIN
       WIDGET_CONTROL, widgetData.ef_zScaleFrom, SENSITIVE=1
       WIDGET_CONTROL, widgetData.ef_zScaleTo, SENSITIVE=1
       WIDGET_CONTROL, widgetData.ef_zScaleFrom, set_value=widgetData.ZSCALEMIN
       WIDGET_CONTROL, widgetData.ef_zScaleTo,   set_value=widgetData.ZSCALEMAX
       IF c NE 0 THEN WIDGET_CONTROL, widgetData.ef_zScaleLb, SENSITIVE=0
     ENDIF ELSE BEGIN
       WIDGET_CONTROL, widgetData.ef_zScaleFrom, SENSITIVE=0
       WIDGET_CONTROL, widgetData.ef_zScaleTo, SENSITIVE=0
       IF c NE 0 THEN WIDGET_CONTROL, widgetData.ef_zScaleLb, SENSITIVE=1
     ENDELSE

     IF widgetData.zLinLog THEN BEGIN
       WIDGET_CONTROL, widgetData.pb_zLog, /SET_BUTTON
     ENDIF ELSE BEGIN
       WIDGET_CONTROL, widgetData.pb_zLin, /SET_BUTTON
     ENDELSE

   endif

;-- update Panel description widget -------------------------------------------
IF keyword_set(DESCR) AND has_descr THEN BEGIN
    IF widgetData.descr THEN $
      WIDGET_CONTROL, widgetData.descr_entry, SENSITIVE=1 $
      ELSE WIDGET_CONTROL, widgetData.descr_entry, SENSITIVE=0
    IF widgetData.ylbl THEN $
      WIDGET_CONTROL, widgetData.ylbl_entry, SENSITIVE=1 $
      ELSE WIDGET_CONTROL, widgetData.ylbl_entry, SENSITIVE=0
    IF widgetData.zlbl THEN $
      WIDGET_CONTROL, widgetData.zlbl_entry, SENSITIVE=1 $
      ELSE WIDGET_CONTROL, widgetData.zlbl_entry, SENSITIVE=0

    wid_ids = [widgetData.rs_base, widgetData.ls_base, widgetData.z_base]
    CASE widgetData.descr_type OF
        0: map = [1, 0, 0]
        1: map = [0, 1, 0]
        2: map = [0, 0, 1]
    ENDCASE
    FOR i = 0, n_elements(map)-1 DO widget_control, wid_ids(i), MAP = map(i)

ENDIF

;-- update data Units widget --------------------------------------------------
   if keyword_set(UNITS) and has_units then begin
     WIDGET_CONTROL, widgetData.mpb_unit,  set_value = widgetData.dataunits
   endif

;-- update Variable select widgets --------------------------------------------
IF keyword_set(VAR_SEL) and has_var_sel THEN BEGIN
    IF VAR_SEL EQ 'ALL' THEN var_list = var_names ELSE var_list = [VAR_SEL]
    FOR i = 0, n_elements(var_list)-1 DO BEGIN
        i1 = where(names EQ var_list(i)+'_SELECT_SET_BASE')
        i2 = where(names EQ var_list(i)+'_SELECT_RANGE')
        i3 = where(names EQ var_list(i)+'_SELECT_FROM_LBL')
        i4 = where(names EQ var_list(i)+'_SELECT_FROM_VAL')
        i5 = where(names EQ var_list(i)+'_SELECT_TO_LBL')
        i6 = where(names EQ var_list(i)+'_SELECT_TO_VAL')
        i7 = where(names EQ var_list(i)+'_SELECT_FRMT')
        f = widgetData.(i7(0))
        WIDGET_CONTROL, widgetData.(i1(0)), SENSITIVE = widgetData.(i2(0))
        out=strformat('Low '+var_list(i)+': '+ $
            string(widgetData.(i4(0)), FORMAT=f),35, /CENTER)
        WIDGET_CONTROL, widgetData.(i3(0)), SET_VALUE = out
        out=strformat('High '+var_list(i)+': '+ $
            string(widgetData.(i6(0)), FORMAT=f),35, /CENTER)
        WIDGET_CONTROL, widgetData.(i5(0)), SET_VALUE = out
    ENDFOR
ENDIF

;-- update Variable average widgets -------------------------------------------
IF keyword_set(VAR_AVE) and has_var_ave THEN BEGIN
    IF VAR_AVE EQ 'ALL' THEN var_list = ave_names ELSE var_list = [VAR_AVE]
    FOR i = 0, n_elements(var_list)-1 DO BEGIN

        i3 = where(names EQ var_list(i)+'_AVERAGE_FROM_LBL')
        i4 = where(names EQ var_list(i)+'_AVERAGE_FROM_VAL')
        i5 = where(names EQ var_list(i)+'_AVERAGE_TO_LBL')
        i6 = where(names EQ var_list(i)+'_AVERAGE_TO_VAL')
        i7 = where(names EQ var_list(i)+'_AVERAGE_FRMT')
        f = widgetData.(i7(0))

        out=strformat('Low '+var_list(i)+': '+ $
            string(widgetData.(i4(0)), FORMAT=f),35, /CENTER)
        WIDGET_CONTROL, widgetData.(i3(0)), SET_VALUE = out
        out=strformat('High '+var_list(i)+': '+ $
            string(widgetData.(i6(0)), FORMAT=f),35, /CENTER)
        WIDGET_CONTROL, widgetData.(i5(0)), SET_VALUE = out
    ENDFOR
ENDIF

;-- update Mag Coords widgets -------------------------------------------------
IF keyword_set(MAG_COORD) and has_mag_coord THEN BEGIN
    IF (widgetData.ticks EQ 0) AND $
       (widgetData.mag_coord_ids(0)) NE -1 THEN BEGIN
        IF widgetData.l_calc THEN BEGIN   ;see if L* is also allowed
            widget_control, widgetData.mag_coord_ids(1), sensitive = 1
            widget_control, widgetData.pitch_plot_base, sensitive = 1
            widget_control, widgetData.mag_pitch_base, sensitive = 1
        ENDIF ELSE BEGIN
            widget_control, widgetData.mag_coord_ids(1), sensitive = 0
            widget_control, widgetData.pitch_plot_base, sensitive = 0
            widget_control, widgetData.mag_pitch_base, sensitive = 0
        ENDELSE
        ;make sure pitch angle L requested is also set to calc
        WIDGET_CONTROL,widgetData.mag_pitch_base,GET_VALUE = set
        set(widgetData.pitch_plot) = 1
        widgetData.pitch_proc = 0
        FOR i = 0, n_elements(set)-1 DO IF set(i) THEN $
          widgetData.pitch_proc = widgetData.pitch_proc+2^i
        WIDGET_CONTROL,widgetData.mag_pitch_base,SET_VALUE = set
    ENDIF
ENDIF

END

;******************************************************************************
PRO papco_panel_exit, widgetData

names=TAG_NAMES(widgetData)

;get the y-values from entry fields. Do only if Tags exist!
index=where(names eq 'EF_YSCALEFROM',count)
if count ne 0 then begin
     WIDGET_CONTROL, widgetData.ef_yscaleFrom, GET_VALUE=from
     WIDGET_CONTROL, widgetData.ef_yscaleTo, GET_VALUE=to
     widgetData.yscaleMin=(from(0)) & widgetData.yscaleMax=(to(0))
     index=where(names eq 'EF_YSCALELB',c)
     IF c NE 0 THEN BEGIN
         WIDGET_CONTROL, widgetData.ef_yscaleLb, GET_VALUE=lb
         widgetData.yscaleLb=(lb(0))
     ENDIF
     if (widgetData.manualYScaling ge 0) and $
       ( widgetData.yLinLog eq 1 ) then begin
       if (widgetData.yscaleMin le 0) or $
          (widgetData.yscaleMax le 0) then begin
           info=WIDGET_MESSAGE(['% papco_panel: ', $
            'invalid Y-range for log scale'])
         ;return, widgetData
       endif
     endif
endif

;get the z-values from entry fields. Do only if Tags exist!
index=where(names eq 'EF_ZSCALEFROM',count)
   if count ne 0 then begin
     WIDGET_CONTROL, widgetData.ef_zscaleFrom, GET_VALUE=from
     WIDGET_CONTROL, widgetData.ef_zscaleTo, GET_VALUE=to
     index=where(names eq 'EF_ZSCALELB',c)
     IF c NE 0 THEN WIDGET_CONTROL, widgetData.ef_zscaleLb, GET_VALUE=lb
     widgetData.zscaleMin=FLOAT(from(0))
     widgetData.zscaleMax=FLOAT(to(0))
     IF c NE 0 THEN widgetData.manualzscaling = $
       (widgetData.manualzscaling /100) * 100 + lb(0)
     if (widgetData.manualZScaling eq 1) and $
       ( widgetData.zLinLog eq 1 ) then begin
       if (widgetData.zscaleMin le 0) or $
         (widgetData.zscaleMax le 0) then begin
         info=WIDGET_MESSAGE(['% papco_panel: ', $
             'invalid Z-range for log scale'])
         ;return, widgetData
       endif
     endif
endif

;get RS panel description from entry field. Do only if Tags exist!
index=where(names EQ 'DESCR_ENTRY',count)
IF count NE 0 THEN BEGIN
    IF widgetdata.descr THEN BEGIN
        WIDGET_CONTROL, widgetdata.descr_entry, GET_VALUE=descr_entry
        descr_str=descr_entry(0)
        IF n_elements(descr_entry) GT 1 THEN $
          FOR i=1,n_elements(descr_entry)-1 DO $
            descr_str=descr_str+'!C'+descr_entry(i)
        widgetdata.descr_str=descr_str
    ENDIF
ENDIF

;get LS panel description from entry field. Do only if Tags exist!
index=where(names EQ 'YLBL_ENTRY',count)
IF count NE 0 THEN BEGIN
    IF widgetdata.ylbl THEN BEGIN
        WIDGET_CONTROL, widgetdata.ylbl_entry, GET_VALUE=descr_entry
        descr_str=descr_entry(0)
        IF n_elements(descr_entry) GT 1 THEN $
          FOR i=1,n_elements(descr_entry)-1 DO $
            descr_str=descr_str+'!C'+descr_entry(i)
        widgetdata.ylbl_str=descr_str
    ENDIF
ENDIF

;get Z description from entry field. Do only if Tags exist!
index=where(names EQ 'ZLBL_ENTRY',count)
IF count NE 0 THEN BEGIN
    IF widgetdata.zlbl THEN BEGIN
        WIDGET_CONTROL, widgetdata.zlbl_entry, GET_VALUE=descr_entry
        descr_str=descr_entry(0)
        IF n_elements(descr_entry) GT 1 THEN $
          FOR i=1,n_elements(descr_entry)-1 DO $
            descr_str=descr_str+'!C'+descr_entry(i)
        widgetdata.zlbl_str=descr_str
    ENDIF
print, widgetdata.zlbl_str
ENDIF

;get panel extensions. Do only if Tags exist!
index=where(names EQ 'DESCR_EXTEND',count)
IF count NE 0 THEN BEGIN
    WIDGET_CONTROL, widgetdata.ef_ext_desc, GET_VALUE=x
    widgetdata.descr_extend = x
    WIDGET_CONTROL, widgetdata.ef_ext_ylbl, GET_VALUE=x
    widgetdata.ylbl_extend = x
ENDIF

END

;******************************************************************************
;* PROCEDURE:
;*      FUNCTION papco_panel_Event, event, widgetData
;*
;* DESCRIPTION:
;*      This routine is called by the panel-editor widget-routines,
;*	to process events for the default widgets.
;*	How this should be done is demonstrated in the file
;*	$papco_PATH/papco_panel_template.pro
;*
;* INPUTS:
;*	event	 	an XMANAGER-event
;*			The event that invoked 'papco_panel_Event'
;*
;*	widgetData	a structure
;*			This is a structure containing at last the
;*			elements for a papco_panel_EDIT-structure
;*			(this is no really existing but a virtual structure
;*			- see the top of this file for its description)
;*
;* KEYWORDS:
;*     none
;*
;* CALLING SEQUENCE:
;*      called by XMANAGER-routines of panel-editors
;*
;* MODIFICATION HISTORY:
;*     written august 1995, Andreas Keese
;*     modified july 1997, Haje Korth, Thickness and Linestyle added
;******************************************************************************
FUNCTION papco_panel_Event, event, widgetData

COMMON config, configkind
COMMON PLOT_COMPOSER, main_widgetData

WIDGET_CONTROL, GET_UVALUE=uval, event.id

forward_FUNCTION papco_pickfile

;test uvalue for structure - select variable sliders return structure uvals!
r = size(uval, /st)
IF r.type_name EQ 'STRUCT' THEN BEGIN
    uval_str = uval
    uval = uval.uvalue
ENDIF

;test for uval begin from var select widget
pos = strpos(uval, '_SELECT_')
IF pos NE -1 THEN BEGIN
    name = strmid(uval, 0, pos)
    ending = strmid(uval, pos+8, strlen(uval))
    uval = 'CW_VAR_SEL_'+ending
END

;test for uval begin from var average widget
pos = strpos(uval, '_AVERAGE_')
IF pos NE -1 THEN BEGIN
    name = strmid(uval, 0, pos)
    ending = strmid(uval, pos+9, strlen(uval))
    uval = 'CW_VAR_AVE_'+ending
END

IF strleft(uval, 9) EQ 'spb_color' THEN BEGIN
    widgetData.channel=fix(strfrom(uval, 9))
    papco_panel_Refresh, widgetData, /COLBAR_STYLE
    goto, done
ENDIF

tags = tag_names(widgetData)

CASE uval OF

    'spb_dynamic': widgetData.mag_dyn = event.select
    'spb_mag_coord': widgetData.mag_coords = event.value
    'spb_mag_proc' : widgetData.mag_proc = event.value
    'spb_mag_pitch' : BEGIN
        WIDGET_CONTROL,widgetData.mag_pitch_base,GET_VALUE = set
        widgetData.pitch_proc = 0
        FOR i = 0, n_elements(set)-1 DO IF set(i) THEN $
          widgetData.pitch_proc = widgetData.pitch_proc+2^i
    END
    'spb_pitch_plot' : BEGIN
        widgetData.pitch_plot = event.value
        papco_panel_Refresh, widgetData, /MAG_COORD
    END

    'spb_lvalues': BEGIN
        widgetData.l_calc = event.value
        papco_panel_Refresh, widgetData, /MAG_COORD
    END

    'mpb_mag_coord':BEGIN
        widget_control, event.id, get_value = set
        widgetData.mag_coords = 0
        FOR i=0,n_elements(set)-1 DO IF set(i) THEN $
          widgetData.mag_coords = widgetData.mag_coords+2^i
        papco_panel_Refresh, widgetData, /MAG_COORD
    END
    'spb_int': BEGIN
        widgetData.internal_model = event.index
    END
    'spb_ext': BEGIN
        widgetData.external_model = event.index
    END


    'CW_VAR_SEL_FROM': BEGIN
        idx = where(tags EQ name+'_SELECT_FROM_VAL')
        widgetData.(idx(0)) = event.value * uval_str.vr + uval_str.v1
        papco_panel_Refresh, widgetData, VAR_SEL = name
    END
    'CW_VAR_SEL_TO': BEGIN
        idx = where(tags EQ name+'_SELECT_TO_VAL')
        widgetData.(idx(0)) = event.value * uval_str.vr + uval_str.v1
        papco_panel_Refresh, widgetData, VAR_SEL = name
    END
    'CW_VAR_SEL_RANGE' : BEGIN
        idx = where(tags EQ name+'_SELECT_'+ending)
        widgetData.(idx(0)) = event.value
        papco_panel_Refresh, widgetData, VAR_SEL = name
    END


    'CW_VAR_AVE_FROM': BEGIN
        idx = where(tags EQ name+'_AVERAGE_FROM_VAL')
        widgetData.(idx(0)) = event.value * uval_str.vr + uval_str.v1
        papco_panel_Refresh, widgetData, VAR_AVE = name
    END
    'CW_VAR_AVE_TO': BEGIN
        idx = where(tags EQ name+'_AVERAGE_TO_VAL')
        widgetData.(idx(0)) = event.value * uval_str.vr + uval_str.v1
        papco_panel_Refresh, widgetData, VAR_AVE = name
    END


    'mpb_yautman': BEGIN
        widgetData.manualYScaling = event.value
        papco_panel_Refresh, widgetData, /YSCALE
    END

    'mpb_ylinlog': BEGIN
        widgetData.yLinLog = event.value
        papco_panel_Refresh, widgetData, /YSCALE
    END

    'pb_zScaleManual' : BEGIN
         widgetData.manualZScaling=100
         ; check limits, and if one zero or negative set linear
         WIDGET_CONTROL, widgetData.ef_zscaleFrom, GET_VALUE=from
         WIDGET_CONTROL, widgetData.ef_zscaleTo,   GET_VALUE=to
         Min=FLOAT(from(0))
         Max=FLOAT(to(0))
         if (Min le 0) or (Max le 0) then widgetData.zLinLog=0
         papco_panel_Refresh, widgetData, /ZSCALE
    END

    'pb_zScaleAuto' : BEGIN
      widgetData.manualZScaling=0
      papco_panel_Refresh, widgetData, /ZSCALE
    END

    'pb_zLog' : BEGIN
         ; check if manual scaling is set. If so, don't allow
         ; log scale if either limit is zero or negative
         WIDGET_CONTROL, widgetData.ef_zscaleFrom, GET_VALUE=from
         WIDGET_CONTROL, widgetData.ef_zscaleTo, GET_VALUE=to
         widgetData.zscaleMin=FLOAT(from(0))
         widgetData.zscaleMax=FLOAT(to(0))
         if (widgetData.manualZScaling eq 1) then begin
           if (widgetData.zscaleMin le 0) or $
              (widgetData.zscaleMax le 0) then widgetData.zLinLog=0 $
           else widgetData.zLinLog=1
         endif else begin
           widgetData.zLinLog=1
         endelse
         papco_panel_Refresh, widgetData, /ZSCALE
    END

    'pb_zLin' : BEGIN
         widgetData.zLinLog=0
         papco_panel_Refresh, widgetData, /ZSCALE
    END

    'pb_help': BEGIN
         done = 'Click to return to '+widgetData.panelKind+' panel editor'
         xdisplayfile, papco_module_helpfile(widgetData.helpfile), $
                       title=widgetData.panelKind+' - Online Help',$
                       group=event.top, height = 48, /modal, $
                       DONE_BUTTON = done
    END

    'pb_config': BEGIN
      papco_Input, /DISABLE
      papco_module_configure, configkind
      papco_Input, /ENABLE
    END

    'pb_datacache': begin
        papco_data_cache_tool, widgetData.datacache, $
          /modal, group_leader=event.top
    end

    'pb_cancel': BEGIN
         WIDGET_CONTROL, event.top, /DESTROY
         widgetData.canceled=1
    END

    'pb_done': BEGIN
        papco_panel_exit, widgetData
        widgetData.canceled=0
        WIDGET_CONTROL, event.top, /DESTROY
    END

    'pb_done_and_cont': BEGIN
        papco_panel_exit, widgetData
        widgetData.canceled=2
        WIDGET_CONTROL, event.top, /DESTROY
    END

    'sl_height' : begin
      widgetData.panelHeight=event.value
      WIDGET_CONTROL, widgetData.sl_label, $
        SET_VALUE=varprt(widgetData.panelHeight)
    end

    'spb_ps_type' : BEGIN
        widgetData.ps_selectColor = event.value
        papco_panel_Refresh, widgetData, /SELECT_PLOT_STYLE
    end

    'sl_ps_psym' : BEGIN
        widgetData.PSYM = event.value
        papco_panel_Refresh, widgetData, /PLOT_STYLE
    END

    'sl_ps_thick' : BEGIN
        widgetData.THICK = event.value
        papco_panel_Refresh, widgetData, /PLOT_STYLE
    END

    'sl_ps_lstyle' : BEGIN
        widgetData.LINESTYLE = event.value
        papco_panel_Refresh, widgetData, /PLOT_STYLE
    END

    'sl_ps_symsize': BEGIN
        widgetData.SYMSIZE = event.value
        papco_panel_Refresh, widgetData, /PLOT_STYLE
    end

    'dw_plotColor' : BEGIN
        geom=widget_info(widgetData.dw_plotColor, /geometry)
        n=widgetData.ps_numberOfColors
        width=float(geom.xsize) / float(n)
        widgetData.COLOR=event.x/width
        papco_panel_Refresh, widgetData, /PLOT_STYLE
    END

    'file_browse' : BEGIN
      if papco_hasdir(widgetData.path) then begin
        fname=dialog_pickfile(GROUP=widgetData.base, GET_PATH=getpath, $
                              _EXTRA = widgetData.extra)

        if fname eq '' then return, widgetData

        papco_sepdir, fname, dir, file, ext
        widgetData.path=getpath
        widgetData.extra.path = getpath
        WIDGET_CONTROL,widgetData.direc_text, SET_VALUE=getpath
        WIDGET_CONTROL,widgetData.direc_text, SET_TEXT_SELECT=[strlen(getpath)]

        if file eq '' then return, widgetData

        ;see if FILTER is set, If so, check filename to make
        ;sure it has extension of FILTER
        IF keyword_set(widgetData.extra.FILTER) THEN BEGIN
            filter = widgetData.extra.FILTER
            filter_ext = strmid(filter, 2, strlen(filter))
            IF ext NE filter_ext THEN ext = filter_ext
        ENDIF

        widgetData.filename=file+'.'+ext
        WIDGET_CONTROL,widgetData.name_text, SET_VALUE=widgetData.filename
        WIDGET_CONTROL,widgetData.name_text, $
          SET_TEXT_SELECT=[strlen(widgetData.filename)]
        widgetData.status=1
        papco_panel_Refresh, widgetData, /ADD_FILE
      endif else begin
        widgetData.status=2
        papco_panel_Refresh, widgetData, /ADD_FILE
        return, widgetData
    ENDELSE
    END

    'fil_delete_spb': BEGIN
        WIDGET_CONTROL, widgetData.direc_text, GET_VALUE=path & path=path(0)
        WIDGET_CONTROL, widgetData.name_text, GET_VALUE=name & name=name(0)
        result = findfile(path+name, count = c)
        IF c NE 0 THEN BEGIN
            spawn, 'rm '+path+name
            print, '% PAPCO_PICKFILE: file deleted'
            widgetData.filename = ''
            WIDGET_CONTROL, widgetData.name_text, SET_VALUE=''
            widgetData.status=0
            papco_panel_Refresh, widgetData, /ADD_FILE
        ENDIF ELSE BEGIN
            print, '% PAPCO_PICKFILE: file not found, cannot delete'
        ENDELSE
    END

    'dir_delete_spb': BEGIN
        WIDGET_CONTROL, widgetData.direc_text, GET_VALUE=path & path=path(0)
        IF papco_finddir(path) THEN BEGIN
            spawn, 'rm -r '+path
            print, '% PAPCO_PICKFILE: directory deleted'
            path = strmid(path, 0, strlen(path)-1)
            pos = strpos(path, '/', /REVERSE_SEARCH)
            path = strmid(path, 0, pos+1)
            WIDGET_CONTROL, widgetData.direc_text, SET_VALUE=path
            widgetData.path=path
            widgetData.extra.path = path
            widgetData.status=3
            papco_panel_Refresh, widgetData, /ADD_FILE
        ENDIF  ELSE BEGIN
            print, '% PAPCO_PICKFILE: directory not found, cannot delete'
        ENDELSE
    END

    'dir_create_spb': BEGIN
        WIDGET_CONTROL, widgetData.direc_text, GET_VALUE=path & path=path(0)
        IF papco_finddir(path) THEN BEGIN
            print, '% PAPCO_PICKFILE: directory already exists'
        ENDIF  ELSE BEGIN
            spawn, 'mkdir '+path
            WIDGET_CONTROL, widgetData.direc_text, SET_VALUE=path
            widgetData.path=path
            widgetData.extra.path = path
            print, '% PAPCO_PICKFILE: directory created'
            widgetData.status=3
            papco_panel_Refresh, widgetData, /ADD_FILE
        ENDELSE
    END

    'file_name' : BEGIN
        ;check if directory exists
        WIDGET_CONTROL, widgetData.direc_text, GET_VALUE=path & path=path(0)
        IF NOT papco_finddir(path) THEN BEGIN
          widgetData.status=2 & papco_panel_Refresh, widgetData, /ADD_FILE
          papco_create_dir, path
          IF path NE '' THEN BEGIN
              widgetData.status=3 & papco_panel_Refresh, widgetData, /ADD_FILE
          ENDIF
        ENDIF
        WIDGET_CONTROL, widgetData.name_text, GET_VALUE=name & name=name(0)
        if name eq '' then BEGIN ;no file name entered
            widgetData.status=4 & papco_panel_Refresh, widgetData, /ADD_FILE
        ENDIF

        papco_sepdir, name, dir, file, ext

        ;see if FILTER is set, If so, check filename to make
        ;sure it has extension of FILTER
        IF keyword_set(widgetData.extra.FILTER) THEN BEGIN
            filter = widgetData.extra.FILTER
            filter_ext = strmid(filter, 2, strlen(filter)-2)
            IF ext NE filter_ext THEN ext = filter_ext
            name = file+'.'+ext
            WIDGET_CONTROL, widgetData.name_text, SET_VALUE = name
        ENDIF

        ;check if file exists
        file = findfile(path+name, count = c)
        IF c EQ 0 THEN BEGIN
            widgetData.status = 0 & papco_panel_Refresh, widgetData, /ADD_FILE
        ENDIF ELSE BEGIN
            widgetData.status = 1 & papco_panel_Refresh, widgetData, /ADD_FILE
        ENDELSE
        widgetData.path=path
        widgetData.filename=name
        widgetData.extra.path = path
    END

    'status_button' : BEGIN
      papco_panel_AddFile_EditFile, widgetData
      papco_panel_Refresh, widgetData, /ADD_FILE
    END

    'mpb_unit': BEGIN
      widgetData.dataUnits=event.value
    END

    'spb_descr': BEGIN
        widgetData.descr_type=event.value
        papco_panel_Refresh, widgetData, /DESCR
    END

    'mpb_descr': BEGIN
      widgetData.descr=event.value
      papco_panel_Refresh, widgetData, /DESCR
    END
    'descr_entry': BEGIN  &  END
    'ef_ext_desc': widgetData.descr_extend=event.value

    'mpb_ylbl': BEGIN
      widgetData.ylbl=event.value
      papco_panel_Refresh, widgetData, /DESCR
    END
    'ylbl_entry': BEGIN  &  END
    'ef_ext_ylbl': widgetData.ylbl_extend=event.value

    'mpb_zlbl': BEGIN
      widgetData.zlbl=event.value
      papco_panel_Refresh, widgetData, /DESCR
    END
    'zlbl_entry': BEGIN  &  END

    'spb_nogap' : widgetData.nogap = event.select

    'spb_ba_option' : BEGIN
        widgetData.ba_option=event.value
    END

    'spb_ba_force' : BEGIN
        widgetData.ba_force = event.select
    END

    'ef_ba_from' : BEGIN
      widgetData.ba_from = event.value
    END

    'spb_dobatch': begin
      widget_control, /hourglass
      ;make options for the procedure call
      redo = widgetData.ba_option
      force = widgetData.ba_force
      proc_call=widgetData.make_batch_call
      panelkind=widgetData.panelkind
      IF widgetData.ba_from NE '' THEN from = widgetData.ba_from ELSE from = 0
      if main_widgetData.default_trap eq 1 then $ ;see if error trapping is on
        catch, error_status else error_status=0
      if error_status ne 0 then begin    ;call error handler
        papco_error_handler, $
          error_status, proc_call, panelkind, panelVector, /NO_ERR_DRAW
        error_status=0 & catch, /CANCEL  ;disable error catching
        goto, done
      endif
      CALL_PROCEDURE, proc_call, widgetData, batch_times, filename,  $
        REDO = redo,  FORCE = force, FROM = from, /verb
      catch, /CANCEL            ;disable error catching

      main_widgetData.BATCHFILENAME=filename

      papco_edit_batch, FILENAME=filename, $
        BATCH_TIMES=batch_times, GROUP=main_widgetdata.base
    end

    ELSE : BEGIN
      message, 'EVENT NOT HANDLED !!!', /cont
      help, /struct, event
    END

ENDCASE
done:
return, widgetData

END
