;******************************************************************************
;******************************************************************************
;* FILE:	 
;*    $papco_PATH/papco_plotEdit.pro
;* 
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:  
;*     contains routines for adding, deleting, modifying plots.
;*
;* FILES BELONGING TO THE MAIN-WINDOW:
;*
;*      papco.pro        
;*         - main-window widget-construction
;*         - main-window event-handlers
;*
;*      papco_plotedit.pro
;*         - adding and editing plotinfos
;*         - deleting plotinfos
;*         - routine for obtaining verbose description of plotinfos
;*
;*      papco_files.pro  
;*         - procedures for saving and loading plotinfos
;*         - procedures for writing and reading default-settings
;*         - batch-mode
;*
;*      papco_settime.pro
;*         - procedures for selecting an orbit and time-range
;*         - procedure for displaying time-range when using time-slider
;*
;*      papco_draw.pro
;*         - routines for outputting plots.
;*
;*      papco_cursor.pro
;*         - routines for handling the mouse-cursor
;*         - routines for zooming into plots
;*         - routines for doing slices of plots
;*
;*      papco_edit_type.pro
;*         - changing the plot-type of a plotinfo
;*
;*    Those procedures are contained:
;*
;*  	FUNCTION papco_DescriptionFor, plotInfo, totalHeight
;*	    -> return a verbose description of a plotinfo
;*	   
;*	PRO papco_setPlotData, aPanelResult, atPosition
;*	    -> copy the data in 'aPanelResult' to the plot at 'atPosition'
;*
;*      PRO papco_EditPlot, plotNumber, CHANGETYPE=CHANGETYPE
;*	    -> invoke the panel-editor for the plotNumber-th plot 
;*
;*	PRO papco_DeletePlot, plotNumber
;*	    -> remove one plot from the list
;*
;*	PRO papco_Check_DEPLO_Time, newPlotData
;*	    -> if a DEPLO-plot was added, check if its time-range shall be used
;*
;*      PRO papco_Add_Plot, panel_name, panelKind, $
;*			    INITIALPLOTINFO=initialPlotInfo
;*	    -> add a new plot
;*
;* MODIFICATION HISTORY:       
;*     august,september 1995, written by A.Keese 
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      papco_DescriptionFor, plotInfo, totalHeight
;* 
;* DESCRIPTION:  
;*	Create a verbose description of the plot specified by plotInfo.
;*	Each plot that is listed in the plot-descriptions - listbox
;*	is described by two lines of text. The first line contains 
;*	a number identifying the plot and the type of the plot (that
;*	is plotInfo.panelKind).
;*	The second line of the description is created by this function.
;*	
;* INPUTS:       
;*	plotInfo	a papco_PLOTINFO-structure
;*			the plot for which a description is wanted
;*
;*	totalHeight	a positive integer
;* 			the descriptions contain the panelplot-vector. 
;*			In order to construct it, the total height of 
;*			the plots is needed. 
;* KEYWORDS:
;* 	none
;*
;* OUTPUT:	 
;*      a string containg the plot-description
;* 
;* CALLING SEQUENCE:
;*	descript=papco_DescriptionFor(aPlotInfoStruct, anInteger)
;*
;* MODIFICATION HISTORY:       
;*     written august, september 1995, Andreas Keese
;***************************************************************************
FUNCTION papco_DescriptionFor, plotInfo, totalHeight

COMMON ADDINS, plot_types
COMMON PLOT_COMPOSER, widgetData

descriptStr=''

; call the routine which gives the descriptor for the panel.
index = where(strchminus(PlotInfo.panelKind) EQ plot_types, count)
IF count ne 0 then BEGIN
    plot_type=plot_types(index(0))
    descriptor_call='descriptor_'+plot_type
    CALL_PROCEDURE, descriptor_call, plotInfo,descriptStr
ENDIF

return, '    '+descriptStr

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_setPlotData, aPanelResult, atPosition
;* 
;* DESCRIPTION:  
;*	This procedure copies the data from a structure that contains some 
;*      or all of the tags that are contained in a PLOTInfo-structure to the 
;*	plot at position 'atPosition', i.e. widgetData.plotInfos(atPosition)
;*
;* INPUTS:       
;*	aPanelResult	a structure.
;*			This structure must contain all fields, that 
;*			the papco_PLOTINFO-structure contains. It MAY contain
;* 			additional fields.
;*
;*	atPosition	a non-negative integer
;*			0<=atPosition<=N_ELEMENTS(widgetData.plotInfos)
;*			the index of 'widgetData.plotInfos', in which the
;*			fields of aPanelResult are to be copied.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	papco_setPlotData, aStructure, anInt
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_setPlotData, aPanelResult, atPosition

COMMON PLOT_COMPOSER, widgetData

dummy=papco_getPlotInfoStruct()

plotInfo_tags=TAG_NAMES(dummy)
np_tags=N_ELEMENTS(plotInfo_tags)
result_tags=TAG_NAMES(aPanelResult)
  
FOR i=0, N_TAGS(aPanelResult)-1 DO BEGIN
    sTag=result_tags(i)
    found=-1
    seek=0
    REPEAT BEGIN
      IF plotInfo_tags(seek) EQ sTag THEN found=seek
      seek=seek+1
    ENDREP UNTIL seek GE np_tags OR found GT -1
    IF found GT -1 THEN begin    
      widgetData.plotInfos(atPosition).(found)=aPanelResult.(i)
    endif  
ENDFOR

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_EditPlot, plotNumber
;* 
;* DESCRIPTION:  
;*      Edit an existing Plot.
;*	According to the plot's panelKind-field, the panel editor is 
;*	opened, in which the plot may be changed.
;*	Afterwards, the display is refreshed.
;*
;* INPUTS:       
;*	plotNumber	anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*			the index of the plot in widgetData.plotInfos.
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	papco_EditPlot, anInteger
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_EditPlot, plotNumber, CHANGETYPE=CHANGETYPE

COMMON PLOT_COMPOSER, widgetData
COMMON CONFIG, configkind
COMMON ADDINS, plot_types
COMMON ADDSUB, sorted_modules ;sorted module names and no of submenu entries
  
newPlotData={canceled:1}
old=widgetData.plotInfos(plotNumber)
r = size(old.USR_PTR1, /st)
IF r.TYPE_NAME EQ 'POINTER' THEN old_ctr = *old.USR_PTR1

oldPanelKind=old.panelKind

IF KEYWORD_SET(CHANGETYPE) THEN BEGIN
    newType=papco_Edit_Type(widgetData.plotInfos(plotNumber),$
                            GROUP=widgetData.main_base)
    IF newType.canceled THEN RETURN
    widgetData.plotInfos(plotNumber).panelKind=newType.panelKind
ENDIF

plotInfo=widgetData.plotInfos(plotNumber)
  
; set the overall configkind for this panel  
panelkind=strchminus(PlotInfo.panelKind)
plot_type_map=findgen(n_elements(plot_types))
i1=0 & i2=0
for i=0, n_elements(sorted_modules)-1 do begin
    i2=i2+sorted_modules(i).n_submenu
    plot_type_map(i1:i2-1)=i
    i1=i2
endfor  
index=where(panelkind eq plot_types,cc)
if cc eq 0 then begin         ;module for this plot type is not loaded
    result=messageBox(['Module for plot type "'+panelKind+'"', $
	               'does not seem loaded - cannot edit this panel' ],  $
		      ['Cancell edit'], /center, title='Module not loaded?')
    return  
endif  
configkind=sorted_modules(plot_type_map(index(0))).name
  
; call the function of the panel editor for this panel Kind.
widget_control, /HOURGLASS
panel_editor_name = plotInfo.panelKind+'_panel'

;check for size of panel, see if scrolling is needed...

;see if size is already determined for this session.
idx = where(plot_types EQ panelKind)
IF widgetData.panel_editor_size(idx(0), 0) NE 0 THEN BEGIN
    SCR_XSIZE = widgetData.panel_editor_size(idx(0), 0)
    SCR_YSIZE = widgetData.panel_editor_size(idx(0), 1)
    GOTO, have_size
ENDIF 

size = 1
catch, error_status
if error_status ne 0 then begin  ;call error handler
    error_status=0 & catch, /CANCEL ;disable error catching
;    message, 'Cannot calculate size of '+panel_editor_name, /cont

    goto, skip_get_panel_size
ENDIF 

newPlotData=CALL_FUNCTION(panel_editor_name, plotInfo, $
                          ACTION='Add '+plotInfo.panelKind, $
                          GROUP=widgetData.main_base, $
                          SIZE_ONLY = size)

error_status=0 & catch, /CANCEL ;disable error catching
SCR_XSIZE = size.SCR_XSIZE & SCR_YSIZE = size.SCR_YSIZE 
widgetData.panel_editor_size(idx(0),0) = SCR_XSIZE
widgetData.panel_editor_size(idx(0),1) = SCR_YSIZE

;if call was succesfull, this feature is supported
have_size:

scroll = 0 & xsize = 0 & ysize = 0
IF SCR_YSIZE GT widgetData.screensize(1) THEN BEGIN
    scroll = 1
    xsize = SCR_XSIZE+20
    ysize = widgetData.screensize(1)-200
        
    newPlotData=CALL_FUNCTION(panel_editor_name, plotInfo, $
                              ACTION='Update '+plotInfo.panelKind, $
                              GROUP=widgetData.main_base, $
                              SCROLL = scroll, $
                              X_SCROLL_SIZE = xsize, $
                              Y_SCROLL_SIZE = ysize )        

    GOTO, done_panel

ENDIF

skip_get_panel_size:

newPlotData = CALL_FUNCTION(panel_editor_name, plotInfo, $
                            ACTION='Update '+plotInfo.panelKind, $
                            GROUP=widgetData.main_base)

done_panel:

IF newPlotData.canceled THEN BEGIN
    widgetData.plotInfos(plotNumber).panelKind=oldPanelKind
    return
ENDIF


;update the plotinfo
papco_SetPlotData, newPlotData, plotNumber
    
; see what type of edit this was (widgetData.edit_choice)
;   0 : edit only this local panel
;   1 : perform the same edits on all panels of the same type
;   2 : perform the same edits on all panels. 

;check for usr_ptr1 tag - update those too where!

CASE widgetData.edit_choice OF
    1: BEGIN 
    END 
    2: BEGIN
        new=widgetData.plotInfos(plotNumber)
        old_tags=TAG_NAMES(old)
            
        ;test all tags
        FOR i=0,n_elements(old_tags)-1 DO BEGIN
            ;test all tag indices - tag can be array
            FOR j=0,n_elements(old.(i))-1 DO BEGIN 
                ;test if tag is a pointer
                r = size(old.(i)(j), /st)
                IF r.TYPE_NAME EQ 'POINTER' THEN BEGIN
                    new_ctr = *new.(i)(j)
                    r = size(new_ctr, /ST)
                    IF r.TYPE_NAME EQ 'INT' THEN GOTO, no_struct
                    ctr_tags = TAG_NAMES(old_ctr)
                    FOR ii = 0, n_elements(ctr_tags)-1 DO BEGIN
                        ;test all control tag indices - tag can be array
                        FOR jj = 0, n_elements(old_ctr.(ii))-1 DO BEGIN 
                            IF old_ctr.(ii)(jj) NE new_ctr.(ii)(jj) THEN BEGIN 
                                message, 'update on all '+ new.panelkind + $
                                  ' '+ ctr_tags(ii), /info
                                ;change in all other panels of same type!
                                FOR k=0, widgetData.numberOfPlots-1 DO BEGIN
                                    panelkind=widgetData.plotinfos(k).panelkind
                                    IF panelkind EQ new.panelkind THEN BEGIN
                                        ctr = *widgetData.plotinfos(k).(i)(j)
                                        ctr.(ii)(jj) = new_ctr.(ii)(jj)
                                        *widgetData.plotinfos(k).(i)(j) = ctr
                                    ENDIF     
                                ENDFOR     
                            ENDIF
                        ENDFOR     
                    ENDFOR
                    no_struct:
                ENDIF ELSE BEGIN
                    IF old.(i)(j) NE new.(i)(j) THEN BEGIN 
                        message, 'update on all '+ new.panelkind + ' ' +$
                                 old_tags(i), /info
                        ;change in all other panels of same type!
                        FOR k=0, widgetData.numberOfPlots-1 DO BEGIN
                            panelkind=widgetData.plotinfos(k).panelkind
                            IF panelkind EQ new.panelkind THEN $
                              widgetData.plotinfos(k).(i)(j)=new.(i)(j)
                       
                        ENDFOR 
                    ENDIF
                ENDELSE  
            ENDFOR  
        ENDFOR           
    END 

    3: begin
    END 
ENDCASE  
    
papco_SetPlotData, newPlotData, plotNumber
papco_refresh, /LIST

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_DeletePlot, plotNumber
;* 
;* DESCRIPTION:  
;*      Remove a plot.
;*	The user has to assure this action in a messagebox. After this is
;*	done, the element at index 'plotNumber' of widgetData.plotInfos is
;*	removed an widgetData.numberOfPlots is decreased.	
;*	Afterwards, the display is refreshed.
;*
;* INPUTS:       
;*	plotNumber	anInteger. 0<=plotNumber<=widgetData.numberOfPlots
;*			the index of the plot in widgetData.plotInfos.
;* 
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	papco_DeletePlot, anInteger
;*
;* MODIFICATION HISTORY:       
;*      written august 1995, Andreas Keese
;******************************************************************************
PRO papco_DeletePlot, plotNumber

  COMMON PLOT_COMPOSER, widgetData

  plotInfo=widgetData.plotInfos(plotNumber)
  IF messageBox(['Do you want to delete the '+plotInfo.panelKind, $
	         'at Position ' + strtrim(string(plotNumber+1), 2), $ 
		 papco_DescriptionFor(plotInfo, 0), '?'],  $
		['Yes', 'No'], /center, title='Delete a Plot') EQ 0 $

  THEN BEGIN
    papco_Cursor_Reset, /FULLY
    widgetData.numberOfPlotsDrawn=0
    FOR i=plotNumber, widgetData.numberOfPlots-2 DO $
      widgetData.plotInfos(i)=widgetData.plotInfos(i+1)
    widgetData.plotInfos(widgetData.numberOfPlots-1)=papco_getPlotInfoStruct()
    widgetData.numberOfPlots=widgetData.numberOfPlots-1
    papco_refresh, /LIST
    ;make all edit buttons inactive if no panles are left after delete!
    if widgetData.numberOfPlots eq 0 then begin
      widgetData.selectedPlot=-1
      widget_control, widgetData.lb_plotInfos, set_list_select=-1
      for i=0, n_elements(widgetData.edit_buttons)-1 do $
        widget_control, widgetData.edit_buttons(i), sensitive=0
    widget_control, widgetData.pb_overPlot, set_button=0
    endif
  ENDIF

END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_Add_Plot, panel_name, panelKind, $
;*			    INITIALPLOTINFO=initialPlotInfo
;* 
;* DESCRIPTION:  
;*      Used to add a new plot. This procedure is called from the main 
;*	event-routine, papco_Event.
;* 
;* INPUTS:       
;*      panel_name	a string
;*			this is the name of the panel-editor-function.
;*			This string will be executed using the 'CALL_FUNCTION'-
;*			command. 
;*
;*      panel_kind	a string
;*			this is the panel-kind.
;*
;* KEYWORDS:
;*      INITIALPLOTINFO a plotInfo-structure
;*			usually, this routine creates the initial plotInfo-
;*			structure by calling 'papco_getPlotInfoStruct()'. 
;*			If you want to use other initial settings, you may
;*			pass a plotInfo-structure to this procedure by 
;*			using this keyword.
;*
;* CALLING SEQUENCE:
;*      papco_Add_Plot, panel_name, panelKind
;*	;-----
;*      defaults=papco_getPlotInfoStruct()
;*      defaults.typeVector(3)=15
;*      papco_Add_Plot, panel_name, panelKind, initialPlotInfo=defaults
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese
;******************************************************************************
PRO papco_Add_Plot, panel_name, panelKind, $
                    INITIALPLOTINFO=initialPlotInfo, MODULE=module

COMMON PLOT_COMPOSER, widgetData
COMMON ADDINS, plot_types     ;contains lables of all plot types

close=0 & firstLoop=0

;check on proper naming - see if panelKind is valid
idx1 = where(plot_types EQ panelKind, c)
IF c EQ 0 THEN BEGIN
    message, 'Naming convention error: Panel '+panelKind+ $
             ' is not a defined valid plot type', /INFO
    print, 'Valid plot types: ', plot_types
    return 
ENDIF 
IF c GT 1 THEN BEGIN
    message, 'Naming convention error: Panel '+panelKind+ $
             ' is defined more than once', /INFO
    print, 'Valid plot types: ', plot_types
    return 
ENDIF

  
IF N_ELEMENTS(initialPlotInfo) GT 0 THEN defaults=initialPlotInfo $
    ELSE defaults=papco_getPlotInfoStruct()
defaults.panelKind=panelKind

;close is "true" as long same panels keep being "add and continued"....
WHILE NOT close do begin

    IF widgetData.numberOfPlots GE N_ELEMENTS(widgetData.plotInfos) THEN BEGIN
        IF NOT firstLoop THEN $
            dummy=messageBox(['Sorry...', 'I am too dumb to handle more', $
                              'Plots at one time :-('], ['What a pity'], $
                               TITLE='Add a Plot')
        return 
    ENDIF    
     
    ;for "add and continue" , make sure a new instance of the plotinfo
    ;pointer variable gets made, and set to the contents of the exisiting one. 
    ;Test for USR_PTR1 ag first - not all mmodules support this!
    idx = where(tag_names(defaults) EQ 'USR_PTR1', c)
    IF c EQ 1 THEN BEGIN 
        old_control = *defaults.USR_PTR1
        new_plotinfo = defaults
        new_plotinfo.USR_PTR1 = ptr_new(old_control)
        defaults = new_plotinfo
    ENDIF  
    firstLoop=1 & widget_control, /HOURGLASS

    ;first call panel to get size of it. Then add keywords for 
    ;scroll bars if needed.

    ;see if size is already determined for this session.
    IF widgetData.panel_editor_size(idx1(0), 0) EQ 1 THEN BEGIN
        SCR_XSIZE = widgetData.panel_editor_size(idx1(0), 0)
        SCR_YSIZE = widgetData.panel_editor_size(idx1(0), 1)
        GOTO, have_size
    ENDIF 
    size = 1

    catch, error_status
    if error_status ne 0 then begin  ;call error handler
        error_status=0 & catch, /CANCEL ;disable error catching
;        message, 'Cannot calculate size of '+panel_name, /cont
        goto, skip_get_panel_size
    ENDIF 

    newPlotData=CALL_FUNCTION(panel_name, defaults, $
                              ACTION='Add '+defaults.panelKind, $
                              GROUP=widgetData.main_base, $
                              SIZE_ONLY = size)

    error_status=0 & catch, /CANCEL ;disable error catching

    SCR_XSIZE = size.SCR_XSIZE & SCR_YSIZE = size.SCR_YSIZE 
    widgetData.panel_editor_size(idx1(0),0) = SCR_XSIZE
    widgetData.panel_editor_size(idx1(0),1) = SCR_YSIZE

    ;if call was succesfull, this feature is supported
    have_size:

    scroll = 0 & xsize = 0 & ysize = 0

    IF SCR_YSIZE GT widgetData.screensize(1) THEN BEGIN
        scroll = 1
        xsize = SCR_XSIZE+20
        ysize = widgetData.screensize(1)-200
        newPlotData=CALL_FUNCTION(panel_name, defaults, $
                                  ACTION='Add '+defaults.panelKind, $
                                  GROUP=widgetData.main_base, $
                                  SCROLL = scroll, $
                                  X_SCROLL_SIZE = xsize, $
                                  Y_SCROLL_SIZE = ysize )
        GOTO, done_panel
   
    ENDIF
    
    skip_get_panel_size:

    if keyword_set(MODULE) then $
      newPlotData=CALL_FUNCTION(panel_name, defaults, $
                                ACTION='Add '+defaults.panelKind, $
                                GROUP=widgetData.main_base, $
                                MODULE=module) $
    else $
      newPlotData=CALL_FUNCTION(panel_name, defaults, $
                                ACTION='Add '+defaults.panelKind, $
                                GROUP=widgetData.main_base)

    done_panel:
     
    if not newPlotData.canceled then begin
        defaults=newPlotData
        IF newPlotData.canceled EQ 0 then close=1   
                       ; .canceled EQ 0 <=> only one plot was to be added. 
                       ; .canceled NE 0 <=> add this and more plots

        papco_SetPlotData, newPlotData, widgetData.numberOfPlots

        n= widgetData.numberOfPlots
        if n gt 0 then begin
            widgetData.plotInfos(0:(n-1)).panelPosition= $
              widgetData.plotInfos(0:(n-1)).panelPosition + $
              widgetData.plotInfos(n).panelHeight
            widgetData.plotInfos(n).panelPosition=0
        ENDIF 

        widgetData.numberOfPlots=widgetData.numberOfPlots+1
        papco_refresh, /LIST
        
    ENDIF ELSE close=1
    
    firstLoop = firstLoop+1

ENDWHILE 

END
