;******************************************************************************
;******************************************************************************
;* FILE:
;*    papco_printer.pro
;*
;* DESCRIPTION:
;*    This file contains the widget for selecting a default printer
;*
;* STRUCTURES:
;*
;*      {papco_PRINTER_INFO, $
;*	     usePrinter:1, $			; 1=printer, 0=file
;*	     printer:1, $			; first printer : 1
;*	     askForFileName:0, $		; 1=open a file-dialog
;*	     fileType:1, $			; 1=PS, 0=PNG
;*	     fileName_tmpl:'~/plot%.ps'
;*           printerData:printerData}
;*      This structure is documented in papco_variables.text.
;*	It may contain either a default-printer. In this case, it describes
;*	a printer as well as a file-name.
;*	It may also be used to describe one output-medium. In this case, the
;*	'usePrinter'-flag decides whether output goes to the printer
;*	referenced by the 'printer'-tag or if output goes to the file
;*	referenced by 'fileName_tmpl' or 'askForFileName'.
;*
;*   	{papco_SELECTPRINTER, $
;*   	         canceled:1, $
;*		 usePrinter:aPrinterInfo.usePrinter, $ ; user printer or file ?
;*		 printer:aPrinterInfo.printer, $       ; number of printer
;*               printerData:aPrinterInfo.printerData, $
;*		 selectDefault:SELECT_DEFAULT, $
;*		 helpfile:'papco_printer.help' , $
;*		 askForFileName:aPrinterInfo.askForFileName, $
;*	         fileType:aPrinterInfo.fileType, $
;*		 fileName_tmpl:aPrinterInfo.fileName_tmpl, $
;*	         spb_printerOrFile:spb_printerOrFile, $
;*		 spb_printer:spb_printer, $
;*		 spb_fileType:spb_fileType, $
;*		 ef_FileName_tmpl:ef_FileName_tmpl}
;*       Used to store the widgetData-information
;*       For its description read papco_variables.text !
;*
;* OVERVIEW:
;*    Those procedures are contained:
;*      FUNCTION papco_getPrinterInfoStruct
;*	    -> create an instance of the papco_PRINTER_INFO-structure.
;*
;*	FUNCTION papco_printerDescription, printerInfo, $
;*	         PRINTER=PRINTER, FILE=FILE
;*          -> return verbose printer-description, substitute templates...
;*
;*      PRO papco_Printer_Refresh
;*	    -> update the display
;*
;*	PRO papco_Printer_Event, Event
;*	    -> event-handler for this window.
;*
;*	FUNCTION papco_Printer, aPrinterInfo, GROUP=GROUP, $
;*					      SELECT_DEFAULT=SELECT_DEFAULT
;*          -> create the widget.
;*
;* MODIFICATION HISTORY:
;*     august,september 1995, written by A.Keese
;*     february 1996, Haje Korth
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_getPrinterInfoStruct
;*
;* DESCRIPTION:
;*      return an instance of the papco_PRINTER_INFO-structure
;*
;*	To enable later modifications of papco,
;*	structures of this kind are NEVER defined explicitly -
;*	always use papco_getPrinterInfoStruct() to create such
;*	a structure.
;*
;* INPUTS:
;*      none
;*
;* OUTPUT:
;*      a papco_PRINTER_INFO-structure
;*
;* CALLING SEQUENCE:
;*      struct=papco_getPrinterInfoStruct()
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;*      february 1996, Haje Korth
;******************************************************************************
FUNCTION papco_getPrinterInfoStruct

papco_readPrinterConfig, printerData
tmp    = {PAPCO_PRINTER_INFO, $
          usePrinter:1, $
          printer:1, $
          askForFileName:0, $
          fileType:1, $			; 1=PS, 0=PNG, 2=both
	  fileName_tmpl:'%.<->.ps', $
          printerData:printerData}
return,tmp

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_printerDescription, printerInfo, $
;*					   PRINTER=PRINTER, FILE=FILE
;*
;* DESCRIPTION:
;*      A description for the printerInfo is returned. This string is used
;*	to label the print-buttons and by the papco_printPlots-routine that
;*	is contained in papco_draw.pro
;*
;*	If printerInfo describes a file, or if the keyword /FILE is set,
;*	The data in COMMON PLOT_COMPOSER, widgetData is used to substitute the
;*	patterns which might be contained in the printerInfo.fileName_tmpl
;*	patterns :
;*		       $     -> is substituted by the product name
;*		       %     -> is substituted by the orbit-number
;*		       <     -> substituted by the widgetData.startTime
;*		       >     -> substituted by the widgetData.endTime
;*                     &     -> substituted by start date only, yyyymmdd
;*
;* INPUTS:
;*      printerInfo	an instance of the papco_PRINTER_INFO-structure
;*			the returned description describes this printer
;*
;* KEYWORDS:
;* 	PRINTER		if not specified, the returned name is the name of
;*	FILE		a printer if printerInfo.usePrinter EQ 1. If the
;*			keyword PRINTER is specified, the returned string is
;*			the name of the printer contained in printerInfo.
;*			If FILE is specified, the returned string is the name
;*			of the file contained in printerInfo
;*
;* OUTPUT:
;*      a string is returned.
;*	this is either the name of a printer or the name of a file
;*
;* CALLING SEQUENCE:
;*	name=papco_printerDescription(aPrinter)
;*	name=papco_printerDescription(aPrinter, /PRINTER)
;*	name=papco_printerDescription(aPrinter, /FILE)
;*
;* MODIFICATION HISTORY:
;*     written august and september 1995, Andreas Keese
;*     february 1996, Haje Korth
;******************************************************************************
FUNCTION papco_printerDescription, printerInfo, PRINTER=PRINTER, FILE=FILE

COMMON PLOT_COMPOSER, widgetData

usePrinter=printerInfo.usePrinter
IF KEYWORD_SET(FILE) THEN usePrinter=0
IF KEYWORD_SET(PRINTER) THEN usePrinter=1

IF usePrinter THEN BEGIN
    printerNames=printerInfo.printerData(0,*)
    return, printerNames(printerInfo.printer)
ENDIF ELSE BEGIN
    substitutePercent='%'
    IF printerInfo.askForFileName THEN $
      return, 'file: <<dialog>>' $
    ELSE BEGIN
      WIDGET_CONTROL, widgetData.ef_orbit, GET_VALUE=orbNr
      orbNr=long(orbNr(0))
;       --- substitute the patterns in fileName_tmpl
      IF (widgetData.minTime.mjd NE widgetData.maxTime.mjd) OR $
         (widgetData.minTime.T NE widgetData.maxTime.T) THEN BEGIN
        fName=widgetData.PrintName

        convert_t90_to_Date, widgetData.startTime, s_year, s_day_oy, $
          hour, minute, second, s_month, s_cmonth, s_day_om, /mjdt
        timeStart=string(s_year,s_month,s_day_om,hour,minute,second, $
          format='(i4.4,i2.2,i2.2,"_",i2.2,i2.2,i2.2)')

        convert_t90_to_Date, widgetData.endTime, e_year, e_day_oy, $
          hour, minute, second, e_month, e_cmonth, e_day_om, /mjdt
        timeEnd=string(e_year,e_month,e_day_om,hour,minute,second, $
                       format='(i4.4,i2.2,i2.2,"_",i2.2,i2.2,i2.2)')

        startDay=string(s_year,s_month,s_day_om,format='(i4.4,i2.2,i2.2)')

        IF widgetData.time_choice_action THEN BEGIN
          if orbNr eq -1 then substitutePercent='' else $
            substitutePercent=string(orbNr,format="(i4.4)")
        ENDIF ELSE BEGIN
          substitutePercent= string(s_day_om, s_cmonth, s_year, e_day_om, $
            e_cmonth, e_year, format='(i2.2, "_", A, "_", i4.4, "-",'+$
                                      'i2.2, "_", A, "_", i4.4)')
        ENDELSE

        if widgetData.saveFileName eq 'none' then $
          substituteDollar='papco' $
        else begin
          papco_sepdir, widgetData.saveFileName, dir, file
          pos=strpos(file,'.papco')
          substituteDollar=strmid(file,0,pos)
        endelse

	pos=strpos(fName, '$')		; substitute '$'
        IF pos GT -1 THEN fName=strleft(fName, pos)+ $
             substituteDollar + strfrom(fName, pos+1)

        pos=strpos(fName, '%')          ; substitute '%'
        IF pos GT -1 THEN fName=strleft(fName, pos)+ $
          substitutePercent + strfrom(fName, pos+1)

        pos=strpos(fName, '<')          ; substitute timeStart
	IF pos GT -1 THEN $
             fName=strleft(fName, pos)+timeStart+strfrom(fName, pos+1)

        pos=strpos(fName, '>')          ; substitute timeEnd
	IF pos GT -1 THEN $
             fName=strleft(fName, pos)+timeEnd+strfrom(fName, pos+1)

        pos=strpos(fName, '&')          ; substitute startDay
	IF pos GT -1 THEN $
             fName=strleft(fName, pos)+startDay+strfrom(fName, pos+1)

	RETURN, fName
      ENDIF ELSE $
        return, printerInfo.fileName_tmpl
    ENDELSE
ENDELSE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Printer_Refresh
;*
;* DESCRIPTION:
;*      update the display of the printer-selection-window
;*
;* INPUTS:
;*      none
;*
;* KEYWORDS:
;*      none
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
PRO papco_Printer_Refresh, FILETYPE=FILETYPE, ALL=ALL

  COMMON PAPCO_SELECTPRINTER, widgetData

  if KEYWORD_SET(ALL) then begin
    FILETYPE=1
  endif

  if KEYWORD_SET(FILETYPE) then begin
    WIDGET_CONTROL, widgetData.ef_FileName_tmpl, GET_VALUE=tmpl
    tmpl=tmpl(0)
    papco_sepdir,tmpl(0),dir, file, ext
    case widgetData.fileType of
      0: begin                  ; png type
        if ext ne '.ps' then ext='.ps'
      end
      1: begin                  ; postscript type
        if ext ne '.png' then ext='.png'
      end
      2: begin                  ; both ps and png output
        if ext ne '.ps+png' then ext='.ps+png'
      end
    endcase
    WIDGET_CONTROL, widgetData.ef_FileName_tmpl, SET_VALUE=dir+file+ext
    widgetData.fileName_tmpl=dir+file+ext
  endif

  ; handle select defaults action
  IF widgetData.selectDefault THEN BEGIN
    ; disable the button that selects if either a file or a printer
    WIDGET_CONTROL, widgetData.destination, SENSITIVE=0
    ; enable the input to the other objects.
    WIDGET_CONTROL, widgetData.printer_select, SENSITIVE=1
    WIDGET_CONTROL, widgetData.file_select, SENSITIVE=1
    ; if file-name-template active, enable the entry-field else, disable it
    IF NOT widgetData.askForFileName THEN BEGIN
      WIDGET_CONTROL, widgetData.file_name_select, SENSITIVE=1
      WIDGET_CONTROL, widgetData.ef_fileName_tmpl, $
		SET_VALUE=widgetData.fileName_tmpl
    ENDIF ELSE WIDGET_CONTROL, widgetData.file_name_select, SENSITIVE=0
  ENDIF ELSE BEGIN ; handle a non-default-printer action
    ; disable the base that selects file destination
    WIDGET_CONTROL, widgetData.destination, SENSITIVE=1
    ; enable the widgets for selecting a printer
    IF widgetData.usePrinter THEN BEGIN
      IF NOT widgetData.selectDefault THEN BEGIN
        WIDGET_CONTROL, widgetData.printer_select, SENSITIVE=1
        WIDGET_CONTROL, widgetData.file_select, SENSITIVE=0
      ENDIF
    ENDIF ELSE BEGIN	; enable the widgets for selecting a file
      WIDGET_CONTROL, widgetData.printer_select, SENSITIVE=0
      WIDGET_CONTROL, widgetData.file_select, SENSITIVE=1
      ; if file-name-template active, enable the entry-field else, disable it
      IF NOT widgetData.askForFileName THEN BEGIN
        WIDGET_CONTROL, widgetData.file_name_select, SENSITIVE=1
        WIDGET_CONTROL, widgetData.ef_fileName_tmpl, $
          SET_VALUE=widgetData.fileName_tmpl
      ENDIF ELSE WIDGET_CONTROL, widgetData.file_name_select, SENSITIVE=0
    ENDELSE
  ENDELSE

END

;******************************************************************************
;* PROCEDURE:
;*      PRO papco_Printer_Event, Event
;*
;* DESCRIPTION:
;*      Process an XMANAGER-Event
;*
;* INPUTS:
;*      Event	 an XMANAGER-Event
;*
;* KEYWORDS:
;*      none
;*
;* CALLING SEQUENCE:
;*      called by XMANAGER
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;******************************************************************************
PRO papco_Printer_Event, Event

  common papco_selectprinter, widgetdata
  common plot_composer, mainwidgetdata

  widget_control,get_uvalue=uval, event.id

  CASE uval OF
      'mpb_file_type' : begin
        widget_control,event.id, GET_VALUE=x
        if x(0) eq 1 then widgetData.fileType=0
        if x(1) eq 1 then widgetData.fileType=1
        if (x(0) eq 1) AND (x(1) eq 1) then widgetData.fileType=2
        papco_printer_refresh, /FILETYPE
      end
      'spb_destination': BEGIN
        widget_control,event.id, GET_VALUE=x
        widgetData.usePrinter=x
        papco_Printer_Refresh
      end
      'spb_printer_select': BEGIN
        widget_control,event.id, GET_VALUE=x
        widgetData.printer=x
      end
      'dialog_printersetup': begin
        x= dialog_printersetup( dialog_parent=event.top )
      end
      'spb_file_choice': BEGIN
        widget_control,event.id, GET_VALUE=x
        widgetData.askForFileName=x
        papco_Printer_Refresh
      end
      'pb_help': BEGIN
        xdisplayfile,papco_helpfile(widgetData.helpfile), $
          title='Printer Selection Online-Help', group=event.top
      END
      'pb_cancel': BEGIN
        widgetData.canceled=1
        WIDGET_CONTROL, event.top, /DESTROY
      END
      'pb_done': BEGIN
        WIDGET_CONTROL, widgetData.ef_FileName_tmpl, GET_VALUE=tmpl
        widgetData.fileName_tmpl=tmpl(0)
        mainwidgetData.printerInfo.fileName_tmpl=tmpl(0)
        WIDGET_CONTROL, widgetData.ef_FileName_dir, GET_VALUE=tmpl
        dir=papco_checkdirname(tmpl(0))
        if papco_finddir(dir) eq 0 then begin ; jbf
          dummy=messageBox('Directory '+ dir +' not found', $
                            ['OK'], /CENTER, TITLE='Error')
          return
        endif
        mainWidgetData.outputFilePath=dir

        mainWidgetData.PrintName= $
          papco_checkdirname(mainWidgetData.outputFilePath) + $
          widgetData.fileName_tmpl
        widgetData.canceled=0
        if widgetData.selectDefault eq 1 then papco_writeDefaultSettings
        WIDGET_CONTROL, event.top, /DESTROY
      END
      'ef_fileName_dir': BEGIN
        dir=papco_checkdirname(event.value)
        if papco_finddir(dir) eq 0 then begin
          dummy=messageBox('Directory '+ dir +' not found', $
                           ['OK'], /CENTER, TITLE='Error')
          return
        endif
        mainWidgetData.outputFilePath=dir
        WIDGET_CONTROL, widgetData.ef_FileName_dir, SET_VALUE=dir
      END
     'file_browse' : BEGIN
       fname=dialog_PICKFILE( /READ, /directory, GROUP=widgetdata.base, $
                      TITLE='File Name', $
                      FILTER='*', $
                      PATH=mainWidgetData.outputFilePath, $
                      GET_PATH=getpath)
       if fname eq '' then return
       mainWidgetData.outputFilePath=getpath
       WIDGET_CONTROL, widgetData.ef_FileName_dir, SET_VALUE=getpath
     END
     ELSE : BEGIN
       message, 'EVENT NOT HANDLED !!!',/cont
       help, /struct, event
     END
  ENDCASE

END

;******************************************************************************
;* FUNCTION:
;*      FUNCTION papco_Printer, aPrinterInfo, $
;*		 GROUP=GROUP, SELECT_DEFAULT=SELECT_DEFAULT
;*
;* DESCRIPTION:
;*      open the printer-selection-dialog.
;*
;*	In this dialog, the user may select if output shall be done to a
;*	printer or to a file. If output is done to a file, the file-format
;*	can be defined (PNG or PS)
;*
;*	If the dialog is called for a default-printer, a printer AND a file-
;*	name may be selected. Else, either a filename or a printer may be
;*	be selected.
;*
;*	If a file is selected, the user may decide, if the filename shall be
;*	obtained by opening a file-selection-box, or if it shall be obtained
;*	by substituting the abbreviations in the file-name-template.
;*      See $PAPCO_BASE/papco/help/papco_printer.help for possible
;*      template formats.
;*
;*
;* INPUTS:
;*      aPrinterInfo	an instance of papco_PRINTER_INFO-structure
;*			the settings in this instance are used as initial
;*			settings for this dialog
;*
;* KEYWORDS:
;* 	GROUP		a widget-base
;*			if set, this will be used as GROUP-Leade for this
;*			window
;*
;*	SELECT_DEFAULT	a boolean (non-existent or 1)
;*			if set, the dialog will be opened for selecting the
;*			default-printer. In this case, a printer as well
;*			as a filename may be chosen.
;*
;*			If it isn't set, either a printer or a filename
;*			can be selected.
;*
;* OUTPUT:
;*      the widget-data is returned.
;*
;* CALLING SEQUENCE:
;*  ;---selecting a default-printer
;*      default_printer=papco_getPrinterInfoStruct()
;*	;....lots of code...
;*	tmp=papco_Printer(default, /SELECT_DEFAULT)
;*	if not tmp.canceled then begin
;*	   default_printer.usePrinter=printer.usePrinter
;*	   default_printer.printer=printer.printer
;*	   default_printer.askForFileName=printer.askForFileName
;*	   default_printer.fileType=printer.fileType
;*	   default_printer.fileName_tmpl=printer.fileName_tmpl
;*	endif
;*
;*  ;---selecting a non-default-printer
;*	printer=papco_Printer(papco_GetPrinterInfoStruct())
;*	if not printer.canceled then begin
;*	   if tmp.usePrinter then begin
;*	      output_toPrinter, tmp.printer
;*	   endif else begin
;*            if tmp.askForFileName then begin
;*		 fileName=pickfile()
;*		 output_toFile, fileName
;*            endif else begin
;*		 ; first, substitute the abbreviations:
;*		 fileName=substitute_templates(tmp.fileName)
;*		 output_toFile, fileName
;*	      endelse
;*	   endelse
;*	endif
;*
;* MODIFICATION HISTORY:
;*      september 1995, written by A.Keese
;*      february 1996, Haje Korth
;*      December 2000, Reiner Friedel
;******************************************************************************
FUNCTION papco_Printer, aPrinterInfo, GROUP=GROUP,SELECT_DEFAULT=SELECT_DEFAULT

  COMMON PAPCO_SELECTPRINTER, widgetData
  COMMON PLOT_COMPOSER, mainWidgetData

  IF N_ELEMENTS(Group) EQ 0 THEN GROUP=0
  IF NOT KEYWORD_SET(SELECT_DEFAULT) THEN SELECT_DEFAULT=0

  base = papco_WIDGET_BASE(GROUP_LEADER=Group, COLUMN=1, /MAP, $
               TITLE='papco - select print destination', UVALUE='base')

; --- Printer or file ? -------------------------------------------------------
  dest_names=[' File',' Printer']
  destination=CW_BGROUP(base, dest_names, ROW=1, /EXCLUSIVE, /FRAME, $
                        LABEL_left='Output destination: ', $
                        UVALUE='spb_destination', $
                        SET_VALUE=aprinterinfo.useprinter)

; --- Which Printer ? ---------------------------------------------------------
  os_printer= !version.os_family + ' printers'

  printer_names= aPrinterInfo.printerData(0, *)

  index = where(printer_names NE '', c)

  if ( printer_names[aprinterinfo.printer] eq '' ) then begin
     aprinterinfo.printer= index[0]
  endif

  IF c GT 0 THEN BEGIN
      printer_names = printer_names(index)
      printer_select=CW_BGROUP(base, printer_names, COLUMN=2, $
                               /FRAME, /EXCLUSIVE, $
                               LABEL_TOP='Select printer Queue', $
                               UVALUE='spb_printer_select', $
                               SET_VALUE=aprinterinfo.printer < $
                               (n_elements(printer_names)-1) )
  ENDIF ELSE BEGIN
      printer_select = $
        widget_text(base, /wrap,  ysize = 3, $
                    VALUE = ['No Printers have been set up',  $
                             'Setup papco_printer_config.dat',  $
                             'in your papco_user directory.'])
  ENDELSE

  r= where( aPrinterInfo.printerData(1,*) eq 'os_printer' )
  if (r[0] ne -1) then begin
      ;; this hangs on unix, so it's not recommended.
      x= widget_text( base, /align_left, /wrap, ysize=3, xsize=80, $
                      value=['Click on the button'+$
                             ' below to run '+$
                             'dialog_printersetup() to'+$
                             ' configure '+$
                             os_printer+'.  Do not use the printer''s landscape setting, instead select landscape under defaults' ] )
      x= widget_button( base, value='Setup Printing (dialog_printersetup())', $
                        uvalue='dialog_printersetup' )
  endif

; ---- File-select ------------------------------------------------------------
  file_select = WIDGET_BASE(base, COLUMN=1)

  file_types=[' Postscript file', ' Portable Network Graphics (PNG) file']
  case aPrinterInfo.fileType of
    0: set=[1,0]  &  1: set=[0,1]  &  2: set=[1,1]
  endcase
  file_type=CW_BGROUP(file_select, file_types, ROW=1, $
                      /NONEXCLUSIVE, /FRAME, $
                      LABEL_TOP='Select output file type(s)', $
                      UVALUE='mpb_file_type', SET_VALUE=set)

  file_choices=[' Use template', ' Ask for']
  file_choice=CW_BGROUP(file_select, file_choices, ROW=1,/EXCLUSIVE, /FRAME, $
                        LABEL_left='Filename Choice: ', $
                        UVALUE='spb_file_choice', $
                        SET_VALUE=aPrinterInfo.askForFileName)

  file_name_select =  WIDGET_BASE(file_select, ROW=2, /FRAME)
  ef_fileName_dir=CW_FIELD(file_name_select, /COLUMN, title='Directory: ', $
			   UVALUE='ef_fileName_dir', XSIZE=50, /RETURN_EVENTS)
  WIDGET_CONTROL, ef_FileName_dir, SET_VALUE=mainWidgetData.outputFilePath
  file_browse=WIDGET_BUTTON(file_name_select, $
                            VALUE=strformat('Browse',10,/CEN), $
                            UVALUE='file_browse')
  ef_fileName_tmpl=CW_FIELD(file_name_select, title='Filename Template: ', $
                            UVALUE='ef_fileName_tmpl', XSIZE=30)
  WIDGET_CONTROL, ef_FileName_tmpl, $
    SET_VALUE=mainWidgetData.printerinfo.fileName_tmpl

; --- Actions -----------------------------------------------------------------
  action_base= widget_base( base, /row )
  ButtonLbls = ['Help', 'Cancel', 'Save as Default']
  if not select_default then  ButtonLbls(2)='Print to this one'
  ButtonUvalues= ['pb_help', 'pb_cancel', 'pb_done']
  buttons=intarr(n_elements(ButtonLbls))
  for i=0, n_elements(ButtonLbls)-1 do $
    buttons(i)=WIDGET_BUTTON(action_base,VALUE=ButtonLbls(i),UVALUE=ButtonUValues(i))

  widgetData={PAPCO_SELECTPRINTER_8, $
              base:base, canceled:1, $
              usePrinter:aPrinterInfo.usePrinter, $   ; user printer or file ?
              printer:aPrinterInfo.printer, $         ; number of printer
              printerData:aPrinterInfo.printerData, $
              selectDefault:select_default, $
              helpfile:'papco_printer.help' , $
              askForFileName:aPrinterInfo.askForFileName, $
              fileType:aPrinterInfo.fileType, $
              fileName_tmpl:aPrinterInfo.fileName_tmpl, $
              file_select:file_select, $
              destination:destination, $
              printer_select:printer_select, $
              file_name_select:file_name_select, $
              ef_FileName_dir:ef_FileName_dir, $
              ef_FileName_tmpl:ef_FileName_tmpl}

  papco_printer_refresh, /ALL
  WIDGET_CONTROL, base, /REALIZE
  papco_xmanager, 'papco_Printer', base

  RETURN, widgetData

END
