;******************************************************************************
;******************************************************************************
;* FILE:         
;*    $PAPCO_PATH/papco_settime.pro
;* 
;*    This file belongs to the main-window of the papco-project. See
;*    papco.pro for further information.
;*
;* DESCRIPTION:  
;*    Contained are procedures for selecting a time-range.
;*
;*    These procedures are contained:
;*
;*      FUNCTION papco_SetOrbit, anOrbitNr
;*	   -> set the orbit to 'anOrbitNr'
;*
;*	FUNCTION SET_MANUAL_TIME, MAXTIME=MAXTIME, MINTIME=MINTIME. $
;*                                DAYTIME=DAYTIME
;*         -> called, when a time-range is entered manually
;*
;*      PRO papco_Orbit_Selected, anOrbitInfo
;*         -> called from the orbit-list window, if an orbit was selected
;*
;*      PRO papco_setTime, newTime, STARTTIME=starttime, ENDTIME=endtime
;*	   -> called, when the time-slider are dragged.
;*
;*	PRO papco_DrawTimeRange, newStart, newEnd, CLEAR_ONLY=CLEAR_ONLY
;*	   -> display the currently chosen time-range in the drawing-widget
;*
;* MODIFICATION HISTORY:       
;*     august,september 1995, written by A.Keese 
;*     january 1997, by R. Friedel, to allow MJDT time format
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION papco_setorbit, anOrbitNr
;* 
;* DESCRIPTION:  
;*      Set the orbit.
;*	This function is invoked, when the user enters an orbit number in
;*	the orbit-entry-field.
;*	The new orbitInfo is read and the display is refreshed.
;*
;* INPUTS:       
;*	anOrbitNr	an Integer 
;*			containing the number of the orbit
;*  
;* OUTPUTS:
;*	a Boolean	0 : the orbit could not be read
;*			1 : all o.k
;*	
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	IF NOT papco_SetOrbit(anInt) THEN print, 'error !'
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
FUNCTION papco_setorbit, anOrbitNr, SATELLITE=SATELLITE, NOWIDGET=NOWIDGET

  COMMON PLOT_COMPOSER, widgetData
  COMMON mjdt, mjdt_start, mjdt_end
  COMMON time, xut1, xut2
  
  if anOrbitNr eq 0 then begin
    widget_control, widgetData.ef_orbit, set_value=0
    widgetData.default_orbitInfo.number=0 
    widgetData.time_choice_action=0
    papco_Refresh, /new_timeRange
    return, 1
  endif  
  
  if keyword_set(SATELLITE) then satellite=SATELLITE else $
    satellite=widgetData.satellite_name
  
;-- get the info. If reading failed, an error-string is returned 
  orbitInfo=papco_getOrbitInfoFor(satellite, anOrbitNr)

  IF N_ELEMENTS(size(orbitInfo)) GT 3 THEN BEGIN
 
    ;convert orbit start/end time to MJDT
    mjdt_start=CONVERT_TZERO_TO_T90(long(orbitInfo.xut1), $
                                    orbitInfo.day, orbitInfo.year, /MJDT)
    mjdt_end  =CONVERT_TZERO_TO_T90(long(orbitInfo.xut2), $
                                    orbitInfo.day, orbitInfo.year, /MJDT)
    ;if year is ge 1990, then also find T() time
    if orbitInfo.year lt 1990 or orbitInfo.year gt 2050 then begin
      xut1=0  &  xut2=0
    endif else begin  
      xut1=CONVERT_TZERO_TO_T90(long(orbitInfo.xut1), $
                                orbitInfo.day, orbitInfo.year)
      xut2=CONVERT_TZERO_TO_T90(long(orbitInfo.xut2), $
                                orbitInfo.day, orbitInfo.year)
    endelse  
    
    if keyword_set(NOWIDGET) then RETURN, 1
    
    papco_DrawTimeRange, 0, 0, /CLEAR_ONLY
    widgetData.default_orbitInfo=orbitInfo
    
    IF widgetData.time_choice_action THEN BEGIN
      widgetData.numberOfPlotsDrawn=0
      widgetData.startTime.t90=xut1
      widgetData.startTime.mjd=mjdt_start.mjd
      widgetData.startTime.t=mjdt_start.t
      widgetData.endTime.t90=xut2
      widgetData.endTime.mjd=mjdt_end.mjd
      widgetData.endTime.t=mjdt_end.t
      widgetData.minTime=widgetData.startTime
      widgetData.maxTime=widgetData.endTime
    ENDIF
   
    widgetData.need_to_draw=1
    papco_Refresh, /new_timeRange
           
    RETURN, 1
  ENDIF ELSE BEGIN
     stop
     IF widgetData.time_choice_action THEN $
       dummy=messageBox(['The requested orbit, '+ $
                         strtrim(string(anOrbitNr), 2)+ ', could', $
                         'not be located !', '', $
                         'The reading module returned this error:', $
                         orbitInfo], ['Sorry!'], $
                         GROUP_LEADER=widgetdata.base, $
                         title='Orbit not located', /center)
     RETURN, 0
  ENDELSE

END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION SET_MANUAL_TIME, MAXTIME=MAXTIME, MINTIME=MINTIME, $
;*                                DAYTIME=DAYTIME
;* 
;* DESCRIPTION:  
;*      this function is connected with the fields for manual entering 
;*      the time-range. It is called from the papco_Event-function when 
;*	one presses enter in this fields.
;*	
;* INPUTS:       
;*      none
;*
;* KEYWORDS:
;* 	MAXTIME		the return-key was pressed in the ef_maxTime-field
;*	MINTIME		enter was pressed in the ef_minTime-field
;*      DAYTIME         enter was pressed in the efDayTime-field
;* 
;* OUTPUT:	 
;*      0	 	the time was entered in an invalid format
;*	1		the entered time has correct format
;*
;* CALLING SEQUENCE:
;*      if SET_MANUAL_TIME(/maxtime) then .....
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      modified to add DAYTIME, may 1996, R. Friedel
;******************************************************************************
FUNCTION set_manual_time, MAXTIME=MAXTIME, MINTIME=MINTIME, DAYTIME=DAYTIME

   COMMON PLOT_COMPOSER, widgetData
  
   IF keyword_set(DAYTIME) THEN BEGIN
      WIDGET_CONTROL, widgetData.ef_DayTime, GET_VALUE=tday
      tday_t90 = CONVERT_timeToSeconds(tday(0), /T90)
      if tday_t90.err ne 0 then  $
         message,'Time outside T90 range', /cont
      tday_mjdt= CONVERT_timeToSeconds(tday(0), /T90, /MJDT)
      if tday_mjdt.err ne 0 then begin
         dummy=messageBox(tday_mjdt.errtext, ['Ok'], $
                          title='wrong format - DayTime')
         return, 0
      endif
      widgetdata.minTime.t90=tDay_t90.value
      widgetdata.maxTime.t90=tDay_t90.value + 86400
      widgetdata.minTime.mjd=tday_mjdt.value.mjd
      widgetdata.maxTime.mjd=tday_mjdt.value.mjd+1
      widgetdata.minTime.t=0.0
      widgetdata.maxTime.t=0.0
      goto,donetime
   ENDIF

   if (not keyword_set(maxTime)) and (not keyword_set(minTime)) then begin
      minTime=1
      maxTime=1
   endif

   WIDGET_CONTROL, widgetData.ef_minTime, GET_VALUE=tminn
   WIDGET_CONTROL, widgetData.ef_maxTime, GET_VALUE=tmaxx

   tmin=CONVERT_timeToSeconds(tminn(0), /T90, /MJDT)
   tmax=CONVERT_timeToSeconds(tmaxx(0), /T90, /MJDT)
   tmin_t90=CONVERT_timeToSeconds(tminn(0), /T90)
   tmax_t90=CONVERT_timeToSeconds(tmaxx(0), /T90)

   if keyword_set(minTime) then begin
      if tmin.err ne 0 then begin
         dummy=messageBox(tmin.errtext, ['Ok'], title='wrong format - minTime')
         return, 0
      endif
      widgetdata.minTime.mjd=tmin.value.mjd
      widgetdata.minTime.t=tmin.value.t
      widgetdata.minTime.t90=tmin_t90.value
      if tmax.err ne 0 then begin
         tmax.value.mjd=widgetData.maxTime.mjd
         tmax.value.t=widgetData.maxTime.t
      endif 
      if (tmax.value.mjd lt tmin.value.mjd) then tmax.value=tmin.value else $
        if (tmax.value.mjd eq tmin.value.mjd) and $
           (tmax.value.t lt tmin.value.t) then tmax.value=tmin.value
      widgetdata.maxTime.mjd=tmax.value.mjd         
      widgetdata.maxTime.t=tmax.value.t
      widgetdata.maxTime.t90=tmax_t90.value
   endif

   if keyword_set(maxTime) then begin
      if tmax.err ne 0 then begin
         dummy=messageBox(tmax.errtext, ['Ok'], title='wrong format - maxTime')
         return, 0
      endif
      widgetdata.maxTime.mjd=tmax.value.mjd         
      widgetdata.maxTime.t=tmax.value.t
      widgetdata.maxTime.t90=tmax_t90.value


      if tmin.err ne 0 then tmin.value=widgetData.minTime
      if (tmax.value.mjd lt tmin.value.mjd) then tmax.value=tmin.value else $
        if (tmax.value.mjd eq tmin.value.mjd) and $
           (tmax.value.t lt tmin.value.t) then tmax.value=tmin.value
      widgetdata.minTime.mjd=tmin.value.mjd
      widgetdata.minTime.t=tmin.value.t
      widgetdata.minTime.t90=tmin_t90.value
   endif

   donetime:

   papco_DrawTimeRange, 0, 0, /CLEAR_ONLY
   widgetData.numberOfPlotsDrawn=0
   widgetData.startTime=widgetData.minTime
   widgetData.endTime=widgetData.maxTime

   papco_Refresh, /NEW_TIMERANGE
   RETURN, 1

END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_Orbit_Selected, anOrbitInfo
;* 
;* DESCRIPTION:  
;*      This procedure is called, when the user chooses an orbit in the orbit
;*	select-window. The number of the orbit is passed to papco_SETOrbit
;*
;* INPUTS:       
;*	anOrbitInfo	a structure of type papco_ORBITINFO           
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	papco_Orbit_Selected, anOrbitInfo
;*	called by papco_SelectOrbit ONLY
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_Orbit_Selected, anOrbitInfo

   dummy=papco_SetOrbit(anOrbitInfo.number)

END

;******************************************************************************
;* PROCEDURE:     
;*      papco_setTime, newTime, STARTTIME=starttime, ENDTIME=endtime
;* 
;* DESCRIPTION:  
;*      Whenever the user changes the start- or the endtime using the
;*	time-sliders or the time-entry-fields, this procedure is invoked
;*	to calculate the new effectively set time.    
;*	This routine guarantees, that the starttime will be earlier than
;*	the endtime, and it guarantees, that both values lie in the range
;*	of the orbit-times.
;*
;* INPUTS:       
;*	newTime 	a long
;*			the new time in seconds.
;* 
;* KEYWORDS:
;* 	STARTIME	when set, the newTime-variable specifies the new
;*			starttime.
;*	ENDTIME		when set, the newTime specifies the new endtime.
;* 
;* CALLING SEQUENCE:
;*	PAPCO_SetTime, seconds, /STARTIME
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_setTime, newTime, STARTTIME=starttime, ENDTIME=endtime

  COMMON PLOT_COMPOSER, widgetData
  COMMON time, xut1,xut2
  COMMON mjdt, mjdt_start, mjdt_end
    
  minTime = long(widgetData.minTime.t)
  maxTime = long(widgetData.maxTime.mjd-widgetData.minTime.mjd)*86400l+ $
                                long(widgetData.maxTime.t)

  newTime=newTime > minTime
  newTime=newTime < maxTime

  newStart = long(widgetData.startTime.mjd-widgetData.minTime.mjd)*86400l+ $
                              long(widgetData.startTime.t)
  newEnd   = long(widgetData.endTime.mjd-widgetData.minTime.mjd)*86400l+ $
                              long(widgetData.endTime.t)

  if KEYWORD_SET(STARTTIME) THEN BEGIN
	newStart=newTime
        if newStart ge newEnd then begin
          if newstart ne maxtime then newEnd=newstart+1 $
            else begin 
              newStart=newStart-1
              newEnd=maxtime
            endelse
        endif
  ENDIF ELSE BEGIN
	newEnd=newTime
        if newStart ge newEnd then begin
          if newEnd ne mintime then newStart=newEnd-1 $
            else begin 
              newEnd=newEnd+1
              newStart=mintime
            endelse
        endif
  ENDELSE

  IF widgetData.numberOfPlotsDrawn GT 0 THEN BEGIN
    papco_DrawTimeRange, newStart, newEnd
  ENDIF

  newSt={papco_TIME, mjd:0l, T:0.0, T90:0l}
  newSt.mjd=widgetData.minTime.mjd + newStart / 86400l
  newSt.t=newStart mod 86400l
  newst_mjd={MJDT,mjd:0l, T:0.0}
  newst_mjd.mjd=newSt.mjd
  newst_mjd.t=newSt.t
  tstr=convert_secondsToTime(newst_mjd,/t90,/mjdt)
  result=convert_timeToSeconds(tstr,/t90)
  newSt.t90=result.value
  widgetData.startTime=newSt

  newEn={papco_TIME, mjd:0l, T:0.0, T90:0l}
  newEn.mjd=widgetData.minTime.mjd + newEnd / 86400l
  newEn.t=newEnd mod 86400l
  newen_mjd={MJDT,mjd:0l, T:0.0}
  newen_mjd.mjd=newEn.mjd
  newen_mjd.t=newEn.t
  tstr=convert_secondsToTime(newen_mjd,/t90,/mjdt)
  result=convert_timeToSeconds(tstr,/t90)
  newEn.t90=result.value
  widgetData.endTime=newEn   
  
;-- set up common start and end time for MJDT users
  mjdt_start=widgetData.startTime
  mjdt_end=widgetData.endTime
;--- set up common start and end time for old T90 users
  xut1=widgetData.startTime.t90
  xut2=widgetData.endTime.t90
    
END

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_DrawTimeRange, newStart, newEnd, CLEAR_ONLY=CLEAR_ONLY
;* 
;* DESCRIPTION:  
;*      when the start- or endtime sliders are dragged and if plots were 
;*      drawn yet, the newly selected time-range is displayed by a colored 
;*	bar in the drawing widget. This functions handles its drawing.
;*
;* INPUTS:
;*      newStart	a number
;*			the new start of the time-range in the T90-format
;*	newEnd		a number
;*			the new end of the time-range in the T90-format
;* 
;* KEYWORDS:
;* 	CLEAR_ONLY	if specified, the timerange-selector is erased
;*
;* CALLING SEQUENCE:
;*      papco_DrawTimeRange, 0, convert_tzero_to_t90(100,2,90)
;*	papco_DrawTimeRange, /CLEAR_ONLY
;*
;* MODIFICATION HISTORY:       
;*     written august 1995, Andreas Keese
;******************************************************************************
PRO papco_DrawTimeRange, newStart, newEnd, CLEAR_ONLY=CLEAR_ONLY

    COMMON PLOT_COMPOSER, widgetData 

    IF widgetData.NumberOfPlotsDrawn EQ 0 THEN RETURN

; check if draw window actually exists

    IF widgetData.no_draw_window eq 0 THEN RETURN

    PAPCO_SET_PLOT, 'X'  
    WIDGET_CONTROL, widgetData.dw_draw,  GET_VALUE=drawIndex
    WSET, drawIndex
    DEVICE, SET_GRAPHICS=6 ; set XOR-Mode for graphics

; clear the timeRange drawn last time, I was called...
    IF widgetData.timeRangeDrawn THEN BEGIN
       startTime=long(widgetData.startTime.mjd-widgetData.minTime.mjd)*86400l+$
                                long(widgetData.startTime.t)
       tmp=papco_Data_To_Normal(0, [startTime,1])
       xl_old=tmp(0)
       endTime=long(widgetData.endTime.mjd-widgetData.minTime.mjd)*86400l+ $
                                long(widgetData.endTime.t)
       tmp=papco_Data_To_Normal(0, [endTime,1])
       xr_old=tmp(0)
       POLYFILL, [xl_old,xl_old,xr_old,xr_old], [0,1,1,0], color=1, /NORMAL
       widgetData.timeRangeDrawn=0
    ENDIF

; now, draw the new time-range
    IF NOT KEYWORD_SET(CLEAR_ONLY) THEN BEGIN
       tmp=papco_Data_To_Normal(0, [newStart,1])
       xl_new=tmp(0)
       tmp=papco_Data_To_Normal(0, [newEnd,1])
       xr_new=tmp(0)
       POLYFILL, [xl_new,xl_new,xr_new,xr_new], [0,1,1,0], color=1, /NORMAL

       ;now, initiate a timer-event, that will clear the drawn time-range
       ;in about 3 seconds.
       WIDGET_CONTROL, widgetData.clearTimeRange, TIMER=3
       widgetData.timeRangeDrawn=1
    ENDIF

    DEVICE, SET_GRAPHICS=3 ; reset the XOR-mode to default

END 

;******************************************************************************
;* PROCEDURE:     
;*      PRO papco_time_adjust
;* 
;* DESCRIPTION:  
;*      This procedure is called when time needs to auto adjusted according
;*      to what is set in default_adjust_time
;*
;* INPUTS:       
;*	none       
;*
;* KEYWORDS:
;* 	none
;* 
;* CALLING SEQUENCE:
;*	papco_time_adjust
;*
;* MODIFICATION HISTORY:       
;*      written march 2000, Reiner Friedel
;******************************************************************************
PRO papco_time_adjust
  
  common plot_composer, widgetdata 
  
;-- modify the timerange according to widgetData.default_adjust_time
  const_sec=long(widgetData.default_adjust_time*3600.0)
  if const_sec gt 0 then begin
    widgetData.starttime=addsubtime(widgetData.starttime, $
                                   {MJD:0,t:abs(const_sec),t90:0},/ADD)
    widgetData.endtime=addsubtime(widgetData.endtime, $
                                   {MJD:0,t:abs(const_sec),t90:0},/SUBTRACT)  
  endif else begin
    widgetData.starttime=addsubtime(widgetData.starttime, $
                                   {MJD:0,t:abs(const_sec),t90:0},/SUBTRACT)
    widgetData.endtime=addsubtime(widgetData.endtime, $
                                   {MJD:0,t:abs(const_sec),t90:0},/ADD)      
  endelse  
    
  widgetData.startTime.t90=(widgetData.startTime.t90+const_sec) > 0
  widgetData.endTime.t90=widgetData.endTime.t90-const_sec
  
  widgetData.minTime=widgetData.starttime
  widgetData.maxTime=widgetData.endtime
  
  papco_refresh, /NEW_TIMERANGE
    
END
