;-------------------------------------------------------------------------
; NAME: WRITE_MYCDF
; PURPOSE:
;       To input an idl structure of the type returned by read_mycdf,
;       and to output it as a CDF file.
; CALLING SEQUENCE:
;       status = write_mycdf(instruct,filename)
; INPUTS:
;       instruct = input data structure
;       filename = the name of the cdf file being created
; KEYWORD PARAMETERS:
;       autoname = if set, then override the filename parameter by
;                  generating the name for the cdf file according to
;                  the ISTP filenaming conventions.  This will also
;                  cause the global attribute LOGICAL_FILE_ID to
;                  be set accordingly.
;       longtime = if set, is used in conjunction with the autoname
;                  keyword, but will cause a deviation from the ISTP
;                  filenaming conventions in that the timestamp in the
;                  filename will also include the starting hour of the data.
;       bothtimes = if set, is used in conjunction with the autoname and
;                   longtime keywords, will cause a deviation from the ISTP
;                   filenaming conventions in that the timestamp in the
;                   filename will include both start and stop times.
;       uppercase = if set, is used in conjunction with the autoname and
;                   longtime keywords such that the automatically deter-
;                   mined filename will be in all uppercase.
;       lowercase = if set, is used in conjunction with the autoname and
;                   longtime keywords such that the automatically deter-
;                   mined filename will be in all lowercase.
;       outdir    = if set, is used in conjunction with the autoname keywords
;                   to create the file in the specified directory.
; OUTPUTS:
;       status = integer indicating success (0) or failure (-1)
; EXAMPLE:
;       a = read_mycdf('','file1.cdf',/all) ; read all vars from file1.cdf
;       s = write_mycdf(a,'file2.cdf')      ; create same file named file2.cdf
;       s = write_mycdf(a,'',/autonamer)    ; create filename based on contents
;                                           ;  of the structure 'a'.
; AUTHOR:
;       Richard J. Burley, NASA/GSFC/Code 632,  June, 1998
;       burley@nssdca.gsfc.nasa.gov    (301)286-2864
; MODIFICATION HISTORY:
;-------------------------------------------------------------------------


; Given an input string, find that structure element name and return number
;TJK commented this function out - the source for this is in tagindex.pro
;
;FUNCTION tagindex, instring, tnames
;instring = STRUPCASE(instring) ; tagnames are always uppercase
;a = where(tnames eq instring,count)
;if count eq 0 then return, -1 $
;else return, a(0)
;end

; Return TRUE(1) or FALSE(0) if the input parameter looks like one of the
; structures returned by the read_mycdf or restore_mystruct functions.
FUNCTION ami_mystruct,a
ds = size(a) & nds = n_elements(ds)
if (ds(nds-2) ne 8) then return,0
for i=0,n_elements(tag_names(a))-1 do begin
  ds = size(a.(i)) & nds = n_elements(ds)
  if (ds(nds-2) ne 8) then return,0
  tnames = tag_names(a.(i))
  w = where(tnames eq 'VARNAME',wc1) & w = where(tnames eq 'CDFTYPE',wc2)
  w = where(tnames eq 'HANDLE' ,wc3) & w = where(tnames eq 'DAT',wc4)
  if wc1 eq 0 then return,0
  if wc2 eq 0 then return,0
  if (wc3 + wc4) ne 1 then return,0
endfor
return,1
end


; Utilize IDL's SAVE procedure to save the structure a into the given filename.
PRO save_mystruct,a,fname
COMMON CDFmySHARE, v0  ,v1, v2, v3, v4, v5, v6, v7, v8, v9,$
                   v10,v11,v12,v13,v14,v15,v16,v17,v18,v19,v20
if tagindex('HANDLE',tag_names(a.(0))) eq -1 then begin
  ; data is stored directly in the structure
  SAVE,a,FILENAME=fname
endif else begin
  ; data is stored in handles.  Retrieve the data from the handles,
  ; and store the data into 'n' local variables, then SAVE.
  tn = tag_names(a) & nt = n_elements(tn) & cmd = 'SAVE,a'
  ; Preallocate some temporary variables.  The EXECUTE command cannot
  ; create new variables...they must already exist.  Lets hope 20 is enough.
;TJK comment this check out since this is now done dynamically
;  if nt ge 20 then begin
;    print,'ERROR= too many handle values in structure to save' & return
;  endif

;  v0=0L  & v1=0L  & v2=0L  & v3=0L  & v4=0L  & v5=0L  & v6=0L  & v7=0L
;  v8=0L  & v9=0L  & v10=0L & v11=0L & v12=0L & v13=0L & v14=0L & v15=0L
;  v16=0L & v17=0L & v18=0L & v19=0L & v20=0L

  for i=0,nt-1 do begin ; retrieve each handle value
    order = 'handle_value,a.(i).HANDLE,v' + strtrim(string(i),2)
     status = EXECUTE(order)
    cmd = cmd + ',v' +  strtrim(string(i),2)
   endfor

  ; Add the filename keyword to save command
  cmd = cmd+",FILENAME='"+fname+"'"
  status = execute(cmd) ; execute the save command
endelse
end



FUNCTION restore_mystruct,fname
; declare variables which exist at top level
COMMON CDFmySHARE, v0  ,v1, v2, v3, v4, v5, v6, v7, v8, v9,$
                   v10,v11,v12,v13,v14,v15,v16,v17,v18,v19,v20
; Use the IDL restore feature to reconstruct the anonymous structure a
RESTORE,FILENAME=fname
; The anonymous structure should now be in the variable 'a'.  Determine
; if the structure contains .DAT or .HANDLE fields
ti = tagindex('HANDLE',tag_names(a.(0)))
if ti ne -1 then begin
  tn = tag_names(a) & nt = n_elements(tn) ; determine number of variables
  for i=0,nt-1 do begin
    a.(i).HANDLE = handle_create()
    order = 'handle_value,a.(i).HANDLE,v' + strtrim(string(i),2) + ',/SET'
    status = EXECUTE(order)
  endfor
endif
return,a
end


; Return the data for the given variable in the given structure
FUNCTION get_mydata,a,var
; Determine the variable number
s = size(var) & ns = n_elements(s) & atags = tag_names(a)
if s(ns-2) eq 7 then begin
  w = where(atags eq var,wc)
  if wc gt 0 then vnum = w(0) $
  else begin
    print,'ERROR>get_mydata:named variable not in structure!' & return,-1
  endelse
endif else vnum = var

; Retrieve the data for the variable
vtags = tag_names(a.(vnum))
ti = tagindex('HANDLE',vtags)
if ti ne -1 then handle_value,a.(vnum).handle,d $
else begin
  ti = tagindex('DAT',vtags)
  if ti ne -1 then d = a.(vnum).dat $
  else begin
    print,'ERROR>get_mydata:variable has neither HANDLE or DAT tag!'
    return,-1
  endelse
endelse
d = reform(d)
return,d
end


; Return the idl sizing information for the data in the given variable
FUNCTION get_mydatasize, a, var
; Determine the variable number
s = size(var) & ns = n_elements(s) & atags = tag_names(a)
if s(ns-2) eq 7 then begin
  w = where(atags eq var,wc)
  if wc gt 0 then vnum = w(0) $
  else begin
    print,'ERROR>get_mydata:named variable not in structure!' & return,-1
  endelse
endif else vnum = var

; Retrieve the idl sizing information for the variable
vtags = tag_names(a.(vnum))
ti = tagindex('HANDLE',vtags) ; search for handle tag
if ti ne -1 then begin
  ti = tagindex('IDLSIZE',vtags) ; search for idlsize tag
  if ti ne -1 then isize = a.(vnum).IDLSIZE $
  else begin ; must retrieve data to get the size
    handle_value,a.(vnum).handle,d & isize = size(d)
  endelse
endif else begin ; search for dat tag
  ti = tagindex('DAT',vtags)
  if ti ne -1 then isize = size(a.(vnum).dat) $
  else begin
    print,'ERROR>get_mydata:variable has neither HANDLE or DAT tag!'
    return,-1
  endelse
endelse
return,isize
end


; IDL always stores structure tags in uppercase.  The ISTP/IACG CDF
; Guidelines show that most required global attributes are not in
; uppercase.  This function performs a case-check on input attribute
; names, and returns the proper case according to the guidelines.
; Unrecognized attribute names are returned without change.
FUNCTION ISTP_gattr_casecheck, a
case a of
  'PROJECT'                    : a = 'Project'
  'DISCIPLINE'                 : a = 'Discipline'
  'SOURCE_NAME'                : a = 'Source_name'
  'DESCRIPTOR'                 : a = 'Descriptor'
  'DATA_TYPE'                  : a = 'Data_type'
  'DATA_VERSION'               : a = 'Data_version'
  'ADID_REF'                   : a = 'ADID_ref'
  'LOGICAL_FILE_ID'            : a = 'Logical_file_id'
  'LOGICAL_SOURCE'             : a = 'Logical_source'
  'LOGICAL_SOURCE_DESCRIPTION' : a = 'Logical_source_description'
  'PI_NAME'                    : a = 'PI_name'
  'PI_AFFILIATION'             : a = 'PI_affiliation'
  'MISSION_GROUP'              : a = 'Mission_group'
  'INSTRUMENT_TYPE'            : a = 'Instrument_type'
  else                         : b = 0 ; do nothing
endcase
return,a
end


; Determine name for a cdf file given the contents of the data structure
; and the ISTP/IACG filenaming conventions.
FUNCTION autoname_mycdf, a, longtime=longtime, bothtimes=bothtimes,  $
                            uppercase=uppercase, lowercase=lowercase

; Determine the variable that contains the timing information
atags = tag_names(a) & tvar = -1
for i=0,n_elements(atags)-1 do begin
  btags = tag_names(a.(i)) & w = where(btags eq 'CDFTYPE',wc)
  if (wc gt 0) then if (a.(i).CDFTYPE eq 'CDF_EPOCH') then tvar = i
endfor
; Now that the 'tvar' is found, Determine the start and stop time of the data
if (tvar ne -1) then begin
  d = get_mydata(a,tvar) & w = where(d gt 0.0D0,wc) & n = n_elements(w)-1
  if (wc le 0) then begin
    stime = '00000000' & ptime = '00000000'
    if keyword_set(LONGTIME) then begin
      stime = stime + '00' & ptime = ptime + '00'
    endif
  endif else begin
    btime = decode_cdfepoch(d(w(0))) & s = parse_mystring(btime)
    stime = s(0) + s(1) + s(2)
    if keyword_set(LONGTIME) then stime = stime + s(3)
    etime = decode_cdfepoch(d(w(n))) & s = parse_mystring(etime)
    ptime = s(0) + s(1) + s(2)
    if keyword_set(LONGTIME) then ptime = ptime + s(3)
  endelse
  d = 0 ; free the data space
endif

; Determine the Logical source for using metadata from the structure
lsource = '' & atags = tag_names(a.(0)) ; get names of the epoch attributes
w = where(atags eq 'LOGICAL_SOURCE',wc)
if (wc gt 0) then lsource = lsource + a.(0).(w(0))
if (strlen(lsource) le 1) then begin ; construct lsource from other info
  s = '' & t = '' & d = ''
  w = where(atags eq 'SOURCE_NAME',wc)
  if (wc gt 0) then begin
    s = break_mystring(a.(0).(w(0)),delimiter='>')
    if strlen(s(0)) gt 2 then s = strmid(s,0,2)
  endif
  w = where(atags eq 'DATA_TYPE',wc)
  if (wc gt 0) then t = break_mystring(a.(0).(w(0)),delimiter='>')
  w = where(atags eq 'DESCRIPTOR',wc)
  if (wc gt 0) then d = break_mystring(a.(0).(w(0)),delimiter='>')
  lsource = s(0) + '_' + t(0) + '_' + d(0)
endif

; Determine the version of the cdf file
v = '01' & w = where(atags eq 'DATA_VERSION',wc)
if (wc gt 0) then v = a.(0).(w(0))
if strlen(v) lt 2 then v = '0' + v

; create the filename
fname = lsource + '_' + stime
if keyword_set(BOTHTIMES) then fname = fname + '_' + ptime
fname = fname + '_v' + v
if keyword_set(LOWERCASE) then fname = strlowcase(fname)
if keyword_set(UPPERCASE) then fname = strupcase(fname)
stop

; update the logical file id attribute
atags = tag_names(a)
for i=0,n_elements(atags)-1 do begin
  btags = tag_names(a.(i)) & w = where(btags eq 'LOGICAL_FILE_ID',wc)
  if (wc gt 0) then a.(i).LOGICAL_FILE_ID = fname
endfor

; create the cdf filename by adding the cdf suffix
fname = fname + '.cdf'
if keyword_set(LOWERCASE) then fname = strlowcase(fname)
if keyword_set(UPPERCASE) then fname = strupcase(fname)
return,fname
end



; Modify the given tag (name or number) in the given variable (name or number)
; in the given structure 'a' with the new value.
FUNCTION modify_mystruct,a,var,tag,value
; Initialize
atags = tag_names(a)

; Determine the variable number and validate
s = size(var) & ns = n_elements(s)
if s(ns-2) eq 7 then begin ; variable is given as a variable name
  w = where(atags eq strupcase(var),wc)
  if wc gt 0 then vnum = w(0) $
  else begin
    print,'ERROR>modify_mystruct:named variable not in structure!' & return,-1
  endelse
endif else begin
  if ((var ge 0)AND(var lt n_elements(atags))) then vnum = var $
  else begin
    print,'ERROR>modify_mystruct:variable# not in structure!' & return,-1
  endelse
endelse
vtags = tag_names(a.(vnum))

; Determine the tag number and validate
s = size(tag) & ns = n_elements(s)
if s(ns-2) eq 7 then begin ; tag is given as a tag name
  w = where(vtags eq strupcase(tag),wc)
  if wc gt 0 then tnum = w(0) $
  else begin
    print,'ERROR>modify_mystruct:named tag not in structure!' & return,-1
  endelse
endif else begin
  if ((tag ge 0)AND(tag lt n_elements(vtags))) then tnum = tag $
  else begin
    print,'ERROR>modify_mystruct:tag# not in structure!' & return,-1
  endelse
endelse

; Create and return new structure with only the one field modified
for i=0,n_elements(atags)-1 do begin ; loop through every variable
  if (i ne vnum) then b = a.(i) $ ; no changes to this variable
  else begin ; must handle this variable field by field
    tnames = tag_names(a.(i))
    for j=0,n_elements(tnames)-1 do begin
      if (j ne tnum) then c = create_struct(tnames(j),a.(i).(j)) $ ; no changes
      else c = create_struct(tnames(j),value) ; new value for this field
      ; Add the structure 'c' to the substructure 'b'
      if (j eq 0) then b = c $ ; create initial structure
      else b = create_struct(b,c) ; append to existing structure
    endfor
  endelse
  ; Add the substructure 'b' to the megastructure
  if (i eq 0) then aa = create_struct(atags(i),b) $ create initial structure
  else begin ; append to existing structure
    c = create_struct(atags(i),b) & aa = create_struct(aa,c)
  endelse
endfor
return,aa
end


; Subset all time dependent variables in the structure 'a' to the times
; specified by the tstart and tstop parameters.
FUNCTION timeslice_mystruct,a,tstart,tstop,NOCOPY=NOCOPY

; Convert tstart to DOUBLE if in string format
s = size(tstart) & ns = n_elements(s)
if s(ns-2) eq 7 then tstart = encode_cdfepoch(tstart) $
else if s(ns-2) ne 5 then begin
  print,'ERROR>timeslice:unknown datatype for the tstart parameter!' & return,a
endif

; Convert tstop to DOUBLE if in string format
s = size(tstop) & ns = n_elements(s)
if s(ns-2) eq 7 then tstop = encode_cdfepoch(tstop) $
else if s(ns-2) ne 5 then begin
  print,'ERROR>timeslice:unknown datatype for the tstop parameter!' & return,a
endif

; Initialize loop
b = a ; copy the input structure for modification
btags = tag_names(b) & nbtags = n_elements(btags)

; Loop through all variables searching for those typed as CDF_EPOCH.
for i=0,nbtags-1 do begin
  vtags = tag_names(b.(i)) & nvtags = n_elements(vtags)
  if b.(i).CDFTYPE eq 'CDF_EPOCH' then begin
    d = get_mydata(b,i) ; retrieve the timing data
    w = where(d ge tstart,wc) ; locate begining record of time span
    if wc eq 0 then begin
      print,'ERROR>timeslice:no data after tstart!' & return,a
    endif else rbegin = w(0)
    w = where(d le tstop,wc) ; locate last record of time span
    if wc eq 0 then begin
      print,'ERROR>timeslice:no data before tstop!' & return,a
    endif else rend = w(n_elements(w)-1)

    ; Subset the variable and plug the data back into a new structure
    d = d(rbegin:rend)
    if (vtags(nvtags-1) eq 'HANDLE') then begin
      newhandle = handle_create()                 ; create new handle
      handle_value,newhandle,d,/set               ; set handle value
      b = modify_mystruct(b,i,'HANDLE',newhandle) ; modify structure
    endif else b = modify_mystruct(b,i,'DAT',d)

    ; Loop through all variables for those which depend on this variable
    for j=0,nbtags-1 do begin
      ti = tagindex('DEPEND_0',tag_names(b.(j)))
      if ti ne -1 then begin
        if b.(j).DEPEND_0 eq b.(i).VARNAME then begin
          d = get_mydata(b,j) ; retrieve the data
          ds = size(d) & nds = n_elements(ds)
          case ds(0) of ; subset the data
            0: print,'ERROR>timeslice:cannot subset vars with 0 dims!'
            1: d = reform(d(rbegin:rend))
            2: d = reform(d(*,rbegin:rend))
            3: d = reform(d(*,*,rbegin:rend))
            else : print,'ERROR>timeslice:Cannot subset vars with > 3 dims!'
          endcase
          if (vtags(nvtags-1) eq 'HANDLE') then begin
            newhandle = handle_create()                 ; create new handle
            handle_value,newhandle,d,/set               ; set handle value
            b = modify_mystruct(b,j,'HANDLE',newhandle) ; modify structure
          endif else b = modify_mystruct(b,j,'DAT',d)
        endif
      endif
    endfor

  endif
endfor
return,b
end


; Prior to destroying or deleting one of the anonymous structures, determine
; if any data handles exists, and if so, free them.
PRO delete_myhandles,a
for i=0,n_elements(tag_names(a))-1 do begin
  ti = tagindex('HANDLE',tag_names(a.(i)))
  if ti ne -1 then begin
    b = handle_info(a.(i).HANDLE,/valid_id)
    if b eq 1 then handle_free,a.(i).HANDLE
  endif
endfor
end


FUNCTION compare_vars, a, b
sa = size(a) & nsa = n_elements(sa)
sb = size(b) & nsb = n_elements(sb)
if (nsa ne nsb) then return,0
for i=0,nsa-1 do if (sa(i) ne sb(i)) then return,0
case sa(0) of
  0    : if (a ne b) then return,0
  1    : for i=0,sa(1)-1 do if (a(i) ne b(i)) then return,0
  2    : begin
         for i=0,sa(1)-1 do begin
           for j=0,sa(2)-1 do if (a(i,j) ne b(i,j)) then return,0
         endfor
         end
  else : print,'WARNING>cannot yet compare vars with > 2 dimensions!'
endcase
return,1
end

; Determine all information about the variable in the varstruct parameter,
; which is required in order to create the variable in a CDF file
FUNCTION create_myCDF_variable,id,varstruct,DEBUG=DEBUG
vid = -1
vname = varstruct.VARNAME ; Determine the name of the variable
; Determine IDL sizing information about the data
ti = tagindex('HANDLE',tag_names(varstruct))
if ti ne -1 then begin
  handle_value,varstruct.HANDLE,d & c = size(d)
endif else begin
  d = varstruct.DAT & c = size(d)
endelse
nc = n_elements(c)

; Determine if this variable is RV or NRV
nrv = 0L & rv = 0L ; initialize
ti = tagindex('VAR_TYPE',tag_names(varstruct))
if (ti ne -1) then begin ; var_type is present
  if (strupcase(varstruct.VAR_TYPE) eq 'METADATA') then nrv=1L else rv=1L
endif else rv=1L ; assume RV

; Determine the dimensionality and the data type based on record variance
if (rv eq 1L) then begin
  case c(0) of
    0   : begin
            print,'ERROR>size of data cannot be 0! - write_mycdf internal error'
            dims = 0L & dvar=[0]
          end
    1   : begin & dims = 0L & dvar=[0] & end
    2   : begin & dims = c(1) & dvar=[1] & end
    3   : begin & dims = [c(1),c(2)] & dvar=[1,1] & end
    4   : begin & dims = [c(1),c(2),c(3)] & dvar=[1,1,1] & end
    else: print,'WARNING>cannot write cdfs with vars with > 3 dimensions!'
  endcase
endif
if (nrv eq 1L) then begin
  case c(0) of
    0   : begin
            print,'ERROR>size of data cannot be 0! - write_mycdf internal error'
            dims = 0L & dvar=[0]
          end
    1   : begin & dims = c(1) & dvar=[1] & end
    2   : begin & dims = [c(1),c(2)] & dvar=[1,1] & end
    3   : begin & dims = [c(1),c(2),c(3)] & dvar=[1,1,1] & end
    else: print,'WARNING>cannot write cdfs with vars with > 3 dimensions!'
  endcase
endif

; Determine the type of the CDF variable
dtype = lonarr(14) & nelems=1L ; initialize
case c(nc-2) of
  0   : print,'ERROR>Undefined data type'
  1   : dtype(0) = 1L ; cdf_byte
  2   : dtype(6) = 1L ; cdf_int2
  3   : dtype(7) = 1L ; cdf_int4
  4   : dtype(8) = 1L ; cdf_real4
  5   : begin ; determine if it is real8 or epoch
          ; determine if a CDFTYPE tag is present, if not then assume real8
          if tagindex('CDFTYPE',tag_names(varstruct)) eq -1 then dtype(9)=1L $
          else begin
            if varstruct.CDFTYPE eq 'CDF_EPOCH' then dtype(3) = 1L $
            else dtype(9) = 1L ; cdf_real8
          endelse
        end
  6   : print,'WARNING>CDF does not have complex_float type'
  7   : begin ; cdf_char
          dtype(1) = 1L
          nelems = strlen(d(0))
        end
  8   : print,'WARNING>CDF does not have structure type'
  9   : print,'WARNING>CDF does not have double_complex_float type'
  else: print,'ERROR>Unknown IDL data type'
endcase

; Create the variable
if keyword_set(DEBUG) then begin
  print,'creating the variable:',vname
  print,'rv=  ',rv,  ' nrv=  ',nrv,' nelems=',nelems
  print,'dims=',dims,' dvary=',dvar
endif
if (dims(0) eq 0) then begin
  vid = cdf_varcreate(id,vname,/ZVARIABLE,NUMELEM=nelems,$
        CDF_BYTE=dtype(0),CDF_CHAR=dtype(1),CDF_DOUBLE=dtype(2),$
        CDF_EPOCH=dtype(3),CDF_FLOAT=dtype(4),CDF_INT1=dtype(5),$
        CDF_INT2=dtype(6),CDF_INT4=dtype(7),CDF_REAL4=dtype(8),$
        CDF_REAL8=dtype(9),CDF_UCHAR=dtype(10),CDF_UINT1=dtype(11),$
        CDF_UINT2=dtype(12),CDF_UINT4=dtype(13),$
        REC_NOVARY=nrv,REC_VARY=rv)
endif else begin
  vid = cdf_varcreate(id,vname,dvar,DIMENSIONS=dims,NUMELEM=nelems,$
        CDF_BYTE=dtype(0),CDF_CHAR=dtype(1),CDF_DOUBLE=dtype(2),$
        CDF_EPOCH=dtype(3),CDF_FLOAT=dtype(4),CDF_INT1=dtype(5),$
        CDF_INT2=dtype(6),CDF_INT4=dtype(7),CDF_REAL4=dtype(8),$
        CDF_REAL8=dtype(9),CDF_UCHAR=dtype(10),CDF_UINT1=dtype(11),$
        CDF_UINT2=dtype(12),CDF_UINT4=dtype(13),$
        REC_NOVARY=nrv,REC_VARY=rv)
endelse

; write the data into the cdf with special handling for character data
if c(nc-2) ne 7 then cdf_varput,id,vname,d $
else begin ; special processing for character data
  if ((c(0) eq 0)AND(d(0) ne '')) then cdf_varput,id,vname,d $
  else begin ; data is a string array
    ; pad all elements to same length and concatenate into single buffer
    maxlength = max(strlen(d)) & buffer = ''
    for j=0,c(1)-1 do begin
      if strlen(d(j)) eq maxlength then buffer = buffer + d(j) $
      else begin
        pad = '' & for g=0,(strlen(d(j))-maxlength)-1 do pad = pad + ' '
        buffer = buffer + d(j) + pad
      endelse
    endfor
    cdf_varput,id,vname,buffer,OFFSET=[0],COUNT=[c(1)]
  endelse
endelse

return,vid
end


; This package of routines complements the read_myCDF package.  read_myCDF
; reads one or more cdf's, and returns all data and metadata as a single
; structure.  write_myCDF will do just the opposite, given a similar structure,
; it will write a cdf.

FUNCTION write_myCDF, a, filename, AUTONAME=AUTONAME,  LONGTIME=LONGTIME,  $
                                   BOTHTIMES=BOTHTIMES,OUTDIR=OUTDIR,      $
                                   UPPERCASE=UPPERCASE,LOWERCASE=LOWERCASE,$
                                   DEBUG=DEBUG

; Identify the global attributes.
b = tag_names(a.(0)) & w = where(b eq 'FIELDNAM',wc) & gattrs=indgen(w(0)-1)+1

; Determine the order of the variables to be written to the CDF.
b = tag_names(a) & nb = n_elements(b) & c = intarr(nb) & d = indgen(nb)
for i=0,nb-1 do begin
  if a.(i).CDFTYPE eq 'CDF_EPOCH' then c(i) = 2 $ ; time variable
  else if strupcase(a.(i).VAR_TYPE) eq 'DATA' then c(i) = 1 ; RV variable
endfor
w2 = where(c eq 2,wc2) & if wc2 gt 0 then s=d(w2)
w1 = where(c eq 1,wc1) & if wc1 gt 0 then s=[s,d(w1)]
w0 = where(c eq 0,wc0) & if wc0 gt 0 then s=[s,d(w0)]
order = s

; Determine the name of the new CDF if autonaming option is turned on
if keyword_set(AUTONAME) then begin
  filename = autoname_mycdf(a,LONGTIME=LONGTIME,BOTHTIMES=BOTHTIMES,$
                            UPPERCASE=UPPERCASE,LOWERCASE=LOWERCASE)
  s = size(filename) & i = n_elements(s)
  if (s(i-2) ne 7) then return,-1 ; fatal error in autoname
endif

; Create the new CDF
if keyword_set(OUTDIR) then filename = OUTDIR + filename
if keyword_set(DEBUG) then print,'Now creating the CDF named:',filename
id = cdf_create(filename,/NETWORK_ENCODING,/SINGLE_FILE,/COL_MAJOR)

; Write global attributes to the CDF
if keyword_set(DEBUG) then print,'Writing global attributes to the CDF...'
b = tag_names(a.(0)) ; get names of attributes
for i=0,n_elements(gattrs)-1 do begin
  g = ISTP_gattr_casecheck(b(gattrs(i)))  ; perform ISTP-case checking
  aid = cdf_attcreate(id,g,/GLOBAL_SCOPE) ; create the attribute
  ; Now put the proper value in the attribute
  s = size(a.(0).(gattrs(i))) & ns = n_elements(s) & c=''
  if (s(ns-2) eq 7) then begin ; special case for string handling
    if s(0) eq 0 then begin ; single string, not an array of strings
      c = a.(0).(gattrs(i)) & if c ne '' then cdf_attput,id,aid,0L,c
    endif else begin ; attribute value is an array of strings
      for j=0,s(1)-1 do begin
        c=a.(0).(gattrs(i))(j) & if c ne '' then cdf_attput,id,aid,j,c
       endfor
    endelse
  endif else cdf_attput,id,aid,0L,a.(0).(gattrs(i))
endfor

; Create the variables
for i=0,n_elements(order)-1 do begin
  b = order(i) & vname = a.(b).VARNAME
  vid = create_myCDF_variable(id, a.(b), DEBUG=DEBUG)
endfor ; create and write all variables

; Write the variable attributes to the CDF
for i=0,n_elements(tag_names(a))-1 do begin ; loop through every variable
  vname = a.(i).VARNAME ; get the case sensitive variable name
  vtags = tag_names(a.(i)) ; get the attribute names
  from = tagindex('FIELDNAM',vtags) ; fieldnam is the first vattr
  to   = tagindex('CDFTYPE' ,vtags) ; cdftype is the next non-vattr
  for j=from,to-1 do begin ; process each variable attribute
    if i eq 0 then aid = cdf_attcreate(id,vtags(j),/VARIABLE_SCOPE) $
    else aid = cdf_attnum(id,vtags(j)) ; get id of existing attribute
    ; Special processing is required for ISTP-stype pointer attributes.
    ; If the current attribute is such an attribute, do not process it here.
    if (amI_ISTPptr(vtags(j)) ne 1) then begin
      s = size(a.(i).(j)) & ns = n_elements(s)
      if (s(ns-2) ne 7) then cdf_attput,id,aid,vname,a.(i).(j) $
      else begin ; special processing for character data
        if s(0) eq 0 then begin
          e = a.(i).(j) & if e ne '' then cdf_attput,id,aid,vname,e
        endif else begin ; data is a string array
          print,'WARNING: ',vtags(j),' vattr not written because of IDL bug!'
        endelse
      endelse
    endif
    endfor
endfor

; Perform special processing for ISTP pointer-type attributes.  When such an
; attribute is located, a new metadata variable may have to be created.  This
; depends on how the original cdf was read.  If the original cdf was read
; with the /all keyword, then all variables, including non-record-varying
; metadata were read into the structure.  If not, then those non-record-
; varying variables may have been lost, in which case new variables must
; be created.
mvcount = 0L
for i=0,n_elements(tag_names(a))-1 do begin ; loop through every variable
  vtags = tag_names(a.(i)) ; get the name of every attribute
  for j=0,n_elements(vtags)-1 do begin
    if ((amI_ISTPptr(vtags(j)) eq 1)AND(a.(i).(j)(0) ne '')) then begin
      ; print,'special processing for istpptr for var ',i,' attr', j

      ; determine if any other variable in the structure has the same
      ; value as a.(i).(j) so that unneeded metavars are not created.
      pvar = -1 ; initialize flag
      ; print,'searching existing variables for correct value'
      for g=0,n_elements(tag_names(a))-1 do begin ; loop through every var
        if compare_vars(a.(i).(j),get_mydata(a,g)) eq 1 then begin
          pvar  = g ; variable with matching value already exists
          vname = (tag_names(a))(g)
          ; print,'value found in the variable ',vname,' pvar=',pvar
        endif
      endfor

      ; if no existing variable in the structure has the correct value,
      ; then determine if any metavar has already been created with that value.
      if pvar eq -1 then begin
        ; print,'searching previously created metavars for correct value'
        for g=0,mvcount-1 do begin
          if compare_vars(a.(i).(j),a.(pv(g)).(pt(g))) eq 1 then begin
            pvar  = g ; same attribute value exists
            vname = pvn(g) ; get name of metavar which already exists
            ; print,'value found in the variable ',vname,' pvar=',pvar
          endif
        endfor
      endif

      ; if pvar still equals -1, then no variable with a matching value exists
      ; in the original structure, or has been previous created as a metavar.
      ; In this case, a new metavar must now be created.
      if (pvar eq -1) then begin
        ; determine the name for new variable
        vname = 'metavar' + strtrim(string(mvcount),2)
        ; print,'creating new metavar named ',vname
        ; create a variable structure
        va = create_struct('VARNAME',vname)
        vb = create_struct('VAR_TYPE','metadata')
        vc = create_struct('DAT',reform(a.(i).(j)))
        v  = create_struct(va,vb) & v = create_struct(v,vc)
        ; create the new variable in the CDF
        vid = create_myCDF_variable(id, v)
        cdf_attput,id,'VAR_TYPE',vname,'metadata'
        cdf_attput,id,'FIELDNAM',vname,vname
        ; record the number of the new variable and attribute tag number
        if mvcount eq 0 then begin
          pv = i & pt = j & pvn = vname
        endif else begin
          pv = [pv,i] & pt = [pt,j] & pvn = [pvn,vname]
        endelse
        mvcount = mvcount + 1 ; increment metavariable count
      endif

      ; point to the correct variable from the istp-pointer type attribute
      ; print,'setting ',a.(i).VARNAME,' ',vtags(j),' to ',vname
      cdf_attput,id,vtags(j),a.(i).VARNAME,vname

    endif
  endfor
endfor

; Close the new CDF
cdf_close,id

return,0
end


