;******************************************************************************
;* PROCEDURE:
;*      papco_autorange
;*
;* DESCRIPTION:
;*	Attempts to establish the range of data values in an array.
;*      In order to discriminate against data glitches
;*      (uncharacteristically low or high values) a top and bottom %
;*      of data pints can be excluded from this process. This also maximizes
;*      the use of the color range.
;*      The minimum for the log range is one decade!
;*
;* INPUTS:
;*	arr       the array to be scaled
;*      
;* OUTPUTS:
;*      min,max   the max and min values for arr
;*
;* KEYWORDS:
;*	LOG       set for logarythmic scaling
;*      NODATA    set to the nodata flag(s) of arr - can be an array
;*      EXCLUDE   set to % of top and bottom data points to exclude
;*      ROBUST_AV normalize all data with respect to a ROBUST_AV per channel
;*      VERBOSE   set to give diagnostic output
;*
;* CALLING SEQUENCE:
;*	papco_autorange,arr,min,max
;*
;* MODIFICATION HISTORY:
;*     written April 1998, Reiner Friedel
;******************************************************************************
pro papco_autorange, arr, min, max, LOG=LOG, NODATA=NODATA, EXCLUDE=EXCLUDE, $
                     ROBUST_AV = ROBUST_AV, VERBOSE=VERBOSE
   
if KEYWORD_SET(NODATA)  then nodata= NODATA  else nodata= 1.0e-20
if KEYWORD_SET(EXCLUDE) then exclude=EXCLUDE else exclude=0
c_tot=n_elements(arr)      ; # aray elements
  
;check if there are ANY vaild data points first
IF n_elements(nodata) EQ 1 THEN idx=where(arr eq nodata, nodat) $
  ELSE BEGIN
      nodat = 0 
      FOR i = 0, n_elements(nodata)-1 DO BEGIN
          index=where(arr eq nodata(i), n)
          IF keyword_set(VERBOSE) THEN $
            message, 'Points flagged '+varprt(nodata(i))+': '+varprt(n), /cont
          IF n NE 0 THEN BEGIN
              nodat = nodat+n
              IF n_elements(idx) EQ 0 THEN idx = index ELSE idx = [idx, index]
          ENDIF
      ENDFOR
  ENDELSE

  IF keyword_set(VERBOSE) THEN $
  message,'Total points: '+varprt(c_tot)+',  Missing: '+varprt(nodat), /cont
  if nodat eq c_tot then goto, no_valid_points

  if KEYWORD_SET(LOG) then begin ; for logarithmic scaling
    IF keyword_set(VERBOSE) THEN $
    message,'Log scaling, clipping '+varprt(exclude)+'%', /cont
    ;check for any negative values and set them to the no data flag.
    ;exclude any nodata values from this test by temporarily setting them
    ;to +ve values
    IF nodat NE 0 THEN BEGIN
        temp = arr(idx) &  arr(idx) = 1
    ENDIF 
    idx_below=where(arr le 0.0,c_below) 
    IF keyword_set(VERBOSE) THEN $
    message,'Data <= zero: '+varprt(c_below)+' out of '+varprt(c_tot),/cont 
    if c_below eq c_tot then goto, no_valid_points
    IF nodat NE 0 THEN idx_nodata = [idx, idx_below] ELSE idx_nodata=idx_below
    if c_below NE 0 then arr(idx_below)=nodata(0) 
    IF nodat NE 0 THEN arr(idx) = temp
    nodat=nodat+c_below 
  endif else begin               ; for linear scaling    
    IF keyword_set(VERBOSE) THEN $
    message,'Lin scaling, clipping '+varprt(exclude)+' %', /cont
    idx_nodata = idx
  endelse   
    
  ;make index array of VALID data points
  idx_data = lindgen(c_tot)
  IF nodat NE 0 THEN BEGIN
      idx_data(idx_nodata) = -1 & valid = where(idx_data NE -1, valdat)
      IF  valdat NE 0 THEN idx_data = idx_data(valid)
  ENDIF ELSE valdat = c_tot

  IF keyword_set(VERBOSE) THEN $
  message,'Total points: '+varprt(c_tot)+',  Valid  : '+varprt(valdat), /cont

  ;find the minimum and maximum value of arr excluding nodata
  ;idx_data=where(arr ne nodata,c_dat)
  minact=min(arr(idx_data),max=maxact)
  min=minact & minact_str=strtrim(string(minact),2)
  max=maxact & maxact_str=strtrim(string(maxact),2) 
  
  ;set the number of point than can be above/below min and find new max/min
  if KEYWORD_SET(EXCLUDE) then BEGIN
    c_dat = valdat
    allowed_beyond=EXCLUDE*c_dat/100.0          
    ;find new min such that EXCLUDE % of points are below min
    c_beyond=0
    while (c_beyond lt allowed_beyond) do begin 
      min=min+min*0.05
      idx=where(arr(idx_data) le min,c_beyond)
    endwhile          
    pc_str=strtrim(string(100*c_beyond/c_dat),2)
    IF keyword_set(VERBOSE) THEN $
    message,'Min: '+minact_str+', New Min: '+varprt(min),/cont
    ;find new max such that EXCLUDE % of points are above max
    c_beyond=0
    while (c_beyond  lt allowed_beyond) do begin 
      max=max-max*0.05
      idx=where(arr(idx_data) ge max,c_beyond)
    endwhile
    pc_str=strtrim(string(100*c_beyond/c_dat),2)      
    IF keyword_set(VERBOSE) THEN $
    message,'Max: '+maxact_str+', New Max: '+varprt(max),/cont        
    if min ge max then begin
      min=minact  &  max=maxact
      IF keyword_set(VERBOSE) THEN $
      message,'Actual max, min used',/cont
    endif  
  endif else     IF keyword_set(VERBOSE) THEN $
    message,'Absolute max: '+maxact_str+'  min: '+minact_str, /cont
            
  if min ge max then begin
    min=minact
    max=maxact
    IF keyword_set(VERBOSE) THEN $
    message,'Actual max, min used, max: '+maxact_str+'  min: '+minact_str,/cont
  endif  

if KEYWORD_SET(LOG) then begin ; for logarithmic scaling
    IF abs(alog10(min)-alog10(max)) LT 1 THEN BEGIN ;less than one decade
        
        min = 10.0^(floor(alog10(min)))
        max = 10.0^(ceil(alog10(max)))
        message,'Full decade, min: '+varprt(min)+'  max: '+varprt(max),/cont
    ENDIF
ENDIF 
    
;do robust average calculation
IF keyword_set(ROBUST_AV) THEN BEGIN
    stop      
    r = size(arr, /st) 
    IF r.N_DIMENSIONS EQ 2 THEN BEGIN
        FOR i = 0, r.DIMENSIONS(1)-1 DO BEGIN
            ch = arr(*, i)
            idx = where(ch NE nodata, c)
            res = moment(ch(idx))
            print,  res(0)
            ch(idx) = ch(idx)/res(0)
            arr(*, i) = ch
        ENDFOR     
    ENDIF 
    idx = where(arr NE nodata)
    min = min( arr(idx), max = max)
ENDIF 

return

; return equal max. min if there are no valid points.
  no_valid_points:  
    max=0 & min=0       ; set max, min and return
    IF keyword_set(VERBOSE) THEN $
    message,'no valid data points', /cont

end

