;+
; NAME: LOGAXIS
;
; PURPOSE:
; This procedure draws nicer labels for logarithmic 
; (and linear) axes for the types given in the IDL-1 AXIS-procedure. 
; The boundaries are taken from the !CRANGE-values.
; Thus the procedure has to be called AFTER
; the PLOT-call. Labels at the top of the axes are only
; plotted if the actual window-boundaries are >= the window-boundaries
; defined by the MPOSITION-common variables.
;	
; CATEGORY: Plotting
;
; CALLING SEQUENCE: LOGAXIS,type [,size,labnum,ticklen=tl,decbit=decbit]
;
; INPUTS:           type: 0: y left lin
;                         1: y left log
;                         2: y right lin
;                         3: y right log
;                         4: x bottom lin
;                         5: x bottom log
;                         6: x top lin
;                         7: x top log
;
; OPTIONAL INPUTS:
;                   size: character size (default:0.9)
;                   labnum: plot only every labnum-th label (def:1)
;
; KEYWORD PARAMETERS:
;       decbit:         (0/1)  for decimal point representation
;       ticklen:	ticklength for 'major' ticks, as a fraction
;			of the plot window size.
;			Default: -0.04
;	
; COMMON BLOCKS:
; The MAIN-viewport boundaries must be defined by:                
;  COMMON MAINVIEW,lposition,mposition
;
; RESTRICTIONS:
; The procedure has to be called AFTER the PLOT-call.
; Labels at the top of the axes are only
; plotted if the actual window-boundaries are >= the window-boundaries
; defined by the MPOSITION-common variables.
;
; PROCEDURE:
;
; MODIFICATION HISTORY:
; 	Written by:M.Fraenz 92 IDL1
;       12-jun-93: IDL3-version setting decimal numbers if the
;                  range is between 0.001 and 100.0, using
;                  TICKNAME and TICKV sysvars.
;
;-
  PRO LOGAXIS,type,size,labnum,ticklen=tl,decbit=decbit
  maxadd=0
;
; The MAIN-viewport boundaries SC20 and SC40 must be defined by:
  COMMON MAINVIEW,lposition,mposition
  IF NOT(keyword_set(decbit)) THEN decbit=0
  pa=!p
  xa=!x
  ya=!y
;
;
  IF N_PARAMS(0) LT 3 THEN labnum=1
  IF N_PARAMS(0) LT 2 THEN size=0.9
  IF N_ELEMENTS(tl) EQ 0 THEN !p.ticklen = -0.04 ELSE !p.ticklen=tl
  !p.charsize=size
;
  type=fix(type)
  linlogbit = (type AND 1)
;
  IF type LT 4 THEN BEGIN  ;******************* Y-axes
   absticks=abs(!y.ticks)
;   IF !p.position(3) GE mposition(3)-0.01 THEN maxadd=0 ELSE maxadd=-1
   dbottom=double(!y.crange(0))
   dtop=double(!y.crange(1))
   !y.tickname=' '
   !y.tickv(*)=0.0
   !y.style=1
  ENDIF ELSE BEGIN        ;******************* X-axes
   absticks=abs(!x.ticks)
;   IF !p.position(2) GE mposition(2)-0.01 THEN maxadd=0 ELSE maxadd=-1
   dbottom=double(!x.crange(0))
   dtop=double(!x.crange(1))
   !x.tickname=' '
   !x.tickv(*)=0.0
   !x.style=1
  ENDELSE
;
  IF linlogbit EQ 0 THEN BEGIN
   IF absticks LE 1 THEN BEGIN
    print,'xticks:',!x.ticks,'yticks:',!y.ticks
    message,' .....Linear Ticks not set..........' 
   ENDIF
   inc=(dtop-dbottom)/absticks*labnum
   absmax=abs(dbottom) > abs(dtop)
   pos =dbottom
   FOR i=0,absticks+maxadd,labnum DO BEGIN
     exponent=fixmin(alog10(pos))
     posfac=pos mod 10.0^exponent
     numstr=fltstr(posfac,exponent,1)
     IF type LT 3 THEN exp_out=numstr $
     ELSE exp_out=STRTRIM(numstr,2)
     IF type LT 4 THEN BEGIN
      !y.tickname(i)=exp_out 
      !y.tickv(i)=pos
     ENDIF ELSE BEGIN
      !x.tickname(i)=exp_out
      !x.tickv(i)=pos
     ENDELSE
     pos=pos+inc
   ENDFOR
   PLOT,!x.crange,!y.crange,/nodata,xstyle=1+4,ystyle=1+4,/noerase
   CASE type OF
    0:	AXIS,yaxis=0,ytype=0
    2:	AXIS,/yaxis,ytype=0
    4:  AXIS,xaxis=0,xtype=0
    6:	AXIS,/xaxis,xtype=0
   ENDCASE
;
  ENDIF ELSE BEGIN   ;************logarithmic
   expbot=FIXmin(dbottom)
   exptop=FIXmax(dtop)
   IF (decbit EQ 0) AND ((exptop-expbot) LE 2) THEN krbit=1 ELSE krbit=0
   exend=(exptop+maxadd) > expbot
; BIG labels:
   j=0
   tenbit=0
   FOR exppos=expbot,exend,labnum DO BEGIN
    FOR i=1,9 DO BEGIN
      pos=i*10.0^exppos
      IF (pos GE float(10.0d^dbottom)) AND (pos LE float(10.0d^dtop)) THEN BEGIN
       IF (tenbit EQ 0) OR (i EQ 1) OR (decbit EQ 1) THEN BEGIN
        IF (type LT 4) THEN BEGIN 
         !y.tickname(j)=FLTSTR(i,exppos,decbit)
         !y.tickv(j)=pos
        ENDIF ELSE BEGIN
         !x.tickname(j)=FLTSTR(i,exppos,decbit)
         !x.tickv(j)=pos
        ENDELSE
        j=j+1
        tenbit=1
       ENDIF
      ENDIF
    ENDFOR
   ENDFOR

   IF (type LT 4) THEN BEGIN
       IF exptop-expbot GT 3 THEN !y.minor=9 ELSE !y.minor=-1
       index=where(!y.tickv GT 0.0,count)
        if count eq 0 then begin
          count=3
          !y.tickv(0)=alog10(!Y.crange(0))
          !y.tickv(1)=alog10(!Y.crange(1))
        endif 
        AXIS,ytype=1,yaxis=(type AND 2)/2,yticks=count-1,/save
   PLOT,!x.crange,10^!y.crange,/nodata,ytype=1,xstyle=5,ystyle=5, /noerase
       AXIS,ytype=1,yaxis=(type AND 2)/2,yticks=count-1,/save
   ENDIF ELSE BEGIN
       IF exptop-expbot GT 3 THEN !x.minor=9 ELSE !x.minor=-1
       index=where(!x.tickv GT 0.0,count)
        if count eq 0 then begin
          count=3
          !x.tickv(0)=alog10(!x.crange(0))
          !x.tickv(1)=alog10(!x.crange(1))
        endif 
        AXIS,ytype=1,yaxis=(type AND 2)/2,yticks=count-1,/save
       PLOT,10.^!x.crange,!y.crange,/nodata,xtype=1,ystyle=5,xstyle=5,/noerase
       AXIS,xtype=1,xaxis=(type AND 2)/2,xticks=count-1,/save
   ENDELSE
   
; small labels:
  IF exptop-expbot LE 3 THEN BEGIN  
   IF (type LT 4) THEN BEGIN 
     !y.tickname(*)=' '
     !y.tickv(*)=0.0
     IF exptop-expbot LT 2 THEN !y.minor=9
     PLOT,!x.crange,10.^!y.crange,/nodata,ytype=1,xstyle=1+4,ystyle=1+4,/noerase
    ENDIF ELSE BEGIN
     !x.tickname(*)=' '
     !x.tickv(*)=0.0
     IF exptop-expbot LT 2 THEN !x.minor=9
     PLOT,10.^!x.crange,!y.crange,/nodata,xtype=1,ystyle=1+4,xstyle=1+4,/noerase
    ENDELSE
    j=0
    FOR exppos=expbot,exptop,labnum DO BEGIN
     tenbit=0
     FOR i=2,9 DO BEGIN
      pos=i*10.0^exppos
      IF (pos GE 10.0^dbottom) AND (pos LE 10.0^dtop) THEN BEGIN
       IF ((i EQ 2) OR (i EQ 4) OR (i EQ 8)) AND (krbit EQ 1) THEN BEGIN
        IF (type LT 4) THEN !y.tickname(j)=STRTRIM(i,2)  $
        ELSE !x.tickname(j)=STRTRIM(i,2) 
       ENDIF
       IF (type LT 4) THEN !y.tickv(j)=pos $
       ELSE !x.tickv(j)=pos
       j=j+1
      ENDIF 
     ENDFOR
    ENDFOR
    !p.ticklen= 0.5*!p.ticklen
    !p.charsize=size*0.6

    IF (type LT 4) THEN BEGIN
        index=where(!y.tickv GT 0.0,count)
        if count eq 0 then begin
          count=3
          !y.tickv(0)=alog10(!Y.crange(0))
          !y.tickv(1)=alog10(!Y.crange(1))
        endif 
        AXIS,ytype=1,yaxis=(type AND 2)/2,yticks=count-1,/save
    ENDIF ELSE BEGIN
        index=where(!x.tickv GT 0.0,count)
        if count eq 0 then begin
          count=3
          !x.tickv(0)=alog10(!x.crange(0))
          !x.tickv(1)=alog10(!x.crange(1))
        endif      
    	AXIS,xtype=1,xaxis=(type AND 2)/2,xticks=count-1,/save
    ENDELSE
  ENDIF ; exptop-expbot 
 ENDELSE ;linlogbit
exit_out:
  !p=pa
  !x=xa
  !y=ya
  RETURN
 END
