;*****************************************************************************
; This file is part of the PAPCO software package. 
;
; PAPCO is a free IDL package to combine data plots written by A. Keese and
; R. Friedel, maintained by R. Friedel
;                                                           
;    For more information, check the papco home page at     
;    http://leadbelly/ccr/software/papco/papco.html       
;    for online Manual and downloadble latest version.      
;                                                           
; Comments and questions can be directed to friedel@lanl.gov  
;  
; You are free to modify and distribute this file. However, any changes not 
; communicated to R. Friedel will not become part of the next release of PAPCO.
;*****************************************************************************
;+
; NAME:
;     papco_y_label
;
; PURPOSE:
;     Attempt to give "nice" y-labels given the range for a plot. 
;
; CATEGORY:
;     Plotting.
;
; CALLING SEQUENCE:
;     papco_y_label, Yst, Yen
;
; INPUTS:
;     Yst:    Start value of Y-axis
;     Yen:    End  value of Y-axis
;
; KEYWORD PARAMETERS:
;     INFO:     Calls doc_library to display this header documentation.
;               1: print to terminal, 2: print to default printer
;
;     VERBOSE:  If set, output informational IF verbose then s to console.
;
;     LOG:      Do for log, full powers of 10 only. Default is for linear.
;
; OUTPUTS:
;     Describe any outputs here.  For example, "This function returns the
;     foobar superflimpt version of the input array."  This is where you
;     should also document the return value for functions.
;
; COMMON BLOCKS:
;     PLOT_COMPOSER: contains the main papco structure
;                    
; SIDE EFFECTS:
;     Alters IDL system variables for plotting:
;     !Y.MINOR, !Y.TICKV, !Y.TICKS  
;
; RESTRICTIONS:
;     Routine exits with !Y.MINOR, !Y.TICKV, !Y.TICKS all set to zero
;     if !Y.TICKS is set to zero (defaults to IDL's internal automatic
;     y-axis labeling)
;
; MODIFICATION HISTORY:
;     Written by:    R. Friedel, February 2000.
;-
;*****************************************************************************

PRO papco_y_label, Yst, Yen, LOG = LOG, INFO = INFO, VERBOSE = VERBOSE, $
                   VALID = VALID
  
;start section common to ALL papco routines ----------------------------------
COMMON PLOT_COMPOSER, widgetData

IF keyword_set(INFO) THEN BEGIN
    IF info EQ 2 THEN print = 1 ELSE print = 0
    doc_library, 'papco_y_label', PRINT = print
    return
ENDIF

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0
IF widgetData.default_verbose THEN verbose = 1
;end section common to ALL papco routines ------------------------------------

!Y.TICKFORMAT = ''

IF !Y.TICKS EQ 1 THEN BEGIN  ;supress tick marks
    !Y.MINOR = -1 & !Y.TICKV = 0 & !Y.TICKFORMAT = 'noticks'
    return                 
ENDIF 

IF !Y.TICKS EQ 0 THEN BEGIN ;allow automatic
    !Y.MINOR=0  &  !Y.TICKV=0  &  return 
ENDIF

IF keyword_set(LOG) THEN BEGIN  ;do for log scales

    ;pretty restrictive, as we only want full powers of ten labeled.
    y1=alog10(Yst)  &  y2=alog10(Yen)  &  diff=y2-y1
    
    valid_incr=ceil(diff/!Y.TICKS)
    labels=[ceil(y1)]  &  nextlabel=labels(0)+valid_incr  &  label_no=0
    
    while nextlabel le y2 do begin
      label_no=label_no+1
      labels=[labels,nextlabel]
      nextlabel=nextlabel+valid_incr
    endwhile
    
    labels=10.0^float(labels)
    !Y.MINOR=valid_incr

ENDIF ELSE BEGIN                ;do for linear scales
    
    IF keyword_set(VALID) THEN valid = VALID ELSE valid=[1,2,4,5,8,10]
    valid_minor=valid
    incr=float(Yen-Yst)/(!Y.TICKS-1)
  
    ;want increment to be nice round numbers.
    i=0
    while incr lt 1 do begin & incr=incr*10 & i=i+1 & endwhile
    j=0
    while incr gt 10 do begin & incr=incr/10 & j=j+1 & endwhile  
  
    diff=abs(valid-incr)  &  result=min(diff,min_idx)
    valid_incr=valid(min_idx)  &  gap=valid_incr  & mult=(10.0^j)/10.0^i
    valid_incr=valid_incr*mult
  
    IF verbose then message, 'Valid increment: '+varprt(valid_incr),/cont
  
    ;now find good staring /end point
    inc_st=round(Yst/valid_incr)
    
    new_Yst=inc_st*valid_incr
    if new_Yst lt Yst then new_Yst=new_Yst+valid_incr
    IF verbose then message,'New Yst: '+varprt(new_Yst),/cont
    
    label_no=1
    labels=[new_Yst]    &    new_label=labels(label_no-1)+valid_incr
    while new_label le Yen do begin
        labels=[labels,new_label]
        new_label=labels(label_no)+valid_incr
        label_no=label_no+1
    endwhile
    label_no=label_no-1
    IF verbose then message,'No of ticks asked: '+varprt(!Y.TICKS)+ $
        ', got: '+varprt(label_no),/cont
     
    ;now determine number of minor ticks.
    test=valid_minor
    if valid_incr lt 1 then valid_incr=valid(min_idx)
    if valid_incr gt 10 then valid_incr=valid_incr/mult
    for i=0,n_elements(valid_minor)-1 do begin
        mgap=valid_minor(i)
        IF valid_incr EQ fix(valid_incr/mgap)*mgap THEN $
          test(i) = 1 ELSE test(i)=0
    ENDFOR 
    index=where(test eq 1)
    valid_minor=valid_minor(index)
    valid_minor=max(valid_minor)
    IF verbose then message,'Minor ticks: '+varprt(valid_minor),/cont
    
    !Y.MINOR=valid_minor
    !Y.TICKV=labels
    !Y.TICKS=label_no
        
ENDELSE 
    
!Y.TICKV=labels
!Y.TICKS=label_no    
    
END
