	FUNCTION FIND_ALL_DIR, PATH, PATH_FORMAT=PATH_FORMAT,	$
		PLUS_REQUIRED=PLUS_REQUIRED
;+
; Project     :	SOHO - CDS
;
; Name        :	FIND_ALL_DIR()
;
; Purpose     :	Finds all directories under a specified directory.
;
; Explanation :	This routines finds all the directories in a directory tree
;		when the root of the tree is specified.  This provides the same
;		functionality as having a directory with a plus in front of it
;		in the environment variable IDL_PATH.
;
; Use         :	Result = FIND_ALL_DIR( PATH )
;
;		PATHS = FIND_ALL_DIR('+mypath', /PATH_FORMAT)
;		PATHS = FIND_ALL_DIR('+mypath1:+mypath2')
;
; Inputs      :	PATH	= The path specification for the top directory in the
;			  tree.  Optionally this may begin with the '+'
;			  character but the action is the same unless the
;			  PLUS_REQUIRED keyword is set.
;
;			  One can also path a series of directories separated
;			  by the correct character ("," for VMS, ":" for Unix)
;
; Opt. Inputs :	None.
;
; Outputs     :	The result of the function is a list of directories starting
;		from the top directory passed and working downward from there.
;		Normally, this will be a string array with one directory per
;		array element, but if the PATH_FORMAT keyword is set, then a
;		single string will be returned, in the correct format to be
;		incorporated into !PATH.
;
; Opt. Outputs:	None.
;
; Keywords    :	PATH_FORMAT	= If set, then a single string is returned, in
;				  the format of !PATH.
;
;		PLUS_REQUIRED	= If set, then a leading plus sign is required
;				  in order to expand out a directory tree.
;				  This is especially useful if the input is a
;				  series of directories, where some components
;				  should be expanded, but others shouldn't.
;
; Calls       :	FIND_WITH_DEF, BREAK_PATH
;
; Common      :	None.
;
; Restrictions:	PATH must point to a directory that actually exists.
;
;		On VMS computers this routine calls a command file,
;		FIND_ALL_DIR.COM, to find the directories.  This command file
;		must be in one of the directories in IDL's standard search
;		path, !PATH.
;
; Side effects:	None.
;
; Category    :	Utilities, Operating_system.
;
; Prev. Hist. :	None.
;
; Written     :	William Thompson, GSFC, 3 May 1993.
;
; Modified    :	Version 1, William Thompson, GSFC, 3 May 1993.
;		Version 2, William Thompson, GSFC, 6 July 1993.
;			Added sort to spawned command under Unix.
;		Version 3, William Thompson, GSFC, 16 May 1995
;			Modified to support multiple directories.
;			Added keyword PLUS_REQUIRED
;
; Version     :	Version 3, 16 May 1995
;-
;
	ON_ERROR, 2
;
	IF N_PARAMS() NE 1 THEN MESSAGE,	$
		'Syntax:  Result = FIND_ALL_DIR( PATH )'
;
;  If more than one directory was passed, then call this routine reiteratively.
;  Then skip directly to the test for the PATH_FORMAT keyword.
;
	PATHS = BREAK_PATH(PATH, /NOCURRENT)
	IF N_ELEMENTS(PATHS) GT 1 THEN BEGIN
		DIRECTORIES = FIND_ALL_DIR(PATHS(0),	$
			PLUS_REQUIRED=PLUS_REQUIRED)
		FOR I = 1,N_ELEMENTS(PATHS)-1 DO DIRECTORIES =	$
			[DIRECTORIES, FIND_ALL_DIR(PATHS(I),	$
				PLUS_REQUIRED=PLUS_REQUIRED)]
		GOTO, TEST_FORMAT
	ENDIF
;
;  Test to see if the first character is a plus sign.  If it is, then remove
;  it.  If it isn't, and PLUS_REQUIRED is set, then remove any trailing '/'
;  character and skip to the end.
;
	DIR = PATH
	IF STRMID(DIR,0,1) EQ '+' THEN BEGIN
		DIR = STRMID(DIR,1,STRLEN(DIR)-1)
	END ELSE IF KEYWORD_SET(PLUS_REQUIRED) THEN BEGIN
		DIRECTORIES = PATH
		IF STRMID(PATH,STRLEN(PATH)-1,1) EQ '/' THEN	$
			DIRECTORIES = STRMID(PATH,0,STRLEN(PATH)-1)
		GOTO, TEST_FORMAT
	ENDIF
;
;  On VMS machines, spawn a command file to find the directories.  Make sure
;  that any logical names are completely translated first.  A leading $ may be
;  part of the name, or it may be a signal that what follows is a logical name.
;
	IF !VERSION.OS EQ 'vms' THEN BEGIN
		REPEAT BEGIN
			IF STRMID(DIR,STRLEN(DIR)-1,1) EQ ':' THEN	$
				DIR = STRMID(DIR,0,STRLEN(DIR)-1)
			TEST = TRNLOG(DIR,VALUE) MOD 2
			IF (NOT TEST) AND (STRMID(DIR,0,1) EQ '$') THEN BEGIN
				TEMP = STRMID(DIR,1,STRLEN(DIR)-1)
				TEST = TRNLOG(TEMP, VALUE) MOD 2
			ENDIF
			IF TEST THEN DIR = VALUE
		ENDREP UNTIL NOT TEST
		COMMAND_FILE = FIND_WITH_DEF('FIND_ALL_DIR.COM',!PATH,'.COM')
		SPAWN,'@' + COMMAND_FILE + ' ' + COMMAND_FILE + ' ' + DIR, $
			DIRECTORIES
;
;  On Unix machines spawn the Bourne shell command 'find'.  Remove any trailing
;  slash character from the first directory.
;
	END ELSE BEGIN
		IF STRMID(DIR,STRLEN(DIR)-1,1) NE '/' THEN DIR = DIR + '/'
		SPAWN,'find ' + DIR + ' -type d -print | sort -',	$
			DIRECTORIES, /SH
		TEMP = DIRECTORIES(0)
		IF STRMID(TEMP,STRLEN(TEMP)-1,1) EQ '/' THEN	$
			DIRECTORIES(0) = STRMID(TEMP,0,STRLEN(TEMP)-1)
	ENDELSE
;
;  If the PATH_FORMAT keyword was set, then reformat the string array into a
;  single string, with the correct separator.
;
TEST_FORMAT:
	IF KEYWORD_SET(PATH_FORMAT) THEN BEGIN
		DIR = DIRECTORIES(0)
		IF !VERSION.OS EQ 'vms' THEN SEP = ',' ELSE SEP = ':'
		FOR I = 1,N_ELEMENTS(DIRECTORIES)-1 DO	$
			DIR = DIR + SEP + DIRECTORIES(I)
		RETURN, DIR
	END ELSE RETURN, DIRECTORIES
;
	END
