	PRO PCLPLOT,FILENAME,DELETE=DELETE,QUEUE=QUEUE,COMMAND=COMMAND
;+
; Project     : SOHO - CDS
;
; Name        : 
;	PCLPLOT
; Purpose     : 
;	Print an HP LaserJet PCL plot file, reset graphics device.
; Explanation : 
;	Sends a PCL plot file generated by IDL to the HP LaserJet PCL laser
;	printer.  The default queue is defined by the logical name/environment
;	variable HPLASER.  The graphics device is reset to what was used
;	previously.
; Use         : 
;	PCLPLOT  [, FILE ]  [, /DELETE ]
;
;	PCL				;Open PCL plot file
;	   ... plotting commands ...	;Create plot
;	PCLPLOT				;Close & plot file, reset to prev. dev.
;	   or
;	PCLCLOSE			;Close w/o printing,  "    "   "    "
;
; Inputs      : 
;	None required.
; Opt. Inputs : 
;	The default filename is either taken from the last call to the PCL
;	routine, or is "idl.pcl".
;
;	A filename other than the default can be passed in one of three ways:
;
;		Explicitly:		e.g. PCLPLOT,'graph.pcl'
;		By number (VMS)		e.g. PCLPLOT,3   for "idl.pcl;3"
;		All versions (VMS)	e.g. PCLPLOT,'*' for "idl.pcl;*"
;		All ".pcl" files (UNIX)	e.g. PCLPLOT,'*' for "*.pcl"
; Outputs     : 
;	A message is printed to the screen.
; Opt. Outputs: 
;	None.
; Keywords    : 
;	DELETE	= If set, then file is deleted after printing.
;	QUEUE	= Name of printer queue to be used in printing the file.
;	COMMAND	= (Unix only.)  Command to be used to send the plot file to the
;		  printer.  If not passed, then the environment variable
;		  PRINTCOM is checked.  If neither of these is set, then the
;		  standard command "lpr" is used.
; Calls       : 
;	SETPLOT
; Common      : 
;	PCL_FILE which contains PCL_FILENAME, the name of the plotting file,
;	and LAST_DEVICE, which is the name of the previous graphics device.
; Restrictions: 
;	The requested plot file must exist.
;
;	In general, the SERTS graphics devices routines use the special system
;	variables !BCOLOR and !ASPECT.  These system variables are defined in
;	the procedure DEVICELIB.  It is suggested that the command DEVICELIB be
;	placed in the user's IDL_STARTUP file.
;
; Side effects: 
;	The plot file is queued on the printer HPLASER.  Also, any files
;	"idl.pcl" that may be open will be closed.  The previous plotting
;	device is reset.
; Category    : 
;	Utilities, Devices.
; Prev. Hist. : 
;	None.
; Written     : 
;	William Thompson, GSFC, 15 June 1993.
; Modified    : 
;	Version 1, William Thompson, GSFC, 15 June 1993.
;		Based on QMPLOT.PRO.
;	Version 2, William Thompson, GSFC, 8 June 1994
;		Added keyword COMMAND
; Version     : 
;	Version 2, 8 June 1994.
;-
;
	COMMON PCL_FILE, PCL_FILENAME, LAST_DEVICE
	SWITCH = ""
	STRING_TYPE = 7
;
;  No parameters passed:  assume either the name from the last time PCL was
;  called, or "idl.pcl".
;
	IF N_PARAMS(0) EQ 0 THEN BEGIN
		IF N_ELEMENTS(PCL_FILENAME) EQ 1 THEN BEGIN
			IF PCL_FILENAME EQ "" THEN FILENAME = "idl.pcl"	$
				ELSE FILENAME = PCL_FILENAME
		END ELSE FILENAME = "idl.pcl"
	ENDIF
;
;  Test and interpret FILENAME.
;
	S = SIZE(FILENAME)
	IF S(0) NE 0 THEN BEGIN
		PRINT,'*** Variable must not be an array, ' +	$
			'name= FILE, routine PCLPLOT.'
		RETURN
;
;  If of type string, then must be the filename.  If the string "*", then the
;  meaning depends on which operating system is being used.
;
	END ELSE IF S(1) EQ STRING_TYPE THEN BEGIN
		IF FILENAME EQ "*" THEN BEGIN
			IF !VERSION.OS EQ "vms" THEN BEGIN
				FILENAME = "idl.pcl;*"
			END ELSE BEGIN
				FILENAME = "*.pcl"
			ENDELSE
		ENDIF
;
;  If numerical, then either it is the file version number (VMS) or it is
;  simply incorrect (UNIX).
;
	END ELSE IF !VERSION.OS EQ "vms" THEN BEGIN
		FILENAME = "idl.pcl;" + TRIM(FILENAME)
	END ELSE BEGIN
		PRINT,'*** Variable must be of type string, ' +		$
			'name= FILENAME, routine PCLPLOT.'
		RETURN
	ENDELSE
;
;  If passed, then check the value of QUEUE.
;
	IF N_ELEMENTS(QUEUE) NE 0 THEN BEGIN
		SQ = SIZE(QUEUE)
		IF S(0) NE 0 THEN BEGIN
			PRINT,'*** Variable must not be an array, ' +	$
				'name= QUEUE, routine PCLPLOT.'
			RETURN
		END ELSE IF S(1) NE STRING_TYPE THEN BEGIN
			PRINT,'*** Variable must be of type string, ' +	$
				'name= QUEUE, routine PCLPLOT.'
			RETURN
		ENDIF
;
;  Otherwise check the logical name/environment variable HPLASER to get the
;  name of the queue.  A queue name is required.
;
	END ELSE BEGIN
		HPLASER = GETENV("HPLASER")
		IF HPLASER NE "" THEN BEGIN
			QUEUE = HPLASER
		END ELSE BEGIN
			PRINT,'*** Logical name HPLASER or keyword QUEUE ' + $
				'must be defined, routine PCLPLOT.'
			RETURN
		ENDELSE
	ENDELSE
;
;  Close any PCL files.
;
	DEVICE = !D.NAME
	IF N_ELEMENTS(LAST_DEVICE) EQ 0 THEN LAST_DEVICE = !D.NAME
	IF !D.NAME NE 'PCL' THEN SETPLOT,'PCL'
	DEVICE,/CLOSE_FILE
	PCL_FILENAME = ""
;
;  Form the print command according to the operating system.
;
	IF !VERSION.OS EQ "vms" THEN BEGIN
		HPLASER = TRIM(STRUPCASE(QUEUE))
		COM_LINE = "PRINT /NOTIFY /QUEUE=" + HPLASER + " "
	END ELSE BEGIN
;
;  In UNIX, use the switch "-P" to control which queue is used.  Otherwise, the
;  default print queue is used.  The printing command can be set by the keyword
;  COMMAND, by the environment variable PRINTCOM, or the standard command "lpr"
;  can be used.
;
		IF N_ELEMENTS(COMMAND) EQ 1 THEN PRINTCOM = COMMAND ELSE $
			PRINTCOM = GETENV("PRINTCOM")
		IF PRINTCOM EQ "" THEN PRINTCOM = "lpr"
		COM_LINE = PRINTCOM + " "
		IF N_ELEMENTS(QUEUE) NE 0 THEN	$
			COM_LINE = COM_LINE + "-P" + QUEUE + " "
	ENDELSE
;
;  Test to see if the DELETE keyword was set.
;
	IF KEYWORD_SET(DELETE) THEN IF !VERSION.OS EQ "vms" THEN	$
		COM_LINE = COM_LINE + "/DELETE "	    ELSE	$
		COM_LINE = COM_LINE + "-r "
	COM_LINE = COM_LINE + FILENAME
	PRINT,"$ " + COM_LINE
	SPAWN,COM_LINE
;
;  Reset the plotting device.
;
	IF DEVICE NE 'PCL' THEN SETPLOT,DEVICE ELSE SETPLOT,LAST_DEVICE
	PRINT,'The plotting device is now set to '+TRIM(LAST_DEVICE)+'.'
;
	RETURN
	END
