;+
; Project     : SOHO - CDS
;
; Name        : SEND_PRINTER
;
; Purpose     : to print a file or array in a device independent way
;
; Category    : Device
;
; Explanation : checks operating system and spawns the appropriate
;               command.
;
; Syntax      : IDL> send_printer,file,que=que,qual=qual,array=array
;
; Example:    : send_printer,'test.doc',que='soho-laser1',qual='h'
;
; Inputs      : FILE = filename to print
;
; Opt. Inputs : None
;
; Outputs     : None
;
; Opt. Outputs: None
;
; Keywords    : QUE  - print que name [def = '']
;               QUAL - qualifier [def = ''] 
;               DEL - set to delete file when done
;               ERR - error message
;               ARRAY - alternative string array to print
;
; Common      : None
;
; Restrictions: Print queues must exist
;
; Side effects: File deleted when /DEL set
;
; History     : Version 1,  4-Sep-1995,  D.M. Zarro.  Written
;
; Contact     : DZARRO@SOLAR.STANFORD.EDU
;-

pro send_printer,file,que=que,qual=qual,delete=delete,err=err,array=array

on_error,1
err='' 

;-- check inputs

if (datatype(array) eq 'STR') and (datatype(file) eq 'STR') then begin
 err='Cannot simultaneously use file or string array as inputs'
 message,err,/cont
 return
endif

if (datatype(array) ne 'STR') and (datatype(file) ne 'STR') then begin
 err='Enter file or string array to print'
 message,err,/cont
 return
endif

if datatype(file) eq 'STR' then begin
 chk=loc_file(file,count=count)
 if count eq 0 then begin
  err='Cannot locate '+file
  message,err,/cont
  return
 endif
endif

;-- print array or file

if datatype(array) eq 'STR' then begin
 del=1
 str2file,array,temp
endif else begin
 temp=file
 del=keyword_set(delete)
endelse

;-- check for print command as environment variable

vms=os_family() eq 'vms'
printcom=chklog('PRINTCOM')
if printcom eq '' then begin
 if vms then printcom='print' else printcom='lpr'
endif

;-- take care of qualifiers and queues
;   usually qualifiers are preceded by '-' in Unix and '/' in VMS
;   usually print queues are specified by '-P' in Unix and '/queue=' in VMS

if not keyword_set(qual) then qual=''
if not keyword_set(que) then que=''

if vms then begin
 if qual ne '' then begin
  chk=strpos(qual,'/')
  if chk eq -1 then aqual='/'+qual else aqual=qual
 endif else aqual=' '
 if que ne '' then aque='/que='+que else aque=' '
endif else begin
 if qual ne '' then begin
  chk=strpos(qual,'-')
  if chk eq -1 then aqual='-'+qual else aqual=qual
 endif else aqual=' '
 if que ne '' then aque='-P '+que else aque=' '
endelse

com=printcom+' '+aqual+' '+aque+' '+temp
dprint,com
espawn,com,out
if out(0) ne '' then begin
 err=out(0)
 message,err,/cont
endif

;-- delete file if requested

if del then begin
 openr,lun,temp,/get_lun,/del
 close,lun & free_lun,lun
endif

return & end


