;+
; Project     : SOHO - CDS     
;                   
; Name        : ANYTIM2UTC()
;               
; Purpose     : Converts (almost) any time format to CDS UTC format.
;               
; Explanation : Tests the type of input and tries to use the appropriate
;               conversion routine to create the date/time in CDS UTC
;               format (either internal (default), external or string)
;               CDS time format.
;               
; Use         : IDL>  utc = anytim2utc(any_format)
;    
; Inputs      : any_format - date/time in any of the acceptable CDS 
;                            time formats -- for acceptable formats see file 
;                            aaareadme.txt.
;               
; Opt. Inputs : None
;               
; Outputs     : Function returns CDS UTC time structure.
;               
; Opt. Outputs: None
;               
; Keywords    : EXTERNAL  = Create output in external format
;               CCSDS     = Create string output in CCSDS format
;               ECS       = Create string output in ECS format
;		VMS	  = Create string output in VMS format
;		STIME	  = Create string output in STIME format
;
;		Only one of the above keywords can be set.  If none of them are
;		set, then the output is in internal format.  The following
;		keywords are only valid if a string format was selected.
;
;		TRUNCATE  = If set, then the time will be truncated to 1 second
;			    accuracy.  Note that this is not the same thing as
;			    rounding off to the nearest second, but is a
;			    rounding down.
;
;		DATE_ONLY = If set, then only the date part of the string is
;			    returned.
;
;		TIME_ONLY = If set, then only the time part of the string is
;			    returned.
;
;		UPPERCASE = If set, then the month field in either the VMS or
;			    STIME format is returned as uppercase.
;
;		The following keyword is always valid.
;
;               ERRMSG    = If defined and passed, then any error messages 
;                           will be returned to the user in this parameter 
;                           rather than being printed to the screen.  If no
;                           errors are encountered, then a null string is
;                           returned.  In order to use this feature, the 
;                           string ERRMSG must be defined first, e.g.,
;
;                                ERRMSG = ''
;                                UTC = ANYTIM2UTC ( DT, ERRMSG=ERRMSG, ...)
;                                IF ERRMSG NE '' THEN ...
;
; Calls       : DATATYPE, INT2UTC, STR2UTC
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, time
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 16-May-94
;               
; Modified    :	Version 1, C D Pike, RAL, 16-May-94
;		Version 2, William Thompson, GSFC, 14 November 1994
;			Changed .DAY to .MJD
;		Version 3, Donald G. Luttermoser, GSFC/ARC, 20 December 1994
;			Added the keyword ERRMSG.  Included ON_ERROR flag.
;		Version 4, Donald G. Luttermoser, GSFC/ARC, 30 January 1995
;			Added ERRMSG keyword to internally called procedures.
;			Made error handling routine more robust.
;		Version 5, Donald G. Luttermoser, GSFC/ARC, 8 February 1995
;			Allowed for input to be either scalar or vector.
;		Version 6, William Thompson, GSFC, 14 March 1995
;			Added keywords VDS, STIME, TRUNCATE, DATE_ONLY,
;			TIME_ONLY, UPPERCASE
;		Version 7, William Thompson, GSFC, 5 May 1995
;			Fixed bug with use of ERRMSG keyword.
;			Made so that TAI times are supported.
;		Version 8, William Thompson, GSFC, 8 May 1995
;			Fixed bug introduced in version 7
;               Version 9 C D Pike, RAL, 17-May-95
;                       Handle time only (no date) string input.
;		Version 10, William Thompson, GSFC, 20 December 1995
;			Fixed bug with use of ERRMSG keyword when string
;			contained no "-" characters.
;
; Version     : Version 10, 20 December 1995
;-            

function anytim2utc, dt, external=external, ccsds=ccsds, ecs=ecs, VMS=VMS, $
	STIME=STIME, TRUNCATE=TRUNCATE, DATE_ONLY=DATE_ONLY,	$
	TIME_ONLY=TIME_ONLY, UPPERCASE=UPPERCASE, errmsg=errmsg
 
;
;  set default return value
;
utc = {cds_int_time, mjd: 0L, time: 0L}

on_error, 2   ;  Return to the caller of this procedure if error occurs.
message=''    ;  Error message returned via ERRMSG if error is encountered.
;
;  see if any parameters were passed
;
if n_params() eq 0 then begin
	message = 'Syntax:  UTC = ANYTIM2UTC(DATE-TIME)'
	goto, handle_error
endif

;
;  determine type of input 
;
type = datatype(dt,1)

;
; see if the input is an array
;
np = n_elements(dt)
if np gt 1 then utc = replicate(utc, np)

;
; act accordingly
;
case type of
      'String':  begin
		    for i=0,np-1 do begin
                    	ndash = n_elements(str_sep(dt(i),'-'))
                    	nslash = n_elements(str_sep(dt(i),'/'))
 
                    	if ndash gt 1 then year = (str_sep(dt(i),'-'))(0) 
                    	if nslash gt 1 then year = (str_sep(dt(i),'/'))(0)
       
                        if ndash eq 1 and nslash eq 1 then begin
                           get_utc,utc
                           date = anytim2cal(utc,form=9,/date)
                           year = strmid(date,7,4)
                           test = str2utc(date+' '+dt(i),/dmy,errmsg=errmsg)
		    	   if n_elements(errmsg) ne 0 then message=errmsg
			   if message eq '' then utc(i) = test
                        endif else begin
                    	   if year lt 31 then dmy_opt = 1 else dmy_opt = 0

                    	   test = str2utc(dt(i),dmy=dmy_opt,errmsg=errmsg)
		    	   if n_elements(errmsg) ne 0 then message=errmsg
			   if message eq '' then utc(i) = test
                        endelse
		    endfor
                 end

   'Structure':  begin
                    case n_tags(dt) of
                       2: utc = dt
                       7: utc = utc2int(dt)
                      else: message='ANYTIM2UTC:  Unknown date/time structure.'
                    endcase
                 end
      'Double':  utc = tai2utc(dt)
          else:  message='ANYTIM2UTC:  Unrecognized input format.'
endcase

if message ne '' then goto, handle_error

if n_elements(errmsg) ne 0 then errmsg = ''
 
if keyword_set(external) or keyword_set(ccsds) or keyword_set(ecs) or $
		keyword_set(vms) or keyword_set(stime) then begin
    return, int2utc(utc,ccsds=ccsds,ecs=ecs,vms=vms,stime=stime,	$
	truncate=truncate,date_only=date_only,time_only=time_only, $
	uppercase=uppercase,errmsg=errmsg)
end else return, utc

;
; Error handling point.
;
handle_error:
if n_elements(errmsg) eq 0 then message, message
errmsg = message
return, utc
;
end
