;+
; Project     : SOHO - CDS     
;                   
; Name        : REM_DUP()
;               
; Purpose     : Function to remove duplicate values from a vector. 
;               
; Explanation : None
;               
; Use         : result = rem_dup( vector, [ flag ] )  
;
;               Eg. Remove duplicate values in vector a.
;                          a = a( rem_dup(a) )
;
;                   Remove duplicates in vector WAVE.  When duplicate values
;                   are found, select the one with the largest intensity, INTE.
;
;                          sub  = rem_dup( wave, inte)
;                          wave = wave( sub )
;                          inte = inte( sub )
;
;    
; Inputs      : vector - vector of values from which duplicates are to be found
;               
; Opt. Inputs : flag   - if supplied then when duplicates occur,
;                        the one with the largest value of flag is selected.
;                        If not supplied the the first occurence of the value
;                        in 'vector' is selected.  Should be a vector with 
;                        the same number of elements as in 'vector'.
;               
; Outputs     : A vector of subscripts in 'vector' is returned.  Each subscript
;               points to a selected value such that vec(rem_dup(vec,flag))
;               has no duplicates.
;               
; Opt. Outputs: None
;               
; Keywords    : None
;
; Calls       : None
;               
; Restrictions: None
;               
; Side effects: The returned subscripts will sort the values in 'vector' in 
;               ascending order with duplicates removed.
;               
; Category    : Util, misc
;               
; Prev. Hist. : D. Lindler  Mar. 87
;
; Written     : CDS version by C D Pike, RAL, 22-Oct-93
;               
; Modified    : Use BSORT instead of SORT to maintain order.  CDP 7-Nov-94
;
; Version     : Version 2, 7-Nov-94
;-            

function rem_dup, a, flag

; 
;  on error return to caller
;
On_error,2

;
; number of input parameters supplied
;
npar = N_params()              

;
;  if no parameters given then prompt with help
;
if npar EQ 0 then begin
   print,'Syntax -  b = rem_dup( a, [ flag ] )
   return, -1
endif

;
;  number of elements in input
;
n = N_elements(a)  

;
;  if only one value then not much to chose from
;
if n lt 2 then return, lonarr(1)    

;
;  if flag parameter not specified , use defaults
;
if npar lt 2 then flag = intarr(n) 

;
;  sort in ascending order
; 
sub = bsort(a)   

;
;  sort input vector
;
aa = a(sub)   
 
;
;  sort the flags
;
ff = flag(sub)    

;
;  values to keep
;
good = lonarr(n)    

;
;  number kept
; 
ngood = 0    

;
; set first value and its flag
; 
val = aa(0)                   
f = ff(0)     

;
;  loop over aa
;
for i = 1,n-1 do begin
   if aa(i) ne val then begin
      val = aa(i)
      f = ff(i)
      ngood = ngood+1
      good(ngood) = i
   endif else begin
      if ff(i) gt f then begin
         f = ff(i)
         good(ngood) = i
      endif
   endelse
endfor

;
;  trim the storage
; 
good = good(0:ngood)

;
;  return subscripts in original a
; 
return, sub(good)              
 
end
