;+
; Project     : SOHO - CDS     
;                   
; Name        : STR_PICK()
;               
; Purpose     : Extract a substring bounded by given characters.
;               
; Explanation : Given an input string and two bounding characters (or
;               substrings), the function returns the extracted substring.  
;               Default delimiters are first and second spaces.  
;               If only one is given then substring returned
;               is from there to the end of the string. If both characters
;               are specified and one of them does not exist then a null
;               string is returned.
;               
; Use         : IDL> text = str_pick(in_text, char1, char2)
;
;               eg print, str_pick('this is <crazy>.', '<','>')
;
;                        --->  crazy
;    
; Inputs      : in_text  -  string from which to extract
;               
;               
; Opt. Inputs : char1 - left boundary character (or substring)
;               char2 - right boundary character (or substring)
;               
; Outputs     : Function returns extracted substring
;               
; Opt. Outputs: None
;               
; Keywords    : None
;
; Calls       : Standard str_ routines
;
; Common      : None
;               
; Restrictions: None
;               
; Side effects: None
;               
; Category    : Util, string
;               
; Prev. Hist. : None
;
; Written     : C D Pike, RAL, 27-Mar-95
;               
; Modified    : Allow boundaries to be substrings.  CDP, 10-Apr-95
;
; Version     : Version 2, 10-Apr-95
;-            

function str_pick, text, char1, char2

;
;  parameter checks
;
if n_params() eq 0 then begin
   print,'Use:  text = str_pick(string, char_left, char_right)
   return,''
endif

if n_params() eq 1 then begin
   char1 = ' ' & char2 = ' '
endif

;
;  easy if only left hand character is given
;
if n_params() eq 2 then begin
   p1 = strpos(text,char1)
   if p1 ge 0 then return,strmid(text,p1+strlen(char1),strlen(text)) else return,''
endif

;
;  if both search characters are the same then pick the substring between
;  the first and second occurrence
;
if char1 eq char2 then begin
   p1 = strpos(text,char1)
   if p1 eq -1 then return,''
   p2 = strpos(text,char2,p1+1)
   if p2 eq -1 then p2 = strlen(text)
   return, strmid(text,p1+strlen(char1),(p2-p1-strlen(char1)))
endif

;
;  two different characters specified so search for them
;
p1 = strpos(text,char1)
if p1 eq -1 then return,''
p2 = strpos(text,char2)
if p2 eq -1 then return,''

return,strmid(text,p1+strlen(char1),p2-p1-strlen(char1))

end
