;+
; Project     :	SOHO - CDS
;
; Name        :	XMATRIX
;
; Purpose     : Widget display of fields within an arbitrary structure
;
; Use         : xmatrix,struct,wbase
;
; Inputs      :
;               STRUCT = input structure
;               WBASE = parent widget base into which place matrix
;
; Opt. Inputs : None.
;
; Outputs     : None.
;
; Opt. Outputs: None.
;
; Keywords    :
;               NX = # of columns by which to arrange widgets (def=2)
;               WTAGS = text widget id's for each tag
;               TITLE = optional widget title
;               EDITABLE = make fields editable
;               ALL      = generate events without return key
;               XSIZE = optional width for text widgets
;               LFONT = font for tag labels
;               TFONT = font for tag fields
; Procedure   :
;               Arranges structures in a matrix format with the tag
;               name in label widget and the tag value in a text widget.
;               If wbase and wtags exist, then widget is just updated
;               with input field values. This procedure is called by XSTRUCT
;
; Calls       : None.
;
; Common      : None.
;
; Restrictions:
;              Input must be a structure.
;              WBASE must be a valid parent base
;
; Side effects: None.
;
; Category    : Widgets
;
; Prev. Hist. : None.
;
; Written     :	Zarro (ARC/GSFC) 20 August 1994
;
; Modified    :
;       Version 2, Liyun Wang, GSFC/ARC, October 12, 1994
;          Made the WIDGET_LABEL right justified by using the 'fixed' font
;       Version 3, Zarro, (GSFC/ARC) 8 March, 1994
;          Added nested structure display
;
; VERSION:
;       Version 3
;-

   pro xmatrix,struct,wbase,nx=nx,wtags=wtags,title=title,editable=editable,$
               xsize=xsize,all=all,lfont=lfont,tfont=tfont

   blank = '                                                              '
   on_error,1

   if (n_elements(struct) ne 1) or (datatype(struct) ne 'STC') then $
      message,'input must be a 1-d structure'
   if n_elements(xsize) eq 0 then xsize=22
   tags=tag_names(struct)
   ntags=n_elements(tags)
   
   if min(xalive(wbase)) eq 0 then message,'define a top level parent base first'

   if n_elements(wtags) eq 0 then wtags=0l
   wtags=long(wtags)
   update=min(xalive(wtags)) ne 0

   if not exist(lfont) then lfont=(get_dfont('7x14bold'))(0)
   if not exist(tfont) then tfont=(get_dfont('7x14bold'))(0)
   if n_elements(nx) eq 0 then nx=2

   nx = nx > 1

;-- get tag definitions

   stc_name=tag_names(struct,/structure_name)
   if stc_name eq '' then stc_name='ANONYMOUS'
   if n_elements(title) eq 0 then title=stc_name
   temp = ntags/nx
   if (nx*fix(temp) eq ntags) then ny = temp else ny=1+temp
   if ntags eq 1 then nx = 1

;-- make label and text widgets

   if (not update) then begin
      wtags=lonarr(ntags)
      wtitle=widget_label(wbase,value=title)
      row=widget_base(wbase,/row)
      i = -1 
      offset = 0
      for k=0,nx-1 do begin
         col=widget_base(row,/column,/frame)
         if (offset+ny) lt ntags then begin
            tail = (offset+ny) 
            real_ny = ny
         endif else begin
            tail = ntags
            real_ny = ntags-offset
         endelse
         c_tags = tags(offset:tail-1)
         tag_len = strlen(c_tags)
         max_len = max(tag_len)
         for j=0, real_ny-1 do begin
            i=i+1
            if i ge ntags then goto,made
            temp=widget_base(col,/row)
            c_tags(j) = c_tags(j)+':'+strmid(blank,0,max_len-tag_len(j))
            label=widget_label(temp,value=c_tags(j),font=lfont)
            wtags(i)=widget_text(temp,value=' ',editable=editable,$
                                 all=all,xsize=xsize,font=tfont)
         endfor
         offset = offset+real_ny
      endfor
   endif

;-- populate text widgets

made:
   if n_elements(wtags) ne ntags then begin
      message,'incompatibility between input structure tags and '+$
         'wbase widget base',/continue
      return
   endif

   i=-1
   for k=0,nx-1 do begin
      for j=0,ny-1 do begin
         i=i+1
         if i ge ntags then goto,done
         field=struct.(i)
         case 1 of
            datatype(field) eq 'STC': begin
               widget_control,wtags(i),set_value='see over',sensitive=0
               xstruct,field,nx=nx,/just_reg,editable=editable,/recur
            end
            (size(field))(0) gt 1 : begin
               widget_control,wtags(i),set_value='   >1d array  ',sensitive=0
            end
            else: begin
               if datatype(field) eq 'BYT' then field=fix(field)
               widget_control,wtags(i),set_value=string(field)
            end
         endcase
      endfor
   endfor
done:   wtitle=widget_info(wbase,/child)

wtype=widget_info(wtitle,/type)
if wtype eq 5 then widget_control,wtitle,set_value=title

return & end
