;---------------------------------------------------------------------------
; Document name: xset_value.pro
; Created by:    Liyun Wang, GSFC/ARC, August 18, 1994
;
; Last Modified: Thu Nov  9 11:47:18 1995 (lwang@achilles.nascom.nasa.gov)
;---------------------------------------------------------------------------
;
;+
; PROJECT:
;       SOHO - CDS
;
; NAME:
;       XSET_VALUE
;
; PURPOSE:
;       Set the value of a variable interactively with a slider.
;
; EXPLANATION:
;
; CALLING SEQUENCE:
;       xset_value, value [, min = min] [, max = max]
;
; INPUTS:
;       VALUE - Current default value; may be changed upon exit
;
; OPTIONAL INPUTS:
;       None.
;
; OUTPUTS:
;       VALUE - Updated value
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORD PARAMETERS:
;       MIN   -- Minimum value of the range (Default: 0)
;       MAX   -- Maximum value of the range (Default: 100)
;       TITLE -- Title of the widget; default: "Number Picker"
;       INSTRUCT -- A brief instruction to the user for running the program
;       GROUP -- Group ID of an upper widget on which this one depends
;
; CALLS:
;       BELL, DATATYPE
;
; COMMON BLOCKS:
;       None.
;
; RESTRICTIONS:
;       None.
;
; SIDE EFFECTS:
;       None.
;
; CATEGORY:
;       Utilities, Widget
;
; PREVIOUS HISTORY:
;       Written August 18, 1994, Liyun Wang, GSFC/ARC
;
; MODIFICATION HISTORY:
;       Version 1, created, Liyun Wang, GSFC/ARC, August 18, 1994
;       Version 2, Liyun Wang, GSFC/ARC, May 2, 1995
;          Gewt rid of common block
;       Version 3, November 9, 1995, Liyun Wang, GSFC/ARC
;          Added check to guarentee the input value is a numerical one
;             and within the allowed range
;          Positioned the widget in the center of the screen
;          Added INSTRUCT keyword
;
; VERSION:
;       Version 3, November 9, 1995
;-

;---------------------------------------------------------------------------
;  Event handler
;---------------------------------------------------------------------------
PRO XSET_VALUE_EVENT, event

   ON_ERROR, 2

   WIDGET_CONTROL, event.top, get_uvalue = unseen

   WIDGET_CONTROL, unseen, get_uvalue = info, /no_copy

   WIDGET_CONTROL, event.id, get_uvalue = uvalue, get_value = cur_value
   CASE (uvalue) OF
      'ACCEPT': BEGIN
         WIDGET_CONTROL, info.sld_id, get_value = value
         IF (value NE info.value_sv) THEN BEGIN
            info.value_sv = value
            WIDGET_CONTROL, unseen, set_uvalue=info, /no_copy
            xkill, event.top
            RETURN
         ENDIF
      END
      'RESET': BEGIN
         value = info.value_sv
         WIDGET_CONTROL, info.sld_id, set_value = value
         WIDGET_CONTROL, info.info_id, set_value = value
      END
      'CANCEL': BEGIN
         WIDGET_CONTROL, unseen, set_uvalue = info, /no_copy
         xkill, event.top
      END
      'SLIDER': BEGIN
         WIDGET_CONTROL, info.sld_id, get_value = value
         WIDGET_CONTROL, info.info_id, set_value = value
      END
      'KEYBOARD': BEGIN
         value = cur_value(0)
         IF valid_num(value) EQ 1 THEN BEGIN
            IF value LT info.min OR value GT info.max THEN BEGIN
               WIDGET_CONTROL, info.text, $
                  set_value='Input value is out of allowed range!'
               WAIT, 2
               WIDGET_CONTROL, info.text, set_value=info.instruct
            ENDIF ELSE BEGIN
               WIDGET_CONTROL, info.info_id, set_value=value
               WIDGET_CONTROL, info.sld_id, set_value=value
            ENDELSE
         ENDIF ELSE BEGIN
            WIDGET_CONTROL, info.text, $
               set_value=STRTRIM(value, 2)+' is not a valid number!'
            WAIT, 2
            WIDGET_CONTROL, info.text, set_value=info.instruct
         ENDELSE
      END
      ELSE: RETURN
   ENDCASE
   IF WIDGET_INFO(event.top, /valid) THEN BEGIN
      WIDGET_CONTROL, unseen, set_uvalue = info, /no_copy
      WIDGET_CONTROL, event.top, set_uvalue = unseen
   ENDIF
END

;---------------------------------------------------------------------------
;  Main program
;---------------------------------------------------------------------------
PRO XSET_VALUE, value, min = min, max = max, title=title, group=group, $
                instruct=instruct

   ON_ERROR, 2

   dtype = datatype(value,2)
   IF ((dtype GE 6) OR (dtype LT 1)) THEN BEGIN
;---------------------------------------------------------------------------
;     data not integer or (double precision) floating number
;---------------------------------------------------------------------------
      MESSAGE,'Wrong data type.',/cont
      RETURN
   ENDIF

   value_sv = value

   IF N_ELEMENTS(title) EQ 0 THEN title = 'Number Picker'
   IF N_ELEMENTS(instruct) EQ 0 THEN instruct = ''

   base = WIDGET_BASE (title = title, /column, space=20)

   row1   = WIDGET_BASE(base, /row, xpad = 30, space = 20, /frame)
   done   = WIDGET_BUTTON(row1, value = 'Accept', uvalue = 'ACCEPT')
   reset  = WIDGET_BUTTON(row1, value = 'Reset', uvalue = 'RESET')
   cancel = WIDGET_BUTTON(row1, value = 'Cancel', uvalue = 'CANCEL')

   base1 = WIDGET_BASE(base, /column, /frame)

   text = WIDGET_TEXT(base1, value = instruct)

   row2 = WIDGET_BASE(base1, /column, /frame, xpad = 30)
   info_id = cw_field(row2, title = 'Current Value', value = value, $
                      xsize = 10,/RETURN, uvalue = 'KEYBOARD')

   slider_y = 30
   IF (dtype LE 2) THEN BEGIN
;---------------------------------------------------------------------------
;     value is a short integer
;---------------------------------------------------------------------------
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100
      sld_id =  WIDGET_SLIDER(base1, minimum = min, maximum = max, $
                              value = value, /frame, uvalue = 'SLIDER', $
                              ysize = slider_y, /drag, /suppress)
   ENDIF ELSE IF (dtype EQ 3) THEN BEGIN
;---------------------------------------------------------------------------
;     value is a long integer
;---------------------------------------------------------------------------
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0l
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100l
      sld_id =  WIDGET_SLIDER(base1, minimum = min, maximum = max, $
                              value = value, /frame, uvalue = 'SLIDER',$
                              ysize = slider_y, /drag, /suppress)
   ENDIF ELSE BEGIN
;---------------------------------------------------------------------------
;     value is a floating point number (either single or double precsion)
;---------------------------------------------------------------------------
      IF (N_ELEMENTS(min) EQ 0) THEN min = 0.0
      IF (N_ELEMENTS(max) EQ 0) THEN max = 100.0
      cur_value = float(value)
      sld_id = CW_FSLIDER(base1, minimum = min, maximum = max, $
                          value = value, uvalue = 'SLIDER',$
                          ysize = slider_y, /frame, /drag, /suppress)
   ENDELSE

   offsets = get_cent_off(base, valid = valid)

   IF valid THEN $
      WIDGET_CONTROL, base, /realize, /map, tlb_set_xoff=offsets(0),$
      tlb_set_yoff=offsets(1), /show $
   ELSE $
      WIDGET_CONTROL, base, /realize, /map, /show

   info = {sld_id:sld_id, info_id:info_id, value_sv:value_sv, text:text,$
           min:min, max:MAX, instruct:instruct}

   unseen = WIDGET_BASE()
   WIDGET_CONTROL, unseen, set_uvalue = info, /no_copy
   WIDGET_CONTROL, base, set_uvalue = unseen
   XMANAGER, 'xset_value', base, group_leader = group, $
      modal = KEYWORD_SET(group)
   WIDGET_CONTROL, unseen, get_uvalue = info, /destroy
   value = info.value_sv
END

;---------------------------------------------------------------------------
; End of 'xset_value.pro'.
;---------------------------------------------------------------------------
