;---------------------------------------------------------------------------
; Document name: xtext.pro
; Created by:    D. M. Zarro, GSFC/ARC, August 20, 1994
;
; Last Modified: Mon Nov 20 10:32:28 1995 (lwang@achilles.nascom.nasa.gov)
;---------------------------------------------------------------------------
;
;+
; PROJECT:
;       SOHO - CDS/SUMER
;
; NAME:
;       XTEXT
;
; PURPOSE:
;       Text display widget with searching capability
;
; CATEGORY:
;       Utility, Widgets
;
; EXPLANATION:
;
; SYNTAX:
;       xtext, array
;
; EXAMPLES:
;       Useful as a pop up text widget.
;
;           xtext,'some text',/just_reg, wbase=wbase, group=event.top
;             ...some processing..
;           xkill, wbase
;
;       This example will pop a text window that will exist during
;       processing, after which it is destroyed by kill
;
; INPUTS:
;       ARRAY - string array to display
;
; OPTIONAL INPUTS:
;       None.
;
; OUTPUTS:
;       None.
;
; OPTIONAL OUTPUTS:
;       None.
;
; KEYWORDS:
;       GROUP       - group leader of text widget parent
;       FONT        - font for text widget
;       TITLE       - title for text widget parent
;       SPACE       - number of lines to space text
;       JUST_REG    - just_reg
;       WBASE       - base widget id
;       XOFF,YOFF   - pixel offset relative to caller
;       WAIT        - secs to wait before killing widget
;       INSTRUCT    - instruction label for exit button [def = Dismiss]
;       XSIZE,YSIZE - X-Y sizes for text widget
;       APPEND      - append to existing text
;
; COMMON:
;       None.
;
; RESTRICTIONS:
;       None.
;
; SIDE EFFECTS:
;       None.
;
; HISTORY:
;       Version 1, August 20, 1994, D. M. Zarro,  GSFC/ARC. Written
;       Version 2, August 25, 1995, Liyun Wang, GSFC/ARC
;          Added the capability of search string
;       Version 3, September 1, 1995, Liyun Wang, GSFC/ARC
;          Added buttons to go top, bottom, or a specific line
;          Fixed the bug that caused X windows protocol error for bad font
;
; CONTACT:
;       Liyun Wang, GSFC/ARC (Liyun.Wang.1@gsfc.nasa.gov)
;-

PRO xtext_event, event
;---------------------------------------------------------------------------
;  event driver routine
;---------------------------------------------------------------------------

   ON_ERROR, 1

   WIDGET_CONTROL, event.top, get_uvalue=unseen
   info=get_pointer(unseen,/no_copy)
   WIDGET_CONTROL, event.id, get_uvalue=uvalue
   
   CASE uvalue OF

      'print': begin                             ;-- print text
        xprint,array=info.text,group=event.top
       end

      'close': BEGIN
         IF xalive(info.group_id) THEN BEGIN
            WIDGET_CONTROL, info.group_id, /sensitive
            xshow, info.group_id
         ENDIF
;---------------------------------------------------------------------------
;        It's important to turn off any selection in the text widget,
;        otherwise you may get "X windows protocol error" for BadFont
;---------------------------------------------------------------------------
         WIDGET_CONTROL, info.wtext, set_text_select=0
         xkill, event.top
      END

      'top': BEGIN
         info.line = 0
         info.pos = 0
         WIDGET_CONTROL, info.wtext, set_text_select=0
         WIDGET_CONTROL, info.wtext, set_text_top_line=0
         WIDGET_CONTROL, info.flnum, set_value='1'
         WIDGET_CONTROL, info.fcnum, set_value='1'
      END

      'bottom': BEGIN
         info.line = info.tline
         np = info.line
         ln = LONG(STRLEN(info.text(info.tline-1)))+1L
         off_set = LONG(TOTAL([STRLEN(info.text), np]))
         WIDGET_CONTROL, info.wtext, set_text_select=off_set
         WIDGET_CONTROL, info.flnum, set_value=STRTRIM(info.tline,2)
         WIDGET_CONTROL, info.fcnum, set_value='1'
      END

      'text': BEGIN
         off_set = event.offset
         os = 0L
         i = 0
         ok = 1
         WHILE (ok AND i LE info.tline-1) DO BEGIN
            os = os+info.line_char(i)+1L
            IF os GT off_set THEN BEGIN
               info.line = i
               info.pos = info.line_char(info.line)-(os-off_set)+1
               WIDGET_CONTROL, info.flnum, set_value=STRTRIM(info.line+1,2)
               WIDGET_CONTROL, info.fcnum, set_value=STRTRIM(info.pos+1,2)
               ok = 0
            ENDIF
            i =i+1
         ENDWHILE
      END

      'gline': BEGIN
         WIDGET_CONTROL, info.lnum, get_value=str
         IF NOT num_chk(str(0),/integer) THEN BEGIN
            line = FIX(STRTRIM(str(0),2))-1 > 0
            IF line LT info.tline THEN info.line = line ELSE $
               info.line = info.tline-1
            IF info.line LE 0 THEN prev_char = 0 ELSE $
               prev_char = info.line_char(0:info.line-1)
            np = info.line
            off_set = LONG(TOTAL([prev_char, np])) < info.tchar_num
            WIDGET_CONTROL, info.wtext, set_text_select=off_set
            WIDGET_CONTROL, info.flnum, set_value=STRTRIM(info.line+1, 2)
            WIDGET_CONTROL, info.fcnum, set_value='1'
         ENDIF 
      END

      'find': BEGIN
         WIDGET_CONTROL, info.search_lb, get_value=tt
         tt = STRTRIM(tt(0),2)
         IF tt NE '' THEN BEGIN
            WIDGET_CONTROL, /hour
            IF NOT info.case_sense THEN tt = STRUPCASE(tt)
            IF info.line GE info.tline THEN BEGIN
               info.line = 0
               info.pos = 0
            ENDIF
            go_on = 1
            WHILE (go_on) DO BEGIN
               IF NOT info.case_sense THEN $
                  text = STRUPCASE(info.text(info.line)) $
               ELSE $
                  text = info.text(info.line)
               IF info.line GE 1 THEN BEGIN
                  prev_char = info.line_char(0:info.line-1)
                  np = info.line
                  off_set = LONG(TOTAL([prev_char, np]))
               ENDIF ELSE off_set = 0L
               idx = STRPOS(text, tt, info.pos)
               IF idx GE 0 THEN BEGIN
                  length = STRLEN(tt)
                  WIDGET_CONTROL, info.wtext, $
                     set_text_select=[off_set+idx, length]
                  WIDGET_CONTROL, info.flnum, $
                     set_value=STRTRIM(info.line+1,2)
                  WIDGET_CONTROL, info.fcnum, set_value=STRTRIM(idx+1,2)
                  info.pos = idx+length
                  go_on = 0
               ENDIF ELSE BEGIN
                  info.pos = 0
                  info.line = info.line+1
                  IF info.line GE info.tline THEN BEGIN
                     info.line = 0
                     WIDGET_CONTROL, info.wtext, set_text_select=0
                     WIDGET_CONTROL, info.wtext, set_text_top_line=0
                     WIDGET_CONTROL, info.flnum, set_value='1'
                     WIDGET_CONTROL, info.fcnum, set_value='1'
                     go_on = 0
                  ENDIF
               ENDELSE
            ENDWHILE
         ENDIF
      END

      'chg_case': BEGIN
         info.case_sense = event.select
      END

      ELSE:
   ENDCASE

   set_pointer,unseen,info,/no_copy

   RETURN
END

;---------------------------------------------------------------------------
;  Main routine
;---------------------------------------------------------------------------
PRO xtext, array, font=font, title=title, group=group, modal=modal, $
           space=space, just_reg=just_reg, scroll=scroll, xoff=xoff, $
           yoff=yoff, append=append, wbase=wbase, WAIT=WAIT, $
           instruct=instruct, xsize=xsize, ysize=ysize

   ON_ERROR, 1

   IF (datatype(array) NE 'STR') THEN MESSAGE, 'input must be a string'
   IF NOT HAVE_WIDGETS() THEN MESSAGE, 'widgets unavailable'

   IF xalive(wbase) THEN update = 1 ELSE update = 0

   IF KEYWORD_SET(just_reg) THEN just_reg = 1 ELSE just_reg = 0
   just_reg_sav = just_reg

   IF update THEN just_reg = 1

   IF (NOT KEYWORD_SET(group)) AND (NOT just_reg) THEN xkill, /all

   append = KEYWORD_SET(append)
   
   IF NOT just_reg THEN space = 0 ELSE BEGIN
      IF N_ELEMENTS(space) EQ 0 THEN space = 3
   ENDELSE

   IF N_ELEMENTS(title) NE 0 THEN wtitle = title ELSE wtitle = 'XTEXT'
   IF (NOT append) AND (space GT 0) THEN BEGIN
      buff = REPLICATE(' ', space) & text=[buff, detabify(array), buff]
   ENDIF ELSE text = detabify(array)
   tline = N_ELEMENTS(text)

   bfont = '-adobe-courier-bold-r-normal--20-140-100-100-m-110-iso8859-1'
   bfont = (get_dfont(bfont))(0)
   
   lfont = '-misc-fixed-bold-r-normal--13-100-100-100-c-70-iso8859-1'
   lfont = (get_dfont(lfont))(0)

   IF datatype(font) EQ 'STR' THEN tfont = (get_dfont(font))(0) ELSE BEGIN
      tfont = '8x13bold'
      tfont = (get_dfont(tfont))(0)
   ENDELSE
   
   get_screen, fspace, fxpad, fypad

   IF NOT update THEN BEGIN
      scroll = N_ELEMENTS(text) GT 20
      IF N_ELEMENTS(ysize) EQ 0 THEN ysize = N_ELEMENTS(text) < 20
      IF N_ELEMENTS(xsize) EQ 0 THEN xsize = MAX(STRLEN(text)) < 80
      wbase = WIDGET_BASE(title=wtitle, /column)
      wtext = WIDGET_TEXT(wbase, /frame, value=text, uvalue='text',$
                          font=tfont, scroll=scroll, /all_event, $
                          ysize=ysize, xsize=xsize)

;---------------------------------------------------------------------------
;     If not just registering then add search and close buttons and
;     call XMANAGER
;---------------------------------------------------------------------------
      IF NOT (just_reg) THEN BEGIN
         case_sense = 0
         line_char = STRLEN(text)
         tchar_num = TOTAL([LONG(line_char), tline])
         temp = WIDGET_BASE(wbase, /row, /frame)
         tmp = WIDGET_LABEL(temp, value=' Go To:', font=bfont)

         tmp = WIDGET_BUTTON(temp, value='Line', uvalue='gline', $
                             font=bfont)
         lnum = WIDGET_TEXT(temp, value='', xsize=5, /edit, uvalue='gline')

         tmp = WIDGET_LABEL(temp, value=' ', font=bfont)
         top = WIDGET_BUTTON(temp, value='Top', uvalue='top',font=bfont)
         tmp = WIDGET_LABEL(temp, value=' ', font=bfont)
         bottom = WIDGET_BUTTON(temp, value='Bottom', uvalue='bottom',$
                                font=bfont)

         temp = WIDGET_BASE(wbase, /column, /frame)
         junk = WIDGET_BASE(temp, /row)
         tmp = WIDGET_BUTTON(junk, value='Find', uvalue='find', font=bfont)
         search_lb = WIDGET_TEXT(junk, value='', xsize=20, /edit, $
                                 uvalue='find', font=lfont)
         
         tmp = WIDGET_LABEL(junk, value=' Case', font=bfont)
         xmenu, 'sensitive', junk, /nonexcl, uvalue='chg_case', $
            buttons=tmp
         WIDGET_CONTROL, tmp(0), set_button=0

         junk = WIDGET_BASE(wbase, /row, /frame)
         tmp = WIDGET_LABEL(junk, value=' Cursor Position:  line', font=lfont)
         flnum = WIDGET_TEXT(junk, value='', xsize=5, font=lfont)
         tmp = WIDGET_LABEL(junk, value='  column', font=lfont)
         fcnum = WIDGET_TEXT(junk, value='', xsize=3, font=lfont)

         junk = WIDGET_BASE(wbase, /row, /frame,xpad=20, space=20)

         IF datatype(instruct) EQ 'STR' THEN bname = instruct ELSE $
            bname = 'Dismiss'
         

         tmp = WIDGET_BUTTON(junk, value=bname, uvalue='close', $
                             font=bfont)

         tmp = WIDGET_BUTTON(junk, value='Print', uvalue='print', $
                             font=bfont)

;---------------------------------------------------------------------------
;        Compile all necessary variables into an info structure for event
;        handler to use
;---------------------------------------------------------------------------
         IF xalive(group) THEN group_id = group ELSE group_id = 0L
      ENDIF

;---------------------------------------------------------------------------
;     determine placement
;---------------------------------------------------------------------------
      IF (N_ELEMENTS(xoff) EQ 0) AND (N_ELEMENTS(yoff) EQ 0) THEN BEGIN
         offsets = get_cent_off(wbase, group, valid=valid)
         IF valid THEN BEGIN
            xoff = offsets(0) & yoff=offsets(1)
         ENDIF
      ENDIF

      realized = WIDGET_INFO(wbase, /realized)
      IF (N_ELEMENTS(xoff) EQ 0) AND (N_ELEMENTS(yoff) EQ 0) THEN $
         WIDGET_CONTROL, wbase,/realize ELSE $
         WIDGET_CONTROL, wbase, /realize, $
         tlb_set_xoff=xoff, tlb_set_yoff=yoff, /map,/show
      
      IF NOT just_reg THEN BEGIN
         make_pointer, unseen
         info = {case_sense:case_sense, search_lb:search_lb, line:0, $
                 text:text, group_id:group_id, tchar_num:tchar_num, $
                 wtext:wtext, pos:0, tline:tline, flnum:flnum, $
                 fcnum:fcnum, lnum:lnum, line_char:line_char}
         set_pointer,unseen,info,/no_copy
         WIDGET_CONTROL, wbase, set_uvalue=unseen
      ENDIF
   ENDIF ELSE BEGIN
      wtext = WIDGET_INFO(wbase, /child)
      temp = WIDGET_INFO(wbase, /sibling)
      WIDGET_CONTROL, wtext, set_value=text, append=append
      WIDGET_CONTROL, wbase, tlb_set_title=wtitle
      xshow, wbase
   ENDELSE

   xmanager, 'xtext', wbase, group=group,just_reg=just_reg,$
    modal=(keyword_set(modal) or xalive(group)) and (not just_reg)
   IF (NOT xalive(group)) AND (NOT just_reg) THEN xmanager

   just_reg = just_reg_sav
   if (not just_reg) then free_pointer,unseen

   IF (N_ELEMENTS(WAIT) GT 0) AND (just_reg) THEN BEGIN
      WAIT, WAIT & xkill, wbase
   ENDIF

   RETURN
END

;---------------------------------------------------------------------------
; End of 'xtext.pro'.
;---------------------------------------------------------------------------
