;******************************************************************************
function papco_gcd1, a_in,b_in, errtot=errtot

   if n_elements( errtot ) eq 0 then errtot=0
   
   a= a_in > b_in
   d= a_in < b_in
   
   r= a mod d

   r_imp= ( r < (d-r) )

   while r gt errtot do begin
       d= r
       r= a mod d
   endwhile
   
   return, d
end


;******************************************************************************
function papco_gcd, A, errtot=errtot, guess=guess 
; gcd: return the greatest common divisor of a set of numbers A.
; I/O: A, array of int, long or float.
;      errtot=errtot  same as errpct, but err is in same units as a,
;                     and is independent of d, specifically:
;                   r < errtot
;        
;      guess=guess  a number that is likely to be the gcd, improve calculation.
;

  if n_elements( guess ) eq 0 then guess= A(0) 
  if n_elements( errtot ) eq 0 then errtot= 0

  gcd= guess
  
  for i=0l,n_elements(A)-1 do gcd= papco_gcd1( gcd, A(i), errtot=errtot )
  
  return, gcd
end

;******************************************************************************
function papco2_hpc_av, X, navg, nodata=nodata, naverage, noavg=noavg, $
                        allow_smooth=allow_smooth, verbose = verbose

; averages array over first dimension (having n elements) down to
; navg elements.

;    x1 x2 x3 x4 x5 x6 x7 x8        <-- original 1D (or 2D array), nX elements
;    \      / \   / \      /        <-- (nX/navg) or (nX/navg+1)
;     \    /   \ /   \    /              averaged together
;       a1     a2      a3           <-- averaged array, navg elements
;
; INPUT:  X      is the array to average
;         navg   is the number of average bins
;         nodata specifies nodata value
;         noavg  don't divide by number of points, just sum
;
;         allow_smooth  (nX/navg+1) values go into each average, so
;                       some values in source array to go into
;                       two averages.  This is somewhat faster.
;
; OUTPUT: naverage   passes out the number of values in average (optional)
;         (returns)  the average array, or if /noavg, the sums, sum(Xi),
;                      to be divided by naverage
;
; ASSUMPTIONS and NOTES:
;  For 2d arrays, ANY value equal to 1e-20 is also treated as nodata.
;  It is also assumed that 1e-20 << X(i)*nsum .
;  Occasionally this routine triggers floating underflow.
;
; HISTORY:
;  Spring 1997   Written, Jeremy Faden, jbf@hydra.physics.uiowa.edu
;  February 2003 aded verbose keyword, R.Friedel

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0

   if n_elements(nodata) eq 0 then nodata= 1e-20
   allow_smooth= keyword_set(allow_smooth)

   sizeX= size(X)
   n= sizeX( 1 )                ; average over the first dimension
   nsum= float(n)/navg
   
   if (nsum eq 1.0) then begin  ; no averaging
       naverage= make_array( size=size(X), /int, value=1 )
       rnv= where( X eq nodata )
       if rnv(0) ne -1 then naverage(rnv)=0
       return, X
   endif 

   if (nsum lt 1.0 ) then begin ; use congrid to expand       
       print, '% papco_hpc_av: I don''t expand'
       return, X
   endif
   
   if ( sizeX(0) lt 1 or sizeX(0) gt 2) then begin
       print, '% papco_hpc_av: only 1- or 2-D arrays please...'
       return, -1
   endif else if sizeX(0) eq 1 then begin ; 1D average
       typeX= sizeX(2)
       saverage= make_array(navg,type=typeX,value=0)
       naverage= make_array(navg,/int)
   endif else begin             ; 2D average
       typeX= sizeX(3)
       ncol= sizeX(2)
       saverage= make_array(navg,ncol,type=typeX)
       naverage= make_array(navg,ncol,/int)
   endelse     
   
   if (allow_smooth) then begin
       IF verbose THEN message, 'smoothing allowed.', /cont
       nsum1=ceil(nsum) 
   endif else nsum1=fix(nsum)-1

   ind= long(indgen(navg)*nsum)
   ind0= [ind,n]                ; initial indeces

   if sizeX(0) eq 1 then begin
       for i=0l, nsum1 do begin
           rv= where(X(ind) ne nodata)
           if (rv(0) ne -1) then begin
               saverage(rv)=saverage(rv)+X(ind(rv))
               naverage(rv)=naverage(rv)+1
           endif
           ind=ind+1
       endfor

       if (not allow_smooth) then begin
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind) ne nodata)
               if (rv(0) ne -1) then begin
                   saverage(r(rv))=saverage(r(rv))+X(ind(rv))
                   naverage(r(rv))=naverage(r(rv))+1
               endif
           endif
       endif
   endif else begin
       if nodata ne 1e-20 then begin
           rnodata=where(X eq nodata) 
           if rnodata(0) ne -1 then X(rnodata)=1e-20
       endif
           
       for i=0l, nsum1-1 do begin
           rv= where(X(ind,*) ne 1e-20)
           if (rv(0) ne -1) then begin
               saverage= saverage+X(ind,*)
               naverage(rv)=naverage(rv)+1
               rz=where(saverage eq 1e-20)
               if rz(0) ne -1 then saverage(rz)=0.0
           endif
           ind=ind+1    
       endfor

       if (not allow_smooth) then begin ; average in the remaining guys
           r= where( ind0(1:*)-ind(0:*) eq 1 ) 
           if r(0) ne -1 then begin
               ind= ind(r)
               rv= where(X(ind,*) ne 1e-20)
               if (rv(0) ne -1) then begin
                   saverage(r,*)= saverage(r,*)+X(ind,*)
                   naverage(r(rv),*)=naverage(r(rv),*)+1
                   rz=where(saverage eq 1e-20) 
                   if rz(0) ne -1 then saverage(rz)=0.0
               endif
           endif
       endif

       if nodata ne 1e-20 then if rnodata(0) ne -1 then X(rnodata)=nodata

   endelse

   r0= where(naverage eq 0)
   if (r0(0) ne -1) then saverage(r0)= nodata
   
   if keyword_set( noavg ) then return, saverage
   
   rv= where(naverage gt 0)
   if (rv(0) ne -1) then saverage(rv)=saverage(rv)/naverage(rv)
   
   return, saverage
end

;******************************************************************************
pro papco_plot_colorspec, zmat_in, xarr_in, yarr_in, ylog=ylog, $
                          dx=dx, dy=dy, $
                          resx=resx, resy=resy, $
                          zlog=zlog, zrange=zrange, $
                          _extra=e, special_colors=special, $
                          nodata=nodata, sample=sample, $
   ;                      colortbl=colortbl, $
                          xfilled=xfilled, $ ; assume data in X is filled.
                          verbose=verbose  ; rf added
  
; Plots a colorplot of data.  Z data must be stored in a 2D array,
;    with columns correponding to intervals in x and rows corresponding
;    to intervals in y.
;
; INPUTS:
;
; zmat is the data, dblarr( n, m )
;
; xarr are the x values for each z column.
;    if xarr=fltarr(n,2) then pairs are start and end of column.
;    if xarr=fltarr(n)   then each is start of column with no gaps,
;                            and the last column is not shown.
; yarr are the y values for each z row
;    same format as xarr. fltarr(m,2) or fltarr(m)
;
; /ylog is handled, but /xlog will give incorrect results.
;
; dx, dy  explicitly specify the bin size for x and y, otherwise
;    the binsize is the greatest common divisor, within pixel
;    precision of the x and y values.
;
; resx, resy  explicitly specify the required accuracy of the x and y
;    bins, otherwise the device pixel resolution sets these.
;
; nodata specifies that a point should be plotted in grey and omitted
;    from averaging.
;
; /sample  do not average zmat if device resolution is not sufficient,
;    instead display samples from zmat.
;
; /verbose give some diagnostic output.
;
; special_colors = fltarr(2,n) can be used to insert special colors,
;                              e.g. saturation.  It is an array of pairs,
;                              each pair is (value,color).  Where
;                              value is found in zmat, use display color.
;
; colortbl=[a,b]  specifies the range of colors to use for the colorbar. 
;    if not specified, then the following algorythm is used:
;        if n_elements(papco_color) gt 0 then use PAPCO color scheme.
;        otherwise, use [0,!d.n_colors]
;
; Notes:
;    No averaging is done on columns (y).
;    Averaging will clobber special color values.
;
; HISTORY:
;    Written, Jeremy Faden, November 1997.  jbf@space-theory.physics.uiowa.edu
  
COMMON papco_color, papco_color, r_papco, g_papco, b_papco 
COMMON papco_color_names, black, red, green, yellow, blue, magenta, cyan, $
                          white, burgundy, olive, dark_green, teal, $
                          royal_blue, violet, dark_grey, grey, $
                          foreground, background

IF keyword_set(VERBOSE) THEN verbose = 1 ELSE verbose = 0

;check on plot structure e - if xstyle, ystyle not set, add them
names=tag_names(e)  
pos=strpos(names,'XSTYLE')  &  index=where(pos ne -1,c) 
if c eq 0 then e=create_struct({XSTYLE:1}, e)
pos=strpos(names,'YSTYLE')  &  index=where(pos ne -1,c) 
if c eq 0 then e=create_struct({YSTYLE:1}, e)

if n_elements(xarr_in) eq 0 then begin ; x and y not specified.
    m= n_elements(zmat_in(*,0))
    xarr_in= findgen(m)
    xarr_in= reform( [ xarr_in, xarr_in+1 ], m, 2)
endif

if n_elements(yarr_in) eq 0 then begin ; x and y not specified.
    n= n_elements(zmat_in(0,*))
    yarr_in= findgen(n)
    yarr_in= reform( [ yarr_in,yarr_in+1 ], n, 2)
endif

; copy inputs to local variables
zmat= zmat_in
xarr= xarr_in
yarr= yarr_in

if not keyword_set(nodata) then nodata=5.43e21

if not keyword_set(zrange) then begin
   zmin=min(zmat,max=zmax)
   zrange=[zmin,zmax]
endif else begin
   zmin= double(zrange(0))
   zmax= double(zrange(1))
endelse

xfilled= keyword_set( xfilled )
if xfilled then begin
    dx= xarr(1)-xarr(0)
endif

zlog= keyword_set(zlog)

sample= keyword_set(sample)     ; sample data, don't average

; find colorbar range
if n_elements(colortbl) eq 0 then begin
    if n_elements( papco_color ) gt 0 then begin
        cindex= where( papco_color(*,2) eq 1 ) ; search the active vector
        if (cindex(0) eq -1) then begin
            print, '% papco_plot_colorspec: Color system failure, '
            print, '  consult jbf@space-theory.physics.uiowa.edu'
            print, '  Using color set 0'
            cindex=0
        endif
        color_range= PAPCO_get_Colorindices( cindex )
    endif else begin
        color_range= [0,!d.n_colors-1]
        grey=0
    endelse
endif else begin
    color_range= colortbl
endelse
colorbase= color_range(0)
ncolors= color_range(1)-color_range(0)
colortop= color_range(1)
    
ylog= keyword_set( ylog )

; reformat xarr and yarr to start, end time format. (if necessary.)
s= size(xarr)
if s(0) eq 1 then begin
    xarr_new= make_array( s(1)-1, 2 )
    xarr_new(*,0)= xarr(0:(s(1)-2))
    xarr_new(*,1)= xarr(1:*)
    xarr=xarr_new
endif

s= size(yarr)
if s(0) eq 1 then begin
    yarr_new= make_array( s(1)-1, 2 )
    yarr_new(*,0)= yarr(0:(s(1)-2))
    yarr_new(*,1)= yarr(1:*)
    yarr=yarr_new
endif

; handle /ylog keyword.
if keyword_set( ylog ) then begin
    yarr=alog10(yarr)
endif

q=n_elements(xarr_in)-1 & r=n_elements(yarr_in)-1
plot, xarr_in([0,q]), yarr_in([0,r]), /nodata, _extra=e, ylog=ylog
xcrange= !x.crange
ycrange= !y.crange

; calculate pixel resolution 
if ylog then ycrange10=10^ycrange else ycrange10=ycrange
lldev= convert_coord( xcrange(0), ycrange10(0), /data, /to_device )
urdev= convert_coord( xcrange(1), ycrange10(1), /data, /to_device )
if not keyword_set( resx ) then $
  resx= ( xcrange(1) - xcrange(0) ) / ( urdev(0)-lldev(0) )
if not keyword_set( resy ) then $
  resy= ( ycrange(1) - ycrange(0) ) / ( urdev(1)-lldev(1) )

; calculate bin size dx and dy
if not keyword_set(dx) then begin
    dxs= xarr(*,1)-xarr(0)
    dx= papco_gcd( dxs, errtot=resx )
endif

if not keyword_set(dy) then begin
    dys= yarr(*,1)-yarr(0)
    dy= papco_gcd( dys, errtot=resy )
endif

if keyword_set(VERBOSE) then begin
  message,'Set dx, dy and resx,resy:', /cont
  print,'  ',dx,dy,resx,resy
endif

; find which xbins appear on the plot
rplotx= where( xarr(*,0) ge xcrange(0) and xarr(*,1) le xcrange(1), nplotx )
if nplotx lt n_elements( xarr(*,0) ) then begin
    if nplotx gt 0 then begin
        xarr=xarr(rplotx,*)
        zmat=zmat(rplotx,*)
    endif else begin
        xarr=xarr(0,*)          ; we'll see that nothing is printed later
        zmat=zmat(0,*)
    endelse
endif

rploty= where( yarr(*,0) ge ycrange(0) and yarr(*,1) le ycrange(1), nploty )
if nploty lt n_elements( yarr(*,0) ) then begin
    if nploty gt 0 then begin
        yarr=yarr(rploty,*)
        zmat=zmat(*,rploty)
    endif else begin
        yarr=yarr(0,*)          ; we'll see that nothing is printed later
        zmat=zmat(*,0)
    endelse
endif

; calculate the number of x and y bins for image and form zimage -> zmat map
if (xfilled) then begin
    nx= n_elements( xarr(*,0) )
    wherex1= lindgen( nx )
    wherex2= lindgen( nx )
endif else begin
    nx= long( ( xarr(n_elements(xarr)-1) - xarr(0) ) / dx + 0.5 ) 
    wherex1= long((xarr(*,0)-xarr(0))/dx+0.5)
    wherex2= (long((xarr(*,1)-xarr(0))/dx+0.5)-1)>wherex1
endelse

ny= long( ( yarr(n_elements(yarr)-1) - yarr(0) ) / dy + 0.5 )
wherey1 = long((yarr(*,0)-yarr(0))/dy+0.5)
wherey2 = (long((yarr(*,1)-yarr(0))/dy+0.5)-1)>wherey1

wherex= lonarr(nx)-1 & wherey= lonarr(ny)-1

for i=0l,n_elements(wherex1)-1 do $
  wherex((wherex1(i) < (nx-1)):(wherex2(i) < (nx-1)))= i
for i=0l,n_elements(wherey1)-1 do wherey(wherey1(i):wherey2(i))= i

zimage= make_array( nx, ny, /float, value=nodata )
rx= where( wherex ge 0 )
ry= where( wherey ge 0 )
for j=0l,n_elements(ry)-1 do begin
    zimage(rx,ry(j))= zmat(wherex(rx),wherey(ry(j)))
endfor

; find the location of tv image of zimage on data coords.
x1= xarr(0) & x2= xarr( n_elements(xarr)-1 )

if ylog then begin
    y1=10^yarr(0) & y2=10^yarr( n_elements(yarr)-1 )
endif else begin
    y1=yarr(0) & y2=yarr( n_elements(yarr)-1 )
endelse

lld= convert_coord( x1, y1, /data, /to_device )
urd= convert_coord( x2, y2, /data, /to_device )

; find device size
xsized= long(urd(0)-lld(0)) & ysized= long(urd(1)-lld(1))

if !d.name eq 'PS' then begin
    if !d.x_px_cm eq 1000 then begin
        IF verbose THEN message, 'reducing PS resolution to 300 dpi', /cont
        xsized = xsized * 300 / 2540 & ysized = ysized * 300 / 2540 
    endif
endif

if keyword_set(VERBOSE) then begin
  message,'Set xsized, nx, ysized:', /cont
  print,'  ',xsized, nx,ysized
endif

; average down the array if there aren't enough pixels to show each 
; measurement.
; note: need to display at least one pixel! RF Nov 2001.
xsized = xsized > 1
if (xsized lt nx) then begin
    if (sample) then begin
        IF verbose THEN message, 'sampling data', /cont
        rrr= long( findgen( xsized ) / xsized * nx )
        zimage= zimage( rrr, * )
    endif else begin
        IF verbose THEN message, 'time-averaging '+strtrim(nx,2)+$
          ' time intervals down to '+strtrim(xsized,2)+'.', /cont 
        zimage= papco2_hpc_av( zimage, xsized, /allow, $
                              nodata=nodata, verbose = verbose)
    endelse
    nx= xsized
endif

; transform data to color
if (zlog) then begin
    zsize= alog10( zmax/zmin )
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rok= where( zimage ge zmin )
    if rok(0) ne -1 then zcolor(rok)= alog10( zimage(rok)/zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
endif else begin
    zsize= zmax-zmin
    zcolor= make_array( nx, ny, /byte, value=colorbase )
    rcol= where( zimage ge zmin )
    zcolor(rcol)= ( zimage(rcol) - zmin ) / $
      ( zsize / ncolors ) + colorbase < colortop
    r= check_math()
    if r ne 0 then message, 'check_math()='+strtrim(r,2), /cont
endelse

; fill color nodata points grey
rnd=where( zimage eq nodata )
if rnd(0) ne -1 then zcolor( rnd ) = grey

; fill special color points with special colors
if n_elements(special) gt 0 then begin
    for i=0l,n_elements(special(0,*))-1 do begin
        value= special(0,i)
        color= byte(special(1,i))
        rval= where( zimage eq value )
        if rval(0) ne -1 then zcolor(rval)=color
    endfor      
endif

if (nplotx gt 0) then begin
    if !d.name eq 'PS' then begin
        lln= convert_coord( lld, /device, /to_normal )
        urn= convert_coord( urd, /device, /to_normal )
        xsizen= urn(0)-lln(0)
        ysizen= urn(1)-lln(1)
        tv, zcolor, lln(0), lln(1), xsize=xsizen, ysize=ysizen, /normal
    endif else BEGIN
        tv, congrid(zcolor, xsized, ysized), lld(0), lld(1)+1 ; +1 is a kludge
    endelse
endif

e.XTICKFORMAT='noticks'

if e.xstyle ne 5 then axis, XAXIS=0, _extra=e 
if e.ystyle ne 5 then axis, YAXIS=0, _extra=e 

end
