;******************************************************************************
;* CALL (papco_color_bar):
;*      papco_color_bar, XPOS=XPOS, WIDTH=WIDTH, TITLEPOS=TITLEPOS, $
;*                       _EXTRA=extra_par
;*
;* DESCRIPTION:
;*      Plots a papco_color_bar on the right side of the screen
;*
;* INPUTS:     
;*      none.    Expects keywords only. 
;*
;* CALLING SEQUENCE:
;*      papco_color_bar, _extra=extra_plotPar
;*
;* KEYWORDS:
;*      uses _extra structure and expects at least the following keywords:
;*                  extra_Par={zrange:[zmin,zmax], $
;*                             zlog:zscl(panel(0),3), $
;*                             ztitle:uztit, $
;*                             ztickformat:'papco_color_bar_log_ticks' }
;*      XPOS   normal coords position seperation from plot
;*      WIDTH  normal coords width of color bar
;*      INFO   print out header of routine
;*	
;* PROCEDURE:  The color sequence is (and must be) linear, even for
;*	       logarithmic z-scaling.
;*             Axes are drawn using the AXIS- and LOGAXIS-procedures.
;*             Viewport and data-screen relation are restored at the end.
;*
;* MODIFICATION HISTORY:
;*       written June 1999, Reiner Friedel
;;*****************************************************************************
PRO papco_color_bar, XPOS=XPOS, WIDTH=WIDTH, POSTITLE=POSTITLE, $
                     SPACETITLE=SPACETITLE, INFO=INFO, _EXTRA=extra_par
  
COMMON PLOT_COMPOSER, widgetData

; see if a manual label was set for this panel, or inhibit was set
currPlotInfo=widgetData.plotInfos(widgetData.current_plotNr)

IF currPlotInfo.descr_extend EQ -1 THEN currPlotInfo.descr = 2

CASE currPlotInfo.descr OF
    0: BEGIN                    ;use automatic setting

    END
    1: BEGIN                    ;use user defined label

    END
    2: BEGIN                    ;inhibit color bar, just do nothing
        return
    END
ENDCASE

; extend label over multiple panels if needed. Simply "panelset" a new
; height.
panel_orig =  widgetData.panelset & panel = widgetData.panelset
panel(2) = abs(currPlotInfo.descr_extend) > panel_orig(2)
panelset, panel	       ;re-sets the panel position viewport

; use standard PAPCO INFO keyword to report usage of routine  
if keyword_set(INFO) then begin
    papco_routine_info,'papco_color_bar','papco_color_bar', $
      ROOT_DIR=getenv('PAPCO_LIB')
    print,''  & return
endif      
  
; handle keywords  
tags=tag_names(extra_par)
  
  index=where(tags eq 'ZRANGE',c)
  if c ne 0 then begin
    zmin=extra_par.zrange(0)  &  zmax=extra_par.zrange(1)
  endif else begin
    zmin=1  &  zmax=10
  endelse  
  
  index=where(tags eq 'ZTITLE',c)
  if c ne 0 then ztitle=extra_par.ZTITLE else ztitle=''
  
  index=where(tags eq 'ZLOG',c)
  if c ne 0 then log=extra_par.ZLOG else log=0
  
  index=where(tags eq 'ZTICKFORMAT',c)
  if c ne 0 then ztickformat=extra_par.ZTICKFORMAT else $
    ztickformat='papco_color_bar_log_ticks'
    
  if keyword_set(XPOS) then xpos=XPOS else xpos=0.065
  if keyword_set(WIDTH) then width=WIDTH else width=1
  if keyword_set(POSTITLE) then postitle=POSTITLE else postitle=0
  if keyword_set(SPACETITLE) then spacetitle=SPACETITLE else spacetitle=0
    
; get starting offset and no of colors for color table. 
  c_index=papco_GetColor_table()   ;get index for papco color table   
  indicies=papco_get_Colorindices(c_index)
  nc=indicies(1)-indicies(0)+1
  c_offset=indicies(0)

; save current viewport and data-screen-relation:
  pa=!p & xa=!x & ya=!y
  !x.title=''  &  !p.TITLE=''  &  !y.title=''
  !p.ticklen=-.2
  !p.position(0)=!p.position(2)+xpos
  !p.position(2)=!p.position(0)+0.025*width
  !x.range=[0.,100.]
  !y.range=[ZMIN,ZMAX]
  !x.style=4+1
  !y.ticks=0
  !y.style=1+4
  zfac= DOUBLE(zmax-zmin)/nc
  PLOT,[0,100],[zmin,zmin],color=!p.background,/noerase

;*******************create color bar (linear color sequence)
  FOR I=c_offset(0),c_offset(0)+nc-1 DO BEGIN
    ZINT1=(I-2-c_offset(0)+2)*zfac+zmin
    ZINT2=(I-1-c_offset(0)+2)*zfac+zmin
    color=FIX(I)
    width=100*width
    POLYFILL,[0.,0.,100.,100.],[ZINT1,ZINT2,ZINT2,ZINT1],col=color
  ENDFOR
 
;*******************axes and annotation (only here do we have to distinguish
;                   linear and logarithmic scaling)
  if log then begin
    axis,yaxis=1,charsize=!P.CHARSIZE, ticklen=-0.6, ylog=1, color=1, $
      yr=[ZMIN,ZMAX], ytickformat=ztickformat, ystyle=1, yticks=0, yminor=0, $
      ytickname=''
  endif else begin
    !x.style=4
    !p.color=1
    axis,yaxis=1,ystyle=1,charsize=1.0
  endelse  
  
  case postitle of
    0:begin                     ;normal title positioning for PAPCO
      ; set position for zaxis label
      rel_label_y=(!p.position(1)+!p.position(3))/2
      rel_label_x=!p.position(2)+0.065 + spacetitle
      ; set size of label to fit into space of panel. Reduce size if required.
      ; get character size in normal coordinates  
      chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
      
      ; get maximum length of a string, ignoring !X formatting characters
      ppp=str_sep(ZTITLE,'!')
      n_char=strlen(ZTITLE)-n_elements(ppp)*2.5
  
      ;work out label height here - scaled by !P.CHARSIZE
      rel_label_height=chrs(0) * n_char; * !P.CHARSIZE
      rel_avail_height=(!p.position(3)-!p.position(1))*0.9
      fit_ratio=(rel_label_height)/rel_avail_height
      ; if label is too long to fit -> Reduce
      if fit_ratio gt 1.0 then char_factor=1/fit_ratio else char_factor=1
      orient=-90  & alig=0.5
    end
    1: begin ;place title on top
      ; set position for zaxis label
      rel_label_y=!p.position(3)+0.02
      rel_label_x=!p.position(0)
      char_factor=1  &  orient=0
      alig=postitle-fix(postitle)
    end
  endcase
  
  XYOUTS, rel_label_x, rel_label_y, ZTITLE, alig=alig, $
    chars=char_factor, orient=orient, color=1, /normal
  
  ;restore original settings
  !x=xa & !y=ya & !p=pa

  ;restore original panel settings
  panel = panel_orig
  panelset,panel	       ;re-sets the panel position viewport   

end   
   
;******************************************************************************
function papco_color_bar_log_ticks, axis, index, t
  exp_label=strtrim(string(fix(ALOG10(t))),2)
  return, '10!U'+exp_label
end
;******************************************************************************

