;******************************************************************************
; NAME: papco_colorbar
;
; PURPOSE:
; This IDL-procedure plots a papco_colorbar on the right side of the screen
;	
; CATEGORY:
;       Plotting (Color)
;
; CALLING SEQUENCE:
;
;       papco_colorbar,ZMIN,ZMAX,relbit [,azimbit,ztitle]
;
; INPUTS:     in accordance with PLOT_CLR
;             ZMIN     float bottom value of color axis
;             ZMAX     float top value of color axis
;             relbit    =0: logarithmic z-scaling (particle fluxes)
;                       =1: linear z-scaling      (relative intensities[%])
;                       =2: linear z-scaling      (particle fluxes)
;                       =3: linear z-scaling      (azimuth angles [deg])
;                      according ytitles are drawn
;
; OPTIONAL INPUTS:
;             azimbit   =0: color values ranging from 2-255(def)
;                       =1: color values ranging from 1-223
;	
; PROCEDURE:  The color sequence is (and must be) linear, even for
;	      logarithmic z-scaling.
;             Axes are drawn using the AXIS- and LOGAXIS-procedures.
;             Viewport and data-screen relation are restored at the end.
;
; MODIFICATION HISTORY:
;       written jan 92 M.Fraenz,N.Krupp
;******************************************************************************
PRO papco_colorbar,ZMIN,ZMAX,relbit,azimbit,ztitle

; get starting offset and no of colors for color table. 
   c_index=papco_GetColor_table()   ;get index for papco color table   
   indicies=papco_get_Colorindices(c_index)
   nc=indicies(1)-indicies(0)+1
   c_offset=indicies(0)

        IF N_PARAMS(0) LT 4 THEN azimbit=0
        IF N_PARAMS(0) LT 5 THEN ztitle=''
        IF azimbit EQ 1 THEN compfac=7.0D/8.0D ELSE compfac=1.0D  
; save current viewport and data-screen-relation:
        pa=!p
        xa=!x
        ya=!y
        !x.title=''
	!p.TITLE=''
        !y.title=''
        !p.ticklen=-.2
        !p.position(0)=!p.position(2)+0.065
        !p.position(2)=!p.position(0)+0.025
	!x.range=[0.,100.]
	!y.range=[ZMIN,ZMAX]
        !x.style=4+1
        !y.ticks=0
        linlogbit = (relbit LT 1)
        !y.style = 1+4
	zfac= DOUBLE(zmax-zmin)/nc
	PLOT,[0,100],[zmin,zmin],color=!p.background,/noerase

;*******************create color bar (linear color sequence)
	FOR I=c_offset(0),c_offset(0)+nc-1 DO BEGIN
         ZINT1=(I-2-c_offset(0)+2)*zfac+zmin
         ZINT2=(I-1-c_offset(0)+2)*zfac+zmin
         color=FIX(I)
	 POLYFILL,[0.,0.,100.,100.],[ZINT1,ZINT2,ZINT2,ZINT1],col=color
	ENDFOR
 
;*******************axes and annotation (only here do we have to distinguish
;                   linear and logarithmic scaling)
       IF linlogbit EQ 1 THEN BEGIN
          AXIS,yaxis=1,charsize=!P.CHARSIZE, ticklen=-0.6, ylog=1, color=1, $
            yrange=[ZMIN,ZMAX],  ytickformat='papco_colorbar_log_ticks'
       ENDIF ELSE BEGIN
         !x.style=4
         !p.color=1
         AXIS,yaxis=1,ystyle=1,charsize=1.0
       ENDELSE
       !p.font=-1
       
; set position for zaxis label
       rel_label_y=(!p.position(1)+!p.position(3))/2
       rel_label_x=!p.position(2)+0.065
       
; set size of label to fit into space of panel. Reduce size if required.
; get character size in normal coordinates  
       chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
       n_char=strlen(ZTITLE)
       res=str_sep(ZTITLE,'!')
       sp_char=(n_elements(res)-1)*2
       n_char=n_char-sp_char*0.9
       
       rel_label_height=chrs(1) * n_char
       rel_avail_height=!p.position(3)-!p.position(1)
       fit_ratio=(rel_label_height)/rel_avail_height
       
       if fit_ratio gt 1.0 then begin  ; label is too long to fit! Reduce!
         char_factor=1/fit_ratio
       endif else begin
         char_factor=1
       endelse  
                
       XYOUTS,rel_label_x, rel_label_y, ZTITLE, alig=0.5, $
         chars=!P.CHARSIZE*1.5*char_factor, $
         orient=-90., color=1, /normal
       
; restore original settings       
   !x=xa
   !y=ya
   !p=pa
   return
   
end   
   
;******************************************************************************
function papco_colorbar_log_ticks,axis,index,t
  
  exp=ALOG10(t)
  exp_label=strtrim(string(fix(exp)),2)
  tick_label='10!U'+exp_label
  return, tick_label
  
end

