;******************************************************************************
;* FUNCTION:
;*      yres,binSizes
;*
;* DESCRIPTION:
;*      Works out the lowest comon denominator for all binsizes, so that
;*      each binszie is a multiple therof to an accuracy of 75%.
;*
;* READ THIS !
;*      When you experience PROBLEMS IN THE RESOLUTION on the y-axis, this
;*  routine is probably the bad guy. Just increase the % accuracy
;*      required. Warning: this may drastically increase your filesizes for
;*      printing, and slow down plotting to screen.
;*
;* INPUTS:
;*      binSizes    : array containing the binsizes of the data bins.
;*
;* OUTPUT:
;*      an Integer
;*
;* CALLING SEQUENCE:
;*      res=yres(binSizes)
;*
;* MODIFICATION HISTORY:
;*      December 1995, written by R.Friedel at Goddard
;******************************************************************************
function yres, binSizes

     indices=where(binSizes NE 0)
     minfac=min(binsizes(indices))
     mindiff=0
     while mindiff lt 85 do begin
        factors=binsizes(indices)/minfac
        newbinsize=fix(factors)*minfac
        diff=100* newbinsize/binsizes(indices)
        mindiff=min(diff)
        res=total(binsizes(indices))/minfac
        minfac=minfac*0.9
     endwhile
     return,fix(res)
end

;******************************************************************************
;* FUNCTION:
;*      calculateHeight, ymin, ymax
;*
;* DESCRIPTION:
;*      This function calculates the height of the plot-array.
;*  The height of this array is the resolution, in which the data on the
;*      y-axis are plotted. When the databins have different sizes, one can't
;*      just choose the number of bins to be the resolution but has to make
;*  sure, that the resolution is big enough for all bins to be visible.
;*  So, the bin-sizes are added in units of the smallest bin, and this
;*  sum is made the y-resolution of the plot-array.
;*
;* READ THIS !
;*      When you experience PROBLEMS IN THE RESOLUTION on the y-axis, this
;*  routine is probably the bad guy. Just increase the returned value to
;*  increase your resolution. (Be careful: when printing, the size of the
;*      created Postscript-files is almost proportional to this function's
;*      return-value.
;*
;* INPUTS:
;*      ymin, ymax  : arrays containing the bounds of the data-bins
;*
;* OUTPUT:
;*      an Integer
;*
;* CALLING SEQUENCE:
;*      yDim=calculateHeight(YMINArr, YMAXArr)
;*
;* MODIFICATION HISTORY:
;*      august 1995, written by A.Keese
;******************************************************************************
FUNCTION calculateHeight, ymin, ymax

  IF !y.type eq 0 THEN BEGIN ; for linear y-scaling
     binSizes=(ymax-ymin)
     sum=yres(binsizes)  ;function that calculates lowest common binsize to 85%
     PRINT, '% plot_clr: Linear scaling; y-bins needed ', sum
     return, sum
  ENDIF ELSE BEGIN       ;for logarithmic y-scaling
     binSizes=(alog10(ymax)-alog10(ymin))
     sum=yres(binsizes)  ;function that calculates lowest common binsize to 85%
     PRINT, '% plot_clr: Logarithmic scaling; y-bins needed ', sum
     return, sum
  ENDELSE

END

;*****************************************************************************
;* PROCEDURE:
;*      PLOT_CLR,XARR,YMINARR,YMAXARR,ZARR,ZMIN,ZMAX,linlogbit [,azimbit,smbit]
;*
;* DESCRIPTION:
;*  plot a two-dimensional representation of three-dimensional data
;*
;* RESTRICTIONS:
;*      The x-coordinates are expected to be linear.
;*      The y-coordinates can be linear or logarithmic, but the exact
;*      y-field boundaries must be specified and be the same for all x.
;*
;*      If the output-device is not the screen or a printer with scalable pixel
;*  sizes, the routine 'PLOT_CLR_ALT' is used, which is extreme slow !
;*
;* HOW DOES THIS PROCEDURE DO ITS WORK - AND WHY ?
;*      Instead of plotting the data to the screen using the POLYFILL-Routines,
;*  the data is plotted into a two-dimensional data-array, 'image', which
;*  is output to the current device afterwards.
;*  Therefore, the dimension of 'image' is the effective resolution the data
;*  is plotted with. As the available RAM is finite, and because the dimension
;*      of image are almost proportional to the size of created Postscript-files,
;*      image can't have the size XResolution X YResolution.
;*  The height of 'image' is calculated by the function 'crres_calculate_Height'.
;*  The width of 'image' is
;*
;* INPUTS:
;*      XARR     FLTARR(xdim) x-coordinates
;*      YMINARR  FLTARR(ydim) lower y-boundaries of color-fields
;*      YMAXARR  FLTARR(ydim) upper y-boundaries of color-fields
;*               the y-coordinates must be the same for all x.
;*               YMINARR and YMAXARR can be unsorted.
;*      ZARR     FLTARR(xdim,ydim) data-values for each field
;*      ZMIN     float bottom value of color axis
;*      ZMAX     float top value of color axis
;*               linlogbit =0: linear z-scaling
;*                         =1: logarithmic z-scaling
;*
;* OPTIONAL INPUTS:
;*      azimbit   =0: color values ranging from 0-255(def)
;*                =1: color values ranging from 0-224
;*  smbit     =0: no smoothing
;*        =1: smoothing (default)
;*
;* KEYWORDS:
;*      USEOLD    when specified, the old routine is used.
;*        e.g.: PLOT_CLR,..., /USEOLD
;*
;* OUTPUT:
;*      a color-coded plot of ZARR on the current
;*      viewport (limits taken from !x.crange,!y.crange).
;*      Axes are drawn depending on the current
;*      value of the !type-variable. Usually x- and y-axes
;*      should be generated by the AXIS- or LOGAXIS-procedures,
;*      the z-axis by the papco_colorbar-procedure.
;*      Color density is influenced by the last parameter
;*      in POLYFILLS (25 for PHASERPS).
;*      Additionally the ZARR is smoothed and interpolated in
;*      y-direction. (the variable ZARR is not changed)
;*  This feature can be turned off by calling with smbit=0
;*
;*
;* MODIFICATION HISTORY:
;*  january 1992, M.Fraenz and N.Krupp
;*                    wrote the routine 'plot_clr_alt' at the end of this
;*            file
;*      august 1995,  written by A.Keese, based on 'plot_clr_alt'
;*            performance-enhancement by using the TV-Routine instead
;*            of the polyfill-routine
;*      18th august 95, A.Keese
;*                corrected the calculation of the vertical resolution
;*      21st august 95, A.Keese
;*            corrected the calculation of the horiz resolution
;*            corrected rounding error while calculating width
;*                    of color-bars
;*      february 1997, R.Friedel
;*                    to work with papco internal color table scheme
;******************************************************************************
PRO PLOT_CLR, XARR, YMINARR, YMAXARR, ZARR, ZMIN, ZMAX, $
                    linlogbit, azimbit, smbit, USEOLD=useold

   COMMON papco_color
   COMMON colors, r_orig,g_orig,b_orig,r_curr,g_curr,b_curr

; search common block for active color table (where papco_color(*,2) is set)
   active_vector=papco_color(*,2)
   c_index=where(active_vector eq 1)        ;set index for papco color table

; set the color table to be used.
   nc_total=!D.N_colors     ;get total number of colors
   nc_avail=nc_total-16-2   ;the top 16 color indices are reserved
                            ;the bottom two colors are reserved
                            ;    0=black for axis and annotation
                            ;    1=200   for background
   nc=nc_avail/4            ;divide by 4 to get no of colors for now.

; set starting offset for color table. Offset to Offset+n defines range
; of color indices this routine may use.
   c_offset=2+c_index*nc

; if the print device is postscript, allow full range for color table
   if !d.name eq 'PS' then begin
        ; do nothing as yet. To Be written.
   endif

; set default - values
   IF N_PARAMS(0) LT 8 THEN azimbit=0  ;for AZIM.VLT
   IF N_PARAMS(0) LT 9 THEN smbit=0    ;no smoothing [def]
   IF azimbit EQ 1 THEN compfac=7.0D/8.0D ELSE compfac=1.0D

; check, if the older routine shall be used...
   IF KEYWORD_SET(USEOLD) THEN BEGIN
      PLOT_CLR_ALT,XARR,YMINARR,YMAXARR,ZARR,ZMIN,ZMAX,linlogbit,azimbit,smbit
      RETURN
   ENDIF

; this procedure works with X-Display and Postscript only !
; if the printer does not supply resizable pixels, the older and slower routine
; must be used.

   IF !D.NAME NE 'X' THEN BEGIN
      PRINT, '% plot_clr: plotting data to printer...'
      IF !D.FLAGS AND 1 EQ 0 THEN $
      BEGIN
         PRINT, '-----------------------------------------------------'
         PRINT, '% plot_clr:  Output to Printer Device !'
         PRINT, "% plot_clr:  I can't use the fast Plot-Procedure"
         PRINT, '% plot_clr:  I am using the old Plotting-Procedure...'
         PRINT, '-----------------------------------------------------'
         PLOT_CLR_ALT,XARR,YMINARR,YMAXARR,ZARR,ZMIN,ZMAX, $
                      linlogbit,azimbit,smbit
     RETURN
      ENDIF
   ENDIF ELSE $
   PRINT, '% plot_clr: plotting data to display...'

; check the dimensions of the data-fields. This is done, cause we don't want
; to be suspected of being wrong when an error occurs :-) Besides, this
; simplifies debugging...

   dimYMIN=size(YMINArr)
   dimYMAX=size(YMAXArr)
   dimX   =size(XARR)
   dimZ   =size(ZARR)
   IF dimYMIN(0) NE 1 OR dimYMAX(0) NE 1 OR dimX(0) NE 1 OR dimZ(0) NE 2 $
   THEN BEGIN
      print, replicate('-',30)
      PRINT, '% plot_clr: PARAMETER Error !'
      PRINT, '            One or more ranges of the dimensions of '
      PRINT, '            YMINArr, YMAXArr, XARR or ZARR are wrong'
      print, replicate('-',30)
      help, /traceback
      print, replicate('-',30)
      return
   ENDIF
   IF dimYMIN(1) NE dimYMAX(1) OR dimZ(1) NE dimX(1) $
   THEN BEGIN
      print, replicate('-',30)
      PRINT, '% plot_clr: PARAMETER-RANGE-Error !'
      PRINT, '            One or more ranges of the dimensions of '
      PRINT, '            YMINArr, YMAXArr, XARR or ZARR are wrong'
      print, replicate('-',30)
      help, /traceback
      print, replicate('-',30)
      return
   ENDIF

; YMINin and YMAXin are yminarr and ymaxarr sorted by yminarr.
; they are the boundaries of the color fields.

   sortIndex=SORT(YMINarr)
   YMINin=YMINarr(sortIndex)
   YMAXin=YMAXarr(sortIndex)
   dim=N_ELEMENTS(YMINin)

; To turn on smoothing in the vertical direction,
; the bounds of the color-fields,
; YMINout and YMAXout, are sectioned finer

   IF smbit THEN BEGIN
     smdim=FIX(100.0/dim)       ; interpolation (1=no interpolation)
     smno=9                     ; smoothing     (1=no smoothing)
                                ; increase the '100' or '9' for finer smoothing
     jmax=smdim-1
     YMINout=FLTARR(smdim*dim)
     YMAXout=FLTARR(smdim*dim)
     zvect=FLTARR(smdim*dim)
     zcount=N_ELEMENTS(zvect)
     FOR i=0,dim-1 DO BEGIN
       FOR j=0,jmax DO BEGIN
         YMINout(i*smdim+j)=((smdim-j)*YMINin(i)+j*YMAXin(i))/smdim
         YMAXout(i*smdim+j)=((smdim-j-1)*YMINin(i)+(j+1)*YMAXin(i))/smdim
       ENDFOR
     ENDFOR
   ENDIF ELSE BEGIN
     YMINout=YMINin
     YMAXout=YMAXin
     zcount=N_ELEMENTS(zarr(0,*))
  ENDELSE ; (smbit)

; if the colors shall be plotted logarithmic, adjust the ranges...

   IF linlogbit EQ 1 THEN BEGIN
     zfac=DOUBLE(ALOG10(ZMAX)-ALOG10(ZMIN))
     logmin=ALOG10(ZMIN)
   ENDIF ELSE BEGIN
     zfac=DOUBLE(ZMAX-ZMIN)
   ENDELSE ;(linlogbit EQ 1)

; get the rectangle in which to plot (in Device-coordinates)
   NormalXY0 = [!p.position(0), !p.position(1)] ; Bottom left in normal coords
   NormalXY1 = [!p.position(2), !p.position(3)] ; Top right in normal coords
   DeviceXY0 = round(convert_coord(NormalXY0, /NORMAL, /TO_DEVICE))
                                                ; bl corner (dev-coords)
   DeviceXY1 = round(convert_coord(NormalXY1, /NORMAL, /TO_DEVICE))
                                                ; tr corner (dev-coords)
   DeviceDX = DeviceXY1(0)-DeviceXY0(0)         ; width of the plot-rectangle
   DeviceDY = DeviceXY1(1)-DeviceXY0(1)         ; its height

;-- Calculate the size of the box in which the image is to be calculated-------
; the image is plotted using the TV-command. The two-dimensional array
; image will be filled with the plotted bitmap.
; we want the width of 'image' to be a multiple or a divisor of DeviceDX,
; and we want it to be close to N. If we would choose imageWidth to be not a
; multiple or divisor of DeviceDX, the output might be influenced by rounding
; errors. The same applies to the image height and the maximal Y-coordinate
; YMAXOutMax

; calculate image-width
  N=N_ELEMENTS(XARR)                 ; number of elements to be plotted

; calWidth is the horizontal resolution we wish to use. It is choosen so that
; each Element of XARR will be visible.
  calWidth=N*(!x.crange(1)-!x.crange(0))/(XARR(N-1)-XARR(0))
  imageWidth=LONG(calWidth) > 1
; now, the horizontal resolution used for plotting is calculated
;  IF DeviceDX GT calWidth THEN $
;     imageWidth=DeviceDX/ROUND(DeviceDX/calWidth) $
;  ELSE $
;     imageWidth=DeviceDX*ROUND(calWidth/DeviceDX)
  widthFactor=DOUBLE(DeviceDX)/DOUBLE(imageWidth)

; calculate image-height
  calImageHeight=calculateHeight(YMINOut, YMAXOut)
  imageHeight=LONG(calImageHeight) > 1
;  IF DeviceDY GT calImageHeight THEN $
;     imageHeight=DeviceDY/ROUND(DeviceDY/calImageHeight) $
;  ELSE $
;     imageHeight=DeviceDY*ROUND(calImageHeight/DeviceDY)
  heightFactor=DOUBLE(DeviceDY)/DOUBLE(imageHeight+1)

  image=intarr(imageWidth, imageHeight+1)

  PRINT, '% plot_clr: Image resolution', $
                       imageWidth,'x',imageHeight

; 'space' is half of the width of a color bar in DATA-coordinates
   space=DOUBLE((DOUBLE(!x.crange(1)-!x.crange(0))/(calWidth+1))/2.0);

; calculate vertical scaling. We use the IDL-routines, so that set coordinate
; systems will be applied. Afterwards, the scaling is transformed to our
; imageHeight by dividing by heightFactor

; if there are as many ybins as there are image bins, the mapping is
; straightforward:

  imageHeight=imageHeight+1

  if imageHeight eq N_ELEMENTS(YMINout) then begin
     yminout=findgen(N_ELEMENTS(YMINout))
     ymaxout=yminout
  endif else begin
     xDummy=indgen(N_ELEMENTS(YMINout))
     YY=convert_coord(xDummy, YMINout, /DATA, /TO_DEVICE)-DeviceXY0(1)
     YMINout=round(yy(1,*)/heightFactor) < (imageHeight-1) ; go sure, that the
     YMINout=YMINout > 0                            ; ranges of YMINout are ok.
     YY=convert_coord(xDummy, YMAXout, /DATA, /TO_DEVICE)-DeviceXY0(1)
     YMAXout=round(yy(1,*)/heightFactor) < (imageHeight-1) ; make sure the ranges
     YMAXout=YMAXout > 0                       ; in YMAXout are ok.
  endelse

; the FOR-NEXT-loop fills the 'image'-variable
  FOR x=0, N-1 DO BEGIN
     IF XARR(x) GT 0.0 THEN BEGIN
       IF smbit EQ 0 THEN BEGIN  ;no smoothing
          zvect=ZARR(x,*)
          zvect=zvect(sortIndex)
       ENDIF ELSE BEGIN          ;apply smoothing
      FOR i=0, dim-1 DO $
             FOR j=0, smdim-1 DO zvect(i*smdim+j)=ZARR(x, i)
          zvect=SMOOTH(zvect, smno)
          zvect=SMOOTH(zvect, smno)
       ENDELSE ;(smbit EQ 0)

       ; by rounding, zvect may become zero. When applying 'alog10' to
       ; zvect, this yields  errors. So, those nasty values are made
       ; infenitesimally small...
       zero=where(zvect LE 0, count)
       IF count GT 0 THEN zvect(zero)=1E-36

; fold data into indices range available for this color bar
       IF linlogbit EQ 1 THEN $
          valcol=c_offset(0)+fix(nc*(alog10(zvect)-logmin)/zfac) $
       ELSE $
          valcol=c_offset(0)+fix(nc*(zvect-zmin)/zfac)

       ; next, restrict zvect's data to the allowed ranges
       exclude=WHERE(zvect LT zmin OR zvect GE zmax, count)
       IF count GT 0 THEN valcol(exclude)=0

       ; now, the rows of 'image' corresponding to x are filled.The IDL-Routine
       ; 'convert_coord' gives the coordinates in the DEVICE-System. These are
       ; transformed to the coordinates of 'image' by dividing by 'widthFactor'
       xp=convert_coord([XARR(x)-space, XARR(x)+space],[0,0],/DATA,/TO_DEVICE)

       xFrom=(xp(0,0) - DeviceXY0(0))/widthFactor
       xFrom=fix(xFrom+1) > 0                   ; check Range and round
       xFrom=xFrom < (imageWidth-1)             ; to higher integer

       xTo  =(xp(0,1) - DeviceXY0(0))/widthFactor
       xTo  =fix(xTo) > 0             ; check Range and round
       xTo  = xTo < (imageWidth-1)            ; to smaller int

       xFrom=xFrom < xTo

; when there are more zvals then bins, only those are plotted, for which
; bins exist
       FOR zval=0,(zcount-1) < N_ELEMENTS(ymaxout) DO BEGIN
         image(xFrom:xTo, YMINout(zval):YMAXout(zval))=valcol(zval)
       ENDFOR
     ENDIF ; IF (XARR(SourceX) LE 0)
   ENDFOR ; (SourceX)


;-- now, the image has been created, and is being plotted after being  scaled
;   to the  display size

   CASE !d.name of
   'WIN' : BEGIN
              TV, CONGRID(image, DeviceDX, DeviceDY), $
                     DeviceXY0(0), DeviceXY0(1), /device
    END
    'X'  : BEGIN
              TV, CONGRID(image, DeviceDX, DeviceDY), $
                     DeviceXY0(0), DeviceXY0(1), /device
    END
    'PS' : BEGIN
              TV, BYTE(image), DeviceXY0(0), DeviceXY0(1), $
                     xsize=DeviceDX, ysize=DeviceDY, /device
    END
    'Z'  : BEGIN
              TV, CONGRID(image, DeviceDX, DeviceDY), $
                     DeviceXY0(0), DeviceXY0(1), /device
    END
   ENDCASE ;(!d.Name)
   RETURN

END

;
; NAME: PLOT_CLR_ALT
;
; PURPOSE:
; This IDL-procedure plots a 2D-color-picture of a 3D-dataarray.
;
; READ THIS !
; This procedure was replaced in august 1995 by PLOT_CLR
;
; CATEGORY: Plotting (color)
;
; CALLING SEQUENCE:
; PLOT_CLR,XARR,YMINARR,YMAXARR,ZARR,ZMIN,ZMAX,linlogbit [,azimbit,smbit]
;
; INPUTS:     XARR     FLTARR(xdim) x-coordinates
;             YMINARR  FLTARR(ydim) lower y-boundaries of color-fields
;             YMAXARR  FLTARR(ydim) upper y-boundaries of color-fields
;                      the y-coordinates must be the same for all x.
;                      YMINARR and YMAXARR can be unsorted.
;             ZARR     FLTARR(xdim,ydim) data-values for each field
;             ZMIN     float bottom value of color axis
;             ZMAX     float top value of color axis
;             linlogbit =0: linear z-scaling
;                       =1: logarithmic z-scaling
;
; OPTIONAL INPUTS:
;             azimbit   =0: color values ranging from 0-255(def)
;                       =1: color values ranging from 0-224
;         smbit =0: no smoothing
;           =1: smoothing (default)
;
; RESTRICTIONS:
; The x-coordinates are expected to be linear and equi-distant.
; The y-coordinates can be linear or logarithmic, but the exact
; y-field boundaries must be specified a be the same for all x.
;
; PROCEDURE:
; OUTPUT    : a color-coded plot of ZARR on the current
;             viewport (limits taken from !x.crange,!y.crange).
;             Axes are drawn depending on the current
;             value of the !type-variable. Usually x- and y-axes
;             should be generated by the AXIS- or LOGAXIS-procedures,
;             the z-axis by the papco_colorbar-procedure.
;             Color density is influenced by the last parameter
;             in POLYFILLS (25 for PHASERPS).
;             Additionally the ZARR is smoothed and interpolated in
;             y-direction. (the variable ZARR is not changed)
;         This feature can be turned off by calling with smbit=0
;
; MODIFICATION HISTORY:
; written jan 92 by M.Fraenz,N.Krupp
  PRO PLOT_CLR_ALT,XARR,YMINARR,YMAXARR,ZARR,ZMIN,ZMAX,linlogbit,azimbit,smbit
; ----------------------------------------------------------
;
   IF N_PARAMS(0) LT 8 THEN azimbit=0  ;for AZIM.VLT
   IF N_PARAMS(0) LT 9 THEN smbit=1    ;smoothing [def]
   IF azimbit EQ 1 THEN compfac=7.0D/8.0D ELSE compfac=1.0D
   index=SORT(YMINARR)
   YMINin=YMINARR(index)
   YMAXin=YMAXARR(index)
   dim=N_ELEMENTS(YMINin)
   IF smbit THEN BEGIN
    smdim=FIX(40.0/dim)            ; interpolation (1=no interpolation)
    smno=9                         ; smoothing     (1=no smoothing)
    jmax=smdim-1
    YMINout=FLTARR(smdim*dim)
    YMAXout=FLTARR(smdim*dim)
    zvect=FLTARR(smdim*dim)
;
    FOR i=0,dim-1 DO BEGIN
     FOR j=0,jmax DO BEGIN
      YMINout(i*smdim+j)=((smdim-j)*YMINin(i)+j*YMAXin(i))/smdim
      YMAXout(i*smdim+j)=((smdim-j-1)*YMINin(i)+(j+1)*YMAXin(i))/smdim
     ENDFOR
    ENDFOR
   ENDIF ELSE BEGIN
    yminout=yminin
    ymaxout=ymaxin
   ENDELSE
;
   !c=0
;   PLOT,!x.crange,!y.crange
;
   IF linlogbit EQ 1 THEN BEGIN
    zfac=DOUBLE(ALOG10(ZMAX)-ALOG10(ZMIN))
    logmin=ALOG10(ZMIN)
   ENDIF ELSE BEGIN
    zfac=DOUBLE(ZMAX-ZMIN)
   ENDELSE
;
   N=N_ELEMENTS(XARR)
   space=(!x.crange(1)-!x.crange(0))/N/2.0
   PRINT, '% plot_clr: plotting using polygonfill...'
   FOR x=0,N-1 DO BEGIN
    print, N-x
    IF XARR(x) GT 0.0 THEN BEGIN
     zin=ZARR(x,*)
     zin=zin(index)
     IF smbit THEN BEGIN
      FOR i=0,dim-1 DO $
       FOR j=0,smdim-1 DO zvect(i*smdim+j)=zin(i)
      zvect=SMOOTH(zvect,smno)
      zvect=SMOOTH(zvect,smno)
     ENDIF ELSE zvect=zin
     zcount=N_elements(zvect)
     FOR zval=0,zcount-1 DO BEGIN
      zint=zvect(zval)
      ymin=YMINout(zval)
;      IF zval GE 1 THEN ymin=ymax
      ymax=YMAXout(zval)
      IF zint GE zmin THEN BEGIN
       IF linlogbit EQ 1 THEN $
        valcol=FIX(254.0*(ALOG10(zint)-logmin)/zfac*compfac) $
       ELSE valcol=FIX(255.0*(zint-zmin)/zfac*compfac)
      ENDIF ELSE valcol=0          ;black(phaserps)
      IF zint GE zmax THEN valcol=0  ;change to white(phaserps)
      POLYFILL,[XARR(x)-space,XARR(x)+space,XARR(x)+space,XARR(x)-space],$
                [ymin,ymin,ymax,ymax],col=valcol
     ENDFOR ; zval
    ENDIF ; XARR(x)
   ENDFOR ; x
;
 RETURN
 END
