;******************************************************************************
;* FILE:	 
;*    comments.pro
;* 
;* DESCRIPTION:  
;*    This is a utility for gaining informations on a tree of IDL-files.
;*    All IDL-Programs (*.pro) that are contained in a certain path AND 
;*    anywhere in its subtree are analyzed. 
;*    The functions, procedures and structures found in theses IDL-programs 
;*    are accumulated in some reference-files.    
;*    
;*    If you start this program by
;*       .run comments
;*       commentsFor, '~/myPrograms', '~/protocol'
;*    those files are written:
;*       ~/protocol.files       - lists all procedures, functions, structures
;*                                by the file they're contained in
;*       ~/protocol.proc_ref    - is a reference: all procedure/functions or
;*				  structure-names are listed alphabetically
;*				  and the files, they're defined in are listed
;*       ~/protocol.definitions - similar to .proc_ref but not only the names
;*				  of those object are listed but their 
;*				  definitions, too
;*	 ~/protocol.def_warn	- warns you, if a procedure or function is
;*				  declared in multiple files
;*	 ~/protocol.file_warn	- lists duplicate file-names
;*
;* USAGE :
;*    Start this utility by typing
;*       commentsFor, dirName, protocolName
;*    e.g.: 
;*       commentsFor, '~/myPro', '~/protocol'
;*
;*    Before starting you have to compile those files:
;*       .run string_functions
;*	 .run comments
;*
;* MODIFICATION HISTORY:       
;*     september 1995, written by A.Keese 
;******************************************************************************

; this is the recursive part of the program
PRO recurseComments, fName
  
   common COMMENTS, fileNames, fNum, statements, sNum

   allfiles=findfile('*')
   directories=strarr(100)
   n_dir=0
   for i=0, n_elements(allFiles)-1 do begin
      f=strtrim(allfiles(i),2)
      if strright(f, 1) EQ ':' then begin
         print, '--->dir:', f
         directories(n_dir)=strleft(f, strlen(f)-1)
         n_dir=n_dir+1
      endif
   endfor
   
   cd, current=currDir
   print, '--------------processing : '+currDir

   thisFiles=findfile('./*.pro')
   for i=0, n_elements(thisFiles)-1 do begin
      f=strtrim(thisFiles(i),2)
      if strleft(f, 2) eq './' then thisfiles(i)=strfrom(f, 2)
      comments, thisFiles(i)
   endfor


   for i=0, n_dir-1 do begin  ; now, start recursion
      cd, currDir
      cd, directories(i)
      recurseComments, fName
   endfor
END


;******************************************************************************
; this is the main-procedure. 
;
PRO commentsFor, directory, protocol_file, PAPCO=PAPCO
; creates those files: 
; protocol_file+'.proc_ref'       -> alle pro / functions / structures
;                                    alphabetisch geordnet
;				     mit referenz auf file-name
; protocol_file+'.definitions'	  -> alle pro / functions  mit function-header
;				     mit referenz auf file-name
; protocol_file+'.def_warnings'   -> warnings ....  (multiple definitions)
; protocol_file+'.file_warnings'   -> warnings ....  (multiple filenames)
; protocol_file+'.files'	  -> alle file-namen mit: pros, functions, stru
;
; If keyword PAPCO is set the local papco setup is used as root
; directory and outout is written to papco_doc/commentsfor
  
   common COMMENTS, fileNames, fNum, statements, sNum
   
   if N_elements(protocol_file) EQ 0 then protocol_file='~/protocol'
   
   if keyword_set(PAPCO) then begin
     temp=directory
     DIR_ROOT=papco_getenv('PAPCO_PATH')
     DIR_ROOT=strmid(DIR_ROOT,0,strlen(DIR_ROOT)-6)
     directory=DIR_ROOT+temp+'/'
     protocol_file=DIR_ROOT+'papco_doc/commentsfor/'+temp
   endif  

   fileNames=strarr(8000, 2)     ; fileNames(*,0) : directory-name
                                ; fileNames(*,1) : file-name
   fNum=0                       ; number of filenames

   statement={STATEMENTS1, $
              name:'', $
              type:0, $         ; 1: Procedure,2:function,3:structure
              definition:strarr(8000), $
              n_definition:0, $ 
              file:-1}
   statements=replicate(statement, 8000)
   snum=0                       ; number of statements
   cd, directory, current=oldDir
   recurseComments, protocol_File
   cd, oldDir

   write_refsByStatement, directory, protocol_file
   write_refsByFile, directory, protocol_file
END


;
; write all references by Procedure-name
;
PRO write_refsByStatement, directory, protocol_file
   common COMMENTS, fileNames, fNum, statements, sNum

   print, '---now writing '+protocol_file+'.proc_ref ---'
   print, '---now writing '+protocol_file+'.def_warnings ---'
   print, '---now writing '+protocol_file+'.definitions ---'
   openw, w_ref,  protocol_file+'.proc_ref', /get_lun 
   openw, w_warn, protocol_file+'.def_warnings', /get_lun
   openw, w_def,  protocol_file+'.definitions', /get_lun

   typeLabels=['PROCEDURES', 'FUNCTIONS', 'STRUCTURES']
   fnLabels=['pro', 'function', 'structure : ']
   stars='***************************************'
   s_line=stars+stars
   
   printf, w_ref, s_line
   printf, w_ref, 'Reference-file for all IDL-files under'
   printf, w_ref, '$DIR_ROOT = '+directory
   printf, w_ref, 'created on : ', systime()
   printf, w_ref, s_line
   
   printf, w_ref, s_line
   printf, w_warn, 'Warnings for all IDL-files under'
   printf, w_warn, '$DIR_ROOT = '+directory
   printf, w_warn, 'created on : ', systime()
   printf, w_warn, s_line
   
   printf, w_def, s_line
   printf, w_def, 'Definitions-file for all IDL-files under '
   printf, w_def, '$DIR_ROOT = '+ directory
   printf, w_def, 'created on : ', systime()
   printf, w_def, s_line

   n_warn=0
   FOR type=1,3 DO BEGIN
      printf, w_ref, ''
      printf, w_ref, ''
      printf, w_ref, s_line
      printf, w_ref, '  '+typeLabels(type-1)
      printf, w_ref, s_line

      printf, w_def, ''
      printf, w_def, ''
      printf, w_def, s_line
      printf, w_def, '  '+typeLabels(type-1)
      printf, w_def, s_line

      printf, w_warn, ''
      printf, w_warn, ''
      printf, w_warn, s_line
      printf, w_warn, '  '+typeLabels(type-1)
      printf, w_warn, s_line

      ind=where(statements.type EQ type)
      arr=statements(ind)
      srt=sort(strlowcase(arr.name))
      
      lastName=''
      lastDuplicate=''
      FOR i=0, N_ELEMENTS(srt)-1 DO BEGIN
         name=arr(srt(i)).name
         
         ; file-name
         n_file=arr(srt(i)).file
         fn=fileNames(n_file, 0)+fileNames(n_file, 1)
         pos=strpos(fn,directory)
         if pos ne -1 then begin
           fn='$DIR_ROOT/'+strmid(fn,strlen(directory),strlen(fn))
         endif  
           
         ; -- short reference --
         fun=fnLabels(type-1)+' ' + arr(srt(i)).name
         fun=strformat(fun, 40, /LEFT)
         printf, w_ref, fun+' ',fn
         
         ; -- warnings --
         if type NE 3 then begin  ; only warn multiple procedure-definitions
            if strlowcase(name) eq strlowcase(lastName) then begin
               if lastDuplicate NE name THEN BEGIN
                  printf, w_warn,  'Duplicate definition : '+strtrim(fun, 2)
                  printf, w_warn,  '     defined in '+fnold
                  n_warn=n_warn+1
               ENDIF
               printf, w_warn,  '            and '+fn
               lastDuplicate=name
            endif
         endif
         
         ; -- long reference --
         for j=0, arr(srt(i)).n_definition-1 do $
            printf, w_def, arr(srt(i)).definition(j)
         printf, w_def, '    ; defined in: ' + fn
         printf, w_def, ''
         
         fnold=fn
         lastname=name
      ENDFOR
   ENDFOR

   printf, w_warn, ''
   printf, w_warn, '--- '+strtrim(string(n_warn), 2)+' warnings encountered --'
   
   close, w_warn   &  free_lun, w_warn
   close, w_ref    &  free_lun, w_ref
   close, w_def    &  free_lun, w_def
   
END


;******************************************************************************
; write all references listed by their file-name
;
PRO write_refsByFile, directory, protocol_file
   common COMMENTS, fileNames, fNum, statements, sNum

   print, '---now writing '+protocol_file+'.files ---'
   openw, w_file,  protocol_file+'.files', /get_lun

   stars='***************************************'
   s_line=stars+stars

   printf, w_file, 'All IDL-files in '+ directory
   printf, w_file, '$DIR_ROOT = ' + directory
   printf, w_file, 'created on : ', systime()
   printf, w_file, s_line

   lastFileName=''
   FOR fileNr=0, fNum-1 DO BEGIN
     fn=fileNames(fileNr, 0)+fileNames(fileNr, 1)
     pos=strpos(fn,directory)
     if pos ne -1 then begin
       fn='$DIR_ROOT/'+strmid(fn,strlen(directory),strlen(fn))
     endif       

     printf, w_file, ''
     printf, w_file, ''
     printf, w_file, s_line
     printf, w_file, '  '+fn
     printf, w_file, s_line

     ind=where(statements.file EQ fileNr, count)
      IF count GT 0 THEN BEGIN
         
         FOR i=0, N_ELEMENTS(ind)-1 DO BEGIN
            stm=statements(ind(i))
            name=stm.name
                                ; -- long reference --
            for j=0, stm.n_definition-1 do $
              printf, w_file, stm.definition(j)
            printf, w_file, ''
         ENDFOR
      ENDIF
   ENDFOR
   close, w_file
   free_lun, w_file


; -- file-warnings
   print, '---now writing '+protocol_file+'.file_warnings ---'
   openw, w_warn, protocol_file+'.file_warnings', /get_lun
   printf, w_warn, 'Warnings for duplicate IDL-files in '+directory
   printf, w_warn, 'created on : ', systime()
   printf, w_warn, s_line
   printf, w_warn, ''

   ind=where(fileNames(*,1) NE '', nfN)
   fN = fileNames(ind, *)
   n_warn=0

   i = 0
   FOR i = 0, nfN-1 DO BEGIN 
       IF fN(i, 1) EQ '' THEN CONTINUE 
       idx1 = where(fN(*, 1 ) EQ fN(i, 1), c1, complement = idx2,  ncomplement = c2)
       IF c1 GT 1 THEN BEGIN
           printf, w_warn, 'Duplicate file(s) : '+fN(idx1(0), 1)
           printf, w_warn,'         in '+fN(idx1(0), 0)
           FOR j = 1, c1-1 DO printf, w_warn,'     and in '+fn(idx1(j), 0)
       ENDIF  
       fN(idx1, 1) = ''
       n_warn=n_warn+c1-1
   ENDFOR 

   printf, w_warn, ''
   printf, w_warn, '--- '+strtrim(string(n_warn), 2)+' warnings encountered ---'
   close, w_warn
   free_lun, w_warn
END


;******************************************************************************
; parse a function-definition for its name
; the function-definition is expected to contain no '$' or ';'
; enclosed in a string (I think, something like that can't appear in a
; valid function-definition
;
; totalLine is an array of on IDL-Statement. 
;           e.g.: totalLine = ['pro a, $', 'b']
FUNCTION functionName, totalLine
   i=0
   n_word=0
   word=''

   repeat begin
      l=strtrim(totalLine(i), 2)+' '
      repeat begin
         ind1=strpos(l, ' ')
         ind2=strpos(l, ',')
         if ind2 gt -1 then if ind2 le ind1 then ind1=ind2

         if ind1 gt -1 then begin
            word=strtrim(strleft(l, ind1), 2)
            l=strtrim(strfrom(l, ind1), 2)+' '
            if strleft(word,1) EQ '$' or strleft(word,1) EQ ";" then begin
               ind1=-1
            endif else $
               n_word=n_word+1
         endif
      endrep until ind1 EQ -1 OR n_word EQ 2
      i=i+1
   endrep until i ge N_ELEMENTS(totalLine) or n_word eq 2

   if strright(word, 1) EQ ',' then word=strleft(word, strlen(word)-1)

   if n_word NE 2 then return, ''
   return, word
END 


;******************************************************************************
; parse a structure-definition for the structure's name
;
; totalLine is an array of on IDL-Statement. 
;           e.g.: totalLine = ['tmp={str, $', 'a:1}']
FUNCTION structureName, totalLine
   n_line=0
   found=0
   word=''
   mode=0 ; mode : 0 -> search for '{'
          ; mode : 1 -> now, search for the next word
          ; mode : 2 -> a word is found, search for ',' or ':'

   repeat begin ; loop : scan through elements of 'totalLine'
      pos=0
      line=strtrim(totalLine(n_line), 2)
      length=strlen(line)
      quoteOpen1=0 
      quoteOpen2=0

      repeat begin ; loop : scan through characters of 'line'
         c=strmid(line, pos, 1)
         
         ; make sure, we're outside strings
         if c eq "'" and not quoteOpen2 then begin 
            if quoteOpen1 then $
              quoteOpen1=0 $
            else $
              quoteOpen1=1
         endif
         if c eq '"' and not quoteOpen1 then begin 
            if quoteOpen2 then $
              quoteOpen2=0 $
            else $
              quoteOpen2=1
         endif
         
         if (not quoteOpen1) and (not quoteOpen2) then begin
            if (c eq ';') or (c eq '$') then begin
               pos=length
            endif else begin
               if c eq '{' then begin ; structure - definition found
                  mode = 1
               endif

               if mode EQ 1 then begin
                  if (c GE 'a' AND c LE 'z') OR $
                     (c GE 'A' AND c LE 'Z') $
                  then begin  ; word-start
                     startpos=pos
                     endword=0
                     pos=pos+1
                     while (pos lt length-1) and (not endword) do begin
                        s=strmid(line, pos, 1)
                        pos=pos+1
                        if NOT ((s ge 'a' and s LE 'z') OR $
                                (s ge 'A' and s le 'Z')) $
                        then begin
                           endword=1
                           if ('0' le s) and ('9' ge s) then endword=0
                           if s eq '_' then endword=0
                        endif
                     endwhile

                     if endword then $ ; not EOL ?
                       pos=pos-2       ; then let pos point to the last char of
                                       ; the last word

                     word=strmid(line, startpos, pos-startpos+1)
                     mode=2
                  endif
               endif 

               if mode EQ 2 then begin
                  if c EQ ':' THEN $ ; Structure without name
                    return, '' 

                  if c EQ ',' THEN $
                    return, word
               endif
            endelse
         endif

         pos=pos+1
      endrep until pos ge length or found
      n_line=n_line+1
   endrep until n_line GE n_Elements(totalLine) or found
   return, ''
END


;******************************************************************************
; process one file - all interestings statements are searched and 
; stored in the common-block COMMENTS
;
; fName : the name of a file in the current directory
;
PRO comments, fname  
   common COMMENTS, fileNames, fNum, statements, sNum

   cd, current=currDir
   currDir=strtrim(currDir, 2)
   if strright(currDir, 1) NE '/' THEN currDir=currDir+'/'
   print, '======='+currDir+fName
   fileNames(fNum, 0)=currDir
   fileNames(fNum, 1)=fName
   fNum=fNum+1

   openr, r_Unit, fname, error=err, /GET_LUN
   IF err NE 0 THEN BEGIN
      print, 'Error: ' + !err_string
      return
   ENDIF

   line=''
   inBlocks=0
   inBrackets1=0   ; ( )
   inBrackets2=0   ; [ ]
   inBrackets3=0   ; { }
   inFunction=0

   while not eof(r_unit) do begin
      parseLine, r_Unit, totalLine, $
        inBrackets1, inBrackets2,  inBrackets3, $
        inBlocks, $
        foundFunction, foundEndBlock, foundStartBlock, $
        foundStructure      

      n_lines=n_elements(totalLine)
      if foundFunction NE 0 then begin
         name=functionName(totalLine)
         if name ne '' then begin
            print, '     Function ' + name
            statements(sNum).name=name
            statements(sNum).type=foundFunction
            for i=0, n_lines-1 do $
              statements(sNum).definition(i)=totalLine(i)
            statements(sNum).n_definition=n_lines
            statements(sNum).file=fNum-1
            sNum=sNum+1
         endif else begin
            print, 'ERROR: ', totalLine
         endelse
      endif
      
      if foundStructure then begin
         name=structureName(totalLine)
         IF (name NE '') AND (name NE 'CW_PDMENU_S') THEN BEGIN
            print, '       Structure ' + name
            statements(sNum).name=name
            statements(sNum).type=3
            for i=0, n_lines-1 do $
              statements(sNum).definition(i)=totalLine(i)
            statements(sNum).n_definition=n_lines
            statements(sNum).file=fNum-1
            sNum=sNum+1
         endif
      endif

   endwhile
   close, r_Unit
   free_LUN, r_Unit
END


;******************************************************************************
; read an IDL-statement, parse it, return data in the parameters.
; INPUT:
;   r_unit    : logical file-unit to read from
;
; OUTPUT :
; totalLine   : returns an array with all file-lines of that IDL-statement
; inBrackets1 : depth in '(' ... ')' (should be zero before and after each 
;               read statement - may be used for expansions.
; inBrackets2 : depth in '[' ... ']' (should be zero before and after each 
;               read statement - may be used for expansions.
; inBrackets3 : depth in '{' ... '}' (should be zero before and after each 
;               read statement - may be used for expansions.
; inBlocks    : this integer is increased or decreased to contain the depth
;               of IDL-blocks (as BEGIN...END or FUNCTION...END...)
; foundFunction: set to one, if 'totalLine' contains a function-definition or
;               a procedure definition
; foundEndBlock: set to one, if the statement is the end of a block like
;               'END', 'ENDFOR', ...
; foundStartBlock: set to one, if the statement is a BEGIN-statement
; foundStructure: set to one, if the statement contains a structure definition
PRO parseLine, r_Unit, totalLine, $                                            
               inBrackets1, inBrackets2, inBrackets3, inBlocks, $
               foundFunction, foundEndBlock, foundStartBlock, $
               foundStructure

   endarray=['end', 'endfor', 'endif', 'endelse', $
             'endcase', 'endrep', 'endwhile']

   totalLine=strarr(8000)
   n_lines=0

   foundFunction=0
   foundEndBlock=0
   foundStartBlock=0
   foundStructure=0

   repeat begin ; read the next IDL-line including continuations --------------
      aline=''
      quoteOpen1=0 
      quoteOpen2=0
   
      readf, r_Unit, aLine
      line=' '+strtrim(aLine, 2)+' '
      line=strlowcase(line)
      length=strlen(line)

      contLine=0
      pos=0
      
      repeat begin ; parse this line of the file ------------------------------
         c=strmid(line, pos, 1)
         
         ; check for string-start / end ----
         if c EQ "'" then begin
            if quoteOpen1 then $
              quoteOpen1=0 $
            else $
              if not quoteOpen2 then quoteOpen1=1
         endif
         if c EQ '"' then begin
            if quoteOpen2 then $
              quoteOpen2=0 $
            else $
              if not quoteOpen1 then quoteOpen2=1
         endif
         
         ; if not inside a string, parse this word ----------------------------
         if (not quoteOpen1) and (not quoteOpen2) then begin
            ;-- check for comments
            if c EQ ';' then begin
;               print, 'Comment : ', strfrom(line, pos)
               line=strleft(line, pos)
            endif

            ; check for line-continuation
            if c EQ '$' then begin
               contLine=1
;               print, 'continuation...'
            endif

            if c EQ '(' then inBrackets1=inBrackets1+1
            if c EQ '[' then inBrackets2=inBrackets2+1
            if c EQ '{' then begin
               inBrackets3=inBrackets3+1
               foundStructure=1
            endif
            if c EQ ')' then inBrackets1=inBrackets1-1
            if c EQ ']' then inBrackets2=inBrackets2-1
            if c EQ '}' then inBrackets3=inBrackets3-1

            if c GE 'a' AND c LE 'z' then begin ; word-start
               startpos=pos
               endword=0
               pos=pos+1
               while (pos lt length-1) and (not endword) do begin
                  s=strmid(line, pos, 1)
                  pos=pos+1
                  if s lt 'a' or s gt 'z' then begin
                     endword=1
                     if ('0' le s) and ('9' ge s) then endword=0
                     if s eq '_' then endword=0
                  endif
               endwhile

               if endword then $ ; not EOL ?
                 pos=pos-1

               word=strmid(line, startpos, pos-startpos)

               ; now, check the word for one of the keywords
               if word eq 'begin' then begin
                  foundStartBlock=1
                  inBlocks=inBlocks+1
               endif else begin               
                  a=where(endarray eq word, count)
                  if count gt 0 then begin
                     foundEndBlock=1
                     inBlocks=inBlocks-1
                  endif
               endelse

               if word eq 'pro' then begin
                  inBlocks=inBlocks+1
                  foundFunction=1 
               endif

               if word eq 'function' then begin
                  inBlocks=inBlocks+1
                  foundFunction=2
               endif
               
            endif
         endif

         pos=pos+1
      endrep until pos ge length  ; end parsing one line of the file ----------

      totalLine(n_lines)=aLine
      n_lines=n_lines+1
   endrep until (not contLine) or eof(r_Unit) ; end parsing one IDL-statement -

   totalLine=totalLine(0:n_Lines-1)
   
END


