;******************************************************************************
;******************************************************************************
;* FILE:	 
;*    convert_time.pro
;* 
;* DESCRIPTION:  
;*    This file contains conversion-utilities for times and dates.
;*
;*    These routines are used with papco. Internally PAPC0 uses two time 
;*    formats:
;*
;*    T90  - seconds since 1990 (a long) - a relic from CRRES processing
;*    MJDT - Modified Julian Date plus time in seconds since the beginning
;*           of the day - a structure of the form
;*
;*           time={MJDT, $
;*                 MJD:0l, $             ; a long
;*                 T:0.0 }               ; a float
;*
;*    a float allows representation of seconds for a day down to milliseconds. 
;*
;*    Another older time format is TZERO, when the time is given in the form
;*    of day of year, year, seconds since midnight. Also a relic from
;*    CRRESS processing.
;*
;*    For compatibility with the older routines, The keyword MJDT is added to
;*    existing routines. 
;*
;*
;*    Provided are routines for converting from TZERO to T90
;*                          for converting time-strings to TZERO or T90
;*                      and for converting T90-times to a date and time
;*                      and for converting date-formats
;*
;*    The T90 and TZERO format are two ways of specifying a time after the
;*    1st January 1990 (CRRES started in 1990, hence this base)
;*    In the TZERO-format, the time consist of three numbers: 
;*       - The day of year (doy)
;*       - The year (a value greater 1990 or (greater 90 and smaller 100))
;*       - and a number which gives the seconds since midnight of 'doy/year'
;*
;*
;* OVERVIEW:
;*    Those procedures are contained:
;* 
;*    FUNCTION convert_TZERO_TO_T90, seconds, day, theYear
;*
;*    FUNCTION convert_T90_TO_TZERO, seconds, day, theYear, $
;*                                   DAY=ForDay, YEAR=ForYear
;*    
;*    PRO convert_t90_to_Date, time90, year, day_oy, hour, minute, second, $
;*    			      month, cmonth, day_om 
;*
;*    FUNCTION convert_date_to_t90, day=day, month=month, doy=doy, $
;*                              year=year, hour=hour, minutes=minutes, $
;*                              seconds=seconds
;*    
;*    FUNCTION convert_secondsToTime, seconds, T90=T90
;*    
;*    FUNCTION convert_timeToSeconds, aTimeString, T90=T90
;*
;*    FUNCTION convert_dayToDate, dayNumber, year
;*
;*    FUNCTION convert_dateToDay, dateString
;*
;* MODIFICATION HISTORY:       
;*      august,september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;******************************************************************************
;******************************************************************************


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_TZERO_TO_T90, seconds, day, theYear
;* 
;* DESCRIPTION:  
;*      Convert a TZERO-time to a T90-time or MJDT time
;* 
;* INPUTS:       
;*      seconds    a number 
;*      day        an integer - the day of year
;*      theYear    an integer - the year
;*
;* KEYWORDS:
;*      MJDT       if set, return time in  the MJDT format instead
;*
;* OUTPUT:	 
;*      T90-format: a long integer. -1 if outside T90 range OR
;*      time={MJDT, MJD:0l, T:0.0 }: a structure  in MJDT-format
;*                                   if keyword MJDT is set
;*
;* CALLING SEQUENCE:
;*      sec=convert_t90_to_tzero(t90_val_orig, sec, doy)
;*      t90_val=convert_tzero_to_t90(sec, doy, year)
;*      if t90_val NE t90_val_orig then print, 'error !'
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;******************************************************************************
function convert_tzero_to_t90, seconds, day, theyear, mjdt=mjdt

   IF theYear LT 100 THEN year=theYear+1900 ELSE year=theYear
      
; when the year GT 2050, the long-variable for T90 may overflow.
; The year must not be smaller than 1990 for T90

   IF NOT keyword_set(MJDT) THEN BEGIN
      IF year LT 1990 OR year GT 2050 THEN BEGIN
         print, ''
         print, '---PARAMETER-ERROR encountered---'
         print, 'You may not use a year after 2050 or before 1990 for T90!'
         print, ''
         return,-1
      ENDIF
      julday=LONG(JULDAY(1, 1, year)+day-1) 
      diffDays=LONG(julday - 2447893)   ;diffDays=LONG(julday-JULDAY(1,1,1990))
      diffSeconds=LONG(diffDays*24*3600)
      RETURN, seconds+diffSeconds
   ENDIF ELSE BEGIN
      time={MJDT, MJD:0l, T:0.0 }
      time.MJD=julday(1, 1, year)+day-1 - 2400001
      time.T=seconds
      if time.T gt 86400.0 then begin
        time.MJD=time.MJD + fix(time.T / 86400)
        time.T=time.T mod 86400.0
      endif  
      return,time
   ENDELSE

END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_T90_TO_TZERO, seconds, day, $
;*                       theYear, DAY=ForDay, YEAR=ForYear
;* 
;* DESCRIPTION:  
;*      This is the inverse function from convert_tzero_to_t90.
;*      A T90-value is converted to TZERO-format or MJD format
;*      if keyword MJDT is set
;*
;* INPUTS:       
;*      seconds  a long number - a T90-value
;* 
;* KEYWORDS:
;*      MJDT          if set, contains input time in the MJDT format 
;*                    then other inputs seconds,day are don't cares.
;*
;* OUTPUT:	 
;*      return-value  The return-value is the seconds since midnight
;*      day           the day of year
;*      theYear       the year
;*
;* KEYWORDS:
;*      DAY, YEAR     set these if you want to force the tzero-value to have
;*                    those values as day or year. The returned value of this
;*                    function will calculated, so that the TZERO and the T90-
;*                    time are the same.
;*                    These keywords were necessary, because some CRRES-orbits
;*                    start on one day and end on the next day. In this case,
;*                    the seconds the orbit's start and end refer to midnight
;*                    of the first day.
;*		      When converting from T90 to TZERO, you can force this 
;*		      by specifying the DAY and YEAR-keywords.
;*		      e.g.: let startTime90, endTime90 be in the T90-format.
;*		            to convert them to TZERO, use this sequence:
;*			    xut1=convert_t90_to_tzero(startTime90, day, year)
;*			    xut2=convert_t90_to_tzero(endTime90, $
;*			                              DAY=day, YEAR=year)
;*
;* CALLING SEQUENCE:
;*      see one paragraph higher...
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;******************************************************************************
FUNCTION convert_T90_TO_TZERO, seconds, day, theYear, $
                               DAY=ForDay, YEAR=ForYear, MJDT=MJDT

   IF NOT keyword_set(MJDT) THEN BEGIN
      numberOfDays=LONG(seconds) / (24L*3600L)
      newSeconds=seconds - numberOfDays*24L*3600L
;    thisday=JULDAY(1,1,1990)+numberOfDays 
      thisday=2447893+numberOfDays
      CALDAT, thisDay, month, dayofmonth, year
      firstDay=JULDAY(1,1,year)
      day=thisDay-firstDay+1
      theYear=year
      IF KEYWORD_SET(ForDay) THEN BEGIN
         IF NOT KEYWORD_SET(ForYear) THEN FYear=year $
                                     ELSE FYear=ForYear
         IF FYear LE 99 THEN FYear=FYear+1900
         diffDays=JULDAY(1,1,Year)+Day-JULDAY(1,1,FYear)-ForDay
         day=ForDay
         theYear=FYear
         newSeconds=newSeconds+24L*3600L*diffDays                  
      ENDIF
      IF TheYear GT 1900 THEN TheYear=TheYear-1900
      RETURN, newSeconds
   ENDIF ELSE BEGIN
      return,MJDT.t
   ENDELSE

END


;******************************************************************************
;* PROCEDURE:     
;*      PRO convert_t90_to_Date, time, year, day_oy, hour, minute, second, $
;*			               month, cmonth, day_om, MJDT=MJDT
;* 
;* DESCRIPTION:  
;*      Convert a T90-time to the time's date to its time of day
;* 
;* INPUTS:       
;*      time      a long number
;*		  This is the time that you wish to convert
;* 
;* KEYWORDS:
;*      MJDT      if set, input time in the MJDT format instead
;*
;* OUTPUTS:
;*      year	  an integer (1990,...2001,...)     the year of the input time 
;*	day_oy	  an integer (1,2,3,...,356)        its day of year
;*
;*	hour	  an integer (0,...,23)         \   these three 
;*	minute	  an integer (0,...,59)          >  variables return
;*	second	  an integer (0,...,59)         /   the time of day
;*
;*      month	  an integer (1,...,12)             the month
;*	cmonth	  a string (e.g. 'Jan' for January) for the month 
;*      day_om	  an integer (1,...,31)	            the day of month
;*
;* CALLING SEQUENCE:
;*      convert_t90_to_Date, convert_timeToSeconds('5.5.1990 11:12:13',/t90), $
;*			     y, doy, h, m, s, m, cm, dom
;*      this assigns y=1990, doy=125, h=11, m=12, s=13, m=5, cm='May', dom=5
;*
;* MODIFICATION HISTORY:       
;*      September 1995, written by A.Keese
;*      January 1997, by R. Friedel, keyword MJDT added 
;******************************************************************************
PRO convert_t90_to_Date, time, year, day_oy, hour, minute, second, $
			       month, cmonth, day_om, MJDT=MJDT 

month_Names  = ['Jan','Feb','Mar','Apr','May','Jun', $
                 'Jul','Aug','Sep','Oct','Nov','Dec']

IF NOT keyword_set(MJDT) THEN BEGIN
    day_90=time / (24L*3600L)
    julian_day=day_90+2447893
    
    ;******Das ist veraendert julian_day -> LONG(...)
    CALDAT, LONG(julian_Day), month, day_om, year
    ;******Ende der Aenderung. H. Korth    
    
    cmonth=month_Names(month-1)
    day_oy=julian_day - JULDAY(1,1,year) + 1
  
    timeZero=time MOD (24L*3600L)
    hour=LONG(timeZero / 3600L)
    minute=LONG((timeZero MOD 3600L) / 60L)
    second=LONG(timeZero MOD 60L)
ENDIF ELSE BEGIN
    jd=time.mjd+2400001
    caldat,jd,month,day_om,year
    cmonth=month_Names(month-1)
    day_oy=jd - JULDAY(1,1,year) +1
    timeZero=time.t
    hour=LONG(timeZero / 3600L)
    minute=LONG((timeZero - hour*3600L) / 60L)
    second=LONG(timeZero MOD 60L)
ENDELSE

END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_date_to_t90, day=day, month=month, doy=doy, $
;*                              year=year, hour=hour, minutes=minutes, $
;*                              seconds=seconds, MJDT=MJDT
;* 
;* DESCRIPTION:  
;*      Convert a time of a given date to the T90-format or MJDT
;* 
;* INPUTS:       
;*      none
;* 
;* KEYWORDS:
;*      You don't have to specify any keyword...
;*
;*      DAY       an integer       if set, this is the day of month. 
;*                                 if omitted, it is assumed to be 1
;*      MONTH     an integer or string
;*                this is the month. You may specify a name as well (either)
;*                in german or in english. Only the first three characters are
;*                significant
;*                If omitted, it is assumed to be 1
;*      DOY       an integer       the day of year. If set, DAY and  MONTH are 
;*                                 ignored. If omitted, it is assumed to be 1
;*      YEAR      an integer       if YEAR LT 1000, 1900 is added
;*                                 If omitted, it is assumed to be 1990
;*      HOUR      an integer       If omitted, it is assumed to be 0
;*      MINUTES   an integer       If omitted, it is assumed to be 0
;*      SECONDS   an integer       If omitted, it is assumed to be 0
;*      MJDT      if set, output time in the MJDT format instead
;*
;* OUTPUT:	 
;*      the seconds since midnight of 1st January 1990 are returned
;*      if keyword MJDT is set, time in MJDT format is returned
;*      (modified Julian Date + seconds since midnight that date
;*
;* CALLING SEQUENCE:
;*      sec=convert_Date_to_T90(day=1, year=1991, hour=12)
;*      sec=convert_Date_to_T90(doy=1, year=1991, hour=12)
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;******************************************************************************
FUNCTION convert_date_to_t90, day=day, month=month, doy=doy, year=year, $
                              hour=hour, minutes=minutes, $
                              seconds=seconds, MJDT=MJDT

  if n_elements(year) eq 0 then year=1990
  if year lt 1000 then year=year+1900 
  IF NOT keyword_set(MJDT) THEN BEGIN
      if year lt 1990 then begin
         print, '------------------ERROR-year GT 1990-----------------------'
         return,-1
      endif
      daysSince1990=0
   ENDIF ELSE BEGIN
      time={MJDT, MJD:0l, T:0.0 }
   ENDELSE

   if n_elements(doy) eq 0 then begin
      theDay=1
      theMonth=1
      if n_elements(day) ne 0 then theDay=day
      if n_elements(month) ne 0 then begin
;    if 'month' is a string, find the number of the month. 
         tmp=size(month)
         if tmp(1) eq 7 then begin
            english_months=['jan', 'feb', 'mar', 'apr', 'may', 'jun', $
                            'jul', 'aug', 'sep', 'oct', 'nov', 'dec']
            ind=where(english_months eq strleft(strlowcase(month),3), count)
            if count gt 0 then begin
               month=ind(0)+1 
            endif else begin
               ger_months=['jan', 'feb', 'mae', 'apr', 'mai', 'jun', $
                           'jul', 'aug', 'sep', 'okt', 'nov', 'dez']
               ind=where(ger_months eq strleft(strlowcase(month),3), count)
               if count gt 0 then month=ind(0)+1 $
               else begin
                  print, '------ ILLEGAL MONTH-name ---------'
                  help, /traceback
                  month=1
               endelse
            endelse 
         endif
         theMonth=month
      endif

      IF NOT keyword_set(MJDT) THEN BEGIN
         daysSince1990=julday(theMonth, theDay, year) - julday(1,1,1990)
      ENDIF ELSE BEGIN
         time.mjd=julday(theMonth, theDay, year) - 2400001l
      ENDELSE

   endif else begin
      if n_elements(day) ne 0 or n_elements(month) ne 0 then begin
         print, '------ ERROR - specified DOY and (MONTH or DAY) -----'
         print, '------ I am using the DOY'
         help, /traceback
      endif
      IF NOT keyword_set(MJDT) THEN $
         daysSince1990=julday(1,1,year) + doy - julday(1,1,1990)-1 ELSE $
         time.mjd=julday(1,1,year) + doy -  2400001 -1
   endelse
   
   if n_elements(hour) eq 0 then hour=0
   if n_elements(minutes) eq 0 then minutes=0
   if n_elements(seconds) eq 0 then seconds=0
   
   IF NOT keyword_set(MJDT) THEN BEGIN
      sec=hour*3600L + minutes*60L+seconds
      return, sec + daysSince1990*24L*3600L 
   ENDIF ELSE BEGIN
      time.t=hour*3600L + minutes*60L+seconds
      return,time
   ENDELSE

end


;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_secondsToTime, seconds, T90=T90, MJDT=MJDT
;* 
;* DESCRIPTION:  
;*      Convert a time to a verbose description. 
;*	If the time is in T90-format, the verbose description will consist
;*	of the date followed by the time of day.
;*	If the time is not in T90-format (i.e., it's seconds since midnight), 
;*	the verbose description will consist of the time of day. In this case, 
;*	the time of day will be preceded by an 'N' if seconds GT 24L*3600L  
;*	(i.e. the time is the time of the next day)
;*
;* INPUTS:       
;*      seconds	     a long number
;*		     a time in the T90 or TZERO-format
;* 
;* KEYWORDS:
;*      T90	     if specified, 'seconds' are interpreted as T90-time.
;*		     Else, 'seconds' are interpreted as TZERO
;*      MJDT         if set, input time in the MJDT format instead
;*      COUNTRY      sets output format to US (1) or EUROPE (0) time style
;*
;* OUTPUT:	 
;*      a string     the string contains the verbose description
;*
;* CALLING SEQUENCE:
;*      ; TZERO - time:
;*	timeString=convert_secondsToTime(seconds)
;*	s1=convert_timeToSeconds(timeString)	
;*      if s1.value NE seconds then print, "this shouldn't happen"
;*	
;*	; T90-time
;*	timeString=convert_secondsToTime(sec, /T90)
;*	s1=convert_timeToSeconds(timeString)	
;*      if s1.value NE seconds then print, "this shouldn't happen"
;*	
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;*      february 1997, by R. Friedel, keyword COUNTRY added
;******************************************************************************
FUNCTION convert_secondsToTime, timein, T90=T90, MJDT=MJDT, COUNTRY=COUNTRY

   time=' 00:00:00'

   IF KEYWORD_SET(T90) THEN BEGIN
      IF keyword_set(MJDT) THEN mjdt=1 else mjdt=0
      convert_t90_to_date, timein, year, d_oy, hour, $
        minutes, sec, month, cmonth, d_om, MJDT=mjdt
      timeH=strtrim(STRING(fix(hour)), 2)
      strput, time, timeH, 3-strlen(timeH)
      timeM=strtrim(STRING(fix(minutes)), 2)
      timeS=strtrim(STRING(fix(sec)), 2)
      strput, time, timeM, 6-strlen(timeM)
      strput, time, timeS, 9-strlen(timeS)
      IF NOT keyword_set(COUNTRY) then COUNTRY=0
      IF COUNTRY eq 0 then begin 
         date=strtrim(string(d_om,format='(i2.2)'),  2) + '.' + $
              strtrim(string(month,format='(i2.2)'),2)+'.'+$
              strtrim(string(year),2)
      ENDIF ELSE BEGIN
         date=strtrim(string(month,format='(i2.2)'),  2) + '/' + $
              strtrim(string(d_om,format='(i2.2)'),2)+'/'+$
              strtrim(string(year),2)
      ENDELSE
      return, date+time
   ENDIF ELSE BEGIN
      IF keyword_set(MJDT) THEN seconds=timein.t else seconds=timein
      timeH=LONG(seconds / 3600L)
      IF timeH GT 24 THEN BEGIN
         timeH=timeH-24
         strput, time, 'N', 0
      ENDIF
      timeH=strtrim(STRING(timeH), 2)
      strput, time, timeH, 3-strlen(timeH)
      timeM=strtrim(STRING(LONG(seconds MOD 3600L) / 60), 2)
      timeS=strtrim(STRING(LONG(seconds MOD 60L)), 2)
      strput, time, timeM, 6-strlen(timeM)
      strput, time, timeS, 9-strlen(timeS)
      return, time
   ENDELSE

END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_timeToSeconds, aTimeString, T90=T90, MJDT=MJDT
;* 
;* DESCRIPTION:  
;*      Convert a verbose time-description to the time in seconds or MJDT.
;* 
;* INPUTS:       
;*      aTimeString	a String
;*			It contains the verbose time in this formats:
;*			If the T90-Keyword is set :
;*			   timeString=date [time
;*			   date=doy"/"y | dom"."mon"."y | mon"/"dom"/"y
;*			   doy=aNumber, dom=aNumber, y=aNumber=mon=aNumber
;*			   (doy:day of year,dom:day of month,mon:month,y:year)
;*			   time=hour":"minutes[":"seconds]
;*			   hour=aNumber, minutes=aNumber, seconds=aNumber
;*			 
;*			   valid examples: 
;*			     aTimeString="1/91 11:12"
;*			     aTimeString="12/30/1990 11:12:31"
;*			     aTimeString="30.12.1990 11:12:"
;*			     aTimeString="1/91"
;*			     aTimeString="1.1.1990 55:12" ; hour may be GT 24 !
;*
;*			   invalid examples:
;*			     aTimeString="30/12/1990 11:12:31"
;*			     aTimeString="12.30.1990 11:12:31"
;*			     aTimeString="11:12:31"
;*
;*			   
;*			If the T90-Keyword is not set:
;*			   timeString=["N"]time
;*			   time=hour":"minutes[":"seconds]
;*			   hour=aNumber, minutes=aNumber, seconds=aNumber
;*
;*			   if N precedes the time, 24 hours will be added to 
;*			   the result - i.e. this is the time of the next day
;*
;*			   valid examples: 
;*			     aTimeString="11:12"
;*			     aTimeString="11:12:31"
;*			     aTimeString="N11:12:31"
;*			     aTimeString="N33:12:31" ; hour may be GT 24 !
;*
;*			   invalid examples: 
;*			     aTimeString="12/30/1990 11:12:31"
;* KEYWORDS:
;*      T90		   if set, the timeString is expected to contain
;*			   date and time. If not set, timeString is expected
;*			   to contain the time only.
;*      MJDT       if set, input time in the MJDT format instead
;*
;* OUTPUT:	 
;*      a structure
;*      return, {err:-1, $
;*	           value:LONG(0), $
;*                 errtext:['text1', 'text2', ...]}
;*      The fields of the returned structure are :
;*	err	   a Boolean (-1 or 0)
;*		   if err EQ -1, timeString did not contain a valid time.
;*			         in this case, 'errtext' contains an 
;*				 explanation of the syntax-error
;*                 if err EQ 0,  timeString was valid. In this case, value 
;*			         contains the time in seconds
;*	value	   a long
;*		   the timeString converted to seconds
;*                 or a MJDT structure if the keyword MJDT is set
;*
;*      errtext	   a string-array
;*		   contains a description of valid formats.
;*
;* CALLING SEQUENCE:
;*	result=convert_timeToSeconds(aString, /T90)
;*	if result.err then begin
;*	   dummy=messagebox(result.errtext, ['Cancel'], title='syntax-error')
;*	   return
;*	endif
;*	print, '-> time-format of your string:', result.value
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;*      january 1997, by R. Friedel, keyword MJDT added
;******************************************************************************
FUNCTION convert_timeToSeconds, aTimeString, T90=T90, MJDT=MJDT
  
  IF NOT KEYWORD_SET(T90) THEN T90=0
  IF keyword_set(MJDT) THEN BEGIN
    mjdt=1 
    time={MJDT, MJD:0l, T:0.0 }
  ENDIF ELSE BEGIN 
    mjdt=0
    time=0l
  ENDELSE
      
  IF KEYWORD_SET(T90) THEN $
    retVal={err:-1, value:time, $
        errtext:['Enter a time in one of these formats:', $
                 'date time, date', $
                 'where time is in one of those formats:', $
                 'h:m:s, h:m', $
                 'and date is in one of those formats:', $
                 'm/d/y, doy/y, d.m.y, m/d/y, yyyymmdd', $
                 'where d is day and doy is day of year, h:m:s is the time', $
                 '(year > 1990 if you use T90 format)', $
                 'e.g: 1/90 14:11, 1/1990 1:1:1, 350/1991 01:00:03', $
                 '     15.1.90 14:11, 1/15/1990 15:3:1']} $
  ELSE retVal={err:-1, value:time, $
        errtext:['Enter a time in one of these formats:', $
                 'h:m:s', 'h:m','for a time of the next day use N as prefix', $
	         'e.g: 14:11  1:1:1  01:00:03', $
	         '    N14:11 N1:1:1 N01:00:03']}

  addToTime=LONG(0)            ; this variable contains the seconds that have
                               ; to be added to seconds specified by the
                               ; time-part of the string in order to get the
                               ; seconds for the correct day.

  timeString=strtrim(aTimeString, 2)+' '

;--- parse a T90-string -------------------------------------------------------
  IF KEYWORD_SET(T90) THEN BEGIN   
    sarr=strarr(3)            ; the first words of the string
    n_s=0                     ; the number of words in the date-part 
    d_type=-1                 ; the type of date - 
                              ;     if the words in 'strarr' were seperated
                              ;        by '.', or d_type=0
                              ;        by '/', d_type=1
                              ;        by somehing else : d_type=-1

    ; find the first words of 'timeString' and extract them into 'sarr'
    end_reached=0
    FOR i=0, 2 DO BEGIN
      ind1=strpos(timeString, '/')
      IF ind1 EQ -1 THEN BEGIN
         ind1=strpos(timeString, '.')
         IF ind1 GT -1 THEN BEGIN
            IF d_type EQ 1 THEN $ ; in this case, the date-part is something
              return, retVal      ; like this: '1/5.1990' => produce error
            d_type=0
         ENDIF 
      ENDIF ELSE BEGIN
         IF d_type EQ 0 THEN $    ; in this case, the date-part is something
           return, retVal         ; like this: '1.5/1990' => produce error
         d_type=1
      ENDELSE 
      
      if not end_reached then begin
         IF ind1 EQ -1 THEN BEGIN ; timeString contains neither '.' nor '/'
            IF (d_type EQ 0 AND n_s EQ 2) OR (d_type EQ 1 AND n_s GE 1) THEN $
              ind1=strpos(timeString, ' ')
              ; find the end of the date by searching for ' '. Some lines 
              ; before, timeString=strtrim(aTimeString, 2)+' ' was executed.
              ; so, ind1 contains the last position of aTimeString or the 
              ; index where the time-part begins
            end_reached=1
         ENDIF 

         IF ind1 GT -1 THEN BEGIN ; found a word...
            val=strtrim(strmid(timeString, 0, ind1), 2)
            IF val NE '' THEN BEGIN
               sarr(n_s)=val ; store the n_s-th part.
               timeString=strtrim(strfrom(timeString, ind1+1), 2)+' '
               IF NOT isDigits(sarr(n_s)) THEN RETURN, retVal
               n_s=n_s+1
            ENDIF
        ENDIF
      endif
    ENDFOR  ; end parsing the first words of 'timeString'
    
    ;now check for first word format of type yyyymmdd
    result=str_sep(strtrim(aTimeString,2), ' ')
    pos=strpos(aTimeString,'/')  ; ddd/yyyy format is also 8 characters long..
    if pos ne -1 then goto, skip1
    if strlen(result(0)) eq 8 then begin
      sarr(2)=strmid(result(0),0,4)
      sarr(1)=strmid(result(0),4,2)
      sarr(0)=strmid(result(0),6,2)
      d_type=0  &  n_s=3  &  end_reached=1  
      IF n_elements(result) EQ 2 THEN TimeString= result(1) ELSE TimeString = '' ;<- A. Aasnes, put in time, if available
    endif  
    skip1:
        
    ; now, sarr contains the first n_s words of 'aTimeString', and timeString
    ; does not contain the date any more 
    IF n_s EQ 2 THEN BEGIN           ; 'day/year' - format
      day=long(sarr(0))
      year=long(sarr(1))
    ENDIF ELSE BEGIN
      IF n_s EQ 3 THEN BEGIN         ; 'd.m.y' - format
        IF d_type EQ 1 THEN BEGIN    ;english ?
          day_om=long(sarr(1))
          month=long(sarr(0))
        ENDIF ELSE BEGIN
          day_om=long(sarr(0))
          month=long(sarr(1))
        ENDELSE
        year=long(sarr(2))
        day=julday(month, day_om, year)-julday(1,1,year)+1
      ENDIF ELSE return, retVal
    ENDELSE

    day=LONG(day)
    year=LONG(year)

    IF year LE 99 THEN year=year+1900
    IF NOT keyword_set(MJDT) then IF year LT 1990 THEN RETURN, retVal

;--- parse a TZERO - string ---------------------------------------------------
   ENDIF ELSE BEGIN 
     IF strleft(timeString, 1) EQ 'n' OR strleft(timeString, 1) EQ 'N' $
       THEN BEGIN
       addToTime=LONG(24L*3600L)
       timeString=strfrom(timeString, 1)
     ENDIF 
   ENDELSE

;--- now, the format-specific part of the string is stripped, and the
;    string should consist only of a time.
   timeString=strtrim(timeString, 2)
   if timeString EQ '' THEN BEGIN ; no time specified
     hour=0L
     minutes=0L
     seconds=0L
   ENDIF ELSE BEGIN
     ind1=strpos(timeString, ':')           
     ind2=strpos(timeString, ':', ind1+1)
     IF ind1 EQ -1 THEN RETURN, retVal
      
     hour=strtrim(strmid(timeString, 0, ind1), 2)
     IF NOT isDigits(hour) THEN return, retVal

     IF ind2 EQ -1 THEN ind2=strlen(timeString)+1 
     minutes=strtrim(strmid(timeString, ind1+1, ind2-ind1-1))
     IF NOT isDigits(minutes) THEN return, retVal

     seconds='0'
     IF ind2 LT strlen(timeString)+1 THEN $
       seconds=strtrim(strmid(timeString, ind2+1, strlen(timeString)-ind2))
     IF NOT isDigits(seconds) THEN return, retVal
  ENDELSE

  retVal.err=0

; maybe, not all of the long-conversions are needed... but I want
; to go sure.
  IF T90 NE 0 THEN BEGIN
    retVal.value=convert_TZERO_TO_T90($
  	LONG(hour)*LONG(3600)+LONG(minutes)*LONG(60)+$
        LONG(seconds), day, year, MJDT=mjdt)
  ENDIF ELSE BEGIN
    retVal.value=LONG(LONG(hour)*LONG(3600)+$
                      LONG(minutes)*LONG(60)+$
                      LONG(seconds)+LONG(addToTime))
  ENDELSE
  retVal.errtext=['ok']
  return, retVal
END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION convert_dayToDate, dayNumber, year
;* 
;* DESCRIPTION:  
;*      convert a given date (in the format 'dayofyear/year') to a 
;*      verbose description
;*
;* INPUTS:       
;*      dayNumber	integer - day of year
;*      year		integer	- year
;*			this is the date to be converted
;*
;* OUTPUT:	 
;*      a string containing a verbose description of this date in the 
;*	format 'day.month.year'
;*
;* CALLING SEQUENCE:
;*      print, convert_dayToDate(100, 1990)
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;******************************************************************************
FUNCTION convert_dayToDate, dayNumber, year

    if year LT 100 then year=year+1900
    firstDayOfYear=julday(1,1, year)
    julianDay=firstDayOfYear + dayNumber - 1
    caldat, julianDay, m, d, y
    date=string(d, m, y, format='(I2.2,".",I2.2,".",I4.4)')
    return, date

END

;******************************************************************************
;* FUNCTION:     
;*      FUNCTION  convert_dateToDay, dateString
;* 
;* DESCRIPTION:  
;*      convert a verbose date-description to the format day_ofYear, year
;*
;* INPUTS:       
;*      dateString	a string containing the date
;*			   date=doy"/"y | dom"."mon"."y
;*			   doy=aNumber, dom=aNumber, y=aNumber, 
;*			   mon=aNumber | aString
;*			   (doy:day of year,dom:day of month,mon:month,y:year)
;*
;* OUTPUT:	 
;*     a structure is returned  err={error:1, day:0, year:0}
;*     fields:
;*     error   a boolean integer (0 or 1)
;*	       if EQ 1, the date-string was not valid
;*	       if EQ 0, the date-string was valid and day,year contain the date
;*
;*     day     an integer
;*	       the day of year of the time-string
;*     year    an integer
;*	       the year of the converted date
;*
;* CALLING SEQUENCE:
;*      a=convert_dateToDay(aString)
;*	if a.err then $
;*	   print, 'ERROR !' $
;*	else $ 
;*	   print, 'day:', a.day, ' year:', a.year
;*
;* MODIFICATION HISTORY:       
;*      september 1995, written by A.Keese 
;******************************************************************************
FUNCTION convert_dateToDay, dateString

  months=['jan', 'feb', 'mar', 'apr', 'may', 'jun', $
          'jul', 'aug', 'sep', 'oct', 'nov', 'dec']

  ind1=strpos(dateString, '/')
  ind2=strpos(dateString, '/', ind1+1)
  IF ind1 EQ -1 THEN BEGIN
      ind1=strpos(dateString, '.')
      ind2=strpos(dateString, '.', ind1+1)
  ENDIF
  err={error:1, day:0, year:0}
  IF ind1 EQ -1 THEN return, err
  day=strtrim(strmid(dateString, 0, ind1), 2)
  IF NOT isDigits(day) THEN return, err

  month=strtrim(strmid(dateString, ind1+1, ind2-ind1-1))
  IF NOT isDigits(month) THEN BEGIN
       i1=WHERE(months EQ month, count)
       IF count GT 0 THEN month=i1(0)+1 $
       ELSE return, err
  ENDIF ELSE month=FIX(month)

  year=strtrim(strfrom(dateString, ind2+1), 2)
  IF NOT isDigits(year) THEN return, err
  year=fix(year)
  IF year LE 100 THEN year=1900+year

  firstDayOfYear = julday(1, 1, year)
  d=julday(month, day, year) - firstDayOfYear + 1

  return, {error:0, day:d, year:year}

END

;******************************************************************************
FUNCTION addSubTime, time1, time2, ADD=ADD, SUBTRACT=SUBTRACT
  
  IF KEYWORD_SET(ADD) THEN BEGIN
      result=time1
      result.mjd=time1.mjd+time2.mjd
      seconds=time1.t+time2.t
      day=long(seconds) / 86400l
      seconds=seconds mod 86400
      result.mjd=result.mjd+day
      case 1 of
        (seconds GE 86400): begin
          result.t=seconds-86400
          result.mjd=result.mjd+1
        end
        (seconds LT 0): begin
          result.t=86400-seconds
          result.mjd=result.mjd-1  
        end
        else:  result.t=seconds
      endcase    
  ENDIF

  IF KEYWORD_SET(SUBTRACT) THEN BEGIN
      result=time1
      result.mjd=time1.mjd-time2.mjd
      seconds=time1.t-time2.t
      day=long(seconds) / 86400l
      seconds=seconds mod 86400
      result.mjd=result.mjd+day
      case 1 of
        (seconds GE 86400): begin
          result.t=seconds-86400
          result.mjd=result.mjd+1
        end
        (seconds LT 0): begin
          result.t=86400+seconds
          result.mjd=result.mjd-1  
        end
        else:  result.t=seconds
      endcase         
  ENDIF
    
  RETURN, result
  
END
