;******************************************************************************
;******************************************************************************
;* FILE:	 
;*      messagebox.pro
;* 
;* DESCRIPTION:  
;*      A window containing a text and buttons is displayed modally.
;*      The index of the pressed button is returned
;*
;* OVERVIEW:
;*      PRO messageBox_event, event
;*	FUNCTION MessageBox, strArray, buttons, CENTER=center
;*	
;* NEEDS:
;*      crres_util
;*
;* COMMON VARIABLES:
;*        COMMON MessageBox, buttonSelected
;*	  	 this COMMON-block is used to return the selected button
;*		 from the event-handler to the messageBox-function
;*               It must not be used by other programs.
;*
;* MODIFICATION HISTORY:       
;*     august 1995, written by A.Keese 
;*     February 1997, modified by R. Friedel to work IDL 4.x / 5.x
;******************************************************************************
;******************************************************************************

;******************************************************************************
;* PROCEDURE:     
;*      messageBox_event
;* 
;* DESCRIPTION:  
;*      this is the event-handler. It is called by XMANAGER only
;* 
;* INPUTS:       
;*      event 	: the event to be handled
;* 
;* OUTPUT:	 
;*      none
;*
;* CALLING SEQUENCE:
;*      this procedure is to be called by XMANAGER only
;*
;******************************************************************************
PRO messageBox_event, event

COMMON MessageBox, buttonSelected

WIDGET_CONTROL, Event.Id, GET_UVALUE=Ev

; check, if the event was initiated by pressing a button
IF strlen(ev) GT 3 THEN $
    IF strmid(ev, 0, 3) EQ 'pb_' THEN BEGIN
        nr=fix(strfrom(ev,3))
        buttonSelected=nr
        WIDGET_CONTROL, Event.top, /DESTROY
    ENDIF

END

;******************************************************************************
;* FUNCTION:     
;*       FUNCTION messageBox, strArray, buttons, CENTER=center
;* 
;* DESCRIPTION:  
;*      A window containing a text and buttons is displayed modally.
;*      The index of the pressed button is returned
;*
;* INPUTS:       
;*	strArray:   An array containing the text to be displayed
;*	buttons :   a string-array containing the buttonlabels
;*
;* KEYWORDS:
;*      SCROL   :   if set, contains lines and scoll
;*	CENTER  :   If set, the displayed text is horizontally centered 
;*      TITLE   :   The title of the message-box
;*
;* OUTPUT:	 
;*      -1	:   if no button was selected
;*      Integer :   if a button was selected, its index is returned 
;*		    the first button has index 0
;*
;* CALLING EXAMPLE:
;*      IF MessageBox(['Exit','Program'], ['Yes', 'No'],$
;*                      TITLE='Exit Message', /CENTER) EQ 0 $
;*         THEN EXIT
;*
;******************************************************************************
FUNCTION messageBox, strArray, buttons, $
                     CENTER=center, TITLE=title, SCROL=scrol, $
                     XPOS=xpos, YPOS=ypos, GROUP_LEADER=group_leader

COMMON MessageBox, buttonSelected
COMMON PLOT_COMPOSER, widgetData

IF keyword_set(GROUP_LEADER) THEN base=group_leader ELSE $
  IF keyword_set(widgetdata) THEN base=widgetdata.base ELSE base = 0 

IF NOT KEYWORD_SET(TITLE) THEN title='Message'   

base = papco_widget_base(COLUMN=1, TITLE=title, $
                         GROUP_LEADER=base, UVALUE='Message Box')

width=60 > max(strlen(strArray))
buttonAnzahl=N_ELEMENTS(buttons)
buttonWidth=max(strlen(buttons))+2
buttonWidthTotal=buttonAnzahl*(buttonWidth)
IF buttonAnzahl GT 1 THEN $
    buttonGap=(width-buttonWidthTotal) / (buttonAnzahl-1) + 1 $
ELSE BEGIN
    buttonGap=(width-buttonWidthTotal) / 2 
    width=width > (buttonWidthTotal+2*buttonGap)
ENDELSE
width=width > (buttonWidthTotal + buttonGap*(buttonAnzahl-1))

IF KEYWORD_SET(CENTER) THEN BEGIN
    strings=strarr(N_ELEMENTS(strArray))
    FOR i=0, N_ELEMENTS(strArray)-1 DO $
        strings(i)=strcenter(strArray(i), width)
ENDIF ELSE BEGIN
   strings=strArray
ENDELSE
 
IF keyword_set(SCROL) THEN BEGIN
    if scrol eq 1 then ysize = 20 else ysize = scrol
    scrol = 1
ENDIF ELSE BEGIN
    scrol = 0
    ysize = N_ELEMENTS(strings)
ENDELSE    

str = WIDGET_TEXT(base, VALUE = strings, SCROLL = scrol, $
                  YSIZE = ysize, XSIZE = width)

butBase=WIDGET_BASE(base, /row)
FOR i=0, buttonAnzahl-1 DO BEGIN
   IF i GT 0 OR buttonAnzahl EQ 1 THEN $
     dummy=WIDGET_LABEL(butBase, VALUE=strcenter('', buttonGap))

   dummy=WIDGET_BUTTON(butBase, $
	               VALUE=strcenter(buttons(i), buttonWidth), $
	               UVALUE='pb_'+strtrim(string(i), 2))
ENDFOR
    
IF buttonAnzahl EQ 1 THEN $
 dummy=WIDGET_LABEL(butBase, VALUE=strcenter('', buttonGap))

geo=WIDGET_INFO(base, /GEOMETRY)
x=600 - geo.xsize/2 & y=300 - geo.ysize/2

if n_elements(xpos) gt 0 then x=xpos
if n_elements(ypos) gt 0 then y=ypos

;sm note: under Mac OS X, if you specify an offset before a pop-up widget
;is realized, the buttons on it don't generate events.
WIDGET_CONTROL, base, /REALIZE, MAP=0              ;sm
WIDGET_CONTROL, base, XOFFSET=x, YOFFSET=y, MAP=1  ;sm

buttonSelected=-1
papco_xmanager, 'messageBox', base
return, buttonSelected 

END 




