;******************************************************************************
;* DESCRIPTION:  
;*      General routine to calculate mag coords and values from models
;*      using the onera fortran library 
;* 
;* INPUTS:       
;*      tai         time in TAI (int. atomic time) format    (can be array)
;*      pos         structure containing position            (can be array)
;*      kcompstr    component format: SPH (spherical), CAR (cartesian)
;*      kinstr      coordinate system of pos: GED, GEO, GEI, MAG, SM, GSM, GSE
;*      
;* KEYWORDS:
;*      -----------------------------------------------------------------------
;*      EXTMOD      external field model index
;*                      0: none
;*                      1: Mead & Fairfield [1975]
;*                      2: Tsyganenko short [1987]
;*                      3: Tsyganenko long [1987]
;*                      4: Tsyganenko [1989c]
;*                      5: Olson & Pfitzer quiet [1977]             (default)
;*                      6: Olson & Pfitzer dynamic [1988]
;*                      7: Tsyganenko [1996]
;*                      8: Ostapenko & Maltsev [1997]
;*                      9: Tsyganenko [2001]
;*                     10: Tsyganenko [2001] storm
;*      -----------------------------------------------------------------------
;*      INTMOD      internal field model index
;*                      0: DGRF/IGRF 1945-2000                      (default)
;*      -----------------------------------------------------------------------
;*      PITCH       pitch angle for which L is calculated (default 90 deg)
;*      KP          kp index value, (default=2)
;*      INT_NAME    returns the name of the internal model
;*      EXT_NAME    returns the name of the external model
;*      KOUTSTR     coordinate system of output mag vector, see kinstr
;*      HAVE_MDATE  time is already in format mdate required by onera
;*      LSTAR       Calculate Roederer L-value, default McIlwain only
;*      -----------------------------------------------------------------------
;*      Inputs for dynamic magnetic field models
;*      -----------------------------------------------------------------------
;*      MAG_DYN      0: use static default values, 1:use OMNI values
;*      -----------------------------------------------------------------------
;*      Any of the following overwrite default or OMNI values
;*      -----------------------------------------------------------------------
;*      KP          single value or array of KP, OMNI format (*10)
;*      DST         single value or array of DST values
;*      PRES        single value or array of solar wind pressure (nP)
;*      DENS        single value or array of solar wind density  (cm-3)
;*      VELO        single value or array of solar wind velocity (lm/s)
;*      IMBX        single value or array of interp. mag field x (nT)
;*      IMBY        single value or array of interp. mag field y (nT)
;*      IMBZ        single value or array of interp. mag field z (nT)
;*      G1          single value or array,used for T01 model
;*      G2          single value or array,used for T01 model
;*      G3          single value or array,used for T01 model
;*      -----------------------------------------------------------------------
;* 
;* OUTPUTS
;*      -----------------------------------------------------------------------
;*      bvec        magnetic field vector in coordinate system B_KOUTSTR
;*                  default  B_KOUTSTR is 'GSM'' - nT
;*      bequ        mag field at equator on field line - nT
;*      Lm          L-value McIlwain
;*      Lstar       L-value Roeder
;*      MLT         Magnetic local time - hour
;*      mlat        Magnetic latitude - deg
;*      error       error status
;*      -----------------------------------------------------------------------
;*
;* CALLING SEQUENCE:
;*      stand-alone
;*
;* MODIFICATION HISTORY:       
;*      June 2004, written by R.Friedel
;******************************************************************************

;******************************************************************************
; G1 is an input paramter used by Tsyganenko [2001] 
; G1=<Vsw*(Bperp/40)^2/(1+Bperp/40)*sin^3(theta/2) > 
;    where the <> mean an average over the previous 1 hour, Vsw is the
;    solar wind speed, Bperp is the transverse IMF component (GSM) and
;    theta it's clock angle.  
FUNCTION get_mag_onera_G1, omni

Vsw   = omni.PLASMA_BULK_SPEED & idx1 = where(Vsw   EQ 1e31, c1)
Bperp = omni.BZ_GSM            & idx2 = where(Bperp EQ 1e31, c2)
;theta = omni.

return, 6

END

;******************************************************************************
; G2 is an input paramter used by Tsyganenko [2001] & Tsyganenko [2001] storm
; G2=< a*Vsw*Bs > 
;    where the <> mean an average over the previous 1 hour, sw is the
;    solar wind speed, Bs=|IMF Bz| when IMF Bz < 0 and Bs=0 when IMF
;    Bz > 0, a=0.005. 
FUNCTION get_mag_onera_G2, omni


return, 10


END

;******************************************************************************
; G3 is an input paramter used by Tsyganenko [2001] storm
; G3=< Vsw*Dsw*Bs /2000.> 
;    where the <> mean an average over the  previous 1 hour, Vsw is
;    the solar wind speed, Dsw is the solar wind  density, Bs=|IMF Bz|
;    when IMF Bz < 0 and Bs=0 when IMF Bz > 0.  
FUNCTION get_mag_onera_G3, omni


return, 10


END

;******************************************************************************

PRO get_mag_onera, tai, pos, kcompstr, kinstr, $                    ; inputs
                   b1, b2, b3, Bmin, Lm, Lstar, MLT, mlat, error, $ ; outputs

                   INTMOD=INTMOD, EXTMOD=EXTMOD,  $
                   INT_NAME=INT_NAME, EXT_NAME=EXT_NAME, $
                   DO_LSTAR = DO_LSTAR, KOUTSTR = KOUTSTR, $
                   MAG_DYN = MAG_DYN, LSTAR_RES = LSTAR_RES, $
                   PITCH=PITCH, VERBOSE=VERBOSE,  HAVE_MDATE = HAVE_MDATE, $
                   DRIFTSHELL = DRIFTSHELL

COMMON get_error, get_err_no, get_err_msg
COMMON omni_data, omni_header, omni
COMMON onera, ext_field_names, onera_ok

;forward_FUNCTION get_mag_onera_g1, get_mag_onera_g2, get_mag_onera_g2

;get onera shareable object location
lib_name = papco_get_libname(/ONERA)

IF lib_name EQ '' THEN BEGIN & get_error = 1 & return & ENDIF

if keyword_set(VERBOSE) then verbose = VERBOSE else verbose = 0     
if keyword_set(INTMOD) then kint = long(INTMOD) else kint = 0l    
if keyword_set(EXTMOD) then kext = long(EXTMOD) else kext = 0l    
IF keyword_set(DO_LSTAR) THEN do_lstar = 1l ELSE do_lstar = 0l
if keyword_set(KOUTSTR) THEN koutstr = KOUTSTR ELSE koutstr = 'GSE'
IF keyword_set(MAG_DYN) THEN mag_dyn = 1 ELSE mag_dyn = 0
IF keyword_set(LSTAR_RES) THEN lstar_res = long(LSTAR_RES) else lstar_res = 0

IF keyword_set(PITCH) THEN pitch = pitch ELSE pitch = [90.0d]
alpha = dblarr(25) & Npa = n_elements(pitch) <  25
alpha(0:Npa-1) = pitch(0:Npa-1)

;make output arrays as needed for number of data points to compute
ntime = n_elements(tai) & mlat = fltarr(ntime)

;set up input position coordinate system
case strupcase(kinstr) OF
    'GED': sysaxes = 0l ;geodetic     (alt/long/lat; km,deg,deg)
    'GEO': sysaxes = 1l ;geographic   (cartesian)
    'GSM': sysaxes = 2l ;geocentric solar magnetospheric
    'GSE': sysaxes = 3l ;geocentric solar ecliptic
    'SM' : sysaxes = 4l ;solar magnetic
    'GEI': sysaxes = 5l ;geocentric equatorial inertial 
    'MAG': sysaxes = 6l ;magnetic coordinates
    ELSE : BEGIN
        message, 'Invalid input coordinate format: '+kinstr, /cont & return
    END
ENDCASE

x1 = dblarr(100000) & x2 = dblarr(100000) & x3 = dblarr(100000) 
x1(0:ntime-1) = transpose(pos[0, *]) 
x2(0:ntime-1) = transpose(pos[1, *]) 
x3(0:ntime-1) = transpose(pos[2, *])
RE = 6371.0d

IF sysaxes GT 0 THEN BEGIN ;convert to R_E
    x1 = x1/RE  &  x2 = x2/RE  &  x3 = x3/RE
ENDIF 

;set up B field output coordinate system
case strupcase(koutstr) OF
    'GEO': kout = 1l ;geographic
    'GEI': kout = 2l ;geocentric equatorial inertial 
    'MAG': kout = 3l ;geomagnetic
    'SM' : kout = 4l ;solar magnetic
    'GSM': kout = 5l ;geocentric solar magnetospheric
    'GSE': kout = 6l ;geocentric solar ecliptic
    ELSE : BEGIN
        message, 'Invalid ouput coordinate format: '+koutstr, /cont & return
    END
ENDCASE

;set up input/output component format
CASE strupcase(kcompstr) OF
    'SPH' : kcomp = 0l ;spherical
    'CAR' : kcomp = 1l ;cartesian
    ELSE : BEGIN
        message, 'Invalid component format: '+kcompstr, /cont & return
    END
ENDCASE 
 
;set up model input parameters. Inputs have to arrays of size 100000
n = 100000
maginput = dblarr(25, n) ;holds all the inputs for mag field models


IF mag_dyn EQ 0 THEN BEGIN ;set default values, not dynamic
    maginput[00, *]  = 2      ;value of Kp as in OMNI2 files 
    maginput[01, *]  = -30    ;Dst index (nT)
    maginput[02, *]  = 25     ;Solar Wind density (cm-3)
    maginput[03, *]  = 300    ;Solar Wind velocity (km/s)
    maginput[04, *]  = 3      ;Solar Wind dynamic pressure (nPa)
    maginput[05, *]  = 0      ;GSM y component of IMF mag. field (nT)
    maginput[06, *]  = 0      ;GSM z component of IMF mag. field (nT)
    maginput[07, *]  = 6      ;G1 for Tsyganenko [2001] 
    maginput[08, *]  = 10     ;G2 for Tsyganenko [2001] & [2001] storm
    maginput[09, *]  = 10     ;G3 for Tsyganenko [2001] storm
ENDIF ELSE BEGIN                ;read from OMNI data base, dynamic inputs.

    new_plotInfo = papco_getplotinfostruct()
    new_plotinfo.typevector(1) = 0 ;do not update
    r_omni, new_PlotInfo, VERBOSE=VERBOSE 
    IF  get_err_no NE 0 THEN BEGIN
        message, 'Error reading OMNI', /info
        error = 1 & return
    ENDIF 

    ot = omni.tai & nodata = 1e31
    Kp = omni.KP
    maginput[00, 0:ntime-1] = papco_interpol(Kp, ot, tai, NODATA = nodata)
    Dst = omni.DST
    maginput[01, 0:ntime-1] =  papco_interpol(Dst, ot, tai, NODATA = nodata)
    dens = omni.PROTON_DENSITY
    maginput[02, 0:ntime-1] =  papco_interpol(dens, ot, tai, NODATA = nodata)
    velo = omni.PLASMA_BULK_SPEED
    maginput[03, 0:ntime-1] = papco_interpol(velo, ot, tai, NODATA = nodata)
    Pdyn = omni.FLOW_PRESSURE
    maginput[04, 0:ntime-1] = papco_interpol(Pdyn, ot, tai, NODATA = nodata)
    ByIMF = omni.BY_GSM
    maginput[05, 0:ntime-1] = papco_interpol(ByIMF, ot, tai, NODATA = nodata)
    BzIMF = omni.BZ_GSM
    maginput[06, 0:ntime-1] = papco_interpol(BzIMF, ot, tai, NODATA = nodata)
    G1 = get_mag_onera_G1(omni)
    G2 = get_mag_onera_G2(omni)
    G3 = get_mag_onera_G3(omni)

ENDELSE 
 
;check & set from input keywords
IF keyword_set(KP)   THEN kp[*]   = kp   
IF keyword_set(DST)  THEN dst[*]  = dst
IF keyword_set(PRES) THEN pres[*] = pres
IF keyword_set(DENS) THEN dens[*] = dens
IF keyword_set(VELO) THEN velo[*] = velo
IF keyword_set(IMBY) THEN imby[*] = imby 
IF keyword_set(IMBZ) THEN imbz[*] = imbz
IF keyword_set(G1)   THEN g1[*]   = g1
IF keyword_set(G2)   THEN g2[*]   = g2
IF keyword_set(G3)   THEN g3[*]   = g3

;make time variable arrays
iyear = lonarr(n) & idoy = lonarr(n) & ut = dblarr(n)
FOR i = 0l, ntime-1 DO BEGIN
    utc = tai2utc(tai(i), /external)
    iyear(i) = long(utc.year)
    idoy(i) = utc2doy(utc)
    ut(i) = utc.hour*3600.0d + utc.minute*60.0d + utc.second
ENDFOR

;declare return values
Lm = dblarr(n) &  Lstar = dblarr(n, 25) &  Blocal = dblarr(n, 25) 
Bmin = dblarr(n) &  XJ = dblarr(n, 25) &  MLT = dblarr(n)
b1 = dblarr(n) & b2 = dblarr(n) & b3 = dblarr(n)

;set control options for make_lstar_shell_splitting
options = lonarr(5)
options(0) = do_lstar   ;turn lstar calc on/off
options(1) = 0          ;update IGRF once per year
options(2) = 1          ;lstar # field lines to trace
;test runs show little improvement in accuracy for higher values. 
options(3) = lstar_res   ;lstar # of steps to trace field line 

message,'lstar_res: '+varprt(lstar_res),/info

;call onera library "make_lstar" routine.

result = call_external(lib_name, 'make_lstar_shell_splitting_', $ 
                       ntime, Npa, kext, options, sysaxes, iyear, idoy, ut, $
                       x1, x2, x3, alpha, maginput, $
                       Lm, Lstar, Blocal, Bmin, XJ, MLT, $ ;return values
                       /f_value)

;call to onera library to get mag field values.
bgeo = dblarr(3) & bl = 0.0d & bvec = dblarr(4, ntime) 

FOR i = 0, ntime-1 DO BEGIN 
bgeo = dblarr(3) & bl = 0.0d
    result = call_external(lib_name, 'get_field_', $

                           kext, options, sysaxes, iyear(i), idoy(i), ut(i), $
                           x1(i), x2(i), x3(i), maginput, $
                           bgeo, bl, $ ;return values
                           /f_value)
    ;convert mag coords geo to gsm
    psi = 0.0d & bgsm = dblarr(3)
    result = call_external(lib_name, 'geo2gsm_', $
                           iyear, idoy, ut, $
                           psi, bgeo, bgsm, $ ;return values
                           /f_value)

    b1(i) = bgsm(0) & b2(i) = bgsm(1) & b3(i) = bgsm(2)
ENDFOR

mlat[*] = -99

;recast all output arrays into actual size used

b1 = b1[0:ntime-1]
b2 = b2[0:ntime-1]
b3 = b3[0:ntime-1]
Bmin = Bmin[0:ntime-1]
Lm = Lm[0:ntime-1]
MLT = MLT[0:ntime-1]
mlat = mlat[0:ntime-1]
IF keyword_set(DO_LSTAR) THEN Lstar = Lstar(0:ntime-1, *)


END

