;*****************************************************************************
;* papco_hilmer_phase_space_density.pro
;*
;* HILMER  PSD CALCULATION
;* here we take values min max values of energy channels and calculate
;* p^2c^2 as in Hilmer et al., Oct 2000 JGR
;*
;*INPUTS:
;*  low_energy_channel_values                   an array containing
;*                                   values of low energy bins (keV)   
;*
;*  top_energy_channel_values                   an array containing
;*                                  values of high energy bins (keV)
;*
;*  diff_flux       data in diff number flux (#)(s-1 cm-2 sr-1 keV-1)
;*
;*OUTPUTS:
;*  PSD   phase space density in units of  s^3 km^-6
;* modified July 2003 to add in GEM conversion factor
;* GEM PSD UNITS (c/MeV cm)^3
;*****************************************************************************
PRO papco_hilmer_phase_space_density, low_energy_channel_values, $
                                      top_energy_channel_values, $
                                      diff_flux, PSD,GEM=GEM
;electron_mass = 1.6726e-31;in kg
;light_speed = 2.9979e5; in km/s

REST_ENERGY = 0.511 ;MeV

;assume energy channel arrays are the same size and consequtive
;this assumes energy channel values are in keV and converts to MeV
KMIN = (low_energy_channel_values)/1000.0 
KMAX = (top_energy_channel_values)/1000.0
size_low_energy_channels = n_elements(KMIN) 
size_top_energy_channels = n_elements(KMAX)
size_diff_flux = size(diff_flux)

IF size_low_energy_channels NE size_top_energy_channels THEN $
  message,'energy bins not fully defined' ELSE $
  message,'energy bins defined', /continue
IF size_low_energy_channels NE size_diff_flux(2) THEN $
  message, 'incorrect number of energy channels in data',/cont ELSE $
  message, 'data bin check complete', /CONTINUE

P_2_c_2 = dblarr(size_low_energy_channels)
;sets up PSD array same size as diff_flux
PSD = dblarr(size_diff_flux(1), size_diff_flux(2))

;----this loop uses Hilmer et al equation to define PSD

FOR i = 0.0, size_low_energy_channels-1 DO BEGIN
    P_2_c_2(i) = (1/2.0)*(KMIN(i)*(KMIN(i)+(2*REST_ENERGY))+KMAX(i)*(KMAX(i)+(2.0*REST_ENERGY)))
    PSD(*, i) =(1.66*1e-10)*diff_flux(*, i)/P_2_c_2(i)
END 

;*******GEM**************************
if keyword_set(GEM) then begin
;USING CORRECTED HILMER-GEM conversion factor, many thanks to Yue Chen!

PSD=PSD*201.48

endif
;*******GEM**************************


;k_min=1.6
;k_max=3.2
;stop
;new_p2c2=0.5*( (k_min)*(k_min + (2*REST_ENERGY))  +(K_max)*(k_max+ (2*rest_energy)))
;L=4.2
;b_hilmer=(0.317/(L^3))
;mu_hilmer_paper=p_2_c_2/(b_hilmer*2*rest_energy)
;new_PSD=(1.66*1e-10)*diff_flux(*, 3)/new_p2c2
;new_hilmer=PSD()*(p_2_c_2(3))/new_p2c2
;PSD is in units of  s^3 km^-6

END
