;*****************************************************************************
; This file is part of the PAPCO software package. 
;
; PAPCO is a free IDL package to combine data plots written and
; maintained by A. Keese and R. Friedel.
;                                                           
;    For more information, check the papco home page at     
;    http://leadbelly/ccr/software/papco/papco.html       
;    for online Manual and downloadble latest version.      
;                                                           
; Comments and questions can be directed to friedel@lanl.gov  
;  
; You are free to modify and distribute this file. However, and
; changes not communicated to R. Friedel will not become part of the
; next release of PAPCO.
;*****************************************************************************
;+
; NAME:
;     PAPCO_MAKE_NEW_MODULE
;
; PURPOSE:
;     Utility program for papco developers - makes a "new" module directory
;     structure. Takes the modules template plot_type and copies it to the 
;     newly created module directory. Then globally changes all occurrences
;     of "plot_type" in both files and file names to the name of the module.
;     By default the new tree is created under the papco_modules directory
;     in the distribution.
;
; CATEGORY:
;     Developers utility
;
; CALLING SEQUENCE:
;
;     PAPCO_MAKE_NEW_MODULE, Name
;
; INPUTS:
;     Name    The new module name
;
; OPTIONAL INPUTS:
;     Parm2:  Describe optional inputs here. If you don't have any, just
;     	      delete this section.
;     
; KEYWORD PARAMETERS:
;                VAR_TYPE   ACTION                         DEFAULT
;  MODULE_PATH:  string     location of new module.        location is   
;                           If = 'own', location           $PAPCO_MODULES
;                           is $PAPCO_HOME
;                               /papco_user_modules
;         INFO:  integer    Calls doc_library to display   0 (no info output)
;                           this header documentation.
;                           1: print to terminal, 
;                           2: print to default printer
;
; OUTPUTS:
;     none. New module constructed.
;
; EXAMPLE:
;     papco_make_new_module, 'crres_meb', MODULE_PATH='own'
;
; MODIFICATION HISTORY:
;     Written by:  August 1998, Reiner Friedel
;-
;*****************************************************************************
pro papco_make_new_module, name, INFO = INFO, MODULE_PATH = MODULE_PATH
  
  IF keyword_set(INFO) THEN BEGIN
      IF info EQ 2 THEN print = 1 ELSE print = 0
      doc_library, 'papco_make_new_module', print = print
      return
  ENDIF
  
  print,''
  message,'Making new module: '+name,/cont
  print,''
    
  path=papco_getenv('PAPCO_MODULES')
  name=strlowcase(name)  &  up_name=strupcase(name)

  if keyword_set(MODULE_PATH) then begin
    if MODULE_PATH eq 'own' then begin     ;set path to papco_user directory
      own_path=papco_getenv('PAPCO_HOME')
      new_module=papco_addpath(own_path,'papco_user_modules')
      MODULE_PATH=new_module
      new_module=papco_addpath(new_module,strlowcase(name))
    endif else begin
      new_module=papco_addpath(papco_checkdirname(MODULE_PATH), $
                               strlowcase(name)) 
    endelse  
  endif else new_module=papco_addpath(path,strlowcase(name))
   
  tmp_module=papco_addpath(path,'plot_type')
   
; make new directory structure
  message,'Making module directory structure',/cont
  cmd='mkdir '+ new_module
  print,cmd  &  spawn, cmd
  cmd='mkdir '+ papco_addpath(new_module,'papco_interface')
  print,cmd  &  spawn, cmd
  cmd='mkdir '+ papco_addpath(new_module,'plot_read')
  print,cmd  &  spawn, cmd
  print,''

; copy and substitute module name new init_module.pro
  message,'Writing new module init file',/cont
  cmd='sed s/plot_type/'+ name +'/g ' + tmp_module + $
    'init_module.pro > ' + new_module + 'init_module.pro'
  print,cmd  &  spawn, cmd
  print,''

; copy and substitute papco_interface files to new module
  message,'Writing new module papco_interface files',/cont
  result=findfile(tmp_module+'papco_interface/*')
  pos=strpos(result,'plot_type')
  tmpname=strarr(n_elements(result))
  for i=0,n_elements(result)-1 do tmpname(i)=strmid(result(i),0,pos(i)) + $
    name + strmid(result(i),pos(i)+9,strlen(result(i)))
  pos=strpos(tmpname,'plot_type')
  for i=0,n_elements(tmpname)-1 do begin
    newname=strmid(tmpname(i),0,pos(i)) + name + $
            strmid(tmpname(i),pos(i)+9,strlen(tmpname(i)))
    if keyword_set(MODULE_PATH) then $
      newname=MODULE_PATH+ strmid(newname,strlen(path),strlen(newname))
    cmd='sed s/plot_type/'+name+'/g ' +result(i) +' > ' + newname
    print,cmd  &  spawn, cmd
  endfor
  print,''
   
; copy and substitute plot_read files to new module
  message,'Writing new module plot_read files',/cont
  result=findfile(tmp_module+'plot_read/*.pro')
  pos=strpos(result,'plot_type')
  tmpname=strarr(n_elements(result))
  for i=0,n_elements(result)-1 do $
    tmpname(i)=strmid(result(i),0,pos(i)) + name + $
                strmid(result(i),pos(i)+9,strlen(result(i)))
  pos=strpos(tmpname,'plot_type')
  for i=0,n_elements(tmpname)-1 do begin
    newname=strmid(tmpname(i),0,pos(i)) + name + $
            strmid(tmpname(i),pos(i)+9,strlen(tmpname(i)))
    if keyword_set(MODULE_PATH) then $
      newname=MODULE_PATH+ strmid(newname,strlen(path),strlen(newname))
    cmd='sed s/plot_type/'+name+'/g ' +result(i) +' > ' + newname
    print,cmd  &  spawn, cmd
    cmd='sed s/PLOT_TYPE/'+up_name+'/g ' +newname +' > '+ new_module +'tmp'
    print,cmd  &  spawn, cmd
    cmd='mv '+ new_module +'tmp ' + newname 
    print,cmd  &  spawn, cmd
  endfor
  print,''

; write defaults file for environmental variable settings
  message,'Writing new module defaults.config file',/cont
  fln=new_module+'plot_read/defaults.config'
  openw,unit,fln,/get_lun
  printf,unit,up_name+'_DATA /net/host/data/'+name
  printf,unit,up_name+ $
     '_DATA_FTP_ARCHIVE ftp://user:password@some.host.site/data/'+name
  close,unit
  free_lun,unit 
  print,''

end

;******************************************************************************
;* CALL (add_sub_plot_type):
;*     PRO add_sub_plot_type, module, sub_type, MODULE_PATH=MODULE_PATH
;*
;* DESCRIPTION:
;*     Utility program for papco developers - adds a new sun plot type
;*     to an exisiting module.
;*     This assumes that the original module was either made with
;*     make_new_module or otherwise adheres to all PAPCO conventions.
;*     
;* INPUTS:
;*    module    The exisiting module name, eg. "polar_ceppad"
;*    sub_type  The new subtype (not including module name) eg. "spec"
;*
;* KEYWORDS:     VAR_TYPE  ACTION                       DEFAULT
;*  MODULE_PATH  string    location of new module.      location is   
;*                         If = 'own', location         $PAPCO_MODULES
;*                         is $PAPCO_HOME
;*                              /papco_user_modules
;*     INFO      boolean   print this header and any    Do nothing
;*                         other info for routine
;*   TEMPLATE    string    use this module as template  use plot_type template
;*
;*   PANEL_KIND  string(2) use panel_kind(0) kind of     ['',sub_type]
;*                         panel, renamed to panel_kind(1) 
;*   PLOT_KIND   string(2) use plot_kind(0) kind of      ['line','line']
;*                         plot, renamed to plot_kind(1) 
;*   READ_KIND   string(2) use read_kind(0) kind of      ['',sub_type]
;*                         read, renamed to read_kind(1) 
;*
;*
;* OUTPUT:
;*     none
;*
;* MODIFICATION HISTORY:
;*     written October 1999, Reiner Friedel
;;*****************************************************************************
pro add_sub_plot_type, module, sub_type, MODULE_PATH=MODULE_PATH, INFO=INFO, $ 
                       TEMPLATE=TEMPLATE, $
                       PANEL_KIND=PANEL_KIND, $
                       PLOT_KIND=PLOT_KIND, $
                       READ_KIND=READ_KIND
  
  COMMON module_paths, paths
  
; print info on routine if needed  
  if keyword_set(INFO)then begin
    papco_routine_info,'make_new_module','add_sub_plot_type', $
      ROOT_DIR=getenv('PAPCO_LIB')
    return
  endif  
  
  print,''
  message,'Adding new sub plot type "'+sub_type+'" to module '+module,/cont
  print,''
  
  path=papco_getenv('PAPCO_MODULES')
  name=strlowcase(module)  &  up_name=strupcase(module)

  if keyword_set(MODULE_PATH) then begin
    if MODULE_PATH eq 'own' then begin     ;set path to papco_user directory
      own_path=papco_getenv('PAPCO_HOME')
      new_module=papco_addpath(own_path,'papco_user_modules')
      MODULE_PATH=new_module
      new_module=papco_addpath(new_module,strlowcase(module))
    endif else begin
      new_module=papco_addpath(papco_checkdirname(MODULE_PATH), $
                               strlowcase(name)) 
    endelse  
  endif else new_module=papco_addpath(path,strlowcase(module))
  
  if keyword_set(TEMPLATE) then begin
    result=strpos(paths, template)
    index=where(result ne -1,c)
    if c eq 0 then begin
      message,'Specified template module ' +template+' not found' ,/cont
      return
    endif  
    tmp_module=paths(index(0))
  endif else tmp_module=papco_addpath(path,'plot_type')
  
  message,'Expanding module ',/cont
  print,'  ',new_module
  
  message,'Using template module: ',/cont
  print,'  ',tmp_module
  print,''

;==============================================================================
;first expand the existing modules's papcoadd interface file.
  
  ;Read into buffer for processing.--------------------------------------------
  fln=new_module+'papco_interface/papcoadd_'+module+'.pro'
  openr,u,fln,/get_lun  &  n=0  &  instr=''
  while not eof(u) do begin
    readf,u,instr
    if n eq 0 then a=instr else a=[a,instr]  &  n=n+1
  endwhile  
  close,u  &  free_lun,u
  
  c=0
  ;now add the sub plot type to the header.------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),6)) ne ';* end' do c=c+1
  if c eq n-1 then begin
    message,'Cannot expand header',/cont
    return
  endif else message,'Expanded header',/cont
  new_sub_plot_no=c-1
  new=';*      papcoadd_'+module+'_'+sub_type
  a=[a(0:c-1),new,a(c:n-1)] & n=n+n_elements(new)
  
  ;expand the drop down menu entries.------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),16)) ne '{ cw_pdmenu_s, 2' $
    AND c ne (n-1) do c=c+1
  if c eq n-1 then begin
    message,'Cannot expand drop down menu',/cont
    return
  endif else message,'Expanded drop down menu',/cont 
  
  new="               { CW_PDMENU_S, 2, '"+sub_type+"' } $"
  
  a(c)=strsubst(a(c),'CW_PDMENU_S, 2,','CW_PDMENU_S, 0,')
  a(c)=strsubst(a(c),'} ','},')
  a=[a(0:c),new,a(c+1:n-1)] & n=n+n_elements(new)
  
  ;expand the case statement in addpanel.--------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),11)) ne 'else: begin' $
    AND c ne (n-1) do c=c+1
  if c eq n-1 then begin
    message,'Cannot expand addpanel procedure.',/cont
    return
  endif else message,'Added new addpanel procedure',/cont  
  
  new=[ $
   "    "+varprt(new_sub_plot_no+1)+": BEGIN"  ,$
   "      panelEditor=panelkind+'_"+sub_type+"_panel'"      ,$
   "      defaults=papco_getPlotInfoStruct()"  ,$
   "      defaults.typeVector=[0,0,0,0]"       ,$     
   "      defaults.channel=0"                  ,$          
   "      papco_Add_Plot, panelEditor, panelKind, initialPlotInfo=defaults" ,$ 
   "    end" ]  
  a=[a(0:c-1),new,a(c:n-1)] & n=n+n_elements(new)
  
  ;add another plottable routine.----------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),18)) ne ';*      pro canget' $
    AND c ne (n-1) do c=c+1  
  if c eq n-1 then begin
    message,'Cannot add new plottable routine.',/cont
    return
  endif else message,'Added new plottable routine',/cont   
  
  new=[ $
   ";-----------------------------------------------------------------------",$
   "pro plottable_"+module+"_"+sub_type+", plottable",$
   "",$
   "  plottable=1",$
   "",$
   "END"]
  a=[a(0:c-3),new,a(c-2:n-1)] & n=n+n_elements(new)
  
  ;add another canget routine.-------------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),22)) ne ';*      pro descriptor' $
    AND c ne (n-1) do c=c+1  
  if c eq n-1 then begin
    message,'Cannot add new canget routine.',/cont
    return
  endif else message,'Added new canget routine',/cont  
  
  new=[ $
   ";-----------------------------------------------------------------------",$
   "pro canget_"+module+"_"+sub_type+", plottable",$
   "",$
   "  canget=   1     ; 1 ",$
   "",$
   "END"]
  a=[a(0:c-3),new,a(c-2:n-1)] & n=n+n_elements(new)
  
  ;add another descriptor routine.---------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),18)) ne ';*      pro editor' $
    AND c ne (n-1) do c=c+1  
  if c eq n-1 then begin
    message,'Cannot add new descriptor routine.',/cont
    return
  endif else message,'Added new descriptor routine',/cont       
  
  new=[ $
   ";-----------------------------------------------------------------------",$
   "pro descriptor_"+module+"_"+sub_type+", plotInfo, descriptStr",$
   "",$
   "  descriptStr=descriptStr+'"+sub_type+" plot'",$
   "",$
   "END"]
  a=[a(0:c-3),new,a(c-2:n-1)] & n=n+n_elements(new)
  
  ;add another editor routine.-------------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),16)) ne ';*      pro draw' $
    AND c ne (n-1) do c=c+1  
  if c eq n-1 then begin
    message,'Cannot add new editor routine.',/cont
    return
  endif else message,'Added new editor routine',/cont     
  
  new=[ $
   ";-----------------------------------------------------------------------",$
   "pro editor_"+module+"_"+sub_type+", plotNumber, newPlotData",$
   "",$
   "  COMMON PLOT_COMPOSER, widgetData",$
   "",$
   "  plotInfo=widgetData.plotInfos(plotNumber)",$
   "  newPlotData="+module+"_"+sub_type+"_panel(plotInfo, $",$
   "                              ACTION='Update '+plotInfo.panelKind, $",$
   "                              GROUP=widgetData.main_base)",$
   "END"]
  a=[a(0:c-3),new,a(c-2:n-1)] & n=n+n_elements(new)
    
  ;add another draw routine.-------------------------------------------------
  ;  (at end of file, no searchc required)
  message,'Added new draw routine',/cont     
  new=[ $
   "",$
   ";-----------------------------------------------------------------------",$
   "pro draw_"+module+"_"+sub_type+", panelVector, currPlotInfo, orbitNo, $",$
   "                    get_data_call, slice_type, IsRead, $",$
   "                    OUTPUT=OUTPUT",$
   "",$
   "  COMMON extra, extra_PlotInfo",$
   "",$
   "  get_data_call='r_"+module+"_"+sub_type+"'",$
   "",$
   "  slice_assigned_value=papco_slice_value('USER Defined Slice A')",$
   "",$
   "  slice_type=1+2+4+8+16",$
   "",$
   "  IF NOT IsRead THEN BEGIN  ",$
   "    r_"+module+"_"+sub_type  ,$
   "    IF papco_Draw_CheckResult(currPlotInfo.panelKind,  $",$
   "                              panelVector, orbitNr,    $",$
   "                              currPlotInfo.typeVector, $",$
   "                              currPlotInfo.channel)    $",$
   "    THEN IsRead=1 ELSE IsRead=0",$
   "  ENDIF",$
   "",$
   "  IF IsRead THEN $",$
   "    p_"+module+"_"+sub_type+", panelVector, channel, OUTPUT=OUTPUT, $",$
   "      _EXTRA=extra_PlotInfo",$
   "",$
   "END",$
   ""]
  
  a=[a(0:n-1),new] & n=n+n_elements(new)
  
;write out new papcoadd file
  print,''
  message,'Writing new papcoadd file to papcoadd_'+module+'.pro',/cont
  print,'  Previous papcoadd... will be moved to old_papcoadd...'
  old_fln=new_module+'papco_interface/old_papcoadd_'+module+'.pro'
  result=findfile(old_fln,count=c)
  input='y'
  if c ne 0 then begin
    message,'old_papcoadd_'+module+'.pro exists',/cont
    read,prompt='  overwrite old_papcoadd_'+module+'.pro ? (y/n): ',input
  endif
  if input eq 'y' then spawn,'mv '+fln+' '+old_fln
  
  openw,u,fln,/get_lun
  for i=0,n-1 do printf,u,a(i)
  close,u  &  free_lun,u  
  
;==============================================================================
;now copy and streamedit the panel editor and plot routines needed  
  
  if keyword_set(PANEL_KIND) then panel_kind=PANEL_KIND $
  else panel_kind=['',sub_type]
  if panel_kind(0) ne '' then panel_kind(0)='_'+panel_kind(0)
  if panel_kind(1) eq '' then panel_kind(1)=sub_type
  
  if keyword_set(PLOT_KIND) then plot_kind=PLOT_KIND $
  else plot_kind=['line','line']
  if plot_kind(0) ne '' then plot_kind(0)='_'+plot_kind(0)
  if plot_kind(1) eq '' then plot_kind(1)=sub_type
  
  if keyword_set(READ_KIND) then read_kind=READ_KIND $
  else read_kind=['',sub_type]
  if read_kind(0) ne '' then read_kind(0)='_'+read_kind(0)
  if read_kind(1) eq '' then read_kind(1)=sub_type
  
  ;copy and substitute panel editor file.--------------------------------------
  s_name=template+panel_kind(0)
  source=tmp_module+'papco_interface/'+s_name+'_panel'+'.pro'
  d_name=module+'_'+panel_kind(1)
  dest=new_module+'papco_interface/'+d_name+ '_panel'+'.pro'
  
  print,''
  message,'Panel Source: '+source,/cont
  message,'Panel Dest. : '+dest,/cont

  result=findfile(source,count=c)
  if c eq 0 then begin
    print,''
    message,'Panel Source not fount',/cont
    return
  endif
  result=findfile(dest,count=c)
  if c eq 1 then begin
    print,''
    message,'Panel Routine Dest. already exists',/cont
    goto, do_plot
  endif    
  cmd='sed s/'+s_name+'/'+d_name+'/g ' +source+' > ' + dest
  print,cmd & spawn,cmd
  
  ;copy and substitute plot routine file.--------------------------------------
do_plot:
  s_name=template+plot_kind(0)
  source=tmp_module+'plot_read/p_'+s_name+'.pro'
  d_name=module+'_'+plot_kind(1)
  dest=new_module+'plot_read/p_'+d_name+'.pro'
  
  print,''
  message,'Plot Routine Source: '+source,/cont
  message,'Plot Routine Dest. : '+dest,/cont

  result=findfile(source,count=c)
  if c eq 0 then begin
    print,''
    message,'Plot Routine Source not fount',/cont
    return
  endif
  result=findfile(dest,count=c)
  if c eq 1 then begin
    print,''
    message,'Plot Routine Dest. already exists',/cont
    goto, do_read
  endif  
  
  cmd='sed s/'+s_name+'/'+d_name+'/g ' +source+' > ' + dest
  print,cmd & spawn,cmd
  
  ;copy and substitute read routine file.--------------------------------------
do_read:
  s_name=template+read_kind(0)
  source=tmp_module+'plot_read/r_'+s_name+'.pro'
  d_name=module+'_'+read_kind(1)
  dest=new_module+'plot_read/r_'+d_name+'.pro'
  
  print,''
  message,'Read Routine Source: '+source,/cont
  message,'Read Routine Dest. : '+dest,/cont

  result=findfile(source,count=c)
  if c eq 0 then begin
    print,''
    message,'Read Routine Source not fount',/cont
    return
  endif
  
  result=findfile(dest,count=c)
  if c eq 1 then begin
    print,''
    message,'Read Routine Dest. already exists',/cont
    goto, do_init
  endif  
  
  cmd='sed s/'+s_name+'/'+d_name+'/g ' +source+' > ' + dest
  print,cmd & spawn,cmd    
  
  ;write new init file for updated module.-------------------------------------
do_init:
  source=new_module+'init_module.pro'
  
  print,''
  message,'Init Routine Source: '+source,/cont  
  
 ;Read into buffer for processing.---------------------------------------------
  openr,u,source,/get_lun  &  n=0  &  instr=''
  while not eof(u) do begin
    readf,u,instr
    if n eq 0 then a=instr else a=[a,instr]  &  n=n+1
  endwhile  
  close,u  &  free_lun,u  &  c=0
  
  ;add the new read routine.---------------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),7)) ne '.run r_' do c=c+1
  if c eq n-1 then begin
    message,'Cannot add new read compile',/cont
    return
  endif else message,'Added new read compile',/cont
  new_sub_plot_no=c-1
  new='.run r_'+d_name+'.pro'
  a=[a(0:c),new,a(c+1:n-1)] & n=n+n_elements(new)  
  
  ;add the new plot routine.---------------------------------------------------
  while strlowcase(strleft(strtrim(a(c),2),7)) ne '.run p_' do c=c+1
  if c eq n-1 then begin
    message,'Cannot add new plot compile',/cont
    return
  endif else message,'Added new plot compile',/cont
  new_sub_plot_no=c-1
  new='.run p_'+d_name+'.pro'
  a=[a(0:c),new,a(c+1:n-1)] & n=n+n_elements(new)  
    
  ;add the new panel routine.--------------------------------------------------
  found=0
  while not found do begin
    pos=strpos(a(c),'panel')
    if pos ne -1 then found=1
    c=c+1
  endwhile  
  if c eq n-1 then begin
    message,'Cannot add new panel compile',/cont
    return
  endif else message,'Added new panel compile',/cont
  new_sub_plot_no=c-1
  new='.run '+d_name+ '_panel'+'.pro'
  a=[a(0:c-1),new,a(c:n-1)] & n=n+n_elements(new)    
  
;write out new papcoadd file
  print,''
  message,'Writing new init file to init_module.pro',/cont
  print,'  Previous init_module.pro will be moved to old_init_module.pro'
  old_source=new_module+'old_init_module.pro'
  result=findfile(old_source,count=c)
  input='y'
  if c ne 0 then begin
    message,'old_init_module.pro exists',/cont
    read,prompt='  overwrite ? (y/n): ',input
  endif
  if input eq 'y' then spawn,'mv '+source+' '+old_source
  openw,u,source,/get_lun
  for i=0,n-1 do printf,u,a(i)
  close,u  &  free_lun,u  
  
  print,''
  message,'To compile the new module enter the following:',/cont
  print,  '@'+source
  print,''
  
END





