;******************************************************************************
;* PROCEDURE:     
;*      papco_mu_calculation, low_channel_values, high_channel_values,
;                             mu_value, particle_data, mag_data
;* 
;* DESCRIPTION:  
;*	program takes energy bin information, particle (time vs
;*      channel) and magnetic field data (of same time range as
;*      particle data), and calculates PSD at selected mu value note
;*      (particle data in format (data, channel)) code calls
;*      papco_hilmer_phase_sapce_density to calculate PSD's   
;*                  
;* INPUTS:
;*      low_channel_values:  energy bin top values in keV
;*      high_channel_values: energy bin bottom values in keV
;*      particle_data:       (data, channel) in dnf (cm-2 s-1 sr-1 KeV-1)
;*      mag_data:            mag field Mag., same timebase as particle_data
;*      mu_value:            mu in MeV/G 
;*        
;*              
;* OUTPUTS: PSD_at_constant_mu, array of data same size as input data
;*         (ie one value per time step) containing PSD values
;*         (s3km-6) for constant mu value input
;*   
;* KEYWORDS:
;*      
;*          
;* CALLING SEQUENCE:
;*     
;*
;* MODIFICATION HISTORY:       
;*     written MARCH 2003 Matt Taylor
;*     MAY 2003 realised error in energy at mu not relativistic           
;******************************************************************************
pro papco_mu_calculation, low_channel_values, high_channel_values, $
                          mu_value, particle_data, mag_data, PSD_at_const_mu,GEM=GEM


IF keyword_set(GEM) THEN begin
mu_value=mu_value/100.0
endif

rest_energy=0.511;MeV
;do check to see if channel arrays are same length

IF n_elements(low_channel_values) NE n_elements(high_channel_values) THEN $
   print, 'channel ERROR!!!'

n_channels=n_elements(low_channel_values);number of energy channels to consider
;*      CALCULATE CENTRE BIN VALUES from LOW_ - high_channel_values. assume
;*      log scaling in channels. Array will hold centre bin values

MAX_ENERGY=high_channel_values(n_channels-1)
MIN_ENERGY=low_channel_values(0)

centre_energy = dblarr(n_channels);
;find centre of log channels
centre_energy=10^((alog10(low_channel_values)+alog10(high_channel_values))/2)

;*    now have array of 'new' energy channels ,with energy at
;*    centre of bin or channel
;*    create dummy array containing 'new' energy channels for each timestep

centre_channels=dblarr((n_elements(mag_data)),n_channels)
FOR i = 0.0, (n_elements(mag_data)-1) DO BEGIN
    centre_channels(i,*) = centre_energy
ENDFOR

energy_at_mu_value = fltarr(n_elements(mag_data),2)
energy_index = fltarr(n_elements(mag_data))

;*  Need to check whether mu value equivalent energy is in energy range
;*  of given channels. Do this by finding position of energy channel of
;*  greater energy than energy of mu value. This gives index between
;*  lowest and uppermost energy channel which is written out to energy
;*  index for each time step
;*  if mu equivalent energy is out of range,timestep index is given -1
;*  value

;first solve quadratic solution of p2c2/(2Brest_energy) to get energy
;were kmin=kmax = equivalent energy
;
;mu= <p2c2>/(B*2*Er)  where Er is rest energy
;so energy at mu given by solving <p2c2> for k_at_mu, where setting kmin=kmax=k_at_mu
;<p2c2>=0.5*( k_min*(k_min + (2*rest_energy)) +  k_max*(k_max + (2*rest_energy)) )
;kmin=kmax=k_at_mu 
;<p2c2>= k_at_mu*(k_at_mu + (2*rest_energy))
;(k_at_mu)^2 + 2*rest_energy*k_at_mu 
;!!!!!!!!!!!!!!ENSURE UNITS ARE CONSISTENT!!!!!!!!!!!!!!!!!!!!
;mu in keV /nT , rest energy in MeV
BBB=2*rest_energy*1000
CCC=BBB*mag_data*mu_value
root_man=(BBB^2 -(4*CCC))

FOR i = 0.0,  ((n_elements(mag_data))-1) DO BEGIN

   ; energy_at_mu_value(i) = mu_value*mag_data(i);assuming mu is in keV/nT
energy_at_mu_value(i,0)= ( (-BBB)+ sqrt(BBB^2 -(-4*CCC(i))) )/2.0
energy_at_mu_value(i,1)= ( (-BBB)- sqrt(BBB^2 -(-4*CCC(i))) )/2.0

   index_energy_value = where(energy_at_mu_value(i,0) lt high_channel_values)  
   index_below_min=where(energy_at_mu_value(i,0) lt low_channel_values(0),count_low)
    energy_index(i) = index_energy_value(0)

    IF count_low ne 0 then energy_index(i)=-1
;    IF energy_index(i) < 0 THEN energy_index(i) = -1

ENDFOR


;*  Create array where mualues exist ie index out -1 values
;*  from energy_index.
data_index =  where(energy_index ne -1,c) 

;* if no indices exit then return with arrya of NaN, so no values of
;* mu in given energy range for given time
if c eq 0 then begin
   null_index=where(energy_index lt 0,cc)
   data_index=null_index 
   PSD_at_const_mu=fltarr(n_elements(mag_data))
   PSD_at_const_mu(*)=!values.F_NAN

   print,'NO VALID DATA IN RANGE '+varprt(mu_value)
   return
ENDIF 

data_index_size = size(data_index)
energy_at_mu_value = energy_at_mu_value(data_index);*
;***energy channel index which is first one 'greater' than given mu value array
;* now needto  re-write energy range with extra energy channel
;* representing mu value,this will be usedto interpolate across energy
;* channels to derive PSD value for given mu

energy_index = energy_index(data_index)
new_energy_array = dblarr(data_index_size(1), n_channels+1)


FOR i = 0.0,  ((n_elements(data_index))-1) DO BEGIN
   
 IF energy_at_mu_value(i) gt centre_energy(energy_index(i)) then begin
   new_energy_array(i,0:energy_index(i))=centre_energy(0:energy_index(i))
   new_energy_array(i,energy_index(i)+1)=energy_at_mu_value(i)
    IF (energy_index(i)+1) eq n_channels then begin
       ;**********   do nothing  *********
     ENDIF else begin
       new_energy_array(i,(energy_index(i)+2):n_channels)=$
       centre_energy((energy_index(i)+1):(n_channels-1))
    ENDELSE 
 ENDIF


 IF energy_at_mu_value(i) lt centre_energy(energy_index(i)) then begin
    if energy_index(i) eq 0 then begin
       new_energy_array(i,0)=energy_at_mu_value(i)
       new_energy_array(i,1:n_channels)=$
        centre_energy
    endif else begin
     new_energy_array(i,0:(energy_index(i)-1))=centre_energy(0:(energy_index(i)-1))
     new_energy_array(i,energy_index(i))=energy_at_mu_value(i)
      IF (energy_index(i)+1) eq n_channels then begin
       new_energy_array(i,(energy_index(i)+1))=centre_energy(n_channels-1)
      ENDIF else begin
       new_energy_array(i,(energy_index(i)+1):n_channels)=$
       centre_energy((energy_index(i)):(n_channels-1))

      ENDELSE 
     endelse
 ENDIF
ENDFOR

;* now have new array of energies (number of time steps,energy
;* channels+mu energy) now call PSD calculation. Use lower_channel,
;* upper_channel bin values as above, 

IF keyword_set (GEM) THEN begin
;------------------------------------------------------------------------------
papco_hilmer_phase_space_density, low_channel_values, high_channel_values, $
                                  particle_data, PSD,/GEM
                ;PARTICLE data now in PSD from HILMER (GEM UNITS)
;--------------------------------------------------------------------------
endif else begin papco_hilmer_phase_space_density, low_channel_values, high_channel_values, $
                                  particle_data, PSD
                ;PARTICLE data now in PSD from HILMER (HILMER UNITS)
;--------------------------------------------------------------------------
endelse

PSD_good = PSD(data_index, *);index data in range
final_PSD_at_mu=fltarr(n_elements(PSD(*,0)));setup array to  hold final data
final_PSD_at_mu(*)=!values.F_NAN; set all values to NaN

size_new_data =n_elements(data_index);** number of data points 'in range'
size_new_energy_array = n_channels+1;*** size of new interpolated array ie extra channel representing mu value.
interpol_PSD_data = dblarr(size_new_data,size_new_energy_array)

;* interpolation of original channels to original channels + 1 (centre
;* bins and mu value  -derived energy gives interpol_PSD_data
;* (PSD,energy channels+'mu energy channel') 

FOR i = 0.0, size_new_data-1 DO begin
    interpol_PSD_data(i, *) = $
      interpol(PSD_good(i, *), centre_channels(i, *), new_energy_array(i, *))
ENDFOR

;***now write PSD at constant mu for each time step

flux_at_const_mu=fltarr(size_new_data)
  
for j=0.0, (size_new_data-1) do begin
    flux_at_const_mu(j) = interpol_PSD_data(j,energy_index(j))
ENDFOR 

;****check for zero's
;****set zero PSD values to NAN
flux_mu_zero_index =where(flux_at_const_mu(*,*) le 0,countz)

if countz gt 0 then begin
    flux_at_const_mu(flux_mu_zero_index)=!values.F_NAN 
endif

final_PSD_at_mu(data_index)=flux_at_const_mu

;****   write out PSD at constant mu
PSD_at_const_mu=final_PSD_at_mu




END
