;******************************************************************************
;* File: papco_panel_label
;*
;* A collection of routines to place labels around papco panels in
;*such a way that they scale to fit into the space available for a panel.
;*
;* Aug 2002 - added routine to put labels into panel window,
;*            e.g. colored labels for each line in panel, like energy channels
;******************************************************************************

;******************************************************************************
;*  NAME:        papco_rs_lbl
;*
;*  DESCRIPTION: Puts a label to mark a panel to the right of the panel.
;*               calculates position form system variable !p.position
;*               Does "correct" new line with enough space
;*               Can attempt to autoscale label into available space.
;*
;*  INPUT:       panel : 3 element vector for position of panel
;*               label : string used as label, IDL Format.
;*
;*  KEYWORDS     ROT90 : rotates output string by 90 degrees, used for plots
;*                       with papco_colorbar
;*               AUTO  : 0 no scaling
;*                       1 attempt to scale into space.
;*               LINE  : add a line and plot the current linestyle there 
;*               _extra: extra keywords to be passed to xyouts procedure
;*
;*  OUTPUT:      plots on current plotting device
;*
;*  DATE:        July 2003
;*
;*  AUTOR:       R. Friedel
;*
;*  SIDEEFFECTS: -
;*
;*  MODIFICATIONS: rewrite using buffers for overplots. 
;******************************************************************************
pro papco_rs_lbl, panel, label, ROT90=ROT90, AUTO=AUTO, LINE = LINE, $
                                NOEXTEND = NOEXTEND, _extra=extra_Par
 
;
 
COMMON PLOT_COMPOSER, widgetData
COMMON right_side_1, rsl_arr, rsl_c

IF keyword_set(ROT90) THEN rot = ROT90 ELSE rot = 0

lbl = label
plotno = widgetData.current_plotNr
currPlotInfo=widgetData.plotInfos(plotno)
IF currPlotInfo.descr_extend LT 0 THEN rot = NOT(rot)

; see if a manual label was set for this panel, or inhibit was set
CASE currPlotInfo.descr OF
    0: BEGIN ;use automatically set label
        widgetData.plotInfos(widgetData.current_plotNr).descr_str=lbl
        ;adopt height from set panelheight for auto.
        currPlotInfo.descr_extend = currPlotInfo.PANELHEIGHT
    END
    1: BEGIN ;use user defined label
        lbl = widgetData.plotInfos(widgetData.current_plotNr).descr_str
        result=strpos(lbl,'$$$')
        IF result NE -1 THEN BEGIN
            new_lbl=str_sep(lbl,'$$$')
            lbl=new_lbl(0)+lbl+new_lbl(1)
        ENDIF
    END
    2: BEGIN ;inhibit labels, just do nothing
        lbl = ''
    END
ENDCASE

; extend label over multiple panels. Simply "panelset" a new height.

;keep original panel and graphics settings
panel_orig = panel & extend = abs(currPlotInfo.descr_extend) > panel_orig(2)
plotted_x = !x  &  plotted_y = !y

IF (extend GT 1) AND NOT keyword_set(NOEXTEND) THEN BEGIN 
     panel(2) = extend
     panelset, panel	       ;re-sets the panel position viewport
ENDIF

;keep in a common block here all the panel label entries for a given
;panel, and redraw when new lables are added. 

;make array of structures to be used - for label and color and line
extra = {PSYM:0, SYMSIZE:0, COLOR:0, THICK:0, LINESTYLE:0}
struct_assign, extra_Par, extra
dmy = {extra:extra, label:'', panelnr:0}

;set up the label arrays each time the first plot is called.
IF rsl_c EQ 0 THEN rsl_arr = [dmy] ELSE rsl_arr = [rsl_arr, dmy]

;load up the label array
rsl_arr(rsl_c).panelnr = panel(0)
IF rsl_arr(rsl_c).label EQ '' THEN rsl_arr(rsl_c).label = lbl $
  ELSE rsl_arr(rsl_c).label = rsl_arr(rsl_c).label+'!C'+lbl

;erase existing right side labels; new ones will be redrawn.
IF NOT rot THEN papco_clear_panel, panel, /RIGHT_SIDE

;collect all exisiting labels for this panel only
idx = where(rsl_arr.panelnr EQ panel(0), nl) ;number of labels is nl
IF nl NE 0 THEN this_rsl_arr = rsl_arr(idx) ELSE return 

;make a combined string for all labels set, this is used for sizing
lbl = ''
FOR i = 0, nl-1 DO BEGIN
    IF i EQ 0 THEN lbl = lbl+this_rsl_arr(i).label $
              ELSE lbl = lbl+'!C'+this_rsl_arr(i).label
ENDFOR 

;increment label counter
rsl_c = rsl_c+1

; get character size in normal coordinates  
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
chrd=chrs*!P.CHARSIZE      
    
; get aspect ratio of plot area
aspect=float(!D.x_size)/float(!D.y_size)
  
; divide up input sting into lines
cr='!C'
res=strpos(lbl,cr)        ; check if string contains carriage returns
if res eq -1 then cr='!c'
res=strpos(lbl,cr)        ; check if string contains carriage returns
  
if res ne -1 then lines=str_sep(lbl,cr) else begin
    lines=strarr(1) & lines(0)=lbl
endelse  
  
; get maximum length of a string, ignoring !X formatting characters
dummy=lines & max_char=intarr(n_elements(lines))
for i=0, n_elements(lines)-1 do begin
    ppp=str_sep(dummy(i),'!')
    max_char(i)=strlen(dummy(i))-(n_elements(ppp)-1)*2
endfor  
max_char=max(max_char)
  
; find the longest line and scale to fit (height of panel for rotated
; labels, width to end of page for horizontal labels)
avail_height=(!p.position(3)-!p.position(1))*0.85
  
IF rot THEN BEGIN
    avail_width=0.035
    length=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    height=max_char*(!D.x_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal)
    x_fac=((avail_height)/box(1)) < 1.0 
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_width)/box(0)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    xcr=chrd(1)*fac/aspect & ycr=0 & deg=-90 
    stpos=[!p.position(2)+0.0625-xcr,!p.position(3)]
endif else begin 
    avail_width=0.16
    length=max_char*(!D.x_ch_size)*!P.CHARSIZE
    height=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal) 
    x_fac=((avail_width)/box(0)) < 1.0
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_height)/box(1)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    xcr=0 & ycr=chrd(1)*fac & deg=0
    stpos=[!p.position(2)+0.02,!p.position(3)-ycr]  
ENDELSE 

;now output the labels. Loop through all labels, and plot using their
;own color, using the sizing determined above!
lines_1 = 0

FOR j = 0, nl-1 DO BEGIN
    lbl = this_rsl_arr(j).label
    ; divide up input sting into lines
    cr='!C'
    res=strpos(lbl,cr)        ; check if string contains carriage returns
    if res eq -1 then cr='!c'
    res=strpos(lbl,cr)        ; check if string contains carriage returns
  
    if res ne -1 then lines=str_sep(lbl,cr) else BEGIN
        lines=strarr(1) & lines(0)=lbl
    ENDELSE
    lno = n_elements(lines)
    

    extra = {color:this_rsl_arr(j).extra.color, orientation:deg, $
             charsize:charsize, normal:1}

    ; check if inside or outside of panel is asked for
    IF widgetData.default_right_label EQ 0 THEN begin ;outside
        FOR i=0,lno-1 DO BEGIN 
            xpos = stpos(0)-xcr*(i+lines_1)
            ypos = stpos(1)-ycr*(i+lines_1)
            xyouts, xpos, ypos, lines(i), _extra=extra

            IF keyword_set(LINE) AND i EQ (lno-1) THEN BEGIN
                ;plot a line representing current line style below label
                old_p = !p.position
                !p.position = [xpos, ypos-ycr, 1.0, ypos]
                plot, [0.1, 0.4, 0.7], [0.5, 0.5, 0.5], yr=[0, 1], xr=[0, 1], $
                      xstyle=5, ystyle=5, /normal, _extra=this_rsl_arr(j).extra
                !p.position = old_p
                lines_1 =lines_1+1 
            ENDIF     
        ENDFOR 
    endif else begin              ;inside
        for i=0,n_elements(lines)-1 do $
          xyouts,stpos(0)-xcr*i-0.02,stpos(1)-ycr*i,lines(i), $ 
          alignment=1.0, _extra=extra_Par
    ENDELSE 
    
    lines_1 = lines_1+n_elements(lines)

ENDFOR 

;restore original panel settings
panel = panel_orig
!x = plotted_x  &  !y = plotted_y 
panelset, panel	       ;re-sets the panel position viewport
    
END 

;******************************************************************************
;*  NAME:        right_side_label
;*
;*  DESCRIPTION: Puts a label to mark a panel to the right of the panel.
;*               calculates position form system variable !p.position
;*               Does "correct" new line with enough space
;*               Can attempt to autoscale label into available space.
;*
;*  INPUT:       panel : 3 element vector for position of panel
;*               label : string to be used as label 
;*
;*  KEYWORDS     ROT90 : rotates output string by 90 degrees, used for plots
;*                       with papco_colorbar
;*               POS   : 0 (default) aligned with top of panel
;*                       1           aligned with bottom of panel
;*               AUTO  : 0 no scaling
;*                       1 attempt to scale into space.
;*               _extra: extra keywords to be passed to xyouts procedure
;*
;*  OUTPUT:      plots on current plotting device
;*
;*  DATE:        August 1995
;*
;*  AUTOR:       R. Friedel
;*
;*  SIDEEFFECTS: -
;*
;*  MODIFICATIONS: Jan 1997 -  made charsize controlled by !P.CHARSIZE
;*                             system variable R. Friedel
;*                 Oct 1997 -  Better New Line spacing added
;*                 Jan 1999 -  Keyword POS added
;*                 Feb 1999 -  Keyword AUTO added.
;******************************************************************************
pro right_side_label, panel, label, ROT90=ROT90, POS=POS, AUTO=AUTO, $
                                    NOEXTEND = NOEXTEND, _extra=extra_Par
  
COMMON PLOT_COMPOSER, widgetData
currPlotInfo=widgetData.plotInfos(widgetData.current_plotNr)
  
if keyword_set(POS) then goto, no_manual     
  
; see if a manual label was set for this panel, or inhibit was set
CASE currPlotInfo.descr OF
    0: BEGIN ;use automatically set label
        widgetData.plotInfos(widgetData.current_plotNr).descr_str=label
    END
    1: BEGIN ;use user defined label
        descr_str = widgetData.plotInfos(widgetData.current_plotNr).descr_str
        result=strpos(descr_str,'$$$')
        if result ne -1 then begin
            new_label=str_sep(descr_str,'$$$')
            label=new_label(0)+label+new_label(1)
        endif else label=descr_str
    END
    2: BEGIN ;inhibit labels, just do nothing
        return
    END
ENDCASE
  
no_manual:

; extend label over multiple panels. Simply "panelset" a new height.
panel_orig = panel
IF keyword_set(currPlotInfo.descr_extend) AND $
   NOT keyword_set(NOEXTEND) THEN BEGIN 
    panel(2) = abs(currPlotInfo.descr_extend) > panel_orig(2)
    IF currPlotInfo.descr_extend LE 0 THEN panel(2) = panel_orig(2)
    panelset, panel	       ;re-sets the panel position viewport
ENDIF

; get character size in normal coordinates  
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
chrd=chrs*!P.CHARSIZE      
    
; get aspect ratio of plot area
aspect=float(!D.x_size)/float(!D.y_size)
  
; divide up input sting into lines
cr='!C'
res=strpos(label,cr)        ; check if string contains carriage returns
  if res eq -1 then cr='!c'
  res=strpos(label,cr)        ; check if string contains carriage returns
  
  if res ne -1 then lines=str_sep(label,cr) else begin
    lines=strarr(1) & lines(0)=label
  endelse  
  
; get maximum length of a string, ignoring !X formatting characters
  dummy=lines & max_char=intarr(n_elements(lines))
  for i=0, n_elements(lines)-1 do begin
    ppp=str_sep(dummy(i),'!')
    max_char(i)=strlen(dummy(i))-(n_elements(ppp)-1)*2
  endfor  
  max_char=max(max_char)
  
; find the longest line and scale to fit (height of panel for rotated
; labels, width to end of page for horizontal labels)
  avail_height=(!p.position(3)-!p.position(1))*0.95
  
  if keyword_set(rot90) then begin
    avail_width=0.045
    length=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    height=max_char*(!D.x_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal)
    x_fac=((avail_height)/box(1)) < 1.0 
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_width)/box(0)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    xcr=chrd(1)*fac/aspect & ycr=0 & deg=-90 
    stpos=[!p.position(2)+0.0625-xcr,!p.position(3)]
  endif else begin 
    avail_width=0.16
    length=max_char*(!D.x_ch_size)*!P.CHARSIZE
    height=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal) 
    x_fac=((avail_width)/box(0)) < 1.0
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_height)/box(1)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    if keyword_set(POS) then begin
      xcr=0 & ycr=-chrd(1)*fac & deg=0
      stpos=[!p.position(2)+0.02,!p.position(3)-ycr/5.]
      stpos(1)=stpos(1)-(!p.position(3)-!p.position(1)-0.005)
    endif else begin 
      xcr=0 & ycr=chrd(1)*fac & deg=0
      stpos=[!p.position(2)+0.02,!p.position(3)-ycr]
    endelse  
  endelse
  
  if keyword_set(extra_Par) then begin
    ;only copy over the color tag if it exists
    tags=tag_names(extra_par)
    res=strpos(tags,'COLOR')
    index=where(res ne -1,count)
    if count ne 0 then col=extra_par.color else col=1    
  endif else col=1
  extra_Par={color:col,orientation:deg,charsize:charsize,normal:1}
    
  ; check if inside or outside of panel is asked for
  if widgetData.default_right_label eq 0 then begin ;outside
    for i=0,n_elements(lines)-1 do begin
      xyouts,stpos(0)-xcr*i,stpos(1)-ycr*i,lines(i), _extra=extra_Par
    endfor
  endif else begin              ;inside
    for i=0,n_elements(lines)-1 do $
      xyouts,stpos(0)-xcr*i-0.02,stpos(1)-ycr*i,lines(i), $ 
      alignment=1.0, _extra=extra_Par
  endelse
  
;restore original panel settings
panel = panel_orig
panelset, panel	       ;re-sets the panel position viewport
    
END 


;******************************************************************************
;*  NAME:        papco_colorbar_label
;*
;*  DESCRIPTION: Puts a label to mark a panel to the right of the panel.
;*               calculates position form system variable !p.position
;*               Does "correct" new line with enough space
;*               Can attempt to autoscale label into available space.
;*
;*  INPUT:       panel : 3 element vector for position of panel
;*               label : string to be used as label 
;*
;*  KEYWORDS     ROT90 : rotates output string by 90 degrees, used for plots
;*                       with papco_colorbar
;*               POS   : 0 (default) aligned with top of panel
;*                       1           aligned with bottom of panel
;*               AUTO  : 0 no scaling
;*                       1 attempt to scale into space.
;*               _extra: extra keywords to be passed to xyouts procedure
;*
;*  OUTPUT:      plots on current plotting device
;*
;*  DATE:        August 1995
;*
;*  AUTOR:       R. Friedel
;*
;*  SIDEEFFECTS: -
;*
;*  MODIFICATIONS: Jan 1997 -  made charsize controlled by !P.CHARSIZE
;*                             system variable R. Friedel
;*                 Oct 1997 -  Better New Line spacing added
;*                 Jan 1999 -  Keyword POS added
;*                 Feb 1999 -  Keyword AUTO added.
;******************************************************************************
pro papco_colorbar_label, panel, label, ROT90=ROT90, POS=POS, AUTO=AUTO, $
                                        NOEXTEND = NOEXTEND, _extra=extra_Par
  
COMMON PLOT_COMPOSER, widgetData
currPlotInfo=widgetData.plotInfos(widgetData.current_plotNr)
  
if keyword_set(POS) then goto, no_manual     
  
; see if a manual label was set for this panel, or inhibit was set
CASE currPlotInfo.zlbl OF
    0: BEGIN ;use automatically set label
        widgetData.plotInfos(widgetData.current_plotNr).zlbl_str=label
    END
    1: BEGIN ;use user defined label
        descr_str = widgetData.plotInfos(widgetData.current_plotNr).zlbl_str
        result=strpos(descr_str,'$$$')
        if result ne -1 then begin
            new_label=str_sep(descr_str,'$$$')
            label=new_label(0)+label+new_label(1)
        endif else label=descr_str
    END
    2: BEGIN ;inhibit labels, just do nothing
        return
    END
ENDCASE

no_manual:

; extend label over multiple panels. Simply "panelset" a new height.
panel_orig = panel
IF keyword_set(currPlotInfo.descr_extend) AND $
   NOT keyword_set(NOEXTEND) THEN BEGIN 
    panel(2) = abs(currPlotInfo.descr_extend) > panel_orig(2)
    IF currPlotInfo.descr_extend LE 0 THEN panel(2) = panel_orig(2)
    panelset, panel	       ;re-sets the panel position viewport
ENDIF

; get character size in normal coordinates  
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
chrd=chrs*!P.CHARSIZE      
    
; get aspect ratio of plot area
aspect=float(!D.x_size)/float(!D.y_size)
  
; divide up input sting into lines
cr='!C'
res=strpos(label,cr)        ; check if string contains carriage returns
if res eq -1 then cr='!c'
res=strpos(label,cr)        ; check if string contains carriage returns
  
if res ne -1 then lines=str_sep(label,cr) else begin
    lines=strarr(1) & lines(0)=label 
ENDELSE 
  
; get maximum length of a string, ignoring !X formatting characters
dummy=lines & max_char=intarr(n_elements(lines))
for i=0, n_elements(lines)-1 do begin
    ppp=str_sep(dummy(i),'!')
    max_char(i)=strlen(dummy(i))-(n_elements(ppp)-1)*2
endfor  
max_char=max(max_char)
  
; find the longest line and scale to fit (height of panel for rotated
; labels, width to end of page for horizontal labels)
avail_height=(!p.position(3)-!p.position(1))*0.95
  
if keyword_set(rot90) then begin
    avail_width=0.045
    length=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    height=max_char*(!D.x_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal)
    x_fac=((avail_height)/box(1)) < 1.0 
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_width)/box(0)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    xcr=chrd(1)*fac/aspect & ycr=0 & deg=-90 
    stpos=[!p.position(2)+0.18-xcr,(!p.position(1)+!p.position(3))/2]
endif else begin 
    avail_width=0.16
    length=max_char*(!D.x_ch_size)*!P.CHARSIZE
    height=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal) 
    x_fac=((avail_width)/box(0)) < 1.0
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_height)/box(1)) < 1.0
    fac=y_fac*x_fac
    charsize=!P.CHARSIZE*fac
    if keyword_set(POS) then begin
      xcr=0 & ycr=-chrd(1)*fac & deg=0
      stpos=[!p.position(2)+0.02,!p.position(3)-ycr/5.]
      stpos(1)=stpos(1)-(!p.position(3)-!p.position(1)-0.005)
    endif else begin 
      xcr=0 & ycr=chrd(1)*fac & deg=0
      stpos=[!p.position(2)+0.02,!p.position(3)-ycr]
    endelse  
  endelse
  
  if keyword_set(extra_Par) then begin
    ;only copy over the color tag if it exists
    tags=tag_names(extra_par)
    res=strpos(tags,'COLOR')
    index=where(res ne -1,count)
    if count ne 0 then col=extra_par.color else col=1    
  endif else col=1
  extra_Par={color:col,orientation:deg,charsize:charsize,normal:1}
    
  ; check if inside or outside of panel is asked for
  if widgetData.default_right_label eq 0 then begin ;outside
    for i=0,n_elements(lines)-1 do begin
      xyouts,stpos(0)-xcr*i,stpos(1)-ycr*i,lines(i), $
        alignment=0.5, _extra=extra_Par
    endfor
  endif else begin              ;inside
    for i=0,n_elements(lines)-1 do $
      xyouts,stpos(0)-xcr*i-0.02,stpos(1)-ycr*i,lines(i), $ 
        alignment=1.0, _extra=extra_Par
  endelse
  
;restore original panel settings
panel = panel_orig
panelset, panel	       ;re-sets the panel position viewport
    
END 



;******************************************************************************
;*  NAME:        left_side_label
;*
;*  DESCRIPTION: same as right_side_label but used for labeling the
;*               lefthand y-axis. 
;*               Check for "overplots" - don't label if overplot
;******************************************************************************
pro left_side_label, panel, label, ROT90 = ROT90, POS = POS, AUTO = AUTO, $
                                   _extra=extra_Par
COMMON PLOT_COMPOSER, widgetData
COMMON extra, extra_PlotInfo           ; passes in papco-set keywords structure

; see if a manual label was set for this panel, or inhibit was set
currPlotInfo=widgetData.plotInfos(widgetData.current_plotNr)

CASE currPlotInfo.ylbl OF
    0: BEGIN ;use automatically set label
        widgetData.plotInfos(widgetData.current_plotNr).ylbl_str = label
    END
    1: BEGIN ;use user defined label
         result=strpos(currPlotInfo.ylbl_str,'$$$')
         if result ne -1 then begin
           new_label=str_sep(currPlotInfo.ylbl_str,'$$$')
          label=new_label(0)+label+new_label(1)
         endif else label=currPlotInfo.ylbl_str
         ;check for "instructions" in label string (starts "**" on own line)
         p = strpos(label, '**')
         IF p NE -1 THEN BEGIN
             inst = strmid(label, p+2, strlen(label))
             label = strmid(label, 0, p-2)
             p2 = strpos(inst, '!')
             IF p2 NE 0 THEN inst = strmid(inst, 0, p2)
             CASE strupcase(inst) OF
                 'ROT90':ROT90 = 1
                 'ROT0':ROT90 = 0                 
                 ELSE: message, 'Instruction '+inst+' not known',/infor
             ENDCASE 
         ENDIF 
    END
    2: BEGIN ;inhibit labels, just do nothing
        return
    END
ENDCASE

; Extend label over multiple panels Simply "panelset" a new height.
panel_orig = panel
panel(2) = currPlotInfo.ylbl_extend  > panel_orig(2)
panelset, panel	       ;re-sets the panel position viewport

; get character size in normal coordinates  
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
chrd=chrs*!P.CHARSIZE      
    
; get aspect ratio of plot area
aspect=float(!D.x_size)/float(!D.y_size)
  
; divide up input sting into lines
cr='!C'
res=strpos(label,cr)        ; check if string contains carriage returns
if res eq -1 then cr='!c'
res=strpos(label,cr)        ; check if string contains carriage returns
  
if res ne -1 then lines=str_sep(label,cr) else begin
    lines=strarr(1) & lines(0)=label
endelse  
  
nl = n_elements(lines)

; get maximum length of a string, ignoring !X formatting characters
dummy=lines & max_char=intarr(n_elements(lines))
for i=0, n_elements(lines)-1 do begin
    ppp=str_sep(dummy(i),'!')
    max_char(i)=strlen(dummy(i))-(n_elements(ppp)-1)*2.3
endfor  
max_char=max(max_char)

avail_height=(!p.position(3)-!p.position(1))
  
if keyword_set(rot90) then begin
    avail_width=0.045
    length=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    height=max_char*(!D.x_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal)
    x_fac=((avail_height)/box(1)) < 1.0 
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_width)/box(0)) < 1.0
    fac=y_fac*x_fac
    ;if not keyword_set(AUTO) then fac=1
    charsize=!P.CHARSIZE*fac
    xcr=chrd(1)*fac/aspect & ycr=0 & deg=90 & align=0.5
    xcr = xcr*(-1)
    stpos=[(!p.position(0)-0.0625+xcr*nl) > 0.02, $
           (!p.position(1)+!p.position(3))/2]
endif else begin 
    avail_width=0.1
    length=max_char*(!D.x_ch_size)*!P.CHARSIZE
    height=n_elements(lines)*(!D.y_ch_size)*!P.CHARSIZE
    box=convert_coord(length,height,/device,/to_normal) 
    x_fac=((avail_width)/box(0)) < 1.0
    box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
    y_fac=((avail_height)/box(1)) < 1.0
    fac=y_fac*x_fac
    ;if not keyword_set(AUTO) then fac=1
    charsize=!P.CHARSIZE*fac
    if keyword_set(pos) then begin
      xcr=0 & ycr=-chrd(1)*fac & deg=0
      stpos=[0.01,!p.position(3)-ycr]
      stpos(1)=stpos(1)-(!p.position(3)-!p.position(1)-0.02)
    endif else begin 
      xcr=0 & ycr=chrd(1)*fac & deg=0
      ypos = !p.position(3)-avail_height/2
      stpos=[0.01,ypos]
    endelse  
endelse
  
IF keyword_set(extra_Par) THEN BEGIN
    ;only copy over the color tag if it exists
    tags=tag_names(extra_par)
    res=strpos(tags,'COLOR')
    index=where(res ne -1,count)
    if count ne 0 then col=extra_par.color else col=1    
ENDIF ELSE col=1
extra_Par={color:col, orientation:deg, charsize:charsize, normal:1}
  
FOR i=0,n_elements(lines)-1 DO BEGIN 
    xyouts,stpos(0)-xcr*i,stpos(1)-ycr*i,lines(i), alignment=align, $
      _extra=extra_Par
ENDFOR

;restore original panel settings
panel = panel_orig
panelset, panel	       ;re-sets the panel position viewport

END 

;******************************************************************************
;* program to draw a label next to the right side end of the x-axis labels
;* used for annotation with multiple x-axis
;* This routine needs to know the x-axis ticklength to position the label.
;* default for most plots is 0.015, use the keyword XTICKLEN to change this.
;******************************************************************************
PRO x_side_label, panel, label, XTICKLEN=XTICKLEN

COMMON PLOT_COMPOSER, widgetData
  
IF NOT keyword_set(XTICKLEN) then ticks=0.015 else ticks=XTICKLEN
  
chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)*!P.CHARSIZE
ticksize=(!p.position(3)-!p.position(1))*ticks
  
; check if inside or outside of panel is asked for.
; If inside, then put labels to the left of plot.
if widgetData.default_right_label eq 0 then begin ;outside
    ;plot x-axis labes on right side
    xyouts,!p.position(2)+chrs(0)*6,!p.position(1)-chrs(1)*1.5-ticksize,$
      label,/normal,color=1,charsize=!P.CHARSIZE
ENDIF ELSE BEGIN
    xyouts,0.05,!p.position(1)-chrs(1)*1.5-ticksize,$
      label,/normal,color=1,charsize=!P.CHARSIZE
    
ENDELSE 

END 

;******************************************************************************
;* program to draw a label to fit/expand into the size of a panel.
;* used by papco_text module. 
;******************************************************************************
pro papco_text_label, panel, label, thick, size
  
 ; get character size in normal coordinates  
  chrs=convert_coord(!D.x_ch_size,!D.y_ch_size,/device,/to_normal)
  chrd=chrs*size
    
; get aspect ratio of plot area
  aspect=float(!D.x_size)/float(!D.y_size)  
  
; get maximum length of a string, ignoring !X formatting characters
  lines=label
  dummy=lines & max_char=intarr(n_elements(lines))
  for i=0, n_elements(lines)-1 do begin
    ppp=str_sep(dummy(i),'!')
    max_char(i)=strlen(dummy(i))-(n_elements(ppp)-1)*2
  endfor  
  max_char=max(max_char)
  
; find the longest line and scale to fit (height of panel for rotated
; labels, width to end of page for horizontal labels)
  avail_height=(!p.position(3)-!p.position(1))*0.9
  avail_width =(!p.position(2)-!p.position(0))*0.9
  
  length=max_char*(!D.x_ch_size)*size
  height=n_elements(lines)*(!D.y_ch_size)*size
  box=convert_coord(length,height,/device,/to_normal) 
  x_fac=((avail_width)/box(0)) < 1.0
    ;print,avail_width, box(0), x_fac
  box=convert_coord(length*x_fac,height*x_fac,/device,/to_normal)    
  y_fac=((avail_height)/box(1)) < 1.0
    ;print, avail_height,box(1), y_fac
  fac=y_fac*x_fac
    ;print,fac
    ;if not keyword_set(AUTO) then fac=1
  charsize=size*fac

  xcr=0 & ycr=chrd(1)*fac 
  stpos=[!p.position(0)+0.02,!p.position(3)-ycr]    
  
  for i=0,n_elements(lines)-1 do begin
    xyouts,stpos(0)-xcr*i,stpos(1)-ycr*i,lines(i), $
      charsize=charsize, charthick=thick, color=1, /NORM
  endfor
   
end 
;******************************************************************************
;* program to draw a label to fit/expand into the size of a panel.
;* used by papco_text module. 
;******************************************************************************
PRO papco_in_side_label, panel, label, ROT90=ROT90, POS=POS, AUTO=AUTO, $
                                       NCOL = NCOL, _extra=extra_plotPar
  
COMMON PLOT_COMPOSER, widgetData
COMMON papco_color_names

n_ch = n_elements(label)

IF keyword_set(NCOL) THEN colors = NCOL ELSE BEGIN
    colors = [fixed_color_index, fixed_color_index, fixed_color_index]
    colors = colors(0:n_ch-1)
ENDELSE 

FOR i = 0, n_ch-1 DO BEGIN

    ch_label = label(i)
    c=i/5  &  mult=i mod 5  &  for j=1,c do ch_label='!C'+ch_label
    
    IF n_ch NE 1 then extra_plotPar.color = colors(i)
    xyouts,!P.position(0)+0.01+0.135*mult,!P.position(3)-0.02,ch_label, $
           /normal,charsize=!p.charsize,color=extra_plotPar.color
ENDFOR 

END 
