;******************************************************************************
;*  NAME:        papco_sat_conj.pro
;*
;*  DESCRIPTION: Routine to find magnetic conjunction between satellites.
;*               Implemented to work between the following satellites: 
;*                  POLAR  HEO
;*
;*
;*
;*  INPUT:       sats array of satellite(s) for conjunctions
;*
;*  OUTPUT:      none (writes conjunctions file)
;*
;*  KEYWORDS:    VERBOSE  -  enables print comments, save/restore comments
;*               L_RNG,MLT_RNG,MLAT_RNG - definition for conjunctions
;*               INTMOD, EXTMOD - mag model to use
;*               DATE_1=DATE_1, DATE_2=DATE_2 - range of time to cover
;*               ADOPT_MJDT - if set, use current time from papco
;*               NEWLOG - if set, start new log, otherwise append old
;*               LOGFLN - if set, use as file name for log
;*               SAVE - collect all interpolated L, MLT, MLAT into a file.
;*               LOAD - use saved interp arrays
;*
;*
;*  DATE:        May 2002
;*
;*  AUTOR:       R. Friedel
;*
;*  CHANGES:     
;******************************************************************************
PRO  papco_sat_conj, sats, DATE_1=DATE_1, DATE_2=DATE_2, $
                     INTMOD=INTMOD, EXTMOD=EXTMOD, $
                     L_RNG = L_RNG, MLT_RNG = MLT_RNG, MLAT_RNG = MLAT_RNG, $
                     NEWLOG = NEWLOG, LOGFLN = LOGFLN, $
                     SAVE = SAVE, LOAD = LOAD, $
                     VERBOSE = VERBOSE

COMMON  mjdt, mjdt_start, mjdt_end
COMMON  get_error, get_err_no, get_err_msg
COMMON  logfile, log_fln  

papco, /set_only & papco_mjdt_start = mjdt_start & papco_mjdt_end = mjdt_end

IF keyword_set(L_RNG) THEN  l_rng=L_RNG ELSE  l_rng=1.0
IF keyword_set(MLT_RNG) THEN  mlt_rng=MLT_RNG ELSE  mlt_rng=4.0
IF keyword_set(MLAT_RNG) THEN  mlat_rng=MLAT_RNG ELSE  mlat_rng=20.0
IF keyword_set(VERBOSE) THEN  verbose=VERBOSE ELSE  verbose=0
IF keyword_set(INTMOD) THEN kint=long(INTMOD) ELSE  kint=0l
IF keyword_set(EXTMOD) THEN kext=long(EXTMOD) ELSE  kext=0l

;------------------------------------------------------------------------------
;set time range, time step for interpolation, output arrays
IF NOT keyword_set(ADOPT_MJDT) THEN BEGIN 
    IF keyword_set(DATE_1) then date_1=DATE_1 else date_1='19971227'
    r = convert_timeToSeconds(date_1, /t90, /mjdt)
    IF r.ERR NE 0 THEN BEGIN
        message, 'wrong time format for DATE_1', /cont & return
    ENDIF ELSE mjdt_start = r.VALUE

    IF keyword_set(DATE_2) then date_2=DATE_2 else date_2='20010601' 
    r = convert_timeToSeconds(date_2, /t90, /mjdt)
    IF r.ERR NE 0 THEN BEGIN
        message, 'wrong time format for DATE_1', /cont & return
    ENDIF ELSE mjdt_end = r.VALUE
ENDIF
tstep = 5.0d * 60.0d                  ;seconds of time step
total = (mjdt_end.mjd-mjdt_start.mjd+1) * (86400 / tstep)

;make mag field coord structure
stn='MODEL_'+varprt(kint)+'_'+varprt(kext)
struct=create_struct(name='UNILB', 'L',0.0,'MLT',0.0,'MLAT',0.0) 
dummy = create_struct(name = 'PAPCO_CONJ', 'TAI', 0.0d, stn, struct)

;set up some names for arrays to be used
conj_str = '' & nsat = n_elements(sats)
r_arr = strarr(nsat) & d_arr = strarr(nsat) 
s_arr = strarr(nsat) & m_arr = strarr(nsat) 

FOR i = 0, nsat-1 DO BEGIN 
    ss = strsplit(sats(i), '_', /extract)
    CASE ss(0) OF 
        'POLAR': BEGIN
            COMMON  polar_neweph, polar_header, polar_data 
            r_arr(i) = 'r_polar_new_eph' 
        END
        'HEO': BEGIN
            COMMON  heo_eph_data, heo_header, heo_data
            r_arr(i) = 'r_heo_eph' 
            m_arr(i) = 'm_aero_heo_eph, INTMOD=INTMOD, EXTMOD=EXTMOD'
        END
        'GPS': BEGIN ;XXXX is for satellite name
            COMMON  gps_eph_data, gps_XXXX_header, gps_XXXX_data
            r_arr(i) = "r_gps_eph, '"+ ss(1)+"'" 
            m_arr(i) = "m_gps_eph, '"+ ss(1)+"'"+ $
              ",/MJDT, INTMOD=INTMOD, EXTMOD=EXTMOD"
        END
        ELSE: BEGIN
            message, 'Satellite '+sat(i)+' not implemented', /cont
            return
        END
    ENDCASE 
    conj_str = conj_str+'_'+sats(i)
    d_arr(i) = sats(i)+'_data'
    s_arr(i) = sats(i)+'_interp' 
ENDFOR 

;make the interp save arrays of structures
FOR i = 0, nsat-1 DO r = execute(s_arr(i) + ' = replicate(dummy, total)')
;filenames for saving of interpolated L, MLT, MLAT
f_arr = s_arr +'_'+ date_1 +'_'+ date_2 + '.idl' 
l_arr = d_arr + '_l' & mlt_arr = d_arr + '_mlt' & mlat_arr = d_arr + '_mlat'

;------------------------------------------------------------------------------
path = '/n/radbelt/dlnk/conj/'
IF keyword_set(LOGFLN) THEN log_fln=LOGFLN ELSE  BEGIN
    log_fln='conj'+conj_str+string(L_RNG,MLT_RNG,MLAT_RNG, $
         format="('_',f5.2,'_',f5.2,'_',f5.2,'.txt')")
    pos=strpos(log_fln,'_ ') 
    WHILE  pos NE -1 DO BEGIN 
        strput,log_fln,'_0',pos  &  pos=strpos(log_fln,'_ ') 
    ENDWHILE  
    log_fln = path+log_fln
ENDELSE   

IF keyword_set(NEWLOG) THEN BEGIN
    print_msg, '# Conjunctions'+conj_str, FILE=2
    print_msg, '#', /FILE
    print_msg, '# L_RNG:    '+varprt(l_rng), /FILE
    print_msg, '# MLT_RNG:  '+varprt(mlt_rng), /FILE
    print_msg, '# MLAT_RNG: '+varprt(mlat_rng), /FILE
    print_msg, '#', /FILE

    print_msg, '#' + strFormat('TIME', 50, /center) + $
      strFormat('L_DIFF' , 19, /center) + $
      strFormat('MLT_DIFF ' , 19, /center) + $
      strFormat('MLAT_DIFF', 19, /center) + $
      strFormat('L_AV', 10, /center) + $
      strFormat('MLT_AV', 10, /center) + $
      strFormat('MLAT_AV', 11, /center), /FILE

    print_msg, '#' + strFormat('FROM', 25, /center) + $
      strFormat('UNTIL', 26, /center) + $
      strFormat('min    max', 19, /center) + $
      strFormat('min    max', 19, /center) + $
      strFormat('min    max', 19, /center) + $
      strFormat('av', 10, /center) + $
      strFormat('av', 10, /center) + $
      strFormat('av', 11, /center), /FILE
      
ENDIF 
;------------------------------------------------------------------------------
;load interp data from arrays
IF keyword_set(LOAD) THEN BEGIN
    FOR i = 0, nsat-1 DO $
      r = execute("restore,'"+path+f_arr(i)+"',/verb")
    step = mjdt_end.mjd-mjdt_start.mjd+1
ENDIF 

;------------------------------------------------------------------------------
;loop through period in step day intervals. 
IF keyword_set(STEP) THEN step =  STEP ELSE step = 1
outer_mjdt_start = mjdt_start & outer_mjdt_end = mjdt_end
sc = 0

FOR i=outer_mjdt_start.mjd, outer_mjdt_end.mjd, step DO BEGIN 

    print, ''
    mjdt_start.mjd=i    & mjdt_start.t = 0
    mjdt_end.mjd=i+step & mjdt_end.t = 0
    tai = utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t}) 
    message, 'Doing for '+TAI2UTC(tai, /ecs)+', Step: '+varprt(step), /cont

    IF keyword_set(LOAD) THEN BEGIN
        ;load up data arrays from restored array
        FOR k = 0, nsat-1 DO BEGIN 
            r = execute('tai='+s_arr(k)+'.tai')
            r = execute(l_arr(k)+'='+s_arr(k)+'.'+stn+'.L')
            r = execute(mlt_arr(k)+'='+s_arr(k)+'.'+stn+'.MLT')
            r = execute(mlat_arr(k)+'='+s_arr(k)+'.'+stn+'.MLAT')
        ENDFOR
        ndat = n_elements(tai)
        GOTO, have_interp
    ENDIF 

    ;--------------------------------------------------------------------------
    ;read in all ephemeris info into commons, check for mag coords
    have_mag = 0
    FOR j = 0, nsat-1 DO BEGIN
        re_read:
        ss = strsplit(sats(j), '_', /extract)
        r = execute(r_arr(j)+',  VERBOSE = VERBOSE')
        IF n_elements(ss) EQ 2 THEN BEGIN
            actual_d = strsubst(d_arr(j), ss(1), 'XXXX')
            r = execute(d_arr(j)+'='+actual_d)
        ENDIF
        r = execute('tags=tag_names('+d_arr(j)+'(0))')
        index=where(tags eq stn,c)
        IF c EQ 0 THEN BEGIN
            message, d_arr(j)+' has no '+stn+' trying to make', /cont
            r = execute(m_arr(j)+',  VERBOSE = VERBOSE')
            IF get_err_no NE 0 THEN continue
            ;re-read the data file.
            GOTO, re_read
        ENDIF
        have_mag = have_mag+1
    ENDFOR

    IF have_mag NE n_elements(r_arr) THEN BEGIN
        message, 'Missing '+stn+', can not do', /cont & continue
    ENDIF 

    ;now use a common time basis
    tai_1 = utc2tai({mjd:mjdt_start.mjd, time:mjdt_start.t}) 
    tai_2 = utc2tai({mjd:mjdt_end.mjd, time:mjdt_end.t})
    ndat = long((tai_2-tai_1) / tstep)
    tai = tai_1+dindgen(ndat)*tstep

    ;fold all L, MLT, MAGLAT data onto common time basis
    ;preserving the range of no data values based on -11 in L

    ;--------------------------------------------------------------------------
    FOR k = 0, nsat-1 DO BEGIN
        r = execute('tai_orig = '+d_arr(k)+'.tai')
        IF r EQ 0 THEN r = execute('tai_orig = '+d_arr(k)+'.time')
        r = execute('l_orig = '+d_arr(k)+'.'+stn+'.L')
        r = execute('mlt_orig = '+d_arr(k)+'.'+stn+'.MLT')
        r = execute('mlat_orig = '+d_arr(k)+'.'+stn+'.MLAT')

        ;get valid interpolated times (within tstep of tai_orig). 
        valid_times = fltarr(ndat) 
        FOR j = 0, ndat-1 DO BEGIN 
            diff = abs(tai_orig-tai(j))
            valid_times(j) = min(diff)
        ENDFOR 
        non_valid_time_idx = where(valid_times GT tstep, c_non_valid_time)
        IF c_non_valid_time NE 0 THEN non_valid_idx = non_valid_time_idx
        
        ;getindices of valid L's in interpolated array
        valid_orig = fltarr(n_elements(tai_orig))
        idx = where(l_orig GT 0, c)
        IF c NE 0 THEN valid_orig(idx) = 1
        valid = interpol(valid_orig, tai_orig, tai)
        non_valid_L_idx = where(valid ne 1, c_non_valid_L)
        IF c_non_valid_L NE 0 THEN IF keyword_set(non_valid_idx) THEN $
          non_valid_idx = [non_valid_idx, non_valid_L_idx] ELSE $
          non_valid_idx = non_valid_L_idx

        c_non_valid =  c_non_valid_time+ c_non_valid_L 

        ;now interpolate L, mlt, mlat; set non-valid's to -99
        r = execute(l_arr(k)+' = interpol(l_orig,tai_orig,tai)') 
        IF c_non_valid NE 0 THEN $
          r = execute(l_arr(k)+'(non_valid_idx) = -99.0')

        r = execute(mlt_arr(k)+' = interpol(mlt_orig,tai_orig,tai)') 
        IF c_non_valid NE 0 THEN $
          r = execute(mlt_arr(k)+'(non_valid_idx) = -99.0')

        r = execute(mlat_arr(k)+' = interpol(mlat_orig,tai_orig,tai)') 
        IF c_non_valid NE 0 THEN $
          r = execute(mlat_arr(k)+'(non_valid_idx) = -99.0')

        ;collect interpolated data in save array
        r = execute(s_arr(k)+'(sc:sc+ndat-1).tai = tai')
        r = execute(s_arr(k)+'(sc:sc+ndat-1).'+stn+'.L = '+l_arr(k))
        r = execute(s_arr(k)+'(sc:sc+ndat-1).'+stn+'.MLT = '+mlt_arr(k))
        r = execute(s_arr(k)+'(sc:sc+ndat-1).'+stn+'.MLAT = '+mlat_arr(k))
    ENDFOR
    sc = sc+ndat

    have_interp:

    ;now loop through common time and find ranges of L, MLT, MAGLAT
    ;for each satellite at each common time.
    lr = fltarr(ndat) & mltr = fltarr(ndat) & mlatr = fltarr(ndat)
    ls = fltarr(nsat) & mlts = fltarr(nsat) & mlats = fltarr(nsat)
    tai_in = tai

    ;--------------------------------------------------------------------------
    FOR k = 0l, ndat-1 DO BEGIN

        ;first, get all L, MLT, MLAT values at a given time.
        FOR j = 0, nsat-1 DO BEGIN 
            r = execute('ls(j) = '+l_arr(j)+'(k)')
            r = execute('mlts(j) = '+mlt_arr(j)+'(k)')
            r = execute('mlats(j) = '+mlat_arr(j)+'(k)')
        ENDFOR 
        
        ;now work out maximum seperation in L, MLT, MLAT.
        idx = where(ls EQ -99, c)
        IF c NE 0 THEN BEGIN
            lr(k) = -99 & mltr(k) = -99 & mlatr(k) = -99
            continue
        ENDIF
        lr(k) = max(ls)-min(ls) 

        ;for mlt more tricky, due to wraparound at midnight
        mlat_diffs = fltarr(nsat, nsat) 
        FOR n = 0, nsat-1 DO BEGIN
            mlat1 = mlts(n) 
            FOR p = 0, nsat-1 DO BEGIN
                mlat2 = mlts(p)
                diff_a = base24subtract(mlat1, mlat2)
                diff_b = base24subtract(mlat2, mlat1)
                IF diff_a LT diff_b THEN mlat_diffs(n, p) = diff_a $
                ELSE mlat_diffs(n, p) = diff_b
            ENDFOR
        ENDFOR
        mltr(k) = max(mlat_diffs)

        mlatr(k) = max(mlats)-min(mlats) 
           
    ENDFOR
   
    ;--------------------------------------------------------------------------
    ;have arrays of max seperation in L, MLT, MLAT @ each time
    ;now can test for conditions required.

    ;test for L condition
    idx_l = where((lr NE -99) AND (lr LE l_rng), c_l)
    print, '  Points in L-range:    '+varprt(c_l)
    IF c_l EQ 0 THEN continue

    ;reduce test population
    tai = tai(idx_l)
    lr = lr(idx_l)
    mltr = mltr(idx_l)
    mlatr = mlatr(idx_l)

    ;test for MLT condition
    idx_mlt = where((mltr NE -99) AND (mltr LE mlt_rng), c_mlt)
    print, '  Points in MLT-range:  '+varprt(c_mlt)
    IF c_mlt EQ 0 THEN continue

    ;reduce test population
    tai = tai(idx_mlt)
    lr = lr(idx_mlt)
    mltr = mltr(idx_mlt)
    mlatr = mlatr(idx_mlt)

    ;test for MLAT condition
    idx_mlat = where((mlatr NE -99) AND (mlatr LE mlat_rng), c_mlat)
    print, '  Points in MLAT-range: '+varprt(c_mlat)
    IF c_mlat EQ 0 THEN continue

    ;reduce test population
    tai = tai(idx_mlat)
    lr = lr(idx_mlat)
    mltr = mltr(idx_mlat)
    mlatr = mlatr(idx_mlat)

    ;--------------------------------------------------------------------------
    ;if we got here, then a conjunction is found. Report. 
    ;could be that more than one conjunction is found -
    ;need to seperate by time!
    ;if times in tai are seperated by more than tstep, need to split. 
    ct_idx = lonarr(c_mlat, 2) & cn = 0 & k = 0 

    IF c_mlat EQ 1 THEN BEGIN
        ct_idx(0, 0) = 0 & ct_idx(0, 1) = 0 & cn = 1
    ENDIF ELSE BEGIN 
        WHILE k LE c_mlat-2 DO BEGIN
            ct_idx(cn, 0) = k
            WHILE tai(k+1) EQ tai(k)+tstep DO BEGIN
                k = k+1 
                IF k GT c_mlat-2 THEN break
            ENDWHILE
            ct_idx(cn, 1) = k
            cn = cn+1 & k = k+1
        ENDWHILE 
        ct_idx = ct_idx(0:cn-1, *)
    ENDELSE
    ;print, ct_idx

    ;--------------------------------------------------------------------------
    ;loop through the cn conjunctions found
    FOR j = 0, cn-1 DO BEGIN 
      
        t = tai(ct_idx(j, *))
        l = lr(ct_idx(j, *))
        mlt = mltr(ct_idx(j, *))
        mlat = mlatr(ct_idx(j, *))

        ;now find the min, max of points of conjunction
        tai_rng_str = tai2utc(t(0), /ECS)+' - '+tai2utc(t(1), /ECS)

        l_max = max(l, min = l_min)
        l_diff_str=string(l_min,l_max, format="(f6.3,' - ',f6.3)")

        mlt_max = max(mlt, min = mlt_min)
        mlt_diff_str=string(mlt_min,mlt_max, format="(f6.3,' - ',f6.3)")

        mlat_max = max(mlat, min = mlat_min)
        mlat_diff_str=string(mlat_min,mlat_max, format="(f6.3,' - ',f6.3)")
    
        ist = where(tai_in EQ tai(ct_idx(j, 0)))
        ien = where(tai_in EQ tai(ct_idx(j, 1)))
        nav = ien(0)-ist(0)+1

        ;and find the average position of the satellites in L, MLT, MLAT
        l_av = 0 & mlt_av = 0 & mlat_av = 0
        FOR k = 0, nsat-1 DO BEGIN
            ex_str = 'l_av = l_av+total('+l_arr(k)+'(ist(0):ien(0)))'
            r = execute(ex_str)
            ex_str = 'mlt_av = mlt_av+total('+mlt_arr(k)+'(ist(0):ien(0)))'
            r = execute(ex_str)
            ex_str = 'mlat_av = mlat_av+total('+mlat_arr(k)+'(ist(0):ien(0)))'
            r = execute(ex_str)
        ENDFOR

        l_av = l_av / (nav*nsat)
        l_av_str =string(l_av, format="(f6.3)")
        mlt_av = mlt_av / (nav*nsat)
        mlt_av_str =string(mlt_av, format="(f6.3)")
        mlat_av = mlat_av / (nav*nsat)
        mlat_av_str =string(mlat_av, format="(f7.3)")
  
        msg =  tai_rng_str+'    '+l_diff_str+'    '+mlt_diff_str+'    '+ $
          mlat_diff_str+'    '+l_av_str+'    '+mlt_av_str+'    '+mlat_av_str
    
        print_msg, msg, /FILE

    ENDFOR 

ENDFOR 
;------------------------------------------------------------------------------
;save collected interpolated L, MLT, MLAT to file
IF keyword_set(SAVE) THEN BEGIN 
    print, ''
    message, 'Saving interpolated data to file',/cont
    FOR i = 0, nsat-1 DO BEGIN
        print, '  ', f_arr(i)
        r = execute(s_arr(i)+'='+s_arr(i)+'(0:sc-1)') ;actual size of arrays
        r = execute("save,file='"+path+f_arr(i)+"',"+s_arr(i)+",/comp,/verb")
    ENDFOR
ENDIF

END 

;------------------------------------------------------------------------------
;program to output to log file
PRO  print_msg, msg, FILE=FILE
  
COMMON  logfile, fln
  
print,msg
if keyword_set(FILE) then file=FILE else file=0
if file ne 0 then BEGIN 
    if file eq 2 then append=0 else append=1
    openw,unit,fln,/get_lun,APPEND=append
    printf,unit,msg
    close,unit & free_lun,unit
ENDIF 
  
END 

;------------------------------------------------------------------------------
PRO  papco_sat_conj_batch
  sats = ['GPS_ns33', 'HEO']
  papco_sat_conj, sats, /NEWLOG, /LOAD, L_RNG=0.25, MLT_RNG=0.5, MLAT_RNG=5
  papco_sat_conj, sats, /NEWLOG, /LOAD, L_RNG=0.50, MLT_RNG=1.0, MLAT_RNG=10
  papco_sat_conj, sats, /NEWLOG, /LOAD, L_RNG=1.00, MLT_RNG=2.0, MLAT_RNG=15
end  

;------------------------------------------------------------------------------
; do a subtraction (a-b) in a base 24 (hour) system.
FUNCTION base24subtract,a,b
if a ge b then return, (a-b) else return, 24 - (b-a)      
END 
